import json
import logging
from datetime import datetime

from requests import Request, Session
from requests_throttler import ThrottledRequest, BaseThrottler

from config import http_params_paginated, http_params
from utils.parse_dependencies import determine_list_of_versions


def find_dependency(response: ThrottledRequest, name):
    try:
        repo_info = response.response.json()
    except json.decoder.JSONDecodeError:
        logging.getLogger().error('Could not decode response')
        return None, None
    if 'full_name' not in repo_info:
        logging.getLogger().error('Libraries.io server error')
        return None, None
    if 'dependencies' not in repo_info:
        logging.getLogger().error('Response does not contain dependencies for: {}'.format(repo_info['full_name']))
        return None, repo_info
    investigated_dep = [dep for dep in repo_info['dependencies'] if dep['project_name'] == name]
    if not len(investigated_dep) > 0:
        logging.getLogger().error('Response does not contain investigated dependency for: {}'.format(repo_info['full_name']))
        return None, repo_info
    return investigated_dep[0], repo_info


def execute_http_request(request: Request, timeout=10):
    session = Session()
    response = None
    for i in range(10):
        try:
            response = session.send(request.prepare(), timeout=timeout)
        except:
            continue
        if response.status_code == 200:
            return response
    print('something went wrong')
    print(response.status_code)
    print(response.text)
    return Exception


def find_repos_with_fixed_versions_for(package_coords: str, fix_release_date: datetime, n=10, current_repos=None):
    if current_repos is None:
        current_repos = []
    repos_with_fixed_version = []
    page = 1
    versions = determine_list_of_versions(package_coords)
    logging.getLogger().debug('Recognised versions: {}'.format(versions))
    while len(repos_with_fixed_version) < n:
        logging.getLogger().info('Currently {} out of {} repos for: {}'.format(len(repos_with_fixed_version), n, package_coords))
        r = execute_http_request(
            Request(method='GET', url='https://libraries.io/api/Maven/{}/dependent_repositories'.format(package_coords),
                    params=http_params_paginated(page))
        )
        page += 1
        repositories = [repo for repo in r.json() if repo['host_type'] == 'GitHub']
        if len(repositories) == 0:
            return repos_with_fixed_version
        with BaseThrottler(name='base-throttler', delay=1.07) as bt:
            dependency_requests = [
                Request(method='GET', url='https://libraries.io/api/github/{}/dependencies'.format(repo['full_name']),
                        params=http_params) for repo in repositories]
            throttled_requests = bt.multi_submit(dependency_requests)

        for dependency_response in throttled_requests:
            dependency, repo_info = find_dependency(dependency_response, package_coords)
            if dependency is None or dependency['requirements'] is None:
                logging.getLogger().warning('Skipping because malformed response or not in requirements: {}'.format(
                    repo_info['full_name'] if repo_info and 'full_name' in repo_info else 'UNKNOWN'
                ))
                continue
            required_version = dependency['requirements']
            if repo_info['full_name'] in current_repos:
                logging.getLogger().warning('Skipping {} - already scanned'.format(repo_info['full_name']))
                continue
            if required_version not in versions or repo_info['fork'] is True:
                logging.getLogger().warning('skipping {} - fork: {} - required version: {}'.format(repo_info['full_name'], repo_info['fork'], required_version))
                logging.getLogger().debug('Dependency requirements: {}'.format(dependency))
                continue
            logging.getLogger().debug('Version required: {} - First fix version date: {} - should include: {}'.format(versions[required_version], fix_release_date, versions[required_version] > fix_release_date))
            if versions[required_version] > fix_release_date:
                logging.getLogger().info('Found repo: {}'.format(repo_info['full_name']))
                repos_with_fixed_version.append({'full_name': repo_info['full_name'], 'pom_path': dependency['filepath']})

    return repos_with_fixed_version
