import logging
import os
import datetime

from git import Repo
from pytz import utc
from tqdm import tqdm

from config import OUT_DIR, get_out_dir_for, CLONE_DIR
from src.utils.parse_dependencies import get_maven_dependencies, determine_list_of_versions
from src.utils.vcs import clone_repository, get_pom_commits, ensure_out_dir, ensure_repo_dir

version_cache = {}


def traverse_pom_history(repo: Repo, package, fixed_date):
    repo_name = repo.working_dir.split('/')[-1]
    print('running for ', repo_name)
    ensure_out_dir(repo_name)
    pom_commits = get_pom_commits(repo.working_dir, 'pom.xml')
    dependency_versions = {}
    delays = {}
    fix_delay = None
    for commit in tqdm(pom_commits):
        repo.git.checkout(commit.hash)
        dependencies = get_maven_dependencies(repo_name, commit.hash)
        for dependency in dependencies:
            key = '{}:{}'.format(dependency['group'], dependency['artifact'])
            if key not in version_cache:
                version_cache[key] = determine_list_of_versions(dependency)
            if key not in version_cache or dependency['version'] not in version_cache[key]:
                logging.getLogger().info('skipping {}'.format(dependency))
                continue
            release_date = version_cache[key][dependency['version']]
            if key not in dependency_versions:
                dependency_versions[key] = (dependency['version'], release_date)
            try:
                previous_version, previous_release_date = dependency_versions[key]
                is_update = dependency['version'] != previous_version and release_date > previous_release_date
            except TypeError:
                logging.getLogger().error('Date comparison failure between new: {}, old: {}'.format(
                    release_date,
                    dependency_versions[key]
                ))
                continue
            if is_update:
                patch_delay = commit.committer_date - version_cache[key][dependency['version']]
                if patch_delay.days > 0:
                    logging.getLogger().info('UPDATE to {} DELAY: {} commit: {} on {}, new: {} - old: {} - release date: {}'.format(
                        key,
                        patch_delay.days,
                        commit.hash,
                        commit.committer_date,
                        dependency,
                        dependency_versions[key],
                        version_cache[key][dependency['version']]
                    ))

                    if key not in delays:
                        delays[key] = {}
                    delays[key][dependency['version']] = patch_delay
                    dependency_versions[key] = (dependency['version'], release_date)

                    if key == package and release_date >= fixed_date and fix_delay is None:
                        fix_delay = patch_delay
    return delays, fix_delay


def main():
    package = 'com.google.guava:guava'
    vulnerability = 'CVE-2020-8908'
    fixed_date = datetime.datetime(2020, 8, 27, tzinfo=utc)

    # repos = find_repos_with_fixed_versions_for(package, '30', 2)
    # repos = [
    #     {'full_name': 'SERG-Delft/jpacman-framework', 'pom_path': 'pom.xml'},
    #     # {'full_name': 'fasten-project/fasten', 'pom_path': 'pom.xml'},
    # ]
    repos = [
        # {'full_name': 'snazy/ohc', 'pom_path': 'pom.xml'},
        {'full_name': 'siom79/japicmp', 'pom_path': 'pom.xml'},
        # {'full_name': 'bonigarcia/selenium-jupiter', 'pom_path': 'pom.xml'},
        # {'full_name': 'neo4j-contrib/cypher-dsl', 'pom_path': 'pom.xml'},
        # {'full_name': 'wnameless/json-flattener', 'pom_path': 'pom.xml'},
        {'full_name': 'openscoring/openscoring', 'pom_path': 'pom.xml'}
    ]
    print(repos)
    for repo_meta in repos:
        repo = clone_repository(repo_meta['full_name'])
        # repo_name = repo.working_dir.split('/')[-1]
        # ensure_out_dir(repo_name)
        # dependency = get_maven_dependencies(repo_name)[0]
        # print(determine_list_of_versions(dependency))

        delays, fix_delay = traverse_pom_history(repo, package, fixed_date)
        days = []
        for delay in delays.values():
            days = days + [delta.days for delta in delay.values()]
        print(days)
        print(sum(days) / len(days))
        print(fix_delay)

        # commit = find_fix_commit(repo, package, vulnerability, repo_meta['pom_path'])

        # patch_delay = commit.committer_date - fix_date
        # print(patch_delay.days)
        # if patch_delay.days > 0:
        #     lag_data.append({'repo': repo_meta['full_name'], 'days': patch_delay.days})

    # plot_lag_distribution(lag_data)


if __name__ == '__main__':
    ensure_out_dir()
    ensure_repo_dir()
    # noinspection PyArgumentList
    logging.basicConfig(filename='../../out/run.log', filemode='w', level=logging.INFO, force=True)

    main()
    # day_delays = np.array([81, 4, 124, 35, 36, 106, 154, 4, 124, 35, 36, 106, 154, 74, 441, 559, 44, 27, 169, 44, 27, 169, 44, 27, 169, 90, 520, 315, 46, 213, 131, 307, 5, 71, 89, 37, 786, 983, 10, 527, 237, 884, 928, 26, 23])
    # print(len(day_delays))
    # fix_delay = 10
    # sns.displot(day_delays, kde=True, binwidth=5)
    # plt.show()


# Research questions important based on evaluation method
# Sliding window calculations on timeframe

# cyper-dsl
# [3, 15, 11, 27, 5, 9, 10, 9, 7, 19, 29, 9, 51, 3, 25, 3, 15, 11, 27, 5, 9, 10, 9, 7, 19, 29, 9, 51, 3, 25, 3, 15, 11, 27, 5, 9, 10, 9, 7, 19, 29, 9, 51, 3, 25, 3, 15, 11, 27, 5, 9, 10, 9, 7, 19, 29, 9, 51, 3, 25, 130, 175, 73, 136, 27, 5, 9, 10, 9, 7, 19, 29, 9, 51, 3, 25, 154, 140, 136, 105, 173, 284, 168, 24, 36, 50, 149, 70, 9, 34, 50, 31, 9, 34, 50, 31, 9, 34, 50, 31, 9, 34, 551, 551, 303, 859, 741, 576, 265, 550, 469, 145, 29, 9, 51, 3, 29, 9, 51, 29, 9, 51, 31, 494, 709, 855, 314, 1593, 250, 25, 301, 301, 168, 61, 61, 23, 411, 172, 738, 35, 35, 35, 35, 35, 35]
# 107.21379310344828
# 145 days, 15:40:18

# Selenium-jupiter
# [3, 15, 11, 27, 5, 9, 10, 9, 7, 19, 29, 9, 51, 3, 25, 3, 15, 11, 27, 5, 9, 10, 9, 7, 19, 29, 9, 51, 3, 25, 3, 15, 11, 27, 5, 9, 10, 9, 7, 19, 29, 9, 51, 3, 25, 3, 15, 11, 27, 5, 9, 10, 9, 7, 19, 29, 9, 51, 3, 25, 130, 175, 73, 136, 27, 5, 9, 10, 9, 7, 19, 29, 9, 51, 3, 25, 154, 140, 136, 105, 173, 284, 168, 24, 36, 50, 149, 70, 9, 34, 50, 31, 9, 34, 50, 31, 9, 34, 50, 31, 9, 34, 551, 551, 303, 859, 741, 576, 265, 550, 469, 145, 29, 9, 51, 3, 29, 9, 51, 29, 9, 51, 31, 494, 709, 855, 314, 1593, 250, 25, 301, 301, 168, 61, 61, 23, 411, 172, 738, 35, 35, 35, 35, 35, 35]
# 107.21379310344828
# 145 days, 15:40:18
# 47 days, 17:30:11

# [41, 37, 49, 90, 91, 62, 7, 153, 83, 94, 151, 118, 103, 59, 60, 102, 128, 37, 11, 94, 405, 336, 195, 241, 965, 132, 257, 114, 38, 22, 26, 38, 22, 26, 38, 22, 26, 153, 11, 5, 36, 36, 36, 36, 36, 36, 36, 39, 39, 39, 39, 39, 39, 39]
# 95.68518518518519
# 7 days, 16:09:37
# running for  formatter-maven-plugin