import datetime
import re
from requests import Request

import pandas as pd

from utils.libraries_api import execute_http_request
from utils.parse_dependencies import determine_list_of_versions


def get_first_fix_data(package_name, cve):
    try:
        print(f'working on {package_name} - {cve}')
        cve_detail = execute_http_request(
            Request(method='GET', url='https://services.nvd.nist.gov/rest/json/cve/1.0/{}'.format(cve)), timeout=5)
        cve_detail = cve_detail.json()
        package_versions = determine_list_of_versions(package_name)
    except ValueError:
        print('invalid json')
        return 'UNKNOWN', 'UNKNOWN'

    try:
        cpe_information = cve_detail['result']['CVE_Items'][0]['configurations']['nodes'][0]['cpe_match'][0]
        version_prefix = cpe_information.get('versionEndExcluding', cpe_information.get('versionEndIncluding'))
        if version_prefix is None:
            raise KeyError('No end version specified for cpe {}'.format(cpe_information))
        version_prefix_is_vulnerable = 'versionEndIncluding' in cpe_information
        maven_fix_version = find_fix_version(list(package_versions.keys()), version_prefix,
                                             version_prefix_is_vulnerable)
        fix_release_date = package_versions[maven_fix_version]
        return maven_fix_version, fix_release_date
    except KeyError as e:
        print('could not decipher things')
        print(e)
        return 'UNKNOWN', 'UNKNOWN'
    except IndexError:
        print('could not find cpe information in {}'.format(cve_detail))
        return 'UNKNOWN', 'UNKNOWN'


def find_fix_version(package_versions, version_prefix, version_vulnerable_included=False):
    l = len(package_versions)
    for index, version in enumerate(package_versions):
        if re.match('{}[^0-9]*'.format(version_prefix), version):
            fix_version = version
            if version_vulnerable_included and index < (l - 1):
                fix_version = package_versions[index + 1]
            return fix_version


if __name__ == '__main__':
    cves = pd.read_csv(open('cves.csv'))
    cves['libraries_io_find_prefix'] = ''
    cves['first_fix_release_date'] = ''
    cves['samples'] = 30

    for i, row in cves.iterrows():
        first_fix_version, first_fix_date = get_first_fix_data(row['package_coords'], row['cve'])
        cves.at[i, 'libraries_io_find_prefix'] = first_fix_version
        cves.at[i, 'first_fix_release_date'] = first_fix_date

    for i, row in cves.iterrows():
        if row['first_fix_release_date'] != 'UNKNOWN':
            print('INSERT INTO vulnerabilities (cve, package_coords, libraries_io_find_prefix, first_fix_release_date) VALUES ({}, {}, {}, {})'.format(
                row['cve'],
                row['package_coords'],
                row['libraries_io_find_prefix'],
                row['first_fix_release_date'].isoformat()
            ))
