import logging
from datetime import datetime

from mysql.connector import connect
from pytz import utc

from utils.core import Status
from utils.libraries_api import find_repos_with_fixed_versions_for
from utils.runner import Runner


class RepoFinder(Runner):
    def _add_repos_to_db(self, repos, cve):
        repos = [{**repo, 'cve': cve}for repo in repos]
        with self._connection.cursor() as cursor:
            cursor.executemany(
                'INSERT INTO repos (full_name, pom_path, cve) VALUES (%(full_name)s, %(pom_path)s, %(cve)s)',
                repos
            )
            cursor.execute(f"UPDATE vulnerabilities SET status='{Status.DONE}' WHERE cve=%s", [cve])
        self._connection.commit()

    def run_once(self):
        with self._connection.cursor() as cursor:
            cursor.execute(f"""
                SELECT v.cve, package_coords, samples, COUNT(r.cve) as current_samples, first_fix_release_date
                FROM vulnerabilities as v
                LEFT JOIN repos r ON r.cve = v.cve
                WHERE v.status = '{Status.NEW}' OR v.status = '{Status.CHANGED}'
                GROUP BY r.cve
            """)
            for (cve, package_coords, wanted_samples, current_samples, first_fix_release_date) in cursor:
                with self._connection.cursor(buffered=True) as sub_cursor:
                    sub_cursor.execute(f"SELECT r.full_name FROM repos as r WHERE cve = '{cve}'")
                    current_repos = [row[0] for row in sub_cursor.fetchall()]
                first_fix_release_date = datetime.combine(first_fix_release_date, datetime.min.time(), utc)
                number_of_new_repos = max(0, wanted_samples - current_samples)
                self._logger.info('Finding {} new repos for {}'.format(number_of_new_repos, cve))
                repos = find_repos_with_fixed_versions_for(package_coords, first_fix_release_date, number_of_new_repos, current_repos)
                self._add_repos_to_db(repos, cve)
            self._logger.info('No more vulnerabilities to find repos for')


if __name__ == '__main__':
    logging.getLogger().setLevel(logging.DEBUG)
    with connect(host='127.0.0.1', port=33062, user='vulnerability-history', database='vulnerability-history',
                 password='secret') as conn:
        repo_finder = RepoFinder(conn, logging.getLogger())
        repo_finder.run_once()
