#include "test/storm_gtest.h"
#include "storm-config.h"

#include "storm-pomdp/modelchecker/BeliefExplorationPomdpModelChecker.h"
#include "storm-pomdp/transformer/MakePOMDPCanonic.h"
#include "storm/api/storm.h"
#include "storm-parsers/api/storm-parsers.h"
#include "storm-pomdp/transformer/GlobalPOMDPSelfLoopEliminator.h"
#include "storm-pomdp/analysis/QualitativeAnalysisOnGraphs.h"
#include "storm-pomdp/transformer/KnownProbabilityTransformer.h"

#include "storm/environment/solver/MinMaxSolverEnvironment.h"


namespace {
    enum class PreprocessingType { None, SelfloopReduction, QualitativeReduction, All };
    
    class DefaultDoubleVIEnvironment {
    public:
        typedef double ValueType;
        static storm::Environment createEnvironment() {
            storm::Environment env;
            env.solver().minMax().setMethod(storm::solver::MinMaxMethod::ValueIteration);
            env.solver().minMax().setPrecision(storm::utility::convertNumber<storm::RationalNumber>(1e-6));
            return env;
        }
        static bool const isExactModelChecking = false;
        static ValueType precision() { return storm::utility::convertNumber<ValueType>(0.12); } // there actually aren't any precision guarantees, but we still want to detect if results are weird.
        static void adaptOptions(storm::pomdp::modelchecker::BeliefExplorationPomdpModelCheckerOptions<ValueType>&) { /* intentionally left empty */ }
        static PreprocessingType const preprocessingType = PreprocessingType::None;
    };
    
    class SelfloopReductionDefaultDoubleVIEnvironment {
    public:
        typedef double ValueType;
        static storm::Environment createEnvironment() {
            storm::Environment env;
            env.solver().minMax().setMethod(storm::solver::MinMaxMethod::ValueIteration);
            env.solver().minMax().setPrecision(storm::utility::convertNumber<storm::RationalNumber>(1e-6));
            return env;
        }
        static bool const isExactModelChecking = false;
        static ValueType precision() { return storm::utility::convertNumber<ValueType>(0.12); } // there actually aren't any precision guarantees, but we still want to detect if results are weird.
        static void adaptOptions(storm::pomdp::modelchecker::BeliefExplorationPomdpModelCheckerOptions<ValueType>& options) { /* intentionally left empty */ }
        static PreprocessingType const preprocessingType = PreprocessingType::SelfloopReduction;
    };
    
    class QualitativeReductionDefaultDoubleVIEnvironment {
    public:
        typedef double ValueType;
        static storm::Environment createEnvironment() {
            storm::Environment env;
            env.solver().minMax().setMethod(storm::solver::MinMaxMethod::ValueIteration);
            env.solver().minMax().setPrecision(storm::utility::convertNumber<storm::RationalNumber>(1e-6));
            return env;
        }
        static bool const isExactModelChecking = false;
        static ValueType precision() { return storm::utility::convertNumber<ValueType>(0.12); } // there actually aren't any precision guarantees, but we still want to detect if results are weird.
        static void adaptOptions(storm::pomdp::modelchecker::BeliefExplorationPomdpModelCheckerOptions<ValueType>& options) { /* intentionally left empty */ }
        static PreprocessingType const preprocessingType = PreprocessingType::QualitativeReduction;
    };
    
    class PreprocessedDefaultDoubleVIEnvironment {
    public:
        typedef double ValueType;
        static storm::Environment createEnvironment() {
            storm::Environment env;
            env.solver().minMax().setMethod(storm::solver::MinMaxMethod::ValueIteration);
            env.solver().minMax().setPrecision(storm::utility::convertNumber<storm::RationalNumber>(1e-6));
            return env;
        }
        static bool const isExactModelChecking = false;
        static ValueType precision() { return storm::utility::convertNumber<ValueType>(0.12); } // there actually aren't any precision guarantees, but we still want to detect if results are weird.
        static void adaptOptions(storm::pomdp::modelchecker::BeliefExplorationPomdpModelCheckerOptions<ValueType>& options) { /* intentionally left empty */ }
        static PreprocessingType const preprocessingType = PreprocessingType::All;
    };

    class FineDoubleVIEnvironment {
    public:
        typedef double ValueType;
        static storm::Environment createEnvironment() {
            storm::Environment env;
            env.solver().minMax().setMethod(storm::solver::MinMaxMethod::ValueIteration);
            env.solver().minMax().setPrecision(storm::utility::convertNumber<storm::RationalNumber>(1e-6));
            return env;
        }
        static bool const isExactModelChecking = false;
        static ValueType precision() { return storm::utility::convertNumber<ValueType>(0.02); } // there actually aren't any precision guarantees, but we still want to detect if results are weird.
        static void adaptOptions(storm::pomdp::modelchecker::BeliefExplorationPomdpModelCheckerOptions<ValueType>& options) {
            options.resolutionInit = 24;
            options.gapThresholdInit = storm::utility::convertNumber<ValueType>(0.001);
        }
        static PreprocessingType const preprocessingType = PreprocessingType::None;
    };

    class RefineDoubleVIEnvironment {
    public:
        typedef double ValueType;
        static storm::Environment createEnvironment() {
            storm::Environment env;
            env.solver().minMax().setMethod(storm::solver::MinMaxMethod::ValueIteration);
            env.solver().minMax().setPrecision(storm::utility::convertNumber<storm::RationalNumber>(1e-6));
            return env;
        }
        static bool const isExactModelChecking = false;
        static ValueType precision() { return storm::utility::convertNumber<ValueType>(0.005); }
        static PreprocessingType const preprocessingType = PreprocessingType::None;
        static void adaptOptions(storm::pomdp::modelchecker::BeliefExplorationPomdpModelCheckerOptions<ValueType>& options) {options.refine = true; options.refinePrecision = precision();}
    };

    class PreprocessedRefineDoubleVIEnvironment {
    public:
        typedef double ValueType;
        static storm::Environment createEnvironment() {
            storm::Environment env;
            env.solver().minMax().setMethod(storm::solver::MinMaxMethod::ValueIteration);
            env.solver().minMax().setPrecision(storm::utility::convertNumber<storm::RationalNumber>(1e-6));
            return env;
        }
        static bool const isExactModelChecking = false;
        static ValueType precision() { return storm::utility::convertNumber<ValueType>(0.005); }
        static PreprocessingType const preprocessingType = PreprocessingType::All;
        static void adaptOptions(storm::pomdp::modelchecker::BeliefExplorationPomdpModelCheckerOptions<ValueType>& options) {options.refine = true; options.refinePrecision = precision();}
    };
    
    class DefaultDoubleOVIEnvironment {
    public:
        typedef double ValueType;
        static storm::Environment createEnvironment() {
            storm::Environment env;
            env.solver().minMax().setMethod(storm::solver::MinMaxMethod::SoundValueIteration);
            env.solver().minMax().setPrecision(storm::utility::convertNumber<storm::RationalNumber>(1e-6));
            env.solver().setForceSoundness(true);
            return env;
        }
        static bool const isExactModelChecking = false;
        static ValueType precision() { return storm::utility::convertNumber<ValueType>(0.12); } // there actually aren't any precision guarantees, but we still want to detect if results are weird.
        static void adaptOptions(storm::pomdp::modelchecker::BeliefExplorationPomdpModelCheckerOptions<ValueType>& options) { /* intentionally left empty */ }
        static PreprocessingType const preprocessingType = PreprocessingType::None;
    };
    
    class DefaultRationalPIEnvironment {
    public:
        typedef storm::RationalNumber ValueType;
        static storm::Environment createEnvironment() {
            storm::Environment env;
            env.solver().minMax().setMethod(storm::solver::MinMaxMethod::PolicyIteration);
            env.solver().setForceExact(true);
            return env;
        }
        static bool const isExactModelChecking = true;
        static ValueType precision() { return storm::utility::convertNumber<ValueType>(0.12); } // there actually aren't any precision guarantees, but we still want to detect if results are weird.
        static void adaptOptions(storm::pomdp::modelchecker::BeliefExplorationPomdpModelCheckerOptions<ValueType>& options) { /* intentionally left empty */ }
        static PreprocessingType const preprocessingType = PreprocessingType::None;
    };
    
    class PreprocessedDefaultRationalPIEnvironment {
    public:
        typedef storm::RationalNumber ValueType;
        static storm::Environment createEnvironment() {
            storm::Environment env;
            env.solver().minMax().setMethod(storm::solver::MinMaxMethod::PolicyIteration);
            env.solver().setForceExact(true);
            return env;
        }
        static bool const isExactModelChecking = true;
        static ValueType precision() { return storm::utility::convertNumber<ValueType>(0.12); } // there actually aren't any precision guarantees, but we still want to detect if results are weird.
        static void adaptOptions(storm::pomdp::modelchecker::BeliefExplorationPomdpModelCheckerOptions<ValueType>& options) { /* intentionally left empty */ }
        static PreprocessingType const preprocessingType = PreprocessingType::All;
    };
    
    template<typename TestType>
    class BeliefExplorationTest : public ::testing::Test {
    public:
        typedef typename TestType::ValueType ValueType;
        BeliefExplorationTest() : _environment(TestType::createEnvironment()) {}
        storm::Environment const& env() const { return _environment; }
        storm::pomdp::modelchecker::BeliefExplorationPomdpModelCheckerOptions<ValueType> options() const {
            storm::pomdp::modelchecker::BeliefExplorationPomdpModelCheckerOptions<ValueType> opt(true, true); // Always compute both bounds (lower and upper)
            TestType::adaptOptions(opt);
            return opt;
        }
        ValueType parseNumber(std::string const& str) {
            return storm::utility::convertNumber<ValueType>(str);
        }
        struct Input {
            std::shared_ptr<storm::models::sparse::Pomdp<ValueType>> model;
            std::shared_ptr<storm::logic::Formula const> formula;
        };
        Input buildPrism(std::string const& programFile, std::string const& formulaAsString, std::string const& constantsAsString = "") const {
            // Parse and build input
            storm::prism::Program program = storm::api::parseProgram(programFile);
            program = storm::utility::prism::preprocess(program, constantsAsString);
            Input input;
            input.formula = storm::api::parsePropertiesForPrismProgram(formulaAsString, program).front().getRawFormula();
            input.model = storm::api::buildSparseModel<ValueType>(program, {input.formula})->template as<storm::models::sparse::Pomdp<ValueType>>();
            
            // Preprocess
            storm::transformer::MakePOMDPCanonic<ValueType> makeCanonic(*input.model);
            input.model = makeCanonic.transform();
            EXPECT_TRUE(input.model->isCanonic());
            if (TestType::preprocessingType == PreprocessingType::SelfloopReduction || TestType::preprocessingType == PreprocessingType::All ) {
                storm::transformer::GlobalPOMDPSelfLoopEliminator<ValueType> selfLoopEliminator(*input.model);
                if (selfLoopEliminator.preservesFormula(*input.formula)) {
                    input.model = selfLoopEliminator.transform();
                } else {
                    EXPECT_TRUE(input.formula->isOperatorFormula());
                    EXPECT_TRUE(input.formula->asOperatorFormula().hasOptimalityType());
                    bool maximizing = storm::solver::maximize(input.formula->asOperatorFormula().getOptimalityType());
                    // Valid reasons for unpreserved formulas:
                    EXPECT_TRUE(maximizing || input.formula->isProbabilityOperatorFormula());
                    EXPECT_TRUE(!maximizing || input.formula->isRewardOperatorFormula());
                }
            }
            if (TestType::preprocessingType == PreprocessingType::QualitativeReduction || TestType::preprocessingType == PreprocessingType::All ) {
                EXPECT_TRUE(input.formula->isOperatorFormula());
                EXPECT_TRUE(input.formula->asOperatorFormula().hasOptimalityType());
                if (input.formula->isProbabilityOperatorFormula() && storm::solver::maximize(input.formula->asOperatorFormula().getOptimalityType())) {
                    storm::analysis::QualitativeAnalysisOnGraphs<ValueType> qualitativeAnalysis(*input.model);
                    storm::storage::BitVector prob0States = qualitativeAnalysis.analyseProb0(input.formula->asProbabilityOperatorFormula());
                    storm::storage::BitVector  prob1States = qualitativeAnalysis.analyseProb1(input.formula->asProbabilityOperatorFormula());
                    storm::pomdp::transformer::KnownProbabilityTransformer<ValueType> kpt;
                    input.model = kpt.transform(*input.model, prob0States, prob1States);
                }
            }
            EXPECT_TRUE(input.model->isCanonic());
            return input;
        }
        ValueType precision() const { return TestType::precision(); }
        ValueType modelcheckingPrecision() const { if (TestType::isExactModelChecking) return storm::utility::zero<ValueType>(); else return storm::utility::convertNumber<ValueType>(1e-6); }
        
    private:
        storm::Environment _environment;
    };
  
    typedef ::testing::Types<
            DefaultDoubleVIEnvironment,
            SelfloopReductionDefaultDoubleVIEnvironment,
            QualitativeReductionDefaultDoubleVIEnvironment,
            PreprocessedDefaultDoubleVIEnvironment,
            FineDoubleVIEnvironment,
            RefineDoubleVIEnvironment,
            PreprocessedRefineDoubleVIEnvironment,
            DefaultDoubleOVIEnvironment,
            DefaultRationalPIEnvironment,
            PreprocessedDefaultRationalPIEnvironment
    > TestingTypes;
    
    TYPED_TEST_SUITE(BeliefExplorationTest, TestingTypes,);
    
    TYPED_TEST(BeliefExplorationTest, simple_Pmax) {
        typedef typename TestFixture::ValueType ValueType;

        auto data = this->buildPrism(STORM_TEST_RESOURCES_DIR "/pomdp/simple.prism", "Pmax=? [F \"goal\" ]", "slippery=0");
        storm::pomdp::modelchecker::BeliefExplorationPomdpModelChecker<storm::models::sparse::Pomdp<ValueType>> checker(data.model, this->options());
        auto result = checker.check(*data.formula);
        
        ValueType expected = this->parseNumber("7/10");
        EXPECT_LE(result.lowerBound, expected + this->modelcheckingPrecision());
        EXPECT_GE(result.upperBound, expected - this->modelcheckingPrecision());
        EXPECT_LE(result.diff(), this->precision()) << "Result [" << result.lowerBound << ", " << result.upperBound << "] is not precise enough. If (only) this fails, the result bounds are still correct, but they might be unexpectedly imprecise.\n";
    }
    
    TYPED_TEST(BeliefExplorationTest, simple_Pmin) {
        typedef typename TestFixture::ValueType ValueType;

        auto data = this->buildPrism(STORM_TEST_RESOURCES_DIR "/pomdp/simple.prism", "Pmin=? [F \"goal\" ]", "slippery=0");
        storm::pomdp::modelchecker::BeliefExplorationPomdpModelChecker<storm::models::sparse::Pomdp<ValueType>> checker(data.model, this->options());
        auto result = checker.check(*data.formula);
        
        ValueType expected = this->parseNumber("3/10");
        EXPECT_LE(result.lowerBound, expected + this->modelcheckingPrecision());
        EXPECT_GE(result.upperBound, expected - this->modelcheckingPrecision());
        EXPECT_LE(result.diff(), this->precision()) << "Result [" << result.lowerBound << ", " << result.upperBound << "] is not precise enough. If (only) this fails, the result bounds are still correct, but they might be unexpectedly imprecise.\n";
    }
    
    TYPED_TEST(BeliefExplorationTest, simple_slippery_Pmax) {
        typedef typename TestFixture::ValueType ValueType;

        auto data = this->buildPrism(STORM_TEST_RESOURCES_DIR "/pomdp/simple.prism", "Pmax=? [F \"goal\" ]", "slippery=0.4");
        storm::pomdp::modelchecker::BeliefExplorationPomdpModelChecker<storm::models::sparse::Pomdp<ValueType>> checker(data.model, this->options());
        auto result = checker.check(*data.formula);
        
        ValueType expected = this->parseNumber("7/10");
        EXPECT_LE(result.lowerBound, expected + this->modelcheckingPrecision());
        EXPECT_GE(result.upperBound, expected - this->modelcheckingPrecision());
        EXPECT_LE(result.diff(), this->precision()) << "Result [" << result.lowerBound << ", " << result.upperBound << "] is not precise enough. If (only) this fails, the result bounds are still correct, but they might be unexpectedly imprecise.\n";
    }
    
    TYPED_TEST(BeliefExplorationTest, simple_slippery_Pmin) {
        typedef typename TestFixture::ValueType ValueType;

        auto data = this->buildPrism(STORM_TEST_RESOURCES_DIR "/pomdp/simple.prism", "Pmin=? [F \"goal\" ]", "slippery=0.4");
        storm::pomdp::modelchecker::BeliefExplorationPomdpModelChecker<storm::models::sparse::Pomdp<ValueType>> checker(data.model, this->options());
        auto result = checker.check(*data.formula);
        
        ValueType expected = this->parseNumber("3/10");
        EXPECT_LE(result.lowerBound, expected + this->modelcheckingPrecision());
        EXPECT_GE(result.upperBound, expected - this->modelcheckingPrecision());
        EXPECT_LE(result.diff(), this->precision()) << "Result [" << result.lowerBound << ", " << result.upperBound << "] is not precise enough. If (only) this fails, the result bounds are still correct, but they might be unexpectedly imprecise.\n";
    }
    
    TYPED_TEST(BeliefExplorationTest, simple_Rmax) {
        typedef typename TestFixture::ValueType ValueType;

        auto data = this->buildPrism(STORM_TEST_RESOURCES_DIR "/pomdp/simple.prism", "Rmax=? [F s>4 ]", "slippery=0");
        storm::pomdp::modelchecker::BeliefExplorationPomdpModelChecker<storm::models::sparse::Pomdp<ValueType>> checker(data.model, this->options());
        auto result = checker.check(*data.formula);
        
        ValueType expected = this->parseNumber("29/50");
        EXPECT_LE(result.lowerBound, expected + this->modelcheckingPrecision());
        EXPECT_GE(result.upperBound, expected - this->modelcheckingPrecision());
        EXPECT_LE(result.diff(), this->precision()) << "Result [" << result.lowerBound << ", " << result.upperBound << "] is not precise enough. If (only) this fails, the result bounds are still correct, but they might be unexpectedly imprecise.\n";
    }
    
    TYPED_TEST(BeliefExplorationTest, simple_Rmin) {
        typedef typename TestFixture::ValueType ValueType;

        auto data = this->buildPrism(STORM_TEST_RESOURCES_DIR "/pomdp/simple.prism", "Rmin=? [F s>4 ]", "slippery=0");
        storm::pomdp::modelchecker::BeliefExplorationPomdpModelChecker<storm::models::sparse::Pomdp<ValueType>> checker(data.model, this->options());
        auto result = checker.check(*data.formula);
        
        ValueType expected = this->parseNumber("19/50");
        EXPECT_LE(result.lowerBound, expected + this->modelcheckingPrecision());
        EXPECT_GE(result.upperBound, expected - this->modelcheckingPrecision());
        EXPECT_LE(result.diff(), this->precision()) << "Result [" << result.lowerBound << ", " << result.upperBound << "] is not precise enough. If (only) this fails, the result bounds are still correct, but they might be unexpectedly imprecise.\n";
    }
    
    TYPED_TEST(BeliefExplorationTest, simple_slippery_Rmax) {
        typedef typename TestFixture::ValueType ValueType;

        auto data = this->buildPrism(STORM_TEST_RESOURCES_DIR "/pomdp/simple.prism", "Rmax=? [F s>4 ]", "slippery=0.4");
        storm::pomdp::modelchecker::BeliefExplorationPomdpModelChecker<storm::models::sparse::Pomdp<ValueType>> checker(data.model, this->options());
        auto result = checker.check(*data.formula);
        
        ValueType expected = this->parseNumber("29/30");
        EXPECT_LE(result.lowerBound, expected + this->modelcheckingPrecision());
        EXPECT_GE(result.upperBound, expected - this->modelcheckingPrecision());
        EXPECT_LE(result.diff(), this->precision()) << "Result [" << result.lowerBound << ", " << result.upperBound << "] is not precise enough. If (only) this fails, the result bounds are still correct, but they might be unexpectedly imprecise.\n";
    }
    
    TYPED_TEST(BeliefExplorationTest, simple_slippery_Rmin) {
        typedef typename TestFixture::ValueType ValueType;

        auto data = this->buildPrism(STORM_TEST_RESOURCES_DIR "/pomdp/simple.prism", "Rmin=? [F s>4 ]", "slippery=0.4");
        storm::pomdp::modelchecker::BeliefExplorationPomdpModelChecker<storm::models::sparse::Pomdp<ValueType>> checker(data.model, this->options());
        auto result = checker.check(*data.formula);
        
        ValueType expected = this->parseNumber("19/30");
        EXPECT_LE(result.lowerBound, expected + this->modelcheckingPrecision());
        EXPECT_GE(result.upperBound, expected - this->modelcheckingPrecision());
        EXPECT_LE(result.diff(), this->precision()) << "Result [" << result.lowerBound << ", " << result.upperBound << "] is not precise enough. If (only) this fails, the result bounds are still correct, but they might be unexpectedly imprecise.\n";
    }
    
    TYPED_TEST(BeliefExplorationTest, maze2_Rmin) {
        typedef typename TestFixture::ValueType ValueType;

        auto data = this->buildPrism(STORM_TEST_RESOURCES_DIR "/pomdp/maze2.prism", "R[exp]min=? [F \"goal\"]", "sl=0");
        storm::pomdp::modelchecker::BeliefExplorationPomdpModelChecker<storm::models::sparse::Pomdp<ValueType>> checker(data.model, this->options());
        auto result = checker.check(*data.formula);
        
        ValueType expected = this->parseNumber("74/91");
        EXPECT_LE(result.lowerBound, expected + this->modelcheckingPrecision());
        EXPECT_GE(result.upperBound, expected - this->modelcheckingPrecision());
        // Use relative difference of bounds for this one
        EXPECT_LE(result.diff(), this->precision()) << "Result [" << result.lowerBound << ", " << result.upperBound << "] is not precise enough. If (only) this fails, the result bounds are still correct, but they might be unexpectedly imprecise.\n";
    }
    
    TYPED_TEST(BeliefExplorationTest, maze2_Rmax) {
        typedef typename TestFixture::ValueType ValueType;

        auto data = this->buildPrism(STORM_TEST_RESOURCES_DIR "/pomdp/maze2.prism", "R[exp]max=? [F \"goal\"]", "sl=0");
        storm::pomdp::modelchecker::BeliefExplorationPomdpModelChecker<storm::models::sparse::Pomdp<ValueType>> checker(data.model, this->options());
        auto result = checker.check(*data.formula);
        
        EXPECT_TRUE(storm::utility::isInfinity(result.lowerBound));
        EXPECT_TRUE(storm::utility::isInfinity(result.upperBound));
    }
    
    TYPED_TEST(BeliefExplorationTest, maze2_slippery_Rmin) {
        typedef typename TestFixture::ValueType ValueType;

        auto data = this->buildPrism(STORM_TEST_RESOURCES_DIR "/pomdp/maze2.prism", "R[exp]min=? [F \"goal\"]", "sl=0.075");
        storm::pomdp::modelchecker::BeliefExplorationPomdpModelChecker<storm::models::sparse::Pomdp<ValueType>> checker(data.model, this->options());
        auto result = checker.check(*data.formula);
        
        ValueType expected = this->parseNumber("80/91");
        EXPECT_LE(result.lowerBound, expected + this->modelcheckingPrecision());
        EXPECT_GE(result.upperBound, expected - this->modelcheckingPrecision());
        // Use relative difference of bounds for this one
        EXPECT_LE(result.diff(), this->precision()) << "Result [" << result.lowerBound << ", " << result.upperBound << "] is not precise enough. If (only) this fails, the result bounds are still correct, but they might be unexpectedly imprecise.\n";
    }
    
    TYPED_TEST(BeliefExplorationTest, maze2_slippery_Rmax) {
        typedef typename TestFixture::ValueType ValueType;

        auto data = this->buildPrism(STORM_TEST_RESOURCES_DIR "/pomdp/maze2.prism", "R[exp]max=? [F \"goal\"]", "sl=0.075");
        storm::pomdp::modelchecker::BeliefExplorationPomdpModelChecker<storm::models::sparse::Pomdp<ValueType>> checker(data.model, this->options());
        auto result = checker.check(*data.formula);
        
        EXPECT_TRUE(storm::utility::isInfinity(result.lowerBound));
        EXPECT_TRUE(storm::utility::isInfinity(result.upperBound));
    }
    
    TYPED_TEST(BeliefExplorationTest, refuel_Pmax) {
        typedef typename TestFixture::ValueType ValueType;

        auto data = this->buildPrism(STORM_TEST_RESOURCES_DIR "/pomdp/refuel.prism", "Pmax=?[\"notbad\" U \"goal\"]", "N=4");
        storm::pomdp::modelchecker::BeliefExplorationPomdpModelChecker<storm::models::sparse::Pomdp<ValueType>> checker(data.model, this->options());
        auto result = checker.check(*data.formula);
        
        ValueType expected = this->parseNumber("38/155");
        EXPECT_LE(result.lowerBound, expected + this->modelcheckingPrecision());
        EXPECT_GE(result.upperBound, expected - this->modelcheckingPrecision());
        // Use relative difference of bounds for this one
        EXPECT_LE(result.diff(), this->precision()) << "Result [" << result.lowerBound << ", " << result.upperBound << "] is not precise enough. If (only) this fails, the result bounds are still correct, but they might be unexpectedly imprecise.\n";
    }
    
    TYPED_TEST(BeliefExplorationTest, refuel_Pmin) {
        typedef typename TestFixture::ValueType ValueType;

        auto data = this->buildPrism(STORM_TEST_RESOURCES_DIR "/pomdp/refuel.prism", "Pmin=?[\"notbad\" U \"goal\"]", "N=4");
        storm::pomdp::modelchecker::BeliefExplorationPomdpModelChecker<storm::models::sparse::Pomdp<ValueType>> checker(data.model, this->options());
        auto result = checker.check(*data.formula);
        
        ValueType expected = this->parseNumber("0");
        EXPECT_LE(result.lowerBound, expected + this->modelcheckingPrecision());
        EXPECT_GE(result.upperBound, expected - this->modelcheckingPrecision());
        // Use relative difference of bounds for this one
        EXPECT_LE(result.diff(), this->precision()) << "Result [" << result.lowerBound << ", " << result.upperBound << "] is not precise enough. If (only) this fails, the result bounds are still correct, but they might be unexpectedly imprecise.\n";
    }
    
    
    
}
