#pragma once

#include "DFTChildren.h"

namespace storm {
    namespace storage {

        /*!
         * Abstract base class for gates.
         */
        template<typename ValueType>
        class DFTGate : public DFTChildren<ValueType> {

            using DFTElementPointer = std::shared_ptr<DFTElement<ValueType>>;
            using DFTElementVector = std::vector<DFTElementPointer>;

        public:
            /*!
             * Constructor.
             * @param id Id.
             * @param name Name.
             * @param children Children.
             */
            DFTGate(size_t id, std::string const& name, DFTElementVector const& children) : DFTChildren<ValueType>(id, name, children) {
                // Intentionally left empty.
            }

            /*!
             * Destructor
             */
            virtual ~DFTGate() {
                // Intentionally left empty.
            }

            virtual bool isGate() const override {
                return true;
            }

            /*!
             * Return whether the gate is a dynamic gate.
             * @return True iff the gate is dynamic.
             */
            bool isDynamicGate() const {
                return !isStaticGateType(this->type());
            }

            virtual void extendSpareModule(std::set<size_t>& elementsInSpareModule) const override {
                if (!this->isSpareGate()) {
                    DFTElement<ValueType>::extendSpareModule(elementsInSpareModule);
                    for (auto const& child : this->children()) {
                        if (elementsInSpareModule.count(child->id()) == 0) {
                            elementsInSpareModule.insert(child->id());
                            child->extendSpareModule(elementsInSpareModule);
                        }
                    }
                }
            }

            virtual bool checkDontCareAnymore(storm::storage::DFTState<ValueType>& state, DFTStateSpaceGenerationQueues<ValueType>& queues) const override {
                if (DFTElement<ValueType>::checkDontCareAnymore(state, queues)) {
                    childrenDontCare(state, queues);
                    return true;
                }
                return false;
            }


        protected:

            void fail(DFTState<ValueType>& state, DFTStateSpaceGenerationQueues<ValueType>& queues) const override {
                for (std::shared_ptr<DFTGate> parent : this->mParents) {
                    if (state.isOperational(parent->id())) {
                        queues.propagateFailure(parent);
                    }
                }
                for (std::shared_ptr<DFTRestriction<ValueType>> restr : this->mRestrictions) {
                    queues.checkRestrictionLater(restr);
                }
                state.setFailed(this->mId);
                this->childrenDontCare(state, queues);
            }

            void failsafe(DFTState<ValueType>& state, DFTStateSpaceGenerationQueues<ValueType>& queues) const override {
                for (std::shared_ptr<DFTGate> parent : this->mParents) {
                    if (state.isOperational(parent->id())) {
                        queues.propagateFailsafe(parent);
                    }
                }
                state.setFailsafe(this->mId);
                this->childrenDontCare(state, queues);
            }

            /*!
             * Propagate Don't Care to children.
             * @param state Current DFT state.
             * @param queues Propagation queues.
             */
            void childrenDontCare(DFTState<ValueType>& state, DFTStateSpaceGenerationQueues<ValueType>& queues) const {
                queues.propagateDontCare(this->children());
            }
        };

    }
}
