"""
Classical Force-field Inspired Descriptors (CFID).

The CFID provide a complete set of chemo-structural descriptors able to
differentiate between structural prototypes. Specifically, the combination
of pairwise radial, nearest neighbor, bond-angle, dihedral-angle and
core-charge distributions, together with hundreds of chemistry-related
descriptors, plays an important role in predicting formation energies,
bandgaps, static refractive indices, magnetic properties, and modulus of
elasticity for 3D materials as well as exfoliation energies of two-dimensional
(2D) layered materials.

The 1557 CFID are described in detail below:

============================================== ===========     ============
Descriptor name                                Array_index     Total number
Chemical (mean_chem)                                0-437          438
Simulation cell-size (cell)                       438-441            4
Radial charge (mean_chg)                          442-819          378
Radial distribution function (rdf)                820-919          100
Angular distribution upto 1st nn cutoff (adfa)   920-1098          179
Angular distribution upto 2nd nn cutoff (adfb)  1099-1277          179
Dihedral distribution upto 1st nn cutoff (ddf)  1278-1456          179
Nearest neighbor distribution (nn)              1457-1556          100
Total                                                   -          1557
============================================== ===========     ============

Details of element based chemical descriptors are below:

===================         ==================================================
Descriptor_name                            Details
jv_enp                      Energy per atom of an element from JARVIS-DFT
KV                          Bulk modulus of an element from JARVIS-DFT
GV                          Shear modulus of an element from JARVIS-DFT
C-m (m=0 to 35)             Elastic constants of element from JARVIS-DFT
                            (total 36)
op_eg                       OptB88vdW bandgap during SCF for an element
mop_eg                      OptB88vdW bandgap during linear optics for element
voro_coord                  Voronoi coordination number of an elemental-crystal
                            structure
ndunfilled                  Number of unfilled d-orbitals
ndvalence                   Number of valence d-orbitals
nsunfilled                  Number of unfilled s-orbitals
nsunfilled                  Number of valence s-orbitals
npunfilled                  Number of unfilled p-orbitals
npvalence                   Number of valence p-orbitals
nfunfilled                  Number of unfilled f-orbitals
nfvalence                   Number of valence f-orbitals
first_ion                   First ionization energy of an element
oq_bg                       OQMD bandgap for an element
elec_aff                    Electron affinity
vol_pa                      Volume per atom of an element
hfus                        Heat of fusion of an element
oq_enp                      OQMD energy per atom
Polariz                     Polarizability
Z                           Atomic number
X                           Electronegativity
row                         Row number in the periodic table
column                      Column number in the periodic table
max_oxid_s                  Maximum oxidation state
min_oxid_s                  Minimum oxidation state
block                       s,p,d,f block assigned to 0,1,2,3 blocks
is_alkali                   Is it alkali element 0/1
is_alkaline                 Is it alkaline element 0/1
is_metalloid                Is it metalloid element 0/1
is_noble_gas                Is it noble gas element 0/1
is_transition_metal         Is it transition element 0/1
is_metalloid                Is it metalloid element 0/1
is_halogen                  Is it halogen element 0/1
is_lanthanoid               Is it lanthanoid element 0/1
is_actinoid                 Is it actinoid element 0/1
atom_mass                   Atomic mass
atom_rad                    Atomic radii
therm_cond                  Thermal conductivity
mol_vol                     Molar volume
bp                          Boiling point
mp                          Melting point
avg_ion_rad                 Average ionic radii
polzbl                      Polarizability
e1                          Static dielectric function in x-direction from
                            JARVIS-DFT using OptB88vdW functional
e2                          Static dielectric function in y-direction from
                            JARVIS-DFT using OptB88vdW functional
e3                          Static dielectric function in z-direction from
                            JARVIS-DFT using OptB88vdW functional
me1                         Static dielectric function in x-direction from
                            JARVIS-DFT using TB-mBJ potential
me2                         Static dielectric function in y-direction from
                            JARVIS-DFT using TB-mBJ potential
me3                         Static dielectric function in z-direction from
                            JARVIS-DFT using TB-mBJ potential
===================         ==================================================

Addition (‘add’), multiplications (‘mult’), subtraction (‘subs’) and quotient
(‘divi’) of hfus, polzbl, first_ion_en, mol_vol, bp,mp, mol_vol, mol_vol,
therm_cond and voro_coord were performed to give additional chemical
descriptors.

Details of simulation cell-size based descriptors are below:

===============   =========================================
Descriptor_name          Details
cell_0            Volume per atom of the cell
cell_1            Logarithm of volume per atom of the cell
cell_2            Packing fraction
cell_3            Density
===============   =========================================

Find details in:
https://journals.aps.org/prmaterials/abstract/10.1103/PhysRevMaterials.2.083801
"""
from jarvis.analysis.structure.neighbors import NeighborsAnalysis
from jarvis.core.specie import get_descrp_arr_name
import numpy as np
from math import log
from jarvis.core.composition import Composition
from jarvis.core.specie import Specie


def get_chem_only_descriptors(
    formula="Al2O3",
    extra=[],
    mean_only=False,
    max_only=False,
    min_only=False,
    source="cfid",
):
    """Get jarvis_cfid or magpie descriptors for each formula."""
    s = Composition.from_string(formula)
    el_dict = s.to_dict()
    arr = []
    sum = 0
    for k, v in el_dict.items():
        sum += v
        des = v * Specie(k, source=source).get_descrp_arr
        arr.append(des)
    names = list(Specie("H", source=source)._data["H"].keys())
    if mean_only:
        chem = np.mean(np.array(arr), axis=0) / sum
        names = ["Mean_" + source + "_" + str(n) for n in names]
    elif max_only:
        chem = np.max(np.array(arr), axis=0) / sum
        names = ["Max_" + source + "_" + str(n) for n in names]
    elif min_only:
        chem = np.min(np.array(arr), axis=0) / sum
        names = ["Min_" + source + "_" + str(n) for n in names]
    else:
        chem = (
            list(np.mean(np.array(arr), axis=0) / sum)
            + list(np.max(np.array(arr), axis=0) / sum)
            + list(np.min(np.array(arr), axis=0) / sum)
        )
        chem = np.array(chem)
        names = (
            ["Mean_" + source + "_" + str(n) for n in names]
            + ["Max_" + source + "_" + str(n) for n in names]
            + ["Min_" + source + "_" + str(n) for n in names]
        )
    chem = list(chem)
    for ii, i in enumerate(extra):
        chem.append(i)
        nm = "extra_" + str(ii)
        names.append(nm)
    chem = np.array(chem)
    return chem, names


class CFID(object):
    """Convert Atoms class into 1557 descriptors."""

    def __init__(self, atoms):
        """Initialize with jarvis.core.atoms.Atoms class."""
        self._atoms = atoms

    def get_comp_descp(
        self,
        jcell=True,
        jmean_chem=True,
        jmean_chg=True,
        jrdf=False,
        jrdf_adf=True,
        print_names=False,
    ):
        """
        Get chemo-structural CFID decriptors.

        Args:
            jcell: whether to use cell-size descriptors

            jmean_chem: whether to use average chemical descriptors

            jmean_chg: whether to use average charge distribution descriptors

            jmean_rdf: whether to use radial distribution descriptors

            jrdf_adf: whether to use radial and angle distribution descriptors

            print_names: whether to print names of descriptors

        Returns:
          cat: catenated final descriptors
        """
        cat = []
        s = self._atoms
        cell = []
        mean_chem = []
        rdf = []
        nn = []
        mean_chg = []
        adfa = []
        adfb = []
        ddf = []
        if jmean_chem:
            el_dict = s.composition._content
            # print (el_dict,type(el_dict))
            arr = []
            for k, v in el_dict.items():
                des = Specie(k).get_descrp_arr
                arr.append(des)
            mean_chem = np.mean(np.array(arr), axis=0)
            # print ('mean_chem',len(mean_chem))

        if jcell:
            v_pa = round(float(s.volume) / float(s.num_atoms), 5)
            logv_pa = round(log(v_pa), 5)
            pf = s.packing_fraction
            density = round(s.density, 5)
            cell = np.array([v_pa, logv_pa, pf, density])
            # print ('jcell',len(cell))

        if jrdf:
            Nbrs = NeighborsAnalysis(s)
            _, distrdf, nn = Nbrs.get_rdf()
            rdf = np.array(distrdf)
            print("rdf", len(rdf))

        if jrdf_adf:
            try:
                adfa = np.zeros(179)
                adfb = np.zeros(179)
                ddf = np.zeros(179)
                rdf = np.zeros(100)
                nn = np.zeros(100)
                distributions = NeighborsAnalysis(s).get_all_distributions
                rdf = distributions["rdf"]
                nn = distributions["nn"]
                adfa = distributions["adfa"]
                adfb = distributions["adfb"]
                ddf = distributions["ddf"]

            except Exception:
                pass
            adfa = np.array(adfa)
            adfb = np.array(adfb)
            rdf = np.array(rdf)
            ddf = np.array(ddf)
            nn = np.array(nn)
            # print ('adfa',len(adfa))
            # print ('ddf',len(ddf))
            # print ('adfb',len(adfb))
            # print ('rdf',len(rdf))
            # print ('nn',len(nn))

        if jmean_chg:
            chgarr = []
            el_dict = s.composition._content
            for k, v in el_dict.items():
                chg = Specie(k).get_chgdescrp_arr
                chgarr.append(chg)
            mean_chg = np.mean(chgarr, axis=0)
            # print ('mean_chg',len(mean_chg))

        if print_names:
            nmes = []
            chem_nm = get_descrp_arr_name()
            for d, nm in zip(
                [mean_chem, cell, mean_chg, rdf, adfa, adfb, ddf, nn],
                [
                    "mean_chem",
                    "cell",
                    "mean_chg",
                    "rdf",
                    "adfa",
                    "adfb",
                    "ddf",
                    "nn",
                ],
            ):
                if len(d) != 0:
                    for ff, dd in enumerate(d):
                        cat.append(dd)
                        if nm == "mean_chem":
                            tag = chem_nm[ff]
                        else:
                            tag = str(nm) + str("_") + str(ff)
                        nmes.append(str(tag))
            cat = np.array(cat).astype(float)
            # print (nmes,len(nmes))
            return nmes
        else:
            for d, nm in zip(
                [mean_chem, cell, mean_chg, rdf, adfa, adfb, ddf, nn],
                [
                    "mean_chem",
                    "cell",
                    "mean_chg",
                    "rdf",
                    "adfa",
                    "adfb",
                    "ddf",
                    "nn",
                ],
            ):
                if len(d) != 0:
                    # if d != []:
                    for ff, dd in enumerate(d):
                        cat.append(dd)
            cat = np.array(cat).astype(float)
        return cat


def feat_names():
    """Names of the 1557 descriptors."""
    names = [
        "bp_mult_atom_rad",
        "hfus_add_bp",
        "elec_aff_mult_voro_coord",
        "mol_vol_subs_atom_mass",
        "is_halogen",
        "atom_mass_subs_first_ion_en",
        "row",
        "mol_vol_mult_atom_mass",
        "voro_coord_divi_therm_cond",
        "voro_coord_subs_mp",
        "polzbl_mult_atom_rad",
        "elec_aff_mult_X",
        "GV",
        "nfunfill",
        "voro_coord_subs_therm_cond",
        "mp_divi_therm_cond",
        "elec_aff_add_X",
        "mol_vol_add_bp",
        "C-9",
        "C-8",
        "C-7",
        "C-6",
        "C-5",
        "C-4",
        "C-3",
        "C-2",
        "C-1",
        "C-0",
        "mp_subs_elec_aff",
        "hfus_add_elec_aff",
        "elec_aff_subs_first_ion_en",
        "therm_cond_subs_mol_vol",
        "X_subs_mol_vol",
        "first_ion_en_subs_hfus",
        "first_ion_en_add_voro_coord",
        "first_ion_en_divi_atom_mass",
        "atom_rad_subs_polzbl",
        "me1",
        "me3",
        "me2",
        "elec_aff_mult_atom_mass",
        "elec_aff_add_atom_mass",
        "therm_cond_subs_elec_aff",
        "atom_rad_divi_atom_mass",
        "atom_rad_divi_voro_coord",
        "max_oxid_s",
        "polzbl_mult_first_ion_en",
        "mp_divi_voro_coord",
        "mol_vol_divi_hfus",
        "mp_divi_atom_mass",
        "therm_cond_subs_mp",
        "npvalence",
        "mol_vol_subs_atom_rad",
        "atom_rad_divi_hfus",
        "mp",
        "hfus_add_atom_rad",
        "atom_mass_add_voro_coord",
        "atom_mass_divi_atom_rad",
        "hfus_divi_atom_rad",
        "polzbl_mult_X",
        "first_ion_en",
        "bp_add_X",
        "therm_cond_subs_atom_rad",
        "first_ion_en_subs_voro_coord",
        "first_ion_en_subs_polzbl",
        "hfus_subs_X",
        "first_ion_en_mult_mol_vol",
        "first_ion_en_mult_mp",
        "bp_mult_atom_mass",
        "bp_subs_mp",
        "polzbl_add_atom_rad",
        "therm_cond_subs_hfus",
        "mol_vol_add_mp",
        "first_ion_en_subs_elec_aff",
        "atom_mass_add_therm_cond",
        "atom_mass_divi_therm_cond",
        "first_ion_en_divi_mol_vol",
        "polzbl_divi_mp",
        "elec_aff_subs_therm_cond",
        "ndunfill",
        "op_eg",
        "atom_mass_subs_mp",
        "mol_vol_subs_voro_coord",
        "jv_enp",
        "polzbl_mult_atom_mass",
        "first_ion_en_add_mp",
        "first_ion_en_add_atom_mass",
        "mol_vol_mult_bp",
        "mol_vol_add_atom_mass",
        "nfvalence",
        "hfus_subs_voro_coord",
        "voro_coord_divi_mol_vol",
        "hfus_mult_first_ion_en",
        "mp_mult_atom_mass",
        "polzbl",
        "therm_cond_subs_first_ion_en",
        "mol_vol_subs_elec_aff",
        "therm_cond_divi_atom_rad",
        "first_ion_en_subs_atom_mass",
        "atom_rad_mult_X",
        "mp_subs_mol_vol",
        "hfus_divi_bp",
        "mol_vol_subs_therm_cond",
        "mol_vol_subs_first_ion_en",
        "oq_bg",
        "therm_cond_mult_X",
        "mol_vol_subs_X",
        "elec_aff_subs_X",
        "atom_rad",
        "bp_subs_elec_aff",
        "polzbl_divi_first_ion_en",
        "mp_mult_atom_rad",
        "polzbl_mult_mp",
        "first_ion_en_divi_bp",
        "atom_mass",
        "bp_subs_therm_cond",
        "bp_add_atom_mass",
        "polzbl_divi_hfus",
        "hfus_mult_atom_rad",
        "first_ion_en_divi_voro_coord",
        "mp_add_therm_cond",
        "hfus_mult_atom_mass",
        "hfus_divi_mol_vol",
        "hfus_divi_polzbl",
        "first_ion_en_divi_hfus",
        "elec_aff_subs_atom_mass",
        "hfus_mult_voro_coord",
        "polzbl_subs_X",
        "elec_aff_mult_therm_cond",
        "atom_rad_subs_atom_mass",
        "is_alkali",
        "voro_coord_subs_atom_mass",
        "mol_vol_mult_X",
        "hfus_subs_mp",
        "mol_vol_divi_atom_rad",
        "atom_mass_subs_polzbl",
        "atom_rad_divi_mol_vol",
        "elec_aff_add_bp",
        "C-13",
        "C-12",
        "C-11",
        "C-10",
        "C-17",
        "C-16",
        "C-15",
        "C-14",
        "hfus_divi_voro_coord",
        "polzbl_subs_atom_mass",
        "C-19",
        "C-18",
        "X_subs_atom_rad",
        "X_subs_elec_aff",
        "voro_coord",
        "mp_add_atom_mass",
        "elec_aff_add_mp",
        "is_noble_gas",
        "polzbl_add_bp",
        "hfus_add_polzbl",
        "X_subs_first_ion_en",
        "mp_subs_first_ion_en",
        "atom_rad_mult_voro_coord",
        "mol_vol_subs_mp",
        "atom_mass_divi_mp",
        "voro_coord_subs_polzbl",
        "bp_divi_first_ion_en",
        "polzbl_add_mp",
        "polzbl_add_therm_cond",
        "hfus_subs_atom_rad",
        "first_ion_en_divi_therm_cond",
        "X_subs_atom_mass",
        "mol_vol_subs_bp",
        "polzbl_mult_therm_cond",
        "bp_subs_voro_coord",
        "e1",
        "e3",
        "e2",
        "atom_rad_subs_X",
        "X_add_voro_coord",
        "mp_mult_voro_coord",
        "mp_subs_atom_mass",
        "mp_add_voro_coord",
        "therm_cond_divi_hfus",
        "bp_divi_atom_mass",
        "first_ion_en_mult_X",
        "hfus_mult_mol_vol",
        "voro_coord_divi_bp",
        "mol_vol_divi_voro_coord",
        "atom_mass_divi_voro_coord",
        "is_lanthanoid",
        "mol_vol_divi_polzbl",
        "atom_mass_subs_therm_cond",
        "atom_rad_divi_therm_cond",
        "atom_mass_divi_polzbl",
        "bp_add_atom_rad",
        "voro_coord_subs_hfus",
        "polzbl_subs_bp",
        "polzbl_divi_bp",
        "elec_aff_mult_mp",
        "ndvalence",
        "atom_mass_divi_mol_vol",
        "mp_divi_mol_vol",
        "therm_cond_subs_voro_coord",
        "atom_mass_subs_atom_rad",
        "polzbl_divi_voro_coord",
        "polzbl_subs_mp",
        "polzbl_subs_therm_cond",
        "KV",
        "min_oxid_s",
        "elec_aff_mult_atom_rad",
        "therm_cond_subs_polzbl",
        "voro_coord_subs_bp",
        "bp_add_voro_coord",
        "nsunfill",
        "mp_mult_X",
        "atom_mass_mult_atom_rad",
        "first_ion_en_mult_voro_coord",
        "elec_aff_subs_voro_coord",
        "bp_divi_mp",
        "atom_rad_subs_mol_vol",
        "bp_subs_hfus",
        "atom_rad_subs_first_ion_en",
        "hfus_divi_mp",
        "first_ion_en_subs_mp",
        "mp_add_atom_rad",
        "hfus_add_atom_mass",
        "bp_subs_mol_vol",
        "hfus_add_first_ion_en",
        "bp_subs_atom_rad",
        "bp_divi_atom_rad",
        "C-26",
        "atom_mass_subs_hfus",
        "mp_subs_therm_cond",
        "polzbl_divi_atom_rad",
        "polzbl_add_mol_vol",
        "first_ion_en_add_bp",
        "hfus_subs_therm_cond",
        "mp_divi_hfus",
        "X_subs_bp",
        "bp_mult_mp",
        "atom_mass_mult_therm_cond",
        "mol_vol_divi_therm_cond",
        "voro_coord_subs_X",
        "first_ion_en_subs_mol_vol",
        "polzbl_add_elec_aff",
        "therm_cond_subs_atom_mass",
        "hfus_subs_bp",
        "atom_mass_subs_mol_vol",
        "X",
        "is_actinoid",
        "hfus_subs_first_ion_en",
        "first_ion_en_divi_mp",
        "atom_rad_add_therm_cond",
        "atom_rad_divi_polzbl",
        "hfus_divi_atom_mass",
        "mp_mult_therm_cond",
        "first_ion_en_add_X",
        "polzbl_add_voro_coord",
        "mol_vol_subs_polzbl",
        "therm_cond_divi_first_ion_en",
        "bp_mult_voro_coord",
        "atom_mass_add_atom_rad",
        "hfus_mult_mp",
        "voro_coord_divi_polzbl",
        "elec_aff_add_therm_cond",
        "C-28",
        "C-29",
        "hfus_add_voro_coord",
        "C-27",
        "C-24",
        "C-25",
        "C-22",
        "C-23",
        "C-20",
        "C-21",
        "therm_cond_divi_polzbl",
        "mol_vol_mult_mp",
        "mp_subs_X",
        "bp_subs_atom_mass",
        "first_ion_en_add_therm_cond",
        "voro_coord_divi_atom_mass",
        "atom_mass_subs_voro_coord",
        "hfus_add_therm_cond",
        "atom_rad_divi_mp",
        "mp_subs_voro_coord",
        "hfus_mult_polzbl",
        "bp_divi_hfus",
        "therm_cond_divi_voro_coord",
        "first_ion_en_subs_atom_rad",
        "atom_rad_subs_bp",
        "atom_rad_add_X",
        "mol_vol_add_voro_coord",
        "avg_ion_rad",
        "bp_divi_mol_vol",
        "atom_rad_subs_mp",
        "atom_mass_mult_voro_coord",
        "therm_cond_add_X",
        "atom_rad_subs_therm_cond",
        "polzbl_add_first_ion_en",
        "polzbl_mult_bp",
        "first_ion_en_mult_elec_aff",
        "elec_aff_subs_mol_vol",
        "nsvalence",
        "mol_vol_subs_hfus",
        "atom_mass_divi_bp",
        "bp_subs_polzbl",
        "bp_add_mp",
        "polzbl_divi_mol_vol",
        "voro_coord_subs_elec_aff",
        "elec_aff_add_atom_rad",
        "mol_vol_add_atom_rad",
        "hfus_mult_X",
        "mp_subs_hfus",
        "voro_coord_divi_first_ion_en",
        "atom_rad_divi_first_ion_en",
        "elec_aff_mult_mol_vol",
        "is_metalloid",
        "hfus_mult_therm_cond",
        "hfus_divi_therm_cond",
        "therm_cond_divi_mol_vol",
        "elec_aff",
        "mp_divi_atom_rad",
        "elec_aff_subs_polzbl",
        "polzbl_divi_atom_mass",
        "atom_rad_subs_elec_aff",
        "bp_divi_therm_cond",
        "atom_mass_mult_X",
        "mp_divi_polzbl",
        "elec_aff_subs_atom_rad",
        "voro_coord_divi_atom_rad",
        "hfus_subs_mol_vol",
        "mol_vol_add_therm_cond",
        "bp_mult_X",
        "therm_cond_divi_mp",
        "mp_subs_polzbl",
        "atom_rad_divi_bp",
        "atom_rad_subs_hfus",
        "polzbl_subs_elec_aff",
        "therm_cond_divi_atom_mass",
        "bp_divi_voro_coord",
        "bp_add_therm_cond",
        "hfus_divi_first_ion_en",
        "X_mult_voro_coord",
        "bp_subs_X",
        "polzbl_subs_voro_coord",
        "coulmn",
        "first_ion_en_divi_polzbl",
        "first_ion_en_mult_atom_mass",
        "mol_vol_mult_therm_cond",
        "bp_mult_therm_cond",
        "mol_vol",
        "mol_vol_mult_voro_coord",
        "atom_mass_subs_elec_aff",
        "atom_mass_divi_first_ion_en",
        "hfus_subs_elec_aff",
        "first_ion_en_mult_bp",
        "polzbl_mult_mol_vol",
        "polzbl_add_atom_mass",
        "elec_aff_mult_bp",
        "voro_coord_subs_atom_rad",
        "mp_add_X",
        "elec_aff_subs_bp",
        "polzbl_divi_therm_cond",
        "polzbl_mult_voro_coord",
        "bp",
        "therm_cond_mult_voro_coord",
        "elec_aff_subs_mp",
        "C-31",
        "C-30",
        "C-33",
        "C-32",
        "C-35",
        "C-34",
        "first_ion_en_subs_X",
        "is_transition_metal",
        "mp_subs_bp",
        "hfus_add_mol_vol",
        "block",
        "atom_rad_subs_voro_coord",
        "X_subs_therm_cond",
        "polzbl_subs_first_ion_en",
        "hfus_subs_atom_mass",
        "mol_vol_divi_bp",
        "first_ion_en_subs_therm_cond",
        "elec_aff_add_voro_coord",
        "voro_coord_divi_hfus",
        "therm_cond",
        "bp_subs_first_ion_en",
        "X_subs_voro_coord",
        "therm_cond_divi_bp",
        "mp_divi_bp",
        "first_ion_en_divi_atom_rad",
        "hfus_subs_polzbl",
        "mol_vol_divi_mp",
        "atom_rad_add_voro_coord",
        "mol_vol_mult_atom_rad",
        "first_ion_en_mult_atom_rad",
        "hfus_mult_bp",
        "voro_coord_subs_first_ion_en",
        "Z",
        "polzbl_subs_hfus",
        "polzbl_add_X",
        "is_alkaline",
        "npunfill",
        "first_ion_en_subs_bp",
        "voro_coord_divi_mp",
        "atom_mass_divi_hfus",
        "oq_enp",
        "elec_aff_subs_hfus",
        "atom_mass_subs_bp",
        "first_ion_en_add_elec_aff",
        "first_ion_en_add_mol_vol",
        "hfus_mult_elec_aff",
        "polzbl_mult_elec_aff",
        "hfus_add_X",
        "hfus_add_mp",
        "therm_cond_subs_bp",
        "bp_divi_polzbl",
        "X_subs_hfus",
        "atom_rad_mult_therm_cond",
        "therm_cond_add_voro_coord",
        "mop_eg",
        "X_subs_polzbl",
        "mol_vol_divi_first_ion_en",
        "mp_divi_first_ion_en",
        "voro_coord_subs_mol_vol",
        "elec_aff_add_mol_vol",
        "mol_vol_add_X",
        "atom_mass_add_X",
        "polzbl_subs_atom_rad",
        "mol_vol_divi_atom_mass",
        "polzbl_subs_mol_vol",
        "mp_subs_atom_rad",
        "X_subs_mp",
        "atom_mass_subs_X",
        "first_ion_en_add_atom_rad",
        "therm_cond_subs_X",
        "hfus",
        "first_ion_en_mult_therm_cond",
        "cell_0",
        "cell_1",
        "cell_2",
        "cell_3",
        "mean_chg_0",
        "mean_chg_1",
        "mean_chg_2",
        "mean_chg_3",
        "mean_chg_4",
        "mean_chg_5",
        "mean_chg_6",
        "mean_chg_7",
        "mean_chg_8",
        "mean_chg_9",
        "mean_chg_10",
        "mean_chg_11",
        "mean_chg_12",
        "mean_chg_13",
        "mean_chg_14",
        "mean_chg_15",
        "mean_chg_16",
        "mean_chg_17",
        "mean_chg_18",
        "mean_chg_19",
        "mean_chg_20",
        "mean_chg_21",
        "mean_chg_22",
        "mean_chg_23",
        "mean_chg_24",
        "mean_chg_25",
        "mean_chg_26",
        "mean_chg_27",
        "mean_chg_28",
        "mean_chg_29",
        "mean_chg_30",
        "mean_chg_31",
        "mean_chg_32",
        "mean_chg_33",
        "mean_chg_34",
        "mean_chg_35",
        "mean_chg_36",
        "mean_chg_37",
        "mean_chg_38",
        "mean_chg_39",
        "mean_chg_40",
        "mean_chg_41",
        "mean_chg_42",
        "mean_chg_43",
        "mean_chg_44",
        "mean_chg_45",
        "mean_chg_46",
        "mean_chg_47",
        "mean_chg_48",
        "mean_chg_49",
        "mean_chg_50",
        "mean_chg_51",
        "mean_chg_52",
        "mean_chg_53",
        "mean_chg_54",
        "mean_chg_55",
        "mean_chg_56",
        "mean_chg_57",
        "mean_chg_58",
        "mean_chg_59",
        "mean_chg_60",
        "mean_chg_61",
        "mean_chg_62",
        "mean_chg_63",
        "mean_chg_64",
        "mean_chg_65",
        "mean_chg_66",
        "mean_chg_67",
        "mean_chg_68",
        "mean_chg_69",
        "mean_chg_70",
        "mean_chg_71",
        "mean_chg_72",
        "mean_chg_73",
        "mean_chg_74",
        "mean_chg_75",
        "mean_chg_76",
        "mean_chg_77",
        "mean_chg_78",
        "mean_chg_79",
        "mean_chg_80",
        "mean_chg_81",
        "mean_chg_82",
        "mean_chg_83",
        "mean_chg_84",
        "mean_chg_85",
        "mean_chg_86",
        "mean_chg_87",
        "mean_chg_88",
        "mean_chg_89",
        "mean_chg_90",
        "mean_chg_91",
        "mean_chg_92",
        "mean_chg_93",
        "mean_chg_94",
        "mean_chg_95",
        "mean_chg_96",
        "mean_chg_97",
        "mean_chg_98",
        "mean_chg_99",
        "mean_chg_100",
        "mean_chg_101",
        "mean_chg_102",
        "mean_chg_103",
        "mean_chg_104",
        "mean_chg_105",
        "mean_chg_106",
        "mean_chg_107",
        "mean_chg_108",
        "mean_chg_109",
        "mean_chg_110",
        "mean_chg_111",
        "mean_chg_112",
        "mean_chg_113",
        "mean_chg_114",
        "mean_chg_115",
        "mean_chg_116",
        "mean_chg_117",
        "mean_chg_118",
        "mean_chg_119",
        "mean_chg_120",
        "mean_chg_121",
        "mean_chg_122",
        "mean_chg_123",
        "mean_chg_124",
        "mean_chg_125",
        "mean_chg_126",
        "mean_chg_127",
        "mean_chg_128",
        "mean_chg_129",
        "mean_chg_130",
        "mean_chg_131",
        "mean_chg_132",
        "mean_chg_133",
        "mean_chg_134",
        "mean_chg_135",
        "mean_chg_136",
        "mean_chg_137",
        "mean_chg_138",
        "mean_chg_139",
        "mean_chg_140",
        "mean_chg_141",
        "mean_chg_142",
        "mean_chg_143",
        "mean_chg_144",
        "mean_chg_145",
        "mean_chg_146",
        "mean_chg_147",
        "mean_chg_148",
        "mean_chg_149",
        "mean_chg_150",
        "mean_chg_151",
        "mean_chg_152",
        "mean_chg_153",
        "mean_chg_154",
        "mean_chg_155",
        "mean_chg_156",
        "mean_chg_157",
        "mean_chg_158",
        "mean_chg_159",
        "mean_chg_160",
        "mean_chg_161",
        "mean_chg_162",
        "mean_chg_163",
        "mean_chg_164",
        "mean_chg_165",
        "mean_chg_166",
        "mean_chg_167",
        "mean_chg_168",
        "mean_chg_169",
        "mean_chg_170",
        "mean_chg_171",
        "mean_chg_172",
        "mean_chg_173",
        "mean_chg_174",
        "mean_chg_175",
        "mean_chg_176",
        "mean_chg_177",
        "mean_chg_178",
        "mean_chg_179",
        "mean_chg_180",
        "mean_chg_181",
        "mean_chg_182",
        "mean_chg_183",
        "mean_chg_184",
        "mean_chg_185",
        "mean_chg_186",
        "mean_chg_187",
        "mean_chg_188",
        "mean_chg_189",
        "mean_chg_190",
        "mean_chg_191",
        "mean_chg_192",
        "mean_chg_193",
        "mean_chg_194",
        "mean_chg_195",
        "mean_chg_196",
        "mean_chg_197",
        "mean_chg_198",
        "mean_chg_199",
        "mean_chg_200",
        "mean_chg_201",
        "mean_chg_202",
        "mean_chg_203",
        "mean_chg_204",
        "mean_chg_205",
        "mean_chg_206",
        "mean_chg_207",
        "mean_chg_208",
        "mean_chg_209",
        "mean_chg_210",
        "mean_chg_211",
        "mean_chg_212",
        "mean_chg_213",
        "mean_chg_214",
        "mean_chg_215",
        "mean_chg_216",
        "mean_chg_217",
        "mean_chg_218",
        "mean_chg_219",
        "mean_chg_220",
        "mean_chg_221",
        "mean_chg_222",
        "mean_chg_223",
        "mean_chg_224",
        "mean_chg_225",
        "mean_chg_226",
        "mean_chg_227",
        "mean_chg_228",
        "mean_chg_229",
        "mean_chg_230",
        "mean_chg_231",
        "mean_chg_232",
        "mean_chg_233",
        "mean_chg_234",
        "mean_chg_235",
        "mean_chg_236",
        "mean_chg_237",
        "mean_chg_238",
        "mean_chg_239",
        "mean_chg_240",
        "mean_chg_241",
        "mean_chg_242",
        "mean_chg_243",
        "mean_chg_244",
        "mean_chg_245",
        "mean_chg_246",
        "mean_chg_247",
        "mean_chg_248",
        "mean_chg_249",
        "mean_chg_250",
        "mean_chg_251",
        "mean_chg_252",
        "mean_chg_253",
        "mean_chg_254",
        "mean_chg_255",
        "mean_chg_256",
        "mean_chg_257",
        "mean_chg_258",
        "mean_chg_259",
        "mean_chg_260",
        "mean_chg_261",
        "mean_chg_262",
        "mean_chg_263",
        "mean_chg_264",
        "mean_chg_265",
        "mean_chg_266",
        "mean_chg_267",
        "mean_chg_268",
        "mean_chg_269",
        "mean_chg_270",
        "mean_chg_271",
        "mean_chg_272",
        "mean_chg_273",
        "mean_chg_274",
        "mean_chg_275",
        "mean_chg_276",
        "mean_chg_277",
        "mean_chg_278",
        "mean_chg_279",
        "mean_chg_280",
        "mean_chg_281",
        "mean_chg_282",
        "mean_chg_283",
        "mean_chg_284",
        "mean_chg_285",
        "mean_chg_286",
        "mean_chg_287",
        "mean_chg_288",
        "mean_chg_289",
        "mean_chg_290",
        "mean_chg_291",
        "mean_chg_292",
        "mean_chg_293",
        "mean_chg_294",
        "mean_chg_295",
        "mean_chg_296",
        "mean_chg_297",
        "mean_chg_298",
        "mean_chg_299",
        "mean_chg_300",
        "mean_chg_301",
        "mean_chg_302",
        "mean_chg_303",
        "mean_chg_304",
        "mean_chg_305",
        "mean_chg_306",
        "mean_chg_307",
        "mean_chg_308",
        "mean_chg_309",
        "mean_chg_310",
        "mean_chg_311",
        "mean_chg_312",
        "mean_chg_313",
        "mean_chg_314",
        "mean_chg_315",
        "mean_chg_316",
        "mean_chg_317",
        "mean_chg_318",
        "mean_chg_319",
        "mean_chg_320",
        "mean_chg_321",
        "mean_chg_322",
        "mean_chg_323",
        "mean_chg_324",
        "mean_chg_325",
        "mean_chg_326",
        "mean_chg_327",
        "mean_chg_328",
        "mean_chg_329",
        "mean_chg_330",
        "mean_chg_331",
        "mean_chg_332",
        "mean_chg_333",
        "mean_chg_334",
        "mean_chg_335",
        "mean_chg_336",
        "mean_chg_337",
        "mean_chg_338",
        "mean_chg_339",
        "mean_chg_340",
        "mean_chg_341",
        "mean_chg_342",
        "mean_chg_343",
        "mean_chg_344",
        "mean_chg_345",
        "mean_chg_346",
        "mean_chg_347",
        "mean_chg_348",
        "mean_chg_349",
        "mean_chg_350",
        "mean_chg_351",
        "mean_chg_352",
        "mean_chg_353",
        "mean_chg_354",
        "mean_chg_355",
        "mean_chg_356",
        "mean_chg_357",
        "mean_chg_358",
        "mean_chg_359",
        "mean_chg_360",
        "mean_chg_361",
        "mean_chg_362",
        "mean_chg_363",
        "mean_chg_364",
        "mean_chg_365",
        "mean_chg_366",
        "mean_chg_367",
        "mean_chg_368",
        "mean_chg_369",
        "mean_chg_370",
        "mean_chg_371",
        "mean_chg_372",
        "mean_chg_373",
        "mean_chg_374",
        "mean_chg_375",
        "mean_chg_376",
        "mean_chg_377",
        "rdf_0",
        "rdf_1",
        "rdf_2",
        "rdf_3",
        "rdf_4",
        "rdf_5",
        "rdf_6",
        "rdf_7",
        "rdf_8",
        "rdf_9",
        "rdf_10",
        "rdf_11",
        "rdf_12",
        "rdf_13",
        "rdf_14",
        "rdf_15",
        "rdf_16",
        "rdf_17",
        "rdf_18",
        "rdf_19",
        "rdf_20",
        "rdf_21",
        "rdf_22",
        "rdf_23",
        "rdf_24",
        "rdf_25",
        "rdf_26",
        "rdf_27",
        "rdf_28",
        "rdf_29",
        "rdf_30",
        "rdf_31",
        "rdf_32",
        "rdf_33",
        "rdf_34",
        "rdf_35",
        "rdf_36",
        "rdf_37",
        "rdf_38",
        "rdf_39",
        "rdf_40",
        "rdf_41",
        "rdf_42",
        "rdf_43",
        "rdf_44",
        "rdf_45",
        "rdf_46",
        "rdf_47",
        "rdf_48",
        "rdf_49",
        "rdf_50",
        "rdf_51",
        "rdf_52",
        "rdf_53",
        "rdf_54",
        "rdf_55",
        "rdf_56",
        "rdf_57",
        "rdf_58",
        "rdf_59",
        "rdf_60",
        "rdf_61",
        "rdf_62",
        "rdf_63",
        "rdf_64",
        "rdf_65",
        "rdf_66",
        "rdf_67",
        "rdf_68",
        "rdf_69",
        "rdf_70",
        "rdf_71",
        "rdf_72",
        "rdf_73",
        "rdf_74",
        "rdf_75",
        "rdf_76",
        "rdf_77",
        "rdf_78",
        "rdf_79",
        "rdf_80",
        "rdf_81",
        "rdf_82",
        "rdf_83",
        "rdf_84",
        "rdf_85",
        "rdf_86",
        "rdf_87",
        "rdf_88",
        "rdf_89",
        "rdf_90",
        "rdf_91",
        "rdf_92",
        "rdf_93",
        "rdf_94",
        "rdf_95",
        "rdf_96",
        "rdf_97",
        "rdf_98",
        "rdf_99",
        "adfa_0",
        "adfa_1",
        "adfa_2",
        "adfa_3",
        "adfa_4",
        "adfa_5",
        "adfa_6",
        "adfa_7",
        "adfa_8",
        "adfa_9",
        "adfa_10",
        "adfa_11",
        "adfa_12",
        "adfa_13",
        "adfa_14",
        "adfa_15",
        "adfa_16",
        "adfa_17",
        "adfa_18",
        "adfa_19",
        "adfa_20",
        "adfa_21",
        "adfa_22",
        "adfa_23",
        "adfa_24",
        "adfa_25",
        "adfa_26",
        "adfa_27",
        "adfa_28",
        "adfa_29",
        "adfa_30",
        "adfa_31",
        "adfa_32",
        "adfa_33",
        "adfa_34",
        "adfa_35",
        "adfa_36",
        "adfa_37",
        "adfa_38",
        "adfa_39",
        "adfa_40",
        "adfa_41",
        "adfa_42",
        "adfa_43",
        "adfa_44",
        "adfa_45",
        "adfa_46",
        "adfa_47",
        "adfa_48",
        "adfa_49",
        "adfa_50",
        "adfa_51",
        "adfa_52",
        "adfa_53",
        "adfa_54",
        "adfa_55",
        "adfa_56",
        "adfa_57",
        "adfa_58",
        "adfa_59",
        "adfa_60",
        "adfa_61",
        "adfa_62",
        "adfa_63",
        "adfa_64",
        "adfa_65",
        "adfa_66",
        "adfa_67",
        "adfa_68",
        "adfa_69",
        "adfa_70",
        "adfa_71",
        "adfa_72",
        "adfa_73",
        "adfa_74",
        "adfa_75",
        "adfa_76",
        "adfa_77",
        "adfa_78",
        "adfa_79",
        "adfa_80",
        "adfa_81",
        "adfa_82",
        "adfa_83",
        "adfa_84",
        "adfa_85",
        "adfa_86",
        "adfa_87",
        "adfa_88",
        "adfa_89",
        "adfa_90",
        "adfa_91",
        "adfa_92",
        "adfa_93",
        "adfa_94",
        "adfa_95",
        "adfa_96",
        "adfa_97",
        "adfa_98",
        "adfa_99",
        "adfa_100",
        "adfa_101",
        "adfa_102",
        "adfa_103",
        "adfa_104",
        "adfa_105",
        "adfa_106",
        "adfa_107",
        "adfa_108",
        "adfa_109",
        "adfa_110",
        "adfa_111",
        "adfa_112",
        "adfa_113",
        "adfa_114",
        "adfa_115",
        "adfa_116",
        "adfa_117",
        "adfa_118",
        "adfa_119",
        "adfa_120",
        "adfa_121",
        "adfa_122",
        "adfa_123",
        "adfa_124",
        "adfa_125",
        "adfa_126",
        "adfa_127",
        "adfa_128",
        "adfa_129",
        "adfa_130",
        "adfa_131",
        "adfa_132",
        "adfa_133",
        "adfa_134",
        "adfa_135",
        "adfa_136",
        "adfa_137",
        "adfa_138",
        "adfa_139",
        "adfa_140",
        "adfa_141",
        "adfa_142",
        "adfa_143",
        "adfa_144",
        "adfa_145",
        "adfa_146",
        "adfa_147",
        "adfa_148",
        "adfa_149",
        "adfa_150",
        "adfa_151",
        "adfa_152",
        "adfa_153",
        "adfa_154",
        "adfa_155",
        "adfa_156",
        "adfa_157",
        "adfa_158",
        "adfa_159",
        "adfa_160",
        "adfa_161",
        "adfa_162",
        "adfa_163",
        "adfa_164",
        "adfa_165",
        "adfa_166",
        "adfa_167",
        "adfa_168",
        "adfa_169",
        "adfa_170",
        "adfa_171",
        "adfa_172",
        "adfa_173",
        "adfa_174",
        "adfa_175",
        "adfa_176",
        "adfa_177",
        "adfa_178",
        "adfb_0",
        "adfb_1",
        "adfb_2",
        "adfb_3",
        "adfb_4",
        "adfb_5",
        "adfb_6",
        "adfb_7",
        "adfb_8",
        "adfb_9",
        "adfb_10",
        "adfb_11",
        "adfb_12",
        "adfb_13",
        "adfb_14",
        "adfb_15",
        "adfb_16",
        "adfb_17",
        "adfb_18",
        "adfb_19",
        "adfb_20",
        "adfb_21",
        "adfb_22",
        "adfb_23",
        "adfb_24",
        "adfb_25",
        "adfb_26",
        "adfb_27",
        "adfb_28",
        "adfb_29",
        "adfb_30",
        "adfb_31",
        "adfb_32",
        "adfb_33",
        "adfb_34",
        "adfb_35",
        "adfb_36",
        "adfb_37",
        "adfb_38",
        "adfb_39",
        "adfb_40",
        "adfb_41",
        "adfb_42",
        "adfb_43",
        "adfb_44",
        "adfb_45",
        "adfb_46",
        "adfb_47",
        "adfb_48",
        "adfb_49",
        "adfb_50",
        "adfb_51",
        "adfb_52",
        "adfb_53",
        "adfb_54",
        "adfb_55",
        "adfb_56",
        "adfb_57",
        "adfb_58",
        "adfb_59",
        "adfb_60",
        "adfb_61",
        "adfb_62",
        "adfb_63",
        "adfb_64",
        "adfb_65",
        "adfb_66",
        "adfb_67",
        "adfb_68",
        "adfb_69",
        "adfb_70",
        "adfb_71",
        "adfb_72",
        "adfb_73",
        "adfb_74",
        "adfb_75",
        "adfb_76",
        "adfb_77",
        "adfb_78",
        "adfb_79",
        "adfb_80",
        "adfb_81",
        "adfb_82",
        "adfb_83",
        "adfb_84",
        "adfb_85",
        "adfb_86",
        "adfb_87",
        "adfb_88",
        "adfb_89",
        "adfb_90",
        "adfb_91",
        "adfb_92",
        "adfb_93",
        "adfb_94",
        "adfb_95",
        "adfb_96",
        "adfb_97",
        "adfb_98",
        "adfb_99",
        "adfb_100",
        "adfb_101",
        "adfb_102",
        "adfb_103",
        "adfb_104",
        "adfb_105",
        "adfb_106",
        "adfb_107",
        "adfb_108",
        "adfb_109",
        "adfb_110",
        "adfb_111",
        "adfb_112",
        "adfb_113",
        "adfb_114",
        "adfb_115",
        "adfb_116",
        "adfb_117",
        "adfb_118",
        "adfb_119",
        "adfb_120",
        "adfb_121",
        "adfb_122",
        "adfb_123",
        "adfb_124",
        "adfb_125",
        "adfb_126",
        "adfb_127",
        "adfb_128",
        "adfb_129",
        "adfb_130",
        "adfb_131",
        "adfb_132",
        "adfb_133",
        "adfb_134",
        "adfb_135",
        "adfb_136",
        "adfb_137",
        "adfb_138",
        "adfb_139",
        "adfb_140",
        "adfb_141",
        "adfb_142",
        "adfb_143",
        "adfb_144",
        "adfb_145",
        "adfb_146",
        "adfb_147",
        "adfb_148",
        "adfb_149",
        "adfb_150",
        "adfb_151",
        "adfb_152",
        "adfb_153",
        "adfb_154",
        "adfb_155",
        "adfb_156",
        "adfb_157",
        "adfb_158",
        "adfb_159",
        "adfb_160",
        "adfb_161",
        "adfb_162",
        "adfb_163",
        "adfb_164",
        "adfb_165",
        "adfb_166",
        "adfb_167",
        "adfb_168",
        "adfb_169",
        "adfb_170",
        "adfb_171",
        "adfb_172",
        "adfb_173",
        "adfb_174",
        "adfb_175",
        "adfb_176",
        "adfb_177",
        "adfb_178",
        "ddf_0",
        "ddf_1",
        "ddf_2",
        "ddf_3",
        "ddf_4",
        "ddf_5",
        "ddf_6",
        "ddf_7",
        "ddf_8",
        "ddf_9",
        "ddf_10",
        "ddf_11",
        "ddf_12",
        "ddf_13",
        "ddf_14",
        "ddf_15",
        "ddf_16",
        "ddf_17",
        "ddf_18",
        "ddf_19",
        "ddf_20",
        "ddf_21",
        "ddf_22",
        "ddf_23",
        "ddf_24",
        "ddf_25",
        "ddf_26",
        "ddf_27",
        "ddf_28",
        "ddf_29",
        "ddf_30",
        "ddf_31",
        "ddf_32",
        "ddf_33",
        "ddf_34",
        "ddf_35",
        "ddf_36",
        "ddf_37",
        "ddf_38",
        "ddf_39",
        "ddf_40",
        "ddf_41",
        "ddf_42",
        "ddf_43",
        "ddf_44",
        "ddf_45",
        "ddf_46",
        "ddf_47",
        "ddf_48",
        "ddf_49",
        "ddf_50",
        "ddf_51",
        "ddf_52",
        "ddf_53",
        "ddf_54",
        "ddf_55",
        "ddf_56",
        "ddf_57",
        "ddf_58",
        "ddf_59",
        "ddf_60",
        "ddf_61",
        "ddf_62",
        "ddf_63",
        "ddf_64",
        "ddf_65",
        "ddf_66",
        "ddf_67",
        "ddf_68",
        "ddf_69",
        "ddf_70",
        "ddf_71",
        "ddf_72",
        "ddf_73",
        "ddf_74",
        "ddf_75",
        "ddf_76",
        "ddf_77",
        "ddf_78",
        "ddf_79",
        "ddf_80",
        "ddf_81",
        "ddf_82",
        "ddf_83",
        "ddf_84",
        "ddf_85",
        "ddf_86",
        "ddf_87",
        "ddf_88",
        "ddf_89",
        "ddf_90",
        "ddf_91",
        "ddf_92",
        "ddf_93",
        "ddf_94",
        "ddf_95",
        "ddf_96",
        "ddf_97",
        "ddf_98",
        "ddf_99",
        "ddf_100",
        "ddf_101",
        "ddf_102",
        "ddf_103",
        "ddf_104",
        "ddf_105",
        "ddf_106",
        "ddf_107",
        "ddf_108",
        "ddf_109",
        "ddf_110",
        "ddf_111",
        "ddf_112",
        "ddf_113",
        "ddf_114",
        "ddf_115",
        "ddf_116",
        "ddf_117",
        "ddf_118",
        "ddf_119",
        "ddf_120",
        "ddf_121",
        "ddf_122",
        "ddf_123",
        "ddf_124",
        "ddf_125",
        "ddf_126",
        "ddf_127",
        "ddf_128",
        "ddf_129",
        "ddf_130",
        "ddf_131",
        "ddf_132",
        "ddf_133",
        "ddf_134",
        "ddf_135",
        "ddf_136",
        "ddf_137",
        "ddf_138",
        "ddf_139",
        "ddf_140",
        "ddf_141",
        "ddf_142",
        "ddf_143",
        "ddf_144",
        "ddf_145",
        "ddf_146",
        "ddf_147",
        "ddf_148",
        "ddf_149",
        "ddf_150",
        "ddf_151",
        "ddf_152",
        "ddf_153",
        "ddf_154",
        "ddf_155",
        "ddf_156",
        "ddf_157",
        "ddf_158",
        "ddf_159",
        "ddf_160",
        "ddf_161",
        "ddf_162",
        "ddf_163",
        "ddf_164",
        "ddf_165",
        "ddf_166",
        "ddf_167",
        "ddf_168",
        "ddf_169",
        "ddf_170",
        "ddf_171",
        "ddf_172",
        "ddf_173",
        "ddf_174",
        "ddf_175",
        "ddf_176",
        "ddf_177",
        "ddf_178",
        "nn_0",
        "nn_1",
        "nn_2",
        "nn_3",
        "nn_4",
        "nn_5",
        "nn_6",
        "nn_7",
        "nn_8",
        "nn_9",
        "nn_10",
        "nn_11",
        "nn_12",
        "nn_13",
        "nn_14",
        "nn_15",
        "nn_16",
        "nn_17",
        "nn_18",
        "nn_19",
        "nn_20",
        "nn_21",
        "nn_22",
        "nn_23",
        "nn_24",
        "nn_25",
        "nn_26",
        "nn_27",
        "nn_28",
        "nn_29",
        "nn_30",
        "nn_31",
        "nn_32",
        "nn_33",
        "nn_34",
        "nn_35",
        "nn_36",
        "nn_37",
        "nn_38",
        "nn_39",
        "nn_40",
        "nn_41",
        "nn_42",
        "nn_43",
        "nn_44",
        "nn_45",
        "nn_46",
        "nn_47",
        "nn_48",
        "nn_49",
        "nn_50",
        "nn_51",
        "nn_52",
        "nn_53",
        "nn_54",
        "nn_55",
        "nn_56",
        "nn_57",
        "nn_58",
        "nn_59",
        "nn_60",
        "nn_61",
        "nn_62",
        "nn_63",
        "nn_64",
        "nn_65",
        "nn_66",
        "nn_67",
        "nn_68",
        "nn_69",
        "nn_70",
        "nn_71",
        "nn_72",
        "nn_73",
        "nn_74",
        "nn_75",
        "nn_76",
        "nn_77",
        "nn_78",
        "nn_79",
        "nn_80",
        "nn_81",
        "nn_82",
        "nn_83",
        "nn_84",
        "nn_85",
        "nn_86",
        "nn_87",
        "nn_88",
        "nn_89",
        "nn_90",
        "nn_91",
        "nn_92",
        "nn_93",
        "nn_94",
        "nn_95",
        "nn_96",
        "nn_97",
        "nn_98",
        "nn_99",
    ]
    return names


"""
if __name__ == "__main__":
    box = [[2.715, 2.715, 0], [0, 2.715, 2.715], [2.715, 0, 2.715]]
    coords = [[0, 0, 0], [0.25, 0.25, 0.25]]
    elements = ["Si", "Si"]
    p = Atoms(lattice_mat=box, coords=coords, elements=elements)
    v = VacuumPadding(p, vacuum=20.0).get_effective_2d_slab()
    v = VacuumPadding(p, vacuum=20.0).get_effective_molecule()



    Time = time.time()
    cfid = CFID(p.atoms).get_comp_descp()
    tot_time = time.time() - Time
    print("tot_time", tot_time)

"""
