#include "MapExtractor.h"

#include <functional>
#include <string>

#include <linux/bpf.h>

#include "polycube/services/table_desc.h"

#define INT "int"
#define CHAR "char"
#define SHORT "short"
#define LONG "long"
#define FLOAT "float"
#define DOUBLE "double"
#define LONG_LONG "long long"
#define LONG_DOUBLE "long double"
#define SIGNED_CHAR "signed char"
#define UNSIGNED_INT "unsigned int"
#define UNSIGNED_CHAR "unsigned char"
#define UNSIGNED_LONG "unsigned long"
#define UNSIGNED_SHORT "unsigned short"
#define UNSIGNED_LONG_LONG "unsigned long long"

using std::ostringstream;
using std::runtime_error;
using namespace polycube::service;
using json = nlohmann::json;
using value_type = json::object_t::value_type;

#pragma region compiletime_string_hashing
/**
 * This region of code exports the hash method which is used at compile time to
 * transform a string into uint64_t value by hashing it; this permits to use
 * the switch statement on strings.
 */
namespace StringHash {
constexpr uint64_t _(uint64_t h, const char *s) {
  return (*s == 0)
         ? h
         : _((h * 1099511628211ull) ^ static_cast<uint64_t>(*s), s + 1);
}

constexpr uint64_t hash(const char *s) {
  return StringHash::_(14695981039346656037ull, s);
}

uint64_t hash(const std::string &s) {
  return StringHash::_(14695981039346656037ull, s.data());
}
}  // namespace StringHash
#pragma endregion

using namespace StringHash;

#define ENUM hash("enum")
#define UNION hash("union")
#define STRUCT hash("struct")
#define STRUCT_PACKED hash("struct_packed")
#define PADDING "__pad_"

typedef enum {
  Property,
  ArrayProperty,
  Value,
  Struct,
  Union,
  Enum,
} objectType;

/**
 * Recognizes the type of a node object contained in a TableDesc's leaf_desc tree
 *
 * @param[object] the object to be recognized
 *
 * @throw std::runtime_error if the object cannot be recognized
 *
 * @returns a objectType
 */
objectType identifyObject(json object) {
  if (object.is_string())
    return Value;
  if (object.is_array()) {
    if (object.size() == 2)
      return Property;
    if (object.size() == 3) {
      if (object[2].is_string()) {
        switch (hash(object[2])) {
        case STRUCT:
        case STRUCT_PACKED:
          return Struct;
        case UNION:
          return Union;
        case ENUM:
          return Enum;
        default:
          throw runtime_error("Unknown type" + object[3].get<string>());
        }
      } else if (object[2].is_array())
        return ArrayProperty;
      else if (object[2].is_number() && object[1].is_string())
        return Property;
      throw runtime_error("Unknown element" + object[0].get<string>());
    }
  }
  throw runtime_error("Unable to identifyObject " + object.dump());
}

json MapExtractor::extractFromArrayMap(const TableDesc &desc, RawTable table,
                                             std::shared_ptr<ExtractionOptions> &extractionOptions) {

  auto value_desc = json::parse(string{desc.leaf_desc});
  json j_entries;

  MapEntry batch(desc.max_entries * desc.key_size,
                 desc.max_entries * desc.leaf_size);
  unsigned int count = desc.max_entries;

  //Trying to retrieve values using batch operations
  if(table.get_batch(batch.getKey(), batch.getValue(), &count) == 0 || errno == ENOENT) {
    //Batch retrieval ok, checking if need to empty
    if(count > 0 && extractionOptions->getEmptyOnRead()) {
      MapEntry reset(0, desc.max_entries * desc.leaf_size);
      table.update_batch(batch.getKey(), reset.getValue(), &count);
    }
    for(int i=0; i<count; i++) {
      int offset = 0;
      auto value = recExtract(value_desc, static_cast<char*>(batch.getValue()) + i*desc.leaf_size, offset);
      j_entries.push_back(value);
    }
    return j_entries;
  }

  // Retrieving values using old method
  std::vector<std::shared_ptr<MapEntry>> entries;
  MapEntry reset_value(0, desc.key_size);
  void *last_key = nullptr;

  //Getting all map values
  while (true) {
    MapEntry entry(desc.key_size, desc.leaf_size);
    if (table.next(last_key, entry.getKey()) > -1) {
      last_key = entry.getKey();
      table.get(entry.getKey(), entry.getValue());
      entries.push_back(std::make_shared<MapEntry>(entry));
      if(extractionOptions->getEmptyOnRead())
        table.set(last_key, reset_value.getValue());
    }else
      break;
  }

  //Parsing retrieved values
  for(auto &entry: entries) {
    int offset = 0;
    auto j_entry = recExtract(value_desc, entry->getValue(), offset);
    j_entries.push_back(j_entry);
  }
  return j_entries;
}

json MapExtractor::extractFromQueueStackMap(const TableDesc &desc, RawQueueStackTable table,
                                        std::shared_ptr<ExtractionOptions> &extractionOptions) {

  auto value_desc = json::parse(string{desc.leaf_desc});
  std::vector<std::shared_ptr<MapEntry>> entries;

  // Retrieving map entries iteratively, DOES NOT SUPPORT BATCH OPERATIONS
  while(true) {
    MapEntry entry(0, desc.leaf_size);
    if (table.pop(entry.getValue()) != 0)
      break;
    entries.push_back(std::make_shared<MapEntry>(entry));
  }

  json j_entries;
  for(auto &entry: entries) {
    int offset = 0;
    auto j_entry = recExtract(value_desc, entry->getValue(), offset);
    j_entries.push_back(j_entry);
  }
  return j_entries;
}

json MapExtractor::extractFromHashMap(const TableDesc &desc, RawTable table,
                                      std::shared_ptr<ExtractionOptions> &extractionOptions) {
  // Getting both key_desc and leaf_desc objects which represents the structure of the map entry
  auto value_desc = json::parse(string{desc.leaf_desc});
  auto key_desc = json::parse(string{desc.key_desc});
  json j_entries;

  MapEntry batch(desc.max_entries * desc.key_size,
                 desc.max_entries * desc.leaf_size);
  unsigned int count = desc.max_entries;

  //Trying to retrieve values using batch operations
  if((extractionOptions->getEmptyOnRead() && (table.get_and_delete_batch(batch.getKey(), batch.getValue(), &count) == 0 || errno == ENOENT)) ||
      (!extractionOptions->getEmptyOnRead() && (table.get_batch(batch.getKey(), batch.getValue(), &count) == 0 || errno == ENOENT))) {
    //Batch retrieval ok
    for(int i=0; i<count; i++) {
      int offset_key = 0, offset_value = 0;
      json entry_obj;
      entry_obj["key"] = recExtract(key_desc, static_cast<char*>(batch.getKey()) + i*desc.key_size, offset_key);
      entry_obj["value"] = recExtract(value_desc, static_cast<char*>(batch.getValue()) + i*desc.leaf_size, offset_value);
      j_entries.push_back(entry_obj);
    }
    return j_entries;
  }

  // Retrieving map entries with old method
  std::vector<std::shared_ptr<MapEntry>> entries;
  void *last_key = nullptr;

  while (true) {
    MapEntry entry(desc.key_size, desc.leaf_size);
    if (table.next(last_key, entry.getKey()) > -1) {
      last_key = entry.getKey();
      table.get(entry.getKey(), entry.getValue());
      entries.push_back(std::make_shared<MapEntry>(entry));
      if(extractionOptions->getEmptyOnRead())
        table.remove(last_key);
    } else
      break;
  }

  for (auto &entry : entries) {
    int val_offset = 0, key_offset = 0;
    json entry_obj;
    entry_obj["key"] = recExtract(key_desc, entry->getKey(), key_offset);
    entry_obj["value"] = recExtract(value_desc, entry->getValue(), val_offset);
    j_entries.push_back(entry_obj);
  }
  return j_entries;
}

json MapExtractor::extractFromPerCPUMap(const TableDesc &desc, RawTable table,
                                        std::shared_ptr<ExtractionOptions> &extractionOptions) {
  // Getting the maximux cpu available
  size_t n_cpus = polycube::get_possible_cpu_count();
  size_t leaf_array_size = n_cpus * desc.leaf_size;
  // Getting key and value description to parse correctly
  auto value_desc = json::parse(string{desc.leaf_desc});
  auto key_desc = json::parse(string{desc.key_desc});
  json j_entries;

  std::vector<std::shared_ptr<MapEntry>> entries;
  MapEntry reset_value(0, leaf_array_size);
  void *last_key = nullptr;

  // Retrieving map entries with old method, DOES NOT SUPPORT BATCH OPERATION
  while (true) {
    MapEntry entry(desc.key_size, leaf_array_size);
    if (table.next(last_key, entry.getKey()) > -1) {
      last_key = entry.getKey();
      table.get(entry.getKey(), entry.getValue());
      entries.push_back(std::make_shared<MapEntry>(entry));
      if(extractionOptions->getEmptyOnRead()) {
        table.set(last_key, reset_value.getValue());
        if(desc.type != BPF_MAP_TYPE_PERCPU_ARRAY) {
          table.remove(last_key);
        }
      }
    } else
      break;
  }

  for(auto& entry : entries) {
    int key_offset = 0;
    json j_entry;

    // Parse the current key and set the id of that entry
    j_entry["key"] = recExtract(key_desc, entry->getKey(), key_offset);

    json j_values;
    // The leaf_size is the size of a single leaf value, but potentially there is an
    // array of values! So the entry->getValue() pointer must be manually shifted to parse
    // all the per_cpu values
    for(int i=0, offset=0; i<n_cpus; i++, offset=0){
      auto val = recExtract(value_desc, static_cast<char*>(entry->getValue()) + (i * desc.leaf_size), offset);
      j_values.emplace_back(val);
    }
    j_entry["value"] = j_values;
    j_entries.push_back(j_entry);
  }
  return j_entries;
}

json MapExtractor::extractFromMap(BaseCube &cube_ref, const string& map_name, int index,
                                  ProgramType type, std::shared_ptr<ExtractionOptions> extractionOptions) {
  // Getting the TableDesc object of the eBPF table
  auto &desc = cube_ref.get_table_desc(map_name, index, type);

  if(desc.type == BPF_MAP_TYPE_HASH || desc.type == BPF_MAP_TYPE_LRU_HASH)
    return extractFromHashMap(desc, cube_ref.get_raw_table(map_name, index, type), extractionOptions);
  else if(desc.type == BPF_MAP_TYPE_ARRAY)
    return extractFromArrayMap(desc, cube_ref.get_raw_table(map_name, index, type), extractionOptions);
  else if(desc.type == BPF_MAP_TYPE_QUEUE || desc.type == BPF_MAP_TYPE_STACK)
    return extractFromQueueStackMap(desc, cube_ref.get_raw_queuestack_table(map_name, index, type), extractionOptions);
  else if(desc.type == BPF_MAP_TYPE_PERCPU_HASH || desc.type == BPF_MAP_TYPE_PERCPU_ARRAY || desc.type == BPF_MAP_TYPE_LRU_PERCPU_HASH)
    return extractFromPerCPUMap(desc, cube_ref.get_raw_table(map_name, index, type), extractionOptions);
  else
    // The map type is not supported yet by Dynmon
    throw runtime_error("Unhandled Map Type " + std::to_string(desc.type) + " extraction.");
}

json MapExtractor::recExtract(json object_description, void *data, int &offset) {
  // Identifying the object
  auto objectType = identifyObject(object_description);

  switch (objectType) {
  case Property: { // the object describes a property of a C struct
    auto propertyName = object_description[0].get<string>();
    auto propertyType = object_description[1];
    return propertyName, recExtract(propertyType, data, offset);
  }
  case ArrayProperty: { // the object describes a property of a C struct which is an array
    auto propertyName = object_description[0].get<string>();
    auto propertyType = object_description[1];
    auto len = object_description[2][0].get<int>();
    // array of primitive type
    if (propertyType.is_string()) // this string is the name of the primitive type
      return valueFromPrimitiveType(propertyType, data, offset, len);
      // array of complex type
    else {
      json array;
      for (int i = 0; i < len; i++)
        array.push_back(recExtract(propertyType, data, offset));
      return value_type(propertyName, array);
    }
  }
  case Value: // the object describes a primitive type
    return valueFromPrimitiveType(object_description.get<string>(), data,
                                  offset);
  case Struct: // the object describes a C struct
    return valueFromStruct(object_description, data, offset);

  case Union: // the object describes a C union
    return valueFromUnion(object_description, data, offset);

  case Enum: // the object describes a C enum
    return valueFromEnum(object_description, data, offset);
  }
  throw runtime_error("Unhandled object type " + std::to_string(objectType) + " - recExtract()");
}

json MapExtractor::valueFromStruct(json struct_description, void *data,
                                   int &offset) {
  json array;
  // Getting the name of the C struct
  // auto structName = struct_description[0].get<string>();
  // Getting the set of properties of the C struct
  auto structProperties = struct_description[1];

  // For each property
  for (auto &property : structProperties) {
    // Getting the property's name
    auto propertyName = property[0].get<std::string>();
    // Getting the property type object
    // auto propertyType = property[1];
    // If the property is an alignment padding -> skipp it increasing the offset
    if (propertyName.rfind(PADDING, 0) == 0) {
      auto paddingSize = property[2][0].get<int>();
      offset += paddingSize;
    } else
      // Adding the property to the returning set extracting its value recursively
      array.push_back(value_type(propertyName, recExtract(property, data, offset)));
  }
  return array;
}

json MapExtractor::valueFromUnion(json union_description, void *data,
                                  int &offset) {
  json array;
  int oldOffset = offset;
  int maxOffset = 0;
  // Getting the name of the C union
  // auto unionName = union_description[0].get<string>();
  // Getting the set of properties of the C union
  auto unionProperties = union_description[1];

  // For each property
  for (auto &property : unionProperties) {
    // Getting the property's name
    auto propertyName = property[0].get<std::string>();
    // Getting the property type object
    // auto propertyType = property[1];
    // Adding the property to the returning set extracting its value recursively
    array.push_back(value_type(propertyName, recExtract(property, data, offset)));
    // Saving the offset in order to increase it to the maximum property size
    if (offset > maxOffset)
      maxOffset = offset;
    // Restoring the previous offset before iterate to the next property
    offset = oldOffset;
  }
  // Setting the offset to the maximum found so far
  offset = maxOffset;
  return array;
}

json MapExtractor::valueFromEnum(json enum_description, void *data,
                                 int &offset) {
  int index = *(int *)((char *)data + offset);
  offset += sizeof(int);
  return enum_description[1][index].get<string>();
}

json MapExtractor::valueFromPrimitiveType(const string& type_name, void *data,
                                          int &offset, int len) {
  char *address = ((char *)data + offset);
  // Casting the memory pointed by *address to the corresponding C type
  switch (hash(type_name)) {
  case hash(INT): {
    if (len == -1) {
      int value = *(int *)address;
      offset += sizeof(int);
      return value;
    } else {
      json vec = json::array();
      for (int i = 0; i < len; i++) {
        int value = *(int *)address;
        offset += sizeof(int);
        address += sizeof(int);
        vec.push_back(value);
      }
      return vec;
    }
  }
  case hash(CHAR): {
    if (len == -1) {
      char value = *address;
      offset += sizeof(char);
      return value;
    } else {
      json vec = json::array();
      for (int i = 0; i < len; i++) {
        char value = *address;
        offset += sizeof(char);
        address += sizeof(char);
        vec.push_back(value);
      }
      return vec;
    }
  }
  case hash(SHORT): {
    if (len == -1) {
      short value = *(short *)address;
      offset += sizeof(short);
      return value;
    } else {
      json vec = json::array();
      for (int i = 0; i < len; i++) {
        short value = *(short *)address;
        offset += sizeof(short);
        address += sizeof(short);
        vec.push_back(value);
      }
      return vec;
    }
  }
  case hash(LONG): {
    if (len == -1) {
      long value = *(long *)address;
      offset += sizeof(long);
      return value;
    } else {
      json vec = json::array();
      for (int i = 0; i < len; i++) {
        long value = *(long *)address;
        offset += sizeof(long);
        address += sizeof(long);
        vec.push_back(value);
      }
      return vec;
    }
  }
  case hash(FLOAT): {
    if (len == -1) {
      float value = *(float *)address;
      offset += sizeof(float);
      return value;
    } else {
      json vec = json::array();
      for (int i = 0; i < len; i++) {
        float value = *(float *)address;
        offset += sizeof(float);
        address += sizeof(float);
        vec.push_back(value);
      }
      return vec;
    }
  }
  case hash(DOUBLE): {
    if (len == -1) {
      double value = *(double *)address;
      offset += sizeof(double);
      return value;
    } else {
      json vec = json::array();
      for (int i = 0; i < len; i++) {
        double value = *(double *)address;
        offset += sizeof(double);
        address += sizeof(double);
        vec.push_back(value);
      }
      return vec;
    }
  }
  case hash(LONG_LONG): {
    if (len == -1) {
      long long value = *(long long *)address;
      offset += sizeof(long long);
      return value;
    } else {
      json vec = json::array();
      for (int i = 0; i < len; i++) {
        long long value = *(long long *)address;
        offset += sizeof(long long);
        address += sizeof(long long);
        vec.push_back(value);
      }
      return vec;
    }
  }
  case hash(LONG_DOUBLE): {
    if (len == -1) {
      long double value = *(long double *)address;
      offset += sizeof(long double);
      return value;
    } else {
      json vec = json::array();
      for (int i = 0; i < len; i++) {
        long double value = *(long double *)address;
        offset += sizeof(long double);
        address += sizeof(long double);
        vec.push_back(value);
      }
      return vec;
    }
  }
  case hash(SIGNED_CHAR): {
    if (len == -1) {
      signed char value = *address;
      offset += sizeof(signed char);
      return value;
    } else {
      signed char *value = (signed char *)address;
      offset += sizeof(signed char) * len;
      return (char *)value;
    }
  }
  case hash(UNSIGNED_INT): {
    if (len == -1) {
      unsigned int value = *(unsigned int *)address;
      offset += sizeof(unsigned int);
      return value;
    } else {
      json vec = json::array();
      for (int i = 0; i < len; i++) {
        unsigned int value = *(unsigned int *)address;
        offset += sizeof(unsigned int);
        address += sizeof(unsigned int);
        vec.push_back(value);
      }
      return vec;
    }
  }
  case hash(UNSIGNED_CHAR): {
    if (len == -1) {
      unsigned char value = *address;
      offset += sizeof(unsigned char);
      return value;
    } else {
      unsigned char *value = (unsigned char *)address;
      offset += sizeof(unsigned char) * len;
      return (char *)value;
    }
  }
  case hash(UNSIGNED_LONG): {
    if (len == -1) {
      unsigned long value = *(unsigned long *)address;
      offset += sizeof(unsigned long);
      return value;
    } else {
      json vec = json::array();
      for (int i = 0; i < len; i++) {
        unsigned long value = *(unsigned long *)address;
        offset += sizeof(unsigned long);
        address += sizeof(unsigned long);
        vec.push_back(value);
      }
      return vec;
    }
  }
  case hash(UNSIGNED_SHORT): {
    if (len == -1) {
      unsigned short value = *(unsigned short *)address;
      offset += sizeof(unsigned short);
      return value;
    } else {
      json vec = json::array();
      for (int i = 0; i < len; i++) {
        unsigned short value = *(unsigned short *)address;
        offset += sizeof(unsigned short);
        address += sizeof(unsigned short);
        vec.push_back(value);
      }
      return vec;
    }
  }
  case hash(UNSIGNED_LONG_LONG): {
    if (len == -1) {
      unsigned long long value = *(unsigned long long *)address;
      offset += sizeof(unsigned long long);
      return value;
    } else {
      json vec = json::array();
      for (int i = 0; i < len; i++) {
        unsigned long long value = *(unsigned long long *)address;
        offset += sizeof(unsigned long long);
        address += sizeof(unsigned long long);
        vec.push_back(value);
      }
      return vec;
    }
  }
  default:
    throw runtime_error("ERROR: " + type_name + " not a valid type!");
  }
}
