package types

import (
	meta_v1 "k8s.io/apimachinery/pkg/apis/meta/v1"
)

// ParsedPolicy is a general policy which contains all data needed to generate
// rules, without being tight to a specific provider (k8s or pcn).
// They are generated by splitting the original policy in as many ParsedPolicy
// as its rules. Example: a pcn policy with three rules, will generate three
// ParsedPolicy. This will make it easier to parse and generate rules.
type ParsedPolicy struct {
	// Name is the generated name of the policy, not its original name
	Name string
	// ParentPolicy contains details about the original policy.
	ParentPolicy ParentPolicy
	// Priority is the priority of this rule in the original
	// policy (read above)
	Priority int32
	// CreationTime specifies when the policy was deployed
	CreationTime meta_v1.Time
	// Direction specifies which direction this policy is targeting
	// (incoming or outgoing packets)
	Direction string
	// PreventDirectAccess tells if the pod should only be called by service
	PreventDirectAccess bool
	// Subject is the pod that must be protected
	Subject PolicySubject
	// Peer is the peer the subject must be protected from/to
	Peer PolicyPeer
	// Templates is a short list of rules pre-calculated based on the
	// rules
	Templates ParsedRules
	// Ports is the ports list specified by the policy
	Ports []ProtoPort
	// Action is the action to take on the packet (forward or drop)
	Action string
}

// ParentPolicy contains details about the original policy
type ParentPolicy struct {
	// Name is the name of the original policy
	Name string
	// Priority is the priority of the original policy.
	// For k8s policies this is always the default value (5)
	Priority int32
	// Provider is the provider of the original policy (pcn or k8s)
	Provider string
}

// PolicySubject contains data about the pod that needs to be protected
type PolicySubject struct {
	// Query contains the query that must be used to find it
	Query *ObjectQuery
	// IsService tells if the policy needs to protect a service
	// only for pcn
	IsService bool
	// Namespace is the name of the namespace where the pod/service lives
	Namespace string
}

// PolicyPeer is the peer the subject must be protected from/to
type PolicyPeer struct {
	// IPBlock tells if the subject needs to be protected from
	// a fixed ip block
	IPBlock []string
	// Peer contains the query that must be used to get them
	Peer *ObjectQuery
	// Namespace is the query that must be used to get its namespace
	Namespace *ObjectQuery
	// Key is a string that can be used to identify this peer
	Key string
}

// ProtoPort defines the protocol and the port
type ProtoPort struct {
	// Protocol is the name of the protocol in lowercase
	Protocol string
	// SPort is the sourec port
	SPort int32
	// DPort is the destination port
	DPort int32
}
