from django.test import TestCase

from apps.accounts.utils import set_group_permissions
from apps.core.testing.mixins import (
    TestListViewMixin,
    TestExportViewMixin,
    TestImportViewMixin,
    TestModelAPIViewMixin,
    TestListAPIViewMixin,
    TestRetrieveAPIViewMixin,
    TestUpdateAPIViewMixin,
    TestDeleteAPIViewMixin
)

from apps.conditions.models import Condition
from apps.options.models import OptionSet

from .models import AttributeEntity, Attribute, Range, VerboseName


class DomainTestCase(TestCase):

    lang = 'en'

    fixtures = (
        'users.json',
        'groups.json',
        'accounts.json',
        'conditions.json',
        'domain.json',
        'options.json',
    )

    users = (
        ('editor', 'editor'),
        ('reviewer', 'reviewer'),
        ('user', 'user'),
        ('api', 'api'),
        ('anonymous', None),
    )

    status_map = {
        'list': {'editor': 200, 'reviewer': 200, 'api': 200, 'user': 403, 'anonymous': 302},
        'export': {'editor': 200, 'reviewer': 200, 'api': 200, 'user': 403, 'anonymous': 302}
    }

    api_status_map = {
        'list': {'editor': 200, 'reviewer': 200, 'api': 200, 'user': 403, 'anonymous': 403},
        'retrieve': {'editor': 200, 'reviewer': 200, 'api': 200, 'user': 403, 'anonymous': 403},
        'create': {'editor': 201, 'reviewer': 403, 'api': 403, 'user': 403, 'anonymous': 403},
        'update': {'editor': 200, 'reviewer': 403, 'api': 403, 'user': 403, 'anonymous': 403},
        'delete': {'editor': 204, 'reviewer': 403, 'api': 403, 'user': 403, 'anonymous': 403}
    }

    def setUp(self):
        set_group_permissions()


class DomainTests(TestListViewMixin, DomainTestCase):

    url_names = {
        'list': 'domain'
    }


class AttributeEntityTests(TestModelAPIViewMixin, DomainTestCase):

    # get entities and order them by level to delete the entities at the bottom of the tree first
    instances = AttributeEntity.objects.filter(attribute=None).order_by('-level')

    api_url_name = 'internal-domain:entity'

    restore_instance = False

    def prepare_create_instance(self, instance):
        instance.key += '_new'
        return instance


class AttributeTests(TestModelAPIViewMixin, DomainTestCase):

    instances = Attribute.objects.all()

    api_url_name = 'internal-domain:attribute'

    def prepare_create_instance(self, instance):
        instance.key += '_new'
        return instance


class RangeTests(TestListAPIViewMixin, TestRetrieveAPIViewMixin, TestUpdateAPIViewMixin, TestDeleteAPIViewMixin, DomainTestCase):

    instances = Range.objects.all()

    api_url_name = 'internal-domain:range'


class VerboseNameTests(TestListAPIViewMixin, TestRetrieveAPIViewMixin, TestUpdateAPIViewMixin, TestDeleteAPIViewMixin, DomainTestCase):

    instances = VerboseName.objects.all()

    api_url_name = 'internal-domain:verbosename'


class ValueTypeTests(TestListAPIViewMixin, DomainTestCase):

    api_url_name = 'internal-domain:valuestype'
    api_status_map = {
        'list': {'editor': 200, 'reviewer': 200, 'reviewer': 200, 'api': 200, 'user': 200, 'anonymous': 403}
    }


class OptionSetTests(TestListAPIViewMixin, TestRetrieveAPIViewMixin, DomainTestCase):

    instances = OptionSet.objects.all()

    api_url_name = 'internal-domain:optionset'


class ConditionTests(TestListAPIViewMixin, TestRetrieveAPIViewMixin, DomainTestCase):

    instances = Condition.objects.all()

    api_url_name = 'internal-domain:condition'


class DomainExportTests(TestExportViewMixin, DomainTestCase):

    url_names = {
        'list': 'domain',
        'export': 'domain_export'
    }

    export_formats = ('xml', 'html', 'rtf', 'csv')


class DomainImportTests(TestImportViewMixin, TestCase):

    import_file = 'testing/xml/domain.xml'


class AttributeEntityAPITests(TestListAPIViewMixin, TestRetrieveAPIViewMixin, DomainTestCase):

    instances = AttributeEntity.objects.filter(attribute=None)

    api_url_name = 'api-v1-domain:entity'


class AttributeAPITests(TestListAPIViewMixin, TestRetrieveAPIViewMixin, DomainTestCase):

    instances = Attribute.objects.all()

    api_url_name = 'api-v1-domain:attribute'
