#Title      : SWEEP023 Pixel Assessment Tool
# License       : CC BY-NC-SA 4.0 - Attribution-NonCommercial-ShareAlike 4.0 International
# Objective : To select 10 % of wrongly classified pixels for re-evaluation before being used in the
#             habitat classification tool
# Created by: Naomi Gatis
# Created on: 24/05/2021

#----------- Project Set Up --------------------------------------------------------------------------------------
# Check that the required packages are installed
list.of.packages <- c( "caret", "here", "packrat", "randomForest",  "raster", "rgdal", "rgeos",
                      "splitstackshape", "svDialogs")
new.packages <- list.of.packages[!(list.of.packages %in% installed.packages()[,"Package"])]
if(length(new.packages)) install.packages(new.packages)
remove(list.of.packages, new.packages)

library(caret)        # Classification and Regression Training - RF with raster
library(here)         # A Simpler Way to Find Your Files
library(packrat)      # A Dependency Management System for Projects and their R Package Dependencies
library(randomForest) # Breiman and Cutler's Random Forests for Classification and Regression
library(raster)       # Geographic Data Analysis and Modeling
library(rgdal)        # Bindings for the Geospatial Database Abstraction Library
library(rgeos)        # Interface to Geometry Engine Open Source ('GEOS')
library(splitstackshape) # Stack and Reshape Datasets After Splitting Concatenated Values
                      # enables random stratified sampling of training/testing pixels
library(svDialogs)    #'SciViews' - Standard Dialogue Boxes for Windows, MacOs and Linuxes

#----------- Get Input Data -------------------------------------------------------------------------------------
pixelsFilenameT1 <- dlg_open("Training_Data", "Select last year's/the most recent training/testing photo-interepreted pixels file (.shp)")$res
RFmodelFilenameT1 <- dlg_open("Models", "Select last year's / the most the most recent Random Forest Model (YYYYRF_model.rds)")$res
springImgFilenameT2 <- dlg_open("Inputs", "Select this year's spring image file")$res
summerImgFilenameT2 <- dlg_open("Inputs", "Select this year's summer image file")$res
aoiFilename <- dlg_open("Inputs/DNPA_Extent", "Shapefile of area of interest")$res

#----------- Load Images and Derive Vegetation Indicies ---------------------------------------------------------
img <- stack(springImgFilenameT2, summerImgFilenameT2)

names(img) <- c("B2_Sp", "B3_Sp", "B4_Sp", "B5_Sp", "B6_Sp", "B7_Sp", "B8_Sp", "B11_Sp", "B8a_Sp",
                "B2_Sum", "B3_Sum", "B4_Sum", "B5_Sum", "B6_Sum", "B7_Sum", "B8_Sum", "B11_Sum", "B8a_Sum")

aoiExtent <- shapefile(aoiFilename)
aoiExtent <- spTransform(aoiExtent, crs(img))
aoiExtentBuff <- gBuffer(aoiExtent, width = 20, quadsegs = 10)
img <- crop(img,aoiExtentBuff)

ExG_Sp <- overlay(img$B3_Sp,img$B4_Sp,img$B2_Sp, fun = function (x,y,z){(2*x)-y-z})
DVI_Sp <- overlay(img$B8_Sp, img$B4_Sp, fun = function (x,y){x-y})
NDWI_Sp <-overlay(img$B3_Sp,img$B8_Sp, fun = function(x,y){(x-y)/(x+y)})
MGVRI_Sp <- overlay(img$B3_Sp,img$B4_Sp, fun = function(x,y){((x*x)-(y*y))/((x*x)+(y*y))})

ExG_Sum <- overlay(img$B3_Sum,img$B4_Sum,img$B2_Sum, fun = function (x,y,z){(2*x)-y-z})
DVI_Sum <- overlay(img$B8_Sum, img$B4_Sum, fun = function (x,y){x-y})
NDWI_Sum <-overlay(img$B3_Sum,img$B8_Sum, fun = function(x,y){(x-y)/(x+y)})
MGVRI_Sum <- overlay(img$B3_Sum,img$B4_Sum, fun = function(x,y){((x*x)-(y*y))/((x*x)+(y*y))})

slope <- raster("./Inputs/DTM/slope.tif")
aspect <- raster("./Inputs/DTM/aspect.tif")
uplandsExtent <- raster("./Inputs/DTM/uplandExtent.tif")
peatExtent <- raster("./Inputs/Peat_Extent/peatExtent.tif")

VI<- stack(ExG_Sp, DVI_Sp, NDWI_Sp, MGVRI_Sp, ExG_Sum, DVI_Sum, NDWI_Sum, MGVRI_Sum,
           slope[[1]], aspect[[1]], peatExtent, uplandsExtent)

names(VI) <- c("ExG_Sp","DVI_Sp","NDWI_Sp", "MGVRI_Sp", "ExG_Sum","DVI_Sum","NDWI_Sum","MGVRI_Sum",
               "Slope", "Aspect", "Peat_Extent", "Uplands_Extent")

img <- stack(img, VI)
img <- brick(img)

#-----------------------------------------------------------------------------------------------------------------
remove(ExG_Sp,DVI_Sp,NDWI_Sp, MGVRI_Sp, ExG_Sum, DVI_Sum, NDWI_Sum, MGVRI_Sum, slope, aspect,
       peatExtent,uplandsExtent, VI, springImgFilenameT2, summerImgFilenameT2)

#----------- Extract info from images for each photo-interpreted pixel--------------------------------------------
traintestPix <- shapefile(pixelsFilenameT1)
traintestPix <- spTransform(traintestPix, crs(img))

responseCol <- "Class"
dfTrainTestPix = data.frame(matrix(nrow = nrow(traintestPix), ncol = length(names(img)) + 1))
colnames(dfTrainTestPix) <- c(names(img),"Class")
for (i in 1:nrow(traintestPix)){
  imglist <- extract(img, traintestPix[i,])
  dfTrainTestPix[i,1:30]<- data.frame(matrix(unlist(imglist), nrow=1, ncol = length(names(img))))
  dfTrainTestPix[i,31] <- traintestPix$Class[i]
}

dfTrainTestPix[is.na(dfTrainTestPix)] <- 99999

#-----------------------------------------------------------------------------------------------------------------
remove(i, responseCol, pixelsFilenameT1, imglist, img)

#----------- Classify images from timeframe 2 using random forest model derived for timeframe 1------------------
modelRFT1 <- readRDS(RFmodelFilenameT1)
predictedPixels <- predict(modelRFT1, dfTrainTestPix)
predictedPixels <-as.numeric(as.character(predictedPixels))

dfTrainTestPix <- cbind(dfTrainTestPix, predictedPixels)

dfTrainTestPix <- dfTrainTestPix[ which ( dfTrainTestPix$Class != "17"),] #remove scrub
dfTrainTestPix$Class[dfTrainTestPix$Class=="30"] <- "29" #group waters

reclassFun <- function (predclass, peat, uplands){
  ifelse((predclass == 1 & peat == 1), 303,
  ifelse((predclass == 14 & peat == 1), 314,
  ifelse((predclass == 1 & uplands == 1), 3,
  ifelse((predclass == 4 & uplands == 1), 5,
  ifelse((predclass == 6 & uplands == 1), 7,
  ifelse((predclass == 14 & uplands != 1), 13, predclass))))))
  }

dfTrainTestPix$Reclass <- reclassFun(dfTrainTestPix$predictedPixels,
                                     dfTrainTestPix$Peat_Extent,
                                     dfTrainTestPix$Uplands_Extent)

#-----------------------------------------------------------------------------------------------------------------
remove(RFmodelFilenameT1, modelRFT1, predictedPixels,reclassFun)

#----------- Compare pixels, photo-interpreted at timeframe T1 vs modelled at T2 ---------------------------------
Correct <- ifelse(dfTrainTestPix$Reclass==dfTrainTestPix$Class,1,0)

pixelCoords <- cbind(coordinates(traintestPix))
pixelMatch <- cbind(dfTrainTestPix$Class, dfTrainTestPix$Reclass, Correct, pixelCoords)
pixelMatch <- as.data.frame(pixelMatch)
colnames(pixelMatch) <- c("Photo_Interpreted_Class","Modelled_Class", "Match", "Easting", "Northing")

###-------Select 10 % of pixels to check where photo-interpreted class is not the same as the modelled class------
pixelsToCheck <- pixelMatch[which(pixelMatch$Match == 0),]
pixelsToCheck <- stratified(pixelsToCheck, "Photo_Interpreted_Class", 0.1)

write.csv(pixelsToCheck, paste0("./Training_Data/", Sys.Date(), "Pixels_to_check.csv"))

remove(dfTrainTestPix, pixelCoords, pixelMatch, pixelsToCheck, traintestPix, Correct, aoiExtent,aoiExtentBuff, aoiFilename)

print("Finished")
