# Title         : SWEEP023 Habitat Change Tool (modal change)
# License       : CC BY-NC-SA 4.0 - Attribution-NonCommercial-ShareAlike 4.0 International
# Objective     : Compares Modal Habitat Classification over two 3-year timeframess to asses habitat decrease,
#               habitat increase and the accuracy of habitat change assessment
# Created by    : Naomi Gatis
# Created on    : 24/05/2021


#----------- Project Set Up --------------------------------------------------------------------------------------
# Check that the required packages are installed
list.of.packages <- c("diffeR", "doParallel", "here", "packrat", "raster", "svDialogs")
new.packages <- list.of.packages[!(list.of.packages %in% installed.packages()[,"Package"])]
if(length(new.packages)) install.packages(new.packages)
remove(list.of.packages, new.packages)

library(diffeR)       # Metric of Difference for Comparing Pairs of Maps
library(doParallel)   # Foreach Parallel Adapter for the parallel Package
library(here)         # A Simpler Way to Find Your Files
library(packrat)      # A Dependency Management System for Projects and their R Package Dependencies
library(raster)       # Geographic Data Analysis and Modeling
library(svDialogs)    # 'SciViews' - Standard Dialogue Boxes for Windows, MacOs and Linuxes

#----------- Get Inputs -----------------------------------------------------------------------------------------

yearT1 <- dlgInput("First 3 year timeframe of interest YYYYtoYYYY", Sys.info()["2017 to 2019"])$res
yearT2 <- dlgInput("Second 3 year timeframe of interest YYYYtoYYYY", Sys.info()["2018 to 2020"])$res

habClassFileT1a <- dlg_open("Habitat_Classification_Maps",
                            "For the first 3-year timeframe select the first Habitat Classification Map (YYYYHabitatClass)")$res
habClassFileT2a <-dlg_open("Habitat_Classification_Maps",
                           "For the first 3-year timeframe select the second Habitat Classification Map (YYYYHabitatClass)")$res
habClassFileT3a <-dlg_open("Habitat_Classification_Maps",
                           "For the first 3-year timeframe select the third Habitat Classification Map (YYYYHabitatClass)")$res
habClassFileT1b <- dlg_open("Habitat_Classification_Maps",
                            "For the second 3-year timeframe select the first Habitat Classification Map (YYYYHabitatClass)")$res
habClassFileT2b <-dlg_open("Habitat_Classification_Maps",
                           "For the second 3-year timeframe select the second Habitat Classification Map (YYYYHabitatClass)")$res
habClassFileT3b <-dlg_open("Habitat_Classification_Maps",
                           "For the second 3-year timeframe select the third Habitat Classification Map (YYYYHabitatClass)")$res
accuracyFileT1a <- dlg_open("Habitat_Classification_Maps",
                            "For the first 3-year timeframe select the first Habitat Classification Accuracy Map (YYYYHabitatClassAccuracy)")$res
accuracyFileT2a <- dlg_open("Habitat_Classification_Maps",
                            "For the first 3-year timeframe select the second Habitat Classification Accuracy Map (YYYYHabitatClassAccuracy)")$res
accuracyFileT3a <- dlg_open("Habitat_Classification_Maps",
                            "For the first 3-year timeframe select the third Habitat Classification Accuracy Map (YYYYHabitatClassAccuracy)")$res
accuracyFileT1b <- dlg_open("Habitat_Classification_Maps",
                            "For the second 3-year timeframe select the first Habitat Classification Accuracy Map (YYYYHabitatClassAccuracy)")$res
accuracyFileT2b <- dlg_open("Habitat_Classification_Maps",
                            "For the second 3-year timeframe select the second Habitat Classification Accuracy Map (YYYYHabitatClassAccuracy)")$res
accuracyFileT3b <- dlg_open("Habitat_Classification_Maps",
                            "For the second 3-year timeframe select the third Habitat Classification Accuracy Map (YYYYHabitatClassAccuracy)")$res

habClassT1a <- raster(habClassFileT1a)
habClassT2a <- raster(habClassFileT2a)
habClassT3a <- raster(habClassFileT3a)
habClassT1 <- stack(habClassT1a, habClassT2a, habClassT3a)

habClassT1b <- raster(habClassFileT1b)
habClassT2b <- raster(habClassFileT2b)
habClassT3b <- raster(habClassFileT3b)
habClassT2 <- stack(habClassT1b, habClassT2b, habClassT3b)

accuracyT1a <- raster(accuracyFileT1a)
accuracyT2a <- raster(accuracyFileT2a)
accuracyT3a <- raster(accuracyFileT3a)
accuracyT1b <- raster(accuracyFileT1b)
accuracyT2b <- raster(accuracyFileT2b)
accuracyT3b <- raster(accuracyFileT3b)
accuracyT12 <- stack(accuracyT1a, accuracyT2a, accuracyT3a,accuracyT1b, accuracyT2b, accuracyT3b)

#-----------------------------------------------------------------------------------------------------------------
remove(habClassFileT1a, habClassFileT2a, habClassFileT3a, habClassFileT1b, habClassFileT2b,
       habClassFileT3b, habClassT1a, habClassT1b, habClassT2a,habClassT2b,habClassT3a,habClassT3b,
       accuracyT1a, accuracyT2a, accuracyT3a, accuracyT1b, accuracyT2b, accuracyT3b,
       accuracyFileT1a, accuracyFileT2a, accuracyFileT3a, accuracyFileT1b, accuracyFileT2b,
       accuracyFileT3b)

#------------Define and run Functions to calculate habitat Decrease, Increase and accuracy -----------------------

Increasefun <- function(x) {ifelse((x[1] != x[2]), x[2], NA) }
Decreasefun <- function(x) {ifelse((x[1] != x[2]), x[1], NA) }
#accuracy is the probability of being correct in timeframe 1 and timeframe 2 given the probability for
#each timeframe is the probability that Y1&Y2 OR Y1&Y3 OR Y1&Y3 or Y1&Y2&Y3 are accurate given not mutually exclusive

AccFun <- function(x){(x[1]*x[2] + x[1]*x[3] + x[2]*x[3] - (2*x[1]*x[2]*x[3])) *
                      (x[4]*x[5] + x[4]*x[6] + x[5]*x[6] - (2*x[4]*x[5]*x[6]))}

cores=detectCores()
cl <- makePSOCKcluster(cores[1]-1) #not to overload your computer
registerDoParallel(cl)

#where the class is different in all years NA as change not discernible from uncertainty
modeT1 <- modal(habClassT1, ties = "NA")
modeT2 <- modal(habClassT2, ties = "NA")
habClassmodes <- stack(modeT1, modeT2)

habIncrease <- overlay(habClassmodes, fun = Increasefun)
habDecrease <- overlay(habClassmodes, fun = Decreasefun)

changeAcc <- overlay(accuracyT12, fun=AccFun)

stopCluster(cl)

writeRaster(habIncrease, paste0("./Habitat_Change_Maps/", yearT1, "_cf_", yearT2, "_Modal_Increase", sep = ""),
            format = "GTiff", overwrite = TRUE)
writeRaster(habDecrease, paste0("./Habitat_Change_Maps/", yearT1, "_cf_", yearT2, "_Modal_Decrease", sep = ""),
            format = "GTiff", overwrite = TRUE)
writeRaster(changeAcc, paste0("./Habitat_Change_Maps/", yearT1, "_cf_", yearT2, "_AccuracyHabitatChange", sep = ""),
            format = "GTiff", overwrite = TRUE)

#-----------------------------------------------------------------------------------------------------------------
remove(Increasefun, Decreasefun, AccFun, cores, cl, habClassmodes, habIncrease, habDecrease,
       accuracyT12, changeAcc, habClassT1, habClassT2)

#----------- Calculate and write transition matrix ---------------------------------------------------------------

continTable <- crosstabm(comp = modeT1, ref = modeT2, percent = FALSE)

conTabPers <- diag(continTable)
conTabIncrease <- colSums(continTable)-diag(continTable)
conTabDecrease <- rowSums(continTable)-diag(continTable)

continTable <- cbind(conTabPers, conTabIncrease, conTabDecrease, continTable)

write.csv(continTable, paste0("./Habitat_Change_Maps/HabClassChange", yearT1, "_cf_", yearT2, ".csv", sep = ""))

remove(continTable, conTabIncrease, conTabDecrease, conTabPers,
       yearT1, yearT2, modeT1, modeT2)

print("Finished")

