clear
cd "LOCAL DIRECTORY" // Please enter your local working directory here
set scheme uncluttered
use results.dta, clear

// note: One observation per participant (country/firm) per period (irrespective of whether agreement was reached or not, irrespective of role)
// ran with stata 17.0, last check Jan 11, 2022

tab playertreatment treatid
label define treatlabel 1 "Price secrecy" 2 "Price transparency" 3 "Full transparency" 
label value treatid treatlabel

// variable labels
label var groupRDcost_real "R&D costs"
label var period "Period"
label var price "Price"
label var investment "Investment level (switching point)"
label var playertreatment "Treatment (text)"
label var treatid "Treatment (dummies)"
label var playeraltu "Altruism (questionnaire based)"
label var playerrisk "Risk aversion (questionnaire based)"
label var playercomp "Competitiveness (questionnaire based)"
label var sessioncode "Session identifier"
label var groupid_in_subsession "Group identifier (within session)"
label var playerlabel "Role"
label var groupresearch_invest "successful investment"
label var playerdrugs_sold "# drugs bought"
label var playerearnings "Earnings/profit"
label var playerbudget "Starting budget"
label var playerage "Age"
label var playerstudy "Study program"
label var playergender "Gender"

// generate id for each group
egen marketgroup=concat(sessioncode groupid_in_subsession), p(_)
egen marketid = group(marketgroup)
label var marketgroup "Unique group identifier"

// generate variable indicating whether price agreement was reached
gen agreement=0
replace agreement=1 if price>0
label var agreement "Agreement dummy"
// generate high budget-country dummy (equal 1 for NL/GE)

gen rich=0
replace rich=1 if playerlabel=="NL"
replace rich=1 if playerlabel=="GE"
label var rich "High budget dummy"
label define richlabel 0 "Low budget" 1 "High budget"
label value rich richlabel

// generate control variables (with averages across countries)
replace playerrisk="10" if playerrisk=="10 -Very willing to take risks"
destring playerrisk, replace
replace playeraltu="10" if playeraltu=="10 - Very willing to do so"
replace playeraltu="0" if playeraltu=="0 - Completely unwilling to do so"
destring playeraltu, replace
replace playercomp="10" if playercomp=="10 - Very competitive"
replace playercomp="0" if playercomp=="0 - Not competitive at all"
destring playercomp, replace

bysort marketid: egen meanrisk = mean(playerrisk) if playerlabel!="Pharma"
sort marketid meanrisk
bysort marketid: replace meanrisk=meanrisk[1]
bysort marketid: egen meanult = mean(playeraltu) if playerlabel!="Pharma"
sort marketid meanult
bysort marketid: replace meanult=meanult[1]
bysort marketid: egen meancomp = mean(playercomp) if playerlabel!="Pharma"
sort marketid meancomp
bysort marketid: replace meancomp=meancomp[1]

replace price=. if price==0

// balancing tests
preserve
keep if period==10 // keep 1 observation per participant
gen genderdummy=(playergender=="Female")
reg genderdummy i.treatid, r
reg playerage i.treatid, r
reg playerrisk i.treatid, r
reg playeraltu i.treatid, r
reg playercompet i.treatid, r
tabulate playerstudy treatid, chi2
restore

// hypothesis 1: non-agreement

// pooling all data
preserve
// keep relevant observations: only countries, only if investment was successful
drop if playerlabel=="Pharma"
keep if groupresearch_invest==1

// statistics on agreement by treatment and overall
bysort playertreatment: sum agreement
sum agreement

restore


// hypothesis 2 / section 2.1: price levels

preserve
// keep relevant observations: only countries, only if investment was successful and agreement occurred
drop if playerlabel=="Pharma"
keep if groupresearch_invest==1
keep if agreement==1

bysort playertreatment: sum price
bysort playertreatment rich: sum price

collapse (mean) price marketid, by(playertreatment marketgroup)

ranksum price if playertreatment=="treatment1" || playertreatment=="treatment2", by(playertreatment)
ranksum price if playertreatment=="treatment1" || playertreatment=="treatment3", by(playertreatment)

restore 

// low/high health budget separately
preserve
drop if playerlabel=="Pharma"
keep if groupresearch_invest==1
keep if agreement==1

bysort playertreatment rich: sum price

collapse (mean) price marketid, by(playertreatment marketgroup rich)

ranksum price if (playertreatment=="treatment1" || playertreatment=="treatment2") & rich==1, by(playertreatment)
ranksum price if (playertreatment=="treatment1" || playertreatment=="treatment2") & rich==0, by(playertreatment)

ranksum price if (playertreatment=="treatment1" || playertreatment=="treatment3") & rich==1, by(playertreatment)
ranksum price if (playertreatment=="treatment1" || playertreatment=="treatment3") & rich==0, by(playertreatment)
restore


// plot
preserve
drop if groupresearch_invest==0
drop if price==0
drop if price==.
keep if agreement==1
drop if playerlabel=="Pharma"

egen playerid=group(playerlabel)

levelsof treatid, local(treatlab)
levelsof playerid, local(plab)
levelsof period, local(periods)
tempname matrix
gen ll = .
gen ul = .
display `treatlab'
foreach t of local treatlab {
	foreach p of local periods {
		foreach plabel of local plab {
		quietly bootstrap _b, r(10000) cluster(marketid) nodots:  mean price if  treatid==`t' &  period==`p' & playerid==`plabel'
        quietly estat bootstrap, all
        matrix `matrix' =  e(ci_bc) 
        quietly replace ll = `matrix'[1, 1] if  treatid==`t' &  period==`p' & playerid==`plabel'
        quietly replace ul = `matrix'[2, 1] if  treatid==`t' &  period==`p' & playerid==`plabel'
		}
	}
}

collapse  (mean) price ll ul marketid, by(treatid playerlabel period)

reshape wide price ll ul, i(marketid period treatid) j(playerlabel) string

label var priceES "Spain"
label var priceGE "Germany"
label var priceNL "the Netherlands"
label var pricePL "Poland"
sort period
graph twoway (line priceES period, lpattern(solid)) (line priceGE period, lpattern(longdash_dot)) (line priceNL period, lpattern(dash)) (line pricePL period, lpattern(shortdash_dot)) (rarea llES ulES period, color(%15)) (rarea llGE ulGE period, color(%15)) (rarea llNL ulNL period, color(%15)) (rarea llPL ulPL period, color(%15)), by(treatid, c(3) note("Prices conditional on agreement, 95% confidence intervals") caption("Figure 1", pos(11) ring(100))) scheme(uncluttered) xtitle(Period) ytitle(Price) note("") ylab(0(25)75) xlab(2(2)11) legend(order(1 "Spain" 2 "Germany" 3 "the Netherlands" 4 "Poland")) subtitle(, bcolor(white)) 

restore

// hypothesis 3: variance of prices / section 2.1: price convergence

preserve
// keep relevant observations: only countries, only if investment was successful
drop if playerlabel=="Pharma"
keep if groupresearch_invest==1
keep if agreement==1

collapse (sd) price (mean) marketid, by(playertreatment marketgroup period)
replace price=price^2 // we test variance (not standard deviation)

collapse (mean) price marketid, by(playertreatment marketgroup)

bysort playertreatment: sum price

ranksum price if playertreatment=="treatment1" || playertreatment=="treatment2", by(playertreatment)
ranksum price if playertreatment=="treatment1" || playertreatment=="treatment3", by(playertreatment)

restore 

// low/high health budget separately
preserve

// keep relevant observations: only countries, only if investment was successful
drop if playerlabel=="Pharma"
keep if groupresearch_invest==1
keep if agreement==1

collapse (sd) price (mean) marketid, by(playertreatment marketgroup period rich)
replace price=price^2 // we test variance (not standard deviation)
collapse (mean) price marketid, by(playertreatment marketgroup rich)

bysort playertreatment rich: sum price

ranksum price if (playertreatment=="treatment1" || playertreatment=="treatment2") & rich==1, by(playertreatment)
ranksum price if (playertreatment=="treatment1" || playertreatment=="treatment2") & rich==0, by(playertreatment)

ranksum price if (playertreatment=="treatment1" || playertreatment=="treatment3") & rich==1, by(playertreatment)
ranksum price if (playertreatment=="treatment1" || playertreatment=="treatment3") & rich==0, by(playertreatment)

restore 

// Hypothesis 6: investment levels / section 2.2 - effect on investment


bysort playertreatment: sum investment if playerlabel=="Pharma"
bysort playertreatment: sum groupresearch_invest if playerlabel=="Pharma"

preserve

collapse (mean) investment marketid, by(playertreatment marketgroup)
ranksum investment if playertreatment=="treatment1" || playertreatment=="treatment2", by(playertreatment)
ranksum investment if playertreatment=="treatment1" || playertreatment=="treatment3", by(playertreatment)

restore

// plot investment

preserve
keep if playerlabel=="Pharma"

collapse (mean) investment, by(period marketid treatid)

levelsof treatid, local(treats)
levelsof period, local(periods)
tempname matrix
gen lo = .
gen hi = .

foreach t of local treats{
	foreach p of local periods {
		quietly bootstrap _b, r(10000) cluster(marketid) nodots:  mean investment if period==`p' & treatid==`t'
        quietly estat bootstrap, all
        matrix `matrix' =  e(ci_bc) 
        quietly replace lo = `matrix'[1, 1] if period==`p' & treatid==`t'
        quietly replace hi = `matrix'[2, 1] if period==`p' & treatid==`t'
	}
}

collapse (mean) investment lo hi, by(period treatid)
graph twoway (line investment period) (rarea lo hi period, color(%15)), by(treatid, c(3) note("") caption("Figure 2", pos(11) ring(100))) xtitle(Period) ytitle("Investment level (switching point)")  ylab(600(300)1800) xlab(2(3)11) scheme(uncluttered)  legend(order(1 "Investment" 2 "95% CI")) subtitle(, bcolor(white))


restore

// Hypothesis 5: rd cost factoring (in section 2.2)

// r&d cost factoring

preserve
keep if groupresearch_invest==1
keep if agreement==1 | playerlabel=="Pharma"
bysort marketid period: egen meanprice=mean(price)
bysort playertreatment: sum meanprice groupRDcost_real
bysort marketid period: egen maxmean=max(meanprice)
keep if playerlabel=="Pharma"
bysort playertreatment: sum meanprice groupRDcost_real
label value treatid treatlabel

reg meanprice c.groupRDcost_real##i.treatid period, cl(marketgroup)
reg meanprice c.groupRDcost_real##i.treatid period meanult meanrisk meancomp playerrisk playeraltu playercompet, cl(marketgroup)

twoway (lfitci meanprice groupRDcost_real, fintensity(55))  (scatter meanprice groupRDcost_real, msymbol(Oh) msize(vsmall)),  scheme(uncluttered) by(treatid, c(3) note("") imargin(medium) caption("Figure 3", pos(11) ring(100))) subtitle(, bcolor(white)) ytitle(Average price) xtitle(R&D cost draw) legend(order(1 "95% CI" 2 "Linear fit" 3 "Average price") c(3)) xsize(8)

restore


// hypothesis 9: welfare - section 2.3

preserve
bysort marketid period: egen patients = sum(playerdrugs_sold)
// profits net of budget
bysort marketid period: egen totalearning = sum(playerearnings - playerbudget)

bysort playertreatment: sum patients totalearning
collapse (mean) patients totalearning marketid, by(playertreatment marketgroup)

ranksum patients if playertreatment=="treatment1" || playertreatment=="treatment2", by(playertreatment)
ranksum patients if playertreatment=="treatment1" || playertreatment=="treatment3", by(playertreatment)

ranksum totalearning if playertreatment=="treatment1" || playertreatment=="treatment2", by(playertreatment)
ranksum totalearning if playertreatment=="treatment1" || playertreatment=="treatment3", by(playertreatment)

restore

// distribution of welfare between pharma and countries

preserve
gen pharma=0
replace pharma=1 if playerlabel=="Pharma"
gen profit = playerearnings-playerbudget
collapse (sum) profit, by(playertreatment pharma period marketgroup)
collapse (mean) profit, by(playertreatment pharma marketgroup)
bysort playertreatment pharma: sum profit
bysort playertreatment marketgroup: egen totalprofit = sum(profit)
gen sharepharma=profit/totalprofit if pharma==1
bysort playertreatment: sum sharepharma
ranksum sharepharma if playertreatment=="treatment1" || playertreatment=="treatment2", by(playertreatment)
ranksum sharepharma if playertreatment=="treatment1" || playertreatment=="treatment3", by(playertreatment)

restore

// plots

preserve
gen hinc=0
replace hinc=1 if playerlabel=="PL" | playerlabel=="ES"
replace hinc=2 if playerlabel=="GE" | playerlabel=="NL"

gen profit = playerearnings-playerbudget
collapse (sum) profit, by(treatid hinc period marketgroup)
collapse (mean) profit, by(treatid hinc)

reshape wide profit, i(treatid) j(hinc)

graph bar profit0 profit1 profit2, over(treatid) stack scheme(uncluttered) legend(order(1 "Pharma" 2 "Low health budget" 3 "High health budget") c(3) on)  intensity(75) ylab(0(1000)3000) title(Overall) caption("Figure 4", pos(11) ring(100))
graph save profitdiv2.gph, replace

restore

preserve

keep if groupresearch_invest==1
gen hinc=0
replace hinc=1 if playerlabel=="PL" | playerlabel=="ES"
replace hinc=2 if playerlabel=="GE" | playerlabel=="NL"

gen profit = playerearnings-playerbudget
collapse (sum) profit, by(treatid hinc period marketgroup)
collapse (mean) profit, by(treatid hinc)

reshape wide profit, i(treatid) j(hinc)

graph bar profit0 profit1 profit2, over(treatid) stack legend(order(1 "Pharma" 2 "Low health budget" 3 "High health budget") c(3) on) scheme(uncluttered) intensity(75) ylab(0(1000)3000) title(Conditional on successful investment)  caption("  ", pos(11) ring(100))
graph save profitdiv_inv2.gph, replace

restore

graph combine profitdiv2.gph profitdiv_inv2.gph, iscale(*1) xsize(9)

// appendix: reproduce tables for figures
eststo clear
// new variable labels
label var meanult "Countries' average altruism"
label var meanrisk "Countries' average risk aversion"
label var meancomp "Countries' average competitiveness"
label var playeraltu "Firm's altruism"
label var playerrisk "Firm's risk aversion"
label var playercomp "Firm's competitiveness"

preserve
keep if groupresearch_invest==1
keep if agreement==1 | playerlabel=="Pharma"
bysort marketid period: egen meanprice=mean(price)
bysort playertreatment: sum meanprice groupRDcost_real
bysort marketid period: egen maxmean=max(meanprice)
keep if playerlabel=="Pharma"
bysort playertreatment: sum meanprice groupRDcost_real
label value treatid treatlabel
label var meanprice "Average price"

reg meanprice c.groupRDcost_real##i.treatid period, cl(marketgroup)
reg meanprice c.groupRDcost_real##i.treatid period meanult meanrisk meancomp playerrisk playeraltu playercompet, cl(marketgroup)

reg meanprice c.groupRDcost_real period meanult meanrisk meancomp playerrisk playeraltu playercompet if treatid==1, cl(marketgroup)
reg meanprice c.groupRDcost_real period meanult meanrisk meancomp playerrisk playeraltu playercompet if treatid==2, cl(marketgroup)
reg meanprice c.groupRDcost_real period meanult meanrisk meancomp playerrisk playeraltu playercompet if treatid==3, cl(marketgroup)

restore


