# coding=utf-8
# Copyright 2020 The HuggingFace Datasets Authors and the current dataset script contributor.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""MultiEURLEX - A multi-lingual and multi-label legal document classification dataset."""


import json
import os

import datasets


_CITATION = """\
@InProceedings{chalkidis-etal-2021-multieurlex,
  author = {Chalkidis, Ilias
                and Fergadiotis, Manos
                and Androutsopoulos, Ion},
  title = {MultiEURLEX -- A multi-lingual and multi-label legal document
               classification dataset for zero-shot cross-lingual transfer},
  booktitle = {Proceedings of the 2021 Conference on Empirical Methods
               in Natural Language Processing},
  year = {2021},
  publisher = {Association for Computational Linguistics},
  location = {Punta Cana, Dominican Republic},
}"""

_DESCRIPTION = """\
MultiEURLEX comprises 65k EU laws in 23 official EU languages (some low-ish resource).
Each EU law has been annotated with EUROVOC concepts (labels) by the Publication Office of EU.
As with the English EURLEX, the goal is to predict the relevant EUROVOC concepts (labels);
this is multi-label classification task (given the text, predict multiple labels).
"""

DATA_URL = "https://zenodo.org/record/5363165/files/multi_eurlex.tar.gz"

_LANGUAGES = [
    "en",
    "da",
    "de",
    "nl",
    "sv",
    "bg",
    "cs",
    "hr",
    "pl",
    "sk",
    "sl",
    "es",
    "fr",
    "it",
    "pt",
    "ro",
    "et",
    "fi",
    "hu",
    "lt",
    "lv",
    "el",
    "mt",
]

_CONCEPTS = {
    "level_1": [
        "100149",
        "100160",
        "100148",
        "100147",
        "100152",
        "100143",
        "100156",
        "100158",
        "100154",
        "100153",
        "100142",
        "100145",
        "100150",
        "100162",
        "100159",
        "100144",
        "100151",
        "100157",
        "100161",
        "100146",
        "100155",
    ],
    "level_2": [
        "100215",
        "100211",
        "100213",
        "100214",
        "100209",
        "100208",
        "100216",
        "100210",
        "100212",
        "100270",
        "100273",
        "100269",
        "100272",
        "100276",
        "100274",
        "100271",
        "100268",
        "100275",
        "100203",
        "100201",
        "100202",
        "100207",
        "100205",
        "100204",
        "100199",
        "100206",
        "100200",
        "100198",
        "100192",
        "100193",
        "100191",
        "100196",
        "100197",
        "100195",
        "100194",
        "100230",
        "100231",
        "100229",
        "100227",
        "100228",
        "100226",
        "100173",
        "100170",
        "100172",
        "100171",
        "100250",
        "100245",
        "100251",
        "100246",
        "100248",
        "100249",
        "100252",
        "100247",
        "100260",
        "100262",
        "100261",
        "100237",
        "100241",
        "100238",
        "100239",
        "100240",
        "100234",
        "100236",
        "100232",
        "100235",
        "100233",
        "100164",
        "100166",
        "100169",
        "100167",
        "100165",
        "100163",
        "100168",
        "100178",
        "100181",
        "100179",
        "100184",
        "100183",
        "100180",
        "100182",
        "100222",
        "100217",
        "100219",
        "100223",
        "100218",
        "100220",
        "100221",
        "100288",
        "100285",
        "100287",
        "100289",
        "100286",
        "100264",
        "100263",
        "100265",
        "100267",
        "100266",
        "100177",
        "100174",
        "100176",
        "100175",
        "100225",
        "100224",
        "100256",
        "100255",
        "100258",
        "100253",
        "100254",
        "100257",
        "100259",
        "100278",
        "100280",
        "100284",
        "100277",
        "100279",
        "100282",
        "100281",
        "100283",
        "100190",
        "100186",
        "100185",
        "100188",
        "100189",
        "100187",
        "100244",
        "100243",
        "100242",
    ],
    "level_3": [
        "1754",
        "5881",
        "c_4ee83dea",
        "2479",
        "1268",
        "181",
        "1415",
        "2789",
        "85",
        "3487",
        "1820",
        "4305",
        "2688",
        "317",
        "3257",
        "2459",
        "108",
        "1004",
        "4050",
        "1909",
        "1916",
        "4184",
        "3307",
        "540",
        "965",
        "2398",
        "4619",
        "2496",
        "490",
        "1690",
        "2475",
        "1687",
        "1221",
        "385",
        "3300",
        "52",
        "3318",
        "2690",
        "4706",
        "1700",
        "2517",
        "1882",
        "1417",
        "3820",
        "4414",
        "3879",
        "4565",
        "3798",
        "3809",
        "3810",
        "1362",
        "1401",
        "1400",
        "3895",
        "1398",
        "5359",
        "1390",
        "1386",
        "1845",
        "1410",
        "1834",
        "5615",
        "1718",
        "5614",
        "2720",
        "2507",
        "3587",
        "4303",
        "1418",
        "1388",
        "3151",
        "1804",
        "1630",
        "1000",
        "1488",
        "2489",
        "2639",
        "1039",
        "2656",
        "2632",
        "929",
        "394",
        "2698",
        "1012",
        "2448",
        "1018",
        "424",
        "2510",
        "4602",
        "2486",
        "1809",
        "1325",
        "1326",
        "1321",
        "2504",
        "1021",
        "1310",
        "8469",
        "285",
        "1452",
        "2495",
        "2149",
        "4370",
        "1016",
        "6332",
        "4671",
        "3239",
        "1850",
        "4380",
        "3156",
        "2520",
        "5842",
        "2519",
        "2292",
        "2128",
        "614",
        "c_4c7717f3",
        "2487",
        "2449",
        "2450",
        "1786",
        "1810",
        "524",
        "2488",
        "87",
        "5268",
        "13",
        "474",
        "4291",
        "138",
        "3193",
        "139",
        "3584",
        "11",
        "3185",
        "1155",
        "54",
        "2474",
        "802",
        "539",
        "3581",
        "4413",
        "1164",
        "1154",
        "813",
        "5985",
        "445",
        "4189",
        "4703",
        "4300",
        "69",
        "2469",
        "3582",
        "554",
        "3451",
        "2628",
        "2647",
        "3423",
        "2464",
        "1346",
        "3461",
        "3474",
        "3467",
        "2189",
        "3476",
        "2500",
        "4697",
        "6722",
        "3450",
        "99",
        "3074",
        "2460",
        "3489",
        "2734",
        "2723",
        "5877",
        "711",
        "1277",
        "2711",
        "2443",
        "2442",
        "1063",
        "651",
        "2505",
        "2972",
        "2477",
        "2493",
        "4363",
        "4412",
        "4630",
        "2014",
        "5962",
        "1372",
        "2320",
        "3544",
        "1652",
        "5913",
        "2476",
        "3605",
        "2551",
        "4358",
        "2814",
        "937",
        "962",
        "2707",
        "2481",
        "2817",
        "4185",
        "2914",
        "2478",
        "3797",
        "4418",
        "3641",
        "3632",
        "3689",
        "4415",
        "3160",
        "2494",
        "3101",
        "3131",
        "4505",
        "3092",
        "1954",
        "2181",
        "2015",
        "3098",
        "4539",
        "4522",
        "2512",
        "4515",
        "2184",
        "82",
        "4543",
        "2178",
        "557",
        "3209",
        "5974",
        "1627",
        "1074",
        "4299",
        "2468",
        "3522",
        "1163",
        "2491",
        "3273",
        "5275",
        "1737",
        "1802",
        "4276",
        "2180",
        "2258",
        "4436",
        "41",
        "3232",
        "2242",
        "53",
        "2246",
        "2166",
        "77",
        "6034",
        "3062",
        "517",
        "2702",
        "2703",
        "4157",
        "1748",
        "695",
        "4750",
        "558",
        "2186",
        "4367",
        "1953",
        "883",
        "2370",
        "1282",
        "2573",
        "3025",
        "1453",
        "2152",
        "2001",
        "4045",
        "4179",
        "4704",
        "5789",
        "3935",
        "4222",
        "5371",
        "2701",
        "1538",
        "5400",
        "3929",
        "523",
        "584",
        "7347",
        "3901",
        "538",
        "3900",
        "3908",
        "3899",
        "6922",
        "565",
        "567",
        "4010",
        "566",
        "1432",
        "3513",
        "3878",
        "3954",
        "573",
        "2351",
        "5923",
        "2472",
        "1422",
        "4486",
        "2467",
        "668",
        "3683",
        "3278",
        "4705",
        "2175",
        "4554",
        "1426",
        "4488",
        "1405",
        "3028",
        "5922",
        "873",
        "778",
        "2059",
        "795",
        "784",
        "4150",
        "494",
        "486",
        "4366",
        "1370",
        "4361",
        "2473",
        "2012",
        "3855",
        "c_415bab24",
        "119",
        "2188",
        "3716",
        "4365",
        "4826",
        "2179",
        "6015",
        "1455",
        "8380",
        "2170",
        "5286",
        "2190",
        "2172",
        "2167",
        "5285",
        "2171",
        "6157",
        "2191",
        "90",
        "2187",
        "5366",
        "2767",
        "3743",
        "1411",
        "2498",
        "3737",
        "2364",
        "1274",
        "1414",
        "744",
        "1412",
        "5350",
        "3783",
        "753",
        "5576",
        "5051",
        "5620",
        "438855",
        "1005",
        "3258",
        "4067",
        "5640",
        "2163",
        "5130",
        "5767",
        "5769",
        "1451",
        "5758",
        "4304",
        "4057",
        "4060",
        "4040",
        "5442",
        "26",
        "4017",
        "5258",
        "8473",
        "525",
        "4479",
        "3956",
        "3928",
        "3952",
        "6382",
        "3941",
        "3949",
        "3946",
        "5017",
        "4314",
        "2775",
        "239",
        "2763",
        "6788",
        "1360",
        "656",
        "2418",
        "2417",
        "5360",
        "2412",
        "1602",
        "2416",
        "2414",
        "2413",
        "1115",
        "2737",
        "2735",
        "2736",
        "1258",
        "4416",
        "6052",
        "8301",
        "1088",
        "7816",
        "8189",
        "2564",
        "2288",
        "6138",
        "6136",
        "1520",
        "1506",
        "1329",
        "337",
        "7815",
        "c_6acbfd25",
        "1184",
        "6135",
        "7879",
        "4840",
        "7861",
        "7855",
        "7868",
        "7817",
        "7814",
        "864",
        "281",
        "5424",
        "311",
        "3778",
        "1087",
        "2287",
        "1086",
        "911",
        "912",
        "913",
        "909",
        "914",
        "1587",
        "1293",
        "1615",
        "5259",
        "1632",
        "1297",
        "5302",
        "4069",
        "1844",
        "4607",
        "2117",
        "8365",
        "c_2b2e17b1",
        "5283",
        "5776",
        "c_964c9649",
        "4690",
        "6205",
        "c_b2c019c8",
        "7207",
        "1794",
        "4587",
        "8369",
        "187",
        "4845",
        "5083",
        "5087",
        "2148",
        "594",
        "5778",
        "8278",
        "c_c385cfb4",
        "5780",
        "8366",
        "5265",
        "1847",
        "1105",
        "1191",
        "8367",
        "2106",
        "2838",
        "727",
        "5448",
        "3791",
        "2858",
        "3889",
        "956",
        "2113",
        "2705",
        "1765",
        "2848",
        "2119",
        "c_6e703074",
        "122",
        "5775",
        "5434",
        "2200",
        "1658",
        "5435",
        "130",
        "2079",
        "5931",
        "4590",
        "c_789ead37",
        "4483",
        "5437",
        "5655",
        "2616",
        "4256",
        "1800",
        "335",
        "427",
        "84",
        "712",
        "2402",
        "2497",
        "637",
        "4006",
        "3013",
        "3659",
        "4362",
        "56",
        "5848",
        "2515",
        "3058",
        "371",
        "343",
        "2090",
        "2524",
        "3549",
        "3150",
        "3146",
        "6011",
        "833",
        "1707",
        "434743",
        "1158",
        "2470",
        "c_98d1408a",
        "2825",
        "5781",
    ],
    "original": [
        "100149",
        "100160",
        "100148",
        "100147",
        "100152",
        "100143",
        "100156",
        "100158",
        "100154",
        "100153",
        "100142",
        "100145",
        "100150",
        "100162",
        "100159",
        "100144",
        "100151",
        "100157",
        "100161",
        "100146",
        "100155",
        "100215",
        "100211",
        "100213",
        "100214",
        "100209",
        "100208",
        "100216",
        "100210",
        "100212",
        "100270",
        "100273",
        "100269",
        "100272",
        "100276",
        "100274",
        "100271",
        "100268",
        "100275",
        "100203",
        "100201",
        "100202",
        "100207",
        "100205",
        "100204",
        "100199",
        "100206",
        "100200",
        "100198",
        "100192",
        "100193",
        "100191",
        "100196",
        "100197",
        "100195",
        "100194",
        "100230",
        "100231",
        "100229",
        "100227",
        "100228",
        "100226",
        "100173",
        "100170",
        "100172",
        "100171",
        "100250",
        "100245",
        "100251",
        "100246",
        "100248",
        "100249",
        "100252",
        "100247",
        "100260",
        "100262",
        "100261",
        "100237",
        "100241",
        "100238",
        "100239",
        "100240",
        "100234",
        "100236",
        "100232",
        "100235",
        "100233",
        "100164",
        "100166",
        "100169",
        "100167",
        "100165",
        "100163",
        "100168",
        "100178",
        "100181",
        "100179",
        "100184",
        "100183",
        "100180",
        "100182",
        "100222",
        "100217",
        "100219",
        "100223",
        "100218",
        "100220",
        "100221",
        "100288",
        "100285",
        "100287",
        "100289",
        "100286",
        "100264",
        "100263",
        "100265",
        "100267",
        "100266",
        "100177",
        "100174",
        "100176",
        "100175",
        "100225",
        "100224",
        "100256",
        "100255",
        "100258",
        "100253",
        "100254",
        "100257",
        "100259",
        "100278",
        "100280",
        "100284",
        "100277",
        "100279",
        "100282",
        "100281",
        "100283",
        "100190",
        "100186",
        "100185",
        "100188",
        "100189",
        "100187",
        "100244",
        "100243",
        "100242",
        "1754",
        "5881",
        "c_4ee83dea",
        "2479",
        "1268",
        "181",
        "1415",
        "2789",
        "85",
        "3487",
        "1820",
        "4305",
        "2688",
        "317",
        "3257",
        "2459",
        "108",
        "1004",
        "4050",
        "1909",
        "1916",
        "4184",
        "3307",
        "540",
        "965",
        "2398",
        "4619",
        "2496",
        "490",
        "1690",
        "2475",
        "1687",
        "1221",
        "385",
        "3300",
        "52",
        "3318",
        "2690",
        "4706",
        "1700",
        "2517",
        "1882",
        "1417",
        "3820",
        "4414",
        "3879",
        "4565",
        "3798",
        "3809",
        "3810",
        "1362",
        "1401",
        "1400",
        "3895",
        "1398",
        "5359",
        "1390",
        "1386",
        "1845",
        "1410",
        "1834",
        "5615",
        "1718",
        "5614",
        "2720",
        "2507",
        "3587",
        "4303",
        "1418",
        "1388",
        "3151",
        "1804",
        "1630",
        "1000",
        "1488",
        "2489",
        "2639",
        "1039",
        "2656",
        "2632",
        "929",
        "394",
        "2698",
        "1012",
        "2448",
        "1018",
        "424",
        "2510",
        "4602",
        "2486",
        "1809",
        "1325",
        "1326",
        "1321",
        "2504",
        "1021",
        "1310",
        "8469",
        "285",
        "1452",
        "2495",
        "2149",
        "4370",
        "1016",
        "6332",
        "4671",
        "3239",
        "1850",
        "4380",
        "3156",
        "2520",
        "5842",
        "2519",
        "2292",
        "2128",
        "614",
        "c_4c7717f3",
        "2487",
        "2449",
        "2450",
        "1786",
        "1810",
        "524",
        "2488",
        "87",
        "5268",
        "13",
        "474",
        "4291",
        "138",
        "3193",
        "139",
        "3584",
        "11",
        "3185",
        "1155",
        "54",
        "2474",
        "802",
        "539",
        "3581",
        "4413",
        "1164",
        "1154",
        "813",
        "5985",
        "445",
        "4189",
        "4703",
        "4300",
        "69",
        "2469",
        "3582",
        "554",
        "3451",
        "2628",
        "2647",
        "3423",
        "2464",
        "1346",
        "3461",
        "3474",
        "3467",
        "2189",
        "3476",
        "2500",
        "4697",
        "6722",
        "3450",
        "99",
        "3074",
        "2460",
        "3489",
        "2734",
        "2723",
        "5877",
        "711",
        "1277",
        "2711",
        "2443",
        "2442",
        "1063",
        "651",
        "2505",
        "2972",
        "2477",
        "2493",
        "4363",
        "4412",
        "4630",
        "2014",
        "5962",
        "1372",
        "2320",
        "3544",
        "1652",
        "5913",
        "2476",
        "3605",
        "2551",
        "4358",
        "2814",
        "937",
        "962",
        "2707",
        "2481",
        "2817",
        "4185",
        "2914",
        "2478",
        "3797",
        "4418",
        "3641",
        "3632",
        "3689",
        "4415",
        "3160",
        "2494",
        "3101",
        "3131",
        "4505",
        "3092",
        "1954",
        "2181",
        "2015",
        "3098",
        "4539",
        "4522",
        "2512",
        "4515",
        "2184",
        "82",
        "4543",
        "2178",
        "557",
        "3209",
        "5974",
        "1627",
        "1074",
        "4299",
        "2468",
        "3522",
        "1163",
        "2491",
        "3273",
        "5275",
        "1737",
        "1802",
        "4276",
        "2180",
        "2258",
        "4436",
        "41",
        "3232",
        "2242",
        "53",
        "2246",
        "2166",
        "77",
        "6034",
        "3062",
        "517",
        "2702",
        "2703",
        "4157",
        "1748",
        "695",
        "4750",
        "558",
        "2186",
        "4367",
        "1953",
        "883",
        "2370",
        "1282",
        "2573",
        "3025",
        "1453",
        "2152",
        "2001",
        "4045",
        "4179",
        "4704",
        "5789",
        "3935",
        "4222",
        "5371",
        "2701",
        "1538",
        "5400",
        "3929",
        "523",
        "584",
        "7347",
        "3901",
        "538",
        "3900",
        "3908",
        "3899",
        "6922",
        "565",
        "567",
        "4010",
        "566",
        "1432",
        "3513",
        "3878",
        "3954",
        "573",
        "2351",
        "5923",
        "2472",
        "1422",
        "4486",
        "2467",
        "668",
        "3683",
        "3278",
        "4705",
        "2175",
        "4554",
        "1426",
        "4488",
        "1405",
        "3028",
        "5922",
        "873",
        "778",
        "2059",
        "795",
        "784",
        "4150",
        "494",
        "486",
        "4366",
        "1370",
        "4361",
        "2473",
        "2012",
        "3855",
        "c_415bab24",
        "119",
        "2188",
        "3716",
        "4365",
        "4826",
        "2179",
        "6015",
        "1455",
        "8380",
        "2170",
        "5286",
        "2190",
        "2172",
        "2167",
        "5285",
        "2171",
        "6157",
        "2191",
        "90",
        "2187",
        "5366",
        "2767",
        "3743",
        "1411",
        "2498",
        "3737",
        "2364",
        "1274",
        "1414",
        "744",
        "1412",
        "5350",
        "3783",
        "753",
        "5576",
        "5051",
        "5620",
        "438855",
        "1005",
        "3258",
        "4067",
        "5640",
        "2163",
        "5130",
        "5767",
        "5769",
        "1451",
        "5758",
        "4304",
        "4057",
        "4060",
        "4040",
        "5442",
        "26",
        "4017",
        "5258",
        "8473",
        "525",
        "4479",
        "3956",
        "3928",
        "3952",
        "6382",
        "3941",
        "3949",
        "3946",
        "5017",
        "4314",
        "2775",
        "239",
        "2763",
        "6788",
        "1360",
        "656",
        "2418",
        "2417",
        "5360",
        "2412",
        "1602",
        "2416",
        "2414",
        "2413",
        "1115",
        "2737",
        "2735",
        "2736",
        "1258",
        "4416",
        "6052",
        "8301",
        "1088",
        "7816",
        "8189",
        "2564",
        "2288",
        "6138",
        "6136",
        "1520",
        "1506",
        "1329",
        "337",
        "7815",
        "c_6acbfd25",
        "1184",
        "6135",
        "7879",
        "4840",
        "7861",
        "7855",
        "7868",
        "7817",
        "7814",
        "864",
        "281",
        "5424",
        "311",
        "3778",
        "1087",
        "2287",
        "1086",
        "911",
        "912",
        "913",
        "909",
        "914",
        "1587",
        "1293",
        "1615",
        "5259",
        "1632",
        "1297",
        "5302",
        "4069",
        "1844",
        "4607",
        "2117",
        "8365",
        "c_2b2e17b1",
        "5283",
        "5776",
        "c_964c9649",
        "4690",
        "6205",
        "c_b2c019c8",
        "7207",
        "1794",
        "4587",
        "8369",
        "187",
        "4845",
        "5083",
        "5087",
        "2148",
        "594",
        "5778",
        "8278",
        "c_c385cfb4",
        "5780",
        "8366",
        "5265",
        "1847",
        "1105",
        "1191",
        "8367",
        "2106",
        "2838",
        "727",
        "5448",
        "3791",
        "2858",
        "3889",
        "956",
        "2113",
        "2705",
        "1765",
        "2848",
        "2119",
        "c_6e703074",
        "122",
        "5775",
        "5434",
        "2200",
        "1658",
        "5435",
        "130",
        "2079",
        "5931",
        "4590",
        "c_789ead37",
        "4483",
        "5437",
        "5655",
        "2616",
        "4256",
        "1800",
        "335",
        "427",
        "84",
        "712",
        "2402",
        "2497",
        "637",
        "4006",
        "3013",
        "3659",
        "4362",
        "56",
        "5848",
        "2515",
        "3058",
        "371",
        "343",
        "2090",
        "2524",
        "3549",
        "3150",
        "3146",
        "6011",
        "833",
        "1707",
        "434743",
        "1158",
        "2470",
        "c_98d1408a",
        "2825",
        "5781",
        "6152",
        "c_28750470",
        "5431",
        "1758",
        "5939",
        "5716",
        "5719",
        "1759",
        "1760",
        "5717",
        "5104",
        "1756",
        "837",
        "5718",
        "5714",
        "1762",
        "1757",
        "7354",
        "c_cbdf29ef",
        "8447",
        "c_9b88f778",
        "6567",
        "5720",
        "5715",
        "1153",
        "4474",
        "838",
        "1304",
        "1859",
        "2923",
        "1852",
        "4455",
        "5970",
        "5907",
        "c_c4afa011",
        "c_31da5694",
        "5764",
        "c_60d3928d",
        "5921",
        "273",
        "3365",
        "4116",
        "5689",
        "2446",
        "6569",
        "c_4523e1dd",
        "1590",
        "4849",
        "1279",
        "1284",
        "465",
        "2318",
        "2785",
        "2773",
        "3593",
        "5821",
        "5487",
        "3597",
        "3594",
        "3595",
        "2790",
        "5785",
        "86",
        "2083",
        "1205",
        "1202",
        "3486",
        "4214",
        "7131",
        "c_6c4d5118",
        "772",
        "1353",
        "6002",
        "1470",
        "1420",
        "4277",
        "2873",
        "1951",
        "2023",
        "2810",
        "4837",
        "2697",
        "3598",
        "3601",
        "4096",
        "1680",
        "5667",
        "439",
        "4097",
        "326",
        "5171",
        "7374",
        "5996",
        "c_827bea7d",
        "4873",
        "3296",
        "5987",
        "c_57946f1a",
        "1508",
        "7397",
        "7387",
        "1454",
        "c_847fc9f2",
        "6561",
        "5030",
        "3297",
        "1536",
        "3904",
        "6009",
        "1244",
        "4101",
        "6560",
        "7371",
        "6563",
        "3200",
        "4089",
        "5322",
        "442",
        "1364",
        "2840",
        "1780",
        "2278",
        "2022",
        "283",
        "6558",
        "4105",
        "6924",
        "102",
        "105",
        "103",
        "4209",
        "3357",
        "2929",
        "4143",
        "826",
        "3361",
        "3358",
        "c_16e35fe6",
        "6233",
        "3360",
        "250",
        "2605",
        "4028",
        "6564",
        "3359",
        "977",
        "1914",
        "1922",
        "2509",
        "1302",
        "6540",
        "1921",
        "724",
        "1912",
        "3322",
        "1948",
        "259",
        "3320",
        "1913",
        "1911",
        "1925",
        "1910",
        "1917",
        "1924",
        "1923",
        "2065",
        "1920",
        "2339",
        "2238",
        "2341",
        "5345",
        "6538",
        "74",
        "2334",
        "2335",
        "3309",
        "3311",
        "7414",
        "719",
        "3315",
        "6536",
        "2099",
        "570",
        "3314",
        "483",
        "4581",
        "117",
        "5296",
        "1818",
        "3317",
        "3316",
        "3312",
        "7352",
        "2239",
        "967",
        "966",
        "5961",
        "6535",
        "4234",
        "5390",
        "2501",
        "3301",
        "5159",
        "6539",
        "6537",
        "3168",
        "861",
        "3281",
        "2391",
        "5173",
        "6033",
        "3124",
        "4167",
        "3376",
        "6035",
        "662",
        "1805",
        "2691",
        "c_d59e7560",
        "847",
        "3377",
        "4443",
        "4618",
        "4442",
        "1456",
        "4850",
        "4058",
        "507",
        "515",
        "1693",
        "5553",
        "1694",
        "1692",
        "556",
        "1695",
        "271",
        "5309",
        "3378",
        "4243",
        "2806",
        "775",
        "2327",
        "3134",
        "4598",
        "4874",
        "1223",
        "1222",
        "1773",
        "2029",
        "589",
        "1991",
        "2092",
        "2484",
        "3460",
        "66",
        "2559",
        "389",
        "2558",
        "3245",
        "3337",
        "1864",
        "3327",
        "5799",
        "4852",
        "3324",
        "405",
        "6544",
        "4231",
        "4343",
        "5314",
        "442958",
        "5325",
        "5634",
        "4374",
        "4728",
        "2823",
        "4475",
        "3352",
        "5800",
        "374",
        "6738",
        "6542",
        "1764",
        "1691",
        "4323",
        "1162",
        "3348",
        "3353",
        "1817",
        "442990",
        "5160",
        "7195",
        "5169",
        "6046",
        "5170",
        "5291",
        "101",
        "432",
        "6549",
        "5382",
        "2270",
        "1990",
        "5323",
        "3250",
        "3329",
        "4702",
        "3334",
        "5903",
        "4470",
        "4634",
        "5166",
        "4245",
        "4564",
        "3179",
        "6552",
        "4086",
        "282",
        "3344",
        "1231",
        "6215",
        "3919",
        "1469",
        "3347",
        "1133",
        "1708",
        "81",
        "3346",
        "580",
        "1340",
        "6823",
        "1886",
        "3837",
        "4084",
        "1883",
        "1885",
        "1884",
        "3824",
        "3831",
        "3828",
        "3827",
        "3826",
        "708",
        "1945",
        "1889",
        "1868",
        "4228",
        "3829",
        "2766",
        "5289",
        "3825",
        "1514",
        "5503",
        "5447",
        "6012",
        "5678",
        "626",
        "1826",
        "4831",
        "154",
        "1830",
        "3880",
        "6916",
        "153",
        "509",
        "1523",
        "3882",
        "1145",
        "510",
        "1512",
        "4211",
        "945",
        "3",
        "6915",
        "2747",
        "28",
        "212",
        "2420",
        "1704",
        "2372",
        "2214",
        "1229",
        "5290",
        "1887",
        "c_b9c60592",
        "4545",
        "1275",
        "5261",
        "4062",
        "1878",
        "2213",
        "51",
        "5035",
        "3817",
        "2740",
        "1143",
        "1396",
        "6914",
        "c_e5d85c14",
        "1387",
        "2739",
        "3808",
        "2716",
        "3861",
        "3859",
        "1833",
        "1828",
        "3864",
        "3865",
        "3876",
        "849",
        "2192",
        "3874",
        "50",
        "1720",
        "2726",
        "1416",
        "2706",
        "5245",
        "1391",
        "2745",
        "5646",
        "1458",
        "2755",
        "2205",
        "3897",
        "1384",
        "2774",
        "1393",
        "1867",
        "1375",
        "2754",
        "6825",
        "3842",
        "1403",
        "1413",
        "2159",
        "2169",
        "1357",
        "1358",
        "5243",
        "1361",
        "1846",
        "3841",
        "1371",
        "3840",
        "155",
        "4584",
        "2378",
        "3759",
        "5440",
        "5397",
        "2549",
        "1077",
        "3846",
        "4579",
        "3851",
        "1832",
        "1994",
        "1724",
        "1723",
        "1725",
        "2758",
        "2770",
        "4224",
        "3627",
        "451069",
        "4264",
        "3290",
        "3707",
        "c_804a9afe",
        "3224",
        "3617",
        "429",
        "2802",
        "3624",
        "2677",
        "8450",
        "1957",
        "1009",
        "1356",
        "1434",
        "2405",
        "1435",
        "1407",
        "1408",
        "3619",
        "3622",
        "1402",
        "1377",
        "1376",
        "5024",
        "1726",
        "3890",
        "2783",
        "4377",
        "1397",
        "4169",
        "1368",
        "3887",
        "315",
        "3886",
        "3706",
        "3477",
        "3590",
        "165",
        "3567",
        "3599",
        "3630",
        "3274",
        "3723",
        "3243",
        "3244",
        "3373",
        "3187",
        "549",
        "34",
        "3222",
        "3673",
        "2906",
        "3242",
        "c_dcf3f7c0",
        "1485",
        "1790",
        "c_3e6af2e7",
        "c_8f89faac",
        "2150",
        "4491",
        "c_834b57c4",
        "3233",
        "4646",
        "2787",
        "728",
        "1459",
        "4263",
        "4187",
        "5156",
        "2002",
        "2502",
        "560",
        "1003",
        "1007",
        "6025",
        "1001",
        "6714",
        "5154",
        "1014",
        "1002",
        "c_10aa91c7",
        "c_39d242fc",
        "2466",
        "1492",
        "1489",
        "1490",
        "1491",
        "1497",
        "1496",
        "1495",
        "2463",
        "1494",
        "1072",
        "233",
        "1354",
        "1025",
        "2456",
        "1348",
        "5155",
        "460",
        "4235",
        "3152",
        "4248",
        "188",
        "2672",
        "2662",
        "2644",
        "2659",
        "2676",
        "2682",
        "3270",
        "2649",
        "2650",
        "2657",
        "2661",
        "2645",
        "2651",
        "2643",
        "4146",
        "4623",
        "2635",
        "2668",
        "2634",
        "2653",
        "2636",
        "2642",
        "2640",
        "2664",
        "2648",
        "3272",
        "2652",
        "2638",
        "2685",
        "2673",
        "3208",
        "2689",
        "2633",
        "2674",
        "2666",
        "2665",
        "2670",
        "3268",
        "2687",
        "2681",
        "4329",
        "2660",
        "2683",
        "2679",
        "4301",
        "933",
        "1675",
        "296",
        "3253",
        "341",
        "3350",
        "173",
        "189",
        "763",
        "402",
        "398",
        "397",
        "403",
        "5059",
        "5061",
        "875",
        "917",
        "3182",
        "144",
        "6028",
        "2574",
        "2615",
        "3558",
        "368",
        "4383",
        "3554",
        "4282",
        "6340",
        "843",
        "2403",
        "5062",
        "5972",
        "5050",
        "4241",
        "1824",
        "4569",
        "4132",
        "3119",
        "3256",
        "5052",
        "3254",
        "58",
        "2319",
        "2067",
        "1017",
        "5054",
        "137",
        "740",
        "1682",
        "5021",
        "729",
        "178",
        "4763",
        "1075",
        "c_2d0e694e",
        "c_6cfce4be",
        "5797",
        "c_c565cff5",
        "5795",
        "5883",
        "5474",
        "5555",
        "204",
        "1799",
        "6335",
        "4390",
        "182",
        "3227",
        "1676",
        "1969",
        "2365",
        "904",
        "1825",
        "c_20fde7af",
        "1322",
        "530",
        "1324",
        "1328",
        "1323",
        "1314",
        "1312",
        "1327",
        "4406",
        "1330",
        "1331",
        "4585",
        "4409",
        "4401",
        "1015",
        "4399",
        "4400",
        "4407",
        "6343",
        "504",
        "3298",
        "4279",
        "935",
        "c_c586f36c",
        "1985",
        "3261",
        "365",
        "3262",
        "6736",
        "561",
        "1234",
        "6342",
        "57",
        "6029",
        "5566",
        "170",
        "924",
        "3685",
        "2230",
        "3264",
        "1316",
        "1315",
        "1320",
        "1319",
        "1313",
        "1317",
        "6713",
        "6339",
        "290",
        "3229",
        "286",
        "2606",
        "295",
        "293",
        "c_896e199b",
        "292",
        "297",
        "289",
        "298",
        "8432",
        "736",
        "4118",
        "3246",
        "872",
        "4738",
        "4195",
        "278",
        "1130",
        "3230",
        "858",
        "2959",
        "3231",
        "3251",
        "4356",
        "2151",
        "2371",
        "3252",
        "408",
        "3612",
        "3994",
        "521",
        "3248",
        "2447",
        "1058",
        "6055",
        "5729",
        "973",
        "3449",
        "3225",
        "1677",
        "739",
        "3186",
        "2219",
        "423",
        "4371",
        "665",
        "6333",
        "4644",
        "4633",
        "4661",
        "4798",
        "1466",
        "4344",
        "4608",
        "219",
        "2069",
        "2387",
        "4078",
        "4405",
        "4793",
        "502",
        "4079",
        "491",
        "1069",
        "5583",
        "3948",
        "4645",
        "3178",
        "4381",
        "4402",
        "4499",
        "1637",
        "3174",
        "5843",
        "4600",
        "4836",
        "4789",
        "1232",
        "4333",
        "3611",
        "3176",
        "1091",
        "4350",
        "161",
        "932",
        "4385",
        "2957",
        "2050",
        "4885",
        "4025",
        "2317",
        "4186",
        "946",
        "4748",
        "613",
        "622",
        "1309",
        "3189",
        "4643",
        "4573",
        "c_34465dac",
        "c_163e1e96",
        "480",
        "3166",
        "1010",
        "5542",
        "2912",
        "1486",
        "4",
        "3123",
        "635",
        "2804",
        "5707",
        "3172",
        "9",
        "10",
        "692",
        "6364",
        "2590",
        "3165",
        "3170",
        "2182",
        "2967",
        "927",
        "619",
        "4605",
        "618",
        "3171",
        "4073",
        "1807",
        "1795",
        "1803",
        "1806",
        "1801",
        "4219",
        "20",
        "1797",
        "5725",
        "2218",
        "1796",
        "6365",
        "1798",
        "6366",
        "522",
        "6062",
        "3570",
        "3167",
        "6268",
        "288",
        "3271",
        "2742",
        "720",
        "2589",
        "893",
        "2756",
        "722",
        "4663",
        "5710",
        "1688",
        "2602",
        "4983",
        "2862",
        "3173",
        "3204",
        "2803",
        "2081",
        "895",
        "1423",
        "6371",
        "3205",
        "1562",
        "4104",
        "3207",
        "5451",
        "2786",
        "3218",
        "1078",
        "5",
        "6010",
        "1055",
        "2045",
        "1684",
        "6",
        "5140",
        "4280",
        "47",
        "6216",
        "2836",
        "4880",
        "4878",
        "3195",
        "c_a62dbeba",
        "5317",
        "4099",
        "3198",
        "4879",
        "3197",
        "140",
        "4198",
        "147",
        "143",
        "146",
        "3202",
        "377",
        "393",
        "396",
        "1135",
        "3201",
        "4208",
        "145",
        "809",
        "3579",
        "810",
        "3577",
        "6081",
        "2771",
        "8",
        "2748",
        "621",
        "2781",
        "4130",
        "1616",
        "6367",
        "1337",
        "5151",
        "620",
        "482",
        "1567",
        "2851",
        "4524",
        "4828",
        "2452",
        "2275",
        "6173",
        "4817",
        "1791",
        "2864",
        "6013",
        "2216",
        "957",
        "4675",
        "1333",
        "5092",
        "5964",
        "55",
        "60",
        "61",
        "3602",
        "3446",
        "75",
        "1635",
        "435330",
        "805",
        "804",
        "803",
        "3006",
        "4574",
        "806",
        "1592",
        "183",
        "1591",
        "3145",
        "2696",
        "5541",
        "1979",
        "2565",
        "2646",
        "2863",
        "2998",
        "4242",
        "2993",
        "2100",
        "588",
        "6134",
        "2135",
        "2134",
        "174",
        "c_03f9a8ac",
        "2165",
        "4360",
        "1165",
        "5480",
        "1782",
        "4855",
        "1811",
        "3443",
        "2629",
        "1161",
        "1156",
        "6064",
        "c_406ad4cc",
        "7926",
        "c_5b447e3a",
        "2792",
        "6910",
        "4286",
        "814",
        "830",
        "829",
        "3005",
        "820",
        "819",
        "824",
        "6754",
        "828",
        "816",
        "821",
        "815",
        "822",
        "5206",
        "827",
        "817",
        "823",
        "2359",
        "1178",
        "1247",
        "4193",
        "227",
        "4202",
        "874",
        "4201",
        "4191",
        "4194",
        "1200",
        "3007",
        "3060",
        "4204",
        "6749",
        "6748",
        "6174",
        "6753",
        "5374",
        "1672",
        "5372",
        "2889",
        "150",
        "7201",
        "115",
        "3121",
        "3543",
        "447",
        "871",
        "3542",
        "3004",
        "1122",
        "2129",
        "110",
        "1193",
        "2264",
        "c_5f90006e",
        "7948",
        "6752",
        "45",
        "1956",
        "7942",
        "4492",
        "7956",
        "c_5a7a0d82",
        "6751",
        "5205",
        "1307",
        "644",
        "215",
        "733",
        "3583",
        "5081",
        "2831",
        "7",
        "1365",
        "2303",
        "4292",
        "3452",
        "3458",
        "5610",
        "5786",
        "3439",
        "3442",
        "5591",
        "3437",
        "5783",
        "3441",
        "7809",
        "3440",
        "3436",
        "4127",
        "2637",
        "5321",
        "3431",
        "3420",
        "1944",
        "4417",
        "2567",
        "6363",
        "3429",
        "3432",
        "7987",
        "c_9104c45f",
        "4156",
        "4822",
        "2905",
        "6380",
        "3447",
        "6379",
        "2499",
        "472",
        "5053",
        "3448",
        "7340",
        "2005",
        "3488",
        "4596",
        "1347",
        "359",
        "5121",
        "1345",
        "4216",
        "3463",
        "3466",
        "3465",
        "5420",
        "2901",
        "5870",
        "5537",
        "2095",
        "1379",
        "5833",
        "6007",
        "2928",
        "2850",
        "5574",
        "3462",
        "1663",
        "3219",
        "2087",
        "3223",
        "3480",
        "5744",
        "3241",
        "3479",
        "3228",
        "3196",
        "c_39046afb",
        "3870",
        "3481",
        "5124",
        "3650",
        "3206",
        "3483",
        "435",
        "3468",
        "3473",
        "3482",
        "3470",
        "5486",
        "2926",
        "5638",
        "6704",
        "6372",
        "5449",
        "5126",
        "c_99e6dc30",
        "4275",
        "4270",
        "3500",
        "2986",
        "5831",
        "8355",
        "3502",
        "8357",
        "3501",
        "c_e78f03db",
        "8354",
        "4269",
        "8356",
        "3499",
        "c_7d7608fa",
        "c_ff638149",
        "2053",
        "1306",
        "1517",
        "2071",
        "3498",
        "406",
        "1869",
        "4696",
        "1743",
        "5773",
        "2485",
        "6024",
        "c_959c23de",
        "3496",
        "3112",
        "6117",
        "1930",
        "5753",
        "5957",
        "3505",
        "3506",
        "3504",
        "3453",
        "6927",
        "411",
        "440",
        "6375",
        "5335",
        "c_47ea173d",
        "498",
        "3495",
        "c_54f7dd1f",
        "1214",
        "c_7cbc24fd",
        "1436",
        "541",
        "1210",
        "3493",
        "3076",
        "5125",
        "3075",
        "c_f7430876",
        "c_50620749",
        "807",
        "3077",
        "899",
        "737",
        "4055",
        "881",
        "5337",
        "949",
        "845",
        "969",
        "922",
        "3003",
        "910",
        "986",
        "500",
        "2909",
        "4221",
        "207",
        "6373",
        "224",
        "210",
        "208",
        "226",
        "5551",
        "220",
        "5546",
        "2285",
        "213",
        "435164",
        "6917",
        "2300",
        "223",
        "c_8f1cd55b",
        "5545",
        "4121",
        "217",
        "4123",
        "216",
        "221",
        "2298",
        "6920",
        "218",
        "211",
        "209",
        "5779",
        "3491",
        "225",
        "4225",
        "1763",
        "324",
        "2625",
        "2131",
        "1031",
        "3409",
        "322",
        "330",
        "2396",
        "7133",
        "323",
        "6321",
        "2025",
        "320",
        "5792",
        "4734",
        "329",
        "6320",
        "318",
        "1260",
        "5462",
        "1755",
        "1856",
        "768",
        "4095",
        "4251",
        "3408",
        "2115",
        "715",
        "721",
        "4427",
        "6322",
        "713",
        "1224",
        "4163",
        "1898",
        "1585",
        "1167",
        "1937",
        "1929",
        "4743",
        "1919",
        "2989",
        "2511",
        "8459",
        "c_8b0ac3e2",
        "2173",
        "2971",
        "2969",
        "2444",
        "2965",
        "1595",
        "2970",
        "2445",
        "6100",
        "2915",
        "4117",
        "638",
        "6311",
        "6308",
        "1849",
        "579",
        "1788",
        "1598",
        "7204",
        "2404",
        "6328",
        "6323",
        "6329",
        "5316",
        "1064",
        "1066",
        "8437",
        "1068",
        "1067",
        "1065",
        "2992",
        "2994",
        "943",
        "6331",
        "563",
        "2995",
        "4352",
        "5219",
        "4336",
        "2920",
        "6326",
        "1513",
        "2893",
        "926",
        "2194",
        "2975",
        "2977",
        "2974",
        "2892",
        "4404",
        "2622",
        "2624",
        "2068",
        "3576",
        "1686",
        "1201",
        "2976",
        "1144",
        "2938",
        "5096",
        "1958",
        "4298",
        "4771",
        "4312",
        "3265",
        "5373",
        "1342",
        "222",
        "319",
        "6317",
        "2419",
        "568",
        "2544",
        "327",
        "328",
        "636",
        "4142",
        "6318",
        "624",
        "2536",
        "2596",
        "370",
        "4490",
        "1521",
        "2842",
        "1483",
        "2925",
        "2410",
        "832",
        "325",
        "4444",
        "4338",
        "4708",
        "4450",
        "4448",
        "4445",
        "4447",
        "4674",
        "152",
        "764",
        "2409",
        "4081",
        "846",
        "c_39cda9ba",
        "2985",
        "2560",
        "842",
        "2211",
        "5034",
        "5252",
        "6918",
        "4612",
        "2718",
        "2310",
        "c_cb1ce6ff",
        "4260",
        "2307",
        "2311",
        "2384",
        "3000",
        "2999",
        "714",
        "2435",
        "313",
        "2395",
        "1961",
        "2411",
        "2305",
        "2308",
        "2561",
        "1035",
        "2921",
        "4790",
        "1159",
        "2879",
        "2556",
        "5255",
        "2455",
        "c_8702d5f7",
        "5228",
        "3679",
        "3682",
        "3424",
        "2725",
        "4257",
        "5906",
        "936",
        "1738",
        "2981",
        "2980",
        "1525",
        "577",
        "6314",
        "2983",
        "939",
        "559",
        "942",
        "615",
        "1947",
        "2655",
        "2978",
        "3009",
        "4811",
        "5226",
        "983",
        "944",
        "4337",
        "938",
        "586",
        "963",
        "1890",
        "964",
        "982",
        "3634",
        "2721",
        "2713",
        "2722",
        "4285",
        "6292",
        "1465",
        "6294",
        "4863",
        "c_b18ab65c",
        "479",
        "4239",
        "2399",
        "2177",
        "5147",
        "3304",
        "3633",
        "5123",
        "c_ab46334d",
        "529",
        "2816",
        "5238",
        "1487",
        "3669",
        "6302",
        "3672",
        "1439",
        "1894",
        "3670",
        "2922",
        "c_de4d6f2f",
        "6305",
        "3668",
        "3671",
        "6306",
        "2916",
        "3667",
        "2924",
        "5237",
        "c_fea6771b",
        "5236",
        "3657",
        "3654",
        "3660",
        "3586",
        "2795",
        "c_7bc27da2",
        "3652",
        "3655",
        "2797",
        "3656",
        "c_99a79cea",
        "2801",
        "4890",
        "4930",
        "3795",
        "3796",
        "4111",
        "3800",
        "3801",
        "6361",
        "3640",
        "3803",
        "2076",
        "3645",
        "3649",
        "4082",
        "866",
        "3997",
        "71",
        "179",
        "1848",
        "4253",
        "3744",
        "180",
        "3695",
        "3694",
        "3691",
        "3696",
        "3698",
        "3697",
        "3692",
        "3693",
        "6299",
        "3909",
        "3651",
        "923",
        "3636",
        "7219",
        "5383",
        "5928",
        "5973",
        "4496",
        "7931",
        "6362",
        "3108",
        "3117",
        "3110",
        "1645",
        "4464",
        "3111",
        "4318",
        "3115",
        "3116",
        "2457",
        "1433",
        "5295",
        "5908",
        "555",
        "3109",
        "4536",
        "2400",
        "5212",
        "6850",
        "648",
        "4033",
        "4379",
        "4387",
        "4805",
        "5324",
        "3100",
        "6020",
        "2325",
        "3143",
        "4162",
        "172",
        "6021",
        "3107",
        "4408",
        "195",
        "c_c8c80a5b",
        "2039",
        "c_ea3f5ed2",
        "5471",
        "3125",
        "1033",
        "5239",
        "5216",
        "5677",
        "3093",
        "4527",
        "4510",
        "4528",
        "4537",
        "4506",
        "4516",
        "4530",
        "6851",
        "2350",
        "5152",
        "3428",
        "4529",
        "4152",
        "818",
        "4535",
        "3114",
        "3113",
        "c_4c441ea0",
        "3120",
        "4531",
        "4511",
        "4509",
        "4541",
        "3097",
        "6848",
        "4651",
        "4526",
        "4532",
        "3614",
        "4517",
        "4518",
        "4508",
        "4519",
        "4521",
        "4540",
        "4514",
        "4533",
        "4827",
        "569",
        "1037",
        "5211",
        "3130",
        "5077",
        "2041",
        "3129",
        "2513",
        "3105",
        "6911",
        "1447",
        "2040",
        "1036",
        "1032",
        "3435",
        "5210",
        "2571",
        "896",
        "4550",
        "1503",
        "5208",
        "592",
        "4546",
        "1653",
        "1926",
        "c_e4f135ba",
        "4039",
        "c_2efea99f",
        "855",
        "3574",
        "4252",
        "2161",
        "238",
        "3566",
        "2082",
        "533",
        "6022",
        "4000",
        "3564",
        "4887",
        "1443",
        "194",
        "5587",
        "2046",
        "98",
        "3572",
        "2266",
        "3374",
        "5982",
        "5975",
        "6344",
        "5983",
        "5981",
        "3515",
        "5980",
        "5979",
        "5976",
        "5977",
        "5978",
        "2284",
        "3534",
        "104",
        "7212",
        "c_090e8e94",
        "3532",
        "6345",
        "7191",
        "1073",
        "7209",
        "7210",
        "6008",
        "4552",
        "5940",
        "3529",
        "3530",
        "4549",
        "3525",
        "506",
        "4547",
        "2904",
        "3519",
        "2427",
        "4029",
        "4700",
        "8549",
        "731",
        "3523",
        "2330",
        "1742",
        "3518",
        "2454",
        "545",
        "284",
        "1646",
        "5329",
        "4332",
        "3626",
        "3623",
        "2267",
        "4090",
        "4273",
        "2342",
        "2943",
        "3562",
        "5475",
        "166",
        "414",
        "3559",
        "3849",
        "3850",
        "3555",
        "3653",
        "687",
        "1866",
        "1026",
        "3553",
        "3556",
        "3688",
        "3557",
        "3664",
        "3843",
        "2619",
        "481",
        "732",
        "5080",
        "2207",
        "4563",
        "6745",
        "3540",
        "c_95b02746",
        "4555",
        "4557",
        "6354",
        "c_9eb9cbaa",
        "4562",
        "4559",
        "2868",
        "2555",
        "2550",
        "1529",
        "4556",
        "3539",
        "1739",
        "1740",
        "4561",
        "4558",
        "735",
        "4054",
        "5207",
        "372",
        "3537",
        "1946",
        "5632",
        "6348",
        "4113",
        "4851",
        "2593",
        "3325",
        "5755",
        "c_a9a17fe5",
        "2126",
        "6350",
        "376",
        "6717",
        "3854",
        "3545",
        "2788",
        "2784",
        "1011",
        "1928",
        "149",
        "2422",
        "5069",
        "6014",
        "244",
        "3964",
        "7343",
        "6888",
        "2261",
        "2259",
        "6886",
        "7378",
        "2255",
        "7377",
        "2262",
        "2257",
        "3050",
        "2254",
        "2253",
        "2260",
        "2251",
        "2250",
        "2252",
        "7376",
        "955",
        "954",
        "1137",
        "582",
        "5357",
        "190",
        "1599",
        "5867",
        "5549",
        "7337",
        "2245",
        "2244",
        "5068",
        "18",
        "5379",
        "1197",
        "2072",
        "7952",
        "3040",
        "1303",
        "1336",
        "3680",
        "1344",
        "1778",
        "3049",
        "4328",
        "7217",
        "4120",
        "362",
        "2968",
        "1046",
        "340",
        "5450",
        "3088",
        "4272",
        "361",
        "206",
        "4136",
        "6894",
        "68",
        "442917",
        "5118",
        "8466",
        "5120",
        "38",
        "7934",
        "5945",
        "5852",
        "c_46f6aaeb",
        "878",
        "c_9829e6b7",
        "390",
        "2296",
        "7353",
        "65",
        "5119",
        "5502",
        "5501",
        "2854",
        "5657",
        "27",
        "391",
        "4451",
        "3031",
        "1172",
        "39",
        "6897",
        "4011",
        "7365",
        "2700",
        "3862",
        "5536",
        "2695",
        "3038",
        "5217",
        "4012",
        "35",
        "1438",
        "3703",
        "2861",
        "1558",
        "4449",
        "6892",
        "4119",
        "4754",
        "4762",
        "1062",
        "4091",
        "3984",
        "4147",
        "5851",
        "3313",
        "4159",
        "1750",
        "1753",
        "1746",
        "4593",
        "697",
        "700",
        "704",
        "5607",
        "699",
        "696",
        "701",
        "5067",
        "4759",
        "4760",
        "4761",
        "3035",
        "7215",
        "5064",
        "1471",
        "360",
        "5984",
        "4766",
        "2268",
        "4765",
        "5105",
        "363",
        "706",
        "5097",
        "4322",
        "702",
        "2966",
        "6890",
        "4368",
        "3044",
        "698",
        "3982",
        "3977",
        "3981",
        "3383",
        "6882",
        "6881",
        "5622",
        "5626",
        "6885",
        "5621",
        "887",
        "5625",
        "5623",
        "7197",
        "4093",
        "381",
        "1588",
        "1601",
        "7216",
        "1829",
        "907",
        "3096",
        "1967",
        "7413",
        "2033",
        "1478",
        "4188",
        "971",
        "c_3a2df4eb",
        "1785",
        "5875",
        "1138",
        "2030",
        "1823",
        "6862",
        "1479",
        "135",
        "633",
        "1614",
        "435136",
        "32",
        "905",
        "3048",
        "2583",
        "3046",
        "5114",
        "2576",
        "2578",
        "2581",
        "2582",
        "2580",
        "2579",
        "2577",
        "382",
        "5483",
        "6883",
        "3002",
        "7386",
        "1977",
        "3016",
        "2020",
        "4950",
        "3020",
        "1174",
        "3413",
        "1966",
        "1173",
        "436",
        "2263",
        "3042",
        "2588",
        "5805",
        "6884",
        "5669",
        "5891",
        "2009",
        "1999",
        "1751",
        "1931",
        "2000",
        "2007",
        "1198",
        "419",
        "2154",
        "7373",
        "2003",
        "2011",
        "2073",
        "7416",
        "470",
        "3056",
        "8428",
        "2010",
        "2004",
        "4153",
        "4729",
        "c_57f3c49f",
        "4730",
        "2162",
        "4087",
        "c_2f00dd5a",
        "c_af502da0",
        "3057",
        "4180",
        "3561",
        "4182",
        "3226",
        "5313",
        "7184",
        "4181",
        "1749",
        "1998",
        "3517",
        "49",
        "5897",
        "1192",
        "6893",
        "7189",
        "309",
        "4959",
        "c_9eea2203",
        "7385",
        "3589",
        "8419",
        "3379",
        "7187",
        "5841",
        "2155",
        "3054",
        "1437",
        "3237",
        "67",
        "6017",
        "7370",
        "7390",
        "7399",
        "3922",
        "578",
        "527",
        "576",
        "1484",
        "7350",
        "6729",
        "6611",
        "6257",
        "1589",
        "c_7afb6cd4",
        "1550",
        "6609",
        "528",
        "148",
        "5813",
        "6023",
        "485",
        "5836",
        "5415",
        "5414",
        "5835",
        "c_0f3b8370",
        "7929",
        "435469",
        "5817",
        "2913",
        "2704",
        "2699",
        "3974",
        "6617",
        "3966",
        "c_6496f5ea",
        "6619",
        "6618",
        "5809",
        "8458",
        "930",
        "5954",
        "1080",
        "5616",
        "3969",
        "7200",
        "5596",
        "2819",
        "3936",
        "2031",
        "2818",
        "234",
        "948",
        "4503",
        "3938",
        "3937",
        "4625",
        "546",
        "2249",
        "2813",
        "5093",
        "3932",
        "2812",
        "2815",
        "3497",
        "186",
        "96",
        "4274",
        "164",
        "5598",
        "2833",
        "564",
        "3902",
        "3915",
        "1626",
        "520",
        "3917",
        "2828",
        "857",
        "7348",
        "3916",
        "c_ba4acdb2",
        "585",
        "3911",
        "3906",
        "1620",
        "1636",
        "1628",
        "1619",
        "1625",
        "583",
        "543",
        "1621",
        "1618",
        "455",
        "454",
        "458",
        "3912",
        "456",
        "5592",
        "3910",
        "459",
        "2834",
        "6254",
        "6230",
        "6231",
        "457",
        "688",
        "516",
        "6628",
        "3914",
        "7411",
        "689",
        "512",
        "513",
        "3913",
        "514",
        "3565",
        "435196",
        "4009",
        "8457",
        "3503",
        "518",
        "4334",
        "8456",
        "542",
        "8411",
        "1633",
        "551",
        "581",
        "4238",
        "537",
        "6923",
        "2034",
        "552",
        "2035",
        "3521",
        "488",
        "7944",
        "8417",
        "245",
        "5571",
        "574",
        "8420",
        "3355",
        "6612",
        "3943",
        "3944",
        "8449",
        "3993",
        "5489",
        "6615",
        "6614",
        "5142",
        "5490",
        "4695",
        "5631",
        "2314",
        "3959",
        "5271",
        "3958",
        "2631",
        "3957",
        "3588",
        "1549",
        "3953",
        "3962",
        "3963",
        "2353",
        "3955",
        "877",
        "3961",
        "575",
        "7349",
        "1537",
        "3990",
        "2441",
        "2124",
        "1735",
        "1938",
        "5141",
        "1543",
        "2982",
        "1539",
        "5692",
        "42",
        "1541",
        "5691",
        "6623",
        "6621",
        "c_bf1e2c81",
        "6140",
        "c_5ea6e5c4",
        "5351",
        "5196",
        "1367",
        "453",
        "3422",
        "3404",
        "156",
        "3585",
        "4330",
        "616",
        "4359",
        "5422",
        "4220",
        "1482",
        "3030",
        "5200",
        "6909",
        "4441",
        "6120",
        "4501",
        "4478",
        "4072",
        "671",
        "6763",
        "5905",
        "5178",
        "269",
        "5177",
        "1597",
        "280",
        "590",
        "3968",
        "5176",
        "5175",
        "1419",
        "854",
        "2397",
        "384",
        "2932",
        "674",
        "673",
        "677",
        "675",
        "669",
        "2313",
        "781",
        "676",
        "779",
        "3279",
        "c_243b7be2",
        "1784",
        "4534",
        "7341",
        "1098",
        "1444",
        "1081",
        "1950",
        "7388",
        "3216",
        "3280",
        "48",
        "1380",
        "1927",
        "5266",
        "4357",
        "2794",
        "2196",
        "684",
        "5094",
        "2973",
        "2195",
        "373",
        "3615",
        "5102",
        "3606",
        "184",
        "446",
        "6037",
        "925",
        "3177",
        "6261",
        "3607",
        "6275",
        "6908",
        "6262",
        "5333",
        "4430",
        "3026",
        "6019",
        "5181",
        "6030",
        "3024",
        "4821",
        "5334",
        "5595",
        "773",
        "1696",
        "5613",
        "3027",
        "5949",
        "c_04ae3ba8",
        "5684",
        "5355",
        "5441",
        "5862",
        "628",
        "631",
        "5687",
        "629",
        "627",
        "782",
        "789",
        "794",
        "787",
        "796",
        "785",
        "679",
        "793",
        "799",
        "792",
        "800",
        "1071",
        "790",
        "801",
        "1070",
        "780",
        "3277",
        "788",
        "682",
        "3276",
        "5603",
        "678",
        "783",
        "672",
        "798",
        "670",
        "680",
        "681",
        "791",
        "2918",
        "1079",
        "c_40f54e0c",
        "6188",
        "774",
        "441",
        "1752",
        "5109",
        "8470",
        "4295",
        "438",
        "4456",
        "2859",
        "1076",
        "4137",
        "5627",
        "2891",
        "5311",
        "5628",
        "7364",
        "1902",
        "1962",
        "6765",
        "7368",
        "4144",
        "1681",
        "7363",
        "487",
        "5277",
        "7379",
        "5665",
        "4905",
        "c_cdec6719",
        "6767",
        "3930",
        "5340",
        "c_25ddd844",
        "1978",
        "452",
        "4168",
        "492",
        "7369",
        "4139",
        "6766",
        "242",
        "31",
        "5356",
        "4626",
        "5066",
        "2492",
        "451",
        "6769",
        "7398",
        "4865",
        "1705",
        "1629",
        "1385",
        "1366",
        "3580",
        "1392",
        "5184",
        "667",
        "1363",
        "5188",
        "2566",
        "c_b12a760a",
        "6088",
        "4424",
        "3255",
        "5793",
        "8418",
        "5191",
        "175",
        "29",
        "2013",
        "2185",
        "3873",
        "2176",
        "3856",
        "2125",
        "3869",
        "c_a02d5941",
        "2392",
        "c_764bcd4e",
        "25",
        "5765",
        "6163",
        "3675",
        "2026",
        "5688",
        "c_d3334ae3",
        "5495",
        "2199",
        "2105",
        "5479",
        "7193",
        "5991",
        "1194",
        "6129",
        "c_9fe65404",
        "4604",
        "2147",
        "1034",
        "3514",
        "5044",
        "2070",
        "5086",
        "7202",
        "5656",
        "c_7bf10a38",
        "6114",
        "1045",
        "3710",
        "6083",
        "2432",
        "6016",
        "1022",
        "7933",
        "2791",
        "5250",
        "8393",
        "2431",
        "4614",
        "4002",
        "258",
        "3709",
        "2885",
        "124",
        "123",
        "8384",
        "2145",
        "1043",
        "2093",
        "2144",
        "2142",
        "2143",
        "968",
        "4616",
        "4595",
        "4591",
        "992",
        "7203",
        "2130",
        "450983",
        "4812",
        "2141",
        "5249",
        "4594",
        "8381",
        "1507",
        "8382",
        "4611",
        "8383",
        "2118",
        "c_acf7832d",
        "5264",
        "6338",
        "7220",
        "5430",
        "7415",
        "4064",
        "2116",
        "3676",
        "7211",
        "6116",
        "c_5bad9438",
        "5738",
        "4872",
        "1094",
        "1190",
        "c_ce38fff4",
        "3725",
        "8402",
        "4615",
        "2094",
        "6336",
        "1838",
        "5279",
        "5446",
        "120",
        "8398",
        "2103",
        "2202",
        "4586",
        "5107",
        "6159",
        "8399",
        "5341",
        "4744",
        "8400",
        "8403",
        "3722",
        "2102",
        "5530",
        "2137",
        "2146",
        "8370",
        "1657",
        "987",
        "1044",
        "1664",
        "8404",
        "312",
        "5282",
        "c_2f64fd66",
        "991",
        "1510",
        "3853",
        "125",
        "6207",
        "906",
        "5365",
        "3720",
        "2183",
        "8488",
        "6091",
        "5557",
        "5460",
        "5248",
        "6160",
        "7186",
        "3718",
        "4597",
        "3715",
        "4588",
        "4859",
        "2123",
        "7185",
        "121",
        "129",
        "c_90ad7755",
        "5284",
        "176",
        "421",
        "2829",
        "8396",
        "5679",
        "450898",
        "4844",
        "2381",
        "1935",
        "3756",
        "3757",
        "3754",
        "3746",
        "5386",
        "5387",
        "3745",
        "5260",
        "4489",
        "3750",
        "3753",
        "3747",
        "2821",
        "3749",
        "5861",
        "6697",
        "c_22f8c0e6",
        "4631",
        "2917",
        "3785",
        "642",
        "534",
        "4507",
        "6044",
        "4293",
        "308",
        "c_d8d35f08",
        "475",
        "3735",
        "6079",
        "1685",
        "4886",
        "5201",
        "3736",
        "746",
        "203",
        "4420",
        "6043",
        "3741",
        "2772",
        "1141",
        "2366",
        "3734",
        "3733",
        "3764",
        "3765",
        "1061",
        "3766",
        "3767",
        "4423",
        "6702",
        "c_eb46b396",
        "749",
        "4900",
        "750",
        "756",
        "757",
        "748",
        "4892",
        "754",
        "4042",
        "c_e0e4608a",
        "4422",
        "3731",
        "2903",
        "3782",
        "751",
        "3786",
        "747",
        "6047",
        "1840",
        "5577",
        "5579",
        "400",
        "401",
        "5057",
        "5452",
        "5825",
        "5590",
        "5158",
        "5554",
        "c_34746c6e",
        "5844",
        "1460",
        "3560",
        "2592",
        "3259",
        "1311",
        "5854",
        "8465",
        "5937",
        "5519",
        "5421",
        "7927",
        "2140",
        "447472",
        "5508",
        "6054",
        "4838",
        "5645",
        "5680",
        "4071",
        "5251",
        "1048",
        "c_871b5612",
        "3717",
        "7938",
        "6128",
        "8284",
        "2164",
        "5426",
        "2153",
        "c_047e5912",
        "6049",
        "441001",
        "448275",
        "c_edee9606",
        "6219",
        "7126",
        "6072",
        "447917",
        "c_f5ed5adb",
        "7985",
        "7937",
        "6076",
        "7928",
        "7221",
        "5310",
        "5354",
        "447756",
        "5641",
        "6256",
        "7936",
        "c_8a658bb0",
        "447958",
        "5427",
        "5757",
        "7222",
        "8439",
        "c_4768a12e",
        "7983",
        "2198",
        "5425",
        "5642",
        "447795",
        "447660",
        "c_96124aaf",
        "6255",
        "c_bbc13d07",
        "4046",
        "5342",
        "7984",
        "4038",
        "2243",
        "128",
        "256",
        "255",
        "5455",
        "114",
        "4177",
        "5762",
        "4175",
        "5885",
        "c_8a6f744c",
        "2324",
        "12",
        "4063",
        "6222",
        "5544",
        "5547",
        "5788",
        "1474",
        "442821",
        "5763",
        "5873",
        "5409",
        "5407",
        "7932",
        "4183",
        "5629",
        "5499",
        "5948",
        "3234",
        "5418",
        "1464",
        "4070",
        "3299",
        "c_59f9c651",
        "6274",
        "5344",
        "5343",
        "5327",
        "40",
        "4125",
        "4030",
        "43",
        "36",
        "5127",
        "7930",
        "2927",
        "3102",
        "440549",
        "440523",
        "449",
        "3484",
        "4014",
        "4015",
        "4020",
        "4022",
        "6284",
        "349",
        "4457",
        "8474",
        "7194",
        "4066",
        "4065",
        "5606",
        "5534",
        "5498",
        "5411",
        "2897",
        "4018",
        "1551",
        "2247",
        "4482",
        "c_777eba81",
        "4032",
        "5871",
        "8553",
        "4068",
        "443137",
        "4035",
        "7127",
        "4481",
        "4480",
        "6142",
        "4031",
        "6725",
        "4034",
        "6726",
        "2369",
        "1245",
        "4205",
        "892",
        "3923",
        "2453",
        "1670",
        "3933",
        "3947",
        "307",
        "4094",
        "2856",
        "1151",
        "c_a935cf3f",
        "2114",
        "1278",
        "1148",
        "6715",
        "3931",
        "6390",
        "1892",
        "6383",
        "3790",
        "632",
        "2367",
        "4921",
        "2232",
        "685",
        "710",
        "1896",
        "3293",
        "5968",
        "3925",
        "1835",
        "709",
        "4110",
        "4109",
        "4108",
        "5966",
        "4107",
        "5301",
        "2376",
        "3292",
        "3291",
        "5287",
        "5018",
        "5019",
        "4315",
        "4317",
        "3793",
        "1170",
        "4316",
        "1563",
        "4319",
        "1511",
        "2847",
        "2845",
        "2846",
        "2849",
        "1176",
        "1814",
        "1175",
        "240",
        "241",
        "2958",
        "1565",
        "4227",
        "1568",
        "4860",
        "1571",
        "5073",
        "1836",
        "1102",
        "301",
        "1573",
        "303",
        "1943",
        "4984",
        "1406",
        "1394",
        "3563",
        "1374",
        "2277",
        "1271",
        "136",
        "5031",
        "1113",
        "1378",
        "93",
        "1395",
        "5967",
        "2710",
        "666",
        "5078",
        "4454",
        "5088",
        "5384",
        "252",
        "1667",
        "1556",
        "1554",
        "4173",
        "4059",
        "5363",
        "5361",
        "4497",
        "2193",
        "4994",
        "1891",
        "4215",
        "1932",
        "1744",
        "4571",
        "3732",
        "3898",
        "1714",
        "4566",
        "1607",
        "1609",
        "1603",
        "1608",
        "5231",
        "1605",
        "1604",
        "4858",
        "4857",
        "1781",
        "1285",
        "2548",
        "6042",
        "2136",
        "3724",
        "4145",
        "1668",
        "4472",
        "2064",
        "2345",
        "4210",
        "4373",
        "2950",
        "2948",
        "1261",
        "5106",
        "1117",
        "1120",
        "1118",
        "1119",
        "693",
        "4582",
        "1116",
        "2888",
        "4207",
        "4681",
        "1907",
        "5728",
        "2121",
        "5263",
        "1564",
        "2302",
        "c_60d786af",
        "387",
        "5232",
        "6790",
        "1654",
        "2732",
        "2749",
        "6789",
        "6787",
        "2730",
        "4311",
        "2738",
        "2731",
        "79",
        "1250",
        "2212",
        "2733",
        "2729",
        "2753",
        "1240",
        "2776",
        "2762",
        "c_16b63d4f",
        "2757",
        "2779",
        "2741",
        "2764",
        "2782",
        "2760",
        "2777",
        "2962",
        "4083",
        "2752",
        "2743",
        "473",
        "2226",
        "386",
        "4726",
        "2887",
        "4294",
        "1857",
        "132",
        "4498",
        "6039",
        "2963",
        "741",
        "683",
        "2133",
        "2658",
        "463",
        "8311",
        "8332",
        "8324",
        "8328",
        "8304",
        "8317",
        "1288",
        "c_8331ae25",
        "5352",
        "c_791e9f27",
        "246",
        "2293",
        "c_bbd07d89",
        "c_be8fda35",
        "c_f2f1fb47",
        "c_0a700c19",
        "2853",
        "c_c44f1704",
        "5041",
        "7842",
        "7847",
        "7850",
        "7849",
        "7846",
        "7845",
        "7841",
        "7840",
        "7844",
        "7853",
        "7843",
        "7848",
        "7851",
        "7852",
        "8491",
        "8490",
        "8494",
        "8496",
        "8497",
        "8493",
        "8492",
        "8495",
        "3395",
        "5076",
        "3394",
        "1730",
        "3393",
        "1177",
        "1187",
        "1209",
        "2210",
        "5037",
        "4781",
        "1101",
        "5637",
        "1248",
        "1249",
        "1660",
        "1189",
        "4632",
        "511",
        "6237",
        "6242",
        "6246",
        "6238",
        "6239",
        "6245",
        "6240",
        "6241",
        "6198",
        "6200",
        "6197",
        "6202",
        "6194",
        "6201",
        "6199",
        "6195",
        "6196",
        "3891",
        "1586",
        "1701",
        "4823",
        "2570",
        "1100",
        "4660",
        "4568",
        "5091",
        "4640",
        "1659",
        "5098",
        "4155",
        "1960",
        "2139",
        "2379",
        "1813",
        "4468",
        "5075",
        "725",
        "5947",
        "2021",
        "113",
        "1610",
        "1230",
        "5932",
        "4825",
        "5470",
        "1239",
        "3711",
        "5880",
        "5730",
        "3918",
        "3713",
        "3903",
        "4847",
        "4835",
        "4560",
        "5879",
        "5039",
        "3726",
        "8280",
        "4693",
        "1",
        "4678",
        "4659",
        "2074",
        "5027",
        "8279",
        "3719",
        "1096",
        "1020",
        "8282",
        "4297",
        "1097",
        "8283",
        "3752",
        "232",
        "4296",
        "4206",
        "8281",
        "7828",
        "7829",
        "7827",
        "7821",
        "7822",
        "7818",
        "7825",
        "7826",
        "7824",
        "7819",
        "7823",
        "7820",
        "c_de77a2c0",
        "c_d4ca9a6c",
        "c_4b33c289",
        "c_e6d358a0",
        "c_46e54685",
        "c_c5c9cfc7",
        "c_d3974887",
        "7889",
        "4458",
        "1183",
        "1295",
        "4171",
        "334",
        "839",
        "2315",
        "7890",
        "495",
        "1717",
        "305",
        "4459",
        "4244",
        "7881",
        "6251",
        "6248",
        "6247",
        "6250",
        "7880",
        "6249",
        "7904",
        "7899",
        "7901",
        "7902",
        "7905",
        "7903",
        "7900",
        "5517",
        "3039",
        "3084",
        "3061",
        "7862",
        "7864",
        "7866",
        "7867",
        "7865",
        "7858",
        "7860",
        "7857",
        "7859",
        "7856",
        "7877",
        "7878",
        "7870",
        "7873",
        "7869",
        "7871",
        "7872",
        "7875",
        "7876",
        "7874",
        "7830",
        "7839",
        "7834",
        "7838",
        "7836",
        "7854",
        "7837",
        "7835",
        "7965",
        "7979",
        "7972",
        "7976",
        "7967",
        "7970",
        "7968",
        "7977",
        "7971",
        "7978",
        "7966",
        "7973",
        "7974",
        "7980",
        "7975",
        "7969",
        "2289",
        "3384",
        "3388",
        "1127",
        "6282",
        "6283",
        "2354",
        "3385",
        "3391",
        "4172",
        "3392",
        "869",
        "3382",
        "1873",
        "5273",
        "3390",
        "3822",
        "3387",
        "2038",
        "3386",
        "5423",
        "291",
        "332",
        "302",
        "342",
        "357",
        "3804",
        "351",
        "3799",
        "3407",
        "8371",
        "8374",
        "5774",
        "4649",
        "3381",
        "1290",
        "7208",
        "1180",
        "2362",
        "1606",
        "1624",
        "261",
        "1716",
        "236",
        "1903",
        "2546",
        "1862",
        "6279",
        "4776",
        "1727",
        "5777",
        "4778",
        "2291",
        "2612",
        "2613",
        "2614",
        "2822",
        "4258",
        "4266",
        "1895",
        "2907",
        "2902",
        "4265",
        "3326",
        "4268",
        "2066",
        "4267",
        "4213",
        "4262",
        "4302",
        "3085",
        "1901",
        "770",
        "3086",
        "1728",
        "6786",
        "1352",
        "1731",
        "634",
        "3400",
        "4250",
        "1041",
        "1040",
        "2910",
        "1038",
        "1421",
        "4254",
        "3080",
        "430",
        "310",
        "6143",
        "2299",
        "5148",
        "2295",
        "2933",
        "358",
        "2569",
        "2297",
        "6781",
        "4223",
        "461",
        "1472",
        "274",
        "5146",
        "3533",
        "5992",
        "3071",
        "3068",
        "835",
        "889",
        "723",
        "862",
        "3069",
        "3070",
        "976",
        "745",
        "797",
        "786",
        "766",
        "8460",
        "995",
        "2407",
        "2393",
        "2408",
        "2406",
        "1409",
        "4077",
        "369",
        "202",
        "2490",
        "2461",
        "c_1138d9d2",
        "2518",
        "410",
        "1463",
        "3066",
        "2465",
        "2458",
        "2844",
        "2462",
        "3194",
        "653",
        "660",
        "658",
        "661",
        "c_87ccc7c3",
        "652",
        "3082",
        "659",
        "647",
        "643",
        "639",
        "7951",
        "655",
        "650",
        "4008",
        "4016",
        "4021",
        "4013",
        "6783",
        "6032",
        "654",
        "1792",
        "646",
        "c_68ddcc11",
        "5840",
        "649",
        "4601",
        "c_8bf65f26",
        "657",
        "640",
        "5564",
        "2062",
        "5056",
        "3684",
        "834",
        "6785",
        "3319",
        "3072",
        "2575",
        "836",
        "4372",
        "3328",
        "3083",
        "2778",
        "59",
        "3333",
        "536",
        "3073",
        "5845",
        "5457",
        "c_a6b5bad0",
        "c_6eb6fb87",
        "5443",
        "c_12742da8",
        "431",
        "1542",
        "5433",
        "462",
        "6782",
        "2516",
        "c_c8363c09",
        "1468",
        "3067",
        "3079",
        "4129",
        "3052",
        "5846",
        "3047",
        "3064",
        "4128",
        "3078",
        "3081",
        "3043",
        "243",
        "412",
        "420",
        "3157",
        "5482",
        "4802",
        "413",
        "416",
        "4165",
        "5958",
        "1335",
        "4800",
        "339",
        "6410",
        "5336",
        "6412",
        "6411",
        "5294",
        "1269",
        "6103",
        "c_fa0acac6",
        "344",
        "6408",
        "6409",
        "347",
        "c_2b70515a",
        "345",
        "612",
        "4141",
        "346",
        "3188",
        "6720",
        "5048",
        "2521",
        "4308",
        "5319",
        "c_be0de7b7",
        "2530",
        "6413",
        "2527",
        "2539",
        "2542",
        "3159",
        "3161",
        "2531",
        "2534",
        "2541",
        "2540",
        "3164",
        "6407",
        "2532",
        "3163",
        "6414",
        "2538",
        "3162",
        "2537",
        "2526",
        "2528",
        "3531",
        "3536",
        "3153",
        "3538",
        "3546",
        "3551",
        "3535",
        "3528",
        "3687",
        "972",
        "865",
        "1030",
        "3149",
        "3154",
        "260",
        "1987",
        "1870",
        "4785",
        "2389",
        "3414",
        "2107",
        "c_93670e37",
        "415",
        "1287",
        "5697",
        "4786",
        "89",
        "3967",
        "4940",
        "664",
        "5741",
        "844",
        "3136",
        "2611",
        "177",
        "2824",
        "2626",
        "2877",
        "6400",
        "434843",
        "434786",
        "434966",
        "434938",
        "434909",
        "2947",
        "3133",
        "718",
        "c_749f2ce9",
        "2869",
        "535",
        "c_bb1a60eb",
        "3139",
        "6396",
        "1157",
        "5682",
        "8550",
        "6397",
        "4074",
        "3140",
        "5794",
        "2919",
        "1160",
        "1709",
        "c_165899a6",
        "c_d8ba2fe4",
        "3144",
        "3141",
        "2841",
        "5225",
        "4801",
        "3142",
        "134",
        "2826",
        "1706",
        "2827",
        "6398",
        "6131",
        "2201",
        "8300",
        "6148",
        "6161",
        "5370",
        "168",
        "6252",
        "690",
        "1298",
        "8373",
        "1168",
        "1291",
        "4848",
        "2547",
        "3823",
        "1019",
        "336",
        "2084",
        "4320",
        "1294",
        "1509",
        "1774",
        "4580",
        "1182",
        "5989",
        "3814",
        "1519",
        "863",
        "2563",
        "1504",
        "4353",
        "4324",
        "3774",
        "1874",
        "1085",
        "4839",
        "1712",
        "2286",
        "1649",
        "1318",
        "1965",
        "2543",
        "5563",
        "5898",
        "5713",
        "1125",
        "5469",
        "5859",
        "5695",
        "6223",
        "3763",
        "5860",
        "1255",
        "5063",
        "5876",
        "5892",
        "5746",
        "5946",
        "5458",
        "5858",
        "1188",
        "5100",
        "1899",
        "888",
        "1338",
        "5969",
        "4707",
        "2089",
        "2368",
        "1767",
        "1254",
        "4164",
        "5049",
        "235",
        "4453",
        "4160",
        "1524",
        "1355",
        "4375",
        "1768",
        "4862",
        "4461",
        "4775",
        "1968",
        "1518",
        "1555",
        "4966",
        "1584",
        "2052",
        "4246",
        "2222",
        "4732",
        "5694",
        "5925",
        "5693",
        "5944",
        "5770",
        "5721",
        "434",
        "1770",
        "5951",
        "6904",
        "5732",
        "6741",
        "5594",
        "3289",
        "5305",
        "5307",
        "5306",
        "5902",
        "c_7dda56e2",
        "6903",
        "6929",
        "c_abfaf2ea",
        "5292",
        "3844",
        "5899",
        "5878",
        "5938",
        "1594",
        "5810",
        "1854",
        "6565",
        "3288",
        "3885",
        "1855",
        "5293",
        "7355",
        "3368",
        "3364",
        "2843",
        "3375",
        "c_41ef4cb8",
        "5731",
        "3372",
        "1181",
        "5894",
        "1084",
        "3366",
        "1256",
        "464",
        "876",
        "1442",
        "2078",
        "c_1af36ac9",
        "4230",
        "961",
        "5734",
        "4733",
        "4635",
        "3815",
        "3813",
        "5914",
        "5818",
        "3596",
        "83",
        "4881",
        "2870",
        "5653",
        "7339",
        "6545",
        "5690",
        "7361",
        "6546",
        "4229",
        "6001",
        "6000",
        "5997",
        "5998",
        "5999",
        "6003",
        "4112",
        "2471",
        "5882",
        "5994",
        "8442",
        "6559",
        "5162",
        "5165",
        "5163",
        "5164",
        "7342",
        "5297",
        "450925",
        "6562",
        "4098",
        "5896",
        "4100",
        "2122",
        "4877",
        "5513",
        "c_e9ccc5a7",
        "5747",
        "4102",
        "4103",
        "1389",
        "2601",
        "3249",
        "7946",
        "3332",
        "7981",
        "1369",
        "3512",
        "3751",
        "2604",
        "1399",
        "3412",
        "7382",
        "c_b0c14f45",
        "185",
        "1462",
        "6541",
        "3323",
        "934",
        "760",
        "4495",
        "6534",
        "8448",
        "6625",
        "4007",
        "2197",
        "761",
        "762",
        "759",
        "3302",
        "3305",
        "4504",
        "162",
        "4278",
        "6737",
        "851",
        "7359",
        "4834",
        "5488",
        "5429",
        "7395",
        "476",
        "477",
        "4741",
        "5712",
        "4709",
        "1703",
        "3380",
        "591",
        "4606",
        "547",
        "2908",
        "6543",
        "4259",
        "2554",
        "974",
        "1993",
        "1992",
        "2480",
        "1949",
        "3464",
        "1865",
        "5280",
        "5281",
        "975",
        "2340",
        "160",
        "2333",
        "2594",
        "1528",
        "758",
        "c_3a9ccfe3",
        "6557",
        "6555",
        "6553",
        "6739",
        "375",
        "6743",
        "5560",
        "5512",
        "1710",
        "2265",
        "3748",
        "3330",
        "2174",
        "2944",
        "3331",
        "1527",
        "3336",
        "c_e2de1ffd",
        "3335",
        "443600",
        "6554",
        "443629",
        "c_67699417",
        "3705",
        "4471",
        "3340",
        "8461",
        "5099",
        "8472",
        "c_6e3cbbca",
        "3339",
        "3608",
        "1373",
        "3343",
        "3341",
        "4767",
        "1404",
        "8471",
        "3118",
        "3702",
        "6121",
        "5723",
        "c_d6ede0c7",
        "3613",
        "5766",
        "853",
        "3338",
        "5396",
        "3351",
        "3349",
        "5636",
        "6550",
        "6548",
        "5484",
        "6113",
        "5395",
        "5060",
        "6253",
        "6105",
        "2514",
        "2156",
        "2424",
        "981",
        "1964",
        "1880",
        "2428",
        "1736",
        "5288",
        "3838",
        "1779",
        "4647",
        "2056",
        "3835",
        "5988",
        "4783",
        "5304",
        "4462",
        "3836",
        "2568",
        "6018",
        "879",
        "4990",
        "1446",
        "316",
        "3834",
        "4577",
        "5065",
        "3832",
        "1059",
        "1515",
        "1516",
        "6821",
        "1827",
        "2425",
        "6822",
        "4856",
        "5993",
        "3690",
        "3881",
        "8435",
        "2744",
        "1042",
        "5971",
        "5046",
        "1499",
        "5055",
        "5045",
        "4676",
        "989",
        "5242",
        "5241",
        "2430",
        "3818",
        "1382",
        "1381",
        "1383",
        "3811",
        "5410",
        "3863",
        "5500",
        "5927",
        "5888",
        "5532",
        "5466",
        "3610",
        "2228",
        "3883",
        "5014",
        "5244",
        "5346",
        "5467",
        "988",
        "3884",
        "1650",
        "5013",
        "3478",
        "169",
        "3471",
        "6824",
        "2765",
        "2751",
        "5819",
        "2780",
        "2936",
        "3600",
        "3616",
        "70",
        "3625",
        "2686",
        "299",
        "1493",
        "4624",
        "4833",
        "850",
        "2712",
        "3621",
        "3620",
        "990",
        "3893",
        "3892",
        "4680",
        "3541",
        "3303",
        "4166",
        "1341",
        "3728",
        "2623",
        "c_a3b85311",
        "c_e749c083",
        "248",
        "262",
        "2098",
        "5362",
        "c_a18525ab",
        "c_f007e95f",
        "3235",
        "5153",
        "947",
        "2946",
        "4493",
        "5157",
        "2386",
        "5240",
        "4749",
        "2890",
        "3240",
        "4226",
        "1952",
        "2394",
        "1132",
        "2805",
        "2798",
        "3148",
        "3674",
        "2678",
        "4818",
        "2675",
        "4397",
        "5011",
        "2684",
        "4391",
        "2799",
        "508",
        "1698",
        "1959",
        "7384",
        "7383",
        "426",
        "2960",
        "c_d1f03d01",
        "5804",
        "5943",
        "5584",
        "6212",
        "c_9a1bc51d",
        "5917",
        "433",
        "1808",
        "1812",
        "4393",
        "2240",
        "c_2ffe4574",
        "1974",
        "1975",
        "1972",
        "903",
        "902",
        "901",
        "900",
        "3220",
        "4392",
        "2900",
        "5647",
        "562",
        "7351",
        "742",
        "3263",
        "4814",
        "5090",
        "4774",
        "4813",
        "4752",
        "3247",
        "4815",
        "5089",
        "4795",
        "4784",
        "4806",
        "c_ee45cd99",
        "c_dd52c1e9",
        "c_dcc650ef",
        "c_7f2d2214",
        "c_14d71455",
        "c_9ffe3bdc",
        "1476",
        "5023",
        "4394",
        "1970",
        "300",
        "1973",
        "587",
        "1918",
        "1300",
        "1858",
        "882",
        "880",
        "2225",
        "1351",
        "5602",
        "6151",
        "4603",
        "4437",
        "5751",
        "3183",
        "5820",
        "5369",
        "1642",
        "1644",
        "3180",
        "5368",
        "3181",
        "6155",
        "2321",
        "4500",
        "5935",
        "97",
        "2322",
        "812",
        "2047",
        "2048",
        "5750",
        "4477",
        "8552",
        "2049",
        "4389",
        "3190",
        "2961",
        "3191",
        "3175",
        "4440",
        "3568",
        "1006",
        "3267",
        "4310",
        "287",
        "3065",
        "4347",
        "2839",
        "2401",
        "5666",
        "3169",
        "1641",
        "5376",
        "6926",
        "5033",
        "2746",
        "4666",
        "4673",
        "4670",
        "4668",
        "4665",
        "4669",
        "3210",
        "4664",
        "4672",
        "4667",
        "5015",
        "1305",
        "5058",
        "7947",
        "1578",
        "5771",
        "771",
        "c_f2c7a2f7",
        "5696",
        "3604",
        "3603",
        "263",
        "3658",
        "72",
        "407",
        "3213",
        "2434",
        "3211",
        "3592",
        "3212",
        "3214",
        "6901",
        "3199",
        "5353",
        "1090",
        "442884",
        "1477",
        "24",
        "1199",
        "3217",
        "4288",
        "4289",
        "1996",
        "1226",
        "1225",
        "4036",
        "2871",
        "c_e5dbcb63",
        "1425",
        "3678",
        "2750",
        "6369",
        "4131",
        "7136",
        "4122",
        "1881",
        "4403",
        "811",
        "3591",
        "5367",
        "1839",
        "4240",
        "3094",
        "46",
        "1136",
        "6115",
        "6145",
        "6146",
        "c_8150867f",
        "4290",
        "1054",
        "1721",
        "1711",
        "2217",
        "6210",
        "c_b3d1a308",
        "c_0e4239f5",
        "c_441fb0c4",
        "266",
        "279",
        "265",
        "1083",
        "264",
        "277",
        "268",
        "267",
        "275",
        "276",
        "4641",
        "5202",
        "409",
        "1815",
        "1057",
        "4884",
        "1673",
        "c_fa8f51b0",
        "63",
        "62",
        "367",
        "4842",
        "2355",
        "392",
        "2911",
        "5270",
        "76",
        "5267",
        "3306",
        "3008",
        "2837",
        "1662",
        "478",
        "64",
        "519",
        "5332",
        "4309",
        "931",
        "354",
        "1980",
        "1982",
        "1981",
        "894",
        "928",
        "918",
        "6755",
        "2016",
        "2358",
        "4200",
        "3896",
        "4199",
        "4196",
        "6757",
        "4192",
        "5658",
        "1051",
        "3122",
        "960",
        "1674",
        "5218",
        "1301",
        "5389",
        "4158",
        "4203",
        "437",
        "380",
        "998",
        "4313",
        "6750",
        "999",
        "3821",
        "5724",
        "4140",
        "3444",
        "4106",
        "2097",
        "3445",
        "1441",
        "2595",
        "3419",
        "2557",
        "366",
        "3426",
        "3425",
        "3427",
        "5453",
        "5660",
        "5468",
        "3434",
        "5952",
        "3433",
        "6377",
        "5801",
        "5901",
        "7412",
        "5827",
        "3644",
        "5525",
        "3637",
        "2835",
        "3367",
        "7935",
        "6898",
        "3472",
        "201",
        "3490",
        "5485",
        "5394",
        "5639",
        "5375",
        "2380",
        "6919",
        "8410",
        "2949",
        "3455",
        "3454",
        "6930",
        "7213",
        "8441",
        "6376",
        "1215",
        "1213",
        "1212",
        "7214",
        "1008",
        "80",
        "2793",
        "4348",
        "958",
        "2294",
        "6374",
        "4843",
        "3295",
        "4627",
        "231",
        "1467",
        "2006",
        "5122",
        "860",
        "6124",
        "2356",
        "5047",
        "8464",
        "994",
        "4576",
        "6112",
        "2988",
        "1612",
        "3308",
        "2990",
        "1233",
        "4854",
        "1888",
        "1908",
        "4747",
        "4745",
        "4746",
        "1988",
        "4395",
        "4236",
        "6048",
        "755",
        "870",
        "6310",
        "6312",
        "6309",
        "191",
        "604",
        "6313",
        "2979",
        "645",
        "4255",
        "1475",
        "984",
        "1789",
        "1445",
        "2426",
        "2415",
        "2997",
        "2996",
        "6324",
        "6721",
        "6327",
        "5016",
        "5950",
        "2935",
        "2934",
        "2621",
        "2937",
        "2620",
        "131",
        "1577",
        "1534",
        "3345",
        "6319",
        "767",
        "765",
        "1719",
        "2204",
        "4484",
        "5227",
        "2987",
        "2357",
        "1146",
        "4388",
        "5012",
        "4637",
        "4650",
        "6928",
        "2438",
        "2437",
        "2436",
        "c_623f2583",
        "1204",
        "2306",
        "2304",
        "4829",
        "4788",
        "133",
        "5257",
        "5254",
        "5253",
        "2309",
        "5256",
        "2872",
        "915",
        "2329",
        "3001",
        "544",
        "2323",
        "997",
        "5229",
        "2984",
        "2727",
        "3275",
        "6315",
        "2709",
        "6316",
        "808",
        "595",
        "228",
        "4876",
        "4446",
        "4327",
        "1179",
        "4339",
        "2360",
        "2017",
        "2237",
        "4613",
        "2667",
        "4287",
        "4281",
        "6293",
        "4232",
        "3629",
        "2708",
        "2714",
        "2717",
        "831",
        "641",
        "3628",
        "4346",
        "2719",
        "7129",
        "c_c95f62c1",
        "1640",
        "1821",
        "3663",
        "167",
        "5042",
        "3095",
        "c_20f6ad03",
        "6304",
        "6264",
        "6271",
        "6270",
        "6267",
        "6269",
        "6272",
        "6263",
        "6303",
        "c_4254375b",
        "8454",
        "304",
        "c_18ac5876",
        "6171",
        "5768",
        "1669",
        "4721",
        "996",
        "4575",
        "4819",
        "3805",
        "2421",
        "3802",
        "2694",
        "2693",
        "2692",
        "3647",
        "3648",
        "5726",
        "3646",
        "2077",
        "1252",
        "2852",
        "867",
        "c_18802d13",
        "3848",
        "6300",
        "c_ab84e157",
        "c_79e507c2",
        "c_50fb9d42",
        "c_d6e5f3ab",
        "5461",
        "3639",
        "4419",
        "4421",
        "3638",
        "6297",
        "2206",
        "c_a02c420c",
        "c_28c51c2a",
        "5887",
        "730",
        "5889",
        "4047",
        "3266",
        "4386",
        "3269",
        "5215",
        "5214",
        "2865",
        "4378",
        "4382",
        "3103",
        "5213",
        "1099",
        "4398",
        "531",
        "2301",
        "2663",
        "2326",
        "6855",
        "c_52792cef",
        "8463",
        "4438",
        "1238",
        "3905",
        "5699",
        "2132",
        "1142",
        "6852",
        "2349",
        "2343",
        "2352",
        "1655",
        "4542",
        "4622",
        "5865",
        "4512",
        "5392",
        "158",
        "5401",
        "3681",
        "3089",
        "6849",
        "4538",
        "254",
        "848",
        "4655",
        "4656",
        "4652",
        "5331",
        "4654",
        "3090",
        "2433",
        "3430",
        "2234",
        "193",
        "4657",
        "3127",
        "4653",
        "3438",
        "2233",
        "1283",
        "5209",
        "4520",
        "1195",
        "7945",
        "2043",
        "4830",
        "4832",
        "985",
        "2282",
        "2283",
        "5103",
        "5101",
        "3700",
        "2316",
        "2044",
        "4736",
        "4735",
        "3321",
        "2728",
        "3550",
        "163",
        "4551",
        "4553",
        "1243",
        "593",
        "2953",
        "1535",
        "4435",
        "1257",
        "1259",
        "3548",
        "1761",
        "825",
        "1334",
        "1853",
        "1280",
        "5612",
        "2857",
        "1273",
        "2375",
        "8416",
        "7188",
        "5588",
        "5527",
        "1689",
        "3417",
        "1186",
        "3573",
        "3575",
        "2279",
        "5604",
        "6026",
        "3569",
        "4233",
        "1424",
        "157",
        "3363",
        "6051",
        "3571",
        "7810",
        "6718",
        "3552",
        "5941",
        "3527",
        "4548",
        "4544",
        "3516",
        "3520",
        "6347",
        "1638",
        "6346",
        "532",
        "6099",
        "1634",
        "3524",
        "1648",
        "1647",
        "7949",
        "6236",
        "1343",
        "6355",
        "6357",
        "2807",
        "6356",
        "5436",
        "6150",
        "1349",
        "625",
        "3845",
        "4820",
        "5869",
        "3847",
        "5085",
        "5084",
        "1783",
        "2209",
        "2208",
        "2276",
        "6351",
        "2553",
        "5986",
        "2552",
        "6353",
        "6887",
        "6891",
        "5830",
        "2363",
        "1481",
        "2876",
        "1995",
        "5558",
        "5117",
        "4694",
        "2598",
        "22",
        "14",
        "21",
        "15",
        "7190",
        "1976",
        "4871",
        "3036",
        "5784",
        "4271",
        "1047",
        "734",
        "5116",
        "1860",
        "2939",
        "5936",
        "5070",
        "5918",
        "6896",
        "126",
        "5959",
        "1939",
        "171",
        "1593",
        "3507",
        "2506",
        "2508",
        "c_cef8ea9d",
        "2587",
        "5995",
        "2800",
        "2811",
        "127",
        "4753",
        "4756",
        "450",
        "3059",
        "2878",
        "4755",
        "2160",
        "489",
        "338",
        "356",
        "1242",
        "2880",
        "6006",
        "159",
        "4321",
        "4757",
        "4751",
        "3045",
        "3037",
        "4758",
        "355",
        "5529",
        "418",
        "5276",
        "694",
        "1679",
        "3041",
        "1359",
        "1082",
        "2796",
        "2808",
        "1013",
        "705",
        "4764",
        "2223",
        "1678",
        "3975",
        "3976",
        "5624",
        "1579",
        "2586",
        "2584",
        "2585",
        "8423",
        "8421",
        "8422",
        "8424",
        "1448",
        "5654",
        "8415",
        "1211",
        "253",
        "2630",
        "769",
        "3053",
        "2991",
        "4452",
        "4467",
        "2440",
        "3055",
        "526",
        "3389",
        "885",
        "7199",
        "5855",
        "214",
        "5609",
        "5748",
        "2332",
        "2269",
        "3260",
        "471",
        "379",
        "2388",
        "5960",
        "5074",
        "7375",
        "8427",
        "6731",
        "6735",
        "6732",
        "6734",
        "6730",
        "6733",
        "5599",
        "3924",
        "2158",
        "6106",
        "571",
        "3099",
        "1697",
        "3699",
        "151",
        "8425",
        "6610",
        "5497",
        "5496",
        "3979",
        "3980",
        "2942",
        "5837",
        "3978",
        "2941",
        "5145",
        "5412",
        "5417",
        "5416",
        "5838",
        "2940",
        "5832",
        "2964",
        "3965",
        "5686",
        "5144",
        "3973",
        "4882",
        "3970",
        "8455",
        "8443",
        "3972",
        "3971",
        "1457",
        "5633",
        "4434",
        "8453",
        "3839",
        "6900",
        "501",
        "3934",
        "5079",
        "3927",
        "497",
        "499",
        "3635",
        "3926",
        "1339",
        "2337",
        "884",
        "5910",
        "3940",
        "2338",
        "3939",
        "5161",
        "5493",
        "437483",
        "c_471a61cf",
        "1777",
        "1129",
        "1281",
        "3475",
        "5582",
        "5559",
        "3920",
        "306",
        "6077",
        "2832",
        "1498",
        "8426",
        "6277",
        "548",
        "717",
        "1622",
        "2881",
        "2051",
        "2086",
        "100",
        "6627",
        "553",
        "1941",
        "1942",
        "1473",
        "c_558c1e00",
        "c_b19d7503",
        "859",
        "5708",
        "5444",
        "5143",
        "1623",
        "3992",
        "4791",
        "3996",
        "611",
        "3995",
        "5405",
        "1114",
        "3677",
        "951",
        "3998",
        "3999",
        "4001",
        "505",
        "2096",
        "2036",
        "4005",
        "4004",
        "3526",
        "4003",
        "2271",
        "3951",
        "8434",
        "c_b499ede2",
        "c_7546272f",
        "3950",
        "5933",
        "3945",
        "5428",
        "5572",
        "1251",
        "6613",
        "8433",
        "2",
        "1092",
        "4742",
        "2383",
        "1600",
        "6727",
        "4349",
        "3960",
        "4114",
        "2597",
        "6899",
        "6925",
        "6728",
        "953",
        "950",
        "1272",
        "3989",
        "2085",
        "1546",
        "1547",
        "1545",
        "1544",
        "3983",
        "1540",
        "6624",
        "5139",
        "44",
        "6622",
        "94",
        "37",
        "3987",
        "1548",
        "3986",
        "3985",
        "c_d02647e2",
        "437539",
        "443",
        "7939",
        "92",
        "437655",
        "6176",
        "6905",
        "716",
        "777",
        "897",
        "88",
        "6276",
        "6126",
        "5197",
        "5520",
        "5071",
        "3023",
        "1429",
        "1431",
        "1427",
        "1428",
        "5473",
        "6778",
        "c_4cd4362b",
        "4426",
        "5198",
        "5358",
        "435299",
        "5199",
        "6779",
        "5920",
        "5919",
        "5476",
        "2157",
        "3701",
        "3033",
        "5915",
        "c_f53bfc88",
        "5863",
        "5929",
        "5893",
        "c_433922a6",
        "630",
        "4151",
        "1582",
        "1581",
        "6189",
        "444",
        "6004",
        "5272",
        "1350",
        "78",
        "617",
        "4135",
        "4134",
        "2860",
        "2896",
        "c_9a9e5627",
        "c_65366e84",
        "2894",
        "7130",
        "6768",
        "5568",
        "4864",
        "3342",
        "4138",
        "5274",
        "3032",
        "493",
        "1533",
        "8414",
        "4869",
        "6067",
        "4868",
        "6764",
        "4867",
        "4866",
        "4870",
        "1332",
        "5187",
        "5186",
        "5569",
        "6104",
        "6110",
        "1745",
        "5183",
        "4384",
        "4133",
        "6771",
        "c_324b44f1",
        "6775",
        "6177",
        "6772",
        "7950",
        "c_a17a2156",
        "6182",
        "6178",
        "6773",
        "3022",
        "4431",
        "5195",
        "3010",
        "6139",
        "4425",
        "4429",
        "5190",
        "30",
        "3860",
        "6774",
        "3018",
        "4502",
        "4433",
        "5180",
        "5194",
        "5192",
        "5193",
        "5348",
        "6770",
        "1984",
        "2883",
        "2599",
        "5320",
        "3014",
        "3012",
        "4432",
        "3015",
        "3868",
        "3871",
        "5477",
        "3867",
        "4331",
        "3872",
        "6130",
        "908",
        "5478",
        "5347",
        "1024",
        "428",
        "3729",
        "434664",
        "5278",
        "8385",
        "8387",
        "23",
        "8386",
        "7137",
        "c_b605a1d4",
        "17",
        "6058",
        "4875",
        "1286",
        "4610",
        "3708",
        "5526",
        "703",
        "4149",
        "8377",
        "8378",
        "8375",
        "7192",
        "3727",
        "7206",
        "1461",
        "7205",
        "450953",
        "1793",
        "6095",
        "6093",
        "6094",
        "257",
        "16",
        "7346",
        "8395",
        "5644",
        "118",
        "6101",
        "3760",
        "4218",
        "1643",
        "3761",
        "3758",
        "1934",
        "1933",
        "1656",
        "6027",
        "952",
        "1060",
        "1831",
        "2945",
        "1241",
        "572",
        "3739",
        "8412",
        "6701",
        "2312",
        "142",
        "378",
        "4629",
        "466",
        "2127",
        "6700",
        "2654",
        "2715",
        "2382",
        "6045",
        "1139",
        "3768",
        "5262",
        "3773",
        "3769",
        "1267",
        "1266",
        "3775",
        "6921",
        "c_ece9dd5b",
        "3770",
        "3779",
        "c_097751fd",
        "2886",
        "743",
        "752",
        "2256",
        "2374",
        "2482",
        "6031",
        "921",
        "2882",
        "2884",
        "3788",
        "1123",
        "776",
        "868",
        "3789",
        "1023",
        "3631",
        "6703",
        "4341",
        "3792",
        "c_3508a934",
        "2429",
        "4617",
        "5578",
        "5575",
        "5580",
        "5787",
        "5904",
        "5866",
        "4126",
        "5561",
        "5856",
        "1052",
        "437422",
        "1851",
        "2609",
        "441716",
        "738",
        "4080",
        "251",
        "5539",
        "5868",
        "2591",
        "5538",
        "5459",
        "5408",
        "5548",
        "5589",
        "5399",
        "5829",
        "5518",
        "5611",
        "6288",
        "447503",
        "c_5cec7cfa",
        "2346",
        "8407",
        "2347",
        "2348",
        "8542",
        "5506",
        "6050",
        "6060",
        "6057",
        "6059",
        "5509",
        "6056",
        "5507",
        "c_e9845504",
        "5807",
        "4124",
        "4044",
        "5128",
        "4174",
        "19",
        "2248",
        "443167",
        "439592",
        "5308",
        "8462",
        "4043",
        "5328",
        "439498",
        "447442",
        "447536",
        "237",
        "5806",
        "447604",
        "4178",
        "6708",
        "6706",
        "6710",
        "5494",
        "5552",
        "8480",
        "5550",
        "5761",
        "5942",
        "6709",
        "c_2e08d60f",
        "8478",
        "c_1c478aa5",
        "4048",
        "5404",
        "5749",
        "5403",
        "116",
        "5681",
        "1450",
        "c_6547ccb7",
        "5491",
        "6286",
        "5864",
        "5543",
        "c_fd848a9f",
        "5772",
        "5413",
        "2451",
        "8409",
        "4061",
        "5137",
        "5826",
        "2008",
        "c_6f7bbe63",
        "c_8d31aed6",
        "5675",
        "2931",
        "4037",
        "350",
        "3106",
        "4024",
        "4019",
        "388",
        "8479",
        "353",
        "352",
        "4026",
        "2930",
        "6133",
        "8468",
        "5204",
        "5812",
        "6265",
        "2227",
        "2618",
        "6266",
        "5811",
        "4023",
        "440710",
        "440657",
        "440631",
        "3494",
        "6290",
        "6289",
        "445228",
        "6285",
        "445203",
        "4027",
        "4473",
        "5129",
        "445259",
        "8489",
        "440605",
        "440579",
        "440265",
        "5815",
        "2809",
        "440238",
        "5115",
        "5511",
        "5822",
        "5456",
        "5203",
        "3087",
        "5524",
        "7128",
        "5136",
        "5953",
        "5739",
        "5581",
        "1989",
        "442850",
        "5661",
        "7362",
        "5663",
        "5803",
        "443080",
        "5662",
        "5664",
        "7391",
        "7392",
        "7393",
        "6395",
        "7394",
        "6394",
        "7941",
        "2019",
        "4888",
        "1580",
        "435265",
        "443048",
        "7389",
        "7812",
        "4176",
        "1276",
        "4770",
        "6391",
        "1936",
        "3294",
        "1147",
        "1152",
        "7360",
        "1150",
        "6389",
        "6387",
        "5269",
        "6388",
        "7218",
        "1149",
        "6386",
        "6381",
        "6392",
        "6384",
        "6716",
        "5608",
        "7955",
        "c_ff4dd793",
        "5338",
        "5740",
        "5028",
        "5651",
        "1246",
        "5303",
        "3282",
        "4810",
        "7410",
        "2373",
        "4889",
        "707",
        "5300",
        "5298",
        "5299",
        "4883",
        "5990",
        "601",
        "1134",
        "4713",
        "1671",
        "2104",
        "1552",
        "1665",
        "606",
        "1553",
        "1265",
        "3806",
        "1262",
        "4861",
        "1270",
        "1572",
        "1569",
        "1575",
        "1576",
        "1570",
        "1566",
        "1574",
        "1109",
        "1103",
        "1108",
        "1112",
        "1104",
        "1111",
        "1110",
        "1107",
        "1106",
        "4773",
        "5364",
        "5000",
        "5010",
        "4682",
        "4691",
        "4687",
        "4688",
        "4685",
        "4684",
        "4689",
        "4692",
        "4683",
        "2954",
        "2955",
        "920",
        "4170",
        "1863",
        "970",
        "1206",
        "4088",
        "2274",
        "2221",
        "1029",
        "1772",
        "5698",
        "5385",
        "4085",
        "417",
        "109",
        "4340",
        "1715",
        "2273",
        "6041",
        "6040",
        "8312",
        "8315",
        "8316",
        "8321",
        "8319",
        "8334",
        "8337",
        "8336",
        "8335",
        "8333",
        "8327",
        "8313",
        "8326",
        "8325",
        "8314",
        "8310",
        "8330",
        "8329",
        "8331",
        "8306",
        "8307",
        "8309",
        "8308",
        "8305",
        "8318",
        "8322",
        "8323",
        "8320",
        "1235",
        "4824",
        "1583",
        "1904",
        "2075",
        "2377",
        "1666",
        "2439",
        "2328",
        "1089",
        "5032",
        "3714",
        "4364",
        "5381",
        "1702",
        "1430",
        "8500",
        "8502",
        "8504",
        "8501",
        "8499",
        "8503",
        "8519",
        "8518",
        "8517",
        "8520",
        "8533",
        "8530",
        "8532",
        "8529",
        "8528",
        "8531",
        "8537",
        "8536",
        "8535",
        "8538",
        "8534",
        "8513",
        "8515",
        "8514",
        "8511",
        "8512",
        "8516",
        "8506",
        "8508",
        "8505",
        "8509",
        "8507",
        "8510",
        "8526",
        "8522",
        "8523",
        "8527",
        "8524",
        "8521",
        "8525",
        "8008",
        "8007",
        "8001",
        "8000",
        "8006",
        "8004",
        "8005",
        "8002",
        "8003",
        "7999",
        "7998",
        "7994",
        "7995",
        "7996",
        "7993",
        "7990",
        "7992",
        "7991",
        "7989",
        "7988",
        "5246",
        "5247",
        "c_82778b62",
        "c_dffba30e",
        "c_f9143718",
        "c_049691c5",
        "c_d3db05e6",
        "c_cf9e250b",
        "c_e4bfb3bb",
        "c_a5c15adc",
        "c_ec0f4970",
        "c_070e15e7",
        "c_23635024",
        "c_496cee2d",
        "c_095c8a84",
        "c_412ed48a",
        "c_7d7c383b",
        "c_fee7acae",
        "c_e300778f",
        "c_b4cffbee",
        "c_7a080afd",
        "c_14e0ae65",
        "c_eb3ada47",
        "7895",
        "7896",
        "7893",
        "7892",
        "7891",
        "5516",
        "5515",
        "5514",
        "1713",
        "2028",
        "1227",
        "1651",
        "5038",
        "5036",
        "1027",
        "2061",
        "1661",
        "1028",
        "4410",
        "5349",
        "1124",
        "5095",
        "1876",
        "4779",
        "3888",
        "1218",
        "111",
        "4351",
        "2027",
        "321",
        "1611",
        "5029",
        "33",
        "2018",
        "4148",
        "4782",
        "1561",
        "1841",
        "1766",
        "5556",
        "1729",
        "1843",
        "4780",
        "4217",
        "5108",
        "4092",
        "2057",
        "2058",
        "1842",
        "1128",
        "3830",
        "1217",
        "1769",
        "1216",
        "1236",
        "4465",
        "4846",
        "1166",
        "4161",
        "1617",
        "249",
        "1639",
        "1734",
        "5709",
        "5706",
        "5619",
        "451941",
        "451991",
        "5652",
        "5445",
        "5454",
        "314",
        "1228",
        "4567",
        "2562",
        "3415",
        "1522",
        "4816",
        "3406",
        "3405",
        "2032",
        "2820",
        "4570",
        "1185",
        "496",
        "3833",
        "2024",
        "3401",
        "3403",
        "2042",
        "2055",
        "1208",
        "691",
        "247",
        "4841",
        "2224",
        "1253",
        "4621",
        "1220",
        "1771",
        "5020",
        "6036",
        "4342",
        "2525",
        "5965",
        "2331",
        "841",
        "1219",
        "4662",
        "2231",
        "5040",
        "1292",
        "1559",
        "6278",
        "1296",
        "2037",
        "3410",
        "6280",
        "205",
        "6281",
        "3858",
        "4466",
        "3416",
        "1237",
        "4772",
        "4583",
        "3411",
        "4648",
        "3857",
        "2088",
        "2229",
        "993",
        "5930",
        "1613",
        "1532",
        "4354",
        "2336",
        "4572",
        "1501",
        "1560",
        "1500",
        "2866",
        "2138",
        "726",
        "4620",
        "4411",
        "1308",
        "623",
        "1955",
        "2101",
        "112",
        "852",
        "5562",
        "4249",
        "1131",
        "6061",
        "5673",
        "6705",
        "6125",
        "2627",
        "5585",
        "c_1137acd8",
        "2390",
        "2483",
        "5886",
        "6108",
        "2281",
        "3721",
        "3310",
        "959",
        "2280",
        "2241",
        "4369",
        "399",
        "2761",
        "2768",
        "c_61b6fae1",
        "5149",
        "c_f05a3667",
        "5472",
        "5847",
        "c_2a1a9fdc",
        "5823",
        "5849",
        "3034",
        "3051",
        "1900",
        "433963",
        "3988",
        "3155",
        "3158",
        "5890",
        "5565",
        "1440",
        "856",
        "5683",
        "686",
        "2063",
        "2523",
        "2522",
        "3469",
        "c_667f2c16",
        "3135",
        "4307",
        "2759",
        "6747",
        "2535",
        "916",
        "2533",
        "5605",
        "5676",
        "c_a2e1def8",
        "5857",
        "5760",
        "5722",
        "1879",
        "5736",
        "5737",
        "5735",
        "2503",
        "1983",
        "2110",
        "2111",
        "2108",
        "2112",
        "2109",
        "4808",
        "4796",
        "4792",
        "4803",
        "4797",
        "4807",
        "2215",
        "5463",
        "c_d701b19c",
        "c_3212cc75",
        "6405",
        "c_9055e9e1",
        "c_0e092eaf",
        "5464",
        "597",
        "5742",
        "4891",
        "5593",
        "2060",
        "434880",
        "435007",
        "5912",
        "3132",
        "5672",
        "5754",
        "6401",
        "272",
        "5533",
        "5839",
        "5782",
        "4075",
        "4076",
        "6402",
        "3138",
        "941",
        "919",
        "1741",
        "141",
        "4853",
        "4485",
        "4628",
        "2235",
        "3457",
        "2830",
        "3128",
        "3784",
        "2203",
        "4376",
        "5072",
        "451486",
        "1557",
        "1449",
        "3780",
        "5955",
        "1526",
        "4460",
        "5026",
        "1502",
        "4326",
        "1963",
        "4325",
        "4679",
        "5172",
        "3284",
        "2855",
        "3287",
        "3286",
        "3356",
        "3285",
        "6570",
        "5900",
        "c_1000336b",
        "3369",
        "5916",
        "c_7951978e",
        "5752",
        "4636",
        "3370",
        "3730",
        "1596",
        "3371",
        "192",
        "7135",
        "1480",
        "5733",
        "1997",
        "6568",
        "c_6f563894",
        "c_aaa8331c",
        "5911",
        "5934",
        "4306",
        "331",
        "3362",
        "4494",
        "1915",
        "4525",
        "4513",
        "7959",
        "4787",
        "4809",
        "5230",
        "4804",
        "1897",
        "4712",
        "2361",
        "4710",
        "4711",
        "1861",
        "c_c825b20c",
        "6556",
        "5168",
        "6547",
        "c_34242feb",
        "5909",
        "5167",
        "4769",
        "5956",
        "4768",
        "3609",
        "1530",
        "6038",
        "1837",
        "4642",
        "1787",
        "c_18180cde",
        "294",
        "5465",
        "3236",
        "6902",
        "2220",
        "5963",
        "4463",
        "4731",
        "1971",
        "230",
        "1816",
        "6359",
        "8429",
        "6334",
        "6360",
        "c_062f6587",
        "4284",
        "5025",
        "4283",
        "3618",
        "6089",
        "1732",
        "1733",
        "229",
        "3215",
        "3643",
        "5573",
        "3642",
        "6711",
        "1631",
        "5393",
        "3283",
        "2724",
        "6756",
        "4190",
        "4115",
        "6378",
        "3421",
        "2603",
        "91",
        "5523",
        "7417",
        "5535",
        "5756",
        "3578",
        "5432",
        "5670",
        "5326",
        "434056",
        "3508",
        "4794",
        "3509",
        "3510",
        "3456",
        "c_4e75f061",
        "2898",
        "6325",
        "2680",
        "4476",
        "1722",
        "4638",
        "4639",
        "6913",
        "3665",
        "3662",
        "3661",
        "3666",
        "5043",
        "550",
        "6307",
        "435098",
        "6295",
        "6296",
        "5235",
        "5234",
        "5233",
        "6298",
        "5481",
        "435061",
        "6912",
        "8452",
        "c_9a27a573",
        "2671",
        "2641",
        "c_681e6d24",
        "5111",
        "5391",
        "5388",
        "467",
        "468",
        "469",
        "7358",
        "4737",
        "4740",
        "4739",
        "4345",
        "2385",
        "6854",
        "4197",
        "4261",
        "4658",
        "c_b3e9d826",
        "106",
        "3354",
        "107",
        "3547",
        "5339",
        "3907",
        "4439",
        "5522",
        "5521",
        "6084",
        "7943",
        "7811",
        "5743",
        "2875",
        "2874",
        "5377",
        "5378",
        "425",
        "5586",
        "5600",
        "3221",
        "3063",
        "383",
        "2895",
        "1093",
        "2091",
        "4701",
        "5528",
        "c_9d09fbc3",
        "7381",
        "7380",
        "7223",
        "5540",
        "886",
        "5853",
        "95",
        "404",
        "348",
        "448",
        "2669",
        "364",
        "7366",
        "2236",
        "3921",
        "1699",
        "5711",
        "5685",
        "5802",
        "5850",
        "1747",
        "5110",
        "1940",
        "5112",
        "5113",
        "6005",
        "5492",
        "3942",
        "422",
        "605",
        "1171",
        "603",
        "598",
        "6626",
        "5330",
        "5649",
        "2867",
        "5796",
        "2272",
        "1095",
        "5570",
        "4727",
        "5659",
        "6132",
        "6616",
        "5884",
        "898",
        "5924",
        "4487",
        "5828",
        "c_715c08c0",
        "3740",
        "c_dea47e63",
        "5635",
        "3704",
        "5601",
        "6192",
        "5671",
        "5705",
        "5174",
        "6190",
        "6191",
        "5185",
        "435228",
        "8413",
        "443345",
        "6907",
        "4698",
        "4699",
        "6776",
        "6777",
        "5745",
        "6906",
        "3875",
        "5182",
        "3011",
        "c_25fe24f4",
        "c_32e52346",
        "3017",
        "5179",
        "395",
        "2600",
        "5808",
        "3019",
        "4428",
        "3021",
        "5189",
        "6337",
        "3755",
        "890",
        "1893",
        "3738",
        "c_6c8b04b9",
        "4469",
        "6699",
        "3776",
        "3772",
        "3771",
        "c_315af034",
        "5419",
        "5872",
        "5926",
        "c_4e3beee2",
        "452085",
        "5643",
        "5138",
        "450652",
        "450658",
        "c_e548b67f",
        "c_b54dc55a",
        "c_d08207d1",
        "978",
        "3238",
        "4237",
        "73",
        "c_4d5b6dbd",
        "2610",
        "2608",
        "2054",
        "2607",
        "6287",
        "5135",
        "5134",
        "5133",
        "6358",
        "6053",
        "7132",
        "7986",
        "5630",
        "6723",
        "6724",
        "8482",
        "8481",
        "8477",
        "8476",
        "7982",
        "7953",
        "8475",
        "7954",
        "5618",
        "4049",
        "5406",
        "8285",
        "5505",
        "5531",
        "5504",
        "5510",
        "4052",
        "4051",
        "4041",
        "5132",
        "5834",
        "c_c4c48e5c",
        "5674",
        "5315",
        "5824",
        "c_2212c2c4",
        "5567",
        "6218",
        "7138",
        "5312",
        "1531",
        "5131",
        "445178",
        "5816",
        "5814",
        "5790",
        "5648",
        "3866",
        "c_70c441cc",
        "c_104476d7",
        "5704",
        "5702",
        "c_b8e8e06e",
        "5700",
        "5703",
        "5701",
        "8438",
        "6385",
        "442791",
        "6744",
        "6260",
        "c_2c5a05a0",
        "442765",
        "6111",
        "4720",
        "5380",
        "4717",
        "4715",
        "4718",
        "4719",
        "4724",
        "4714",
        "4716",
        "4725",
        "4723",
        "4722",
        "2952",
        "2951",
        "1263",
        "1264",
        "2956",
        "4686",
        "4212",
        "484",
        "5617",
        "891",
        "4578",
        "1819",
        "1196",
        "3816",
        "1822",
        "1207",
        "8547",
        "3402",
        "1986",
        "1299",
        "451885",
        "333",
        "451912",
        "454784",
        "7198",
        "451462",
        "1289",
        "4523",
        "4154",
        "2899",
        "4592",
        "1121",
        "503",
        "7134",
        "6784",
        "3686",
        "2769",
        "5150",
        "2572",
        "1140",
        "5798",
        "5650",
        "2529",
        "5439",
        "5727",
        "8446",
        "8445",
        "5895",
        "5874",
        "450869",
        "1775",
        "8444",
        "5222",
        "5224",
        "5223",
        "5221",
        "5220",
        "1875",
        "1776",
        "1871",
        "1877",
        "1872",
        "c_1c2d26cf",
        "5791",
        "c_3897ba70",
        "6404",
        "608",
        "1169",
        "610",
        "599",
        "609",
        "607",
        "c_25e0bb6d",
        "600",
        "3147",
        "602",
        "1683",
        "1050",
        "2080",
        "840",
        "3137",
        "4335",
        "940",
        "c_82916123",
        "8451",
        "1505",
        "2290",
        "5438",
        "663",
        "7396",
        "6566",
        "c_8c634c9b",
        "6740",
        "4799",
        "2545",
        "c_8f56da2a",
        "6370",
        "434004",
        "6273",
        "2617",
        "5597",
        "c_ccac12ef",
        "c_12e208c8",
        "3777",
        "5668",
        "1056",
        "4056",
        "980",
        "979",
        "197",
        "7813",
        "196",
        "200",
        "8430",
        "3812",
        "3787",
        "5022",
        "6406",
        "8303",
        "6403",
        "1049",
        "2423",
        "c_d978c1fc",
        "1906",
        "596",
        "3396",
        "8498",
        "3399",
        "4777",
        "1905",
        "3398",
        "3397",
        "4247",
        "199",
        "4053",
        "198",
        "4355",
        "5318",
    ],
}


class MultiEURLEXConfig(datasets.BuilderConfig):
    """BuilderConfig for MultiEURLEX."""

    def __init__(self, language: str, languages=None, label_level="level_1", **kwargs):
        """BuilderConfig for MultiEURLEX.

        Args:
        language: One of ar,bg,de,el,en,es,fr,hi,ru,sw,th,tr,ur,vi,zh, or all_languages
          **kwargs: keyword arguments forwarded to super.
        """
        super(MultiEURLEXConfig, self).__init__(**kwargs)
        self.language = language
        self.label_level = label_level
        if language != "all_languages":
            self.languages = [language]
        else:
            self.languages = languages if languages is not None else _LANGUAGES


class MultiEURLEX(datasets.GeneratorBasedBuilder):
    """MultiEURLEX - A multi-lingual and multi-label legal document classification dataset. Version 1.0"""

    VERSION = datasets.Version("1.0.0", "")
    BUILDER_CONFIG_CLASS = MultiEURLEXConfig
    BUILDER_CONFIGS = [
        MultiEURLEXConfig(
            name=lang,
            language=lang,
            version=datasets.Version("1.0.0", ""),
            description=f"Plain text import of MultiEURLEX for the {lang} language",
        )
        for lang in _LANGUAGES
    ] + [
        MultiEURLEXConfig(
            name="all_languages",
            language="all_languages",
            version=datasets.Version("1.0.0", ""),
            description="Plain text import of MultiEURLEX for all languages",
        )
    ]

    def _info(self):
        if self.config.language == "all_languages":
            features = datasets.Features(
                {
                    "celex_id": datasets.Value("string"),
                    "text": datasets.Translation(
                        languages=_LANGUAGES,
                    ),
                    "labels": datasets.features.Sequence(
                        datasets.ClassLabel(names=_CONCEPTS[self.config.label_level])
                    ),
                }
            )
        else:
            features = datasets.Features(
                {
                    "celex_id": datasets.Value("string"),
                    "text": datasets.Value("string"),
                    "labels": datasets.features.Sequence(
                        datasets.ClassLabel(names=_CONCEPTS[self.config.label_level])
                    ),
                }
            )
        return datasets.DatasetInfo(
            description=_DESCRIPTION,
            features=features,
            supervised_keys=None,
            homepage="https://github.io/iliaschalkidis",
            citation=_CITATION,
        )

    def _split_generators(self, dl_manager):
        data_dir = dl_manager.download_and_extract(DATA_URL)
        return [
            datasets.SplitGenerator(
                name=datasets.Split.TRAIN,
                # These kwargs will be passed to _generate_examples
                gen_kwargs={"filepath": os.path.join(data_dir, "train.jsonl"), "split": "train"},
            ),
            datasets.SplitGenerator(
                name=datasets.Split.TEST,
                # These kwargs will be passed to _generate_examples
                gen_kwargs={"filepath": os.path.join(data_dir, "test.jsonl"), "split": "test"},
            ),
            datasets.SplitGenerator(
                name=datasets.Split.VALIDATION,
                # These kwargs will be passed to _generate_examples
                gen_kwargs={"filepath": os.path.join(data_dir, "dev.jsonl"), "split": "dev"},
            ),
        ]

    def _generate_examples(self, filepath, split):
        """This function returns the examples in the raw (text) form."""

        if self.config.language == "all_languages":
            with open(filepath, encoding="utf-8") as f:
                for id_, row in enumerate(f):
                    data = json.loads(row)
                    yield id_, {
                        "celex_id": data["celex_id"],
                        "text": {lang: data["text"][lang] for lang in self.config.languages},
                        "labels": data["eurovoc_concepts"][self.config.label_level],
                    }
        else:
            with open(filepath, encoding="utf-8") as f:
                for id_, row in enumerate(f):
                    data = json.loads(row)
                    if data["text"][self.config.language] is not None:
                        yield id_, {
                            "celex_id": data["celex_id"],
                            "text": data["text"][self.config.language],
                            "labels": data["eurovoc_concepts"][self.config.label_level],
                        }
