# coding=utf-8
# Copyright 2020 The HuggingFace Datasets Authors and the current dataset script contributor.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""\
Greek_Legal_Code contains 47k classified legal resources from Greek Legislation. Its origin is “Permanent Greek Legislation Code - Raptarchis”,
a collection of Greek legislative documents classified into multi-level (from broader to more specialized) categories.
"""


import json
import os

import datasets


_CITATION = """\
@inproceedings{papaloukas-etal-2021-glc,
    title = "Multi-granular Legal Topic Classification on Greek Legislation",
    author = "Papaloukas, Christos and Chalkidis, Ilias and Athinaios, Konstantinos and Pantazi, Despina-Athanasia and Koubarakis, Manolis",
    booktitle = "Proceedings of the 3rd Natural Legal Language Processing (NLLP) Workshop",
    year = "2021",
    address = "Punta Cana, Dominican Republic",
    publisher = "",
    url = "",
    doi = "",
    pages = ""
}
"""

_DESCRIPTION = """\
Greek_Legal_Code contains 47k classified legal resources from Greek Legislation. Its origin is “Permanent Greek Legislation Code - Raptarchis”,
a collection of Greek legislative documents classified into multi-level (from broader to more specialized) categories.
"""

_LABEL_NAMES = {
    "volume": [
        "ΚΟΙΝΩΝΙΚΗ ΠΡΟΝΟΙΑ",
        "ΓΕΩΡΓΙΚΗ ΝΟΜΟΘΕΣΙΑ",
        "ΡΑΔΙΟΦΩΝΙΑ ΚΑΙ ΤΥΠΟΣ",
        "ΒΙΟΜΗΧΑΝΙΚΗ ΝΟΜΟΘΕΣΙΑ",
        "ΥΓΕΙΟΝΟΜΙΚΗ ΝΟΜΟΘΕΣΙΑ",
        "ΠΟΛΕΜΙΚΟ ΝΑΥΤΙΚΟ",
        "ΤΑΧΥΔΡΟΜΕΙΑ - ΤΗΛΕΠΙΚΟΙΝΩΝΙΕΣ",
        "ΔΑΣΗ ΚΑΙ ΚΤΗΝΟΤΡΟΦΙΑ",
        "ΕΛΕΓΚΤΙΚΟ ΣΥΝΕΔΡΙΟ ΚΑΙ ΣΥΝΤΑΞΕΙΣ",
        "ΠΟΛΕΜΙΚΗ ΑΕΡΟΠΟΡΙΑ",
        "ΝΟΜΙΚΑ ΠΡΟΣΩΠΑ ΔΗΜΟΣΙΟΥ ΔΙΚΑΙΟΥ",
        "ΝΟΜΟΘΕΣΙΑ ΑΝΩΝΥΜΩΝ ΕΤΑΙΡΕΙΩΝ ΤΡΑΠΕΖΩΝ ΚΑΙ ΧΡΗΜΑΤΙΣΤΗΡΙΩΝ",
        "ΠΟΛΙΤΙΚΗ ΑΕΡΟΠΟΡΙΑ",
        "ΕΜΜΕΣΗ ΦΟΡΟΛΟΓΙΑ",
        "ΚΟΙΝΩΝΙΚΕΣ ΑΣΦΑΛΙΣΕΙΣ",
        "ΝΟΜΟΘΕΣΙΑ ΔΗΜΩΝ ΚΑΙ ΚΟΙΝΟΤΗΤΩΝ",
        "ΝΟΜΟΘΕΣΙΑ ΕΠΙΜΕΛΗΤΗΡΙΩΝ ΣΥΝΕΤΑΙΡΙΣΜΩΝ ΚΑΙ ΣΩΜΑΤΕΙΩΝ",
        "ΔΗΜΟΣΙΑ ΕΡΓΑ",
        "ΔΙΟΙΚΗΣΗ ΔΙΚΑΙΟΣΥΝΗΣ",
        "ΑΣΦΑΛΙΣΤΙΚΑ ΤΑΜΕΙΑ",
        "ΕΚΚΛΗΣΙΑΣΤΙΚΗ ΝΟΜΟΘΕΣΙΑ",
        "ΕΚΠΑΙΔΕΥΤΙΚΗ ΝΟΜΟΘΕΣΙΑ",
        "ΔΗΜΟΣΙΟ ΛΟΓΙΣΤΙΚΟ",
        "ΤΕΛΩΝΕΙΑΚΗ ΝΟΜΟΘΕΣΙΑ",
        "ΣΥΓΚΟΙΝΩΝΙΕΣ",
        "ΕΘΝΙΚΗ ΑΜΥΝΑ",
        "ΣΤΡΑΤΟΣ ΞΗΡΑΣ",
        "ΑΓΟΡΑΝΟΜΙΚΗ ΝΟΜΟΘΕΣΙΑ",
        "ΔΗΜΟΣΙΟΙ ΥΠΑΛΛΗΛΟΙ",
        "ΠΕΡΙΟΥΣΙΑ ΔΗΜΟΣΙΟΥ ΚΑΙ ΝΟΜΙΣΜΑ",
        "ΟΙΚΟΝΟΜΙΚΗ ΔΙΟΙΚΗΣΗ",
        "ΛΙΜΕΝΙΚΗ ΝΟΜΟΘΕΣΙΑ",
        "ΑΣΤΙΚΗ ΝΟΜΟΘΕΣΙΑ",
        "ΠΟΛΙΤΙΚΗ ΔΙΚΟΝΟΜΙΑ",
        "ΔΙΠΛΩΜΑΤΙΚΗ ΝΟΜΟΘΕΣΙΑ",
        "ΔΙΟΙΚΗΤΙΚΗ ΝΟΜΟΘΕΣΙΑ",
        "ΑΜΕΣΗ ΦΟΡΟΛΟΓΙΑ",
        "ΤΥΠΟΣ ΚΑΙ ΤΟΥΡΙΣΜΟΣ",
        "ΕΘΝΙΚΗ ΟΙΚΟΝΟΜΙΑ",
        "ΑΣΤΥΝΟΜΙΚΗ ΝΟΜΟΘΕΣΙΑ",
        "ΑΓΡΟΤΙΚΗ ΝΟΜΟΘΕΣΙΑ",
        "ΕΡΓΑΤΙΚΗ ΝΟΜΟΘΕΣΙΑ",
        "ΠΟΙΝΙΚΗ ΝΟΜΟΘΕΣΙΑ",
        "ΕΜΠΟΡΙΚΗ ΝΟΜΟΘΕΣΙΑ",
        "ΕΠΙΣΤΗΜΕΣ ΚΑΙ ΤΕΧΝΕΣ",
        "ΕΜΠΟΡΙΚΗ ΝΑΥΤΙΛΙΑ",
        "ΣΥΝΤΑΓΜΑΤΙΚΗ ΝΟΜΟΘΕΣΙΑ",
    ],
    "chapter": [
        "ΜΕΤΑΛΛΕΙΑ ΚΑΙ ΟΡΥΧΕΙΑ",
        "ΣΤΑΤΙΩΤΙΚΕΣ ΣΧΟΛΕΣ",
        "ΠΑΡΟΧΕΣ ΑΝΕΡΓΙΑΣ",
        "ΣΙΔΗΡΟΔΡΟΜΙΚΑ ΔΙΚΤΥΑ",
        "ΕΙΔΙΚΑ ΣΤΡΑΤΙΩΤΙΚΑ ΑΔΙΚΗΜΑΤΑ",
        "ΚΡΑΤΙΚΕΣ ΠΡΟΜΗΘΕΙΕΣ",
        "ΑΓΡΟΤΙΚΗ ΑΠΟΚΑΤΑΣΤΑΣΗ",
        "ΑΞΙΩΜΑΤΙΚΟΙ ΠΟΛΕΜΙΚΟΥ ΝΑΥΤΙΚΟΥ",
        "ΣΧΕΔΙΑ ΠΟΛΕΩΝ",
        "ΣΥΚΑ",
        "ΠΡΟΛΗΨΙΣ ΚΑΙ ΔΙΩΞΙΣ ΤΟΥ ΕΓΚΛΗΜΑΤΟΣ",
        "ΔΙΕΘΝΕΙΣ ΜΕΤΑΦΟΡΕΣ",
        "ΓΕΝΙΚΗ ΣΥΓΚΟΙΝΩΝΙΑ ΚΑΙ ΔΙΑΤΑΞΕΙΣ",
        "ΚΛΗΡΟΝΟΜΙΚΟ ΔΙΚΑΙΟ",
        "ΚΟΙΝΩΝΙΚΗ ΑΝΤΙΛΗΨΗ",
        "ΝΑΥΤΙΛΙΑΚΕΣ ΣΗΜΑΝΣΕΙΣ",
        "ΔΙΕΘΝΕΣ ΠΟΙΝΙΚΟ ΔΙΚΑΙΟ",
        "ΑΣΦΑΛΙΣΤΙΚΟΙ ΟΡΓΑΝΙΣΜΟΙ Ε.Ν",
        "ΣΩΜΑΤΙΚΗ ΑΓΩΓΗ",
        "ΣΠΟΡΟΠΑΡΑΓΩΓΗ",
        "ΥΠΗΡΕΣΙΑΙ ΔΗΜΟΣΙΩΝ ΕΡΓΩΝ",
        "ΤΑΜΕΙΑ ΣΥΝΤΑΞΕΩΝ ΤΡΑΠΕΖΩΝ",
        "ΠΥΡΟΣΒΕΣΤΙΚΟ ΣΩΜΑ",
        "ΔΙΑΦΟΡΕΣ ΒΙΟΜΗΧΑΝΙΕΣ",
        "ΕΚΤΕΛΕΣΗ ΚΑΙ ΣΥΝΕΠΕΙΕΣ ΤΗΣ ΠΟΙΝΗΣ",
        "ΔΙΕΘΝΕΙΣ ΑΣΦΑΛΙΣΤΙΚΕΣ ΣΥΜΒΑΣΕΙΣ",
        "ΝΟΜΟΘΕΣΙΑ",
        "ΒΑΜΒΑΚΙ",
        "ΠΑΡΟΧΕΣ ΣΥΝΤΑΞΕΩΝ",
        "ΝΟΜΙΣΜΑ",
        "ΣΥΜΒΑΣΗ ΝΑΥΤΙΚΗΣ ΕΡΓΑΣΙΑΣ",
        "ΟΡΓΑΝΙΣΜΟΊ ΚΟΙΝΩΝΙΚΉΣ ΑΣΦΑΛΊΣΕΩΣ",
        "ΑΓΡΟΤΙΚΗ ΑΣΦΑΛΕΙΑ",
        "ΥΓΕΙΟΝΟΜΙΚΟΣ ΕΛΕΓΧΟΣ ΕΙΣΕΡΧΟΜΕΝΩΝ",
        "ΜΟΥΣΕΙΑ ΚΑΙ ΣΥΛΛΟΓΕΣ",
        "ΠΡΟΣΩΠΙΚΟ Ι.Κ.Α",
        "ΞΕΝΟΔΟΧΕΙΑ",
        "ΚΡΑΤΙΚΗ ΑΣΦΑΛΕΙΑ",
        "ΣΥΝΕΤΑΙΡΙΣΜΟΙ",
        "ΠΟΛΥΕΘΝΕΙΣ ΣΥΜΦΩΝΙΕΣ",
        "ΕΤΕΡΟΔΟΞΟΙ",
        "ΜΕΣΗ ΕΚΠΑΙΔΕΥΣΙΣ",
        "ΓΕΩΡΓΙΚΟΙ ΟΡΓΑΝΙΣΜΟΙ",
        "ΓΕΝΙΚΟ ΛΟΓΙΣΤΗΡΙΟ",
        "ΡΥΘΜΙΣΗ ΤΗΣ ΑΓΟΡΑΣ ΕΡΓΑΣΙΑΣ",
        "ΠΑΡΟΧΟΙ ΚΙΝΗΤΩΝ ΤΗΛΕΠΙΚΟΙΝΩΝΙΩΝ",
        "ΕΜΠΡΑΓΜΑΤΟΣ ΑΣΦΑΛΕΙΑ",
        "ΦΟΡΟΛΟΓΙΑ ΑΚΑΘΑΡΙΣΤΟΥ ΠΡΟΣΟΔΟΥ",
        "ΚΤΗΜΑΤΙΚΕΣ ΤΡΑΠΕΖΕΣ",
        "ΣΤΑΤΙΣΤΙΚΗ",
        "ΚΕΡΑΙΕΣ – ΣΤΑΘΜΟΙ ΚΕΡΑΙΩΝ",
        "ΠΟΙΝΙΚΟΣ ΝΟΜΟΣ",
        "ΜΕΣΑ ΔΙΔΑΣΚΑΛΙΑΣ",
        "ΕΜΠΟΡΙΟ ΦΑΡΜΑΚΩΝ",
        "ΔΙΑΦΟΡΑ",
        "ΔΗΜΟΣΙΑ ΚΤΗΜΑΤΑ",
        "ΕΙΣΦΟΡΕΣ Ι.Κ.Α",
        "ΚΑΤΑΓΓΕΛΙΑ ΣΥΜΒΑΣΕΩΣ ΕΡΓΑΣΙΑΣ",
        "ΠΡΟΣΩΠΙΚΟ ΠΟΛΙΤΙΚΗΣ ΑΕΡΟΠΟΡΙΑΣ",
        "ΔΗΜΟΣΙΟ ΧΡΕΟΣ",
        "ΑΠΟΤΑΜΙΕΥΣΗ",
        "ΑΛΛΟΘΡΗΣΚΟΙ",
        "ΠΛΟΗΓΙΚΗ ΥΠΗΡΕΣΙΑ",
        "ΤΥΠΟΣ ΚΑΙ ΠΛΗΡΟΦΟΡΙΕΣ",
        "ΤΡΟΠΟΠΟΙΗΣΗ ΚΑΙ ΚΑΤΑΡΓΗΣΗ ΤΗΣ ΠΟΙΝΗΣ",
        "ΑΣΦΑΛΙΣΤΙΚΑ ΤΑΜΕΙΑ ΤΥΠΟΥ",
        "ΟΙΚΟΓΕΝΕΙΑΚΟ ΔΙΚΑΙΟ",
        "ΔΙΟΙΚΗΣΗ ΕΘΝΙΚΗΣ ΟΙΚΟΝΟΜΙΑΣ",
        "ΥΠΟΥΡΓΕΙΟ ΕΘΝΙΚΗΣ ΑΜΥΝΑΣ",
        "ΔΙΕΘΝΕΙΣ ΣΥΜΒΑΣΕΙΣ ΠΡΟΝΟΙΑΣ",
        "ΠΡΟΣΩΠΙΚΟ ΤΩΝ ΔΙΚΑΣΤΗΡΙΩΝ",
        "ΠΡΟΣΤΑΣΙΑ ΠΡΟΣΩΠΩΝ ΕΙΔΙΚΩΝ ΚΑΤΗΓΟΡΙΩΝ",
        "ΠΑΡΟΧΕΣ ΑΣΘΕΝΕΙΑΣ",
        "ΜΕΤΑΝΑΣΤΕΥΣΗ",
        "ΥΠΟΥΡΓΕΙΟ ΠΑΙΔΕΙΑΣ",
        "ΑΣΦΑΛΕΙΑ ΝΑΥΣΙΠΛΟΪΑΣ",
        "ΟΔΟΠΟΙΪΑ",
        "ΣΤΡΑΤΟΔΙΚΕΙΑ",
        "ΜΙΣΘΩΣΗ",
        "ΕΙΣΠΡΑΞΗ ΔΗΜΟΣΙΩΝ ΕΣΟΔΩΝ",
        "ΟΠΛΙΤΕΣ ΚΑΙ ΑΝΘΥΠΑΣΠΙΣΤΕΣ",
        "ΟΡΓΑΝΙΣΜΟΣ ΤΗΛΕΠΙΚΟΙΝΩΝΙΩΝ ΕΛΛΑΔΑΣ (Ο.Τ.Ε.)",
        "ΌΡΓΑΝΑ ΆΣΚΗΣΗΣ ΔΙΑΧΕΙΡΙΣΤΙΚΟΎ ΕΛΈΓΧΟΥ ΟΡΓΑΝΙΣΜΏΝ ΚΑΙ ΕΠΙΧΕΙΡΉΣΕΩΝ",
        "ΠΟΙΝΙΚΗ ΝΟΜΟΘΕΣΙΑ ΤΥΠΟΥ",
        "ΕΞΑΓΩΓΙΚΟ ΕΜΠΟΡΙΟ",
        "ΑΕΡΟΠΟΡΙΚΕΣ ΣΥΜΒΑΣΕΙΣ",
        "ΓΕΩΡΓΙΚΟΙ ΣΥΝΕΤΑΙΡΙΣΜΟΙ ΑΓΡΟΤΙΚΕΣ ΣΥΝΕΤΑΙΡΙΣΤΙΚΕΣ ΟΡΓΑΝΩΣΕΙΣ",
        "ΟΙΚΟΝΟΜΙΚΕΣ ΥΠΗΡΕΣΙΕΣ",
        "ΟΧΥΡΩΣΕΙΣ",
        "ΕΚΤΑΚΤΟΙ ΠΟΙΝΙΚΟΙ ΝΟΜΟΙ",
        "ΕΚΤΕΛΕΣΗ",
        "ΔΙΟΙΚΗΤΙΚΟΙ ΚΑΝΟΝΙΣΜΟΙ",
        "ΥΔΡΑΥΛΙΚΑ ΕΡΓΑ",
        "ΑΣΦΑΛΙΣΤΙΚΑ ΤΑΜΕΙΑ ΔΗΜΟΣΙΩΝ ΥΠΑΛΛΗΛΩΝ",
        "ΕΚΚΑΘΑΡΙΣΕΙΣ ΔΗΜΟΣΙΩΝ ΥΠΑΛΛΗΛΩΝ",
        "ΔΙΟΙΚΗΣΗ ΕΜΠΟΡΙΚΟΥ ΝΑΥΤΙΚΟΥ",
        "ΑΝΩΤΑΤΟ ΕΙΔΙΚΟ ΔΙΚΑΣΤΗΡΙΟ",
        "ΑΡΤΟΣ",
        "ΕΙΣΑΓΩΓΙΚΟ ΕΜΠΟΡΙΟ",
        "ΑΛΙΕΙΑ",
        "ΕΚΚΛΗΣΙΑΣΤΙΚΗ ΠΕΡΙΟΥΣΙΑ",
        "ΔΙΑΦΟΡΑ ΔΗΜΟΣΙΑ ΕΡΓΑ",
        "ΜΟΝΕΣ",
        "ΠΡΟΕΔΡΟΣ ΤΗΣ ΔΗΜΟΚΡΑΤΙΑΣ ΚΑΙ ΠΡΟΕΔΡΙΑ ΤΗΣ ΔΗΜΟΚΡΑΤΙΑΣ",
        "ΠΟΛΥΕΘΝΕΙΣ ΟΡΓΑΝΙΣΜΟΙ",
        "ΑΡΧΑΙΟΤΗΤΕΣ",
        "ΝΑΟΙ ΚΑΙ ΛΕΙΤΟΥΡΓΟΙ ΑΥΤΩΝ",
        "ΕΚΚΛΗΣΙΑΣΤΙΚΗ ΕΚΠΑΙΔΕΥΣΗ",
        "ΕΝΙΣΧΥΣΙΣ ΤΗΣ ΓΕΩΡΓΙΑΣ",
        "ΕΚΘΕΣΕΙΣ",
        "ΠΡΟΣΤΑΣΙΑ ΤΩΝ ΣΥΝΑΛΛΑΓΩΝ",
        "ΑΣΦΑΛΙΣΗ",
        "ΚΤΗΝΟΤΡΟΦΙΑ",
        "ΕΚΠΑΙΔΕΥΤΙΚΑ ΤΕΛΗ",
        "ΔΙΟΙΚΗΣΗ ΕΚΠΑΙΔΕΥΣΕΩΣ",
        "ΤΑΜΕΙΟ ΠΑΡΑΚΑΤΑΘΗΚΩΝ ΚΑΙ ΔΑΝΕΙΩΝ",
        "ΑΓΑΘΟΕΡΓΑ ΙΔΡΥΜΑΤΑ",
        "ΦΟΡΟΛΟΓΙΚΑ ΔΙΚΑΣΤΗΡΙΑ",
        "ΦΟΡΟΙ ΚΑΤΑΝΑΛΩΣΕΩΣ",
        "ΒΙΒΛΙΟΘΗΚΕΣ-ΠΡΟΣΤΑΣΙΑ ΒΙΒΛΙΟΥ-ΔΙΑΔΟΣΗ ΛΟΓΟΤΕΧΝΙΑΣ",
        "ΤΗΛΕΠΙΚΟΙΝΩΝΙΑΚΕΣ ΚΑΙ ΤΑΧΥΔΡΟΜΙΚΕΣ ΣΥΜΒΑΣΕΙΣ",
        "ΙΔΙΩΤΙΚΗ ΕΚΠΑΙΔΕΥΣΗ",
        "ΤΗΛΕΠΙΚΟΙΝΩΝΙΕΣ",
        "ΑΣΥΡΜΑΤΟΣ",
        "ΑΠΟΔΟΧΕΣ ΔΗΜΟΣΙΩΝ ΥΠΑΛΛΗΩΝ",
        "ΥΓΕΙΟΝΟΜΙΚΗ ΥΠΗΡΕΣΙΑ ΣΤΡΑΤΟΥ",
        "ΦΑΡΜΑΚΕΙΑ",
        "ΔΗΜΟΣΙΟ ΛΟΓΙΣΤΙΚΟ",
        "ΝΑΥΤΙΚΗ ΕΚΠΑΙΔΕΥΣΗ",
        "ΕΞΥΠΗΡΕΤΗΣΗ ΠΟΛΙΤΙΚΗΣ ΑΕΡΟΠΟΡΙΑΣ",
        "ΠΑΡΟΧΕΣ Ι.Κ.Α",
        "ΓΕΝΙΚΑ ΥΓΕΙΟΝΟΜΙΚΑ ΜΕΤΡΑ",
        "ΕΚΜΕΤΑΛΛΕΥΣΗ ΘΑΛΑΣΣΙΩΝ ΣΥΓΚΟΙΝΩΝΙΩΝ",
        "ΠΡΟΣΩΠΙΚΟ ΤΑΧΥΔΡΟΜΕΙΩΝ",
        "ΕΚΤΕΛΕΣΤΙΚΗ ΕΞΟΥΣΙΑ",
        "ΣΥΣΤΑΣΗ ΚΑΙ ΕΔΡΑ ΤΟΥ ΚΡΑΤΟΥΣ",
        "ΦΟΡΟΛΟΓΙΑ ΔΙΑΣΚΕΔΑΣΕΩΝ",
        "ΤΗΛΕΦΩΝΑ",
        "ΣΤΡΑΤΟΛΟΓΙΑ",
        "ΕΚΠΑΙΔΕΥΣΗ ΕΡΓΑΤΩΝ",
        "ΥΠΟΥΡΓΕΙΟ ΠΟΛΙΤΙΣΜΟΥ",
        "ΦΟΡΟΛΟΓΙΑ ΟΙΝΟΠΝΕΥΜΑΤΩΔΩΝ ΠΟΤΩΝ",
        "ΥΠΟΥΡΓΕΙΟ ΓΕΩΡΓΙΑΣ",
        "ΣΩΜΑΤΕΙΑ",
        "ΕΙΔΙΚΕΣ ΜΟΡΦΕΣ ΑΠΑΣΧΟΛΗΣΗΣ",
        "ΥΠΟΥΡΓΕΙΟ ΔΙΚΑΙΟΣΥΝΗΣ",
        "ΝΑΥΤΙΛΙΑΚΟΙ ΟΡΓΑΝΙΣΜΟΙ",
        "ΤΟΥΡΙΣΜΟΣ",
        "ΚΑΠΝΟΣ",
        "ΠΡΟΣΤΑΣΙΑ ΗΘΩΝ",
        "ΕΙΔΙΚΕΣ ΥΠΗΡΕΣΙΕΣ ΝΑΥΤΙΚΟΥ",
        "ΑΠΟΔΟΧΕΣ ΣΤΡΑΤΙΩΤΙΚΩΝ",
        "ΠΡΟΝΟΙΑ ΠΛΗΡΩΜΑΤΩΝ Ε.Ν",
        "ΕΙΔΙΚΕΣ ΔΙΑΤΑΞΕΙΣ ΠΕΡΙ ΑΝΩΝ.ΕΤΑΙΡΕΙΩΝ",
        "ΔΗΜΟΣΙΑ ΔΙΟΙΚΗΣΗ",
        "ΤΟΠΙΚΑ ΣΧΕΔΙΑ ΠΟΛΕΩΝ",
        "ΠΡΟΣΤΑΣΙΑ ΠΑΙΔΙΚΗΣ ΗΛΙΚΙΑΣ",
        "ΕΛΛΗΝΙΚΗ ΑΣΤΥΝΟΜΙΑ",
        "ΛΙΜΕΝΙΚΟ ΣΩΜΑ",
        "ΤΟΥΡΙΣΤΙΚΗ ΑΣΤΥΝΟΜΙΑ",
        "ΒΙΟΜΗΧΑΝΙΑ",
        "ΣΧΟΛΕΣ ΠΑΝΕΠΙΣΤΗΜΙΟΥ ΑΘΗΝΩΝ",
        "ΑΣΦΑΛΙΣΤΙΚΟΙ ΟΡΓΑΝΙΣΜΟΙ ΣΤΡΑΤΟΥ",
        "ΑΛΥΚΕΣ",
        "ΕΣΩΤΕΡΙΚΟ ΕΜΠΟΡΙΟ",
        "ΕΘΝΙΚΟ ΣΥΣΤΗΜΑ ΥΓΕΙΑΣ",
        "ΝΟΜΟΘΕΤΙΚΗ ΕΞΟΥΣΙΑ",
        "ΔΙΟΙΚΗΣH ΚΟΙΝΩΝIKΗΣ ΠΡΟΝΟΙΑΣ",
        "ΠΛΗΡΩΜΑΤΑ",
        "ΜΑΘΗΤΙΚΗ ΠΡΟΝΟΙΑ",
        "ΔΙΟΙΚΗΣΗ ΤΥΠΟΥ ΚΑΙ ΤΟΥΡΙΣΜΟΥ",
        "ΕΠΟΙΚΙΣΜΟΣ",
        "ΤΡΟΧΙΟΔΡΟΜΟΙ",
        "ΕΠΑΓΓΕΛΜΑΤΙΚΗ ΕΚΠΑΙΔΕΥΣΗ",
        "ΑΕΡΟΠΟΡΙΚΗ ΕΚΠΑΙΔΕΥΣΗ",
        "ΥΠΟΥΡΓΕΙΟ ΕΘΝΙΚΗΣ ΟΙΚΟΝΟΜΙΑΣ",
        "ΘΕΑΤΡΟ",
        "ΥΔΡΕΥΣΗ",
        "ΔΙΕΘΝΕΙΣ ΣΤΡΑΤΙΩΤΙΚΕΣ ΣΥΜΒΑΣΕΙΣ",
        "ΕΘΝΙΚΟ ΜΕΤΣΟΒΙΟ ΠΟΛΥΤΕΧΝΕΙΟ",
        "ΥΠΟΥΡΓΕΙΟ ΕΞΩΤΕΡΙΚΩΝ",
        "ΕΥΡΩΠΑΪΚΟΙ ΠΟΛΥΕΘΝΕΙΣ ΟΡΓΑΝΙΣΜΟΙ",
        "ΕΛΕΥΘΕΡΙΑ ΤΗΣ ΕΡΓΑΣΙΑΣ",
        "ΥΠΟΥΡΓΕΙΟ ΕΣΩΤΕΡΙΚΩΝ ΔΗΜ.ΔΙΟΙΚΗΣΗΣ ΚΑΙ ΑΠΟΚΕΝΤΡΩΣΗΣ",
        "ΔΙΑΦΟΡΕΣ ΕΝΟΧΙΚΕΣ ΣΧΕΣΕΙΣ",
        "ΛΗΞΙΑΡΧΕΙΑ",
        "ΕΙΔΙΚΟΙ ΚΑΝΟΝΙΣΜΟΙ",
        "ΤΕΛΩΝΕΙΑΚΕΣ ΣΥΜΒΑΣΕΙΣ",
        "ΝΑΥΤΙΚΟ ΠΟΙΝΙΚΟ ΔΙΚΑΙΟ",
        "ΣΤΕΓΑΣΗ ΔΗΜΟΣΙΩΝ ΥΠΗΡΕΣΙΩΝ",
        "ΠΛΗΡΩΜΑΤΑ ΠΟΛΕΜΙΚΟΥ ΝΑΥΤΙΚΟΥ",
        "ΣΥΝΤΑΓΜΑΤΙΚΟΣ ΧΑΡΤΗΣ",
        "ΗΛΕΚΤΡΙΣΜΟΣ",
        "ΑΣΦΑΛΙΣΤΙΚΑ ΔΙΚΑΣΤΗΡΙΑ",
        "ΛΕΣΧΕΣ ΕΛΛΗΝΙΚΗΣ ΑΣΤΥΝΟΜΙΑΣ",
        "ΥΠΟΥΡΓΕΙΟ ΔΗΜΟΣΙΑΣ TAΞΗΣ",
        "ΕΚΤΕΛΕΣ ΔΗΜΟΣΙΩΝ ΕΡΓΩΝ",
        "ΠΑΝΕΠΙΣΤΗΜΙΟ ΘΕΣΣΑΛΟΝΙΚΗΣ",
        "ΔΑΣΙΚΗ ΝΟΜΟΘΕΣΙΑ",
        "ΕΙΔΙΚΕΣ ΑΝΩΤΑΤΕΣ ΣΧΟΛΕΣ",
        "ΕΔΑΦΟΣ ΤΟΥ ΕΛΛΗΝΙΚΟΥ ΚΡΑΤΟΥΣ",
        "ΔΙΚΗΓΟΡΟΙ",
        "ΔΙΚΑΙΟ ΤΩΝ ΠΡΟΣΩΠΩΝ",
        "ΔΙΟΙΚΗΣΗ ΤΑΧΥΔΡΟΜΙΚΗΣ, ΤΗΛΕΓΡΑΦΙΚΗΣ",
        "ΣΧΟΛΙΚΑ ΚΤΙΡΙΑ ΚΑΙ ΤΑΜΕΙΑ",
        "ΑΕΡΟΛΙΜΕΝΕΣ",
        "ΥΠΟΘΗΚΟΦΥΛΑΚΕΙΑ",
        "ΑΣΦΑΛΙΣΤΙΚΑ ΤΑΜΕΙΑ ΠΡΟΣΩΠΙΚΟΥ ΥΠΟΥΡΓΕΙΟΥ ΔΗΜΟΣΙΑΣ ΤΑΞΗΣ",
        "ΔΙΑΧΕΙΡΙΣΕΙΣ ΤΟΥ ΔΗΜΟΣΙΟΥ",
        "ΕΜΠΡΑΓΜΑΤΟ ΔΙΚΑΙΟ",
        "ΦΟΡΤΟΕΚΦΟΡΤΩΣΕΙΣ",
        "ΑΝΩΝΥΜΕΣ ΕΤΑΙΡΕΙΕΣ",
        "ΕΙΔΙΚΟΙ ΕΠΙΣΙΤΙΣΤΙΚΟΙ ΝΟΜΟΙ",
        "ΕΚΚΛΗΣΙΕΣ ΑΛΛΟΔΑΠΗΣ",
        "ΔΙΕΘΝΕΙΣ ΣΥΜΒΑΣΕΙΣ",
        "ΟΡΓΑΝΙΣΜΟΣ ΑΣΦΑΛΙΣΗΣ ΕΛΕΥΘΕΡΩΝ ΕΠΑΓΓΕΛΜΑΤΙΩΝ",
        "ΑΣΦΑΛΕΙΑ ΑΕΡΟΠΛΟΪΑΣ",
        "ΤΑΜΕΙΑ ΑΣΦΑΛΙΣΕΩΣ ΚΑΙ ΑΡΩΓΗΣ",
        "ΑΝΩΤΑΤΗ ΕΚΠΑΙΔΕΥΣΗ",
        "ΠΟΛΕΜΙΚΗ ΔΙΑΘΕΣΙΜΟΤΗΤΑ",
        "ΠΟΙΝΙΚΟ ΚΑΙ ΠΕΙΘΑΡΧΙΚΟ ΔΙΚΑΙΟ",
        "ΦΟΡΟΛΟΓΙΑ ΕΠΙΤΗΔΕΥΜΑΤΟΣ",
        "ΕΚΤΑΚΤΕΣ ΦΟΡΟΛΟΓΙΕΣ",
        "ΠΟΙΝΙΚΗ ΔΙΚΟΝΟΜΙΑ",
        "ΣΤΟΙΧΕΙΩΔΗΣ ΕΚΠΑΙΔΕΥΣΗ",
        "ΣΥΜΒΟΥΛΙΟ ΕΠΙΚΡΑΤΕΙΑΣ ΚΑΙ ΔΙΟΙΚΗΤΙΚΑ ΔΙΚΑΣΤΗΡΙΑ",
        "ΝΟΜΙΚΑ ΠΡΟΣΩΠΑ ΚΑΙ ΕΚΜΕΤΑΛΛΕΥΣΕΙΣ",
        "ΟΙΚΟΝΟΜΙΚΗ ΔΙΟΙΚΗΣΗ ΝΑΥΤΙΚΟΥ",
        "ΤΥΠΟΣ",
        "ΤΑΜΕΙΟ ΑΣΦΑΛΙΣΕΩΣ ΕΠΑΓΓΕΛΜΑΤΙΩΝ",
        "ΠΑΝΕΠΙΣΤΗΜΙΟ ΙΩΑΝΝΙΝΩΝ",
        "ΧΡΕΩΓΡΑΦΑ",
        "ΠΡΟΪΟΝΤΑ ΕΛΑΙΑΣ",
        "ΕΚΚΛΗΣΙΑ ΙΟΝΙΩΝ ΝΗΣΩΝ",
        "ΔΙΟΙΚΗΣH ΥΓΙΕΙΝΗΣ",
        "ΑΕΡΟΠΟΡΙΚΟ ΠΟΙΝΙΚΟ ΔΙΚΑΙΟ",
        "ΚΑΤΑΠΟΛΕΜΗΣΗ ΝΟΣΩΝ ΚΑΤ’ ΙΔΙΑΝ",
        "ΕΙΔΙΚΟΙ ΠΟΙΝΙΚΟΙ ΝΟΜΟΙ",
        "ΘΗΡΑ",
        "ΥΓΙΕΙΝΗ ΚΑΙ ΑΣΦΑΛΕΙΑ ΕΡΓΑΖΟΜΕΝΩΝ",
        "ΔΙΟΙΚΗΣΗ ΣΥΓΚΟΙΝΩΝΙΩΝ",
        "ΑΠΟΣΤΟΛΙΚΗ ΔΙΑΚΟΝΙΑ ΕΚΚΛΗΣΙΑΣ ΤΗΣ ΕΛΛΑΔΟΣ",
        "ΠΡΟΣΩΡΙΝΕΣ ΑΤΕΛΕΙΕΣ",
        "ΤΑΧΥΔΡΟΜΙΚΑ ΤΑΜΙΕΥΤΗΡΙΑ",
        "ΑΝΩΤΑΤΗ ΣΧΟΛΗ ΚΑΛΩΝ ΤΕΧΝΩΝ",
        "ΔΙΟΙΚΗΣΗ ΕΡΓΑΣΙΑΣ",
        "ΑΓΙΟΝ ΟΡΟΣ",
        "ΣΧΟΛΕΣ Π. ΝΑΥΤΙΚΟΥ",
        "ΤΡΑΠΕΖΕΣ",
        "ΕΛΕΓΧΟΣ ΚΙΝΗΣΕΩΣ ΜΕ ΤΟ ΕΞΩΤΕΡΙΚΟ",
        "ΕΙΔΙΚΑΙ ΚΑΤΗΓΟΡΙΑΙ ΠΛΟΙΩΝ",
        "ΓΕΩΡΓΙΚΗ ΥΓΙΕΙΝΗ",
        "ΕΞΟΔΑ ΠΟΙΝΙΚΗΣ ΔΙΑΔΙΚΑΣΙΑΣ",
        "ΕΡΓΑΣΙΑ ΓΥΝΑΙΚΩΝ ΚΑΙ ΑΝΗΛΙΚΩΝ",
        "ΔΙΟΙΚΗΣΗ ΕΦΟΔΙΑΣΜΟΥ",
        "ΕΜΠΟΡΙΚΑ ΕΠΑΓΓΕΛΜΑΤΑ",
        "ΕΚΤΕΛΩΝΙΣΤΕΣ",
        "ΦΟΡΟΛΟΓΙΑ ΚΛΗΡΟΝΟΜΙΩΝ, ΔΩΡΕΩΝ ΚΛΠ",
        "ΟΡΓΑΝΙΣΜΟΙ ΥΠΟΥΡΓΕΙΟΥ ΕΡΓΑΣΙΑΣ",
        "ΕΝΙΣΧΥΣΗ ΕΠΙΣΤΗΜΩΝ ΚΑΙ ΤΕΧΝΩΝ",
        "ΔΙΑΦΟΡΟΙ ΦΟΡΟΛΟΓΙΚΟΙ ΝΟΜΟΙ",
        "ΤΕΧΝΙΚΕΣ ΠΡΟΔΙΑΓΡΑΦΕΣ",
        "ΜΗΤΡΩΑ ΔΗΜΟΤΩΝ",
        "ΚΑΤΑΣΤΑΣΗ ΔΗΜΟΣΙΩΝ ΥΠΑΛΛΗΛΩΝ",
        "ΠΡΟΣΩΠΙΚΟΝ ΔΗΜΩΝ ΚΑΙ ΚΟΙΝΟΤΗΤΩΝ",
        "ΥΓΕΙΟΝΟΜΙΚΗ ΑΝΤΙΛΗΨΗ",
        "ΤΕΛΗ ΧΑΡΤΟΣΗΜΟΥ",
        "ΣΤΡΑΤΙΩΤΙΚΟΙ ΓΕΝΙΚΑ",
        "ΛΙΜΕΝΙΚΕΣ ΑΡΧΕΣ",
        "ΕΛΕΓΧΟΣ ΚΥΚΛΟΦΟΡΙΑΣ",
        "ΤΑΜΕΙΟ ΣΥΝΤΑΞΕΩΣ ΚΑΙ ΑΥΤΑΣΦΑΛΙΣΕΩΣ ΥΓΕΙΟΝΟΜΙΚΩΝ",
        "ΠΟΛΙΤΙΚΗ ΚΑΙ ΟΙΚΟΝΟΜΙΚΗ ΕΠΙΣΤΡΑΤΕΥΣΗ",
        "ΤΗΛΕΓΡΑΦΟΙ",
        "ΣΕΙΣΜΟΠΛΗΚΤΟΙ",
        "ΙΑΜΑΤΙΚΕΣ ΠΗΓΕΣ",
        "ΙΔΙΩΤΙΚΟ ΝΑΥΤΙΚΟ ΔΙΚΑΙΟ",
        "ΔΙΕΘΝΕΙΣ ΥΓΕΙΟΝΟΜΙΚΕΣ ΣΥΜΒΑΣΕΙΣ",
        "ΝΟΜΙΚΑ ΠΡΟΣΩΠΑ ΔΗΜΟΣΙΟΥ ΔΙΚΑΙΟΥ",
        "ΕΚΚΛΗΣΙΑ ΚΡΗΤΗΣ",
        "ΠΡΟΣΤΑΣΙΑ ΝΟΜΙΣΜΑΤΟΣ",
        "ΠΡΟΣΤΑΣΙΑ ΠΡΟΪΟΝΤΩΝ ΑΜΠΕΛΟΥ",
        "ΑΝΑΠΗΡΟΙ ΚΑΙ ΘΥΜΑΤΑ ΠΟΛΕΜΟΥ",
        "ΠΑΡΟΧΕΣ ΔΙΑΦΟΡΕΣ",
        "ΤΟΠΙΚΗ ΑΥΤΟΔΙΟΙΚΗΣΗ",
        "OΡΓΑΝΩΣΗ ΣΤΡΑΤΟΥ ΞΗΡΑΣ",
        "ΔΙΑΚΟΠΕΣ ΤΗΣ ΕΡΓΑΣΙΑΣ",
        "ΟΡΓΑΝΙΣΜΟΣ ΠΟΛΕΜΙΚΗΣ ΑΕΡΟΠΟΡΙΑΣ",
        "ΕΠΙΜΕΛΗΤΗΡΙΑ",
        "ΕΚΚΛΗΣΙΑ ΤΗΣ ΕΛΛΑΔΟΣ",
        "ΝΑΡΚΩΤΙΚΑ",
        "ΕΚΜΕΤΑΛΛΕΥΣΗ ΤΑΧΥΔΡΟΜΕΙΩΝ",
        "ΜΟΥΣΙΚΗ",
        "ΝΟΜΑΡΧΙΕΣ",
        "ΠΡΟΣΩΠΙΚΟ ΠΟΛΕΜΙΚΟΥ ΝΑΥΤΙΚΟΥ",
        "ΓΕΝΙΚΟ ΧΗΜΕΙΟ ΤΟΥ ΚΡΑΤΟΥΣ",
        "ΚΡΑΤΙΚΗ",
        "ΔΙΟΙΚΗΣΗ ΠΟΛΕΜΙΚΟΥ ΝΑΥΤΙΚΟΥ",
        "ΠΑΡΟΧΟΙ ΣΤΑΘΕΡΩΝ ΗΛΕΚΤΡΟΝΙΚΩΝ ΕΠΙΚΟΙΝΩΝΙΩΝ",
        "ΕΠΑΓΓΕΛΜΑΤΙΚΟΣ ΚΙΝΔΥΝΟΣ",
        "ΕΝΟΧΕΣ ΣΕ ΧΡΥΣΟ ΚΑΙ ΣΥΝΑΛΛΑΓΜΑ",
        "ΙΠΠΟΠΑΡΑΓΩΓΗ",
        "ΑΥΤΟΚΙΝΗΤΑ",
        "ΑΓΟΡΑΝΟΜΙΚΕΣ ΔΙΑΤΑΞΕΙΣ",
        "ΠΡΟΣΦΥΓΕΣ",
        "ΔΙΑΦΟΡΑ ΣΤΡΑΤΙΩΤΙΚΑ ΘΕΜΑΤΑ",
        "ΓΕΝ. ΓΡΑΜΜ. ΒΙΟΜΗΧΑΝΙΑΣ - ΓΕΝ. ΓΡΑΜΜ. ΕΡΕΥΝΑΣ ΚΑΙ ΤΕΧΝΟΛΟΓΙΑΣ",
        "ΔΙΑΜΕΤΑΚΟΜΙΣΗ",
        "ΔΙΚΑΙΟΣΤΑΣΙΟ",
        "ΥΔΑΤΑ",
        "ΦΟΡΟΛΟΓΙΚΕΣ ΔΙΕΥΚΟΛΥΝΣΕΙΣ ΚΑΙ ΑΠΑΛΛΑΓΕΣ",
        "ΜΟΝΟΠΩΛΙΑ",
        "ΕΙΔΙΚΕΣ ΔΙΑΔΙΚΑΣΙΕΣ",
        "ΠΡΟΝΟΙΑ ΓΙΑ ΤΟΥΣ ΣΤΡΑΤΙΩΤΙΚΟΥΣ",
        "ΠΟΛΙΤΙΚΗ ΔΙΚΟΝΟΜΙΑ",
        "ΟΡΓΑΝΩΣΗ ΧΡΟΝΟΥ ΕΡΓΑΣΙΑΣ",
        "ΠΡΟΣΩΠΙΚΟ ΤΥΠΟΥ",
        "ΔΙΚΑΣΤΙΚΟΙ ΕΠΙΜΕΛΗΤΕΣ",
        "ΛΟΥΤΡΟΠΟΛΕΙΣ",
        "ΤΕΛΩΝΕΙΑΚΟΣ ΚΩΔΙΚΑΣ",
        "ΤΑΜΕΙΟ ΣΥΝΤΑΞΕΩΝ ΝΟΜΙΚΩΝ",
        "ΔΙΑΦΟΡΟΙ ΤΕΛΩΝΕΙΑΚΟΙ ΝΟΜΟΙ",
        "ΔΙΟΙΚΗΣΗ ΠΟΛΙΤΙΚΗΣ ΑΕΡΟΠΟΡΙΑΣ",
        "ΑΕΡΟΠΟΡΙΚΕΣ ΕΚΜΕΤΑΛΛΕΥΣΕΙΣ",
        "ΕΜΠΟΡΙΚΕΣ ΠΡΑΞΕΙΣ",
        "ΔΙΚΑΣΤΗΡΙΑ",
        "ΒΑΣΙΛΕΙΑ ΚΑΙ ΑΝΤΙΒΑΣΙΛΕΙΑ",
        "ΠΡΟΣΩΠΙΚΟ ΠΟΛΕΜΙΚΗΣ ΑΕΡΟΠΟΡΙΑΣ",
        "ΠΡΟΣΤΑΣΙΑ ΚΑΙ ΚΙΝΗΤΡΑ ΙΔΙΩΤΙΚΩΝ ΕΠΕΝΔΥΣΕΩΝ",
        "ΒΑΣΙΛΙΚΑ ΙΔΡΥΜΑΤΑ",
        "ΣΙΔΗΡΟΔΡΟΜΟΙ ΓΕΝΙΚΑ",
        "ΠΝΕΥΜΑΤΙΚΗ ΙΔΙΟΚΤΗΣΙΑ",
        "ΔΙΑΦΟΡΑ ΑΣΦΑΛΙΣΤΙΚΑ ΤΑΜΕΙΑ",
        "ΥΓΕΙΟΝΟΜΙΚΑ ΕΠΑΓΓΕΛΜΑΤΑ",
        "ΦΟΡΟΛΟΓΙΑ ΚΑΠΝΟΥ",
        "ΟΙΚΟΝΟΜΙΚΗ ΔΙΟΙΚΗΣΗ",
        "ΧΩΡΟΦΥΛΑΚΗ",
        "ΤΕΛΩΝΕΙΑΚΗ ΥΠΗΡΕΣΙΑ",
        "ΠΑΝΕΠΙΣΤΗΜΙΟ ΠΑΤΡΩΝ",
        "ΤΑΜΕΙΟ ΕΠΙΚΟΥΡΙΚΗΣ ΑΣΦΑΛΙΣΕΩΣ ΑΣΦΑΛΙΣΤΩΝ",
        "ΑΣΦΑΛΙΣΤΙΚΟΙ ΟΡΓΑΝΙΣΜΟΙ",
        "ΣΤΡΑΤΙΩΤΙΚΑ ΕΡΓΑ ΚΑΙ ΠΡΟΜΗΘΕΙΕΣ",
        "ΥΠΟΝΟΜΟΙ",
        "ΦΟΡΟΛΟΓΙΑ ΚΕΦΑΛΑΙΟΥ",
        "ΕΤΑΙΡΕΙΕΣ ΠΕΡΙΩΡΙΣΜΕΝΗΣ ΕΥΘΥΝΗΣ",
        "ΥΠΟΥΡΓΕΊΟ ΚΟΙΝΩΝΙΚΏΝ ΑΣΦΑΛΊΣΕΩΝ",
        "ΣΥΜΒΟΛΑΙΟΓΡΑΦΟΙ",
        "ΤΑΜΕΙΟ ΑΣΦΑΛΙΣΕΩΣ ΑΡΤΕΡΓΑΤΩΝ",
        "ΕΡΓΑ ΚΑΙ ΠΡΟΜΗΘΕΙΕΣ ΔΗΜΩΝ ΚΑΙ ΚΟΙΝΟΤΗΤΩΝ",
        "ΕΛΕΓΚΤΙΚΟ ΣΥΝΕΔΡΙΟ",
        "ΔΙΑΦΟΡΑ ΕΠΙΣΤΗΜΟΝΙΚΑ ΙΔΡΥΜΑΤΑ",
        "ΑΞΙΩΜΑΤΙΚΟΙ ΕΝΟΠΛΩΝ ΔΥΝΑΜΕΩΝ",
        "ΤΑΜΕΙΟ ΑΣΦΑΛΙΣΕΩΣ ΕΜΠΟΡΩΝ (Τ.Α.Ε)",
        "ΣΤΡΑΤΙΩΤΙΚΗ ΠΟΙΝΙΚΗ",
        "ΦΟΡΟΛΟΓΙΑ ΟΙΝΟΠΝΕΥΜΑΤΟΣ",
        "ΟΡΓΑΝΙΣΜΟΣ ΓΕΩΡΓΙΚΩΝ ΑΣΦΑΛΙΣΕΩΝ",
        "ΣΥΛΛΟΓΙΚΕΣ ΣΥΜΒΑΣΕΙΣ ΕΡΓΑΣΙΑΣ",
        "ΧΡΗΜΑΤΙΣΤΗΡΙΑ",
        "ΠΟΛΙΤΙΚΑΙ ΚΑΙ ΣΤΡΑΤΙΩΤΙΚΑΙ ΣΥΝΤΑΞΕΙΣ",
        "ΚΟΙΝΩΝΙΚΗ  ΣΤΕΓΑΣΤΙΚΗ ΣΥΝΔΡΟΜΗ",
        "ΚΑΤΟΧΥΡΩΣΗ ΕΠΑΓΓΕΛΜΑΤΩΝ",
        "ΦΟΡΟΛΟΓΙΑ ΚΑΘΑΡΑΣ ΠΡΟΣΟΔΟΥ",
        "ΠΕΡΙΦΕΡΕΙΕΣ",
        "ΕΚΚΛΗΣΙΑΣΤΙΚΗ ΔΙΚΑΙΟΣΥΝΗ",
        "ΥΠΟΥΡΓΕΙΟ ΟΙΚΟΝΟΜΙΚΩΝ",
        "ΕΘΝΙΚΑ ΚΛΗΡΟΔΟΤΗΜΑΤΑ",
        "ΕΓΓΕΙΟΒΕΛΤΙΩΤΙΚΑ ΕΡΓΑ",
        "ΛΙΜΕΝΕΣ",
        "ΦΥΛΑΚΕΣ",
        "ΓΕΩΡΓΙΚΗ ΕΚΠΑΙΔΕΥΣΗ",
        "ΠΛΗΡΩΜΗ ΕΡΓΑΣΙΑΣ",
        "ΕΜΠΟΡΙΚΟΣ ΝΟΜΟΣ",
        "ΙΔΡΥΜΑ ΚΟΙΝΩΝΙΚΩΝ ΑΣΦΑΛΙΣΕΩΝ",
        "ΑΣΦΑΛΙΣΤΙΚΑ ΤΑΜΕΙΑ ΤΡΑΠΕΖΩΝ",
        "ΕΙΔΙΚΟΙ ΑΓΡΟΤΙΚΟΙ ΝΟΜΟΙ",
        "ΔΙΕΘΝΕΙΣ ΔΙΚΟΝΟΜΙΚΕΣ ΣΥΜΒΑΣΕΙΣ",
        "ΥΠΟΥΡΓΕΙΑ ΜΑΚΕΔΟΝΙΑΣ–ΘΡΑΚΗΣ, ΑΙΓΑΙΟΥ Κ.Λ.Π",
        "ΑΣΤΥΝΟΜΙΚΟΊ ΣΚΎΛΟΙ",
        "ΔΙΑΦΟΡΑ ΘΕΜΑΤΑ",
        "ΕΚΔΟΣΗ ΕΓΚΛΗΜΑΤΙΩΝ",
        "ΑΓΟΡΑΝΟΜΙΑ",
        "ΔΙΚΑΣΤΙΚΟ ΤΟΥ ΔΗΜΟΣΙΟΥ",
        "ΑΣΤΙΚΟΣ ΚΩΔΙΚΑΣ",
        "ΤΕΛΩΝΕΙΑΚΕΣ ΑΤΕΛΕΙΕΣ",
        "ΑΓΡΟΤΙΚΕΣ ΜΙΣΘΩΣΕΙΣ",
        "ΛΕΩΦΟΡΕΙΑ",
        "ΓΕΝΙΚΟΙ ΕΠΙΣΙΤΙΣΤΙΚΟΙ ΝΟΜΟΙ",
        "ΑΣΤΥΝΟΜΙΑ ΠΟΛΕΩΝ",
        "ΜΗΧΑΝΙΚΟΙ ΚΑΙ ΕΡΓΟΛΑΒΟΙ",
        "ΠΟΛΕΜΙΚΕΣ ΣΥΝΤΑΞΕΙΣ",
    ],
    "subject": [
        "ΜΕΤΟΧΙΚΟ ΤΑΜΕΙΟ Π.Ν",
        "ΜΕΤΑΝΑΣΤΕΥΣΗ ΣΤΟ ΒΕΛΓΙΟ",
        "ΝΑΥΤΙΚΕΣ ΦΥΛΑΚΕΣ",
        "ΚΑΝΟΝΙΣΜΟΣ ΕΚΤΕΛΕΣΕΩΣ ΣΤΡΑΤΙΩΤΙΚΩΝ ΕΡΓΩΝ",
        "ΔΙΟΙΚΗΤΙΚΗ ΚΑΙ ΟΙΚΟΝΟΜΙΚΗ ΥΠΗΡΕΣΙΑ",
        "ΑΣΚΗΣΗ ΠΟΙΝΙΚΗΣ ΑΓΩΓΗΣ",
        "ΚΑΝΟΝΙΣΜΟΣ ΕΣΩΤΕΡΙΚΗΣ ΥΠΗΡΕΣΙΑΣ ΕΠΙΒΑΤΗΓΩΝ ΠΛΟΙΩΝ",
        "ΚΩΔΙΚΑΣ ΠΟΛΙΤΙΚΗΣ ΔΙΚΟΝΟΜΙΑΣ - ΠΑΛΑΙΟΣ",
        "ΚΑΤΑΣΤΑΤΙΚΟ ΤΑΜΕΙΟΥ ΑΣΦΑΛΙΣΕΩΣ ΕΜΠΟΡΩΝ (Τ.Α.Ε)",
        "ΜΗΧΑΝΟΛΟΓΟΙ, ΗΛΕΚΤΡΟΛΟΓΟΙ, ΝΑΥΠΗΓΟΙ ΚΑΙ ΜΗΧΑΝΟΔΗΓΟΙ",
        "ΣΤΕΓΑΣΗ ΠΑΡΑΠΗΓΜΑΤΟΥΧΩΝ",
        "ΝΟΜΙΣΜΑΤΙΚΗ ΕΠΙΤΡΟΠΗ",
        "ΠΕΡΙΦΕΡΕΙΑΚΑ ΤΑΜΕΙΑ",
        "ΜΗΤΡΩΑ ΑΡΡΕΝΩΝ",
        "ΔΙΚΑΣΤΙΚΕΣ ΔΙΑΚΟΠΕΣ",
        "ΣΥΜΦΩΝΙΑ ΠΕΡΙ ΠΡΟΞΕΝΙΚΩΝ ΣΧΕΣΕΩΝ",
        "ΠΑΛΑΙΟΙ ΑΣΤΙΚΟΙ ΚΩΔΙΚΕΣ",
        "ΚΛΑΔΟΣ ΕΠΙΚΟΥΡΙΚΗΣ ΑΣΦΑΛΙΣΕΩΣ ΔΙΚΗΓΟΡΩΝ (Κ.Ε.Α.Δ.)",
        "ΟΙΚΟΝΟΜΙΚΕΣ ΑΡΜΟΔΙΟΤΗΤΕΣ ΣΤΡΑΤΙΩΤΙΚΩΝ ΑΡΧΩΝ",
        "ΥΠΟΝΟΜΟΙ ΘΕΣΣΑΛΟΝΙΚΗΣ",
        "ΔΙΑΦΟΡΑ ΥΔΡΑΥΛΙΚΑ ΤΑΜΕΙΑ",
        "ΕΛΕΓΧΟΣ ΘΕΑΤΡΙΚΩΝ ΕΡΓΩΝ ΚΑΙ ΔΙΣΚΩΝ",
        "ΥΠΗΡΕΣΙΑ ΙΠΠΟΠΑΡΑΓΩΓΗΣ",
        "ΣΩΜΑΤΙΚΗ ΑΓΩΓΗ",
        "ΕΚΔΙΚΑΣΗ ΤΕΛΩΝΕΙΑΚΩΝ ΠΑΡΑΒΑΣΕΩΝ",
        "ΚΙΝΗΤΡΑ ΙΔΙΩΤΙΚΩΝ ΕΠΕΝΔΥΣΕΩΝ ΣΤΗΝ ΠΕΡΙΦΕΡΕΙΑ",
        "ΜΕΛΗ ΟΙΚΟΓΕΝΕΙΑΣ ΑΣΦΑΛΙΣΜΕΝΩΝ",
        "ΚΕΡΜΑΤΑ",
        "ΕΠΙΔΟΜΑ ΑΝΑΠΡΟΣΑΡΜΟΓΗΣ",
        "ΕΚΤΕΛΕΣΗ ΔΑΣΙΚΩΝ ΕΡΓΩΝ",
        "ΛΙΠΑΣΜΑΤΑ",
        "ΕΠΙΧΟΡΗΓΗΣΗ ΣΠΟΥΔΑΣΤΩΝ ΤΕΚΝΩΝ ΕΡΓΑΖΟΜΕΝΩΝ",
        "ΠΡΟΣΤΑΣΙΑ ΟΙΝΟΥ",
        "ΠΤΗΤΙΚΟ ΚΑΙ ΚΑΤΑΔΥΤΙΚΟ ΕΠΙΔΟΜΑ",
        "ΤΑΜΕΙΟ ΕΠΙΚΟΥΡΙΚΗΣ ΑΣΦΑΛΙΣΕΩΣ ΥΠΑΛΛΗΛΩΝ ΕΜΠΟΡΙΚΩΝ ΚΑΤΑΣΤΗΜΑΤΩΝ (Τ.Ε.Α.Υ.Ε.Κ.)",
        "ΕΚΚΟΚΚΙΣΗ ΒΑΜΒΑΚΟΣ",
        "ΜΟΝΟΠΩΛΙΟ ΚΙΝΙΝΟΥ",
        "ΙΝΣΤΙΤΟΥΤΑ ΔΙΕΘΝΟΥΣ ΔΙΚΑΙΟΥ",
        "ΙΑΠΩΝΙΑ – ΙΝΔΙΑ –ΙΟΡΔΑΝΙΑ Κ.ΛΠ",
        "ΕΠΙΔΟΜΑ ΣΤΟΛΗΣ",
        "ΑΝΑΓΝΩΡΙΣΕΙΣ",
        "ΤΑΜΕΙΟ ΠΡΟΝΟΙΑΣ ΕΡΓΟΛΗΠΤΩΝ",
        "ΑΝΑΣΤΟΛΗ ΤΗΣ ΠΟΙΝΗΣ",
        "ΠΟΤΑΜΟΠΛΟΙΑ",
        "ΕΙΔΙΚΗ ΤΕΛΩΝΕΙΑΚΗ ΠΑΡΑΚΟΛΟΥΘΗΣΗ",
        "ΕΠΙΘΕΩΡΗΣΗ ΦΑΡΜΑΚΕΙΩΝ",
        "ΣΥΝΤΑΞΕΙΣ ΘΥΜΑΤΩΝ ΕΘΝΙΚΩΝ",
        "ΑΠΛΟΠΟΙΗΣΗ ΤΕΛΩΝΕΙΑΚΩΝ ΔΙΑΤΥΠΩΣΕΩΝ",
        "ΚΛΑΔΟΣ ΑΣΘΕΝΕΙΑΣ Τ.Α.Κ.Ε",
        "ΥΠΗΡΕΣΙΑ ΥΠΟΔΟΧΗΣ ΠΛΟΙΩΝ ΚΑΙ ΠΟΛΕΜΙΚΗ ΧΡΗΣΗ ΛΙΜΕΝΩΝ",
        "ΦΑΡΜΑΚΕΙΟ ΠΟΛΕΜΙΚΟΥ ΝΑΥΤΙΚΟΥ",
        "ΤΑΜΕΙΟ ΑΠΟΚΑΤΑΣΤΑΣΕΩΣ ΠΡΟΣΦΥΓΩΝ ΣΥΜΒΟΥΛΙΟΥ ΤΗΣ ΕΥΡΩΠΗΣ",
        "ΝΑΥΤΙΚΕΣ ΕΤΑΙΡΕΙΕΣ",
        "ΙΣΡΑΗΛΙΤΙΚΕΣ ΚΟΙΝΟΤΗΤΕΣ",
        "ΣΕΙΣΜΟΠΛΗΚΤΟΙ  ΣΤΕΡΕΑΣ ΕΛΛΑΔΑΣ (ΑΤΤΙΚΗΣ, ΒΟΙΩΤΙΑΣ Κ.Λ.Π.)",
        "ΔΙΑΦΟΡΕΣ ΣΧΟΛΕΣ Π.Ν",
        "ΤΑΜΕΙΟ ΠΡΟΝΟΙΑΣ ΠΡΟΣΩΠΙΚΟΥ ΕΜΠΟΡ.ΚΑΙ ΒΙΟΜ.- ΕΠΑΓΓΕΛ. ΚΑΙ ΒΙΟΤΕΧΝ. ΕΠΙΜΕΛΗΤΗΡΙΩΝ ΤΟΥ ΚΡΑΤΟΥΣ",
        "ΕΘΝΙΚΗ ΚΤΗΜΑΤΙΚΗ ΤΡΑΠΕΖΑ",
        "ΝΑΥΤΙΚΟΙ ΑΚΟΛΟΥΘΟΙ",
        "ΔΗΜΟΣΙΕΣ ΝΑΥΤΙΚΕΣ ΣΧΟΛΕΣ",
        "ΜΙΚΡΟΦΩΤΟΓΡΑΦΙΕΣ",
        "ΚΑΤΑΣΤΑΤΙΚΟΙ ΝΟΜΟΙ-Τ.Σ.Α.Υ",
        "ΚΑΤΑΣΤΑΣΗ ΑΞΙΩΜΑΤΙΚΩΝ Π.Ν",
        "ΕΛΛΗΝΙΚΑ ΣΧΟΛΕΙΑ ΑΛΛΟΔΑΠΗΣ",
        "ΟΡΓΑΝΙΣΜΟΣ ΟΙΚΟΝΟΜΙΚΗΣ",
        "ΕΘΝΙΚΗ ΤΡΑΠΕΖΑ ΤΗΣ ΕΛΛΑΔΟΣ",
        "ΓΕΝΙΚΕΣ ΔΙΑΤΑΞΕΙΣ Ν.Π.Δ.Δ",
        "ΠΡΟΣΩΠΙΚΟ ΜΕ ΣΧΕΣΗ ΙΔΙΩΤΙΚΟΥ ΔΙΚΑΙΟΥ",
        "ΤΑΜΕΙΟ ΕΠΙΚΟΥΡΙΚΗΣ ΑΣΦΑΛΙΣΕΩΣ ΠΡΟΣΩΠΙΚΟΥ ΕΤΑΙΡΕΙΑΣ ΥΔΡΕΥΣΗΣ ΚΑΙ ΑΠΟΧΕΤΕΥΣΗΣ ΠΡΩΤΕΥΟΥΣΗΣ (Τ.Ε.Α.Π.Ε.Υ.Α.Π.)",
        "ΣΩΜΑ ΟΙΚΟΝΟΜΙΚΟΥ ΕΛΕΓΧΟΥ",
        "ΣΥΜΒΑΣΗ ΠΕΡΙ ΔΙΕΚΔΙΚΗΣΕΩΣ ΔΙΑΤΡΟΦΗΣ",
        "ΙΣΟΤΗΤΑ ΤΩΝ ΔΥΟ ΦΥΛΩΝ",
        "ΤΑΜΕΙΟ ΑΡΩΓΗΣ ΚΑΙ ΕΠΙΚΟΥΡΙΚΟ ΤΑΜΕΙΟ",
        "ΤΟΥΡΙΣΤΙΚΟ ΔΕΛΤΙΟ",
        "ΔΙΑΦΟΡΟΙ ΝΟΜΟΙ",
        "ΟΡΓΑΝΙΣΜΟΣ ΛΙΜΕΝΟΣ ΠΕΙΡΑΙΩΣ ΑΝΩΝΥΜΗ ΕΤΑΙΡΙΑ",
        "ΕΚΚΑΘΑΡΙΣΙΣ ΔΙΟΡΙΣΜΩΝ ΚΑΙ ΠΡΟΑΓΩΓΩΝ ΚΑΤΟΧΗΣ",
        "ΤΑΞΙΝΟΜΗΣΗ ΒΑΜΒΑΚΟΣ",
        "ΠΡΥΤΑΝΕΙΣ ΚΑΙ ΚΟΣΜΗΤΟΡΕΣ",
        "ΥΠΗΡΕΣΙΑΚΟ ΣΥΜΒΟΥΛΙΟ ΕΚΚΛΗΣΙΑΣΤΙΚΗΣ ΕΚΠΑΙΔΕΥΣΗΣ",
        "ΩΡΕΣ ΕΡΓΑΣΙΑΣ ΣΤΗΝ ΒΙΟΜΗΧΑΝΙΑ ΚΑΙ ΒΙΟΤΕΧΝΙΑ",
        "ΧΑΡΤΗΣ ΟΡΓΑΝΙΣΜΟΥ ΟΙΚΟΝΟΜΙΚΗΣ ΣΥΝΕΡΓΑΣΙΑΣ",
        "ΓΥΜΝΑΣΙΟ ΑΠΟΔΗΜΩΝ ΕΛΛΗΝΟΠΑΙΔΩΝ",
        "ΚΑΝΟΝΙΣΜΟΣ ΑΣΘΕΝΕΙΑΣ",
        "ΕΚΔΟΣΕΙΣ ΥΠΟΥΡΓΕΙΟΥ ΕΜΠΟΡΙΚΗΣ ΝΑΥΤΙΛΙΑΣ",
        "ΠΛΗΤΤΟΜΕΝΟΙ ΑΠΟ ΘΕΟΜΗΝΙΕΣ ΚΑΙ ΑΛΛΑ ΕΚΤΑΚΤΑ ΓΕΓΟΝΟΤΑ",
        "ΩΡΕΣ ΕΡΓΑΣΙΑΣ ΠΡΟΣΩΠΙΚΟΥ",
        "ΓΕΩΜΗΛΑ",
        "ΦΟΡΟΛΟΓΙΑ ΑΝΑΤΙΜΗΣΗΣ ΑΚΙΝΗΤΩΝ",
        "ΠΑΝΩΛΗΣ",
        "ΣΧΟΛΕΣ ΝΗΠΙΑΓΩΓΩΝ",
        "ΦΑΡΜΑΚΑΠΟΘΗΚΕΣ",
        "ΦΡΟΝΤΙΣΤΗΡΙΑ ΝΟΜΙΚΩΝ ΣΠΟΥΔΩΝ",
        "ΟΙΚΟΓΕΝΕΙΑΚΑ ΕΠΙΔΟΜΑΤΑ ΜΙΣΘΩΤΩΝ",
        "ΗΛΕΚΤΡΟΚΙΝΗΤΑ ΛΕΩΦΟΡΕΙΑ ΑΘΗΝΩΝ – ΠΕΙΡΑΙΩΣ (Η.Λ.Π.Α.Π.)",
        "ΑΣΤΙΚΑ ΔΙΚΑΙΩΜΑΤΑ ΑΛΛΟΔΑΠΩΝ",
        "ΠΟΛΙΤΙΚΟ ΠΡΟΣΩΠΙΚΟ ΑΕΡΟΠΟΡΙΑΣ",
        "ΔΙΚΑΣΤΙΚΗ ΕΚΠΡΟΣΩΠΗΣΗ Ι.Κ.Α",
        "ΥΓΕΙΟΝΟΜΙΚΗ ΥΠΗΡΕΣΙΑ Π.Σ",
        "ΥΓΕΙΟΝΟΜΙΚΟΙ ΣΤΑΘΜΟΙ",
        "ΙΕΡΑΡΧΙΑ ΚΑΙ ΠΡΟΑΓΩΓΕΣ ΜΟΝΙΜΩΝ ΥΠΑΞΙΩΜΑΤΙΚΩΝ ΚΑΙ ΑΝΘΥΠΑΣΠΙΣΤΩΝ",
        "ΤΑΜΕΙΟ ΕΠΙΚΟΥΡΙΚΗΣ ΑΣΦΑΛΙΣΕΩΣ ΕΡΓΑΤΟΤΕΧΝΙΤΩΝ ΚΑΙ ΥΠΑΛΛΗΛΩΝ ΔΕΡΜΑΤΟΣ ΕΛΛΑΔΑΣ (Τ.Ε.Α.Ε.Υ.Δ.Ε.)",
        "ΠΡΑΤΗΡΙΑ ΑΡΤΟΥ",
        "ΠΛΗΡΩΜΗ ΜΕ ΕΠΙΤΑΓΗ",
        "ΤΕΧΝΙΚΗ ΕΚΜΕΤΑΛΛΕΥΣΗ ΕΛΙΚΟΠΤΕΡΩΝ",
        "ΔΙΕΘΝΕΙΣ ΤΑΧΥΔΡΟΜΙΚΕΣ ΣΥΜΒΑΣΕΙΣ",
        "ΔΙΚΑΣΤΙΚΟΙ ΑΝΤΙΠΡΟΣΩΠΟΙ ΤΟΥ ΔΗΜΟΣΙΟΥ",
        "ΩΡΕΣ ΕΡΓΑΣΙΑΣ ΣΕ ΔΙΑΦΟΡΑ ΕΠΑΓΓΕΛΜΑΤΑ",
        "ΔΙΕΥΘΥΝΣΗ ΚΤΗΝΟΤΡΟΦΙΑΣ",
        "ΕΠΙΘΕΩΡΗΣΗ ΣΦΑΓΙΩΝ",
        "ΠΛΩΙΜΟΤΗΤΑ ΑΕΡΟΣΚΑΦΩΝ",
        "ΑΓΟΡΑΝΟΜΙΚΟΣ ΚΩΔΙΚΑΣ",
        "ΔΙΕΘΝΕΙΣ ΜΕΤΑΦΟΡΕΣ ΕΠΙΒΑΤΩΝ ΚΑΙ ΕΜΠΟΡΕΥΜΑΤΩΝ",
        "ΠΡΟΜΗΘΕΙΕΣ",
        "ΔΙΑΦΟΡΕΣ ΔΙΑΤΑΞΕΙΣ",
        "ΔΙΑΙΤΗΣΙΑ ΣΥΛΛΟΓΙΚΩΝ ΔΙΑΦΟΡΩΝ - ΜΕΣΟΛΑΒΗΤΕΣ ΔΙΑΙΤΗΤΕΣ",
        "ΣΟΥΛΤΑΝΙΝΑ",
        "ΜΕΤΑΓΡΑΦΗ",
        "ΕΙΣΑΓΩΓΗ ΕΠΙΣΤΗΜΟΝΙΚΟΥ ΥΛΙΚΟΥ",
        "ΔΙΑΡΘΡΩΣΗ ΥΠΗΡΕΣΙΩΝ  Ο.Γ.Α",
        "ΔΙΚΑΣΤΙΚΟΙ ΛΕΙΤΟΥΡΓΟΙ - ΕΘΝΙΚΗ ΣΧΟΛΗ ΔΙΚΑΣΤΩΝ",
        "ΠΙΣΤΟΠΟΙΗΤΙΚΑ ΚΑΙ ΔΙΚΑΙΟΛΟΓΗΤΙΚΑ",
        "ΑΣΚΗΣΗ ΙΑΤΡΙΚΟΥ ΕΠΑΓΓΕΛΜΑΤΟΣ",
        "ΣΥΝΕΤΑΙΡΙΣΜΟΙ ΔΗΜΟΣΙΩΝ ΥΠΑΛΛΗΛΩΝ",
        "ΣΧΟΛΗ ΕΠΙΣΤΗΜΩΝ ΥΓΕΙΑΣ ΠΑΝΜΙΟΥ ΠΑΤΡΩΝ",
        "ΑΛΛΟΔΑΠΕΣ ΝΑΥΤΙΛΙΑΚΕΣ ΕΠΙΧΕΙΡΗΣΕΙΣ",
        "ΛΑΤΟΜΕΙΑ",
        "ΕΚΜΕΤΑΛΛΕΥΣΗ ΙΑΜΑΤΙΚΩΝ ΠΗΓΩΝ",
        "ΠΩΛΗΣΗ ΧΡΕΩΓΡΑΦΩΝ ΜΕ ΔΟΣΕΙΣ",
        "ΝΟΜΟΘΕΣΙΑ ΠΕΡΙ ΤΡΑΠΕΖΩΝ (ΓΕΝΙΚΑ)",
        "ΕΙΔΙΚΑ ΜΕΤΑΛΛΕΙΑ",
        "YΠΟΥΡΓΕΙΟ ΥΓΙΕΙΝΗΣ",
        "ΛΗΞΙΑΡΧΙΚΕΣ ΠΡΑΞΕΙΣ",
        "ΓΕΝΙΚΕΣ ΔΙΑΤΑΞΕΙΣ ΓΙΑ ΤΟΝ ΤΥΠΟ",
        "ΕΘΝΙΚΟ ΣΥΣΤΗΜΑ ΕΠΑΓΓΕΛΜΑΤΙΚΗΣ ΕΚΠΑΙΔΕΥΣΗΣ-ΚΑΤΑΡΤΙΣΗΣ",
        "ΑΡΟΥΡΑΙΟΙ ΚΑΙ ΑΚΡΙΔΕΣ",
        "ΠΡΟΣΤΑΣΙΑ ΦΥΜΑΤΙΚΩΝ ΝΑΥΤΙΚΩΝ",
        "ΑΠΟΡΡΗΤΟ ΕΠΙΣΤΟΛΩΝ ΚΑΙ ΕΠΙΚΟΙΝΩΝΙΩΝ",
        "ΠΟΡΘΜΕΙΑ ΚΑΙ ΟΧΗΜΑΤΑΓΩΓΑ",
        "ΜΕΤΡΑ ΕΞΟΙΚΟΝΟΜΗΣΗΣ ΕΝΕΡΓΕΙΑΣ",
        "ΣΤΟΙΧΕΙΑ ΠΡΟΣΩΠΙΚΟΥ ΔΗΜΟΣΙΩΝ ΥΠΗΡΕΣΙΩΝ ΚΑΙ Ν.Π.Δ.Δ",
        "ΠΑΓΙΕΣ ΑΜΟΙΒΕΣ ΔΙΚΗΓΟΡΩΝ",
        "ΟΡΓΑΝΙΣΜΟΣ ΣΧΟΛΗΣ ΕΥΕΛΠΙΔΩΝ",
        "ΟΙΚΟΝΟΜΙΚΟ ΕΠΙΜΕΛΗΤΗΡΙΟ ΤΗΣ ΕΛΛΑΔΑΣ",
        "ΓΡΑΦΕΙΑ ΕΥΡΕΣΕΩΣ ΕΡΓΑΣΙΑΣ",
        "ΔΙΑΦΗΜΙΣΕΙΣ",
        "ΔΙΑΦΟΡΕΣ ΥΠΟΤΡΟΦΙΕΣ",
        "ΦΟΡΤΗΓΑ ΑΚΤΟΠΛΟΙΚΑ ΠΛΟΙΑ (ΜS) ΜΕΧΡΙ 500 Κ.Ο.Χ",
        "ΕΠΙΤΡΟΠΗ ΣΥΝΕΡΓΑΣΙΑΣ UNICEF",
        "ΥΓΙΕΙΝΗ ΘΕΡΕΤΡΩΝ",
        "ΕΠΙΣΤΗΜΟΝΙΚΗ ΕΡΕΥΝΑ ΚΑΙ ΤΕΧΝΟΛΟΓΙΑ",
        "ΑΠΑΓΟΡΕΥΣΕΙΣ ΕΞΑΓΩΓΗΣ",
        "ΑΜΠΕΛΟΥΡΓΙΚΟ ΚΤΗΜΑΤΟΛΟΓΙΟ",
        "ΥΠΟΥΡΓΕΙΟ ΥΓΕΙΑΣ ΚΑΙ ΠΡΟΝΟΙΑΣ",
        "ΔΙΕΘΝΗΣ ΝΑΥΤΙΛΙΑΚΟΣ ΟΡΓΑΝΙΣΜΟΣ",
        "ΔΙΕΥΘΥΝΣΗ ΤΕΛΩΝΕΙΑΚΟΥ ΕΛΕΓΧΟΥ",
        "ΔΕΛΤΙΑ ΤΑΥΤΟΤΗΤΟΣ Π. ΝΑΥΤΙΚΟΥ",
        "ΑΝΩΤΑΤΗ ΥΓΕΙΟΝΟΜΙΚΗ ΕΠΙΤΡΟΠΗ",
        "ΠΡΟΣΤΑΣΙΑ ΕΦΕΔΡΩΝ ΑΞΙΩΜΑΤΙΚΩΝ, ΑΝΑΠΗΡΩΝ ΠΟΛΕΜΟΥ ΚΑΙ ΑΓΩΝΙΣΤΩΝ ΕΘΝ. ΑΝΤΙΣΤΑΣΗΣ",
        "ΦΟΡΟΙ ΥΠΕΡ ΤΡΙΤΩΝ",
        "ΑΓΡΟΛΗΨΙΕΣ ΙΟΝΙΩΝ ΝΗΣΙΩΝ",
        "ΤΑΜΕΙΟ ΕΠΙΚΟΥΡΙΚΗΣ ΑΣΦΑΛΙΣΕΩΣ ΥΠΑΛΛΗΛΩΝ ΕΜΠΟΡΙΟΥ ΤΡΟΦΙΜΩΝ (Τ.Ε.Α.Υ.Ε.Τ)",
        "ΑΝΩΤΑΤΟ ΕΙΔΙΚΟ ΔΙΚΑΣΤΗΡΙΟ",
        "ΕΙΣΑΓΩΓΗ ΓΥΝΑΙΚΩΝ ΣΤΙΣ ΑΝΩΤΑΤΕΣ ΣΤΡΑΤΙΩΤΙΚΕΣ ΣΧΟΛΕΣ",
        "ΣΧΟΛΗ ΑΞΙΩΜΑΤΙΚΩΝ ΝΟΣΗΛΕΥΤΙΚΗΣ (Σ.Α.Ν.)",
        "ΔΙΑΔΙΚΑΣΙΑ ΔΙΟΙΚΗΤΙΚΩΝ ΔΙΚΑΣΤΗΡΙΩΝ",
        "ΠΡΟΣΤΑΣΙΑ ΕΡΓΑΖΟΜΕΝΟΥ ΠΑΙΔΙΟΥ",
        "ΑΜΝΗΣΤΙΑ",
        "ΣΧΟΛΕΣ ΚΑΛΛΙΤΕΧΝΙΚΗΣ ΕΚΠΑΙΔΕΥΣΗΣ",
        "ΧΑΡΗ ΚΑΙ ΜΕΤΡΙΑΣΜΟΣ",
        "ΤΥΦΛΟΙ",
        "ΣΥΜΒΟΥΛΙΟ ΤΗΣ ΕΥΡΩΠΗΣ",
        "ΕΡΓΟΣΤΑΣΙΑ ΕΚΡΗΚΤΙΚΩΝ ΥΛΩΝ",
        "ΜΗΤΡΩΑ Π. ΝΑΥΤΙΚΟΥ",
        "ΥΓΡΗ ΑΜΜΩΝΙΑ",
        "ΠΕΙΡΑΜΑΤΙΚΑ ΣΧΟΛΕΙΑ",
        "ΤΑΜΕΙΟ ΠΡΟΝΟΙΑΣ ΑΞΙΩΜΑΤΙΚΩΝ Ε.Ν",
        "ΕΠΑΓΓΕΛΜΑΤΙΚΟΣ ΠΡΟΣΑΝΑΤΟΛΙΣΜΟΣ ΚΑΙ ΚΑΤΑΡΤΙΣΗ",
        "ΤΕΛΩΝΕΙΑΚΗ ΕΠΙΒΛΕΨΗ",
        "ΠΡΟΣΩΡΙΝΕΣ ΕΥΡΩΠΑΙΚΕΣ ΣΥΜΦΩΝΙΕΣ",
        "ΜΟΝΟΠΩΛΙΟ ΠΑΙΓΝΙΟΧΑΡΤΩΝ",
        "ΛΕΙΤΟΥΡΓΙΑ ΤΟΥΡΙΣΤΙΚΗΣ ΑΣΤΥΝΟΜΙΑΣ",
        "ΕΚΠΟΙΗΣΗ ΕΚΚΛΗΣΙΑΣΤΙΚΩΝ ΚΙΝΗΤΩΝ ΚΑΙ ΑΚΙΝΗΤΩΝ",
        "ΣΥΛΛΟΓΙΚΕΣ ΣΥΜΒΑΣΕΙΣ (ΓΕΝΙΚΑ)",
        "ΟΔΟΙΠΟΡΙΚΑ ΚΑΙ ΑΠΟΖΗΜΙΩΣΕΙΣ ΕΚΤΟΣ ΕΔΡΑΣ",
        "ΣΤΕΓΑΣΤΙΚΗ ΑΠΟΚΑΤΑΣΤΑΣΗ ΠΡΟΣΦΥΓΩΝ",
        "ΑΝΩΤΑΤΑ ΣΥΜΒΟΥΛΙΑ ΕΚΠΑΙΔΕΥΣΕΩΣ",
        "ΑΡΧΕΙΑ ΥΠΟΥΡΓΕΙΟΥ ΟΙΚΟΝΟΜΙΚΩΝ",
        "ΓΕΝΙΚΗ ΓΡΑΜΜΑΤΕΙΑ ΥΠΟΥΡΓΙΚΟΥ ΣΥΜΒΟΥΛΙΟΥ",
        "ΠΕΡΙΠΤΕΡΑ ΑΝΑΠΗΡΩΝ ΠΟΛΕΜΟΥ",
        "ΕΠΑΓΓΕΛΜΑΤΙΚΕΣ ΟΡΓΑΝΩΣΕΙΣ ΕΜΠΟΡΩΝ, ΒΙΟΤΕΧΝΩΝ ΚΑΙ ΛΟΙΠΩΝ ΕΠΑΓΓΕΛΜΑΤΙΩΝ",
        "ΙΔΙΩΤΙΚΟΙ ΣΤΑΘΜΟΙ ΠΑΡΑΓΩΓΗΣ ΗΛΕΚΤΡΙΚΗΣ ΕΝΕΡΓΕΙΑΣ",
        "ΘΕΑΤΡΙΚΑ ΕΡΓΑ",
        "ΜΕ ΤΗ ΝΕΑ ΖΗΛΑΝΔΙΑ",
        "ΦΟΡΟΣ ΚΑΤΑΝΑΛΩΣΕΩΣ ΣΑΚΧΑΡΕΩΣ",
        "ΝΟΜΑΡΧΙΑΚΑ ΤΑΜΕΙΑ",
        "ΑΓΩΓΕΣ ΚΑΚΟΔΙΚΙΑΣ",
        "ΚΩΔΙΚΑΣ ΦΟΡΟΛΟΓΙΚΗΣ ΔΙΚΟΝΟΜΙΑΣ",
        "ΑΤΟΜΑ ΒΑΡΙΑ ΝΟΗΤΙΚΑ ΚΑΘΥΣΤΕΡΗΜΕΝΑ",
        "ΜΕ ΤΗ ΣΟΥΗΔΙΑ",
        "ΑΕΡΟΝΑΥΤΙΚΗ ΜΕΤΕΩΡΟΛΟΓΙΑ",
        "ΙΔΙΩΤΙΚΕΣ ΣΧΟΛΕΣ ΓΥΜΝΑΣΤΙΚΗΣ",
        "ΠΕΡΙΟΥΣΙΑ ΔΗΜΩΝ ΚΑΙ ΚΟΙΝΟΤΗΤΩΝ",
        "ΑΓΟΡΑΠΩΛΗΣΙΕΣ ΚΑΤΟΧΗΣ",
        "ΕΚΚΛΗΣΙΑ ΠΑΡΙΣΙΩΝ",
        "ΔΙΕΘΝΕΙΣ ΣΥΜΒΑΣΕΙΣ ΠΡΟΣΤΑΣΙΑΣ ΦΥΤΩΝ",
        "ΚΑΤΟΧΥΡΩΣΗ ΘΡΗΣΚΕΥΤΙΚΗΣ ΕΛΕΥΘΕΡΙΑΣ",
        "ΥΓΕΙΟΝΟΜΙΚΗ ΕΞΕΤΑΣΗ ΜΗ ΙΠΤΑΜΕΝΟΥ ΠΡΟΣΩΠΙΚΟΥ",
        "ΣΥΝΤΑΞΕΙΣ ΘΥΜΑΤΩΝ ΠΟΛΕΜΟΥ 1940",
        "ΥΔΡΑΥΛΙΚΕΣ ΕΓΚΑΤΑΣΤΑΣΕΙΣ",
        "ΚΟΙΝΩΝΙΚΟΙ ΛΕΙΤΟΥΡΓΟΙ - ΚΟΙΝΩΝΙΚΟΙ ΣΥΜΒΟΥΛΟΙ",
        "ΔΙΑΦΟΡΕΣ ΠΡΟΣΩΡΙΝΕΣ ΑΤΕΛΕΙΕΣ",
        "ΟΙΚΟΝΟΜΙΚΗ ΔΙΑΧΕΙΡΙΣΗ ΚΑΙ ΛΟΓΙΣΤΙΚΟ",
        "ΕΞΗΛΕΚΤΡΙΣΜΟΣ ΝΗΣΩΝ",
        "ΕΚΠΑΙΔΕΥΣΗ ΣΤΕΛΕΧΩΝ",
        "ΩΡΕΣ ΕΡΓΑΣΙΑΣ ΚΑΤΑΣΤΗΜΑΤΩΝ ΚΑΙ ΓΡΑΦΕΙΩΝ",
        "ΗΜΕΡΟΛΟΓΙΟ ΓΕΦΥΡΑΣ",
        "ΠΡΟΣΤΑΣΙΑ ΤΗΣ ΣΤΑΦΙΔΑΣ",
        "ΠΑΛΑΙΟΙ ΔΙΚΟΝΟΜΙΚΟΙ ΝΟΜΟΙ",
        "ΤΑΜΕΙΟ ΕΠΙΚ. ΑΣΦΑΛΙΣΕΩΣ ΠΡΟΣΩΠΙΚΟΥ ΟΡΓΑΝΙΣΜΩΝ ΚΟΙΝΩΝΙΚΗΣ ΑΣΦΑΛΙΣΕΩΣ (Τ.Ε.Α.Π.Ο.Κ.Α.)",
        "ΠΑΡΟΧΕΣ ΥΓΕΙΑΣ ΑΣΦΑΛΙΣΤΙΚΩΝ ΟΡΓΑΝΙΣΜΩΝ",
        "ΠΛΑΝΟΔΙΟΙ ΙΧΘΥΟΠΩΛΕΣ",
        "ΔΙΑΦΟΡΟΙ ΝΟΜΟΙ ΠΕΡΙ ΑΞΙΩΜΑΤΙΚΩΝ Π.Ν",
        "ΥΠΟΧΡΕΩΣΕΙΣ ΕΦΟΠΛΙΣΤΩΝ ΣΕ ΑΣΘΕΝΕΙΑ Η ΘΑΝΑΤΟ ΝΑΥΤΙΚΩΝ",
        "ΠΡΟΣΤΑΣΙΑ ΚΑΤΑ ΤΗΣ ΑΣΘΕΝΕΙΑΣ",
        "ΓΕΝΙΚΑ ΠΕΡΙ ΣΧΕΔΙΩΝ ΠΟΛΕΩΝ",
        "ΕΞΑΙΡΕΣΕΙΣ ΑΠΟ ΤΗΝ ΕΡΓΑΤΙΚΗ ΝΟΜΟΘΕΣΙΑ",
        "ΑΓΡΟΤΙΚΟ ΚΤΗΜΑΤΟΛΟΓΙΟ",
        "ΣΥΝΤΑΓΜΑΤΙΚΕΣ ΔΙΑΤΑΞΕΙΣ ΕΚΚΛΗΣΙΑΣ ΤΗΣ ΕΛΛΑΔΟΣ",
        "ΠΑΝΑΓΙΟΣ ΤΑΦΟΣ",
        "ΣΥΝΕΡΓΕΙΑ Π. ΝΑΥΤΙΚΟΥ",
        "ΕΠΙΘΕΩΡΗΣΙΣ ΣΤΡΑΤΟΥ",
        "ΣΥΝΘΕΣΗ ΠΛΗΡΩΜΑΤΩΝ",
        "ΟΡΓΑΝΙΣΜΟΣ ΕΡΓΑΤΙΚΗΣ ΕΣΤΙΑΣ",
        "ΔΙΑΦΟΡΑ ΥΔΡΑΥΛΙΚΑ ΕΡΓΑ",
        "ΔΙΚΑΙΩΜΑ ΤΟΥ ΣΥΝΕΡΧΕΣΘΑΙ",
        "ΚΟΙΝΩΝΙΚΟΠΟΙΗΣΗ - ΑΠΟΚΡΑΤΙΚΟΠΟΙΗΣΗ ΕΠΙΧΕΙΡΗΣΕΩΝ ΔΗΜΟΣΙΟΥ ΧΑΡΑΚΤΗΡΑ",
        "ΛΑΙΚΗ ΚΑΤΟΙΚΙΑ",
        "ΦΟΡΟΛΟΓΙΑ ΚΕΡΔΩΝ",
        "ΤΕΧΝΙΚΗ ΥΠΗΡΕΣΙΑ",
        "ΜΕΤΕΚΠΑΙΔΕΥΣΗ ΔΗΜΟΔΙΔΑΣΚΑΛΩΝ",
        "ΣΥΝΤΑΞΕΙΣ ΥΠΟΥΡΓΩΝ ΚΑΙ ΒΟΥΛΕΥΤΩΝ",
        "ΟΡΙΟ ΗΛΙΚΙΑΣ",
        "ΣΤΡΑΤΙΩΤΙΚΕΣ ΠΡΟΜΗΘΕΙΕΣ",
        "ΑΠΟΣΤΟΛΑΙ ΕΞΩΤΕΡΙΚΟΥ",
        "ΦΟΡΟΛΟΓΙΑ ΑΚΙΝΗΤΗΣ ΠΕΡΙΟΥΣΙΑΣ",
        "ΧΡΟΝΟΣ ΕΡΓΑΣΙΑΣ - ΑΔΕΙΕΣ ΠΡΟΣΩΠΙΚΟΥ ΕΛΛΗΝΙΚΗΣ ΑΣΤΥΝΟΜΙΑΣ",
        "ΝΑΥΤΙΚΑ ΕΡΓΑ ΚΑΙ ΠΡΟΜΗΘΕΙΕΣ",
        "ΟΙΚΟΝΟΜΙΚΗ ΔΙΟΙΚΗΣΗ ΚΑΙ ΛΟΓΙΣΤΙΚΟ",
        "ΔΑΣΜΟΛΟΓΙΚΕΣ ΣΥΜΒΑΣΕΙΣ",
        "ΤΑΜΕΙΟ ΑΣΦΑΛΙΣΕΩΣ ΧΡΗΜΑΤΙΣΤΩΝ ,ΜΕΣΙΤΩΝ,ΑΝΤΙΚΡΥΣΤΩΝ ΚΑΙ ΥΠΑΛΛΗΛΩΝ ΧΡΗΜΑΤΙΣΤΗΡΙΟΥ ΑΘΗΝΩΝ (Τ.Α.Χ.Μ.Α.)",
        "ΚΡΑΤΙΚΗ ΣΧΟΛΗ ΟΡΧΗΣΤΙΚΗΣ ΤΕΧΝΗΣ",
        "ΕΘΝΙΚΗ ΛΥΡΙΚΗ ΣΚΗΝΗ",
        "ΑΕΡΟΝΑΥΤΙΚΕΣ ΤΗΛΕΠΙΚΟΙΝΩΝΙΕΣ",
        "ΚΕΝΤΡΟ ΒΙΟΤΕΧΝΙΚΗΣ ΑΝΑΠΤΥΞΗΣ",
        "ΑΡΧΑΙΟΛΟΓΙΚΟ ΜΟΥΣΕΙΟ",
        "ΥΠΕΡΩΚΕΑΝΕΙΑ",
        "ΔΑΣΗ",
        "ΑΣΚΗΣΗ ΚΤΗΝΙΑΤΡΙΚΟΥ ΕΠΑΓΓΕΛΜΑΤΟΣ",
        "ΚΤΗΣΗ ΚΑΙ ΑΠΩΛΕΙΑ",
        "ΡΑΔΙΟΤΗΛΕΓΡΑΦΙΚΗ ΥΠΗΡΕΣΙΑ",
        "ΑΕΡΟΛΙΜΕΝΑΣ ΑΘΗΝΩΝ",
        "ΠΡΩΤΟΒΑΘΜΙΑ ΕΚΠΑΙΔΕΥΣΗ",
        "ΣΤΕΛΕΧΟΣ ΕΦΕΔΡΩΝ ΑΞΙΩΜΑΤΙΚΩΝ",
        "ΠΤΩΧΕΥΣΗ ΚΑΙ ΣΥΜΒΙΒΑΣΜΟΣ",
        "ΠΟΛΙΤΙΚΟΣ ΓΑΜΟΣ",
        "ΙΔΙΩΤΙΚΗ ΕΠΙΧΕΙΡΗΣΗ ΑΣΦΑΛΙΣΕΩΣ",
        "ΠΛΟΙΑ ΠΟΛΕΜΙΚΟΥ ΝΑΥΤΙΚΟΥ",
        "ΙΑΤΡΙΚΕΣ ΑΜΟΙΒΕΣ",
        "ΕΛΛΗΝΙΚΟΣ ΕΡΥΘΡΟΣ ΣΤΑΥΡΟΣ",
        "ΑΝΩΜΑΛΕΣ ΚΑΤΑΘΕΣΕΙΣ ΣΕ ΧΡΥΣΟ",
        "ΣΥΜΒΟΥΛΙΟ ΤΙΜΗΣ ΑΞΙΩΜΑΤΙΚΩΝ Π.Ν",
        "ΔΙΑΦΟΡΟΙ ΑΡΔΕΥΤΙΚΟΙ ΝΟΜΟΙ",
        "ΚΥΒΕΡΝΗΤΙΚΟΣ ΕΠΙΤΡΟΠΟΣ",
        "ΕΚΤΕΛΕΣΗ ΣΥΓΚΟΙΝΩΝΙΑΚΩΝ ΕΡΓΩΝ",
        "ΤΑΜΕΙΟ ΣΥΝΤΑΞΕΩΝ ΚΑΙ ΑΡΩΓΗΣ",
        "ΔΑΣΙΚΕΣ ΜΕΤΑΦΟΡΕΣ",
        "ΜΕ ΤΗ ΔΗΜΟΚΡΑΤΙΑ ΤΟΥ ΚΕΜΠΕΚ",
        "ΕΠΑΝΕΞΑΓΟΜΕΝΑ ΜΕ ΕΓΓΥΗΣΗ",
        "ΔΙΑΝΟΜΗ ΗΛΕΚΤΡΙΚΗΣ ΕΝΕΡΓΕΙΑΣ",
        "ΑΡΣΗ ΣΥΓΚΡΟΥΣΕΩΣ ΚΑΘΗΚΟΝΤΩΝ",
        "ΕΚΠΑΙΔΕΥΤΙΚΑ ΠΛΟΙΑ",
        "ΚΕΝΤΡΟ ΜΕΤΑΦΡΑΣΗΣ",
        "ΕΙΣΦΟΡΕΣ ΚΑΙ ΝΑΥΛΩΣΕΙΣ",
        "ΜΕΤΕΓΓΡΑΦΕΣ ΦΟΙΤΗΤΩΝ ΑΝΩΤ. ΕΚΠΑΙΔΕΥΤΙΚΩΝ ΙΔΡΥΜΑΤΩΝ",
        "ΤΜΗΜΑΤΑ ΕΠΙΣΤΗΜΗΣ ΦΥΣΙΚΗΣ ΑΓΩΓΗΣ - ΑΘΛΗΤΙΣΜΟΥ",
        "ΨΥΧΙΑΤΡΕΙΑ",
        "ΦΟΡΟΛΟΓΙΑ ΚΕΦΑΛΑΙΟΥ ΑΝΩΝ. ΕΤΑΙΡΕΙΩΝ",
        "ΤΥΠΟΙ ΣΥΜΒΟΛΑΙΩΝ",
        "ΚΑΝΟΝΙΣΜΟΣ ΕΠΙΘΕΩΡΗΣΕΩΣ",
        "ΜΟΥΣΕΙΟ ΕΛΛΗΝΙΚΗΣ ΛΑΙΚΗΣ ΤΕΧΝΗΣ",
        "ΠΑΝΕΠΙΣΤΗΜΙΟ ΠΕΛΟΠΟΝΝΗΣΟΥ",
        "ΟΡΓΑΝΙΣΜΟΣ ΕΡΓΑΤΙΚΗΣ ΚΑΤΟΙΚΙΑΣ",
        "ΑΣΦΑΛΕΙΑ ΕΡΓΑΖΟΜΕΝΩΝ ΣΕ ΟΙΚΟΔΟΜΕΣ",
        "ΣΤΕΓΑΝΗ ΥΠΟΔΙΑΙΡΕΣΗ ΠΛΟΙΩΝ",
        "ΔΙΟΙΚΗΣΗ ΠΡΩΤΕΥΟΥΣΗΣ",
        "ΔΙΔΑΚΤΟΡΙΚΕΣ - ΜΕΤΑΠΤΥΧΙΑΚΕΣ ΣΠΟΥΔΕΣ ΕΘΝΙΚΟΥ ΜΕΤΣΟΒΙΟΥ",
        "ΕΙΣΦΟΡΑ ΚΑΤΟΧΩΝ ΕΙΔΩΝ ΠΡΩΤΗΣ ΑΝΑΓΚΗΣ",
        "ΔΙΑΦΟΡΟΙ ΔΙΚΟΝΟΜΙΚΟΙ ΝΟΜΟΙ",
        "ΔΙΕΘΝΕΙΣ ΛΙΜΕΝΙΚΕΣ ΣΥΜΒΑΣΕΙΣ",
        "ΥΓΕΙΟΝΟΜΙΚΗ ΥΠΗΡΕΣΙΑ ΕΛ.ΑΣ",
        "ΕΛΛΗΝΙΚΑ ΤΑΧΥΔΡΟΜΕΙΑ (ΕΛ.ΤΑ)",
        "ΜΙΣΘΟΙ ΚΑΙ ΕΠΙΔΟΜΑΤΑ Π. ΝΑΥΤΙΚΟΥ",
        "ΓΕΩΡΓΙΚΑ ΤΑΜΕΙΑ",
        "ΑΤΕΛΕΙΕΣ ΥΠΕΡ ΜΕΤΑΛΛΕΥΤΙΚΩΝ ΕΠΙΧΕΙΡΗΣΕΩΝ",
        "ΑΠΟΒΑΡΟ",
        "ΤΑΜΕΙΟ ΕΠΙΚΟΥΡΙΚΗΣ ΑΣΦΑΛΙΣΕΩΣ ΕΚΠΡΟΣΩΠΩΝ ΚΑΙ ΥΠΑΛΛΗΛΩΝ",
        "ΚΩΔΙΚΑΣ ΠΕΡΙ ΔΙΚΗΓΟΡΩΝ",
        "ΙΕΡΑΡΧΙΑ ΚΑΙ ΠΡΟΒΙΒΑΣΜΟΙ",
        "ΙΣΡΑΗΛΙΤΕΣ",
        "ΣΩΜΑ ΚΤΗΝΙΑΤΡΙΚΟ",
        "ΝΟΡΒΗΓΙΑ - ΝΕΑ ΖΗΛΑΝΔΙΑ – ΝΙΓΗΡΙΑ Κ.ΛΠ",
        "ΕΝΤΥΠΑ ΚΑΙ ΒΙΒΛΙΟΘΗΚΕΣ ΝΑΥΤΙΚΟΥ",
        "ΥΠΟΥΡΓΕΙΟ ΤΥΠΟΥ ΚΑΙ ΜΕΣΩΝ ΜΑΖΙΚΗΣ ΕΝΗΜΕΡΩΣΗΣ",
        "ΝΑΥΤΙΚΕΣ ΠΕΙΘΑΡΧΙΚΕΣ ΠΟΙΝΕΣ",
        "ΜΙΣΘΩΣΕΙΣ ΑΓΡΟΤΙΚΩΝ ΑΚΙΝΗΤΩΝ",
        "ΔΙΑΦΟΡΟΙ ΣΥΝΕΤΑΙΡΙΣΜΟΙ",
        "ΑΓΡΟΤΙΚΗ ΠΙΣΤΗ",
        "ΛΑΙΚΕΣ ΑΓΟΡΕΣ-ΤΑΜΕΙΟ ΛΑΙΚΩΝ ΑΓΟΡΩΝ",
        "ΚΑΝΟΝΙΣΜΟΣ ΠΕΙΘΑΡΧΙΑΣ ΧΩΡΟΦΥΛΑΚΗΣ",
        "ΑΔΙΚΗΜΑΤΑ ΚΑΤΑ ΤΗΣ ΔΗΜΟΣΙΑΣ ΑΣΦΑΛΕΙΑΣ",
        "ΕΝΟΙΚΙΑΣΗ ΦΟΡΟΥ ΔΗΜΟΣΙΩΝ ΘΕΑΜΑΤΩΝ",
        "ΕΥΡΩΠΑΙΚΗ ΣΥΜΒΑΣΗ ΚΟΙΝΩΝΙΚΗΣ ΚΑΙ ΙΑΤΡΙΚΗΣ ΑΝΤΙΛΗΨΕΩΣ",
        "ΕΠΙΒΑΤΗΓΑ ΑΕΡΟΣΤΡΩΜΝΑ ΟΧΗΜΑΤΑ",
        "ΕΦΕΔΡΟΙ",
        "ΣΤΡΑΤΙΩΤΙΚΕΣ ΛΕΣΧΕΣ",
        "ΠΡΟΣΩΠΙΚΟ ΦΥΛΑΚΩΝ",
        "ΑΝΑΘΕΩΡΗΣΗ ΤΙΜΩΝ",
        "ΜΑΛΑΚΙΑ ΚΑΙ ΜΑΛΑΚΟΣΤΡΑΚΑ",
        "ΚΩΔΙΚΑΣ ΔΗΜΟΣΙΟΥ ΝΑΥΤΙΚΟΥ ΔΙΚΑΙΟΥ",
        "ΔΙΑΦΟΡΑ ΣΩΜΑΤΕΙΑ",
        "ΓΕΝΙΚΕΣ ΔΙΑΤΑΞΕΙΣ",
        "ΚΩΔΙΚΟΠΟΙΗΣΗ ΑΓΟΡΑΝΟΜΙΚΩΝ ΔΙΑΤΑΞΕΩΝ",
        "ΕΚΠΑΙΔΕΥΣΗ ΣΤΗΝ ΑΛΛΟΔΑΠΗ",
        "ΔΙΔΑΚΤΙΚΑ ΒΙΒΛΙΑ",
        "ΣΥΝΤΑΞΙΟΔΟΤΙΚΑ ΚΑΙ ΑΣΦΑΛΙΣΤΙΚΑ ΘΕΜΑΤΑ ΠΡΟΣΩΠΙΚΟΥ Ν.Π.Δ.Δ",
        "ΕΠΙΔΟΜΑ ΟΙΚΟΓΕΝΕΙΩΝ ΣΤΡΑΤΙΩΤΙΚΩΝ ΕΞΑΦΑΝΙΣΘΕΝΤΩΝ ΚΑΙ ΑΙΧΜΑΛΩΤΩΝ",
        "ΔΙΕΘΝΕΙΣ ΣΥΜΒΑΣΕΙΣ",
        "ΚΕΝΤΡΟ ΔΙΠΛΩΜΑΤΙΚΩΝ ΣΠΟΥΔΩΝ",
        "ΓΕΝ. ΔΙΕΥΘΥΝΣΗ ΤΥΠΟΥ ΚΑΙ ΠΛΗΡΟΦΟΡΙΩΝ",
        "ΑΡΧΕΙΑ ΤΕΛΩΝΕΙΑΚΩΝ ΑΡΧΩΝ",
        "ΕΙΔΙΚΕΣ ΤΙΜΕΣ ΚΑΥΣΙΜΩΝ",
        "ΣΤΕΓΗ ΥΓΕΙΟΝΟΜΙΚΩΝ",
        "ΓΕΝΙΚΑ ΠΕΡΙ ΣΥΜΒΟΛΑΙΟΓΡΑΦΩΝ",
        "ΒΟΥΛΗ",
        "ΕΠΙΛΟΓΗ & ΑΞΙΟΛΟΓΗΣΗ ΑΣΤΥΝΟΜΙΚΟΥ ΠΡΟΣΩΠΙΚΟΥ ΕΛ.ΑΣ",
        "ΧΟΙΡΟΤΡΟΦΙΑ",
        "ΦΟΡΟΣ ΚΑΤΑΝΑΛΩΣΕΩΣ ΠΕΤΡΕΛΑΙΟΕΙΔΩΝ",
        "ΕΠΙΒΟΛΗ ΤΕΛΩΝΙΑΚΩΝ ΔΑΣΜΩΝ",
        "ΑΕΡΟΠΟΡΙΚΗ ΣΤΡΑΤΟΛΟΓΙΑ",
        "ΔΙΕΘΝΕΙΣ ΣΥΜΒΑΣΕΙΣ ΓΙΑ ΤΑ ΝΑΡΚΩΤΙΚΑ",
        "ΔΙΑΦΟΡΕΣ ΤΡΑΠΕΖΕΣ",
        "ΟΙΝΟΛΟΓΟΙ",
        "ΤΕΛΩΝΟΦΥΛΑΚΗ",
        "ΤΑΜΕΙΟ ΕΘΝΙΚΗΣ ΑΜΥΝΑΣ (T.EΘ.A.) - ΕΘΝΙΚΗ ΕΠΙΤΡΟΠΗ ΕΞΟΠΛΙΣΜΟΥ ΕΝΟΠΛΩΝ ΔΥΝΑΜΕΩΝ (Ε.Ε.Ε.Ε.Δ.)",
        "ΕΚΤΕΛΕΣΗ ΤΗΣ ΠΟΙΝΗΣ",
        "ΙΣΟΛΟΓΙΣΜΟΙ ΑΝΩΝΥΜΩΝ ΕΤΑΙΡΕΙΩΝ",
        "ΑΡΧΙΤΕΚΤΟΝΙΚΟΙ ΔΙΑΓΩΝΙΣΜΟΙ",
        "ΚΑΤΑΡΓΗΣΗ ΦΥΛΕΤΙΚΩΝ ΔΙΑΚΡΙΣΕΩΝ",
        "ΕΠΑΓΓΕΛΜΑΤΙΚΑ ΔΙΚΑΙΩΜΑΤΑ ΑΠΟΦΟΙΤΩΝ",
        "ΜΟΝΑΣΤΗΡΙΑΚΗ  ΠΕΡΙΟΥΣΙΑ   ΣΑΜΟΥ",
        "ΣΥΝΤΑΞΗ ΔΗΜΟΤΙΚΩΝ ΚΑΙ ΚΟΙΝΟΤΙΚΩΝ ΥΠΑΛΛΗΛΩΝ",
        "ΟΙΚΟΝΟΜΙΚΕΣ ΕΦΟΡΙΕΣ",
        "ΦΡΟΝΤΙΣΤΗΡΙΑ ΕΦΑΡΜΟΓΩΝ",
        "ΝΟΜΑΡΧΙΕΣ ΑΤΤΙΚΗΣ",
        "ΦΥΜΑΤΙΩΣΗ",
        "ΕΛΕΓΧΟΣ ΑΝΑΤΙΜΗΣΕΩΝ",
        "ΑΤΕΛΕΙΕΣ ΥΠΕΡ ΤΗΣ ΝΑΥΤΙΛΙΑΣ",
        "ΚΩΦΑΛΑΛΟΙ",
        "ΙΑΤΡΙΚΗ ΔΕΟΝΤΟΛΟΓΙΑ",
        "ΕΞΟΔΑ ΔΗΜΟΣΙΑΣ ΑΣΦΑΛΕΙΑΣ",
        "ΜΕ ΤΗΝ ΑΡΓΕΝΤΙΝΗ",
        "ΚΛΑΔΟΣ ΥΓΕΙΟΝΟΜΙΚΗΣ ΠΕΡΙΘΑΛΨΗΣ Τ.Α.Ε",
        "ΥΠΗΡΕΣΙΑ ΕΚΚΑΘΑΡΙΣΕΩΣ ΝΑΡΚΟΠΕΔΙΩΝ",
        "ΤΑΜΕΙΟ ΑΡΩΓΗΣ ΥΠΑΛΛΗΛΩΝ ΑΣΤΥΝΟΜΙΑΣ ΠΟΛΕΩΝ Τ.Α.Υ.Α.Π",
        "ΠΡΟΣΤΑΣΙΑ ΔΗΜΟΣΙΩΝ ΚΤΗΜΑΤΩΝ",
        "ΒΙΒΛΙΑ ΕΝΔΙΚΩΝ ΜΕΣΩΝ",
        "ΕΛΛΗΝΙΚΟΣ ΟΡΓΑΝΙΣΜΟΣ ΜΙΚΡΟΜΕΣΑΙΩΝ ΜΕΤΑΠΟΙΗΤΙΚΩΝ ΕΠΙΧΕΙΡΗΣΕΩΝ ΚΑΙ ΧΕΙΡΟΤΕΧΝΙΑΣ",
        "ΔΗΜΟΣΙΟΓΡΑΦΙΚΟΣ ΧΑΡΤΗΣ",
        "ΦΟΡΟΣ ΓΑΜΙΚΩΝ ΣΥΜΦΩΝΩΝ ΙΣΡΑΗΛΙΤΩΝ",
        "ΥΠΟΤΡΟΦΙΑΙ ΚΤΗΝΙΑΤΡΙΚΗΣ",
        "ΑΠΟΔΟΧΕΣ ΠΡΟΣΩΠΙΚΟΥ ΙΔΙΩΤΙΚΟΥ ΔΙΚΑΙΟΥ",
        "ΕΠΙΒΑΤΗΓΑ ΑΚΤΟΠΛΟΙΚΑ ΠΛΟΙΑ",
        "ΠΑΛΑΙΟΙ ΔΗΜΟΣΙΟΥΠΑΛΛΗΛΙΚΟΙ ΝΟΜΟΙ",
        "ΚΩΔΙΚΑΣ ΠΕΡΙ ΚΛΗΡΟΔΟΤΗΜΑΤΩΝ",
        "ΟΙΚΟΝΟΜΙΚΗ ΕΠΙΘΕΩΡΗΣΗ",
        "ΚΤΗΜΑΤΟΓΡΑΦΗΣΗ ΔΑΣΩΝ",
        "ΟΡΓΑΝΙΚΕΣ ΘΕΣΕΙΣ",
        "ΠΕΡΙΟΡΙΣΜΟΣ ΧΡΗΣΗΣ ΟΡΙΣΜΕΝΩΝ ΣΥΜΒΑΤΙΚΩΝ ΟΠΛΩΝ",
        "ΑΓΙΟΝ ΟΡΟΣ",
        "ΚΥΡΩΣΕΙΣ ΦΟΡΟΛΟΓΙΚΩΝ ΠΑΡΑΒΑΣΕΩΝ",
        "ΚΑΤΑΣΤΑΣΗ ΠΡΟΣΩΠΙΚΟΥ Ο.Γ.Α",
        "ΕΠΑΝΑΠΑΤΡΙΣΜΟΣ ΚΕΦΑΛΑΙΩΝ",
        "ΜΑΘΗΤΕΣ ΤΕΧΝΙΤΕΣ",
        "ΔΙΑΒΙΒΑΣΕΙΣ",
        "ΕΜΜΙΣΘΟΙ ΚΑΙ ΠΟΙΝΙΚΟΙ ΔΙΚ. ΕΠΙΜΕΛΗΤΕΣ",
        "ΣΥΜΒΑΣΕΙΣ ΔΙΚΑΣΤΙΚΗΣ ΣΥΝΔΡΟΜΗΣ",
        "ΔΗΜΟΣΙΑ ΕΠΙΧΕΙΡΗΣΗ ΠΕΤΡΕΛΑΙΟΥ",
        "ΕΛΛΗΝΙΚΗ ΤΡΑΠΕΖΑ ΒΙΟΜΗΧΑΝΙΚΗΣ ΑΝΑΠΤΥΞΕΩΣ ΑΝΩΝΥΜΟΣ ΕΤΑΙΡΕΙΑ (Ε.Τ.Β.Α.  Α.Ε.)",
        "ΕΙΔΙΚΟΤΗΤΕΣ ΚΑΙ ΤΡΟΠΟΣ ΕΙΣΟΔΟΥ ΣΤΕΛΕΧΩΝ",
        "ΠΡΟΣΤΑΣΙΑ ΕΡΓΑΖΟΜΕΝΩΝ ΣΤΗΝ ΗΜΕΔΑΠΗ - ΣΩΜΑ ΕΠΙΘΕΩΡΗΣΗΣ ΕΡΓΑΣΙΑΣ",
        "ΙΝΣΤΙΤΟΥΤΟ ΩΚΕΑΝΟΓΡΑΦΙΚΩΝ ΚΑΙ ΑΛΙΕΥΤΙΚΩΝ ΕΡΕΥΝΩΝ",
        "ΕΛΕΓΧΟΣ ΑΠΟΛΥΣΕΩΝ ΜΙΣΘΩΤΩΝ",
        "ΠΑΝΕΛΛΗΝΙΑ ΕΚΘΕΣΗ ΛΑΜΙΑΣ",
        "ΚΥΡΙΑΚΗ ΑΡΓΙΑ ΚΑΙ ΑΛΛΕΣ ΥΠΟΧΡΕΩΤΙΚΕΣ ΑΡΓΙΕΣ",
        "ΚΛΑΔΟΣ ΥΓΕΙΑΣ Ο.Α.Ε.Ε",
        "ΟΡΚΟΣ ΣΤΡΑΤΙΩΤΙΚΩΝ",
        "ΕΜΠΟΡΙΚΑ ΒΙΒΛΙΑ",
        "ΥΓΕΙΟΝΟΜΙΚΕΣ ΕΠΙΤΡΟΠΕΣ ΕΝΟΠΛΩΝ ΔΥΝΑΜΕΩΝ",
        "ΑΓΙΟΣ ΒΙΚΕΝΤΙΟΣ-ΓΡΕΝΑΔΙΝΟΙ, ΑΓΙΟΣ ΜΑΡΙΝΟΣ Κ.ΛΠ",
        "ΑΠΟΖΗΜΙΩΣΗ ΔΙΑΤΕΛΕΣΑΝΤΩΝ ΠΡΩΘΥΠΟΥΡΓΩΝ",
        "ΑΣΦΑΛΙΣΗ ΛΟΓΟΤΕΧΝΩΝ ΚΑΙ ΚΑΛΛΙΤΕΧΝΩΝ",
        "ΠΕΙΘΑΡΧΙΚΑ ΣΥΜΒΟΥΛΙΑ",
        "ΕΤΑΙΡΙΕΣ ΧΡΗΜΑΤΟΔΟΤΙΚΗΣ ΜΙΣΘΩΣΗΣ",
        "ΚΟΙΝΩΝΙΚΗ ΥΠΗΡΕΣΙΑ ΦΥΛΑΚΩΝ",
        "ΚΑΝΟΝΙΣΜΟΣ ΥΠΗΡΕΣΙΩΝ ΑΓΡΟΦΥΛΑΚΗΣ",
        "ΑΣΦΑΛΙΣΗ ΣΤΟ ΙΚΑ",
        "ΕΜΠΟΡΙΚΟΙ ΣΥΜΒΟΥΛΟΙ ΚΑΙ ΑΚΟΛΟΥΘΟΙ",
        "ΕΠΙΚΟΥΡΟΙ ΠΑΡΑΤΗΡΗΤΕΣ",
        "ΥΠΟΤΡΟΦΙΕΣ",
        "ΚΕΝΤΡΟ ΠΡΟΓΡΑΜΜΑΤΙΣΜΟΥ",
        "ΠΡΩΤΕΣ ΥΛΕΣ ΣΟΚΟΛΑΤΟΠΟΙΙΑΣ",
        "ΕΠΙΤΡΟΠΗ ΚΗΠΩΝ ΚΑΙ ΔΕΝΔΡΟΣΤΟΙΧΙΩΝ",
        "ΚΙΝΗΤΟ ΕΠΙΣΗΜΑ",
        "ΣΥΝΔΙΚΑΛΙΣΜΟΣ ΔΗΜΟΣΙΩΝ ΥΠΑΛΛΗΛΩΝ",
        "ΤΑΜΕΙΟ ΠΡΟΝΟΙΑΣ Π.Ν",
        "ΟΡΓΑΝΙΚΕΣ ΔΙΑΤΑΞΕΙΣ ΤΑΜΕΙΟΥ ΠΑΡΑΚΑΤΑΘΗΚΩΝ ΚΑΙ ΔΑΝΕΙΩΝ",
        "ΑΔΕΙΕΣ ΗΝΙΟΧΙΑΣ",
        "ΥΠΗΡΕΣΙΑ ΠΡΟΓΡΑΜΜΑΤΙΣΜΟΥ ΚΑΙ ΜΕΛΕΤΩΝ",
        "ΚΡΑΤΙΚΑ ΑΥΤΟΚΙΝΗΤΑ",
        "ΑΤΟΜΙΚΗ ΚΑΤΑΓΓΕΛΙΑ ΣΥΜΒΑΣΕΩΣ ΕΡΓΑΣΙΑΣ",
        "ΠΟΛΥΤΕΚΝΟΙ",
        "ΙΣΤΟΡΙΚΟ ΑΡΧΕΙΟ ΜΑΚΕΔΟΝΙΑΣ",
        "ΑΣΦΑΛΙΣΗ ΑΥΤΟΚΙΝΗΤΙΚΩΝ ΑΤΥΧΗΜΑΤΩΝ",
        "ΔΑΝΕΙΑ ΕΣΩΤΕΡΙΚΑ",
        "ΕΚΚΛΗΣΙΑ ΚΡΗΤΗΣ",
        "ΦΟΡΟΛΟΓΙΑ ΣΤΑΦΙΔΑΣ",
        "ΕΚΠΑΙΔΕΥΤΙΚΕΣ ΑΔΕΙΕΣ",
        "ΑΕΡΟΔΙΚΕΙΑ",
        "ΕΠΙΔΟΜΑ ΑΣΘΕΝΕΙΑΣ",
        "ΘΕΣΕΙΣ ΣΥΜΒΟΛΑΙΟΓΡΑΦΩΝ",
        "ΑΓΟΡΑ ΣΥΝΑΛΛΑΓΜΑΤΟΣ",
        "ΝΟΜΙΚΟ ΣΥΜΒΟΥΛΙΟ ΤΟΥ ΚΡΑΤΟΥΣ (Ν.Σ.Κ.)",
        "ΦΟΡΟΛΟΓΙΑ ΜΕΤΑΒΙΒΑΣΗΣ",
        "ΣΥΜΒΟΥΛΙΑ - ΕΠΙΤΡΟΠΕΣ - ΙΝΣΤΙΤΟΥΤΑ ΕΡΓΑΣΙΑΣ ΚΑΙ ΚΟΙΝΩΝΙΚΩΝ ΑΣΦΑΛΙΣΕΩΝ",
        "ΤΕΛΗ ΕΙΣΙΤΗΡΙΩΝ ΚΑΙ ΚΟΜΙΣΤΡΩΝ",
        "ΟΙΚΟΝΟΜΙΚΗ ΥΠΗΡΕΣΙΑ ΥΓΕΙΟΝΟΜΙΚΟΥ ΣΩΜΑΤΟΣ",
        "ΠΡΟΣΩΠΙΚΟ ΣΩΜΑΤΩΝ ΑΣΦΑΛΕΙΑΣ ΜΕ ΣΧΕΣΗ ΙΔΙΩΤΙΚΟΥ ΔΙΚΑΙΟΥ",
        "ΑΡΤΕΡΓΑΤΕΣ",
        "ΕΥΚΟΛΙΕΣ ΣΕ ΦΟΙΤΗΤΕΣ",
        "ΣΥΝΕΤΑΙΡΙΣΜΟΙ ΚΟΙΝΗΣ ΧΟΡΤΟΝΟΜΗΣ ΚΑΙ ΣΥΝΙΔΙΟΚΤΗΣΙΑΣ",
        "ΤΑΜΕΙΟ ΑΣΦΑΛΙΣΕΩΣ ΠΡΟΣΩΠΙΚΟΥ ΠΕΡΙΦΕΡΕΙΑΚΟΥ ΓΕΝΙΚΟΥ ΝΟΣΟΚΟΜΕΙΟΥ Ο ΕΥΑΓΓΕΛΙΣΜΟΣ",
        "ΠΡΟΣΚΟΠΙΣΜΟΣ",
        "ΣΥΜΒΟΥΛΙΑ ΕΠΑΓΓΕΛΜΑΤΙΚΗΣ ΚΑΙ ΤΕΧΝΙΚΗΣ ΕΚΠΑΙΔΕΥΣΕΩΣ",
        "ΚΡΑΤΙΚΟΣ ΟΡΓΑΝΙΣΜΟΣ ΜΗΧΑΝΗΜΑΤΩΝ ΔΗΜΟΣΙΩΝ ΕΡΓΩΝ",
        "ΑΤΟΜΙΚΑ ΕΓΓΡΑΦΑ ΑΝΘΥΠΑΣΠΙΣΤΩΝ-ΥΠΑΞΙΩΜΑΤΙΚΩΝ",
        "ΔΙΑΦΟΡΕΣ ΣΧΟΛΕΣ",
        "ΒΙΒΛΙΑ ΔΗΜΟΣΙΕΥΣΕΩΣ ΔΙΑΘΗΚΩΝ",
        "ΚΑΝΟΝΙΣΜΟΙ ΠΡΟΣΩΠΙΚΟΥ ΣΥΓΚΟΙΝΩΝΙΑΚΩΝ ΕΠΙΧΕΙΡΗΣΕΩΝ",
        "ΤΟΥΡΙΣΤΙΚΟΙ ΤΟΠΟΙ",
        "ΙΝΣΤΙΤΟΥΤΟ ΞΕΝΩΝ ΓΛΩΣΣΩΝ ΚΑΙ ΦΙΛΟΛΟΓΙΩΝ",
        "ΚΑΠΝΟΠΩΛΕΣ",
        "ΑΓΩΓΕΣ ΓΙΑΤΡΩΝ",
        "ΣΥΣΤΑΣΗ ΚΑΙ ΑΠΟΔΟΣΗ ΠΑΡΑΚΑΤΑΘΗΚΩΝ ΑΠΟ Τ.Π. ΚΑΙ Δ",
        "ΑΔΙΚΗΜΑΤΑ ΔΙΑΠΡΑΤΤΟΜΕΝΑ ΣΤΑ ΚΡΑΤΗ-ΜΕΛΗ",
        "ΑΝΑΣΤΟΛΕΣ ΤΟΥ ΣΥΝΤΑΓΜΑΤΟΣ - ΚΑΤΑΣΤΑΣΗ ΠΟΛΙΟΡΚΙΑΣ",
        "ΣΥΜΒΑΣΕΙΣ ΠΑΡΟΧΗΣ ΑΣΦΑΛΕΙΑΣ (ΕΝΕΧΥΡΟ, ΥΠΟΘΗΚΗ Κ.ΛΠ.)",
        "ΤΑΜΕΙΟ ΑΣΦΑΛΙΣΕΩΣΝΑΥΤΙΚΩΝ ΠΡΑΚΤΟΡΩΝ ΚΑΙ ΥΠΑΛΛΗΛΩΝ (Τ.Α.Ν.Π.Υ.)",
        "ΑΝΩΤΑΤΟ ΣΥΓΚΟΙΝΩΝΙΑΚΟ ΣΥΜΒΟΥΛΙΟ",
        "ΠΡΕΒΕΝΤΟΡΙΑ",
        "ΑΝΑΒΟΛΗ ΣΤΡΑΤΕΥΣΕΩΣ",
        "ΕΙΔΙΚΑ ΛΗΞΙΑΡΧΕΙΑ",
        "ΓΕΩΤΕΧΝΙΚΟ ΕΠΙΜΕΛΗΤΗΡΙΟ",
        "ΥΓΕΙΟΝΟΜΙΚΑ ΔΙΚΑΙΩΜΑΤΑ",
        "ΤΑΜΕΙΟ ΕΠΙΚΟΥΡΙΚΗΣ ΑΣΦΑΛΙΣΗΣ ΕΚΠΑΙΔΕΥΤΙΚΩΝ",
        "ΚΑΖΑΚΣΤΑΝ – ΚΑΜΕΡΟΥΝ – ΚΑΝΑΔΑΣ Κ.ΛΠ",
        "ΣΥΝΤΑΞΕΙΣ ΘΥΜΑΤΩΝ ΑΠΟ ΤΟΝ ΑΜΑΧΟ ΠΛΗΘΥΣΜΟ",
        "ΦΙΛΟΣΟΦΙΚΗ ΣΧΟΛΗ",
        "ΕΚΤΕΛΩΝΙΣΜΟΣ ΤΑΧΥΔΡΟΜΙΚΩΝ ΔΕΜΑΤΩΝ",
        "ΥΔΡΕΥΣΗ ΘΕΣΣΑΛΟΝΙΚΗΣ",
        "ΣΥΜΦΩΝΙΕΣ ΠΕΡΙ ΠΛΩΤΩΝ ΟΔΩΝ",
        "ΑΝΑΚΗΡΥΞΗ ΤΗΣ ΑΝΕΞΑΡΤΗΣΙΑΣ",
        "ΕΠΙΤΡΟΠΗ ΟΛΥΜΠΙΑΚΩΝ ΑΓΩΝΩΝ",
        "ΟΙΝΟΠΑΡΑΓΩΓΗ ΑΤΤΙΚΟΒΟΙΩΤΙΑΣ",
        "ΕΚΠΤΩΣΕΙΣ ΥΠΕΡ ΕΞΑΓΩΓΕΩΝ",
        "ΦΟΡΟΛΟΓΙΑ ΚΛΗΡΟΝΟΜΙΩΝ, ΔΩΡΕΩΝ, ΓΟΝΙΚΩΝ ΠΑΡΟΧΩΝ",
        "ΟΡΦΑΝΟΤΡΟΦΕΙΑ ΚΑΙ ΟΙΚΟΤΡΟΦΕΙΑ",
        "ΜΕ ΤΗΝ ΟΥΡΑΓΟΥΑΗ",
        "ΜΕ ΤΗΝ ΑΥΣΤΡΙΑΚΗ",
        "ΔΙΑΦΟΡΟΙ ΦΟΡΟΙ ΚΑΤΑΝΑΛΩΣΕΩΣ",
        "ΔΙΕΥΘΥΝΣΗ ΕΦΕΔΡΩΝ - ΠΟΛΕΜΙΣΤΩΝ - ΑΓΩΝΙΣΤΩΝ",
        "ΑΓΡΟΤΙΚΕΣ ΟΙΚΟΚΥΡΙΚΕΣ ΣΧΟΛΕΣ",
        "ΞΥΛΕΙΑ",
        "ΒΙΒΛΙΑΡΙΑ ΥΓΕΙΑΣ ΕΡΓΑΤΩΝ",
        "ΣΧΟΛΗ ΑΞΙΩΜΑΤΙΚΩΝ ΣΤΡΑΤΙΩΤΙΚΩΝ ΥΠΗΡΕΣΙΩΝ",
        "ΝΟΜΑΡΧΙΑΚΕΣ ΚΑΙ ΔΗΜΟΤΙΚΕΣ ΕΚΛΟΓΕΣ",
        "ΕΓΓΥΗΣΕΙΣ ΚΑΙ ΔΑΝΕΙΑ ΤΟΥ ΔΗΜΟΣΙΟΥ",
        "ΥΠΟΥΡΓΕΙΟ ΑΝΑΠΤΥΞΗΣ",
        "ΤΑΚΤΙΚΑ ΔΙΟΙΚΗΤΙΚΑ ΔΙΚΑΣΤΗΡΙΑ - ΓΕΝΙΚΕΣ ΔΙΑΤΑΞΕΙΣ",
        "ΤΡΟΦΟΔΟΣΙΑ ΠΛΗΡΩΜΑΤΩΝ ΠΛΟΙΩΝ",
        "ΔΙΑΦΟΡΟΙ ΛΙΜΕΝΕΣ ΚΑΙ ΛΙΜΕΝΙΚΑ ΤΑΜΕΙΑ",
        "ΗΛΕΚΤΡΙΚΕΣ ΕΚΜΕΤΑΛΛΕΥΣΕΙΣ",
        "ΠΡΟΥΠΟΘΕΣΕΙΣ ΑΣΚΗΣΗΣ ΔΙΑΦΟΡΩΝ ΕΠΑΓΓΕΛΜΑΤΩΝ",
        "ΤΕΛΩΝΕΙΑΚΗ ΥΠΗΡΕΣΙΑ ΑΕΡΟΣΚΑΦΩΝ",
        "ΕΠΙΤΡΟΠΗ ΔΑΣΜΟΛΟΓΙΟΥ",
        "ΝΑΥΠΗΓΕΙΑ Π. ΝΑΥΤΙΚΟΥ",
        "ΒΙΟΜΗΧΑΝΙΚΕΣ ΚΑΙ ΕΠΙΧΕΙΡΗΜΑΤΙΚΕΣ ΠΕΡΙΟΧΕΣ",
        "ΙΑΤΡΟΔΙΚΑΣΤΕΣ",
        "ΑΘΛΗΤΙΣΜΟΣ ΕΝΟΠΛΩΝ ΔΥΝΑΜΕΩΝ",
        "ΟΡΓΑΝΙΣΜΟΣ ΣΥΚΩΝ",
        "ΚΑΝΟΝΙΣΜΟΣ ΑΣΘΕΝΕΙΑΣ ΤΑΜΕΙΟΥ ΣΥΝΤΑΞΕΩΝ ΕΦΗΜΕΡΙΔΟΠΩΛΩΝ ΚΑΙ ΥΠΑΛΛΗΛΩΝ ΠΡΑΚΤΟΡΕΙΩΝ (Τ.Σ.Ε.Υ.Π.)",
        "ΑΔΕΙΕΣ ΜΙΣΘΩΤΩΝ",
        "ΠΡΟΣΤΑΣΙΑ ΚΕΦΑΛΑΙΩΝ ΕΞΩΤΕΡΙΚΟΥ",
        "ΑΠΟΔΕΙΚΤΙΚΑ ΦΟΡΟΛΟΓΙΚΗΣ ΕΝΗΜΕΡΟΤΗΤΑΣ",
        "ΟΡΓΑΝΩΣΗ ΚΑΙ ΛΕΙΤΟΥΡΓΙΑ ΤΩΝ ΤΗΛΕΠΙΚΟΙΝΩΝΙΩΝ ΕΘΝΙΚΗ ΕΠΙΤΡΟΠΗ ΤΗΛΕΠΙΚΟΙΝΩΝΙΩΝ ΚΑΙ ΤΑΧΥΔΡΟΜΕΙΩΝ (Ε.Ε.Τ.Τ.)",
        "ΠΡΟΣΩΠΙΚΟ Ο.Τ.Ε",
        "ΒΑΣΙΛΙΚΑ ΙΔΡΥΜΑΤΑ",
        "ΑΠΟΚΑΤΑΣΤΑΣΗ ΠΛΗΓΕΝΤΩΝ ΑΠΟ ΕΚΡΗΞΗ ΠΛΟΙΟΥ ΣΤΗΝ ΚΡΗΤΗ",
        "ΕΚΜΕΤΑΛΛΕΥΣΗ ΔΥΝΑΜΕΩΣ ΡΕΟΝΤΩΝ ΥΔΑΤΩΝ",
        "ΚΑΚΟΥΡΓΙΟΔΙΚΕΙΑ",
        "ΚΕΝΤΡΙΚΕΣ ΑΓΟΡΕΣ ΑΛΛΩΝ ΠΟΛΕΩΝ",
        "ΤΑΜΕΙΟ ΑΛΛΗΛΟΒΟΗΘΕΙΑΣ Π.Ν",
        "ΕΚΛΟΓΙΚΟΙ ΚΑΤΑΛΟΓΟΙ ΚΑΙ ΒΙΒΛΙΑΡΙΑ",
        "ΥΠΗΡΕΣΙΑ ΕΓΓΕΙΩΝ ΒΕΛΤΙΩΣΕΩΝ",
        "ΤΟΥΡΙΣΤΙΚΗ ΑΝΑΠΤΥΞΗ",
        "ΝΟΜΟΘΕΣΙΑ ΠΕΡΙ ΣΥΜΒΑΣΕΩΣ ΕΡΓΑΣΙΑΣ",
        "ΕΛΕΓΧΟΣ ΕΚΡΗΚΤΙΚΩΝ ΥΛΩΝ",
        "ΜΑΚΕΔΟΝΙΚΟΙ ΣΙΔΗΡΟΔΡΟΜΟΙ",
        "ΔΙΕΥΚΟΛΥΝΣΕΙΣ ΣΕ ΔΗΜΟΣΙΟΥΣ ΥΠΑΛΛΗΛΟΥΣ",
        "ΣΤΡΑΤΙΩΤΙΚΕΣ ΥΠΟΧΡΕΩΣΕΙΣ ΕΠΑΝΑΠΑΤΡΙΖΟΜΕΝΩΝ",
        "ΔΙΑΚΡΙΣΗ ΕΜΠΟΡΙΚΩΝ ΠΡΑΞΕΩΝ",
        "ΟΡΓΑΝΙΣΜΟΣ ΕΛΛΗΝΙΚΩΝ ΓΕΩΡΓΙΚΩΝ ΑΣΦΑΛΙΣΕΩΝ (Ε.Λ.Γ.Α.)",
        "ΕΞΩΣΧΟΛΙΚΗ ΣΩΜΑΤΙΚΗ ΑΓΩΓΗ",
        "ΔΡΑΧΜΟΠΟΙΗΣΗ",
        "ΜΕ ΤΗ ΒΡΑΖΙΛΙΑ",
        "ΕΚΚΛΗΣΙΑΣΤΙΚΗ ΑΚΑΔΗΜΙΑ",
        "ΑΝΤΑΛΛΑΓΗ ΘΕΡΑΠΕΥΤΙΚΩΝ ΟΥΣΙΩΝ",
        "ΓΑΛΛΙΑ, ΓΕΡΜΑΝΙΑ Κ.ΛΠ",
        "ΝΟΜΟΠΑΡΑΣΚΕΥΑΣΤΙΚΕΣ ΕΠΙΤΡΟΠΕΣ",
        "ΚΥΒΕΡΝΕΙΟ ΘΕΣΣΑΛΟΝΙΚΗΣ",
        "ΣΤΡΑΤΙΩΤΙΚΟΙ ΑΚΟΛΟΥΘΟΙ",
        "ΔΙΑΘΕΣΗ ΑΠΟΣΤΡΑΓΓΙΖΟΜΕΝΩΝ ΓΑΙΩΝ",
        "ΓΕΝΙΚΕΣ ΔΙΑΤΑΞΕΙΣ ΓΙΑ ΡΑΔΙΟΦΩΝΙΑ – ΤΗΛΕΟΡΑΣΗ",
        "ΓΝΩΜΟΔΟΤΙΚΟ ΣΥΜΒΟΥΛΙΟ ΦΑΡΜΑΚΩΝ",
        "ΣΥΜΒΑΣΕΙΣ ΔΙΑΦΟΡΕΣ",
        "ΠΡΑΞΕΙΣ ΚΑΤΑ ΤΗΣ ΑΣΦΑΛΕΙΑΣ ΤΗΣ ΑΕΡΟΠΟΡΙΑΣ",
        "ΙΑΤΡΟΙ ΙΑΜΑΤΙΚΩΝ ΠΗΓΩΝ",
        "ΚΕΝΤΡΙΚΟ ΣΥΜΒΟΥΛΙΟ ΥΓΕΙΑΣ (ΚΕ.Σ.Υ.)",
        "ΑΝΩΤΑΤΟ ΣΥΜΒΟΥΛΙΟ ΔΗΜΟΣΙΩΝ ΥΠΑΛΛΗΛΩΝ",
        "ΥΠΟΥΡΓΕΙΟ ΕΝΕΡΓΕΙΑΣ ΚΑΙ ΦΥΣΙΚΩΝ ΠΟΡΩΝ",
        "ΤΕΧΝΙΚΗ ΕΚΜΕΤΑΛΛΕΥΣΗ ΕΛΑΦΡΩΝ ΑΕΡΟΠΛΑΝΩΝ Δ.Χ",
        "ΠΟΛΥΕΘΝΕΙΣ ΜΟΡΦΩΤΙΚΕΣ ΣΥΜΦΩΝΙΕΣ",
        "ΕΚΠΑΙΔΕΥΣΗ Λ.Σ",
        "ΠΡΟΣΤΑΣΙΑ ΕΛΕΥΘΕΡΟΥ ΑΝΤΑΓΩΝΙΣΜΟΥ",
        "ΕΘΝΙΚΗ ΕΠΙΤΡΟΠΗ ΔΙΕΘΝΟΥΣ ΕΜΠΟΡΙΚΟΥ ΕΠΙΜΕΛΗΤΗΡΙΟΥ",
        "ΟΡΓΑΝΙΣΜΟΣ",
        "ΤΕΛΩΝΕΙΑΚΕΣ ΠΑΡΑΚΑΤΑΘΗΚΕΣ",
        "ΕΛΕΓΧΟΣ ΟΡΓΑΝΙΣΜΩΝ ΚΟΙΝΩΝΙΚΗΣ ΠΟΛΙΤΙΚΗΣ",
        "ΕΝΩΣΕΙΣ ΑΠΟΣΤΡΑΤΩΝ ΑΞΙΩΜΑΤΙΚΩΝ ΕΝΟΠΛΩΝ ΔΥΝΑΜΕΩΝ",
        "ΦΥΛΛΑ ΠΟΙΟΤΗΤΑΣ ΑΞΙΩΜΑΤΙΚΩΝ Π.Ν",
        "ΙΝΣΤΙΤΟΥΤΟ ΓΕΩΛΟΓΙΚΩΝ ΚΑΙ ΜΕΤΑΛΛΕΥΤΙΚΩΝ ΕΡΕΥΝΩΝ",
        "ΛΑΟΓΡΑΦΙΚΟ ΚΑΙ ΕΘΝΟΛΟΓΙΚΟ ΜΟΥΣΕΙΟ ΜΑΚΕΔΟΝΙΑΣ - ΘΡΑΚΗΣ",
        "ΠΡΩΤΕΣ ΥΛΕΣ ΤΑΠΗΤΟΥΡΓΙΑΣ",
        "ΠΑΝΕΠΙΣΤΗΜΙΟ ΚΡΗΤΗΣ",
        "ΚΩΔΙΚΑΣ ΟΔΙΚΗΣ ΚΥΚΛΟΦΟΡΙΑΣ",
        "ΦΑΡΜΑΚΕΥΤΙΚΗ ΠΕΡΙΘΑΛΨΗ",
        "ΜΕΛΕΤΕΣ ΠΡΟΓΡΑΜΜΑΤΟΣ ΔΗΜΟΣΙΩΝ ΕΠΕΝΔΥΣΕΩΝ",
        "ΕΠΙΔΟΣΗ ΔΙΑ ΤΟΥ ΤΑΧΥΔΡΟΜΕΙΟΥ",
        "ΠΑΝΕΠΙΣΤΗΜΙΟ ΘΡΑΚΗΣ",
        "ΗΘΙΚΕΣ ΑΜΟΙΒΕΣ",
        "ΔΗΜΟΣΙΑ ΚΤΗΜΑΤΑ ΣΤΗ ΔΩΔΕΚΑΝΗΣΟ",
        "ΣΥΜΒΑΣΕΙΣ ΔΙΚΑΣΤΙΚΗΣ ΑΝΤΙΛΗΨΕΩΣ",
        "ΠΕΡΙΟΡΙΣΜΟΙ ΑΛΙΕΙΑΣ",
        "ΠΥΡΗΝΙΚΕΣ ΕΓΚΑΤΑΣΤΑΣΕΙΣ",
        "ΩΡΕΣ ΕΡΓΑΣΙΑΣ ΠΡΟΣΩΠΙΚΟΥ ΑΥΤΟΚΙΝΗΤΩΝ",
        "ΕΓΓΡΑΦΕΣ, ΕΞΕΤΑΣΕΙΣ, ΑΝΑΛΥΤΙΚΑ ΠΡΟΓΡΑΜΜΑΤΑ",
        "ΔΙΚΑΙΩΜΑΤΑ ΤΕΛΩΝΕΙΑΚΩΝ ΕΡΓΑΣΙΩΝ",
        "ΤΑΜΕΙΟ ΑΣΦΑΛΙΣΕΩΣ ΑΥΤΟΚΙΝΗΤΙΣΤΩΝ (Τ.Σ.Α.)",
        "ΤΗΛΕΦΩΝΙΚΟΣ ΚΑΝΟΝΙΣΜΟΣ",
        "ΦΟΡΟΛΟΓΙΑ ΑΣΦΑΛΙΣΤΡΩΝ",
        "ΔΙΕΘΝΗΣ ΥΔΡΟΓΡΑΦΙΚΟΣ ΟΡΓΑΝΙΣΜΟΣ",
        "ΕΠΑΡΧΙΕΣ",
        "ΑΓΡΟΤ. ΑΠΟΚΑΤΑΣΤΑΣΗ ΠΡΟΣΦΥΓΩΝ",
        "ΓΕΝΙΚΑ ΓΙΑ ΤΑ ΘΕΑΤΡΑ",
        "ΣΥΜΒΑΣΕΙΣ ΔΙΩΞΕΩΣ ΛΑΘΡΕΜΠΟΡΙΟΥ",
        "ΜΗΧΑΝΕΣ ΠΡΟΠΛΗΡΩΜΗΣ ΤΕΛΩΝ",
        "ΟΡΓΑΝΙΣΜΟΣ ΚΡΑΤΙΚΩΝ ΘΕΑΤΡΩΝ",
        "ΚΕΝΤΡΟ ΗΛΕΚΤΡΟΝΙΚΟΥ ΥΠΟΛΟΓΙΣΤΟΥ ΚΟΙΝΩΝΙΚΩΝ ΥΠΗΡΕΣΙΩΝ",
        "ΦΟΡΟΣ ΠΡΟΣΤΙΘΕΜΕΝΗΣ ΑΞΙΑΣ",
        "ΤΑΜΕΙΑ ΑΡΩΓΗΣ ΤΤΤ. ΥΠΑΛΛΗΛΩΝ",
        "ΣΩΜΑ ΟΡΚΩΤΩΝ ΕΛΕΓΚΤΩΝ ΛΟΓΙΣΤΩΝ (Σ.Ο.Ε.Λ.), ΕΠΙΤΡΟΠΗ ΛΟΓΙΣΤΙΚΗΣ ΤΥΠΟΠΟΙΗΣΗΣ ΚΑΙ ΕΛΕΓΧΩΝ (Ε.Λ.Τ.Ε.)",
        "ΑΓΡΟΤΙΚΑ ΝΗΠΙΟΤΡΟΦΕΙΑ",
        "ΣΧΕΔΙΟ ΠΟΛΕΩΣ ΑΘΗΝΩΝ ΠΕΙΡΑΙΩΣ",
        "ΜΙΣΘΩΣΕΙΣ ΑΚΙΝΗΤΩΝ Ο.Δ.Ε.Π",
        "ΕΛΕΓΧΟΣ ΣΠΟΡΟΠΑΡΑΓΩΓΗΣ",
        "ΑΜΥΝΤΙΚΕΣ ΠΕΡΙΟΧΕΣ ΚΑΙ Ν. ΟΧΥΡΑ",
        "ΟΔΟΙΠΟΡΙΚΑ",
        "ΠΟΡΟΙ ΟΡΓΑΝΙΣΜΩΝ ΤΟΥΡΙΣΜΟΥ",
        "ΔΙΕΘΝΕΣ ΔΙΚΑΣΤΗΡΙΟ",
        "ΟΙΚΟΝΟΜΙΚΗ ΜΕΡΙΜΝΑ ΕΝΟΠΛΩΝ ΔΥΝΑΜΕΩΝ",
        "ΓΕΝΙΚΟ ΝΟΣΟΚΟΜΕΙΟ ΕΜΠΟΡΙΚΟΥ ΝΑΥΤΙΚΟΥ",
        "ΝΟΜΙΚΗ ΒΟΗΘΕΙΑ ΣΕ ΠΟΛΙΤΕΣ ΧΑΜΗΛΟΥ ΕΙΣΟΔΗΜΑΤΟΣ",
        "ΣΥΜΒΟΛΑΙΟΓΡΑΦΙΚΟΙ ΣΥΛΛΟΓΟΙ",
        "ΥΠΟΥΡΓΕΙΟ ΣΤΡΑΤΙΩΤΙΚΩΝ",
        "ΠΡΟΣΩΠΙΚΟ Ε.Μ.Π",
        "ΥΠΟΥΡΓΕΙΟ ΕΡΓΑΣΙΑΣ",
        "ΑΓΟΝΕΣ ΓΡΑΜΜΕΣ",
        "ΜΟΝΟΠΩΛΙΟ ΠΕΤΡΕΛΑΙΟΥ",
        "ΠΡΟΛΗΨΗ ΡΥΠΑΝΣΗΣ ΤΗΣ ΘΑΛΑΣΣΑΣ",
        "ΧΩΡΙΚΗ ΔΙΚΑΙΟΔΟΣΙΑ ΤΕΛΩΝΕΙΑΚΩΝ ΑΡΧΩΝ",
        "ΕΠΑΓΓΕΛΜΑΤΙΚΑ ΣΩΜΑΤΕΙΑ",
        "ΥΠΗΡΕΣΙΑ ΑΓΡΟΤΙΚΗΣ ΑΣΦΑΛΕΙΑΣ",
        "ΑΞΙΟΠΟΙΗΣΗ ΕΚΚΛΗΣΙΑΣΤΙΚΗΣ ΠΕΡΙΟΥΣΙΑΣ",
        "ΕΜΠΟΡΙΚΟΙ ΑΝΤΙΠΡΟΣΩΠΟΙ",
        "ΕΝΩΣΕΙΣ ΕΦΕΔΡΩΝ ΑΞΙΩΜΑΤΙΚΩΝ",
        "ΑΤΕΛΕΙΕΣ ΥΠΕΡ ΤΗΣ ΒΙΟΜΗΧΑΝΙΑΣ",
        "ΛΟΓΙΣΤΙΚΟ ΕΙΔΙΚΩΝ ΤΑΜΕΙΩΝ Ν.Π.Δ.Δ",
        "ΣΥΜΒΑΣΗ ΓΙΑ ΔΕΙΓΜΑΤΑ ΚΛΠ",
        "ΕΡΓΟΛΗΠΤΕΣ ΔΗΜΟΣΙΩΝ ΕΡΓΩΝ",
        "ΕΠΑΝΕΠΟΙΚΙΣΜΟΣ ΠΑΡΑΜΕΘΟΡΙΩΝ ΠΕΡΙΟΧΩΝ",
        "ΦΑΡΙΚΑ ΤΕΛΗ",
        "ΛΑΤΟΜΕΙΑ ΜΑΡΜΑΡΩΝ",
        "ΠΟΣΟΣΤΟ ΣΥΜΜΕΤΟΧΗΣ ΑΣΦΑΛΙΣΜΕΝΩΝ",
        "ΑΣΦΑΛΕΙΑ ΑΝΘΡΩΠΙΝΗΣ ΖΩΗΣ ΣΤΗ ΘΑΛΑΣΣΑ",
        "ΟΡΓΑΝΙΚΟΙ ΝΟΜΟΙ ΠΕΡΙ ΦΥΛΑΚΩΝ",
        "ΛΑΘΡΕΜΠΟΡΙΑ",
        "ΑΣΦΑΛΙΣΤΙΚΕΣ ΔΙΑΤΑΞΕΙΣ ΓΕΝΙΚΑ",
        "ΕΙΣΑΓΩΓΗ ΧΛΩΡΙΚΟΥ ΚΑΛΙΟΥ",
        "ΙΝΣΤΙΤΟΥΤΟ ΓΕΩΠΟΝΙΚΩΝ ΕΠΙΣΤΗΜΩΝ",
        "ΕΠΙΔΟΜΑ ΠΑΣΧΑ - ΧΡΙΣΤΟΥΓΕΝΝΩΝ",
        "ΓΕΩΡΓΙΚΟΙ ΣΥΝΕΤΑΙΡΙΣΜΟΙ ΑΛΛΗΛΑΣΦΑΛΕΙΑΣ",
        "ΟΡΓΑΝΙΣΜΟΣ ΦΟΡΟΛΟΓΙΚΩΝ ΔΙΚΑΣΤΗΡΙΩΝ",
        "ΕΠΙΔΟΣΗ",
        "ΙΔΡΥΜΑ ΚΡΑΤΙΚΩΝ ΥΠΟΤΡΟΦΙΩΝ",
        "ΥΓΕΙΟΝΟΜΙΚΟΣ ΚΑΝΟΝΙΣΜΟΣ ΑΕΡΟΥΓΕΙΟΝΟΜΕΙΩΝ",
        "ΟΦΕΙΛΕΣ ΠΡΟΣ ΤΟ ΔΗΜΟΣΙΟ",
        "ΠΡΑΚΤΟΡΕΙΑ ΕΙΔΗΣΕΩΝ",
        "ΕΛΕΓΧΟΣ ΚΑΙ ΕΠΟΠΤΕΙΑ ΞΕΝΟΔΟΧΕΙΩΝ ΚΛΠ",
        "ΚΟΙΝΑ ΤΑΜΕΙΑ ΕΚΜΕΤΑΛΛΕΥΣΕΩΣ ΛΕΩΦΟΡΕΙΩΝ (Κ.Τ.Ε.Λ.)",
        "ΚΑΤΩΤΑΤΑ ΟΡΙΑ ΜΙΣΘΩΝ ΚΑΙ ΗΜΕΡΟΜΙΣΘΙΩΝ",
        "ΣΥΝΤΗΡΗΤΙΚΗ ΚΑΤΑΣΧΕΣΗ ΠΛΟΙΩΝ",
        "ΥΠΗΡΕΣΙΑ ΠΡΟΣΤΑΣΙΑΣ ΕΡΓΑΖΟΜΕΝΩΝ ΣΤΗΝ ΑΛΛΟΔΑΠΗ",
        "ΕΥΡΩΠΑΙΚΟΣ ΟΡΓΑΝΙΣΜΟΣ ΠΥΡΗΝΙΚΩΝ ΕΡΕΥΝΩΝ",
        "ΒΙΒΛΙΑ ΓΕΩΡΓΙΚΩΝ ΣΥΝΕΤΑΙΡΙΣΜΩΝ",
        "ΠΟΛΙΤΙΚΕΣ ΚΑΙ ΣΤΡΑΤΙΩΤΙΚΕΣ ΣΥΝΤΑΞΕΙΣ",
        "ΜΕΤΑΤΡΟΠΗ ΜΕΤΟΧΩΝ ΣΕ ΟΝΟΜΑΣΤΙΚΕΣ",
        "ΕΙΔΙΚΟΙ ΦΡΟΥΡΟΙ",
        "ΥΠΗΡΕΣΙΑ ΕΘΝΙΚΗΣ ΑΣΦΑΛΕΙΑΣ",
        "ΡΥΘΜΙΣΤΙΚΟΣ ΦΟΡΟΣ",
        "ΛΙΜΑΝΙ ΗΡΑΚΛΕΙΟΥ ΚΡΗΤΗΣ ΚΑΙ",
        "ΕΚΚΛΗΣΙΑΣΤΙΚΕΣ ΥΠΟΤΡΟΦΙΕΣ",
        "ΦΟΡΟΛΟΓΙΑ ΟΙΝΟΥ",
        "ΔΙΕΘΝΗΣ ΥΓΕΙΟΝΟΜΙΚΗ ΣΥΜΒΑΣΗ ΑΕΡΟΝΑΥΤΙΛΙΑΣ",
        "ΤΑΜΕΙΟ ΑΡΩΓΗΣ ΥΠΑΛΛΗΛΩΝ",
        "ΚΟΙΝΩΝΙΚΗ ΑΣΦΑΛΙΣΗ ΑΓΡΟΤΩΝ",
        "ΚΥΡΟΣ ΣΥΜΒΟΛΑΙΟΓΡΑΦΙΚΩΝ ΠΡΑΞΕΩΝ",
        "ΦΟΡΟΛΟΓΙΑ ΥΠΕΡΑΞΙΑΣ ΑΚΙΝΗΤΩΝ",
        "ΝΗΠΙΑΓΩΓΕΙΑ",
        "ΕΚΘΕΜΑΤΑ ΚΑΙ ΔΕΙΓΜΑΤΑ",
        "ΥΓΕΙΟΝΟΜΙΚΟ ΣΩΜΑ ΑΕΡΟΠΟΡΙΑΣ",
        "ΠΛΗΡΩΜΗ ΜΙΣΘΩΝ ΚΑΙ ΗΜΕΡΟΜΙΣΘΙΩΝ",
        "ΚΩΔΙΚΑΣ ΦΟΡΟΛΟΓΙΑΣ ΚΑΠΝΟΥ",
        "ΟΡΙΑ",
        "ΔΙΚΑΙΟΣΤΑΣΙΑ ΣΕΙΣΜΟΠΑΘΩΝ, ΠΥΡΟΠΑΘΩΝ, ΠΡΟΣΦΥΓΩΝ ΚΛΠ",
        "ΧΡΕΗ ΚΛΗΡΟΝΟΜΙΩΝ",
        "ΠΡΟΣΩΠΙΚΟΝ ΙΔΡΥΜΑΤΩΝ ΠΑΙΔΙΚΗΣ ΠΡΟΣΤΑΣΙΑΣ",
        "ΜΙΣΘΩΣΕΙΣ ΚΑΙ ΑΓΟΡΕΣ",
        "ΠΑΛΑΙΟΤΕΡΑΙ ΕΚΚΑΘΑΡΙΣΕΙΣ",
        "ΟΙΚΟΝΟΜΙΚΗ ΑΠΟΚΑΤΑΣΤΑΣΗ ΑΓΡΟΤΩΝ",
        "ΑΠΑΛΛΟΤΡΙΩΣΕΙΣ ΓΙΑ ΔΗΜΟΤΙΚΑ ΚΑΙ ΚΟΙΝΟΤΙΚΑ ΕΡΓΑ",
        "ΜΗΤΡΩΟ ΑΓΡΟΤΩΝ",
        "ΚΑΝΟΝΙΣΜΟΣ ΔΙΕΥΚΟΛΥΝΣΕΩΝ",
        "ΚΡΑΤΙΚΟ ΕΡΓΟΣΤΑΣΙΟ ΑΕΡΟΠΛΑΝΩΝ",
        "ΕΠΑΓΓΕΛΜΑΤΙΚΑ ΕΝΔΕΙΚΤΙΚΑ",
        "ΑΥΘΑΙΡΕΤΕΣ ΚΑΤΑΣΚΕΥΕΣ",
        "ΕΓΚΑΤΑΛΕΛΕΙΜΜΕΝΕΣ ΕΚΤΑΣΕΙΣ",
        "ΥΠΟΥΡΓΕΙΟ ΔΗΜΟΣΙΩΝ ΄ΕΡΓΩΝ",
        "ΠΡΟΝΟΙΑ Β. ΕΛΛΑΔΟΣ",
        "ΔΙΚΑΣΤΙΚΟ ΕΝΣΗΜΟ - ΑΓΩΓΟΣΗΜΟ",
        "ΤΑΧΥΔΡΟΜΙΚΗ ΑΝΤΑΠΟΚΡΙΣΗ",
        "ΕΣΩΤΕΡΙΚΗ ΝΟΜΟΘΕΣΙΑ",
        "ΦΟΡΟΛΟΓΙΑ ΤΣΙΓΑΡΟΧΑΡΤΟΥ",
        "ΟΡΓΑΝΙΚΕΣ ΘΕΣΕΙΣ ΑΞΙΩΜΑΤΙΚΩΝ",
        "ΜΑΙΕΥΤΙΚΗ ΠΕΡΙΘΑΛΨΗ",
        "ΑΔΕΙΕΣ ΣΤΡΑΤΙΩΤΙΚΩΝ",
        "ΟΡΓΑΝΙΚΕΣ ΔΙΑΤΑΞΕΙΣ ΕΛΛΗΝΙΚΗΣ ΑΣΤΥΝΟΜΙΑΣ",
        "ΠΟΙΝΙΚΟΣ ΚΑΙ ΠΕΙΘΑΡΧΙΚΟΣ ΚΩΔΙΚΑΣ",
        "ΑΝΥΠΟΤΑΚΤΟΙ",
        "ΔΙΕΥΘΥΝΣΗ ΤΕΛΩΝΕΙΩΝ ΘΕΣΣΑΛΟΝΙΚΗΣ",
        "ΠΕΡΙΦΕΡΕΙΕΣ ΛΙΜΕΝΙΚΩΝ ΑΡΧΩΝ",
        "ΑΣΦΑΛΙΣΗ ΚΑΙ ΕΙΣΠΡΑΞΗ ΠΟΡΩΝ Τ.Ε.Β.Ε",
        "ΣΙΔΗΡΟΣ",
        "ΓΕΝΙΚΗ ΓΡΑΜΜΑΤΕΙΑ ΕΜΠΟΡΙΟΥ",
        "ΔΙΑΧΕΙΡΙΣΗ ΙΣΡΑΗΛΙΤΙΚΩΝ ΠΕΡΟΥΣΙΩΝ",
        "ΛΙΠΟΤΑΞΙΑ",
        "ΒΑΡΕΑ ΚΑΙ ΑΝΘΥΓΙΕΙΝΑ ΕΠΑΓΓΕΛΜΑΤΑ",
        "ΕΙΔΙΚΟ ΤΑΜΕΙΟ ΜΗΧΑΝΗΜΑΤΩΝ",
        "ΛΕΩΦΟΡΕΙΑ ΠΕΡΙΟΧΗΣ ΠΡΩΤΕΥΟΥΣΑΣ",
        "ΑΝΑΜΟΡΦΩΤΙΚΑ ΚΑΤΑΣΤΗΜΑΤΑ",
        "ΥΓΕΙΟΝΟΜΙΚΟ ΣΩΜΑ",
        "ΟΡΓΑΝΙΣΜΟΣ ΥΠΟΥΡΓΕΙΟΥ ΕΡΓΑΣΙΑΣ",
        "ΔΙΩΡΥΓΑ ΚΟΡΙΝΘΟΥ",
        "ΠΕΡΙΘΑΛΨΗ ΦΥΜΑΤΙΚΩΝ ΑΣΦΑΛΙΣΜΕΝΩΝ",
        "ΚΟΙΝΩΝΙΚΟΣ ΕΛΕΓΧΟΣ ΔΙΟΙΚΗΣΗΣ - ΑΝΤΙΓΡΑΦΕΙΟΚΡΑΤΙΚΑ ΜΕΤΡΑ -ΕΚΚΑΘΑΡΙΣΗ ΑΡΧΕΙΩΝ",
        "ΒΙΒΛΙΑ ΥΠΟΘΕΣΕΩΝ ΕΚΟΥΣΙΑΣ ΔΙΚΑΙΟΔΟΣΙΑΣ",
        "ΖΑΧΑΡΗ",
        "ΒΟΡΕΙΟΑΤΛΑΝΤΙΚΗ ΑΜΥΝΤΙΚΗ ΟΡΓΑΝΩΣΗ (Ν.Α.Τ.Ο)",
        "ΤΑΜΕΙΟ ΣΥΝΤΑΞΕΩΝ ΠΡΟΣΩΠΙΚΟΥ ΕΤΑΙΡΕΙΑΣ ΓΕΝΙΚΩΝ ΑΠΟΘΗΚΩΝ",
        "ΝΟΜΙΚΗ ΚΑΤΑΣΤΑΣΗ ΠΡΟΣΦΥΓΩΝ",
        "ΔΙΚΑΣΤΗΡΙΟ ΛΕΙΩΝ",
        "ΔΙΕΘΝΗΣ ΟΡΓΑΝΩΣΗ ΕΡΓΑΣΙΑΣ",
        "ΠΡΟΜΗΘΕΙΕΣ–ΜΙΣΘΩΣΕΙΣ–ΕΡΓΑ Ο.Γ.Α",
        "ΠΕΡΙΘΑΛΨΗ ΠΡΟΣΩΠΙΚΟΥ Ο.Γ.Α",
        "ΧΟΡΗΓΗΣΗ ΔΑΝΕΙΩΝ ΑΠΟ Τ.Π. ΚΑΙ ΔΑΝΕΙΩΝ",
        "ΤΕΛΟΣ ΕΠΙΤΗΔΕΥΜΑΤΟΣ",
        "ΕΛΕΥΘΕΡΑ ΤΕΛΩΝΕΙΑΚΑ ΣΥΓΚΡΟΤΗΜΑΤΑ",
        "ΦΟΡΟΛΟΓΙΚΑ ΚΙΝΗΤΡΑ ΣΥΓΧΩΝΕΥΣΕΩΣ Η ΜΕΤΑΤΡΟΠΗΣ ΕΠΙΧΕΙΡΗΣΕΩΝ",
        "ΚΑΤΑΣΤΑΤΙΚΕΣ ΔΙΑΤΑΞΕΙΣ T.E.B.E",
        "ΝΑΥΤΙΚΟ ΕΠΙΜΕΛΗΤΗΡΙΟ",
        "ΠΡΟΣΩΠΙΚΟ Υ.Ε.Ν",
        "ΛΕΙΤΟΥΡΓΟΙ ΜΕΣΗΣ ΕΚΠΑΙΔΕΥΣΗΣ",
        "ΚΟΙΝΟΠΡΑΞΙΑ ΓΕΩΡΓΙΚΩΝ ΣΥΝΕΤΑΙΡΙΣΜΩΝ",
        "ΤΑΜΕΙΟ ΑΣΦΑΛΙΣΕΩΣ ΕΠΙΧΕΙΡΗΜΑΤΙΩΝ ΚΙΝΗΜΑΤΟΓΡΑΦΟΥ",
        "ΒΟΣΚΟΤΟΠΟΙ",
        "ΕΠΙΤΟΚΙΑ ΤΡΑΠΕΖΩΝ",
        "ΚΑΠΝΙΚΟΙ ΟΡΓΑΝΙΣΜΟΙ",
        "ΣΤΑΘΜΟΙ ΑΥΤΟΚΙΝΗΤΩΝ",
        "ΕΥΛΟΓΙΑ",
        "ΠΕΡΙΦΕΡΕΙΑΚΕΣ ΥΠΗΡΕΣΙΕΣ ΥΠΟΥΡΓΕΙΟΥ ΒΙΟΜΗΧΑΝΙΑΣ",
        "ΤΑΜΕΙΟ ΑΕΡΟΠΟΡΙΚΗΣ ΑΜΥΝΑΣ",
        "ΟΡΓΑΝΙΣΜΟΣ ΚΕΝΤΡΙΚΗΣ ΥΠΗΡΕΣΙΑΣ",
        "ΤΑΜΕΙΟ ΕΡΓΑΣΙΑΣ ΗΘΟΠΟΙΩΝ",
        "ΤΕΛΩΝΙΣΜΟΣ ΕΙΔΩΝ ΑΤΟΜΙΚΗΣ ΧΡΗΣΕΩΣ",
        "ΦΟΡΟΛΟΓΙΑ ΠΡΟΣΟΔΟΥ ΑΠΟ ΠΛΟΙΑ",
        "ΔΙΟΙΚΗΤΙΚΗ ΔΙΑΙΡΕΣΗΣ",
        "ΟΡΓΑΝΙΣΜΟΣ ΑΥΤΟΚΙΝΗΤΟΔΡΟΜΙΩΝ ΕΛΛΑΔΟΣ (Ο.Α.Ε.)",
        "ΕΘΝΙΚΟ ΚΕΝΤΡΟ ΑΜΕΣΗΣ ΒΟΗΘΕΙΑΣ (Ε.Κ.Α.Β.)",
        "ΓΝΩΜΟΔΟΤΙΚΟ ΣΥΜΒΟΥΛΙΟ ΟΙΚΟΝΟΜΙΚΗΣ ΑΝΑΠΤΥΞΗΣ",
        "ΔΙΑΘΗΚΗ",
        "ΑΓΩΓΕΣ ΔΙΑΤΡΟΦΗΣ",
        "ΦΑΡΜΑΚΕΥΤΙΚΟΙ ΣΥΛΛΟΓΟΙ",
        "ΤΑΜΕΙΟ ΣΥΝΤΑΞΕΩΝ ΚΑΙ  ΕΠΙΚΟΥΡΙΚΗΣ ΑΣΦΑΛΙΣΕΩΣ ΠΡΟΣΩΠΙΚΟΥ ΓΕΩΡΓΙΚΩΝ ΣΥΝΕΤΑΙΡΙΣΤΙΚΩΝ ΟΡΓΑΝΩΣΕΩΝ (Τ.Σ.Ε.Α.Π.Γ.Σ.Ο)",
        "ΕΠΙΔΟΜΑΤΑ ΔΙΑΦΟΡΑ",
        "ΠΕΙΘΑΡΧΙΚΟ ΔΙΚΑΙΟ",
        "ΤΑΜΕΙΟ ΕΠΙΚΟΥΡΙΚΗΣ ΑΣΦΑΛΙΣΕΩΣ ΧΗΜΙΚΩΝ (Τ.Ε.Α.Χ)",
        "ΠΡΟΑΓΩΓΕΣ ΚΑΙ ΠΡΟΣΟΝΤΑ ΠΥΡΟΣΒΕΣΤΙΚΟΥ ΠΡΟΣΩΠΙΚΟΥ",
        "ΟΔΟΙΠΟΡΙΚΑ ΕΞΟΔΑ ΠΡΟΣΩΠΙΚΟΥ ΣΩΜΑΤΩΝ ΑΣΦΑΛΕΙΑΣ",
        "ΝΟΣΗΛΕΥΤΙΚΑ ΙΔΡΥΜΑΤΑ ΚΑΤ’ ΙΔΙΑΝ",
        "ΠΡΟΣΤΑΣΙΑ ΚΑΤΑ ΤΗΣ ΦΥΛΛΟΞΗΡΑΣ",
        "ΟΡΓΑΝΙΣΜΟΣ  ΤΑΜΕΙΟΥ  ΝΟΜΙΚΩΝ",
        "ΠΡΑΤΗΡΙΑ ΥΓΡΩΝ ΚΑΥΣΙΜΩΝ",
        "ΘΡΗΣΚΕΥΤΙΚΟ ΣΩΜΑ ΕΝΟΠΛΩΝ ΔΥΝΑΜΕΩΝ",
        "ΔΙΑΔΙΚΑΣΙΑ ΑΝΑΓΚΑΣΤΙΚΩΝ ΑΠΑΛΛΟΤΡΙΩΣΕΩΝ ΑΚΙΝΗΤΩΝ",
        "ΔΙΕΡΜΗΝΕΙΣ",
        "ΣΧΕΔΙΑ ΑΛΛΩΝ ΠΟΛΕΩΝ",
        "ΤΑΜΕΙΟ ΑΛΛΗΛΟΒΟΗΘΕΙΑΣ ΣΤΡΑΤΙΩΤΙΚΩΝ ΑΕΡΟΠΟΡΙΑΣ",
        "ΗΜΕΡΟΛΟΓΙΟ ΜΗΧΑΝΗΣ",
        "ΚΕΝΤΡΟ ΕΛΛΗΝΙΚΗΣ ΓΛΩΣΣΑΣ",
        "ΩΡΕΣ ΕΡΓΑΣΙΑΣ ΣΕ ΑΡΤΟΠΟΙΕΙΑ",
        "ΓΕΝΙΚΗ ΓΡΑΜΜΑΤΕΙΑ",
        "ΜΕΤΑΦΡΑΣΤΙΚΑ ΓΡΑΦΕΙΑ",
        "ΠΡΟΔΙΑΓΡΑΦΕΣ ΜΕΛΕΤΩΝ",
        "ΣΥΝΤΑΞΕΙΣ ΘΥΜΑΤΩΝ ΕΘΝΙΚΗΣ",
        "ΤΑΜΕΙΟ ΠΡΟΝΟΙΑΣ ΣΥΜΒΟΛΑΙΟΓΡΑΦΩΝ",
        "ΙΑΤΡΟΔΙΚΑΣΤΙΚΗ ΑΜΟΙΒΗ",
        "ΕΦΟΡΙΕΣ ΚΑΠΝΟΥ – ΚΑΠΝΕΡΓΟΣΤΑΣΙΑ",
        "ΠΟΙΜΝΙΟΣΤΑΣΙΑ",
        "ΚΕΝΤΡΑ ΕΡΕΥΝΑΣ - ΕΡΕΥΝΗΤΙΚΑ ΙΝΣΤΙΤΟΥΤΑ",
        "ΤΑΜΕΙΑ ΠΡΟΝΟΙΑΣ ΔΙΚΗΓΟΡΩΝ",
        "ΟΙΝΟΠΑΡΑΓΩΓΗ ΣΑΜΟΥ",
        "ΙΜΑΤΙΣΜΟΣ Π. ΝΑΥΤΙΚΟΥ",
        "ΜΗΧΑΝΙΚΟΙ,ΑΡΧΙΤΕΚΤΟΝΕΣ,ΤΟΠΟΓΡΑΦΟΙ",
        "ΠΑΝΤΕΙΟ ΠΑΝΕΠΙΣΤΗΜΙΟ ΚΟΙΝΩΝΙΚΩΝ ΚΑΙ ΠΟΛΙΤΙΚΩΝ ΕΠΙΣΤΗΜΩΝ",
        "ΝΕΟΙ ΧΡΗΜΑΤΟΠΙΣΤΩΤΙΚΟΙ ΘΕΣΜΟΙ",
        "ΥΠΗΡΕΣΙΑ ΠΟΛΙΤΙΚΗΣ ΑΕΡΟΠΟΡΙΑΣ",
        "ΟΡΓΑΝΙΣΜΟΣ ΥΠΟΘΗΚΟΦΥΛΑΚΕΙΩΝ",
        "ΑΤΥΧΗΜΑΤΑ ΣΕ ΔΗΜΟΣΙΑ ΕΡΓΑ",
        "ΑΡΕΙΟΣ ΠΑΓΟΣ",
        "ΥΠΑΓΩΓΗ ΣΕ ΑΣΦΑΛΙΣΗ ΚΑΙ",
        "ΔΙΕΘΝΕΙΣ ΣΙΔΗΡΟΔΡΟΜΙΚΕΣ ΜΕΤΑΦΟΡΕΣΔΙΕΥΡΩΠΑΙΚΟ ΣΙΔΗΡΟΔΡΟΜΙΚΟ ΣΥΣΤΗΜΑ",
        "ΟΙΚΟΝΟΜΙΚΗ ΕΠΙΘΕΩΡΗΣΗ Π. ΝΑΥΤΙΚΟΥ",
        "ΑΝΑΠΤΥΞΙΑΚΗ ΚΑΙ ΒΙΟΜΗΧΑΝΙΚΗ ΠΟΛΙΤΙΚΗ",
        "ΒΕΒΑΙΩΣΗ ΚΑΙ ΕΙΣΠΡΑΞΗ ΠΟΙΝΙΚΩΝ ΕΞΟΔΩΝ",
        "ΝΑΥΤΙΚΟ ΧΗΜΕΙΟ",
        "ΛΑΧΕΙΑ",
        "ΤΡΟΧΙΟΔΡΟΜΟΙ ΑΘΗΝΩΝ – ΠΕΙΡΑΙΩΣ",
        "ΤΑΜΕΙΟ  ΠΡΟΝΟΙΑΣ ΠΡΟΣΩΠΙΚΟΥ ΕΤΑΙΡΕΙΩΝ ΛΙΠΑΣΜΑΤΩΝ ΤΑ.Π.Π.Ε.Λ",
        "ΔΙΕΥΚΟΛΥΝΣΕΙΣ ΓΙΑ ΑΝΟΙΚΟΔΟΜΗΣΗ",
        "ΑΓΟΡΑΠΩΛΗΣΙΑ ΚΑΠΝΟΥ",
        "ΠΕΡΙ ΟΡΩΝ ΕΡΓΑΣΙΑΣ ΠΡΟΣΩΠΙΚΟΥ ΔΙΕΘΝΩΝ ΜΕΤΑΦΟΡΩΝ",
        "ΑΛΙΕΥΤΙΚΟΣ ΚΩΔΙΚΑΣ",
        "ΣΥΜΒΟΥΛΙΑ ΚΑΙ ΕΠΙΤΡΟΠΕΣ",
        "ΠΕΡΙΦΕΡΕΙΑΚΕΣ ΥΠΗΡΕΣΙΕΣ ΥΠΟΥΡΓΕΙΟΥ ΟΙΚΟΝΟΜΙΚΩΝ",
        "ΣΥΜΒΑΣΕΙΣ ΠΕΡΙ ΑΣΕΜΝΩΝ ΔΗΜΟΣΙΕΥΜΑΤΩΝ",
        "ΓΕΩΡΓΙΚΟΙ ΣΤΑΘΜΟΙ",
        "ΝΑΞΙΩΤΙΚΗ ΣΜΥΡΙΔΑ",
        "ΑΝΑΣΤΟΛΗ ΠΡΟΣΕΛΕΥΣΕΩΣ ΕΦΕΔΡΩΝ",
        "ΕΚΠΑΙΔΕΥΣΗ ΧΩΡΟΦΥΛΑΚΗΣ",
        "ΑΣΦΑΛΙΣΗ ΕΞΑΓΩΓΙΚΩΝ ΠΙΣΤΩΣΕΩΝ",
        "ΘΕΡΑΠΑΙΝΙΔΕΣ ΑΝΑΠΗΡΩΝ ΠΟΛΕΜΟΥ",
        "ΕΠΙΤΡΟΠΗ ΑΤΟΜΙΚΗΣ ΕΝΕΡΓΕΙΑΣ",
        "ΚΑΝΟΝΙΣΜΟΣ ΑΣΤΥΝΟΜΙΑΣ ΠΟΛΕΩΝ",
        "ΦΥΛΛΑ ΠΟΙΟΤΗΤΑΣ ΥΠΑΞΙΩΜΑΤΙΚΩΝ Π.Ν",
        "ΕΠΙΘΕΩΡΗΣΕΙΣ ΚΤΗΝΙΑΤΡΙΚΗΣ",
        "ΜΕΡΙΚΗ ΑΠΑΣΧΟΛΗΣΗ - ΦΑΣΟΝ - ΤΗΛΕΡΓΑΣΙΑ ΚΑΤ’ ΟΙΚΟΝ ΑΠΑΣΧΟΛΗΣΗ",
        "ΗΛΕΚΤΡΙΚΗ ΕΤΑΙΡΕΙΑ ΑΘΗΝΩΝ - ΠΕΙΡΑΙΩΣ",
        "ΠΡΟΚΑΤΑΣΚΕΥΑΣΜΕΝΑΙ ΟΙΚΙΑΙ",
        "ΤΡΑΠΕΖΑ ΤΗΣ ΕΛΛΑΔΟΣ",
        "ΣΥΜΦΩΝΙΕΣ ΠΡΟΣΤΑΣΙΑΣ ΤΟΥ ΠΕΡΙΒΑΛΛΟΝΤΟΣ",
        "ΛΙΓΝΙΤΗΣ",
        "ΤΑΜΕΙΟ ΕΠΑΓΓΕΛΜΑΤΙΚΗΣ ΑΣΦΑΛΙΣΗΣ ΠΡΟΣΩΠΙΚΟΥ ΕΛΤΑ",
        "ΜΕΛΕΤΕΣ ΤΕΧΝΙΚΩΝ ΕΡΓΩΝ",
        "ΠΛΗΡΩΜΑΤΑ ΑΕΡΟΣΚΑΦΩΝ",
        "ΕΞΑΓΩΓΗ ΣΤΑΦΙΔΑΣ",
        "ΤΑΜΕΙΟΝ ΠΡΟΝΟΙΑΣ ΔΗΜΟΣΙΩΝ ΥΠΑΛΛΗΛΩΝ",
        "ΔΙΑΧΕΙΡΙΣΗ ΠΕΡΙΟΥΣΙΑΣ",
        "ΟΡΓΑΝΙΚΟΙ ΝΟΜΟΙ",
        "ΥΠΗΡΕΣΙΕΣ ΑΙΜΟΔΟΣΙΑΣ",
        "ΣΩΜΑΤΕΙΑ ΔΗΜΟΣΙΩΝ ΥΠΑΛΛΗΛΩΝ",
        "ΠΕΖΟΔΡΟΜΙΑ",
        "ΔΙΑΘΕΣΗ ΑΠΟΡΡΙΜΜΑΤΩΝ",
        "ΤΡΟΧΙΟΔΡΟΜΟΙ ΘΕΣΣΑΛΟΝΙΚΗΣ",
        "ΓΕΝΙΚΗ ΔΙΕΥΘΥΝΣΗ ΔΗΜΟΣΙΟΥ ΛΟΓΙΣΤΙΚΟΥ",
        "ΡΥΜΟΥΛΚΑ - ΛΑΝΤΖΕΣ",
        "ΠΕΤΡΕΛΑΙΟΕΙΔΗ",
        "ΓΕΝΙΚΑ ΑΡΧΕΙΑ ΤΟΥ ΚΡΑΤΟΥΣ",
        "ΚΑΤΑΣΤΑΤΙΚΕΣ ΔΙΑΤΑΞΕΙΣ Ο.Τ.Ε. - ΣΧΕΣΕΙΣ Ο.Τ.Ε. ΜΕ ΑΛΛΟΥΣ ΠΑΡΟΧΟΥΣ",
        "ΥΠΗΡΕΣΙΑ ΑΥΤΟΚΙΝΗΤΩΝ",
        "ΑΚΑΔΗΜΙΑ ΑΘΗΝΩΝ",
        "ΜΟΝΟΠΩΛΙΟ ΖΑΧΑΡΙΝΗΣ",
        "ΟΙΚΙΣΤΙΚΕΣ ΠΕΡΙΟΧΕΣ",
        "ΑΤΕΛΕΙΕΣ ΥΠΕΡ ΤΗΣ ΑΛΙΕΙΑΣ",
        "ΔΙΑΦΟΡΕΣ ΕΚΤΑΚΤΕΣ ΦΟΡΟΛΟΓΙΕΣ",
        "ΒΙΒΛΙΑ ΔΗΜΩΝ ΚΑΙ ΚΟΙΝΟΤΗΤΩΝ",
        "ΕΡΓΑΤΙΚΑ ΑΤΥΧΗΜΑΤΑ",
        "ΝΟΣΗΛΕΥΤΕΣ",
        "ΣΥΝΔΙΚΑΛΙΣΤΙΚΕΣ ΕΛΕΥΘΕΡΙΕΣ",
        "ΕΘΝΙΚΟ ΣΥΜΒΟΥΛΙΟ ΕΝΕΡΓΕΙΑΣ",
        "ΤΑΜΕΙΟ ΕΠΙΚΟΥΡΙΚΗΣ ΑΣΦΑΛΙΣΕΩΣ ΕΡΓΑΤΟΤΕΧΝΙΤΩΝ ΥΑΛΟΥΡΓΩΝ",
        "ΑΓΩΓΕΣ ΑΣΦΑΛΙΣΤΡΩΝ",
        "ΣΩΜΑΤΕΜΠΟΡΙΑ ΓΥΝΑΙΚΩΝ",
        "ΑΤΕΛΕΙΕΣ ΕΡΓΩΝ ΑΜΥΝΤΙΚΟΥ ΠΡΟΓΡΑΜΜΑΤΟΣ",
        "ΤΕΧΝΙΚΗ ΕΚΠΑΙΔΕΥΣΗ ΑΞΙΩΜΑΤΙΚΩΝ ΣΕ ΑΝΩΤΑΤΕΣ ΣΧΟΛΕΣ",
        "ΔΙΚΑΙΩΜΑΤΑ ΚΗΡΥΚΩΝ ΚΛΠ",
        "ΓΕΝΙΚΕΣ ΔΙΑΤΑΞΕΙΣ ΤΑΜΕΙΟΥ ΝΟΜΙΚΩΝ",
        "ΝΑΥΤΕΣ ΚΑΙ ΛΙΜΕΝΟΦΥΛΑΚΕΣ",
        "ΠΑΝΕΠΙΣΤΗΜΙΑΚΗ ΣΧΟΛΗ ΑΓΡΙΝΙΟΥ",
        "ΠΟΛΥΤΕΧΝΙΚΗ ΣΧΟΛΗ",
        "ΜΕΙΩΣΗ ΕΙΣΦΟΡΩΝ",
        "ΚΕΝΤΡΑ ΛΗΨΕΩΣ ΤΙΜΩΝ ΣΦΑΓΕΙΩΝ",
        "ΑΠΟΔΗΜΙΑ ΣΤΡΑΤΕΥΣΙΜΩΝ",
        "ΤΑΜΕΙΟ ΕΠΙΚΟΥΡΙΚΗΣ ΑΣΦΑΛΙΣΕΩΣ ΠΡΟΝΟΙΑΣ ΚΑΙ ΚΟΙΝΗΣ ΔΙΑΝΟΜΗΣ ΠΩΛΗΤΩΝ ΒΕΝΖΙΝΗΣ ΑΘΗΝΩΝ - ΠΕΙΡΑΙΩΣ ΚΑΙ ΠΕΡΙΧΩΡΩΝ",
        "ΙΑΤΡΟΦΑΡΜΑΚΕΥΤΙΚΗ ΠΕΡΙΘΑΛΨΗ",
        "ΝΟΣΗΛΕΥΤΙΚΑ ΙΔΡΥΜΑΤΑ",
        "ΓΕΝΙΚΑ ΠΕΡΙ ΜΟΥΣΕΙΩΝ",
        "ΑΣΦΑΛΕΙΑ ΟΧΥΡΩΝ ΘΕΣΕΩΝ",
        "ΓΕΩΡΓΙΚΑ ΜΗΧΑΝΗΜΑΤΑ",
        "ΤΑΜΕΙΑ ΣΥΝΕΡΓΑΣΙΑΣ",
        "ΙΔΙΩΤΙΚΕΣ ΚΛΙΝΙΚΕΣ ΚΑΙ ΕΡΓΑΣΤΗΡΙΑ",
        "ΥΓΕΙΟΝΟΜΙΚΗ ΕΞΕΤΑΣΗ ΙΠΤΑΜΕΝΩΝ",
        "ΔΙΑΦΟΡΕΣ ΑΕΡΟΠΟΡΙΚΕΣ ΣΧΟΛΕΣ",
        "ΓΥΝΑΙΚΕΣ ΝΟΣΟΚΟΜΟΙ",
        "ΦΟΙΤΗΣΗ, ΒΑΘΜΟΛΟΓΙΑ, ΕΞΕΤΑΣΕΙΣ ΚΛΠ. Α.Σ.Κ.Τ",
        "ΣΕΙΣΜΟΠΛΗΚΤΟΙ ΔΙΑΦΟΡΟΙ",
        "ΟΡΓΑΝΙΣΜΟΣ ΥΠΟΥΡΓΕΙΟΥ ΓΕΩΡΓΙΑΣ",
        "ΚΩΔΙΚΟΠΟΙΗΣΗ ΤΗΣ ΝΟΜΟΘΕΣΙΑΣ",
        "ΜΕΤΑ ΤΗΣ ΓΑΛΛΙΑΣ",
        "ΓΕΩΓΡΑΦΙΚΗ ΥΠΗΡΕΣΙΑ ΣΤΡΑΤΟΥ",
        "ΕΙΔΗ ΠΑΡΑΔΙΔΟΜΕΝΑ ΣΤΗΝ ΕΛΕΥΘΕΡΗ ΧΡΗΣΗ",
        "ΜΟΝΟΠΩΛΙΟ ΣΠΙΡΤΩΝ",
        "ΚΑΤΑΣΤΑΤΙΚΟΝ Τ.Α.Κ.Ε",
        "ΕΠΙΚΟΥΡΙΚΟ ΤΑΜΕΙΟ ΥΠΑΛΛΗΛΩΝ ΑΣΤΥΝΟΜΙΑΣ ΠΟΛΕΩΝ (Ε.Τ.Υ.Α.Π.)",
        "ΜΙΣΘΟΔΟΣΙΑ ΙΕΡΕΩΝ – ΕΝΟΡΙΑΚΗ ΕΙΣΦΟΡΑ",
        "ΥΓΕΙΟΝΟΜΙΚΟΣ ΚΑΝΟΝΙΣΜΟΣ",
        "ΝΟΜΟΣ ΠΕΡΙ ΚΤΗΜΑΤΙΚΩΝ ΤΡΑΠΕΖΩΝ",
        "ΔΙΕΘΝΗΣ ΣΥΜΒΑΣΗ ΠΕΡΙ ΥΔΡΑΥΛΙΚΩΝ ΔΥΝΑΜΕΩΝ",
        "ΑΝΑΠΗΡΟΙ ΑΞΙΩΜΑΤΙΚΟΙ ΚΑΙ ΟΠΛΙΤΕΣ ΕΙΡΗΝΙΚΗΣ ΠΕΡΙΟΔΟΥ",
        "ΠΟΙΝΙΚΗ ΚΑΙ ΠΕΙΘΑΡΧΙΚΗ ΔΩΣΙΔΙΚΙΑ Λ.Σ",
        "ΔΑΣΙΚΟ ΠΡΟΣΩΠΙΚΟ",
        "ΑΟΠΛΗ ΘΗΤΕΙΑ-ΑΝΤΙΡΡΗΣΙΕΣ ΣΥΝΕΙΔΗΣΗΣ",
        "ΝΕΟΙ ΠΡΟΣΦΥΓΕΣ",
        "ΤΕΧΝΙΚΕΣ ΥΠΗΡΕΣΙΕΣ ΣΤΡΑΤΟΥ",
        "ΜΕΤΟΧΙΚΟ ΤΑΜΕΙΟ ΠΟΛΙΤΙΚΩΝ ΥΠΑΛΛΗΛΩΝ",
        "ΠΡΟΣΩΠΙΚΟ ΙΔΙΩΤΙΚΟΥ ΔΙΚΑΙΟΥ",
        "ΚΩΔΙΚΑΣ ΑΓΡΟΤΙΚΗΣ ΑΣΦΑΛΕΙΑΣ",
        "ΟΡΓΑΝΙΚΕΣ ΔΙΑΤΑΞΕΙΣ ΑΠΟΣΤΟΛΙΚΗΣ ΔΙΑΚΟΝΙΑΣ",
        "ΥΠΟΥΡΓΕΙΟ ΑΙΓΑΙΟΥ",
        "ΓΑΜΟΙ  ΔΩΔΕΚΑΝΗΣΟΥ",
        "ΩΡΕΣ ΕΡΓΑΣΙΑΣ ΚΡΕΟΠΩΛΕΙΩΝ",
        "ΚΩΔΙΚΑΣ ΤΕΛΩΝ ΧΑΡΤΟΣΗΜΟΥ",
        "ΔΕΛΤΙΟ ΑΝΩΝΥΜΩΝ ΕΤΑΙΡΕΙΩΝ",
        "ΑΡΜΟΔΙΟΤΗΤΑ ΝΟΜΑΡΧΗ ΣΕ ΕΡΓΑΤΙΚΑ ΖΗΤΗΜΑΤΑ",
        "ΤΡΟΦΟΔΟΣΙΑ Π. ΝΑΥΤΙΚΟΥ",
        "ΣΥΜΦΩΝΙΑ ΠΕΡΙ ΔΙΠΛΩΜΑΤΙΚΩΝ ΣΧΕΣΕΩΝ",
        "ΕΦΕΔΡΟΙ ΚΑΙ ΕΠΙΚΟΥΡΟΙ ΑΞΙΩΜΑΤΙΚΟΙ Π.Ν",
        "ΤΟΥΡΙΣΤΙΚΕΣ ΕΠΙΧΕΙΡΗΣΕΙΣ",
        "ΔΙΕΘΝΕΣ ΠΟΙΝΙΚΟ ΔΙΚΑΣΤΗΡΙΟ",
        "ΔΙΟΙΚΗΤΙΚΕΣ ΠΡΑΞΕΙΣ",
        "ΝΟΣΟΚΟΜΕΙΑ ΠΟΛΕΜΙΚΟΥ ΝΑΥΤΙΚΟΥ",
        "ΣΥΜΒΟΥΛΙΟ ΧΑΛΥΒΑ",
        "ΤΕΜΑΧΙΣΜΟΣ ΚΡΕΑΤΩΝ",
        "ΕΛΕΓΧΟΣ ΚΑΤΟΧΗΣ ΟΠΛΩΝ",
        "ΑΝΑΠΡΟΣΑΡΜΟΓΕΣ ΤΗΣ ΔΡΑΧΜΗΣ",
        "ΕΦΟΔΙΑΣΜΟΣ ΠΛΟΙΩΝ",
        "ΣΕΙΣΜΟΠΛΗΚΤΟΙ ΙΟΝΙΩΝ ΝΗΣΩΝ",
        "ΔΗΜΟΣΙΑ ΕΠΙΧΕΙΡΗΣΗ ΚΙΝΗΤΩΝ ΑΞΙΩΝ ΑΝΩΝΥΜΗ ΕΤΑΙΡΕΙΑ (Δ.Ε.Κ.Α. Α.Ε.)",
        "ΕΤΑΙΡΕΙΑ – ΕΥΡΩΠΑΙΚΟΣ ΟΜΙΛΟΣ",
        "ΔΙΕΥΘΥΝΣΗ ΑΛΙΕΙΑΣ",
        "ΕΠΙΜΕΛΗΤΗΡΙΟ ΤΟΥΡΙΣΤΙΚΩΝ ΚΑΤΑΣΤΗΜΑΤΩΝ",
        "ΔΙΕΘΝΕΙΣ ΣΥΜΒΑΣΕΙΣ ΕΛΑΙΟΛΑΔΟΥ",
        "ΠΤΗΤΙΚΗ ΙΚΑΝΟΤΗΤΑ",
        "ΕΚΚΛΗΣΙΑΣΤΙΚΕΣ ΣΧΟΛΕΣ",
        "ΔΙΑΤΙΜΗΣΗ ΙΑΤΡΙΚΩΝ ΠΡΑΞΕΩΝ",
        "ΑΔΙΚΗΜΑΤΑ ΤΥΠΟΥ",
        "ΕΞΑΝΘΗΜΑΤΙΚΟΣ ΤΥΦΟΣ",
        "ΟΙΚΟΣ ΝΑΥΤΟΥ",
        "ΜΑΣΤΙΧΑ",
        "ΣΥΛΛΟΓΟΙ ΚΑΙ ΟΜΟΣΠΟΝΔΙΑ ΔΙΚΑΣΤΙΚΩΝ ΕΠΙΜΕΛΗΤΩΝ",
        "ΕΜΠΟΡΙΚΑ ΚΑΙ ΒΙΟΜΗΧΑΝΙΚΑ ΣΗΜΑΤΑ",
        "ΟΡΓΑΝΩΣΗ ΚΑΙ ΛΕΙΤΟΥΡΓΙΑ ΑΝΩΤΑΤΩΝ ΕΚΠΑΙΔΕΥΤΙΚΩΝ ΙΔΡΥΜΑΤΩΝ",
        "ΥΓΕΙΟΝΟΜΙΚΗ ΑΠΟΘΗΚΗ",
        "ΓΕΝ. ΔΙΕΥΘΥΝΣΗ ΠΟΙΝΙΚΗΣ ΔΙΚΑΙΟΣΥΝΗΣ",
        "ΑΕΡΟΠΟΡΙΚΟ ΔΙΚΑΙΟ",
        "ΜΕΛΕΤΗ ΚΑΙ ΕΠΙΒΛΕΨΗ ΜΗΧΑΝΟΛΟΓΙΚΩΝ ΕΓΚΑΤΑΣΤΑΣΕΩΝ",
        "ΑΘΕΜΙΤΟΣ ΑΝΤΑΓΩΝΙΣΜΟΣ",
        "ΠΟΛΕΜΙΚΗ ΔΙΑΘΕΣΙΜΟΤΗΤΑ",
        "ΛΕΣΧΕΣ ΚΑΙ ΠΡΑΤΗΡΙΑ ΕΛ.ΑΣ",
        "ΚΑΥΣΙΜΑ",
        "ΥΓΕΙΟΝΟΜΙΚΑ ΜΕΤΡΑ",
        "ΚΑΤΑΣΤΑΣΗ ΑΞΙΩΜΑΤΙΚΩΝ",
        "ΕΙΣΠΡΑΞΗ ΠΟΡΩΝ ΤΑΜΕΙΟΥ ΝΟΜΙΚΩΝ",
        "ΔΙΟΙΚΗΤΙΚΗ ΡΥΘΜΙΣΗ ΑΠΟΔΟΧΩΝ ΚΑΙ ΟΡΩΝ ΕΡΓΑΣΙΑΣ",
        "ΓΕΝΙΚΗ ΔΙΕΥΘΥΝΣΗ ΤΑΧΥΔΡΟΜΕΙΩΝ",
        "ΟΡΓΑΝΙΣΜΟΣ ΛΙΜΕΝΟΣ ΘΕΣΣΑΛΟΝΙΚΗΣ ΑΝΩΝΥΜΗ ΕΤΑΙΡΙΑ (Ο.Λ.Θ. Α.Ε.)",
        "ΣΧΟΛΗ ΕΘΝΙΚΗΣ ΑΜΥΝΑΣ",
        "ΚΑΘΟΛΙΚΟΙ",
        "ΕΚΚΛΗΣΙΑΣΤΙΚΑ ΜΟΥΣΕΙΑ",
        "ΔΙΕΘΝΗΣ ΕΚΘΕΣΗ ΘΕΣΣΑΛΟΝΙΚΗΣ Α.Ε. – XELEXPO Α.Ε",
        "ΕΥΕΡΓΕΤΙΚΟΣ ΥΠΟΛΟΓΙΣΜΟΣ ΗΜΕΡΩΝ ΕΡΓΑΣΙΑΣ",
        "ΕΙΣΦΟΡΑ ΕΠΑΓΓΕΛΜΑΤΙΚΟΥ ΚΙΝΔΥΝΟΥ",
        "ΑΠΑΛΛΟΤΡΙΩΣΕΙΣ ΓΙΑ ΤΟΥΡΙΣΤΙΚΟΥΣ ΣΚΟΠΟΥΣ",
        "ΑΠΟΛΥΜΑΝΤΗΡΙΑ",
        "ΕΚΠΟΙΗΣΗ ΠΛΟΙΩΝ ΔΗΜΟΣΙΟΥ",
        "ΔΙΑΚΟΝΟΙ",
        "ΥΔΡΕΥΣΗ ΔΙΑΦΟΡΩΝ ΠΟΛΕΩΝ",
        "ΠΡΩΤΕΣ ΥΛΕΣ ΚΛΩΣΤΟΥΦΑΝΤΟΥΡΓΙΑΣ",
        "ΨΕΥΔΗΣ ΒΕΒΑΙΩΣΗ ΕΝΩΠΙΟΝ ΑΡΧΗΣ",
        "ΑΠΩΛΕΣΘΕΙΣΕΣ ΚΑΙ ΠΑΡΑΓΡΑΦΕΙΣΕΣ ΑΞΙΕΣ",
        "ΦΟΙΤΗΤΙΚΗ ΛΕΣΧΗ",
        "ΤΑΜΕΙΟ ΥΓΕΙΑΣ ΤΑΧΥΔΡΟΜΙΚΟΥ ΠΡΟΣΩΠΙΚΟΥ",
        "ΕΛΕΓΧΟΣ ΔΕΝΔΡΩΔΩΝ ΚΑΛΛΙΕΡΓΕΙΩΝ",
        "ΚΑΤΑΠΟΛΕΜΗΣΗ ΑΝΑΛΦΑΒΗΤΙΣΜΟΥΛΑΙΚΗ ΕΠΙΜΟΡΦΩΣΗ",
        "ΕΠΙΚΟΥΡΙΚΟΣ ΟΡΓΑΝΙΣΜΟΣ ΜΕΤΑΦΟΡΩΝ",
        "ΦΟΙΤΗΤΙΚΕΣ ΛΕΣΧΕΣ",
        "ΔΙΕΘΝΕΙΣ ΣΥΜΒΑΣΕΙΣ ΓΙΑ ΤΗΝ ΠΡΟΣΤΑΣΙΑ ΤΩΝ ΕΡΓΑΖΟΜΕΝΩΝ ΓΥΝΑΙΚΩΝ",
        "ΛΗΣΤΕΙΑ",
        "ΑΓΩΓΕΣ ΑΠΟ ΣΥΝΑΛΛΑΓΜΑΤΙΚΕΣ ΚΑΙ ΓΡΑΜΜΑΤΙΑ",
        "ΕΚΜΙΣΘΩΣΗ ΔΗΜΟΣΙΩΝ ΜΕΤΑΛΛΕΙΩΝ",
        "ΚΟΛΥΜΒΗΤΙΚΕΣ ΔΕΞΑΜΕΝΕΣ",
        "ΕΡΑΝΟΙ ΚΑΙ ΛΑΧΕΙΟΦΟΡΟΙ Η ΦΙΛΑΝΘΡΩΠΙΚΕΣ  ΑΓΟΡΕΣ",
        "ΠΡΟΣΤΑΣΙΑ ΕΠΙΒΑΤΗΓΟΥ ΝΑΥΤΙΛΙΑΣ",
        "ΓΕΝΙΚΟΙ ΝΟΜΟΙ ΠΕΡΙ ΞΕΝΟΔΟΧΕΙΩΝ-ΕΠΙΠΛ. ΔΩΜΑΤΙΩΝ ΚΛΠ",
        "ΙΕΡΑΡΧΙΑ ΚΑΙ ΠΡΟΑΓΩΓΕΣ ΑΞΙΩΜΑΤΙΚΩΝ",
        "ΣΥΝΕΡΓΑΤΕΣ (ΓΡΑΜΜΑΤΕΙΣ) ΒΟΥΛΕΥΤΩΝ-ΕΥΡΩΒΟΥΛΕΥΤΩΝ",
        "ΣΧΟΛΗ ΙΚΑΡΩΝ",
        "ΟΡΓΑΝΙΣΜΟΣ ΣΙΔΗΡΟΔΡΟΜΩΝ ΕΛΛΑΔΟΣ (Ο.Σ.Ε.)ΣΙΔΗΡΟΔΡΟΜΙΚΕΣ ΕΠΙΧΕΙΡΗΣΕΙΣ",
        "ΥΓΕΙΟΝΟΜΙΚΟΣ ΚΑΝΟΝΙΣΜΟΣ ΚΑΤΑ ΘΑΛΑΣΣΑΝ ΚΑΙ ΚΑΤΑ ΞΗΡΑΝ",
        "ΚΑΝΟΝΙΣΜΟΣ ΜΕΤΑΛΛΕΥΤΙΚΩΝ ΕΡΓΑΣΙΩΝ",
        "ΑΠΟΦΥΓΗ ΣΥΓΚΡΟΥΣΕΩΝ",
        "ΤΟΜΑΤΟΠΑΡΑΓΩΓΗ",
        "ΔΙΑΦΟΡΕΣ ΔΙΑΤΑΞΕΙΣ ΓΙΑ ΤΑ ΑΥΤΟΚΙΝΗΤΑ",
        "ΚΑΤΑΤΑΞΗ ΓΥΝΑΙΚΩΝ ΣΤΟ Λ.Σ",
        "ΕΤΑΙΡΕΙΕΣ ΔΙΟΙΚΟΥΜΕΝΕΣ ΑΠΟ ΤΟΥΣ ΠΙΣΤΩΤΕΣ",
        "ΒΑΛΚΑΝΙΚΕΣ ΣΥΜΦΩΝΙΕΣ",
        "ΜΕΤΑΦΟΡΑ ΣΥΝΤΕΛΕΣΤΗ ΔΟΜΗΣΗΣ",
        "ΠΡΟΜΗΘΕΥΤΙΚΟΣ ΟΡΓΑΝΙΣΜΟΣ Π.Ν",
        "ΠΡΟΣΩΠΙΚΟ ΦΑΡΜΑΚΕΙΩΝ",
        "ΔΙΔΑΣΚΟΜΕΝΑ ΜΑΘΗΜΑΤΑ",
        "ΕΚΛΟΓΗ ΒΟΥΛΕΥΤΩΝ - ΕΥΡΩΒΟΥΛΕΥΤΩΝ",
        "ΦΑΡΜΑΚΟΠΟΙΟΙ",
        "ΣΤΡΑΤΙΩΤΙΚΑ ΠΡΑΤΗΡΙΑ",
        "ΚΑΡΚΙΝΟΣ",
        "ΤΑΜΕΙΟ ΕΠΙΚΟΥΡΙΚΗΣ ΑΣΦΑΛΙΣΕΩΣ ΠΡΟΣΩΠΙΚΟΥ Α.Ε. ΟΙΝΟΠΟΙΙΑΣ, ΖΥΘΟΠΟΙΙΑΣ ΚΑΙ ΟΙΝΟΠΝΕΥΜΑΤΟΠΟΙΙΑΣ",
        "ΧΕΙΡΙΣΤΕΣ ΑΣΥΡΜΑΤΟΥ",
        "ΠΟΛΙΤΙΚΗ ΕΠΙΣΤΡΑΤΕΥΣΗ-ΠΑΛΛΑΙΚΗ ΑΜΥΝΑ",
        "ΟΡΓΑΝΙΣΜΟΙ ΕΓΓΕΙΩΝ ΒΕΛΤΙΩΣΕΩΝ",
        "ΟΜΟΓΕΝΕΙΣ ΠΑΛΛΙΝΟΣΤΟΥΝΤΕΣ",
        "ΕΥΡΩΠΑΙΚΟΣ ΚΟΙΝΩΝΙΚΟΣ ΧΑΡΤΗΣ",
        "ΟΡΓΑΝΙΚΕΣ ΔΙΑΤΑΞΕΙΣ",
        "ΕΞΑΙΡΕΣΗ ΔΙΚΑΣΤΩΝ",
        "ΓΕΝΙΚΕΣ ΕΠΙΘΕΩΡΗΣΕΙΣ – ΔΙΕΥΘΥΝΣΕΙΣ ΣΤΟΙΧΕΙΩΔΟΥΣ ΕΚΠΑΙΔΕΥΣΗΣ",
        "ΚΑΝΟΝΙΣΜΟΣ ΕΠΙΘΕΩΡΗΣΕΩΣ ΚΑΙ ΑΣΦΑΛΕΙΑΣ",
        "ΤΑΜΕΙΟ ΠΡΟΝΟΙΑΣ ΠΡΟΣΩΠΙΚΟΥ ΑΥΤΟΝΟΜΟΥ ΣΤΑΦΙΔΙΚΟΥ ΟΡΓΑΝΙΣΜΟΥ (Τ.Α.Π.Α.Σ.Ο)",
        "ΤΑΜΕΙΟΝ ΠΡΟΝΟΙΑΣ ΟΡΘΟΔΟΞΟΥ ΕΦΗΜΕΡΙΑΚΟΥ",
        "ΣΧΟΛΙΚΗ ΣΩΜΑΤΙΚΗ ΑΓΩΓΗ",
        "ΚΕΝΤΡΟ ΠΑΡΑΓΩΓΙΚΟΤΗΤΑΣ",
        "ΤΑΜΕΙΟ ΑΣΦΑΛΙΣΕΩΣ ΙΔΙΟΚΤΗΤΩΝ",
        "ΒΟΣΚΗ ΕΝΤΟΣ ΔΑΣΩΝ",
        "ΕΛΕΓΧΟΣ ΕΞΑΓΟΜΕΝΩΝ ΓΕΩΡΓΙΚΩΝ ΠΡΟΙΟΝΤΩΝ",
        "ΠΑΙΔΑΓΩΓΙΚΑ ΤΜΗΜΑΤΑ Α.Ε.Ι",
        "ΥΠΟΤΡΟΦΙΕΣ ΚΛΗΡΟΔΟΤΗΜΑΤΟΣ Π. ΒΑΣΣΑΝΗ",
        "ΑΤΥΧΗΜΑ ΑΠΟ ΔΟΛΟ ΤΟΥ ΕΡΓΟΔΟΤΗ",
        "ΒΥΖΑΝΤΙΝΟ ΚΑΙ ΧΡΙΣΤΙΑΝΙΚΟ ΜΟΥΣΕΙΟ",
        "ΕΙΡΗΝΕΥΤΙΚΕΣ ΑΠΟΣΤΟΛΕΣ",
        "ΥΓΕΙΟΝΟΜΙΚΟΣ ΄ΕΛΕΓΧΟΣ ΕΙΣΕΡΧΟΜΕΝΩΝ",
        "ΟΡΚΟΣ ΤΟΥ ΠΟΛΙΤΗ",
        "ΥΓΕΙΟΝΟΜΙΚΗ ΠΕΡΙΘΑΛΨΗ ΣΠΟΥΔΑΣΤΩΝ",
        "ΠΑΡΑΧΑΡΑΞΗ ΚΑΙ ΚΙΒΔΗΛΙΑ",
        "ΔΙΑΜΕΡΙΣΜΑΤΑ ΠΛΟΙΑΡΧΩΝ ΚΑΙ ΠΛΗΡΩΜΑΤΩΝ",
        "ΚΛΑΔΟΣ ΑΡΩΓΗΣ Τ.Α.Κ.Ε",
        "ΟΡΓΑΝΙΣΜΟΣ ΒΑΜΒΑΚΟΣ",
        "ΝΟΣΗΛΕΙΑ ΣΤΡΑΤΙΩΤΙΚΩΝ",
        "ΤΑΜΕΙΟ ΑΣΦΑΛΙΣΕΩΣ",
        "ΠΟΛΥΕΘΝΕΙΣ ΑΕΡΟΠΟΡΙΚΕΣ ΣΥΜΒΑΣΕΙΣ",
        "ΝΑΥΤΙΚΟ ΑΠΟΜΑΧΙΚΟ ΤΑΜΕΙΟ",
        "ΥΓΙΕΙΝΗ ΑΡΤΟΠΟΙΕΙΩΝ",
        "ΝΟΜΑΡΧΙΑΚΑ ΣΥΜΒΟΥΛΙΑ",
        "ΛΕΣΧΗ ΑΞΙΩΜΑΤΙΚΩΝ Π.Ν",
        "ΚΑΤΩΤΕΡΟ ΔΙΔΑΚΤΙΚΟ ΠΡΟΣΩΠΙΚΟ",
        "ΓΕΝΙΚΑ ΠΕΡΙ ΚΥΚΛΟΦΟΡΙΑΣ ΑΥΤΟΚΙΝΗΤΩΝ",
        "ΤΑΜΕΙΟ ΝΟΣΗΛΕΙΑΣ ΣΠΟΥΔΑΣΤΩΝ",
        "ΕΠΑΓΓΕΛΜΑΤΙΚΑ ΚΑΙ ΒΙΟΤΕΧΝΙΚΑ ΕΠΙΜΕΛΗΤΗΡΙΑ",
        "ΑΚΤΟΠΛΟΙΑ",
        "ΠΡΟΣΤΑΣΙΑ ΑΛΙΕΙΑΣ",
        "ΜΕ ΤΗ ΝΟΡΒΗΓΙΑ",
        "ΗΘΙΚΕΣ ΑΜΟΙΒΕΣ ΠΡΟΣΩΠΙΚΟΥ (΄ΕΝΟΠΛΟΥ-ΠΟΛΙΤΙΚΟΥ) ΥΠΟΥΡΓΕΙΟΥ ΔΗΜΟΣΙΑΣ ΤΑΞΗΣ",
        "ΛΕΩΦΟΡΕΙΑ ΙΔΙΩΤΙΚΗΣ ΧΡΗΣΕΩΣ",
        "ΕΡΓΑΤΙΚΕΣ ΔΙΑΦΟΡΕΣ",
        "ΡΑΔΙΟΗΛΕΚΤΡΟΛΟΓΟΙ-ΡΑΔΙΟΤΕΧΝΙΤΕΣ",
        "ΠΡΟΓΝΩΣΤΙΚΑ ΠΟΔΟΣΦΑΙΡΟΥ",
        "ΤΑΜΕΙΟ ΣΥΝΤΑΞΕΩΝ ΚΑΙ ΠΡΟΝΟΙΑΣ ΠΡΟΣΩΠΙΚΟΥ ΤΗΣ ΑΓΡΟΤΙΚΗΣ ΤΡΑΠΕΖΑΣ ΤΗΣ ΕΛΛΑΔΑΣ (Τ.Σ.Π. – Α.Τ.Ε.)",
        "ΥΔΡΕΥΣΗ ΛΕΚΑΝΟΠΕΔΙΟΥ ΑΘΗΝΩΝ",
        "ΤΡΑΠΕΖΑ ΟΦΘΑΛΜΩΝ",
        "ΕΘΝΙΚΟ ΚΕΝΤΡΟ ΧΑΡΤΩΝ ΚΑΙ ΧΑΡΤΟΓΡΑΦΙΚΗΣ ΚΛΗΡΟΝΟΜΙΑΣ - ΕΘΝΙΚΗ ΧΑΡΤΟΘΗΚΗ",
        "ΚΑΝΟΝΙΣΜΟΙ ΑΠΟΦΥΓΗΣ ΣΥΓΚΡΟΥΣΕΩΝ",
        "ΓΡΑΦΕΙΟ ΕΓΚΛΗΜΑΤΙΩΝ ΠΟΛΕΜΟΥ",
        "ΑΓΡΟΤΙΚΕΣ ΣΥΝΔΙΚΑΛΙΣΤΙΚΕΣ ΟΡΓΑΝΩΣΕΙΣ",
        "ΤΑΥΤΟΤΗΤΕΣ",
        "ΔΑΣΙΚΟΙ ΣΥΝΕΤΑΙΡΙΣΜΟΙ",
        "ΣΥΜΒΟΛΑΙΟΓΡΑΦΙΚΑ ΔΙΚΑΙΩΜΑΤΑ",
        "ΙΔΙΟΚΤΗΣΙΑ ΚΑΤ’ ΟΡΟΦΟ",
        "ΣΧΟΛΙΚΑ ΤΑΜΕΙΑ",
        "ΑΡΧΕΙΟΦΥΛΑΚΕΙΑ ΔΙΑΦΟΡΑ",
        "ΑΠΟΖΗΜΙΩΣΗ ΑΝΤΑΛΛΑΞΙΜΩΝ",
        "ΣΧΟΛΙΚΑ ΚΤΙΡΙΑ",
        "ΦΟΡΟΛΟΓΙΑ ΟΙΚΟΔΟΜΩΝ",
        "ΠΡΟΤΥΠΑ ΔΗΜΟΤΙΚΑ",
        "ΠΡΩΤΕΣ ΥΛΕΣ ΒΥΡΣΟΔΕΨΙΑΣ - ΔΕΡΜΑΤΑ",
        "ΣΥΜΒΙΒΑΣΜΟΣ ΚΑΙ ΔΙΑΙΤΗΣΙΑ",
        "ΚΑΤΑΣΤΑΣΗ ΔΗΜΟΤΙΚΩΝ ΚΑΙ ΚΟΙΝΟΤΙΚΩΝ ΥΠΑΛΛΗΛΩΝ",
        "ΕΣΟΔΑ ΔΗΜΩΝ ΚΑΙ ΚΟΙΝΟΤΗΤΩΝ",
        "ΣΤΑΔΙΑ ΚΑΙ ΓΥΜΝΑΣΤΗΡΙΑ",
        "ΚΟΙΝΗ ΑΓΡΟΤΙΚΗ ΠΟΛΙΤΙΚΗ",
        "ΑΤΟΜΑ ΜΕ ΕΙΔΙΚΕΣ ΑΝΑΓΚΕΣ - ΥΠΕΡΗΛΙΚΕΣ - ΧΡΟΝΙΑ ΠΑΣΧΟΝΤΕΣ",
        "ΕΚΚΛΗΣΙΑΣΤΙΚΑ ΔΙΚΑΣΤΗΡΙΑ",
        "ΣΥΜΒΑΣΕΙΣ ΓΙΑ ΤΗΝ ΑΠΟΦΥΓΗ ΔΙΠΛΗΣ ΦΟΡΟΛΟΓΙΑΣ",
        "ΠΡΟΣΤΑΣΙΑ ΒΑΜΒΑΚΟΠΑΡΑΓΩΓΗΣ",
        "ΝΑΥΤΙΚΗ ΣΤΡΑΤΟΛΟΓΙΑ",
        "ΝΟΣΟΚΟΜΕΙΑΚΗ ΠΕΡΙΘΑΛΨΗ ΑΣΦΑΛΙΣΜΕΝΩΝ Ο.Γ.Α",
        "ΦΥΣΙΚΑ ΟΡΓΑΝΙΚΑ ΛΙΠΑΣΜΑΤΑ",
        "ΤΑΜΕΙΟ ΕΠΙΚΟΥΡΙΚΗΣ ΑΣΦΑΛΙΣΕΩΣ ΜΙΣΘΩΤΩΝ ΕΣΤΙΑΤΟΡΙΩΝ, ΖΑΧΑΡΟΠΛΑΣΤΕΙΩΝ, ΚΑΦΕΝΕΙΩΝ Κ.ΛΠ. (Τ.Ε.Α.Μ.Ε.Ζ.)",
        "ΤΕΧΝΙΚΑΙ ΥΠΗΡΕΣΙΑΙ",
        "ΣΥΓΚΕΝΤΡΩΣΗ ΠΡΟΙΟΝΤΩΝ",
        "ΥΔΡΟΓΡΑΦΙΚΗ ΥΠΗΡΕΣΙΑ",
        "ΥΠΗΡΕΣΙΑ ΕΛΕΓΧΟΥ ΚΑΤΑΣΚΕΥΗΣ ΑΞΙΩΝ ΤΟΥ ΔΗΜΟΣΙΟΥ",
        "ΕΠΙΣΚΟΠΙΚΑ ΓΡΑΦΕΙΑ",
        "ΒΕΛΓΙΟ, ΒΕΝΕΖΟΥΕΛΑ Κ.ΛΠ",
        "ΔΗΜΟΤΙΚΟΣ ΚΑΙ ΚΟΙΝΟΤΙΚΟΣ ΚΩΔΙΚΑΣ",
        "ΠΡΟΔΟΣΙΑ",
        "ΜΙΣΘΟΣ ΔΗΜΟΣΙΩΝ ΥΠΑΛΛΗΛΩΝ",
        "ΠΟΛΙΤΙΚΟ ΠΡΟΣΩΠΙΚΟ ΝΑΥΤΙΚΟΥ",
        "ΑΝΑΖΗΤΗΣΗ ΚΑΙ ΔΙΑΦΥΛΑΞΗ ΑΡΧΑΙΟΤΗΤΩΝ",
        "ΑΔΕΙΕΣ ΛΙΑΝΙΚΗΣ ΠΩΛΗΣΗΣ ΤΣΙΓΑΡΩΝ ΚΑΙ ΕΙΔΩΝ ΜΟΝΟΠΩΛΙΟΥ",
        "ΕΠΟΠΤΙΚΑ ΜΕΣΑ ΔΙΔΑΣΚΑΛΙΑΣ",
        "ΕΚΛΟΓΟΔΙΚΕΙΑ",
        "Ο.Γ.Α ΚΑΤΑΣΤΑΤΙΚΕΣ ΔΙΑΤΑΞΕΙΣ",
        "ΙΝΣΤΙΤΟΥΤΟ ΥΓΕΙΑΣ ΤΟΥ ΠΑΙΔΙΟΥ",
        "ΣΧΟΛΗ ΘΕΤΙΚΩΝ ΕΠΙΣΤΗΜΩΝ ΠΑΝΕΠΙΣΤΗΜΙΟΥ ΠΑΤΡΩΝ",
        "ΕΣΠΕΡΙΔΟΕΙΔΗ-ΟΠΩΡΟΚΗΠΕΥΤΙΚΑ",
        "ΕΠΙΔΟΜΑΤΑ ΣΤΡΑΤΕΥΟΜΕΝΩΝ",
        "ΠΡΟΛΗΨΗ ΕΡΓΑΤΙΚΩΝ ΑΤΥΧΗΜΑΤΩΝ ΤΩΝ ΝΑΥΤΙΚΩΝ",
        "ΥΠΗΡΕΣΙΑ ΑΠΟΜΑΓΝΗΤΙΣΕΩΣ ΠΛΟΙΩΝ",
        "ΔΙΑΦΟΡΕΣ ΕΙΔΙΚΕΣ ΔΙΑΔΙΚΑΣΙΕΣ",
        "ΓΕΝΙΚΗ ΔΙΕΥΘΥΝΣΗ ΤΗΛΕΠΙΚΟΙΝΩΝΙΩΝ",
        "ΕΘΝΙΚΗ ΥΠΗΡΕΣΙΑ ΠΛΗΡΟΦΟΡΙΩΝ (Ε.Υ.Π.)",
        "ΤΑΜΕΙΟ ΕΠΙΚΟΥΡΙΚΗΣ ΑΣΦΑΛΙΣΕΩΣ ΜΙΣΘΩΤΩΝ (T.E.A.M)",
        "ΑΣΦΑΛΙΣΗ ΚΑΤΑ ΤΗΣ ΑΝΕΡΓΙΑΣ - ΟΡΓΑΝΙΣΜΟΣ ΑΠΑΣΧΟΛΗΣΗΣ ΕΡΓΑΤΙΚΟΥ ΔΥΝΑΜΙΚΟΥ",
        "ΣΩΜΑΤΙΚΗ ΙΚΑΝΟΤΗΤΑ ΠΡΟΣΩΠΙΚΟΥ ΣΤΡΑΤΕΥΜΑΤΟΣ",
        "ΟΙΚΟΝΟΜΙΚΗ ΥΠΗΡΕΣΙΑ Π. ΝΑΥΤΙΚΟΥ",
        "ΔΑΣΙΚΗ ΦΟΡΟΛΟΓΙΑ",
        "ΑΤΕΛΕΙΕΣ ΥΠΕΡ ΤΗΣ ΚΤΗΝΟΤΡΟΦΙΑΣ, ΜΕΛΙΣΣΟΚΟΜΙΑΣ Κ.Λ.Π",
        "ΠΟΛΙΤΙΚΑ ΔΙΚΑΙΩΜΑΤΑ ΤΩΝ ΓΥΝΑΙΚΩΝ",
        "ΜΕΤΑΘΕΣΕΙΣ ΕΚΠΑΙΔΕΥΤΙΚΩΝ",
        "ΔΙΕΘΝΕΣ ΚΕΝΤΡΟ ΥΠΟΛΟΓΙΣΜΟΥ",
        "ΔΙΑΧΕΙΡΙΣΗ ΔΑΣΩΝ",
        "ΔΟΥΛΕΙΑ",
        "ΜΕ ΤΗ ΠΟΛΩΝΙΑ",
        "ΑΝΑΔΙΑΝΟΜΗ ΚΤΗΜΑΤΩΝ",
        "ΥΠΟΑΠΑΣΧΟΛΟΥΜΕΝΟΙ ΜΙΣΘΩΤΟΙ",
        "ΟΡΓΑΝΙΣΜΟΙ ΠΡΩΗΝ Υ.Β.Ε.Τ. - Γ.Γ.Β. - Γ.Γ.Ε.Τ",
        "ΠΑΝΕΠΙΣΤΗΜΙΑΚΗ ΒΙΒΛΙΟΘΗΚΗ ΑΘΗΝΩΝ",
        "ΤΑΜΕΙΟ ΑΣΦΑΛΙΣΕΩΣ ΠΡΟΣΩΠΙΚΟΥ ΑΣΦΑΛΙΣΤ.ΕΤΑΙΡΕΙΑΣ Η ΕΘΝΙΚΗ (Τ.Α.Π.Α.Ε. Η ΕΘΝΙΚΗ)",
        "ΤΕΛΗ ΣΧΟΛΑΖΟΥΣΩΝ ΚΛΗΡΟΝΟΜΙΩΝ",
        "ΞΕΝΕΣ ΓΛΩΣΣΕΣ",
        "ΚΑΤΑΣΚΗΝΩΣΕΙΣ - ΠΑΙΔΙΚΕΣ ΕΞΟΧΕΣ",
        "ΔΙΚΑΣΤΗΡΙΑ ΑΝΗΛΙΚΩΝ",
        "ΣΥΜΒΑΣΕΙΣ ΕΚΤΕΛΕΣΕΩΣ ΑΛΛΟΔΑΠΩΝ ΑΠΟΦΑΣΕΩΝ",
        "ΦΟΡΟΣ ΕΙΣΟΔΗΜΑΤΟΣ ΝΟΜΙΚΩΝ ΠΡΟΣΩΠΩΝ",
        "ΘΕΩΡΗΤΙΚΑ ΚΑΙ ΙΣΤΟΡΙΚΑ ΜΑΘΗΜΑΤΑ",
        "ΑΦΡΟΔΙΣΙΑ",
        "ΦΑΡΟΙ",
        "ΔΗΜΟΣΙΟΓΡΑΦΙΚΟ ΕΠΑΓΓΕΛΜΑ",
        "ΚΑΤΑΣΤΑΤΙΚΟΣ ΝΟΜΟΣ ΕΚΚΛΗΣΙΑΣ ΤΗΣ ΕΛΛΑΔΟΣ",
        "ΕΛΕΓΧΟΣ ΣΚΟΠΙΜΟΤΗΤΑΣ ΙΔΡΥΣΕΩΣ ΒΙΟΜΗΧΑΝΙΩΝ",
        "ΓΥΜΝΑΣΙΑ ΚΑΙ ΛΥΚΕΙΑ",
        "ΑΕΡΟΝΑΥΤΙΚΕΣ ΠΛΗΡΟΦΟΡΙΕΣ",
        "ΚΑΤΑΣΤΑΣΗ ΥΠΑΞΙΩΜΑΤΙΚΩΝ Π.Ν",
        "ΥΠΟΥΡΓΕΙΟ ΧΩΡΟΤΑΞΙΑΣ",
        "ΕΚΤΕΛΕΣΗ ΄ΕΡΓΩΝ",
        "ΜΙΣΘΟΔΟΣΙΑ ΥΠΑΛΛΗΛΩΝ ΣΕ ΕΠΙΣΤΡΑΤΕΥΣΗ",
        "ΚΟΙΜΗΤΗΡΙΑ",
        "ΑΣΦΑΛΙΣΤΙΚΟΣ ΟΡΓΑΝΙΣΜΟΣ ΚΙΝΔΥΝΩΝ ΠΟΛΕΜΟΥ",
        "ΣΥΜΦΩΝΙΑ ΓΙΑ ΑΝΙΘΑΓΕΝΕΙΣ",
        "ΝΟΜΑΡΧΙΑΚΗ ΑΥΤΟΔΙΟΙΚΗΣΗ",
        "ΣΧΟΛΗ ΤΟΥΡΙΣΤΙΚΩΝ ΕΠΑΓΓΕΛΜΑΤΩΝ",
        "ΤΑΜΕΙΟ ΕΠΙΚΟΥΡΙΚΗΣ ΑΣΦΑΛΙΣΕΩΣ ΜΙΣΘΩΤΩΝ ΠΑΡΑΓΩΓΗΣ ΚΑΙ ΕΜΠΟΡΙΑΣ ΟΠΩΡΟΚΗΠΕΥΤΙΚΩΝ",
        "ΑΠΟΛΥΜΑΝΣΗ ΥΔΑΤΩΝ",
        "ΠΟΛΕΟΔΟΜΙΚΕΣ ΕΠΙΤΡΟΠΕΣ",
        "ΟΡΓΑΝΙΣΜΟΣ ΕΚΔΟΣΕΩΣ ΣΧΟΛΙΚΩΝ ΒΙΒΛΙΩΝ",
        "ΥΠΑΛΛΗΛΟΙ ΝΟΜ. ΠΡΟΣΩΠΩΝ ΔΗΜΟΣΙΟΥ ΔΙΚΑΙΟΥ",
        "ΑΝΤΙΣΤΑΘΜΙΣΤΙΚΗ ΕΙΣΦΟΡΑ",
        "ΠΡΟΣΩΠΙΚΟ ΙΔΙΩΤΙΚΩΝ ΕΚΠΑΙΔΕΥΤΗΡΙΩΝ",
        "ΔΙΕΘΝΕΙΣ ΣΥΜΒΑΣΕΙΣ ΓΙΑ ΤΑ ΑΥΤΟΚΙΝΗΤΑ",
        "ΕΞΩΣΧΟΛΙΚΗ ΑΓΩΓΗ",
        "ΑΣΦΑΛΙΣΤΙΚΗ ΑΡΜΟΔΙΟΤΗΤΑ",
        "ΕΛΙΕΣ ΚΑΙ ΕΛΑΙΑ",
        "ΓΑΜΟΙ ΙΣΡΑΗΛΙΤΩΝ",
        "ΤΑΜΕΙΟ ΑΡΤΟΥ",
        "ΚΑΝΟΝΙΣΜΟΣ ΕΠΙΤΡΟΠΩΝ",
        "ΣΥΜΒΑΣΗ ΚΑΤΑ ΔΑΓΚΕΙΟΥ",
        "ΕΘΝΙΚΟΙ ΔΡΥΜΟΙ",
        "ΑΠΑΛΛΑΓΕΣ ΤΕΛΩΝ ΧΑΡΤΟΣΗΜΟΥ",
        "ΔΙΕΘΝΗΣ ΟΡΓΑΝΙΣΜΟΣ ΑΝΑΠΤΥΞΕΩΣ",
        "ΚΑΝΟΝΙΣΜΟΣ ΕΡΓΑΣΙΑΣ ΕΠΙ ΦΟΡΤΗΓΩΝ ΠΛΟΙΩΝ",
        "ΛΥΣΣΑ",
        "ΑΓΡΟΚΤΗΜΑ",
        "ΚΑΘΗΓΗΤΕΣ ΚΑΙ ΥΦΗΓΗΤΕΣ",
        "ΠΑΙΔΙΚΟΙ - ΒΡΕΦΟΝΗΠΙΑΚΟΙ ΣΤΑΘΜΟΙ",
        "ΚΕΝΤΡΟ ΒΥΖΑΝΤΙΝΩΝ ΕΡΕΥΝΩΝ",
        "ΙΔΡΥΣΗ ΕΛΕΥΘΕΡΗΣ ΖΩΝΗΣ ΣΕ ΔΙΑΦΟΡΑ ΛΙΜΑΝΙΑ ΤΗΣ ΧΩΡΑΣ",
        "ΣΧΟΛΙΚΑ ΛΕΩΦΟΡΕΙΑ",
        "ΣΦΑΓΕΙΑ",
        "ΕΠΙΚΥΡΩΣΗ ΝΟΜΟΘΕΤΗΜΑΤΩΝ",
        "ΕΓΓΡΑΦΑ ΤΑΥΤΟΤΗΤΑΣ ΝΑΥΤΙΚΩΝ",
        "ΑΤΟΜΙΚΑ ΔΙΚΑΙΩΜΑΤΑ - ΔΕΔΟΜΕΝΑ ΠΡΟΣΩΠΙΚΟΥ ΧΑΡΑΚΤΗΡΑ",
        "ΙΑΤΡΟΦΑΡΜΑΚΕΥΤΙΚΗ - ΝΟΣΟΚΟΜΕΙΑΚΗ ΠΕΡΙΘΑΛΨΗ - ΕΞΟΔΑ ΚΗΔΕΙΑΣ",
        "ΥΠΗΡΕΣΙΑ ΔΙΑΧΕΙΡΙΣΕΩΣ ΑΝΤΑΛΛΑΞΙΜΩΝ ΚΤΗΜΑΤΩΝ",
        "ΣΤΟΛΕΣ ΠΡΟΣΩΠΙΚΟΥ Λ.Σ",
        "ΠΕΡΙΦΡΑΞΗ ΟΙΚΟΠΕΔΩΝ",
        "ΣΙΔΗΡΟΔΡΟΜΟΙ ΑΤΤΙΚΗΣ",
        "ΤΡΑΧΩΜΑΤΑ",
        "ΝΑΥΑΓΙΑ-ΝΑΥΑΓΙΑΙΡΕΣΗ",
        "ΥΠΟΜΗΧΑΝΙΚΟΙ",
        "ΤΑΙΝΙΟΘΗΚΗ ΤΗΣ ΕΛΛΑΔΟΣ",
        "ΚΑΝΟΝΙΣΜΟΣ ΤΗΛΕΓΡΑΦΙΚΗΣ ΥΠΗΡΕΣΙΑΣ",
        "ΣΥΝΤΑΞΕΙΣ ΘΥΜΑΤΩΝ ΤΡΟΜΟΚΡΑΤΙΑΣ",
        "ΚΑΝΟΝΙΣΜΟΣ ΠΥΡΙΜΑΧΟΥ ΠΡΟΣΤΑΣΙΑΣ ΕΠΙΒΑΤΗΓΩΝ ΠΛΟΙΩΝ",
        "ΑΤΟΜΙΚΑ ΒΙΒΛΙΑΡΙΑ",
        "ΕΠΑΓΓΕΛΜΑΤΙΚΑ ΒΙΒΛΙΑΡΙΑ ΑΡΤΕΡΓΑΤΩΝ ΚΛΠ",
        "ΦΟΡΟΛΟΓΙΑ ΑΜΥΛΟΣΙΡΟΠΙΟΥ, ΣΤΑΦΙΔΙΝΗΣ ΚΛΠ",
        "ΜΟΥΣΕΙΟ ΕΛΛΗΝΙΚΩΝ ΛΑΙΚΩΝ ΟΡΓΑΝΩΝ",
        "ΕΠΙΚΟΥΡΙΚΟ ΤΑΜΕΙΟ ΠΡΟΝΟΙΑΣ ΚΑΙ ΠΕΡΙΘΑΛΨΕΩΣ ΠΡΟΣΩΠΙΚΟΥ ΕΛΛΗΝ. ΗΛΕΚΤΡ. ΕΤΑΙΡΙΑΣ (Ε.Η.Ε.)",
        "ΤΑΜΕΙΑ ΜΟΝΙΜΩΝ ΟΔΟΣΤΡΩΜΑΤΩΝ",
        "ΟΡΓΑΝΙΚΕΣ ΘΕΣΕΙΣ ΑΞΙΩΜΑΤΙΚΩΝ Π.Ν",
        "ΤΑΜΕΙΟ ΣΥΝΤΑΞΕΩΝ ΠΡΟΣΩΠΙΚΟΥ ΤΡΑΠΕΖΑΣ ΑΘΗΝΩΝ",
        "ΠΟΛΙΟΜΥΕΛΙΤΙΔΑ",
        "ΠΡΟΑΓΩΓΑΙ ΑΞΙΩΜΑΤΙΚΩΝ ΧΩΡΟΦΥΛΑΚΗΣ",
        "ΕΠΙΔΟΜΑ ΑΔΕΙΑΣ",
        "ΕΞΕΤΑΣΕΙΣ ΓΙΑ ΤΗΝ ΠΡΟΣΛΗΨΗ ΠΡΟΣΩΠΙΚΟΥ",
        "ΕΛΕΓΧΟΣ ΕΞΑΓΩΓΙΚΟΥ ΕΜΠΟΡΙΟΥ",
        "ΡΑΔΙΟΦΩΝΙΚΟΙ ΣΤΑΘΜΟΙ",
        "ΚΑΝΟΝΙΣΜΟΣ ΔΙΟΙΚΗΤΙΚΗΣ ΟΡΓΑΝΩΣΕΩΣ Τ.Σ.Α.Υ",
        "Φ.Κ.Π. ΑΝΩΝΥΜΩΝ ΕΤΑΙΡΕΙΩΝ",
        "ΔΙΑΦΟΡΟΙ ΠΟΛΥΕΘΝΕΙΣ ΟΡΓΑΝΙΣΜΟΙ",
        "ΧΟΛΕΡΑ",
        "EΝΙΑΙΟΣ ΔΗΜΟΣΙΟΓΡΑΦΙΚΟΣ ΟΡΓΑΝΙΣΜΟΣ",
        "ΑΤΕΛΕΙΕΣ ΔΗΜΟΣΙΩΝ ΥΠΗΡΕΣΙΩΝ",
        "ΤΑΜΕΙΟ ΠΡΟΝΟΙΑΣ ΜΗΧΑΝΟΔΗΓΩΝ ΟΔΟΣΤΡΩΤΗΡΩΝ ΚΛΠ",
        "ΝΟΣΟΚΟΜΟΙ",
        "ΝΟΣΟΚΟΜΕΙΑ ΦΥΛΑΚΩΝ",
        "ΑΠΟΚΑΤΑΣΤΑΣΗ ΚΤΗΝΟΤΡΟΦΩΝ",
        "ΤΕΛΗ ΚΑΙ ΕΙΣΦΟΡΕΣ",
        "ΑΚΑΤΑΣΧΕΤΑ",
        "ΞΕΝΟΔΟΧΕΙΑΚΟ ΕΠΙΜΕΛΗΤΗΡΙΟ ΤΗΣ ΕΛΛΑΔΑΣ",
        "ΔΗΜΟΤΟΛΟΓΙΑ",
        "ΣΤΑΤΙΣΤΙΚΗ ΥΠΗΡΕΣΙΑ",
        "ΚΡΑΤΙΚΟ ΕΡΓΑΣΤΗΡΙΟ ΕΛΕΓΧΟΥ ΦΑΡΜΑΚΩΝ",
        "ΑΕΡΟΠΟΡΙΚΗ ΑΣΤΥΝΟΜΙΑ",
        "ΕΚΤΑΚΤΕΣ ΕΙΣΦΟΡΕΣ",
        "ΣΥΝΤΑΞΕΙΣ ΥΠΑΛΛΗΛΩΝ Τ.Τ.Τ",
        "ΜΕΤΡΑ ΚΑΤΑ ΤΗΣ ΦΟΡΟΔΙΑΦΥΓΗΣ",
        "ΕΔΑΦΙΚΗ ΕΠΕΚΤΑΣΗ ΝΟΜΟΘΕΣΙΑΣ",
        "ΜΙΚΡΟΔΙΑΦΟΡΕΣ",
        "ΤΑΤΖΙΚΙΣΤΑΝ – ΤΑΥΛΑΝΔΗ – ΤΟΥΡΚΙΑ Κ.ΛΠ",
        "ΣΥΜΒΑΣΗ ΔΙΕΘΝΟΥΣ ΜΕΤΑΦΟΡΑΣ ΕΜΠΟΡΕΥΜΑΤΩΝ ΟΔΙΚΩΣ",
        "ΚΩΔΙΚΑΣ ΙΔΙΩΤΙΚΟΥ ΝΑΥΤΙΚΟΥ ΔΙΚΑΙΟΥ",
        "ΚΕΝΤΡΑ ΓΕΩΡΓΙΚΗΣ ΕΚΠΑΙΔΕΥΣΗΣ-Ο.Γ.Ε.Ε.Κ.Α",
        "ΤΑΜΕΙΟ ΕΠΙΚΟΥΡΙΚΗΣ ΑΣΦΑΛΙΣΕΩΣ ΠΡΟΣΩΠΙΚΟΥ ΙΔΡΥΜΑΤΩΝ ΕΜΠΟΡΙΚΟΥ ΝΑΥΤΙΚΟΥ",
        "ΓΡΑΦΕΙΟ ΔΙΑΡΚΗ ΚΩΔΙΚΑ ΝΟΜΟΘΕΣΙΑΣ",
        "ΕΡΕΥΝΑ ΙΔΙΩΤΙΚΩΝ ΜΕΤΑΛΛΕΙΩΝ",
        "ΔΙΕΥΘΥΝΣΗ ΔΗΜΟΣΙΩΝ ΕΡΓΩΝ ΑΕΡΟΠΟΡΙΑΣ",
        "ΠΕΡΙ ΝΟΜΑΡΧΩΝ",
        "ΣΥΝΤΑΞΕΙΣ ΘΥΜΑΤΩΝ ΑΠΟ ΕΣΩΤΕΡΙΚΕΣ ΔΙΑΜΑΧΕΣ",
        "ΔΙΑΧΕΙΡΙΣΗ ΕΦΟΔΙΩΝ ΕΞΩΤΕΡΙΚΟΥ",
        "ΟΡΓΑΝΩΣΗ ΥΠΗΡΕΣΙΩΝ ΠΟΛΕΜΙΚΟΥ ΝΑΥΤΙΚΟΥ",
        "ΦΟΡΤΗΓΑ ΠΛΟΙΑ ΑΝΩ ΤΩΝ 4.500 ΤΟΝΝΩΝ",
        "ΡΑΔΙΟΤΗΛΕΓΡΑΦΙΚΗ ΥΠΗΡΕΣΙΑ ΠΛΟΙΩΝ",
        "ΕΠΑΓΓΕΛΜΑΤΙΚΕΣ ΣΧΟΛΕΣ",
        "ΔΙΑΦΟΡΕΣ ΒΙΟΜΗΧΑΝΙΕΣ",
        "ΣΥΝΤΗΡΗΣΗ ΑΕΡΟΣΚΑΦΩΝ",
        "ΟΛΥΜΠΙΑΚΗ ΑΕΡΟΠΟΡΙΑ",
        "ΟΡΓΑΝΙΣΜΟΣ ΧΩΡΟΦΥΛΑΚΗΣ",
        "ΠΕΡΙΘΑΛΨΗ ΦΥΜΑΤΙΚΩΝ ΤΑΧΥΔΡΟΜΙΚΩΝ ΥΠΑΛΛΗΛΩΝ",
        "ΟΡΓΑΝΙΣΜΟΣ ΧΡΗΜΑΤΟΔΟΤΗΣΗΣ ΟΙΚΟΝΟΜΙΚΗΣ ΑΝΑΠΤΥΞΗΣ",
        "ΠΡΩΤΕΣ ΥΛΕΣ ΞΥΛΙΝΩΝ ΒΑΡΕΛΙΩΝ",
        "ΤΑΜΕΙΟ ΑΣΦΑΛΙΣΕΩΣ ΤΕΧΝΙΚΩΝ ΤΥΠΟΥ ΑΘΗΝΩΝ (Τ.Α.Τ.Τ.Α.)",
        "ΠΡΟΠΑΡΑΣΚΕΥΑΣΤΙΚΗ ΣΧΟΛΗ ΚΑΛΩΝ ΤΕΧΝΩΝ ΤΗΝΟΥ",
        "ΟΙΚΟΝΟΜΙΚΕΣ ΑΝΤΙΠΡΟΣΩΠΕΙΕΣ ΕΞΩΤΕΡΙΚΟΥ",
        "ΚΑΛΛΙΤΕΧΝΙΚΟΙ ΣΤΑΘΜΟΙ",
        "ΣΥΜΒΑΣΕΙΣ ΓΙΑ ΤΗ ΒΙΑ ΤΩΝ",
        "ΠΡΟΣΤΑΣΙΑ ΑΜΠΕΛΟΥΡΓΙΚΗΣ ΠΑΡΑΓΩΓΗΣ",
        "ΔΙΑΦΟΡΑ ΑΔΙΚΗΜΑΤΑ",
        "ΑΣΤΥΝΟΜΙΑ ΚΑΙ ΑΣΦΑΛΕΙΑ ΣΙΔΗΡΟΔΡΟΜΩΝ",
        "ΜΕΤΟΧΙΚΟ ΤΑΜΕΙΟ ΒΑΣΙΛΙΚΗΣ ΑΕΡΟΠΟΡΙΑΣ",
        "ΥΠΟΘΗΚΗ ΜΗΧΑΝΙΚΩΝ ΕΓΚΑΤΑΣΤΑΣΕΩΝ",
        "ΕΥΘΥΝΗ ΑΠΟ Τ’ΑΥΤΟΚΙΝΗΤΑ",
        "ΠΡΟΣΤΑΣΙΑ ΜΗΤΡΟΤΗΤΟΣ ΚΑΙ ΒΡΕΦΩΝ",
        "ΜΕ ΤΗ ΦΙΛΑΝΔΙΑ",
        "ΕΠΑΡΧΙΑΚΟΣ ΤΥΠΟΣ",
        "ΕΠΙΘΕΩΡΗΣΗ ΤΕΛΩΝΕΙΩΝ",
        "ΕΠΙΤΡΟΠΕΙΕΣ ΤΟΠΩΝΥΜΙΩΝ",
        "ΜΕΤΑΝΑΣΤΕΥΣΗ ΚΑΙ ΑΠΟΔΗΜΙΑ",
        "ΔΙΚΗΓΟΡΙΚΟΙ ΣΥΛΛΟΓΟΙ",
        "ΠΡΟΣΩΠΙΚΟ ΥΠΟΥΡΓΕΙΟΥ ΓΕΩΡΓΙΑΣ",
        "ΤΜΗΜΑ ΟΙΚΟΝΟΜΙΚΩΝ ΕΠΙΣΤΗΜΩΝ ΠΑΝΜΙΟΥ ΠΑΤΡΩΝ",
        "ΜΑΛΑΚΤΕΣ",
        "ΕΛΑΙΑ",
        "ΑΤΟΜΙΚΑ ΕΓΓΡΑΦΑ ΑΞΙΩΜΑΤΙΚΩΝ",
        "ΑΓΡΟΤΙΚΗ ΤΡΑΠΕΖΑ ΤΗΣ ΕΛΛΑΔΟΣ",
        "ΟΠΤΙΚΟΙ - ΚΑΤΑΣΤΗΜΑΤΑ ΟΠΤΙΚΩΝ ΕΙΔΩΝ",
        "ΔΗΜΟΣΙΕΣ ΕΠΕΝΔΥΣΕΙΣ",
        "ΚΡΑΤΙΚΗ ΟΡΧΗΣΤΡΑ ΘΕΣΣΑΛΟΝΙΚΗΣ",
        "ΝΗΟΛΟΓΙΑ-ΥΠΟΘΗΚΟΛΟΓΙΑ-ΣΗΜΑΤΟΛΟΓΗΣΗ",
        "ΤΑΜΕΙΟ ΣΥΝΤΑΞΕΩΝ ΠΡΟΣΩΠΙΚΟΥ ΕΤΑΙΡΕΙΑΣ ΔΙΑΧΕΙΡΙΣΕΩΣ ΕΙΔΩΝ ΜΟΝΟΠΩΛΙΟΥ (Τ.Α.Π.-Ε.Δ.Ε.Μ.Ε.)",
        "ΕΙΣΠΡΑΞΗ ΑΞΙΩΝ",
        "ΥΓΕΙΟΝΟΜΙΚΟΣ ΕΛΕΓΧΟΣ ΤΡΟΦΙΜΩΝ-ΠΟΤΩΝ-ΝΕΡΩΝ",
        "ΛΟΓΙΣΤΕΣ - ΦΟΡΟΤΕΧΝΙΚΟΙ",
        "ΕΙΔΙΚΕΣ ΔΙΚΟΝΟΜΙΚΕΣ ΔΙΑΤΑΞΕΙΣ ΓΙΑ ΤΟ ΔΗΜΟΣΙΟ",
        "ΣΧΟΛΕΣ ΣΩΜΑΤΩΝ ΑΣΦΑΛΕΙΑΣ",
        "ΤΑΜΕΙΟΝ ΚΟΙΝΩΦΕΛΩΝ ΕΡΓΩΝ ΛΕΥΚΑΔΟΣ",
        "ΕΙΔΙΚΗ ΑΓΩΓΗ, ΕΙΔΙΚΗ ΕΠΑΓΓΕΛΜΑΤΙΚΗ",
        "ΥΠΗΡΕΣΙΑ ΚΡΑΤΙΚΩΝ ΠΡΟΜΗΘΕΙΩΝ",
        "ΟΙΝΟΛΟΓΙΚΑ ΙΔΡΥΜΑΤΑ",
        "ΣΥΝΘΗΚΕΣ ΕΚΔΟΣΕΩΣ",
        "ΑΞΙΩΜΑΤΙΚΟΙ ΚΑΙ ΥΠΑΞΙΩΜΑΤΙΚΟΙ Λ.Σ",
        "ΥΓΕΙΟΝΟΜΙΚΗ ΕΞΕΤΑΣΗ ΠΡΟΣΩΠΙΚΟΥ",
        "ΞΕΝΑ ΣΧΟΛΕΙΑ ΗΜΕΔΑΠΗΣ",
        "Ε.Σ.Υ.-ΓΕΝΙΚΕΣ ΔΙΑΤΑΞΕΙΣ",
        "ΤΑΜΕΙΑ ΕΦΑΡΜΟΓΗΣ ΣΧΕΔΙΩΝ ΠΟΛΕΩΝ",
        "ΣΥΝΕΤΑΙΡΙΣΜΟΙ ΣΤΡΑΤΙΩΤΙΚΩΝ ΕΙΔΩΝ",
        "ΣΥΝΘΗΚΗ ΠΕΡΙ ΔΙΑΣΤΗΜΑΤΟΣ",
        "ΔΙΑΧΕΙΡΙΣΗ ΑΝΤΑΛΛΑΞΙΜΩΝ ΚΤΗΜΑΤΩΝ",
        "ΠΡΟΣΩΠΙΚΟΝ ΔΙΟΙΚΗΣΕΩΣ",
        "ΣΧΟΛΗ ΕΚΠΤΙΚΩΝ ΛΕΙΤΟΥΡΓΩΝ",
        "ΤΑΜΕΙΟ ΑΣΦΑΛΙΣΕΩΣ ΞΕΝΟΔΟΧΟΥΠΑΛΛΗΛΩΝ (Τ.Α.Ξ.Υ.)",
        "ΣΩΜΑΤΙΚΗ ΙΚΑΝΟΤΗΤΑ ΑΞΙΩΜΑΤΙΚΩΝ",
        "ΒΕΒΑΙΩΣΗ ΕΣΟΔΩΝ ΔΗΜΟΣΙΟΥ ΑΠΟ ΜΕΤΑΛΛΕΙΑ ΚΑΙ ΛΑΤΟΜΕΙΑ",
        "ΔΙΑΦΟΡΟΙ ΕΠΟΙΚΙΣΤΙΚΟΙ ΝΟΜΟΙ",
        "ΕΠΙΚΟΥΡΙΚΟ ΤΑΜΕΙΟ ΑΣΦΑΛΙΣΕΩΣ ΚΡΕΟΠΩΛΩΝ ΚΑΙ ΕΡΓΑΤΟΥΠΑΛΛΗΛΩΝ ΚΡΕΑΤΟΣ (Ε.Τ.Α.Κ.Ε.Κ)",
        "ΟΙΚΟΝΟΜΙΚΟ ΠΑΝΕΠΙΣΤΗΜΙΟ ΑΘΗΝΩΝ",
        "ΓΕΝΙΚΕΣ ΑΠΟΘΗΚΕΣ",
        "ΤΑΜΕΙΑΚΗ ΥΠΗΡΕΣΙΑ",
        "ΓΕΝΙΚΕΣ ΔΙΑΤΑΞΕΙΣ ΠΕΡΙ ΑΝΩΝΥΜΩΝ ΕΤΑΙΡΕΙΩΝ",
        "ΤΟΜΕΑΣ ΕΠΙΚΟΥΡΙΚΗΣ ΑΣΦΑΛΙΣΕΩΣ ΜΙΣΘΩΤΩΝ (ΙΚΑ-ΤΕΑΜ)ΕΙΔΙΚΟΣ ΤΟΜΕΑΣ ΕΠΙΚΟΥΡΙΚΗΣ ΑΣΦΑΛΙΣΕΩΣ ΜΙΣΘΩΤΩΝ (ΙΚΑ-ΕΤΕΑΜ)",
        "ΒΑΡΒΑΚΕΙΟ ΛΥΚΕΙΟ",
        "ΚΩΔΙΚΑΣ ΔΙΚΩΝ ΤΟΥ ΔΗΜΟΣΙΟΥ",
        "ΔΙΕΘΝΕΣ ΤΑΜΕΙΟΝ ΠΕΡΙΘΑΛΨΕΩΣ ΤΟΥ ΠΑΙΔΙΟΥ",
        "ΣΙΔΗΡΟΔΡΟΜΟΙ ΕΛΛΗΝΙΚΟΥ ΚΡΑΤΟΥΣ",
        "ΑΡΔΕΥΣΕΙΣ",
        "ΤΑΜΕΙΟ ΑΡΧΑΙΟΛΟΓΙΚΩΝ ΠΟΡΩΝ ΚΑΙ ΑΠΑΛΛΟΤΡΙΩΣΕΩΝ",
        "ΙΔΡΥΜΑ ΒΥΖΑΝΤΙΝΗΣ ΜΟΥΣΙΚΟΛΟΓΙΑΣ",
        "ΚΥΒΕΡΝΗΤΙΚΟ ΣΥΜΒΟΥΛΙΟ ΕΛΕΓΧΟΥ ΤΙΜΩΝ",
        "ΕΙΔΙΚΟ ΤΑΜΕΙΟ ΕΠΟΙΚΙΣΜΟΥ",
        "ΚΤΗΜΑΤΟΛΟΓΙΑ ΔΗΜΩΝ ΚΑΙ ΚΟΙΝΟΤΗΤΩΝ",
        "ΚΑΤΑΣΚΕΥΗ ΣΤΑΦΙΔΙΝΗΣ",
        "ΔΙΕΘΝΗΣ ΥΓΕΙΟΝΟΜΙΚΟΣ ΚΑΝΟΝΙΣΜΟΣ",
        "ΕΠΕΤΗΡΙΔΑ",
        "ΠΑΓΚΟΣΜΙΟΣ ΟΡΓΑΝΙΣΜΟΣ ΤΟΥΡΙΣΜΟΥ",
        "ΕΝΙΣΧΥΣΗ ΑΠΡΟΣΤΑΤΕΥΤΩΝ ΠΑΙΔΙΩΝ",
        "ΔΙΑΦΟΡΟΙ ΕΠΙΣΙΤΙΣΤΙΚΟΙ ΝΟΜΟΙ",
        "ΔΙΠΛΩΜΑΤΙΚΕΣ ΑΤΕΛΕΙΕΣ",
        "ΜΕΤΑ ΤΟΥ ΒΕΛΓΙΟΥ",
        "ΚΑΝΝΑΒΙΣ",
        "ΕΚΤΕΛΕΣΗ",
        "ΤΟΥΡΙΣΤΙΚΕΣ ΕΓΚΑΤΑΣΤΑΣΕΙΣ ΡΟΔΟΥ",
        "ΠΟΙΝΙΚΟ ΜΗΤΡΩΟ",
        "ΑΝΩΜΑΛΕΣ ΔΙΚΑΙΟΠΡΑΞΙΕΣ ΔΩΔΕΚΑΝΗΣΟΥ",
        "ΕΜΠΟΡΙΚΑ ΚΑΙ ΒΙΟΜΗΧΑΝΙΚΑ ΕΠΙΜΕΛΗΤΗΡΙΑ",
        "ΣΥΝΤΟΝΙΣΜΟΣ ΠΡΟΓΡΑΜΜΑΤΩΝ ΚΑΙ ΕΡΓΑΣΙΩΝ ΟΔΩΝ ΚΑΙ ΕΡΓΩΝ ΚΟΙΝΗΣ ΩΦΕΛΕΙΑΣ",
        "ΠΡΟΣΩΠΙΚΟ ΞΕΝΟΔΟΧΕΙΩΝ",
        "ΙΝΣΤΙΤΟΥΤΟ ΦΥΣΙΚΗΣ ΤΟΥ ΣΤΕΡΕΟΥ ΦΛΟΙΟΥ ΤΗΣ ΓΗΣ",
        "ΕΠΙΚΙΝΔΥΝΕΣ ΟΙΚΟΔΟΜΕΣ",
        "ΑΡΧΕΙΑ ΔΙΚΑΣΤΗΡΙΩΝ",
        "ΣΚΟΠΟΒΟΛΗ",
        "ΑΠΟΝΟΜΗ ΣΥΝΤΑΞΕΩΝ ΤΑΜΕΙΟΥ ΝΟΜΙΚΩΝ",
        "ΣΗΡΟΤΡΟΦΙΑ",
        "ΕΣΩΤΕΡΙΚΟΣ ΚΑΝΟΝΙΣΜΟΣ",
        "ΠΡΟΣΤΑΣΙΑ ΤΗΣ ΚΤΗΝΟΤΡΟΦΙΑΣ",
        "ΧΑΡΤΗΣ",
        "ΥΠΗΡΕΣΙΑ ΕΓΚΛΗΜΑΤΟΛΟΓΙΚΩΝ ΑΝΑΖΗΤΗΣΕΩΝ",
        "ΥΓΕΙΟΝΟΜΙΚΗ ΠΕΡΙΘΑΛΨΗ ΒΟΥΛΕΥΤΩΝ",
        "ΔΙΚΑΙΟΣΤΑΣΙΟ ΠΟΛΕΜΟΥ 1940",
        "ΧΗΜΕΙΟ ΣΤΡΑΤΟΥ",
        "ΕΠΑΡΧΙΑΚΕΣ ΓΕΝΙΚΕΣ ΣΥΝΕΛΕΥΣΕΙΣ",
        "ΛΟΓΑΡΙΑΣΜΟΣ ΑΡΩΓΗΣ ΟΙΚΟΓΕΝΕΙΩΝ ΣΤΡΑΤΙΩΤΙΚΩΝ ΕΝΟΠΛΩΝ ΔΥΝΑΜΕΩΝ",
        "ΚΑΤ’ ΙΔΙΑΝ ΝΑΟΙ",
        "ΠΛΗΡΩΜΗ ΜΕ ΕΠΙΤΑΓΕΣ",
        "ΕΘΝΙΚΕΣ ΣΥΛΛΟΓΙΚΕΣ ΣΥΜΒΑΣΕΙΣ",
        "ΣΩΜΑ ΣΤΡΑΤΟΛΟΓΙΑΣ",
        "ΟΔΟΝΤΙΑΤΡΟΙ",
        "ΤΑΜΕΙΟ ΕΘΝΙΚΟΥ ΣΤΟΛΟΥ",
        "ΣΥΜΠΛΗΡΩΜΑΤΙΚΕΣ ΠΑΡΟΧΕΣ ΜΗΤΡΟΤΗΤΑΣ",
        "ΜΕΤΑΤΡΕΨΙΜΟΤΗΤΑ ΚΑΤΑΘΕΣΕΩΝ",
        "ΠΤΗΝΟΤΡΟΦΙΑ",
        "ΠΤΥΧΙΟΥΧΟΙ ΑΛΛΟΔΑΠΩΝ ΠΑΝΕΠΙΣΤΗΜΙΩΝ - ΔΙΑΠΑΝΕΠΙΣΤΗΜΙΑΚΟ ΚΕΝΤΡΟ ΑΝΑΓΝΩΡΙΣΕΩΣ",
        "ΦΟΡΤΗΓΑ ΑΥΤΟΚΙΝΗΤΑ",
        "ΥΠΗΡΕΣΙΑ ΜΗΧΑΝΙΚΗΣ ΚΑΛΛΙΕΡΓΕΙΑΣ",
        "ΕΛΕΓΧΟΣ ΚΙΝΗΜΑΤΟΓΡΑΦΩΝ",
        "ΔΗΜΟΣΙΟΓΡΑΦΙΚΕΣ ΟΡΓΑΝΩΣΕΙΣ",
        "ΝΑΥΤΙΛΙΑΚΕΣ ΤΡΑΠΕΖΕΣ",
        "ΛΕΙΤΟΥΡΓΙΑ ΥΔΡΟΘΕΡΑΠΕΥΤΗΡΙΩΝ",
        "ΣΥΜΒΟΥΛΙΟ ΕΜΠΟΡΙΚΗΣ ΝΑΥΤΙΛΙΑΣ",
        "ΕΓΓΕΙΟΣ ΦΟΡΟΛΟΓΙΑ ΚΑΠΝΟΥ",
        "ΤΕΛΟΣ ΑΔΕΙΩΝ ΟΙΚΟΔΟΜΩΝ",
        "ΕΘΝΙΚΟΤΗΤΑ ΠΛΟΙΩΝ",
        "ΠΟΛΙΤΙΚΑ ΚΟΜΜΑΤΑ",
        "ΣΧΟΛΗ ΘΕΤΙΚΩΝ ΕΠΙΣΤΗΜΩΝ",
        "ΝΗΟΓΝΩΜΟΝΕΣ",
        "ΔΙΑΦΟΡΟΙ ΠΟΙΝΙΚΟΙ ΝΟΜΟΙ",
        "ΠΡΟΣΩΡΙΝΗ ΑΠΟΛΥΣΗ",
        "ΤΑΜΕΙΟ ΑΛΛΗΛΟΒΟΗΘΕΙΑΣ ΣΤΡΑΤΟΥ ΞΗΡΑΣ",
        "ΥΠΑΞΙΩΜΑΤΙΚΟΙ ΑΕΡΟΠΟΡΙΑΣ",
        "ΦΟΡΟΛΟΓΙΑ ΧΡΗΜΑΤΙΣΤΗΡΙΑΚΩΝ ΣΥΜΒΑΣΕΩΝ",
        "ΠΤΥΧΙΑ ΙΠΤΑΜΕΝΟΥ ΠΡΟΣΩΠΙΚΟΥ",
        "ΚΡΕΑΤΑ ΣΕ ΠΑΚΕΤΑ",
        "ΕΛΕΓΧΟΣ ΟΠΛΟΦΟΡΙΑΣ",
        "ΑΝΑΣΤΟΛΕΣ ΔΗΜΟΣΙΟΥ ΧΡΕΟΥΣ",
        "ΗΛΕΚΤΡΙΚΟΙ ΣΙΔΗΡΟΔΡΟΜΟΙ ΑΘΗΝΩΝ-ΠΕΙΡΑΙΩΣ (Η.Σ.Α.Π)",
        "ΔΙΑΘΕΣΗ ΛΥΜΑΤΩΝ ΚΑΙ ΑΠΟΒΛΗΤΩΝ",
        "ΕΠΙΘΕΩΡΗΣΗ ΤΕΧΝΙΚΗΣ ΕΚΠΑΙΔΕΥΣΗΣ",
        "ΤΕΛΗ ΑΔΕΙΩΝ ΕΞΑΓΩΓΗΣ",
        "ΠΡΟΙΟΝΤΑ ΓΑΛΑΚΤΟΣ",
        "ΓΕΩΡΓΙΚΑ ΕΠΙΜΕΛΗΤΗΡΙΑ",
        "ΙΕΡΑΡΧΙΚΟΣ ΄ΕΛΕΓΧΟΣ",
        "ΣΤΡΑΤΙΩΤΙΚΕΣ ΦΥΛΑΚΕΣ",
        "ΤΑΜΕΙΟ ΕΠΙΚ. ΑΣΦΑΛΙΣΕΩΣ ΥΠΑΛΛΗΛΩΝ ΚΑΠΝΕΜΠΟΡΙΚΩΝ ΕΠΙΧΕΙΡΗΣΕΩΝ",
        "ΤΑΜΕΙΟ ΠΡΟΝΟΙΑΣ ΚΑΙ ΕΠΙΚΟΥΡΙΚΗΣ ΑΣΦΑΛΙΣΕΩΣ ΠΡΟΣΩΠΙΚΟΥ ΙΠΠΟΔΡΟΜΙΩΝ (Τ.Α.Π.Ε.Α.Π.Ι.)",
        "ΑΠΟΧΩΡΗΤΗΡΙΑ",
        "ΦΟΡΟΣ ΕΙΣΟΔΗΜΑΤΟΣ ΦΥΣΙΚΩΝ ΚΑΙ ΝΟΜΙΚΩΝ ΠΡΟΣΩΠΩΝ",
        "ΚΑΤΑΣΤΑΤΙΚΕΣ ΔΙΑΤΑΞΕΙΣ ΠΑΡΟΧΩΝ",
        "ΑΤΤΙΚΟ ΜΕΤΡΟ",
        "ΒΟΥΣΤΑΣΙΑ",
        "ΑΠΟΣΤΡΑΤΕΙΕΣ - ΕΠΑΝΑΦΟΡΕΣ",
        "ΤΡΑΠΕΖΙΤΙΚΑ ΔΑΝΕΙΑ ΣΕ ΧΡΥΣΟ ΚΛΠ",
        "ΔΙΚΑΙΟΣΤΑΣΙΟ ΠΟΛΕΜΩΝ",
        "ΕΘΝΙΚΟ ΑΣΤΕΡΟΣΚΟΠΕΙΟ",
        "ΙΔΙΩΤΙΚΕΣ ΕΠΙΧΕΙΡΗΣΕΙΣ ΠΑΡΟΧΗΣ ΥΠΗΡΕΣΙΩΝ ΑΣΦΑΛΕΙΑΣ",
        "ΔΑΝΕΙΑ ΕΞΩΤΕΡΙΚΑ",
        "ΠΝΕΥΜΑΤΙΚΟ ΚΕΝΤΡΟ ΑΘΗΝΩΝ",
        "ΑΠΟΣΒΕΣΕΙΣ",
        "ΔΙΑΦΟΡΟΙ ΟΙΝΙΚΟΙ ΚΑΙ ΣΤΑΦΙΔΙΚΟΙ ΝΟΜΟΙ",
        "ΑΚΑΔΗΜΙΑ ΣΩΜΑΤΙΚΗΣ ΑΓΩΓΗΣ",
        "ΑΜΜΟΛΗΨΙΑ",
        "ΠΡΟΣΩΠΙΚΟ ΠΛΟΗΓΙΚΗΣ ΥΠΗΡΕΣΙΑΣ",
        "ΗΘΙΚΕΣ ΑΜΟΙΒΕΣ ΑΕΡΟΠΟΡΙΑΣ",
        "ΚΩΔΙΚΑΣ ΦΟΡΟΛΟΓΙΑΣ ΟΙΝΟΠΝΕΥΜΑΤΟΣ",
        "ΛΙΜΕΝΙΚΑ ΤΑΜΕΙΑ – ΛΙΜΕΝΙΚΑ ΕΡΓΑ",
        "ΤΑΜΕΙΟ ΕΠΙΚ. ΑΣΦΑΛΙΣΕΩΣ ΥΠΑΛΛΗΛΩΝ ΕΘΝΙΚΟΥ ΟΡΓΑΝΙΣΜΟΥ ΚΑΠΝΟΥ  (Τ.Ε.Α.ΥΕ.Ο.Κ)",
        "ΕΛΕΓΧΟΣ ΤΗΣ ΠΙΣΤΕΩΣ",
        "ΣΤΡΑΤΙΩΤΙΚΗ ΣΧΟΛΗ ΑΞΙΩΜΑΤΙΚΩΝ ΣΩΜΑΤΩΝ",
        "ΒΟΗΘΗΤΙΚΑ ΠΡΟΣΩΠΑ ΤΗΣ ΔΙΚΗΣ",
        "ΟΡΓΑΝΙΣΜΟΣ ΣΧΟΛΙΚΩΝ ΚΤΙΡΙΩΝ",
        "ΒΙΟΜΗΧΑΝΙΕΣ ΔΩΔΕΚΑΝΗΣΟΥ",
        "ΥΓΙΕΙΝΗ ΚΑΙ ΑΣΦΑΛΕΙΑ ΧΩΡΩΝ ΕΡΓΑΣΙΑΣ ΚΑΙ ΕΡΓΑΖΟΜΕΝΩΝ",
        "ΜΕΤΑΤΡΟΠΗ ΤΗΣ ΠΟΙΝΗΣ",
        "ΑΥΤΟΝΟΜΟΣ ΟΙΚΟΔΟΜΙΚΟΣ ΟΡΓΑΝΙΣΜΟΣ ΑΞΙΩΜΑΤΙΚΩΝ",
        "ΟΔΙΚΕΣ ΜΕΤΑΦΟΡΕΣ-ΜΕΤΑΦΟΡΕΙΣ",
        "ΑΡΜΑ ΘΕΣΠΙΔΟΣ",
        "ΔΗΜΟΤΙΚΑ & ΚΟΙΝΟΤΙΚΑ",
        "ΠΕΡΙΦΕΡΕΙΑΚΕΣ ΥΠΗΡΕΣΙΕΣ",
        "ΣΧΟΛΗ ΑΝΘΡΩΠΙΣΤΙΚΩΝ ΚΑΙ ΚΟΙΝΩΝΙΚΩΝ ΕΠΙΣΤΗΜΩΝ",
        "ΣΤΡΑΤΕΥΟΜΕΝΟΙ ΦΟΙΤΗΤΑΙ",
        "ΓΕΝΙΚΑ",
        "ΚΑΤΑΠΟΛΕΜΗΣΗ ΕΠΙΖΩΟΤΙΩΝ",
        "ΟΡΓΑΝΙΣΜΟΣ ΔΙΟΙΚΗΣΕΩΣ ΕΚΚΛΗΣΙΑΣΤΙΚΗΣ ΚΑΙ ΜΟΝΑΣΤΗΡΙΑΚΗΣ ΠΕΡΙΟΥΣΙΑΣ",
        "ΑΠΑΓΟΡΕΥΣΗ ΧΡΗΣΗΣ ΕΠΙΒΛΑΒΩΝ ΟΥΣΙΩΝ",
        "ΨΥΧΟΛΟΓΟΙ",
        "ΠΥΡΑΣΦΑΛΕΙΑ ΕΠΙΧΕΙΡΗΣΕΩΝ ΚΑΙ ΑΠΟΘΗΚΩΝ",
        "ΑΠΟΚΑΤΑΣΤΑΣΙΣ ΑΠΟΡΩΝ ΚΟΡΑΣΙΔΩΝ",
        "ΜΕ ΤΗ ΒΕΝΕΖΟΥΕΛΑ",
        "ΔΙΚΑΙΟ ΤΩΝ ΣΥΝΘΗΚΩΝ",
        "ΚΤΗΝΙΑΤΡΙΚΑ ΜΙΚΡΟΒΙΟΛΟΓΙΚΑ ΕΡΓΑΣΤΗΡΙΑ",
        "ΕΡΓΑΣΤΗΡΙΑ",
        "ΚΑΝΟΝΙΣΜΟΙ TELEX ΚΑΙ TELEFAX",
        "ΟΠΛΑ ΚΑΙ ΣΩΜΑΤΑ ΣΤΡΑΤΟΥ ΞΗΡΑΣ",
        "ΕΚΠΑΙΔΕΥΣΗ ΤΑΧΥΔΡΟΜΙΚΩΝ ΥΠΑΛΛΗΛΩΝ",
        "ΤΙΜΟΛΟΓΙΑ ΠΑΡΟΧΩΝ",
        "ΜΟΥΣΟΥΛΜΑΝΙΚΕΣ ΚΟΙΝΟΤΗΤΕΣ",
        "ΣΤΡΑΤΙΩΤΙΚΑ ΕΡΓΑ ΕΝ ΓΕΝΕΙ",
        "ΣΤΡΑΤΙΩΤΙΚΑ ΝΟΣΟΚΟΜΕΙΑ",
        "ΔΙΟΙΚΗΣΗ ΔΗΜΟΣΙΩΝ ΚΤΗΜΑΤΩΝ –",
        "ΕΙΔΙΚΕΣ ΤΙΜΕΣ ΚΑΥΣΙΜΩΝ ΚΑΙ ΗΛΕΚΤΡΙΚΗΣ ΕΝΕΡΓΕΙΑΣ",
        "ΕΓΓΡΑΦΗ ΣΠΟΥΔΑΣΤΩΝ",
        "ΔΗΜΟΤΙΚΑ-ΚΟΙΝΟΤΙΚΑ ΔΑΣΗ ΚΑΙ ΚΗΠΟΙ",
        "ΔΗΜΟΣΙΑ ΕΠΙΧΕΙΡΗΣΗ ΠΟΛΕΟΔΟΜΙΑΣ ΚΑΙ ΣΤΕΓΑΣΕΩΣ",
        "ΣΥΝΤΑΞΙΟΔΟΤΗΣΗ ΠΡΟΣΩΠΙΚΟΥ Ι.Κ.Α",
        "ΕΞΕΤΑΣΤΙΚΕΣ ΕΠΙΤΡΟΠΕΣ ΒΟΥΛΗΣ",
        "ΜΕΤΡΑ ΚΑΤΑ ΤΩΝ ΠΥΡΚΑΙΩΝ ΔΑΣΩΝ",
        "ΥΠΟΥΡΓΕΙΟ ΕΘΝΙΚΗΣ ΟΙΚΟΝΟΜΙΑΣ",
        "ΣΥΓΚΕΝΤΡΩΣΗ ΠΕΡΙΟΥΣΙΑΣ ΤΟΥ ΔΗΜΟΣΙΟΥ",
        "ΚΑΤΑΣΚΕΥΗ ΚΑΙ ΣΥΝΤΗΡΗΣΗ ΟΔΩΝ",
        "ΤΕΛΩΝΕΙΑΚΑ ΚΤΙΡΙΑ",
        "ΤΑΜΕΙΟ ΣΥΝΤΑΞΕΩΝ ΕΚΤΕΛΩΝΙΣΤΩΝ (Τ.Σ.Ε.)",
        "ΚΑΘΗΓΗΤΙΚΕΣ ΕΔΡΕΣ",
        "ΝΑΥΤΙΚΗ ΕΡΓΑΣΙΑ ΝΕΩΝ",
        "ΕΚΤΕΛΕΣΗ ΘΑΝΑΤΙΚΗΣ ΠΟΙΝΗΣ",
        "ΕΠΙΘΕΩΡΗΣΗ ΠΛΟΙΩΝ",
        "ΔΙΠΛΩΜΑΤΑ ΚΑΙ ΑΔΕΙΕΣ ΝΑΥΤΙΚΗΣ ΙΚΑΝΟΤΗΤΑΣ",
        "ΙΣΤΟΡΙΚΟ ΚΑΙ ΕΘΝΟΛΟΓΙΚΟ ΜΟΥΣΕΙΟ",
        "ΠΡΟΣΤΑΣΙΑ ΕΡΓΑΖΟΜΕΝΗΣ ΝΕΑΣ",
        "ΥΠΗΡΕΣΙΑ ΕΠΙΜΕΛΗΤΩΝ ΑΝΗΛΙΚΩΝ",
        "ΑΣΤΙΚΗ ΕΥΘΥΝΗ ΑΠΟ ΠΥΡΗΝΙΚΗ ΕΝΕΡΓΕΙΑ",
        "ΚΩΔΙΚΑΣ ΦΟΡΟΛΟΓΙΑΣ ΚΑΘΑΡΑΣ ΠΡΟΣΟΔΟΥ",
        "ΕΠΙΘΕΩΡΗΣΗ Υ.Ε.Ν",
        "ΚΑΤΑΓΓΕΛΙΑ ΣΥΜΒΑΣΕΩΣ ΕΡΓΑΣΙΑΣ ΣΥΝΔΙΚΑΛΙΣΤΙΚΩΝ ΣΤΕΛΕΧΩΝ",
        "ΥΓΕΙΟΝΟΜΙΚΕΣ ΔΙΑΤΑΞΕΙΣ",
        "ΔΙΔΑΣΚΑΛΕΙΟ ΜΕΣΗΣ ΕΚΠΑΙΔΕΥΣΗΣ",
        "ΥΠΟΒΡΥΧΙΑ",
        "ΥΠΗΡΕΣΙΑ ΑΠΩΛΕΙΩΝ, ΝΕΚΡΟΤΑΦΕΙΩΝ ΚΛΠ",
        "ΑΓΡΟΤ. ΑΠΟΚΑΤΑΣΤΑΣΗ ΣΤΑ  ΔΩΔΕΚΑΝΗΣΑ",
        "ΕΙΔΙΚΕΣ ΑΠΑΛΛΟΤΡΙΩΣΕΙΣ",
        "ΣΤΕΓΑΣΗ ΤΑΧΥΔΡΟΜΙΚΩΝ ΥΠΗΡΕΣΙΩΝ",
        "ΔΙΑΜΕΤΑΚΟΜΙΣΗ ΝΑΡΚΩΤΙΚΩΝ",
        "ΜΕΤΑΜΟΣΧΕΥΣΗ ΒΙΟΛΟΓΙΚΩΝ ΟΥΣΙΩΝ",
        "ΒΡΑΒΕΙΑ ΚΑΙ ΧΟΡΗΓΙΕΣ",
        "ΕΥΡΩΠΑΙΚΗ ΜΟΡΦΩΤΙΚΗ ΣΥΜΒΑΣΗ",
        "ΤΑΜΕΙΟ ΕΠΙΚΟΥΡΙΚΗΣ ΑΣΦΑΛΙΣΕΩΣ ΠΡΟΣΩΠΙΚΟΥ ΕΛΛΗΝ. ΕΡΥΘΡΟΥ ΣΤΑΥΡΟΥ (Τ.Ε.Α.Π.Ε.Ε.Σ.)",
        "ΑΤΕΛΕΙΕΣ ΕΙΔΩΝ ΒΟΗΘΕΙΑΣ",
        "ΕΚΤΕΛΕΣΗ ΕΡΓΩΝ ΟΧΥΡΩΣΗΣ",
        "ΡΟΥΑΝΤΑ – ΡΟΥΜΑΝΙΑ Κ.ΛΠ",
        "ΜΟΝΙΜΕΣ ΑΝΤΙΠΡΟΣΩΠΕΙΕΣ",
        "ΠΡΟΣΤΑΣΙΑ ΕΦΕΔΡΩΝ ΙΠΤΑΜΕΝΩΝ",
        "ΤΡΑΠΕΖΕΣ ΕΞΩΤΕΡΙΚΟΥ ΕΜΠΟΡΙΟΥ",
        "ΙΑΤΡΙΚΟΝ ΠΡΟΣΩΠΙΚΟΝ ΔΗΜΟΣΙΟΥ ΚΑΙ Ν.Π.Δ.Δ",
        "ΔΙΑΦΟΡΑ  ΜΟΝΑΣΤΗΡΙΑ",
        "ΕΤΑΙΡΕΙΕΣ ΕΠΕΝΔΥΣΕΩΝ - ΧΑΡΤΟΦΥΛΑΚΙΟΥ ΚΑΙ ΑΜΟΙΒΑΙΩΝ ΚΕΦΑΛΑΙΩΝ",
        "ΑΝΑΓΝΩΡΙΣΗ ΤΗΣ ΕΛΛΗΝΙΚΗΣ ΠΟΛΙΤΕΙΑΣ",
        "ΔΙΕΘΝΗΣ ΣΥΜΒΑΣΗ",
        "ΛΙΜΕΝΑΡΧΕΙΑ",
        "ΣΕΙΣΜΟΠΛΗΚΤΟΙ ΘΕΣΣΑΛΙΑΣ",
        "ΣΤΡΑΤΕΥΣΗ ΓΥΝΑΙΚΩΝ",
        "ΣΤΡΑΤΙΩΤΙΚΗ ΥΠΗΡΕΣΙΑ ΚΑΤΑΣΚΕΥΗΣ ΕΡΓΩΝ ΑΝΑΣΥΓΚΡΟΤΗΣΗΣ",
        "ΠΡΟΣΤΑΣΙΑ ΤΗΣ ΤΙΜΗΣ ΤΟΥ ΠΟΛΙΤΙΚΟΥ ΚΟΣΜΟΥ",
        "ΕΠΙΜΟΡΦΩΣΗ ΛΕΙΤΟΥΡΓΩΝ Μ.Ε",
        "ΕΝΙΣΧΥΣΗ ΕΞΑΓΩΓΗΣ",
        "ΗΛΕΚΤΡΟΦΩΤΙΣΜΟΣ ΔΙΑΦΟΡΩΝ ΠΟΛΕΩΝ",
        "ΜΕ ΤΙΣ ΚΑΤΩ ΧΩΡΕΣ",
        "ΝΑΥΠΗΓΟΥΜΕΝΑ ΠΛΟΙΑ-ΝΑΥΠΗΓΟΕΠΙΣΚΕΥΑΣΤΙΚΕΣ",
        "ΕΛΕΓΧΟΣ ΠΩΛΗΣΕΩΝ ΕΠΙ ΠΙΣΤΩΣΕΙ",
        "ΕΛΕΓΧΟΣ ΒΙΟΜΗΧΑΝΙΚΩΝ ΕΓΚΑΤΑΣΤΑΣΕΩΝ",
        "ΔΙΕΘΝΗΣ ΟΙΚΟΝΟΜΙΚΗ ΕΠΙΤΡΟΠΗ",
        "ΓΡΑΦΕΙΑ ΕΥΡΕΣΗΣ ΕΡΓΑΣΙΑΣ - ΣΥΜΒΟΥΛΟΙ ΕΡΓΑΣΙΑΣ",
        "ΜΟΝΟΠΩΛΙΟ ΝΑΡΚΩΤΙΚΩΝ",
        "ΑΠΑΛΛΑΓΕΣ ΦΟΡΟΛΟΓΙΑΣ ΚΛΗΡΟΝΟΜΙΩΝ",
        "ΠΑΓΚΟΣΜΙΑ ΟΡΓΑΝΩΣΗ ΥΓΕΙΑΣ",
        "ΕΘΝΙΚΟ ΙΔΡΥΜΑ ΕΡΕΥΝΩΝ",
        "ΝΟΜΟΘΕΣΙΑ ΠΕΡΙ ΣΥΛΛΟΓΙΚΗΣ ΣΥΜΒΑΣΕΩΣ",
        "ΕΘΝΙΚΟΣ ΟΡΓΑΝΙΣΜΟΣ ΦΑΡΜΑΚΩΝ",
        "ΔΙΑΦΟΡΑ ΓΥΜΝΑΣΙΑ & ΛΥΚΕΙΑ",
        "ΞΕΝΕΣ ΣΧΟΛΕΣ ΓΕΩΠΟΝΙΑΣ ΚΑΙ ΔΑΣΟΛΟΓΙΑΣ",
        "ΠΡΟΣΤΑΣΙΑ ΑΝΕΡΓΩΝ",
        "ΦΙΛΑΝΘΡΩΠΙΚΑ ΚΑΤΑΣΤΗΜΑΤΑ ΚΕΦΑΛΛΗΝΙΑΣ",
        "ΚΑΝΟΝΙΣΜΟΣ ΠΑΡΟΧΩΝ Τ.Ε.Β.Ε",
        "ΩΔΕΙΑ ΚΛΠ. ΜΟΥΣΙΚΑ ΙΔΡΥΜΑΤΑ",
        "ΠΡΟΣΚΥΝΗΜΑΤΙΚΑ ΙΔΡΥΜΑΤΑ",
        "ΤΑΜΕΙΟ ΕΠΙΚΟΥΡΙΚΗΣ ΑΣΦΑΛΙΣΕΩΣ ΠΡΟΣΩΠΙΚΟΥ ΑΝΩΝ. ΥΔΡΟΗΛΕΚΤΡ. ΕΤ. ΓΛΑΥΚΟΣ",
        "ΠΡΕΣΒΕΙΕΣ ΚΑΙ ΠΡΟΞΕΝΕΙΑ",
        "ΥΠΟΥΡΓΕΙΑ ΤΥΠΟΥ ΚΑΙ ΤΟΥΡΙΣΜΟΥ",
        "ΖΩΝΕΣ ΕΝΕΡΓΟΥ ΠΟΛΕΟΔΟΜΙΑΣ",
        "ΕΚΚΛΗΣΙΑ ΙΟΝΙΩΝ ΝΗΣΩΝ",
        "ΕΠΙΤΡΟΠΑΙ ΑΣΦΑΛΕΙΑΣ",
        "ΥΠΟΥΡΓΟΙ",
        "ΠΟΙΝΙΚΗ ΔΙΑΤΙΜΗΣΗ",
        "ΤΑΜΕΙΟ ΑΣΦΑΛΙΣΕΩΣ ΕΡΓΑΤΩΝ ΚΕΡΑΜΟΠΟΙΩΝ",
        "ΠΡΩΤΕΣ ΥΛΕΣ ΠΑΙΓΝΙΟΧΑΡΤΩΝ",
        "ΚΡΥΠΤΟΓΡΑΦΙΚΗ ΥΠΗΡΕΣΙΑ",
        "ΔΙΕΘΝΗΣ ΕΠΙΤΡΟΠΗ ΠΡΟΣΩΠΙΚΗΣ ΚΑΤΑΣΤΑΣΕΩΣ",
        "ΕΛΕΓΧΟΣ ΗΛΕΚΤΡΙΚΩΝ ΕΓΚΑΤΑΣΤΑΣΕΩΝ",
        "ΔΙΑΧΕΙΡΙΣΗ ΙΔΡΥΜΑΤΩΝ ΚΑΙ ΚΛΗΡΟΔΟΤΗΜΑΤΩΝ",
        "ΤΕΛΩΝΕΙΑΚΗ ΣΤΑΤΙΣΤΙΚΗ",
        "ΙΔΙΩΤΙΚΕΣ ΝΑΥΤΙΚΕΣ ΣΧΟΛΕΣ",
        "ΑΕΡΟΠΟΡΙΚΑ ΑΤΥΧΗΜΑΤΑ",
        "ΑΝΩΤΕΡΟ ΔΙΔΑΚΤΙΚΟ ΠΡΟΣΩΠΙΚΟ",
        "ΔΙΑΦΟΡΟΙ ΔΙΟΙΚΗΤΙΚΟΙ ΕΡΓΑΤΙΚΟΙ ΝΟΜΟΙ",
        "ΣΥΜΒΟΥΛΙΟ ΓΕΩΓΡΑΦΙΚΩΝ ΥΠΗΡΕΣΙΩΝ",
        "ΕΚΚΛΗΣΙΑΣΤΙΚΕΣ ΒΙΒΛΙΟΘΗΚΕΣ",
        "ΤΜΗΜΑ ΕΠΙΣΤΗΜΗΣ ΦΥΣΙΚΗΣ ΑΓΩΓΗΣ ΚΑΙ ΑΘΛΗΤΙΣΜΟΥ",
        "ΠΕΡΙΟΡΙΣΜΟΣ ΣΥΝΘΕΣΕΩΣ ΥΠΗΡΕΣΙΩΝ",
        "ΤΑΜΕΙΑ ΕΠΑΡΧΙΑΚΗΣ ΟΔΟΠΟΙΙΑΣ",
        "ΤΙΜΟΛΟΓΙΑ Ο.Τ.Ε - ΚΟΣΤΟΛΟΓΗΣΗ ΥΠΗΡΕΣΙΩΝ Ο.Τ.Ε",
        "ΕΘΝΙΚΗ ΒΙΒΛΙΟΘΗΚΗ",
        "ΔΗΜΟΣΙΕΣ ΣΧΟΛΕΣ ΥΠΟΜΗΧΑΝΙΚΩΝ",
        "ΑΝΑΦΟΡΕΣ ΠΡΟΣ ΤΙΣ ΑΡΧΕΣ",
        "ΚΡΑΤΙΚΗ ΕΚΜΕΤΑΛΛΕΥΣΗ ΛΕΩΦΟΡΕΙΑΚΩΝ ΓΡΑΜΜΩΝ",
        "ΔΙΑΦΟΡΑ ΕΠΙΔΟΜΑΤΑ",
        "ΙΔΙΩΤΙΚΗ ΑΕΡΟΠΟΡΙΑ – ΑΕΡΟΛΕΣΧΕΣ",
        "ΤΜΗΜΑ ΔΙΟΙΚΗΤΙΚΗΣ ΤΩΝ ΕΠΙΧΕΙΡΗΣΕΩΝ",
        "ΔΙΕΘΝΕΙΣ ΑΕΡΟΠΟΡΙΚΕΣ ΣΥΜΒΑΣΕΙΣ",
        "ΠΡΟΙΚΟΔΟΤΗΣΕΙΣ ΕΞ ΕΘΝΙΚΩΝ ΓΑΙΩΝ",
        "ΔΙΟΡΘΩΣΗ ΑΣΥΜΦΩΝΙΩΝ",
        "ΕΠΙΤΡΟΠΗ ΔΙΟΙΚΗΣΕΩΣ",
        "ΜΕΤΑ ΤΗΣ ΓΕΡΜΑΝΙΑΣ",
        "ΟΙΚΟΔΟΜΙΚΟΙ ΣΥΝΕΤΑΙΡΙΣΜΟΙ",
        "ΚΑΤΑΣΤΑΤΙΚΟΙ ΝΟΜΟΙ",
        "ΑΞΙΩΜΑΤΙΚΟΙ ΓΡΑΦΕΙΟΥ",
        "ΚΑΝΟΝΙΣΜΟΣ ΕΝΑΕΡΙΟΥ ΚΥΚΛΟΦΟΡΙΑΣ",
        "ΔΙΑΧΕΙΡΙΣΗ ΚΑΥΣΙΜΩΝ",
        "ΟΜΟΛΟΓΙΑΚΑ ΔΑΝΕΙΑ",
        "ΕΡΓΑ",
        "ΣΧΟΛΗ ΝΑΥΤΙΚΩΝ ΔΟΚΙΜΩΝ",
        "ΠΩΛΗΣΗ ΦΑΡΜΑΚΩΝ ΑΠΟ ΙΑΤΡΟΥΣ",
        "ΣΗΜΑΤΑ ΕΘΝΙΚΟΤΗΤΑΣ ΚΑΙ ΝΗΟΛΟΓΗΣΕΩΣ",
        "ΛΕΙΤΟΥΡΓΟΙ ΣΤΟΙΧΕΙΩΔΟΥΣ",
        "ΕΦΕΤΕΙΑ ΚΑΙ ΠΡΩΤΟΔΙΚΕΙΑ",
        "ΥΠΟΥΡΓΕΙΟ ΠΡΟΕΔΡΙΑΣ ΚΥΒΕΡΝΗΣΕΩΣ",
        "ΜΟΡΦΩΤΙΚΟΣ – ΚΙΝΗΜΑΤΟΓΡΑΦΟΣ",
        "ΚΑΤΑΜΕΤΡΗΣΗ ΧΩΡΗΤΙΚΟΤΗΤΑΣ",
        "ΦΩΤΑΕΡΙΟ",
        "ΠΑΘΗΤΙΚΗ ΑΕΡΑΜΥΝΑ",
        "ΠΡΟΣΩΠΙΚΟ ΝΟΣΗΛΕΥΤΙΚΩΝ ΙΔΡΥΜΑΤΩΝ",
        "ΜΕ ΤΗΝ ΚΥΠΡΟ",
        "ΚΟΛΛΗΓΟΙ (ΕΠΙΜΟΡΤΟΙ ΚΑΛΛΙΕΡΓΗΤΕΣ)",
        "ΤΑΜΕΙΟ ΑΡΩΓΗΣ Λ.Σ",
        "ΙΧΘΥΟΣΚΑΛΕΣ",
        "ΣΧΗΜΑ ΚΑΙ ΤΙΜΗ ΠΩΛΗΣΗΣ ΕΦΗΜΕΡΙΔΩΝ",
        "ΥΙΟΘΕΣΙΑ",
        "ΕΚΤΕΛΕΣΗ ΕΡΓΩΝ ΑΡΜΟΔΙΟΤΗΤΑΣ ΕΚΚΛΗΣΙΑΣ",
        "ΤΑΜΕΙΟ ΣΥΝΤΑΞΕΩΝ ΠΡΟΣΩΠΙΚΟΥ",
        "ΔΙΑΦΟΡΕΣ ΕΥΡΩΠΑΙΚΕΣ ΣΥΜΦΩΝΙΕΣ",
        "ΕΓΓΕΙΟΣ ΦΟΡΟΛΟΓΙΑ",
        "ΠΑΙΔΑΓΩΓΙΚΕΣ ΑΚΑΔΗΜΙΕΣ",
        "ΤΑΜΕΙΟ ΠΡΟΝΟΙΑΣ ΕΡΓΑΤΟΥΠΑΛΛΗΛΩΝ ΜΕΤΑΛΛΟΥ (ΤΑ.Π.Ε.Μ.)",
        "ΤΕΧΝΙΚΗ ΕΚΜΕΤΑΛΛΕΥΣΗ ΑΕΡΟΣΚΑΦΩΝ",
        "ΕΝΩΣΗ ΑΠΟΣΤΡΑΤΩΝ ΑΞΙΩΜΑΤΙΚΩΝ Β.Α",
        "ΑΣΦΑΛΙΣΗ ΕΡΓΑΤΩΝ ΓΕΩΡΓΙΑΣ",
        "ΟΡΓΑΝΩΣΗ ΚΑΛΛΙΤΕΧΝΙΚΩΝ ΕΚΔΗΛΩΣΕΩΝ-ΦΕΣΤΙΒΑΛ",
        "ΠΕΡΙΟΥΣΙΑΚΕΣ ΣΥΝΕΠΕΙΕΣ ΤΗΣ ΠΟΙΝΗΣ",
        "ΤΗΛΕΓΡΑΦΙΚΗ ΑΝΤΑΠΟΚΡΙΣΗ",
        "ΕΠΙΘΕΩΡΗΣΗ ΔΗΜΟΣΙΩΝ ΥΠΟΛΟΓΩΝ",
        "ΜΕ ΤΟΝ ΚΑΝΑΔΑ",
        "ΑΛΛΗΛΟΓΡΑΦΙΑ Υ.Ε.Ν",
        "ΤΕΧΝΙΚΟ ΠΡΟΣΩΠΙΚΟ ΑΕΡΟΠΟΡΙΑΣ",
        "ΚΛΑΔΟΣ ΑΥΤΟΤΕΛΩΣ ΑΠΑΣΧΟΛΟΥΜΕΝΩΝ, ΕΛΕΥΘΕΡΩΝ ΚΑΙ ΑΝΕΞΑΡΤΗΤΩΝ",
        "ΣΧΟΛΕΙΑ ΒΑΡΥΚΟΩΝ Η ΚΩΦΩΝ",
        "ΤΑΜΕΙΟ ΠΡΟΝΟΙΑΣ ΚΑΤΩΤΕΡΩΝ ΠΛΗΡΩΜΑΤΩΝ Ε.Ν",
        "ΤΟΥΡΙΣΤΙΚΑ ΠΛΟΙΑ - ΣΚΑΦΗ ΑΝΑΨΥΧΗΣ - ΤΟΥΡΙΣΤΙΚΟΙ ΛΙΜΕΝΕΣ (ΜΑΡΙΝΕΣ)",
        "ΕΠΙΔΟΜΑΤΑ ΕΟΡΤΩΝ ΧΡΙΣΤΟΥΓΕΝΝΩΝ ΚΑΙ ΠΑΣΧΑ",
        "ΕΠΙΜΕΛΗΤΗΡΙΑ - ΓΕΝΙΚΕΣ ΔΙΑΤΑΞΕΙΣ",
        "ΥΠΟΥΡΓΕΙΟ ΕΡΕΥΝΑΣ ΚΑΙ ΤΕΧΝΟΛΟΓΙΑΣ",
        "ΣΤΕΓΑΣΗ ΑΞΙΩΜΑΤΙΚΩΝ",
        "ΠΑΡΑΡΤΗΜΑΤΑ ΓΕΝΙΚΟΥ ΧΗΜΕΙΟΥ",
        "ΚΑΘΑΡΙΣΤΡΙΕΣ",
        "ΚΑΝΟΝΙΣΜΟΣ ΝΑΥΤΟΔΙΚΕΙΟΥ",
        "ΑΜΟΙΒΕΣ ΜΗΧΑΝΙΚΩΝ",
        "ΕΠΙΜΟΡΦΩΣΗ ΔΗΜΟΣΙΩΝ ΥΠΑΛΛΗΛΩΝ",
        "ΚΑΝΟΝΙΣΜΟΙ ΕΠΙΒΑΤΗΓΩΝ ΠΛΟΙΩΝ",
        "ΤΑΜΕΙΟ ΣΥΝΤΑΞΕΩΝ ΠΡΟΣΩΠΙΚΟΥ ΕΤΑΙΡΙΑΣ ΕΛΛ. ΚΑΛΥΚΟΠΟΙΕΙΟΥ-ΠΥΡΙΤΙΔΟΠΟΙΕΙΟΥ",
        "ΠΡΟΣΩΠΙΚΟ ΤΡΑΠΕΖΩΝ",
        "ΛΥΣΣΙΑΤΡΕΙΑ",
        "ΣΥΝΟΡΙΑΚΕΣ ΥΓΕΙΟΝΟΜΙΚΕΣ ΣΥΜΒΑΣΕΙΣ",
        "ΠΟΛΕΜΙΚΟ ΜΟΥΣΕΙΟ",
        "ΚΑΘΗΚΟΝΤΑ ΤΕΛΩΝΕΙΑΚΩΝ ΥΠΑΛΛΗΛΩΝ",
        "ΕΠΕΚΤΑΣΗ ΤΗΣ ΑΣΦΑΛΙΣΕΩΣ",
        "ΦΟΡΟΛΟΓΙΚΕΣ ΑΠΑΛΛΑΓΕΣ",
        "ΕΠΙΔΟΜΑ ΣΤΡΑΤΕΥΣΗΣ",
        "ΔΙΑΡΚΗ ΣΤΡΑΤΟΔΙΚΕΙΑ",
        "ΣΥΝΤΑΞΙΟΔΟΤΗΣΗ ΠΡΟΣΩΠΙΚΟΥ Ο.Γ.Α",
        "ΑΣΤΥΝΟΜΙΑ ΕΜΠΟΡΙΚΗΣ ΝΑΥΤΙΛΙΑΣ",
        "ΦΡΟΝΤΙΣΤΕΣ ΜΟΝΑΔΩΝ",
        "ΑΡΑΒΟΣΙΤΟΣ",
        "ΜΗΤΡΟΠΟΛΕΙΣ",
        "ΦΙΛΑΝΘΡΩΠΙΚΑ ΣΩΜΑΤΕΙΑ",
        "ΔΙΑΦΟΡΟΙ ΠΟΛΥΕΘΝΕΙΣ ΣΥΜΒΑΣΕΙΣ",
        "ΕΞΥΓΙΑΝΤΙΚΑ ΕΡΓΑ",
        "ΦΥΛΛΑ ΠΟΙΟΤΗΤΑΣ ΝΑΥΤΩΝ",
        "ΦΙΛΑΝΘΡΩΠΙΚΑ ΙΔΡΥΜΑΤΑ ΚΑΙ ΣΩΜΑΤΕΙΑ",
        "ΕΣΤΙΑ ΝΑΥΤΙΚΩΝ",
        "ΓΛΥΚΑ ΚΑΙ ΚΟΝΣΕΡΒΕΣ",
        "ΠΡΟΣΤΑΣΙΑ ΥΠΟΒΡΥΧΙΩΝ ΚΑΛΩΔΙΩΝ",
        "ΕΠΕΞΕΡΓΑΣΙΑ ΚΑΙ ΕΜΠΟΡΙΑ ΣΥΚΩΝ",
        "ΧΑΡΟΚΟΠΕΙΟ",
        "ΔΙΑΜΕΤΑΚΟΜΙΣΗ ΣΤΗΝ ΑΛΒΑΝΙΑ",
        "ΕΠΙΘΕΩΡΗΣΗ ΦΥΛΑΚΩΝ",
        "ΔΙΕΘΝΕΙΣ ΣΥΜΒΑΣΕΙΣ ΠΕΡΙ ΚΥΡΙΑΚΗΣ ΑΡΓΙΑΣ",
        "ΚΙΝΗΜΑΤΟΓΡΑΦΙΚΗ ΒΙΟΜΗΧΑΝΙΑ",
        "ΠΙΣΤΟΠΟΙΗΤΙΚΑ ΠΡΟΕΛΕΥΣΕΩΣ",
        "ΤΟΥΡΙΣΤΙΚΗ ΠΡΟΠΑΓΑΝΔΑ",
        "ΕΙΣΦΟΡΑ ΕΙΣΑΓΩΓΕΩΝ",
        "ΚΑΖΙΝΟ",
        "ΜΕ ΤΗΝ ΕΛΒΕΤΙΑ",
        "ΔΙΚΑΣΤΙΚΟΙ ΕΠΙΜΕΛΗΤΕΣ",
        "ΚΩΔΙΚΑΣ ΠΟΙΝΙΚΗΣ ΔΙΚΟΝΟΜΙΑΣ",
        "ΤΟΠΙΚΕΣ ΔΙΟΙΚΗΤΙΚΕΣ ΕΠΙΤΡΟΠΕΣ",
        "ΕΤΑΙΡΕΙΕΣ ΚΕΦΑΛΑΙΟΠΟΙΗΣΕΩΣ",
        "ΟΡΥΖΑ",
        "ΔΙΟΙΚΗΤΙΚΟ ΣΥΜΒΟΥΛΙΟ Ο.Γ.Α",
        "ΕΚΠΑΙΔΕΥΤΙΚΟ ΠΡΟΣΩΠΙΚΟ ΣΧΟΛΩΝ Π.Ν",
        "ΒΑΣΙΛΕΙΑ ΚΑΙ ΑΝΤΙΒΑΣΙΛΕΙΑ",
        "ΥΠΗΡΕΣΙΑ ΣΤΙΣ ΕΠΑΡΧΙΕΣ Τ.Π. ΚΑΙ Δ",
        "ΓΕΩΡΓΙΚΕΣ ΒΙΟΜΗΧΑΝΙΕΣ",
        "ΒΟΥΛΕΥΤΗΡΙΟ",
        "ΠΟΡΘΜΕΙΑ",
        "ΕΚΤΕΛΕΣΗ ΥΔΡΑΥΛΙΚΩΝ ΕΡΓΩΝ",
        "ΙΝΣΤΙΤΟΥΤΑ ΚΡΗΤΙΚΟΥ ΔΙΚΑΙΟΥ - ΑΙΓΑΙΟΥ ΚΑΙ ΔΙΑΦΟΡΑ ΕΡΕΥΝΗΤΙΚΑ ΚΕΝΤΡΑ",
        "ΑΤΕΛΕΙΕΣ ΔΙΑΦΟΡΕΣ",
        "ΚΕΝΤΡΑ ΠΑΡΑΘΕΡΙΣΜΟΥ -",
        "ΣΧΟΛΕΣ ΑΕΡΟΠΟΡΙΑΣ",
        "ΛΕΠΡΑ",
        "ΑΙΣΘΗΤΙΚΟΙ",
        "ΕΚΚΑΘΑΡΙΣΗ ΠΟΙΝΙΚΩΝ ΕΞΟΔΩΝ",
        "ΓΕΝ. ΟΙΚΟΔΟΜΙΚΟΣ ΚΑΝΟΝΙΣΜΟΣ",
        "ΕΛΕΓΧΟΣ ΔΑΠΑΝΩΝ ΤΟΥ ΚΡΑΤΟΥΣ",
        "ΠΕΤΡΕΛΑΙΟΚΙΝΗΤΑ ΚΑΙ ΙΣΤΙΟΦΟΡΑ",
        "ΚΑΛΛΙΕΡΓΕΙΑ ΚΑΠΝΟΥ",
        "ΔΙΟΙΚΗΣΗ  ΜΟΝΑΣΤΗΡΙΩΝ",
        "ΚΤΗΝΙΑΤΡΙΚΑ ΙΔΙΟΣΚΕΥΑΣΜΑΤΑ",
        "ΜΟΝΙΜΟΙ ΚΑΙ ΕΘΕΛΟΝΤΕΣ",
        "ΦΟΡΟΛΟΓΙΑ ΚΕΡΔΩΝ ΕΙΣΑΓΩΓΕΩΝ",
        "ΑΓΩΓΕΣ ΕΞΩΣΕΩΣ ΜΙΣΘΩΤΩΝ",
        "ΟΡΓΑΝΩΣΗ ΕΞΩΤΕΡΙΚΟΥ ΕΜΠΟΡΙΟΥ",
        "ΑΓΩΓΕΣ ΜΗΧΑΝΙΚΩΝ",
        "ΝΑΥΤΙΚΗ ΣΧΟΛΗ ΠΟΛΕΜΟΥ",
        "ΜΕΤΑΦΟΡΑ ΘΕΣΕΩΝ",
        "ΕΙΣΑΓΩΓΗ ΕΠΑΓΓΕΛΜΑΤΙΚΟΥ ΥΛΙΚΟΥ",
        "ΣΥΓΚΡΟΤΗΣΗ ΚΑΙ ΛΕΙΤΟΥΡΓΙΑ",
        "ΤΑΜΕΙΟ ΕΠΙΚΟΥΡΙΚΗΣ ΑΣΦΑΛΙΣΕΩΣ ΠΡΟΣΩΠΙΚΟΥ ΑΕΡΟΠΟΡΙΚΩΝ ΕΠΙΧΕΙΡΗΣΕΩΝ (T.Ε.Α.Π.Α.Ε.)",
        "ΣΥΛΛΟΓΗ ΚΑΙ ΔΙΑΚΙΝΗΣΗ ΠΕΤΡΕΛΑΙΟΕΙΔΩΝ ΕΡΜΑΤΩΝ",
        "ΚΕΝΤΡΑ ΑΔΥΝΑΤΙΣΜΑΤΟΣ – ΔΙΑΙΤΟΛΟΓΙΑΣ",
        "ΟΜΑΔΙΚΗ ΚΑΤΑΓΓΕΛΙΑ ΣΥΜΒΑΣΕΩΣ ΕΡΓΑΣΙΑΣ",
        "ΔΙΑΦΟΡΑ ΜΟΥΣΕΙΑ",
        "ΒΕΒΑΙΩΣΗ ΚΑΙ ΕΙΣΠΡΑΞΗ ΕΣΟΔΩΝ",
        "ΓΡΑΦΕΙΑ ΤΥΠΟΥ",
        "ΔΙΟΙΚΗΤΙΚΟ ΠΡΟΣΩΠΙΚΟ",
        "ΣΥΝΕΡΓΕΙΑ ΕΠΙΣΚΕΥΩΝ",
        "ΤΑΜΕΙΟ ΕΠΙΚΟΥΡΙΚΗΣ ΑΣΦΑΛΙΣΗΣ ΠΡΟΝΟΙΑΣ ΚΑΙ ΑΣΘΕΝΕΙΑΣ ΕΡΓΑΖΟΜΕΝΩΝ ΣΤΑ ΛΙΜΑΝΙΑ (Τ.Ε.Α.Π.Α.Ε.Λ.)",
        "ΑΣΦΑΛΙΣΗ ΚΑΠΝΕΡΓΑΤΩΝ",
        "ΑΝΤΙΣΗΚΩΜΑΤΑ (ΕΞΑΓΟΡΑ ΘΗΤΕΙΑΣ)",
        "ΡΥΜΟΥΛΚΟΥΜΕΝΑ ΟΧΗΜΑΤΑ",
        "ΝΟΜΟΙ ΑΝΑΦΕΡΟΜΕΝΟΙ ΣΕ ΠΟΛΛΕΣ ΦΟΡΟΛΟΓΙΕΣ",
        "ΟΙΚΟΣΥΣΤΗΜΑΤΑ–ΒΙΟΤΟΠΟΙ",
        "ΠΡΟΣΤΑΣΙΑ ΠΡΟΣΩΠΩΝ",
        "ΕΘΝΙΚΟ ΤΥΠΟΓΡΑΦΕΙΟ",
        "ΔΙΚΑΣΤΙΚΑ ΚΑΤΑΣΤΗΜΑΤΑ",
        "ΠΡΟΣΤΑΣΙΑ ΒΙΒΛΙΟΥ-ΕΘΝΙΚΟ ΚΕΝΤΡΟ ΒΙΒΛΙΟΥ-ΛΟΓΟΤΕΧΝΙΑ",
        "ΔΑΣΜΟΙ ΑΝΤΙΝΤΑΜΠΙΓΚ",
        "ΔΑΣΗ ΠΑΡΑΜΕΘΟΡΙΩΝ ΠΕΡΙΟΧΩΝ",
        "ΘΕΟΛΟΓΙΚΗ ΣΧΟΛΗ",
        "ΟΡΟΙ - ΠΡΟΔΙΑΓΡΑΦΕΣ ΤΥΠΟΠΟΙΗΣΗΣ",
        "ΦΟΡΟΛΟΓΙΑ ΒΥΝΗΣ ΚΑΙ ΖΥΘΟΥ",
        "ΑΠΟΘΗΚΗ ΚΤΗΝΙΑΤΡΙΚΩΝ ΕΦΟΔΙΩΝ",
        "ΠΑΡΟΧΗ ΤΗΛΕΦΩΝΙΚΩΝ ΣΥΝΔΕΣΕΩΝ",
        "ΠΑΡΑΧΩΡΗΣΗ ΙΑΜΑΤΙΚΩΝ ΠΗΓΩΝ",
        "ΜΑΘΗΤΙΚΑ ΣΥΣΣΙΤΙΑ",
        "ΠΡΟΣΛΗΨΗ ΕΦΕΔΡΩΝ, ΑΝΑΠΗΡΩΝ ΠΟΛΕΜΟΥ, ΠΟΛΥΤΕΚΝΩΝ ΚΑΙ ΑΛΛΩΝ ΑΤΟΜΩΝ ΜΕ ΕΙΔΙΚΕΣ ΑΝΑΓΚΕΣ",
        "ΕΡΤ – 3",
        "ΣΧΟΛΗ ΠΟΛΕΜΟΥ ΑΕΡΟΠΟΡΙΑΣ",
        "ΤΟΠΟΘΕΤΗΣΕΙΣ - ΜΕΤΑΤΑΞΕΙΣ",
        "ΔΙΕΘΝΕΙΣ ΣΥΜΒΑΣΕΙΣ ΠΡΟΣΤΑΣΙΑΣ",
        "ΦΥΣΙΚΟ ΑΕΡΙΟ",
        "ΤΕΧΝΙΚΑ ΕΡΓΑ",
        "ΔΙΠΛΩΜΑΤΟΥΧΟΙ ΑΝΩΤΑΤΩΝ",
        "ΕΘΝΙΚΟ ΝΟΜΙΣΜΑΤΙΚΟ ΜΟΥΣΕΙΟ",
        "ΟΙΚΟΝΟΜΙΚΗ ΑΣΤΥΝΟΜΙΑ ΣΤΗ ΘΑΛΑΣΣΑ",
        "ΑΣΦΑΛΕΙΑ, ΛΕΙΤΟΥΡΓΙΑ ΚΑΙ ΕΚΜΕΤΑΛΛΕΥΣΗ",
        "ΕΙΔΙΚΑ ΠΡΟΝΟΜΙΑ ΑΝΩΝΥΜΩΝ ΕΤΑΙΡΕΙΩΝ",
        "ΓΡΑΜΜΑΤΕΙΑ ΤΩΝ ΔΙΚΑΣΤΗΡΙΩΝ ΚΑΙ ΕΙΣΑΓΓΕΛΙΩΝ",
        "ΑΛΙΠΑΣΤΑ",
        "ΕΠΙΔΟΣΗ ΔΙΚΟΓΡΑΦΩΝ",
        "ΚΕΝΤΡΙΚΟ ΤΑΜΕΙΟ ΓΕΩΡΓΙΑΣ",
        "ΣΤΡΑΤΙΩΤΙΚΑ ΣΥΜΒΟΥΛΙΑ",
        "ΤΑΜΕΙΑΚΗ ΥΠΗΡΕΣΙΑ ΤΕΛΩΝΕΙΩΝ",
        "ΝΟΣΗΛΕΥΤΙΚΟ ΙΔΡΥΜΑ Μ.Τ.Σ",
        "ΔΙΚΑΙΟ ΘΑΛΑΣΣΑΣ-ΥΦΑΛΟΚΡΗΠΙΔΑ",
        "ΕΙΔΙΚΟΣ ΦΟΡΟΣ ΚΑΤΑΝΑΛΩΣΗΣ",
        "ΜΕΙΟΝΟΤΙΚΑ ΣΧΟΛΕΙΑ",
        "ΓΡΑΦΕΙΑ ΕΜΠΟΡΙΚΩΝ ΠΛΗΡΟΦΟΡΙΩΝ",
        "ΣΥΝΤΟΝΙΣΤΙΚΟΝ ΣΥΜΒΟΥΛΙΟΝ ΝΕΩΝ ΠΡΟΣΦΥΓΩΝ",
        "ΠΕΡΙΘΑΛΨΗ ΑΠΟΡΩΝ ΚΑΙ ΑΝΑΣΦΑΛΙΣΤΩΝ",
        "ΦΟΡΟΛΟΓΙΑ ΚΕΝΤΡΩΝ ΔΙΑΣΚΕΔΑΣΕΩΣ ΚΑΙ ΠΟΛΥΤΕΛΕΙΑΣ",
        "ΣΠΟΓΓΑΛΙΕΥΤΙΚΑ – ΔΥΤΕΣ",
        "ΔΙΕΘΝΕΣ ΝΟΜΙΣΜΑΤΙΚΟ ΤΑΜΕΙΟ",
        "ΒΙΒΛΙΟ ΔΙΕΚΔΙΚΗΣΕΩΝ",
        "ΕΓΚΑΤΑΣΤΑΣΗ - ΛΕΙΤΟΥΡΓΙΑ ΚΑΤΑΣΚΕΥΩΝ ΚΕΡΑΙΩΝ",
        "ΕΝΩΣΗ ΔΗΜΩΝ ΚΑΙ ΚΟΙΝΟΤΗΤΩΝ",
        "ΛΟΓΙΣΤΙΚΟΣ ΚΑΙ ΟΙΚΟΝΟΜΙΚΟΣ ΚΑΝΟΝΙΣΜΟΣ",
        "ΚΑΤΩΤΕΡΑ ΟΡΓΑΝΑ ΣΩΜΑΤΩΝ ΑΣΦΑΛΕΙΑΣ",
        "ΥΠΟΥΡΓΕΙΟ ΕΜΠΟΡΙΚΗΣ ΝΑΥΤΙΛΙΑΣ",
        "ΟΡΓΑΝΙΣΜΟΣ ΕΛΕΓΚΤΙΚΟΥ ΣΥΝΕΔΡΙΟΥ",
        "ΑΓΟΡΕΣ ΑΓΡΟΤΙΚΩΝ ΠΡΟΙΟΝΤΩΝ",
        "ΤΑΜΕΙΟ ΕΠΙΚΟΥΡΙΚΗΣ ΑΣΦΑΛΙΣΕΩΣ ΜΙΣΘΩΤΩΝ ΚΛΩΣΤΟΥΦΑΝΤΟΥΡΓΙΑΣ",
        "ΞΕΝΑΓΟΙ ΚΑΙ ΔΙΕΡΜΗΝΕΙΣ",
        "ΠΟΛΕΜΙΚΕΣ ΣΥΝΤΑΞΕΙΣ",
        "ΑΣΤΙΚΕΣ ΣΥΓΚΟΙΝΩΝΙΕΣ ΑΘΗΝΩΝ-ΠΕΙΡΑΙΩΣ ΚΑΙ ΠΕΡΙΧΩΡΩΝ-Ο.Α.Σ.Α",
        "ΚΑΤΑΣΤΑΤΙΚΕΣ ΔΙΑΤΑΞΕΙΣ ΤΑΜΕΙΟΥ ΑΣΦΑΛΙΣΕΩΣ ΑΡΤΕΡΓΑΤΩΝ Κ.Λ.Π",
        "ΑΤΥΧΗΜΑΤΑ ΣΕ ΜΕΤΑΛΛΕΙΑ ΚΛΠ",
        "ΦΟΡΟΛΟΓΙΑ ΠΟΛΕΜΙΚΩΝ ΚΕΡΔΩΝ",
        "ΣΧΕΔΙΟ ΠΟΛΕΩΣ ΘΕΣΣΑΛΟΝΙΚΗΣ",
        "ΟΙΚΟΝΟΜΙΚΗ ΔΙΟΙΚΗΣΗ ΑΓΡΟΤ. ΑΣΦΑΛΕΙΑΣ",
        "ΚΡΑΤΙΚΟ ΩΔΕΙΟ ΘΕΣΣΑΛΟΝΙΚΗΣ",
        "ΚΕΝΤΡΑ ΑΝΩΤΕΡΗΣ ΤΕΧΝΙΚΗΣ ΕΚΠΑΙΔΕΥΣΗΣ (Κ.A.Τ.Ε.)",
        "ΤΗΛΕΦΩΝΙΚΗ ΑΝΤΑΠΟΚΡΙΣΗ",
        "ΟΙΚΟΝΟΜΙΚΑ ΓΥΜΝΑΣΙΑ",
        "ΒΙΒΛΙΑ ΚΑΙ ΕΥΡΕΤΗΡΙΑ ΣΥΝΕΤΑΙΡΙΣΜΩΝ",
        "ΕΠΙΔΟΜΑ ΑΝΕΡΓΙΑΣ",
        "ΕΓΓΡΑΦΕΣ, ΕΞΕΤΑΣΕΙΣ, ΠΡΟΓΡΑΜΜΑΤΑ ΚΛΠ",
        "ΣΧΟΛΗ ΜΟΝΙΜΩΝ ΥΠΑΞΙΩΜΑΤΙΚΩΝ",
        "ΕΚΚΛΗΣΙΑ ΑΜΕΡΙΚΗΣ",
        "ΜΕΤΟΧΙΚΟ ΤΑΜΕΙΟ ΣΤΡΑΤΟΥ",
        "ΝΟΣΗΛΕΙΑ",
        "ΣΧΟΛΗ ΕΥΕΛΠΙΔΩΝ",
        "ΥΠΟΥΡΓΕΙΟ ΕΡΓΑΣΙΑΣ ΚΑΙ ΚΟΙΝΩΝΙΚΩΝ ΑΣΦΑΛΙΣΕΩΝ",
        "ΚΑΝΟΝΙΣΜΟΣ ΧΡΗΜΑΤΙΣΤΗΡΙΟΥ ΑΞΙΩΝ ΑΘΗΝΩΝ",
        "ΑΝΤΙΣΕΙΣΜΙΚΟΣ ΚΑΝΟΝΙΣΜΟΣ",
        "ΦΑΡΜΑΚΕΥΤΙΚΗ ΔΕΟΝΤΟΛΟΓΙΑ",
        "ΦΟΡΟΛΟΓΙΑ ΕΛΑΙΩΔΩΝ ΠΡΟΙΟΝΤΩΝ",
        "ΕΙΔΙΚΑ ΡΑΔΙΟΤΗΛΕΦΩΝΙΚΑ ΔΙΚΤΥΑ",
        "ΤΕΧΝΙΚΕΣ ΥΠΗΡΕΣΙΕΣ",
        "ΑΡΧΕΙΑ ΥΓΙΕΙΝΗΣ",
        "ΟΔΟΙΠΟΡΙΚΑ ΚΑΙ ΑΠΟΖΗΜΙΩΣΕΙΣ ΑΠΟΣΤΟΛΩΝ ΕΞΩΤΕΡΙΚΟΥ",
        "ΔΙΑΦΟΡΟΙ ΛΟΓΙΣΤΙΚΟΙ ΝΟΜΟΙ",
        "ΕΚΚΛΗΣΙΑΣΤΙΚΟΙ ΥΠΑΛΛΗΛΟΙ",
        "ΝΑΥΤΙΚΑ ΕΠΑΓΓΕΛΜΑΤΙΚΑ ΣΩΜΑΤΕΙΑ ΚΑΙ ΟΜΟΣΠΟΝΔΙΕΣ",
        "ΤΕΛΗ ΧΡΗΣΗΣ ΑΕΡΟΛΙΜΕΝΩΝ",
        "ΠΡΟΑΙΡΕΤΙΚΗ ΑΣΦΑΛΙΣΗ",
        "ΜΕ ΤΗ ΛΙΒΥΗ",
        "ΠΟΤΑΜΟΠΛΟΙΑ ΦΟΡΤΙΟΥ ΥΓΡΩΝ ΚΑΥΣΙΜΩΝ",
        "ΤΑΜΕΙΟ ΣΥΝΤΑΞΕΩΝ ΠΡΟΣΩΠΙΚΟΥ ΕΛΛΗΝΙΚΩΝ ΗΛΕΚΤΡΙΚΩΝ ΣΙΔΗΡΟΔΡΟΜΩΝ ΑΘΗΝΩΝ-ΠΕΙΡΑΙΩΣ (Τ.Σ.Π.-Η.Σ.Α.Π)",
        "ΜΕΣΑΖΟΝΤΕΣ",
        "ΣΤΡΑΤΙΩΤΙΚΟΣ ΠΟΙΝΙΚΟΣ",
        "ΔΙΚΑΙΩΜΑΤΑ ΚΑΙ ΚΑΘΗΚΟΝΤΑ ΦΟΙΤΗΤΩΝ",
        "ΠΡΟΕΔΡΙΑ ΔΗΜΟΚΡΑΤΙΑΣ",
        "ΚΩΔΙΚΑΣ ΕΜΠΟΡΙΚΟΥ ΝΟΜΟΥ",
        "ΣΥΝΤΑΞΙΟΔΟΤΗΣΗ Ο.Γ.Α",
        "ΣΑΝΑΤΟΡΙΑ",
        "ΕΛΕΓΧΟΣ ΕΜΠΟΡΙΟΥ ΕΙΔΩΝ ΠΡΩΤΗΣ ΑΝΑΓΚΗΣ",
        "ΒΑΛΑΝΙΔΙΑ",
        "ΠΟΛΥΤΕΧΝΙΚΗ ΣΧΟΛΗ ΠΑΝΕΠΙΣΤΗΜΙΟΥ ΠΑΤΡΩΝ",
        "ΣΕΙΣΜΟΠΛΗΚΤΟΙ ΠΕΛΟΠΟΝΝΗΣΟΥ",
        "ΔΙΕΘΝΗΣ ΟΡΓΑΝΙΣΜΟΣ ΧΡΗΜΑΤΟΔΟΤΗΣΕΩΣ",
        "ΜΕΤΑΦΟΡΑ ΣΤΟ ΕΣΩΤΕΡΙΚΟ",
        "ΙΣΤΟΡΙΚΟ ΑΡΧΕΙΟ ΥΔΡΑΣ",
        "ΕΓΚΑΤΑΣΤΑΣΗ ΚΑΙ ΚΙΝΗΣΗ ΑΛΛΟΔΑΠΩΝ",
        "ΣΧΟΛΗ ΤΕΧΝΙΚΗΣ ΕΚΠΑΙΔΕΥΣΗΣ ΑΞΙΩΜΑΤΙΚΩΝ",
        "ΓΑΜΟΣ ΣΤΡΑΤΙΩΤΙΚΩΝ",
        "ΑΠΑΓΟΡΕΥΣΗ ΕΞΟΔΟΥ ΟΦΕΙΛΕΤΩΝ",
        "ΠΡΩΤΕΣ ΥΛΕΣ ΨΕΚΑΣΤΗΡΩΝ",
        "ΦΙΛΕΚΠΑΙΔΕΥΤΙΚΗ ΕΤΑΙΡΕΙΑ",
        "ΑΔΕΙΕΣ ΟΔΗΓΩΝ ΑΥΤΟΚΙΝΗΤΩΝ",
        "ΕΘΝΙΚΗ ΠΙΝΑΚΟΘΗΚΗ ΚΑΙ ΜΟΥΣΕΙΟ ΑΛ. ΣΟΥΤΣΟΥ",
        "ΤΑΧΥΔΡΟΜΙΚΑ ΔΕΜΑΤΑ",
        "ΕΙΣΠΡΑΞΗ ΠΟΡΩΝ",
        "ΟΡΓΑΝΩΣΗ ΚΑΙ ΛΕΙΤΟΥΡΓΙΑ ΤΕΧΝΙΚΩΝ ΣΧΟΛΩΝ",
        "ΔΙΑΘΕΣΗ ΓΑΙΩΝ ΣΤΗ ΘΕΣΣΑΛΙΑ",
        "ΔΙΑΚΡΙΣΗ ΑΣΦΑΛΙΣΜΕΝΩΝ",
        "ΑΓΑΘΟΕΡΓΑ ΙΔΡΥΜΑΤΑ ΚΕΡΚΥΡΑΣ",
        "ΥΠΑΙΘΡΙΟ-ΠΛΑΝΟΔΙΟ ΕΜΠΟΡΙΟ ΚΑΙ ΕΜΠΟΡΟΠΑΝΗΓΥΡΕΙΣ",
        "ΕΞΑΓΩΓΙΚΑ ΤΕΛΗ",
        "ΥΠΟΥΡΓΙΚΟ ΣΥΜΒΟΥΛΙΟ - ΟΡΓΑΝΩΣΗ ΥΠΟΥΡΓΕΙΩΝ - ΚΥΒΕΡΝΗΤΙΚΕΣ ΕΠΙΤΡΟΠΕΣ",
        "ΑΥΤΟΚΙΝΗΤΑ ΚΑΙ ΑΜΑΞΙΔΙΑ ΑΝΑΠΗΡΩΝ ΠΟΛΕΜΟΥ",
        "ΥΠΗΡΕΣΙΕΣ ΠΕΡΙΦΕΡΕΙΑΚΗΣ ΑΝΑΠΤΥΞΗΣ",
        "ΔΙΑΤΙΜΗΣΗ ΦΑΡΜΑΚΩΝ",
        "ΦΟΡΟΛΟΓΙΑ ΕΙΔΩΝ ΠΟΛΥΤΕΛΕΙΑΣ",
        "ΝΑΥΤΙΚΗ ΠΟΙΝΙΚΗ ΝΟΜΟΘΕΣΙΑ",
        "ΤΑΜΕΙΟ ΕΠΙΚΟΥΡΙΚΗΣ ΑΣΦΑΛΙΣΕΩΣ ΠΡΟΣΩΠΙΚΟΥ ΕΤΑΙΡΕΙΩΝ ΠΕΤΡΕΛΑΙΟΕΙΔΩΝ",
        "ΔΩΡΟ ΕΟΡΤΩΝ ΕΦΗΜΕΡΙΔΟΠΩΛΩΝ",
        "ΔΙΕΥΚΟΛΥΝΣΕΙΣ ΓΙΑ ΤΗΝ ΑΝΟΙΚΟΔΟΜΗΣΗ",
        "ΕΠΙΣΚΕΥΑΣΤΕΣ - ΣΥΝΕΡΓΕΙΑ ΕΠΙΣΚΕΥΗΣ ΑΥΤΟΚΙΝΗΤΩΝΟΔΙΚΗ ΒΟΗΘΕΙΑ ΟΧΗΜΑΤΩΝ",
        "ΠΑΡΑΧΩΡΗΣΗ ΔΑΣΩΝ",
        "ΤΑΜΕΙΟ ΑΣΦΑΛΙΣΕΩΣ ΑΣΘΕΝΕΙΑΣ ΠΡΟΣΩΠΙΚΟΥ ΤΡΑΠΕΖΩΝ ΠΙΣΤΕΩΣ, ΓΕΝΙΚΗΣ ΚΑΙ ΑΜΕΡΙΚΑΝ ΕΞΠΡΕΣ",
        "ΠΛΗΤΤΟΜΕΝΑ ΑΠΟ ΤΗΝ ΑΝΕΡΓΙΑ ΕΠΑΓΓΕΛΜΑΤΑ",
        "ΤΑΜΕΙΑ Κ.Α.Τ.Ε",
        "ΕΙΔΙΚΟΙ ΣΤΡΑΤΙΩΤΙΚΟΙ ΟΡΓΑΝΙΣΜΟΙ",
        "ΤΑΜΕΙΟ ΑΣΦΑΛΙΣΕΩΣ ΠΡΟΣΩΠΙΚΟΥ ΙΟΝΙΚΗΣ ΚΑΙ ΛΑΙΚΗΣ ΤΡΑΠΕΖΑΣ (Τ.Α.Π.- Ι.Λ.Τ.)",
        "ΠΡΟΣΤΑΣΙΑ ΑΠΟ ΑΚΤΙΝΟΒΟΛΙΕΣ",
        "ΚΡΑΤΙΚΟ ΘΕΑΤΡΟ Β. ΕΛΛΑΔΟΣ",
        "ΥΓΕΙΟΝΟΜΙΚΟΣ ΕΛΕΓΧΟΣ ΦΟΙΤΗΤΩΝ",
        "ΔΙΑΦΟΡΑ",
        "ΤΕΛΩΝΕΙΑΚΗ ΥΠΗΡΕΣΙΑ ΣΙΔΗΡΟΔΡΟΜΩΝ",
        "ΕΦΕΥΡΕΣΕΙΣ ΑΦΟΡΩΣΑΙ ΕΘΝ. ΑΜΥΝΑ",
        "ΥΠΟΒΡΥΧΙΟΣ ΤΗΛΕΓΡΑΦΟΣ",
        "ΑΔΕΙΕΣ ΟΙΚΟΔΟΜΗΣ ΞΕΝΟΔΟΧΕΙΩΝ",
        "ΙΝΣΤΙΤΟΥΤΟ ΒΥΖΑΝΤΙΝΩΝ ΣΠΟΥΔΩΝ",
        "ΣΧΟΛΗ ΓΕΩΤΕΧΝΙΚΩΝ ΕΠΙΣΤΗΜΩΝ ΠΑΝΜΙΟΥ ΘΕΣΝΙΚΗΣ",
        "ΒΙΒΛΙΟΘΗΚΕΣ",
        "ΤΑΜΕΙΑ ΑΝΕΓΕΡΣΕΩΣ ΔΙΔΑΚΤΗΡΙΩΝ",
        "ΕΠΙΔΟΜΑ ΒΙΒΛΙΟΘΗΚΗΣ",
        "ΚΑΤΑΣΤΗΜΑΤΑ ΑΦΟΡΟΛΟΓΗΤΩΝ ΕΙΔΩΝ",
        "ΕΠΙΧΕΙΡΗΣΕΙΣ ΠΕΡΙΘΑΛΨΕΩΣ ΗΛΙΚΙΩΜΕΝΩΝ Η ΑΝΑΠΗΡΩΝ",
        "ΛΙΜΕΝΙΚΟΙ ΣΤΑΘΜΟΙ",
        "ΝΟΜΟΘΕΤΙΚΕΣ ΕΞΟΥΣΙΟΔΟΤΗΣΕΙΣ",
        "ΘΑΛΑΜΟΙ ΡΑΔΙΟΙΣΟΤΟΠΩΝ",
        "ΔΙΟΙΚΗΣΗ ΕΚΚΛΗΣΙΑΣΤΙΚΗΣ ΕΚΠΑΙΔΕΥΣΗΣ",
        "ΑΠΑΓΟΡΕΥΜΕΝΕΣ ΚΑΙ",
        "ΗΘΟΠΟΙΟΙ",
        "ΣΥΜΒΑΣΕΙΣ ΠΕΡΙ ΔΙΕΘΝΩΝ ΕΚΘΕΣΕΩΝ",
        "ΣΦΡΑΓΙΣΤΟΣ ΧΑΡΤΗΣ",
        "ΕΤΑΙΡΕΙΕΣ ΔΙΑΧΕΙΡΙΖΟΜΕΝΕΣ ΔΗΜΟΣΙΑ ΣΥΜΦΕΡΟΝΤΑ",
        "ΤΕΛΩΝΕΙΑΚΕΣ ΔΙΕΥΚΟΛΥΝΣΕΙΣ",
        "ΔΕΞΑΜΕΝΟΠΛΟΙΑ",
        "ΚΕΝΤΡΟ ΔΙΕΘΝΟΥΣ ΚΑΙ ΕΥΡΩΠΑΙΚΟΥ",
        "ΕΠΙΒΑΤΗΓΑ ΜΕΣΟΓΕΙΑΚΑ ΚΑΙ ΤΟΥΡΙΣΤΙΚΑ ΠΛΟΙΑ",
        "ΕΠΙΘΕΩΡΗΣΗ ΔΙΚΑΣΤΙΚΩΝ ΥΠΑΛΛΗΛΩΝ",
        "ΚΑΝΟΝΙΣΜΟΣ ΘΕΑΤΡΩΝ ΚΙΝΗΜΑΤΟΓΡΑΦΩΝ ΚΛΠ",
        "ΜΕΤΑΛΛΕΥΤΙΚΟΣ ΚΩΔΙΚΑΣ",
        "ΚΑΤΑΣΤΑΤΙΚΟ Τ.Ε.Α.Α.Π.Α.Ε",
        "ΠΑΝΕΠΙΣΤΗΜΙΑΚΗ ΛΕΣΧΗ",
        "ΕΜΠΟΡΙΚΑ ΚΑΙ ΒΙΟΜΗΧΑΝΙΚΑ ΣΗΜΑΤΑ - (ΔΙΕΘΝΕΙΣ ΣΥΜΒΑΣΕΙΣ)",
        "ΕΠΙΔΟΜΑΤΑ ΑΠΟΛΥΟΜΕΝΩΝ ΟΠΛΙΤΩΝ ΩΣ ΑΝΙΚΑΝΩΝ",
        "ΣΥΜΒΟΥΛΙΟ ΕΝΕΡΓΕΙΑΣ",
        "ΣΧΟΛΗ ΝΟΜΙΚΩΝ,ΟΙΚΟΝΟΜΙΚΩΝ ΚΑΙ ΠΟΛΙΤΙΚΩΝ ΕΠΙΣΤΗΜΩΝ",
        "ΠΡΟΠΛΗΡΩΜΕΣ ΚΑΙ ΠΡΟΚΑΤΑΒΟΛΕΣ",
        "ΚΛΑΔΟΣ ΑΣΘΕΝΕΙΑΣ Τ.Ε.Β.Ε",
        "ΔΙΑΝΟΜΗ ΓΑΙΩΝ ΚΩΠΑΙΔΑΣ",
        "ΠΡΟΣΩΠΙΚΟ ΑΣΦΑΛΕΙΑΣ Ν.Π.Δ.Δ. - ΟΡΓΑΝΙΣΜΩΝ & ΕΠΙΧΕΙΡΗΣΕΩΝ",
        "ΥΠΟΥΡΓΕΙΟ ΥΠΟΔΟΜΩΝ, ΜΕΤΑΦΟΡΩΝ ΚΑΙ ΔΙΚΤΥΩΝ",
        "ΑΕΡΟΝΑΥΑΓΟΣΩΣΤΙΚΗ ΜΟΝΑΔΑ",
        "ΚΟΥΡΕΙΑ, ΚΟΜΜΩΤΗΡΙΑ Κ.Λ.Π",
        "ΤΑΜΕΙΟ ΠΡΟΝΟΙΑΣ ΔΙΚΑΣΤΙΚΩΝ ΕΠΙΜΕΛΗΤΩΝ",
        "ΕΙΔΙΚΑ ΣΥΝΕΡΓΕΙΑ",
        "ΚΑΤΕΨΥΓΜΕΝΑ ΚΡΕΑΤΑ",
        "ΜΕΣΟΓΕΙΑΚΑ ΔΡΟΜΟΛΟΓΙΑ ΕΠΙΒΑΤΗΓΩΝ ΠΛΟΙΩΝ",
        "ΣΥΓΚΡΟΤΗΣΗ ΠΡΟΣΩΠΙΚΟΥ ΑΕΡΟΠΟΡΙΑΣ",
        "ΥΠΑΛΛΗΛΙΚΟΣ ΚΩΔΙΚΑΣ",
        "ΓΕΝΙΚΕΣ ΔΙΑΤΑΞΕΙΣ ΠΕΡΙ ΦΑΡΜΑΚΕΙΩΝ",
        "ΔΙΑΦΟΡΟΙ ΣΤΕΓΑΣΤΙΚΟΙ ΝΟΜΟΙ",
        "ΥΠΟΥΡΓΕΙΟ ΣΥΝΤΟΝΙΣΜΟΥ",
        "ΠΡΟΣΛΗΨΕΙΣ ΣΤΟ ΔΗΜΟΣΙΟ",
        "ΤΑΜΕΙΟ ΕΠΙΚ. ΑΣΦΑΛ. ΠΡΟΣΩΠ. Ο.Ε.Α.Σ. ΚΑΙ ΥΠΑΛΛ. ΓΡΑΦΕΙΩΝ ΚΟΙΝΩΝ ΤΑΜΕΙΩΝ ΙΔΙΩΤΙΚΩΝ ΛΕΩΦΟΡΕΙΩΝ",
        "ΣΤΡΑΤΙΩΤΙΚΗ ΑΣΤΥΝΟΜΙΑ",
        "ΝΟΜΙΣΜΑΤΙΚΕΣ ΣΥΜΒΑΣΕΙΣ",
        "ΑΡΧΗ ΔΙΑΣΦΑΛΙΣΗΣ ΑΠΟΡΡΗΤΟΥ ΕΠΙΚΟΙΝΩΝΙΩΝ (Α.Δ.Α.Ε.)",
        "ΣΤΡΑΤΙΩΤΙΚΑ ΣΥΝΕΡΓΕΙΑ",
        "ΠΡΟΣΩΠΙΚΗ ΚΡΑΤΗΣΗ",
        "ΕΦΗΜΕΡΙΔΑ ΤΗΣ ΚΥΒΕΡΝΗΣΕΩΣ",
        "ΑΝΩΤΑΤΟ ΥΓΕΙΟΝΟΜΙΚΟ ΣΥΜΒΟΥΛΙΟ",
        "ΓΡΑΜΜΑΤΕΙΣ ΣΤΡΑΤΟΔΙΚΕΙΩΝ",
        "ΚΑΤΑΣΤΑΣΗ ΔΙΟΠΩΝ, ΝΑΥΤΩΝ ΚΑΙ ΝΑΥΤΟΠΑΙΔΩΝ",
        "ΠΕΡΙΠΤΩΣΕΙΣ ΑΜΟΙΒΑΙΑΣ ΣΥΝΔΡΟΜΗΣ",
        "ΥΠΟΝΟΜΟΙ ΠΡΩΤΕΥΟΥΣΑΣ",
        "ΤΕΛΗ ΔΙΑΔΡΟΜΗΣ ΕΝΑΕΡΙΟΥ ΧΩΡΟΥ",
        "ΥΓΕΙΟΝΟΜΙΚΑΙ ΕΠΙΤΡΟΠΑΙ",
        "ΙΑΤΡΙΚΕΣ ΕΙΔΙΚΟΤΗΤΕΣ",
        "ΕΡΤ – 2",
        "ΕΚΤΕΛΕΣΗ ΕΡΓΩΝ Ο.Σ.Ε.ΚΑΙ ΣΥΝΔΕΔΕΜΕΝΩΝ ΕΠΙΧΕΙΡΗΣΕΩΝ",
        "ΓΕΩΡΓΙΚΕΣ ΣΧΟΛΕΣ",
        "ΣΥΜΜΕΤΟΧΗ ΣΥΝΕΤΑΙΡΙΣΜΩΝ ΣΕ ΠΡΟΜΗΘΕΙΕΣ ΔΗΜΟΣΙΟΥ",
        "ΔΙΚΑΙΩΜΑ ΧΟΡΤΟΝΟΜΗΣ",
        "ΟΙΚΟΚΥΡΙΚΕΣ ΣΧΟΛΕΣ",
        "ΚΕΝΤΡΑ ΥΓΕΙΑΣ-ΠΟΛΥΙΑΤΡΕΙΑ",
        "ΔΙΚΑΣΤΗΡΙΟ ΣΥΝΔΙΑΛΛΑΓΗΣ ΚΑΙ ΔΙΑΙΤΗΣΙΑΣ",
        "ΕΠΙΘΕΩΡΗΣΗ ΙΧΘΥΩΝ",
        "ΣΥΝΕΤΑΙΡΙΣΜΟΙ ΕΞΕΥΓΕΝΙΣΜΟΥ ΔΕΝΔΡΩΝ",
        "ΦΟΙΤΗΤΕΣ",
        "ΔΟΜΗΣΗ ΕΠΙ ΡΥΜΟΤΟΜΟΥΜΕΝΩΝ ΑΚΙΝΗΤΩΝ",
        "ΑΠΑΣΧΟΛΗΣΗ - ΕΞΕΙΔΙΚΕΥΣΗ - ΚΑΤΑΡΤΙΣΗ ΑΝΕΡΓΩΝ",
        "ΤΑΜΕΙΟ ΕΠΙΚΟΥΡΙΚΗΣ ΑΣΦΑΛΙΣΕΩΣ ΥΠΑΛΛΗΛΩΝ ΦΑΡΜΑΚΕΥΤΙΚΩΝ ΕΡΓΑΣΙΩΝ (Τ.Ε.Α.Υ.Φ.Ε.)",
        "ΝΟΜΙΣΜΑΤΙΚΟ ΣΥΣΤΗΜΑ",
        "ΑΠΟΓΡΑΦΗ ΝΑΥΤΙΚΩΝ",
        "ΕΘΝΙΚΟ ΘΕΑΤΡΟ",
        "ΥΠΗΡΕΣΙΑ ΕΠΙΣΤΗΜΟΝΙΚΗΣ ΄ΕΡΕΥΝΑΣ ΚΑΙ ΑΝΑΠΤΥΞΕΩΣ",
        "ΠΑΡΟΧΕΣ ΑΣΤΥΝΟΜΙΚΟΥ ΠΡΟΣΩΠΙΚΟΥ ΕΛΛΗΝΙΚΗΣ ΑΣΤΥΝΟΜΙΑΣ",
        "ΣΙΒΙΤΑΝΙΔΕΙΟΣ ΣΧΟΛΗ",
        "ΣΤΡΑΤΙΩΤΙΚΗ ΙΑΤΡΙΚΗ ΣΧΟΛΗ",
        "ΥΠΟΥΡΓΕΙΟ ΚΟΙΝΩΝΙΚΩΝ ΥΠΗΡΕΣΙΩΝ",
        "ΑΠΑΓΟΡΕΥΣΗ ΑΠΑΛΛΟΤΡΙΩΣΗΣ ΠΛΟΙΩΝ",
        "ΠΑΝΕΠΙΣΤΗΜΙΑΚΑ ΣΥΓΓΡΑΜΜΑΤΑ",
        "ΜΟΥΣΟΥΛΜΑΝΟΙ",
        "ΔΙΚΑΣΤΙΚΟΙ ΣΥΜΒΟΥΛΟΙ ΠΟΛΕΜΙΚΟΥ ΝΑΥΤΙΚΟΥ",
        "ΑΕΡΟΠΟΡΙΚΑ ΕΡΓΑ ΚΑΙ ΠΡΟΜΗΘΕΙΕΣ",
        "ΤΟΠΙΚΑ ΕΓΓΕΙΟΒΕΛΤΙΩΤΙΚΑ ΕΡΓΑ",
        "ΦΟΡΟΛΟΓΙΑ ΖΩΩΝ",
        "ΣΥΝΤΑΓΜΑ",
        "ΝΟΜΟΙ ΠΕΡΙ ΧΡΗΜΑΤΙΣΤΗΡΙΟΥ - ΕΠΙΤΡΟΠΗ ΚΕΦΑΛΑΙΑΓΟΡΑΣ - ΧΡΗΜΑΤΙΣΤΗΡΙΑΚΗ ΑΓΟΡΑ ΠΑΡΑΓΩΓΩΝ",
        "ΓΕΩΤΡΗΣΕΙΣ",
        "ΤΑΜΕΙΑ ΕΠΙΚΟΥΡΙΚΗΣ ΑΣΦΑΛΙΣΕΩΣ ΚΑΙ ΠΡΟΝΟΙΑΣ ΠΡΟΣΩΠΙΚΟΥ ΕΜΠΟΡΙΚΗΣ ΤΡΑΠΕΖΑΣ ΕΛΛΑΔΑΣ (Τ.Ε.Α.Π.Ε.Τ.Ε ΚΑΙ Τ.Α.Π.Ε.Τ.Ε.)",
        "ΕΦΕΔΡΟΙ ΑΕΡΟΠΟΡΙΑΣ",
        "ΚΑΤ’ ΙΔΙΑΝ ΙΔΙΩΤΙΚΑ ΕΚΠΑΙΔΕΥΤΗΡΙΑ",
        "ΣΧΟΛΗ ΝΟΜΙΚΩΝ ΚΑΙ ΟΙΚΟΝΟΜΙΚΩΝ ΕΠΙΣΤΗΜΩΝ",
        "ΚΑΤΑΒΟΛΗ ΕΙΣΦΟΡΩΝ ΜΕ ΔΟΣΕΙΣ",
        "ΠΑΛΑΙΟΤΕΡΕΣ ΑΕΡΟΠΟΡΙΚΕΣ ΕΤΑΙΡΕΙΕΣ",
        "ΤΡΟΜΟΚΡΑΤΙΑ - ΟΡΓΑΝΩΜΕΝΗ",
        "ΤΑΜΕΙΑ ΕΛΙΑΣ-ΔΑΚΟΚΤΟΝΙΑ",
        "ΓΡΑΦΕΙΑ ΕΥΡΕΣΕΩΣ ΝΑΥΤΙΚΗΣ ΕΡΓΑΣΙΑΣ",
        "ΑΡΤΟΠΟΙΕΙΑ",
        "ΦΟΡΟΛΟΓΙΑ ΚΥΚΛΟΥ ΕΡΓΑΣΙΩΝ",
        "ΣΥΝΑΛΛΑΓΜΑΤΙΚΗ ΚΑΙ ΓΡΑΜΜΑΤΙΟ ΣΕ ΔΙΑΤΑΓΗ",
        "ΠΕΡΙΦΕΡΕΙΑΚΕΣ ΥΠΗΡΕΣΙΕΣ ΥΠΟΥΡΓΕΙΟΥ ΜΕΤΑΦΟΡΩΝ ΚΑΙ ΕΠΙΚΟΙΝΩΝΙΩΝ",
        "ΕΛΛΗΝΙΚΟΣ ΟΡΓΑΝΙΣΜΟΣ ΤΟΥΡΙΣΜΟΥ",
        "ΠΡΟΣΤΑΣΙΑ ΤΡΑΥΜΑΤΙΩΝ, ΑΙΧΜΑΛΩΤΩΝ ΚΑΙ ΑΜΑΧΟΥ ΠΛΗΘΥΣΜΟΥ",
        "ΚΑΝΟΝΙΣΜΟΣ ΛΕΙΤΟΥΡΓΙΑΣ Τ.Ε.Β.Ε",
        "ΣΤΕΓΑΣΗ ΑΝΑΠΗΡΩΝ ΠΟΛΕΜΟΥ",
        "ΑΘΛΗΤΙΣΜΟΣ ΚΑΙ ΨΥΧΑΓΩΓΙΑ Π. ΝΑΥΤΙΚΟΥ",
        "ΑΝΕΛΚΥΣΤΗΡΕΣ - ΑΝΥΨΩΤΙΚΑ ΜΕΣΑ ΚΑΙ ΜΗΧΑΝΗΜΑΤΑ",
        "ΣΥΝΤΑΞΕΙΣ ΠΛΗΡΩΜΑΤΩΝ ΕΠΙΤΑΚΤΩΝ ΠΛΟΙΩΝ",
        "ΔΙΚΑΙΩΜΑΤΑ ΥΠΕΡΗΜΕΡΙΑΣ",
        "ΚΩΔΙΚΑΣ ΠΟΛΕΜΙΚΩΝ ΣΥΝΤΑΞΕΩΝ",
        "ΚΑΠΝΟΣ",
        "ΠΡΟΣΤΑΣΙΑ ΣΕΙΣΜΟΠΛΗΚΤΩΝ",
        "ΑΠΟΣΤΡΑΤΕΙΕΣ ΚΑΙ ΑΠΟΚΑΤΑΣΤΑΣΕΙΣ",
        "ΠΡΟΣΩΠΙΚΟ ΕΠΑΓΓΕΛΜΑΤΙΚΩΝ ΣΧΟΛΩΝ",
        "ΔΙΕΘΝΕΙΣ ΣΥΜΒΑΣΕΙΣ ΓΙΑ ΤΗΝ ΠΡΟΣΤΑΣΙΑ ΤΩΝ ΕΡΓΑΖΟΜΕΝΩΝ ΑΝΗΛΙΚΩΝ",
        "ΚΕΝΤΡΙΚΗ ΑΓΟΡΑ ΑΘΗΝΩΝ",
        "ΕΝΙΣΧΥΣΗ ΕΛΑΙΟΠΑΡΑΓΩΓΗΣ",
        "ΑΝΟΙΚΤΑ ΣΩΦΡΟΝΙΣΤΙΚΑ ΚΑΤΑΣΤΗΜΑΤΑ",
        "ΦΙΛΑΝΘΡΩΠΙΚΑ ΙΔΡΥΜΑΤΑ ΖΑΚΥΝΘΟΥ",
        "ΔΙΑΦΟΡΑ ΕΙΔΗ ΤΡΟΦΙΜΩΝ, ΠΟΤΩΝ & ΑΝΤΙΚΕΙΜΕΝΩΝ",
        "ΦΟΡΟΛΟΓΙΑ ΕΠΙΧΕΙΡΗΣΕΩΝ ΤΥΠΟΥ",
        "ΠΕΡΙΟΡΙΣΜΟΙ ΕΙΣΑΓΩΓΗΣ",
        "ΠΡΟΣΩΡΙΝΗ ΕΙΣΔΟΧΗ ΕΜΠΟΡΕΥΜΑΤΩΝ",
        "ΑΡΧΕΙΟ",
        "ΔΙΥΛΙΣΤΗΡΙΑ ΠΕΤΡΕΛΑΙΟΥ",
        "ΕΙΣΑΓΩΓΗ ΠΑΙΔΑΓΩΓΙΚΟΥ ΥΛΙΚΟΥ",
        "ΕΠΙΘΕΩΡΗΣΗ ΚΛΗΡΟΔΟΤΗΜΑΤΩΝ",
        "ΣΙΔΗΡΟΔΡΟΜΟΙ ΒΟΡΕΙΟΔΥΤΙΚΗΣ ΕΛΛΑΔΟΣ",
        "ΤΑΜΕΙΟ ΕΠΙΚΟΥΡΙΚΗΣ ΑΣΦΑΛΙΣΕΩΣ ΕΡΓΑΤΟΤΕΧΝΙΤΩΝ ΔΟΜΙΚΩΝ ΚΑΙ ΞΥΛΟΥΡΓΙΚΩΝ ΕΡΓΑΣΙΩΝ (Τ.Ε.Α.Ε.Δ.Ξ.Ε.)",
        "ΤΑΜΕΙΑ ΠΡΟΝΟΙΑΣ ΣΤΙΣ ΠΡΕΣΒΕΙΕΣ",
        "ΟΙΚΟΓΕΝΕΙΑΚΟΣ ΠΡΟΓΡΑΜΜΑΤΙΣΜΟΣ - ΥΓΕΙΑ ΠΑΙΔΙΟΥ",
        "ΑΡΧΙΕΡΕΙΣ",
        "ΣΥΜΒΟΥΛΙΑ ΥΠΟΥΡΓΕΙΟΥ ΔΙΚΑΙΟΣΥΝΗΣ",
        "ΝΟΣΟΚΟΜΕΙΑΚΗ ΠΕΡΙΘΑΛΨΗ",
        "ΚΑΤΑΣΤΗΜΑΤΑ ΠΩΛΗΣΕΩΣ ΟΙΝΟΠΝΕΥΜΑΤΩΔΩΝ ΠΟΤΩΝ ΚΑΙ ΚΕΝΤΡΑ ΔΙΑΣΚΕΔΑΣΕΩΣ",
        "ΠΡΩΤΕΥΟΥΣΑ",
        "ΠΟΛΥΤΕΧΝΕΙΟ ΚΡΗΤΗΣ",
        "ΤΑΜΕΙΟ ΕΠΙΚΟΥΡΙΚΗΣ ΑΣΦΑΛΙΣΕΩΣ ΠΡΟΣΩΠΙΚΟΥ ΕΤΑΙΡΕΙΩΝ ΤΣΙΜΕΝΤΩΝ (Τ.Ε.Α.Π.Ε.Τ.)",
        "ΕΛΛΗΝΙΚΟΣ ΤΑΠΗΤΟΥΡΓΙΚΟΣ ΟΡΓΑΝΙΣΜΟΣ",
        "ΕΦΑΡΜΟΓΗ ΔΗΜΟΣΙΟΥΠΑΛΛΗΛΙΚΟΥ ΚΩΔΙΚΑ",
        "ΗΛΕΚΤΡΟΛΟΓΙΚΟ ΕΡΓΑΣΤΗΡΙΟ",
        "ΤΑΜΕΙΟ ΣΥΝΤΑΞΕΩΝ ΜΗΧΑΝΙΚΩΝ ΚΑΙ ΕΡΓΟΛΗΠΤΩΝ",
        "ΜΕΣΙΤΕΣ ΑΣΤΙΚΩΝ ΣΥΜΒΑΣΕΩΝ",
        "ΠΛΩΤΕΣ ΔΕΞΑΜΕΝΕΣ",
        "ΚΑΝΟΝΙΣΜΟΙ ΦΟΡΤΩΣΕΩΝ",
        "ΕΙΔΙΚΑ ΕΠΙΔΟΜΑΤΑ",
        "ΠΟΙΝΙΚΟΣ ΚΩΔΙΚΑΣ",
        "ΕΙΔΙΚΟΣ ΛΟΓΑΡΙΑΣΜΟΣ ΠΡΟΝΟΙΑΣ (Τ.Σ.Ε.Υ.Π.)",
        "ΕΘΝΙΚΗ ΑΝΤΙΣΤΑΣΗ",
        "ΟΡΓΑΝΙΣΜΟΣ ΒΙΟΜΗΧΑΝΙΚΗΣ ΑΝΑΠΤΥΞΗΣ",
        "ΕΡΓΑ ΚΟΙΝΗΣ ΥΠΟΔΟΜΗΣ",
        "ΔΙΕΥΘΥΝΣΗ TΕΛΩΝΕΙΩΝ ΠΕΙΡΑΙΑ",
        "ΙΑΤΡΙΚΗ ΣΧΟΛΗ ΙΩΑΝΝΙΝΩΝ",
        "ΖΩΟΚΛΟΠΗ ΚΑΙ ΖΩΟΚΤΟΝΙΑ",
        "ΡΥΘΜΙΣΙΣ ΚΙΝΗΣΕΩΣ ΕΝ ΟΔΟΙΣ",
        "ΕΤΑΙΡΕΙΕΣ ΠΡΟΣΤΑΣΙΑΣ ΚΡΑΤΟΥΜΕΝΩΝ - ΑΠΟΦΥΛΑΚΙΖΟΜΕΝΩΝ",
        "ΔΑΣΙΚΗ ΔΙΕΥΘΕΤΗΣΗ ΧΕΙΜΑΡΡΩΝ",
        "ΣΥΝΟΡΙΑΚΟΙ ΦΥΛΑΚΕΣ",
        "ΣΧΟΛΗ ΘΕΤΙΚΩΝ ΕΠΙΣΤΗΜΩΝ ΠΑΝΜΙΟΥ ΙΩΑΝΝΙΝΩΝ",
        "ΕΚΠΑΙΔΕΥΣΗ Π.ΝΑΥΤΙΚΟΥ",
        "ΔΙΚΑΙΟΣΤΑΣΙΟ ΕΠΙΣΤΡΑΤΕΥΣΕΩΣ 1974",
        "ΡΑΔΙΟΤΗΛΕΓΡΑΦΙΚΗ ΚΑΙ ΡΑΔΙΟΤΗΛΕΦΩΝΙΚΗ ΥΠΗΡΕΣΙΑ",
        "ΦΑΡΜΑΚΑ-ΙΔΙΟΣΚΕΥΑΣΜΑΤΑ",
        "ΣΥΝΤΕΛΕΣΤΕΣ ΚΕΡΔΟΥΣ ΕΠΑΓΓΕΛΜΑΤΙΩΝ",
        "ΕΘΝΙΚΟ ΚΕΝΤΡΟ ΚΟΙΝΩΝΙΚΩΝ ΕΡΕΥΝΩΝ",
        "ΚΕΦΑΛΑΙΟ ΝΑΥΤΙΚΗΣ ΕΚΠΑΙΔΕΥΣΕΩΣ",
        "ΕΙΣΠΡΑΞΗ ΕΣΟΔΩΝ ΠΑΡΕΛΘΟΥΣΩΝ ΧΡΗΣΕΩΝ",
        "ΟΡΓΑΝΙΣΜΟΣ ΗΝΩΜΕΝΩΝ ΕΘΝΩΝ",
        "ΣΕΙΣΜΟΠΛΗΚΤΟΙ ΝΗΣΟΥ ΘΗΡΑΣ",
        "ΚΕΝΤΡΙΚΗ ΑΓΟΡΑ ΘΕΣΣΑΛΟΝΙΚΗΣ",
        "ΔΙΑΦΘΟΡΑ ΑΛΛΟΔΑΠΩΝ ΔΗΜΟΣΙΩΝ ΛΕΙΤΟΥΡΓΩΝ",
        "ΓΕΩΠΟΝΙΚΟ ΠΑΝΕΠΙΣΤΗΜΙΟ ΑΘΗΝΩΝ",
        "ΚΑΝΟΝΙΣΜΟΣ ΣΤΡΑΤΟΔΙΚΕΙΩΝ",
        "ΔΙΑΦΟΡΕΣ ΥΓΕΙΟΝΟΜΙΚΕΣ ΣΥΜΒΑΣΕΙΣ",
        "ΤΟΥΡΙΣΤΙΚΑ ΛΕΩΦΟΡΕΙΑ",
        "ΔΑΝΕΙΑ ΑΠΟ ΕΚΔΟΤΙΚΕΣ ΤΡΑΠΕΖΕΣ",
        "ΕΠΙΘΑΛΑΣΣΙΑ ΑΡΩΓΗ - ΡΥΜΟΥΛΚΗΣΗ ΠΛΟΙΩΝ",
        "ΠΡΟΣΤΑΣΙΑ ΤΟΥ ΚΑΘΕΣΤΩΤΟΣ",
        "ΣΥΜΒΑΣΕΙΣ ΠΕΡΙ ΥΛΙΚΟΥ ΕΥΗΜΕΡΙΑΣ ΝΑΥΤΙΛΛΟΜΕΝΩΝ",
        "ΜΕΣΙΤΕΣ ΕΓΧΩΡΙΩΝ ΠΡΟΙΟΝΤΩΝ",
        "ΚΡΑΤΙΚΗ ΟΡΧΗΣΤΡΑ ΑΘΗΝΩΝ",
        "ΤΜΗΜΑΤΑ ΜΟΥΣΙΚΩΝ - ΘΕΑΤΡΙΚΩΝ ΣΠΟΥΔΩΝ ΚΑΙ ΕΠΙΚΟΙΝΩΝΙΑΣ - ΜΕΣΩΝ ΜΑΖΙΚΗΣ ΕΝΗΜΕΡΩΣΗΣ",
        "ΠΕΙΘΑΡΧΙΚΗ ΕΞΟΥΣΙΑ ΛΙΜΕΝΙΚΩΝ ΑΡΧΩΝ",
        "ΙΝΣΤΙΤΟΥΤΟ ΑΜΥΝΤΙΚΩΝ ΑΝΑΛΥΣΕΩΝ (Ι.Α.Α.)",
        "ΙΔΙΩΤΙΚΟΙ ΣΤΑΘΜΟΙ ΑΣΥΡΜΑΤΟΥ - ΧΡΗΣΗ ΡΑΔΙΟΣΥΧΝΟΤΗΤΩΝ",
        "ΑΝΑΓΝΩΡΙΣΗ ΞΕΝΩΝ ΚΑΤΑΜΕΤΡΗΣΕΩΝ",
        "ΓΕΝΟΚΤΟΝΙΑ",
        "ΕΠΕΞΕΡΓΑΣΙΑ ΚΑΠΝΟΥ",
        "ΣΥΜΒΟΥΛΙΟ ΕΠΙΚΡΑΤΕΙΑΣ",
        "ΙΑΤΡΟΙ Ι.Κ.Α",
        "ΥΠΟΘΗΚΗ",
        "ΑΡΜΟΔΙΟΤΗΤΑ ΛΙΜΕΝΙΚΟΥ ΣΩΜΑΤΟΣ",
        "ΕΙΣΑΓΩΓΕΣ ΓΙΑ ΕΚΘΕΣΕΙΣ, ΣΥΝΕΔΡΙΑ ΚΛΠ",
        "ΕΥΡΩΠΑΙΚΗ ΤΡΑΠΕΖΑ ΑΝΑΣΥΓΚΡΟΤΗΣΗ-ΑΝΑΠΤΥΞΗ",
        "ΑΕΡΟΔΡΟΜΙΟ ΣΠΑΤΩΝ",
        "ΤΜΗΜΑ ΔΗΜΟΣΙΟΓΡΑΦΙΑΣ - ΜΕΣΩΝ ΜΑΖΙΚΗΣ ΕΠΙΚΟΙΝΩΝΙΑΣ",
        "ΤΟΚΟΣ",
        "ΕΝΙΣΧΥΣΗ ΠΟΛΕΜΟΠΑΘΩΝ ΚΛΠ. ΑΓΡΟΤΩΝ",
        "ΕΞΟΔΑ ΚΗΔΕΙΑΣ ΣΤΡΑΤΙΩΤΙΚΩΝ",
        "ΠΑΡΟΧΕΣ ΥΠΑΛΛΗΛΩΝ",
        "ΠΡΟΣΤΑΣΙΑ ΣΙΤΟΠΑΡΑΓΩΓΗΣ",
        "ΑΣΦΑΛΙΣΗ Ο.Γ.Α ΑΠΟ ΑΝΕΜΟΘΥΕΛΛΑ ΚΑΙ ΠΛΗΜΜΥΡΑ",
        "ΔΙΕΥΘΥΝΣΗ ΚΑΤΑΣΚΕΥΩΝ ΚΑΙ ΕΞΟΠΛΙΣΜΟΥ",
        "ΤΕΛΩΝΕΙΑΚΟΙ ΥΠΟΛΟΓΟΙ",
        "ΓΕΝΙΚΗ ΓΡΑΜΜΑΤΕΙΑ ΑΘΛΗΤΙΣΜΟΥ",
        "ΣΥΝΤΑΞΕΙΣ",
        "ΑΔΕΙΕΣ ΠΡΟΣΩΠΙΚΟΥ Λ.Σ",
        "ΣΥΝΤΑΞΕΙΣ ΣΤΡΑΤΙΩΤΙΚΩΝ ΠΑΘΟΝΤΩΝ ΣΤΗΝ",
        "ΑΣΦΑΛΙΣΗ ΕΠΙΒΑΤΩΝ",
        "ΑΠΑΛΛΟΤΡΙΩΣΗ ΑΚΙΝΗΤΩΝ",
        "ΣΧΟΛΗ ΕΠΙΣΤΗΜΩΝ ΥΓΕΙΑΣ",
        "ΕΝΟΙΚΙΟΣΤΑΣΙΟ ΒΟΣΚΩΝ",
        "ΤΑΜΕΙΟ ΣΥΝΤΑΞΕΩΝ ΗΘΟΠΟΙΩΝ - ΣΥΓΓΡΑΦΕΩΝ ΤΕΧΝΙΚΩΝ ΘΕΑΤΡΟΥ",
        "ΕΥΡΩΠΑΙΚΟ ΕΝΤΑΛΜΑ ΣΥΛΛΗΨΗΣ",
        "ΑΝΤΙΚΕΙΜΕΝΑ ΔΕΔΗΛΩΜΕΝΗΣ ΑΞΙΑΣ  ΑΝΤΙΚΑΤΑΒΟΛΕΣ",
        "ΓΕΝΙΚΗ ΔΙΕΥΘΥΝΣΗ ΜΕΤΑΦΟΡΩΝ",
        "ΟΡΓΑΝΙΣΜΟΣ ΥΠΟΥΡΓΕΙΟΥ ΔΙΚΑΙΟΣΥΝΗΣ",
        "ΕΥΘΥΝΗ ΥΠΟΥΡΓΩΝ",
        "ΤΜΗΜΑ ΚΤΗΝΙΑΤΡΙΚΗΣ",
        "ΔΙΚΑΣΤΙΚΟ ΣΩΜΑ ΕΝΟΠΛΩΝ ΔΥΝΑΜΕΩΝ",
        "ΕΝΟΡΙΑΚΟΙ ΝΑΟΙ ΚΑΙ ΕΦΗΜΕΡΙΟΙ",
        "ΥΓΕΙΟΝΟΜΙΚΕΣ ΕΠΙΤΡΟΠΕΣ ΝΑΥΤΙΚΟΥ",
        "ΤΑΜΕΙΟ ΕΠΙΚΟΥΡΙΚΗΣ ΑΣΦΑΛΙΣΕΩΣ ΚΑΙ ΠΡΟΝΟΙΑΣ ΠΡΟΣΩΠΙΚΟΥ ΕΛΛΗΝΙΚΗΣ ΡΑΡΙΟΦΩΝΙΑΣ-ΤΗΛΕΟΡΑΣΕΩΣ-ΤΟΥΡΙΣΜΟΥ (Τ.Ε.Α.Π.Π. Ε.Ρ.Τ. Τ.)",
        "ΣΤΡΑΤΙΩΤΙΚΗ ΒΟΗΘΕΙΑ Η.Π.Α",
        "ΣΥΝΤΑΞΕΙΣ ΠΡΟΣΩΠΙΚΟΥ",
        "ΧΡΗΜΑΤΙΚΗ ΔΙΑΧΕΙΡΙΣΗ Π. ΝΑΥΤΙΚΟΥ",
        "ΠΟΛΙΤΙΚΟ ΓΡΑΦΕΙΟ ΠΡΩΘΥΠΟΥΡΓΟΥ",
        "ΛΟΥΤΡΟΘΕΡΑΠΕΙΑ ΚΑΙ ΑΕΡΟΘΕΡΑΠΕΙΑ",
        "ΣΥΜΒΟΥΛΙΟ ΚΟΙΝΩΝΙΚΩΝ ΑΣΦΑΛΙΣΕΩΝ",
        "ΕΝΤΟΚΑ ΓΡΑΜΜΑΤΙΑ",
        "ΣΩΦΡΟΝΙΣΤΙΚΟΣ ΚΩΔΙΚΑΣ",
        "ΔΗΜΟΤΙΚΕΣ ΕΠΙΧΕΙΡΗΣΕΙΣ",
        "ΚΩΔΙΚΑΣ ΠΟΛΙΤΙΚΗΣ ΔΙΚΟΝΟΜΙΑΣ - ΝΕΟΣ",
        "ΤΑΜΕΙΟ ΕΠΙΚΟΥΡΙΚΗΣ ΑΣΦΑΛΙΣΕΩΣ ΥΠΑΛΛΗΛΩΝ ΚΟΥΡΕΙΩΝ ΚΑΙ ΚΟΜΜΩΤΗΡΙΩΝ",
        "ΠΡΟΣΩΠΙΚΟ ΣΙΔΗΡΟΔΡΟΜΩΝ- Ο.Σ.Ε.- ΣΙΔΗΡΟΔΡΟΜΙΚΩΝ ΕΠΙΧΕΙΡΗΣΕΩΝ",
        "ΔΙΑΦΟΡΟΙ ΝΟΜΟΙ ΓΙΑ ΤΟΝ ΤΥΠΟ",
        "ΤΑΧΥΔΡΟΜΙΚΑ ΔΕΛΤΑΡΙΑ",
        "ΤΑΜΕΙΟ ΑΣΦΑΛΙΣΕΩΣ ΠΡΟΣΩΠΙΚΟΥ ΗΛΕΚΤΡ. ΕΤ. ΑΘΗΝΩΝ - ΠΕΙΡΑΙΩΣ ΚΑΙ ΕΛΛΗΝ. ΗΛΕΚΤΡ. ΕΤΑΙΡΙΑΣ (Τ.Α.Π  Η.Ε.Α.Π.- Ε.Η.Ε.)",
        "ΤΑΜΕΙΟ ΕΠΙΚΟΥΡΙΚΗΣ ΑΣΦΑΛΙΣΗΣ ΑΡΤΟΠΟΙΩΝ",
        "ΔΗΜΟΤΙΚΟΙ ΚΑΙ ΚΟΙΝΟΤΙΚΟΙ ΑΡΧΟΝΤΕΣ",
        "ΜΕΤΑΦΟΡΑ ΤΑΧΥΔΡΟΜΕΙΟΥ",
        "ΚΑΝΟΝΙΣΜΟΣ ΠΑΡΟΧΩΝ ΤΑΜΕΙΟΥ ΕΠΙΚΟΥΡΙΚΗΣ ΑΣΦΑΛΙΣΕΩΣ ΑΣΦΑΛΙΣΤΩΝ ΚΑΙ ΠΡΟΣΩΠΙΚΟΥ ΑΣΦΑΛΙΣΤΙΚΩΝ ΕΠΙΧΕΙΡΗΣΕΩΝ (Τ.Ε.Α.Α.Π.Α.Ε.)",
        "ΠΡΟΣΩΠΙΚΟ",
        "ΔΗΜΟΣΙΑ ΕΠΙΧΕΙΡΗΣΗ ΗΛΕΚΤΡΙΣΜΟΥ",
        "ΚΑΝΟΝΙΣΜΟΙ ΕΡΓΩΝ ΩΠΛΙΣΜΕΝΟΥ ΣΚΥΡΟΔΕΜΑΤΟΣ",
        "ΑΛΕΥΡΑ-ΑΡΤΟΣ",
        "ΤΕΛΗ ΠΡΟΣΟΡΜΙΣΕΩΣ, ΠΑΡΑΒΟΛΗΣ ΚΑΙ ΠΑΡΟΠΛΙΣΜΟΥ",
        "ΙΔΙΩΤΙΚΑ ΕΚΠΑΙΔΕΥΤΗΡΙΑ ΦΡΟΝΤΙΣΤΗΡΙΑ",
        "ΑΡΧΑΙΟΛΟΓΙΚΗ ΥΠΗΡΕΣΙΑ",
        "ΤΑΜΕΙΟ ΑΣΦΑΛΙΣΕΩΣ ΤΥΠΟΓΡΑΦΩΝ ΚΑΙ ΜΙΣΘΩΤΩΝ ΓΡΑΦΙΚΩΝ ΤΕΧΝΩΝ (Τ.Α.Τ. & Μ.Γ.Τ)",
        "ΕΙΔΙΚΕΣ ΕΦΑΡΜΟΓΕΣ ΚΥΡΙΑΚΗΣ ΑΡΓΙΑΣ",
        "ΔΙΑΦΟΡΟΙ ΝΟΜΟΙ ΓΙΑ ΤΑ ΠΛΗΡΩΜΑΤΑ",
        "ΑΣΤΙΚΑ ΣΧΟΛΕΙΑ",
        "ΤΑΜΕΙΑ ΣΥΝΤΑΞΕΩΝ ΕΦΗΜΕΡΙΔΟΠΩΛΩΝ ΚΑΙ ΥΠΑΛΛΗΛΩΝ ΠΡΑΚΤΟΡΕΙΩΝ ΑΘΗΝΩΝ-ΘΕΣΝΙΚΗΣ (Τ.Σ.Ε.Υ.Π.)",
        "ΔΟΜΙΚΑ ΕΡΓΑ",
        "ΝΑΥΣΤΑΘΜΟΣ",
        "ΑΝΤΙΓΡΑΦΙΚΑ ΔΙΚΑΙΩΜΑΤΑ",
        "ΕΠΙΔΟΜΑ ΟΙΚΟΓΕΝΕΙΑΚΩΝ ΒΑΡΩΝ",
        "ΕΛΛΗΝΙΚΗ-ΕΥΡΩΠΑΙΚΗ ΦΑΡΜΑΚΟΠΟΙΙΑ",
        "ΔΕΛΤΙΑ ΤΑΥΤΟΤΗΤΟΣ",
        "ΣΧΟΛΙΑΤΡΙΚΗ ΥΠΗΡΕΣΙΑ",
        "ΥΔΡΟΓΟΝΑΝΘΡΑΚΕΣ",
        "ΓΕΝΙΚΑ ΠΕΡΙ ΕΚΘΕΣΕΩΝ",
        "ΦΟΡΟΛΟΓΙΚΕΣ ΔΙΕΥΚΟΛΥΝΣΕΙΣ",
        "ΛΣΜΟΣ ΠΡΟΝΟΙΑΣ ΠΡΟΣΩΠΙΚΟΥ Ι.Κ.Α",
        "ΕΛΕΓΧΟΣ ΚΤΙΡΙΑΚΩΝ ΕΡΓΩΝ",
        "ΤΑΜΕΙΟ ΕΠΙΚΟΥΡΙΚΗΣ ΑΣΦΑΛΙΣΗΣ",
        "ΕΛΑΙΟΠΥΡΗΝΕΣ",
        "ΕΜΦΥΤΕΥΤΙΚΑ ΚΤΗΜΑΤΑ",
        "ΤΟΥΡΙΣΤΙΚΕΣ ΣΥΜΒΑΣΕΙΣ",
        "ΚΛΑΔΟΣ ΑΣΦΑΛΙΣΕΩΣ ΤΕΧΝΙΚΩΝ ΤΥΠΟΥ ΘΕΣΣΑΛΟΝΙΚΗΣ (Κ.Α.Τ.Τ.Θ.)",
        "ΜΕΤΕΩΡΟΛΟΓΙΚΗ ΥΠΗΡΕΣΙΑ",
        "ΑΓΡΟΤΙΚΟΣ ΚΩΔΙΚΑΣ",
        "ΤΕΧΝΙΚΟ ΕΠΙΜΕΛΗΤΗΡΙΟ",
        "ΕΛΕΓΧΟΣ ΝΟΜΙΜΟΦΡΟΣΥΝΗΣ",
        "ΑΡΧΑΙΟΛΟΓΙΚΗ ΕΤΑΙΡΙΑ",
        "ΣΧΟΛΑΖΟΥΣΕΣ ΚΛΗΡΟΝΟΜΙΕΣ",
        "ΓΕΦΥΡΑ ΡΙΟΥ - ΑΝΤΙΡΡΙΟΥ",
        "ΦΟΙΤΗΣΗ, ΕΞΕΤΑΣΕΙΣ ΚΛΠ",
        "ΤΥΧΕΡΑ, ΜΙΚΤΑ ΚΑΙ ΤΕΧΝΙΚΑ ΠΑΙΓΝΙΑ",
        "ΟΡΓΑΝΙΚΟΙ ΑΡΙΘΜΟΙ ΥΠΑΞΙΩΜΑΤΙΚΩΝ",
        "ΦΟΡΟΛΟΓΙΑ ΚΙΝΗΤΗΣ ΚΑΙ ΑΚΙΝΗΤΗΣ ΠΕΡΙΟΥΣΙΑΣ",
        "ΑΤΕΛΕΙΕΣ ΑΓΙΟΥ ΟΡΟΥΣ",
        "ΜΟΝΟΠΩΛΙΟ ΑΛΑΤΙΟΥ",
        "ΑΣΦΑΛΙΣΗ ΕΛΛΗΝΩΝ ΕΞΩΤΕΡΙΚΟΥ",
        "ΔΙΕΘΝΕΣ ΚΕΝΤΡΟ ΑΝΩΤΑΤΩΝ",
        "ΑΝΑΠΡΟΣΑΡΜΟΓΕΣ ΣΥΝΤΑΞΕΩΝ",
        "ΓΕΝΙΚΕΣ ΕΠΙΘΕΩΡΗΣΕΙΣ-ΔΙΕΥΘΥΝΣΕΙΣ",
        "ΣΩΜΑ ΟΡΚΩΤΩΝ ΛΟΓΙΣΤΩΝ",
        "ΣΕΙΣΜΟΠΛΗΚΤΟΙ ΒΟΡΕΙΟΥ ΕΛΛΑΔΟΣ",
        "ΠΑΝΕΠΙΣΤΗΜΙΑ ΠΕΙΡΑΙΩΣ-ΜΑΚΕΔΟΝΙΑΣ",
        "ΧΩΡΟΤΑΞΙΑ ΚΑΙ ΠΕΡΙΒΑΛΛΟΝ",
        "ΕΣΩΤΕΡΙΚΟΙ ΚΑΝΟΝΙΣΜΟΙ ΕΡΓΑΣΙΑΣ",
        "ΕΛΕΓΧΟΣ ΝΑΥΤΙΚΩΝ ΑΤΥΧΗΜΑΤΩΝ",
        "ΠΝΕΥΜΑΤΙΚΑ ΚΕΝΤΡΑ",
        "ΠΛΟΗΓΙΚΑ ΔΙΚΑΙΩΜΑΤΑ",
        "ΣΤΡΑΤΕΥΟΜΕΝΟΙ ΔΙΚΗΓΟΡΟΙ",
        "ΣΥΣΤΑΤΙΚΑ ΑΥΤΟΚΙΝΗΤΩΝ",
        "ΣΙΔΗΡΟΔΡΟΜΟΙ ΠΕΛΟΠΟΝΝΗΣΟΥ",
        "ΤΜΗΜΑ ΜΕΘΟΔΟΛΟΓΙΑΣ, ΙΣΤΟΡΙΑΣ ΚΑΙ ΘΕΩΡΙΑΣ ΤΗΣ ΕΠΙΣΤΗΜΗΣ",
        "ΕΥΡΩΠΑΙΚΟ ΠΟΛΙΤΙΣΤΙΚΟ ΚΕΝΤΡΟ ΔΕΛΦΩΝ",
        "ΣΥΝΕΤΑΙΡΙΣΜΟΙ ΕΓΓΕΙΩΝ ΒΕΛΤΙΩΣΕΩΝ",
        "ΤΑΜΕΙΟ ΕΠΙΚΟΥΡΙΚΗΣ ΑΣΦΑΛΙΣΗΣ ΔΗΜΟΣΙΩΝ ΥΠΑΛΛΗΛΩΝ (Τ.Ε.Α.Δ.Υ.)",
        "ΙΕΡΟΚΗΡΥΚΕΣ",
        "ΕΙΡΗΝΟΔΙΚΕΙΑ - ΠΤΑΙΣΜΑΤΟΔΙΚΕΙΑ",
        "ΑΓΟΡΑΝΟΜΙΚΗ ΝΟΜΟΘΕΣΙΑ",
        "ΤΡΑΠΕΖΙΤΙΚΗ ΕΠΙΤΑΓΗ",
        "ΝΑΥΑΓΟΣΩΣΤΙΚΑ ΚΑΙ ΡΥΜΟΥΛΚΑ",
        "ΦΟΡΟΛΟΓΙΚΕΣ ΔΙΑΦΟΡΕΣΙ",
        "ΜΕΤΡΑ ΚΑΙ ΣΤΑΘΜΑ",
        "ΓΕΝΙΚΟ ΧΗΜΕΙΟ ΤΟΥ ΚΡΑΤΟΥΣ",
        "ΣΥΜΦΩΝΙΑ ΓΙΑ ΙΣΑ ΟΙΚΟΝΟΜΙΚΑ ΚΟΙΝΩΝΙΚΑ",
        "ΣΥΝΟΡΙΑΚΟΙ ΣΤΑΘΜΟΙ",
        "ΑΞΙΩΜΑΤΙΚΟΙ ΣΩΜΑΤΩΝ ΑΣΦΑΛΕΙΑΣ",
        "ΥΠΗΡΕΣΙΑΚΑ ΣΥΜΒΟΥΛΙΑ",
        "ΕΙΣΑΓΩΓΙΚΟΣ ΝΟΜΟΣ",
        "ΚΤΗΜΑΤΟΛΟΓΙΟ",
        "ΕΤΑΙΡΕΙΑ ΔΙΑΧΕΙΡΙΣΕΩΣ ΥΠΕΓΓΥΩΝ ΠΡΟΣΟΔΩΝ",
        "ΥΠΟΥΡΓΕΙΟ ΜΑΚΕΔΟΝΙΑΣ – ΘΡΑΚΗΣ",
        "ΤΟΥΡΙΣΤΙΚΑ ΓΡΑΦΕΙΑ ΚΑΙ ΣΩΜΑΤΕΙΑ",
        "ΔΑΝΕΙΑ ΑΝΑΣΥΓΚΡΟΤΗΣΗΣ",
        "ΑΣΤΙΚΕΣ ΣΥΓΚΟΙΝΩΝΙΕΣ ΘΕΣΣΑΛΟΝΙΚΗΣ-Ο.Α.Σ.Θ",
        "ΕΘΕΛΟΝΤΕΣ ΑΕΡΟΠΟΡΙΑΣ",
        "ΣΗΜΕΙΩΤΕΣ",
        "ΤΕΛΗ ΕΓΚΑΤΑΣΤΑΣΗΣ - ΛΕΙΤΟΥΡΓΙΑΣ ΚΕΡΑΙΩΝ",
        "Η.Π.Α",
        "ΠΑΝΕΠΙΣΤΗΜΙΑ ΑΙΓΑΙΟΥ, ΙΟΝΙΟΥ ΚΑΙ ΘΕΣΣΑΛΙΑΣ",
        "ΤΑΜΕΙΟ ΠΡΟΝΟΙΑΣ ΞΕΝΟΔΟΧΩΝ",
        "ΣΥΜΒΟΥΛΙΑ ΣΤΕΓΑΣΕΩΣ",
        "ΤΕΧΝΙΚΗ ΕΚΜΕΤΑΛΛΕΥΣΗ ΙΔΙΩΤΙΚΩΝ ΑΕΡΟΠΛΑΝΩΝ",
        "ΦΟΡΟΛΟΓΙΑ ΔΗΜΟΣΙΩΝ ΘΕΑΜΑΤΩΝ",
        "ΣΤΡΑΤΟΛΟΓΙΑ ΟΠΛΙΤΩΝ ΧΩΡΟΦΥΛΑΚΗΣ",
        "ΓΥΜΝΑΣΙΑ ΑΡΙΣΤΟΥΧΩΝ",
        "ΣΧΟΛΙΚΗ ΑΝΤΙΛΗΨΗ",
        "ΕΥΘΥΝΗ ΣΤΡΑΤΙΩΤΙΚΩΝ",
        "ΣΤΑΘΜΟΙ ΕΠΙΒΗΤΟΡΩΝ",
        "ΒΕΒΑΙΩΣΗ ΠΤΑΙΣΜΑΤΩΝ ΑΠΟ",
        "ΔΙΑΖΥΓΙΟ",
        "ΔΙΕΘΝΗΣ ΣΥΜΒΑΣΗ ΠΕΡΙ ΑΝΑΓΚΑΣΤΙΚΗΣ ΕΡΓΑΣΙΑΣ",
        "ΔΙΕΥΚΟΛΥΝΣΗ ΔΙΕΘΝΟΥΣ ΝΑΥΤΙΛΙΑΚΗΣ ΚΙΝΗΣΕΩΣ",
        "ΕΝΟΙΚΙΟΣΤΑΣΙΟ",
        "ΕΚΘΕΣΕΙΣ ΖΑΠΠΕΙΟΥ ΜΕΓΑΡΟΥ",
        "ΔΙΑΧΕΙΡΙΣΗ ΥΛΙΚΟΥ Π. ΝΑΥΤΙΚΟΥ",
        "ΕΦΕΔΡΙΚΑ ΤΑΜΕΙΑ ΚΡΗΤΗΣ",
        "ΣΙΤΑΡΙ",
        "ΦΟΡΤΗΓΑ 501-4500 ΤΟΝΝΩΝ",
        "ΤΡΑΠΕΖΑ ΕΡΓΑΣΙΑΣ",
        "ΑΤΕΛΕΙΕΣ ΥΠΕΡ ΤΗΣ ΓΕΩΡΓΙΑΣ",
        "ΑΙΓΙΑΛΟΣ ΚΑΙ ΠΑΡΑΛΙΑ",
        "ΔΑΣΗ ΙΔΡΥΜΑΤΩΝ",
        "ΙΧΘΥΟΤΡΟΦΕΙΑ",
        "ΑΠΟΓΡΑΦΕΣ Π. ΝΑΥΤΙΚΟΥ",
        "ΣΗΜΑΤΑ ΚΑΙ ΔΕΛΤΙΑ ΑΝΑΠΗΡΩΝ ΠΟΛΕΜΟΥ",
        "ΠΕΙΘΑΡΧΙΚΟ ΔΙΚΑΙΟ ΑΣΤΥΝΟΜΙΚΟΥ ΠΡΟΣΩΠΙΚΟΥ ΕΛΛΗΝΙΚΗΣ ΑΣΤΥΝΟΜΙΑΣ",
        "ΑΤΜΟΛΕΒΗΤΕΣ",
        "ΤΑΧΥΔΡΟΜΙΚΗ ΥΠΗΡΕΣΙΑ ΣΤΡΑΤΟΥ",
        "ΠΡΟΣΤΑΣΙΑ ΠΙΝΑΚΙΔΩΝ",
        "ΑΓΡΟΤΙΚΑ ΚΤΗΝΙΑΤΡΕΙΑ",
        "ΧΡΗΜΑΤΙΣΤΗΡΙΑΚΑ ΔΙΚΑΣΤΗΡΙΑ",
        "ΕΓΓΡΑΦΗ ΠΡΟΕΡΧΟΜΕΝΩΝ ΑΠΟ ΤΗΝ ΑΛΛΟΔΑΠΗ",
        "ΟΡΓΑΝΙΣΜΟΣ ΔΙΑΧΕΙΡΙΣΗΣ ΔΗΜΟΣΙΟΥ ΥΛΙΚΟΥ",
        "ΠΑΝΕΠΙΣΤΗΜΙΟ ΚΥΠΡΟΥ",
        "ΚΑΤΕΡΓΑΣΙΑ ΞΗΡΑΣ ΣΤΑΦΙΔΑΣ",
        "ΤΕΛΩΝΕΙΑΚΗ ΔΙΑΙΡΕΣΗ",
        "ΑΖΗΤΗΤΑ",
        "ΜΕΛΙΣΣΟΤΡΟΦΙΑ",
        "ΔΙΕΥΘΥΝΣΗ ΘΑΛΑΣΣΙΩΝ ΚΡΑΤΙΚΩΝ ΜΕΤΑΦΟΡΩΝ",
        "ΕΚΜΕΤΑΛΛΕΥΣΗ ΜΕΤΑΛΛΕΙΩΝ ΜΕ ΕΓΓΥΗΣΗ",
        "ΙΔΙΩΤΙΚΕΣ ΕΠΑΓΓΕΛΜΑΤΙΚΕΣ ΣΧΟΛΕΣ",
        "ΔΙΑΘΕΣΗ ΑΧΡΗΣΤΟΥ ΥΛΙΚΟΥ",
        "ΤΑΧΥΔΡΟΜΙΚΕΣ ΜΕΤΑΦΟΡΕΣ",
        "ΕΡΥΘΡΟ ΠΙΠΕΡΙ",
        "ΠΙΚΠΑ-ΕΟΠ-ΚΕΝΤΡΟ ΒΡΕΦΩΝ Η ΜΗΤΕΡΑ-ΕΛΕΠΑΠ",
        "ΣΥΜΜΕΤΟΧΗ ΣΕ ΣΥΜΒΟΥΛΙΑ",
        "ΓΥΜΝΑΣΤΗΡΙΟ",
        "ΙΑΤΡΙΚΟΙ- ΟΔΟΝΤΙΑΤΡΙΚΟΙ ΣΥΛΛΟΓΟΙ",
        "ΕΙΣΑΓΩΓΗ ΦΟΙΤΗΤΩΝ",
        "ΕΛΛΗΝΙΚΟ ΄ΙΔΡΥΜΑ ΠΟΛΙΤΙΣΜΟΥ",
        "ΛΟΙΜΟΚΑΘΑΡΤΗΡΙΑ ΖΩΩΝ",
        "ΔΙΕΘΝΗΣ ΟΡΓΑΝΙΣΜΟΣ ΑΤΟΜΙΚΗΣ ΕΝΕΡΓΕΙΑΣ",
        "ΤΑΜΕΙΟ ΕΞΟΔΟΥ ΚΑΙ ΕΠΙΚΟΥΡΙΚΗΣ ΑΣΦΑΛΙΣΕΩΣ ΜΙΣΘΩΤΩΝ ΒΙΟΜΗΧΑΝΙΑΣ ΚΑΠΝΟΥ",
        "ΚΑΘΗΓΗΤΕΣ Ε.Μ.Π",
        "ΟΙΚΟΝΟΜΙΚΗ ΔΙΟΙΚΗΣΗ",
        "ΒΕΒΑΙΩΣΗ ΦΟΡΟΛΟΓΙΑΣ ΚΑΘΑΡΑΣ ΠΡΟΣΟΔΟΥ",
        "ΤΑΜΕΙΟ ΣΥΝΤΑΞΕΩΝ ΠΡΟΣΩΠΙΚΟΥ ΤΡΑΠΕΖΩΝ ΕΛΛΑΔΟΣ ΚΑΙ ΚΤΗΜΑΤΙΚΗΣ",
        "ΔΗΜΟΨΗΦΙΣΜΑΤΑ",
        "ΕΛΛΗΝΙΚΟ ΑΝΟΙΚΤΟ ΠΑΝΕΠΙΣΤΗΜΙΟ",
        "ΚΑΛΛΙΤΕΧΝΙΚΟ ΕΠΑΓΓΕΛΜΑΤΙΚΟ ΕΠΙΜΕΛΗΤΗΡΙΟ",
        "ΑΝΟΙΚΟΔΟΜΗΣΙΣ",
        "ΔΑΣΙΚΟΣ ΚΩΔΙΚΑΣ",
        "ΚΑΝΟΝΙΣΜΟΣ ΠΥΡΟΣΒΕΣΤΙΚΩΝ ΜΕΣΩΝ ΤΩΝ ΠΛΟΙΩΝ",
        "ΔΙΦΘΕΡΙΤΙΔΑ",
        "ΒΙΒΛΙΑ ΚΑΙ ΦΟΡΟΛΟΓΙΚΑ ΣΤΟΙΧΕΙΑ",
        "ΕΛΕΓΧΟΣ ΕΞΑΓΟΜΕΝΩΝ ΕΛΑΙΩΝ",
        "ΕΠΙΔΟΜΑΤΑ ΟΙΚΟΓΕΝΕΙΩΝ ΣΤΡΑΤΙΩΤΙΚΩΝ",
        "ΕΥΡΩΠΑΙΚΕΣ ΣΥΜΦΩΝΙΕΣ ΠΟΥ ΑΦΟΡΟΥΝ ΤΗΝ ΤΗΛΕΟΡΑΣΗ",
        "ΕΚΤΑΚΤΑ ΣΤΡΑΤΟΔΙΚΕΙΑ",
        "ΠΟΛΕΜΙΚΗ ΒΙΟΜΗΧΑΝΙΑ",
        "ΑΣΕΜΝΟΙ ΓΥΝΑΙΚΕΣ",
        "ΑΠΕΛΕΥΘΕΡΩΣΗ ΑΓΟΡΑΣ ΗΛΕΚΤΡΙΚΗΣ ΕΝΕΡΓΕΙΑΣ ΕΝΕΡΓΕΙΑΚΗ ΠΟΛΙΤΙΚΗ Ρ.Α.Ε",
        "ΠΡΟΕΙΣΠΡΑΞΗ ΔΙΚΗΓΟΡΙΚΗΣ ΑΜΟΙΒΗΣ",
        "ΕΘΝΙΚΗ ΣΧΟΛΗ ΔΗΜΟΣΙΑΣ ΥΓΕΙΑΣ (Ε.Σ.Δ.Υ.)",
        "ΠΡΟΜΗΘΕΙΑ ΘΕΙΟΥ ΚΑΙ ΘΕΙΙΚΟΥ ΧΑΛΚΟΥ",
        "ΧΗΜΙΚΟΙ - ΧΗΜΙΚΕΣ ΒΙΟΜΗΧΑΝΙΕΣ",
        "ΑΣΦΑΛΙΣΗ ΚΑΤΑ ΤΗΣ ΑΣΘΕΝΕΙΑΣ",
        "ΤΑΜΕΙΟ ΑΛΛΗΛΟΒΟΗΘΕΙΑΣ ΠΡΟΣΩΠΙΚΟΥ ΕΘΝΙΚΟΥ ΤΥΠΟΓΡΑΦΕΙΟΥ (Τ.Α.Π.Ε.Τ.)",
        "ΟΡΓΑΝΙΣΜΟΣ ΥΠΟΥΡΓΕΙΟΥ ΟΙΚΟΝΟΜΙΚΩΝ",
        "ΠΕΡΙΕΧΟΜΕΝΟ ΔΗΛΩΣΗΣ ΦΟΡΟΥ ΕΙΣΟΔΗΜΑΤΟΣ",
        "ΠΡΩΤΕΣ ΥΛΕΣ ΣΙΔΕΡΕΝΙΩΝ ΒΑΡΕΛΙΩΝ",
        "ΕΥΡΩΠΑΙΚΟΣ ΚΩΔΙΚΑΣ ΚΟΙΝΩΝΙΚΗΣ ΑΣΦΑΛΕΙΑΣ",
        "ΔΙΑΦΟΡΟΙ ΓΕΩΡΓΙΚΟΙ ΣΥΝΕΤΑΙΡΙΣΜΟΙ",
        "ΣΧΕΔΙΑ ΠΟΛΕΩΝ ΙΟΝΙΩΝ ΝΗΣΩΝ",
        "ΕΥΡΩΠΑΙΚΗ ΟΙΚΟΝΟΜΙΚΗ ΚΟΙΝΟΤΗΤΑ ΕΥΡΩΠΑΙΚΗ ΕΝΩΣΗ",
        "ΣΧΟΛΗ ΔΙΟΙΚΗΣΕΩΣ ΝΟΣΗΛΕΥΤ. ΙΔΡΥΜΑΤΩΝ",
        "ΔΙΑΦΟΡΟΙ ΝΟΜΟΙ ΕΜΠΡΑΓΜΑΤΟΥ ΔΙΚΑΙΟΥ",
        "ΕΠΙΜΕΛΗΤΕΙΑ ΚΑΙ ΟΙΚΟΝΟΜΙΚΕΣ ΥΠΗΡΕΣΙΕΣ",
        "ΔΙΑΔΙΚΑΣΙΑ ΑΤΕΛΕΙΑΣ",
        "ΠΑΙΔΙΚΕΣ ΕΞΟΧΕΣ",
        "ΤΑΜΕΙΟ ΣΥΝΤΑΞΕΩΝ ΠΡΟΣΩΠΙΚΟΥ ΕΘΝΙΚΗΣ ΤΡΑΠΕΖΑΣ ΤΗΣ ΕΛΛΑΔΟΣ",
        "ΚΡΑΤΙΚΗ ΕΚΜΕΤΑΛΛΕΥΣΗ ΔΑΣΩΝ",
        "ΑΝΕΞΑΡΤΗΣΙΑ ΤΗΣ ΕΚΚΛΗΣΙΑΣ ΤΗΣ ΕΛΛΑΔΟΣ",
        "ΤΕΧΝΙΚΑ ΠΤΥΧΙΑ",
        "ΕΠΙΒΑΤΙΚΑ ΑΥΤΟΚΙΝΗΤΑ (ΔΗΜΟΣΙΑΣ ΚΑΙ ΙΔΙΩΤΙΚΗΣ ΧΡΗΣΗΣ)",
        "ΣΥΜΒΑΣΕΙΣ ΒΟΥΛΕΥΤΩΝ",
        "ΟΡΓΑΝΙΣΜΟΣ ΤΩΝ ΔΙΚΑΣΤΗΡΙΩΝ",
        "ΕΚΠΑΙΔΕΥΤΙΚΟΙ ΛΕΙΤΟΥΡΓΟΙ ΕΝ ΓΕΝΕΙ",
        "ΑΡΜΟΔΙΟΤΗΤΑ ΤΕΛΩΝΕΙΑΚΩΝ ΑΡΧΩΝ",
        "ΕΙΔΙΚΑ ΕΦΕΤΕΙΑ",
        "ΑΞΙΩΜΑΤΙΚΟΙ ΑΕΡΟΠΟΡΙΑΣ",
        "ΠΑΝΕΠΙΣΤΗΜΙΑΚΗ ΒΙΒΛΙΟΘΗΚΗ",
        "ΕΠΙΤΡΟΠΗ ΣΥΝΤΑΞΗΣ ΣΧΕΔΙΟΥ ΚΩΔΙΚΑ ΕΡΓΑΣΙΑΣ",
        "ΕΛΟΝΟΣΙΑ",
        "ΝΑΥΛΟΣΥΜΦΩΝΑ",
        "ΣΙΔΗΡΟΔΡΟΜΟΙ ΘΕΣΣΑΛΙΚΟΙ",
        "ΡΑΔΙΟΦΩΝΙΚΕΣ ΣΥΜΒΑΣΕΙΣ",
        "ΠΡΟΩΘΗΣΗ ΓΕΩΡΓΙΚΗΣ ΠΑΡΑΓΩΓΗΣ-ΕΘ.Ι.ΑΓ.Ε",
        "ΕΠΟΧΙΑΚΩΣ ΕΡΓΑΖΟΜΕΝΟΙ ΜΙΣΘΩΤΟΙ",
        "ΔΙΔΑΚΤΙΚΟ ΠΡΟΣΩΠΙΚΟ",
        "ΚΩΔΙΚΑΣ ΚΕΝΤΡΙΚΗΣ, ΠΡΕΣΒΕΥΤΙΚΗΣ ΚΑΙ",
        "ΠΟΛΙΤΙΚΟ ΠΡΟΣΩΠΙΚΟ ΥΠΟΥΡΓΕΙΟΥ ΕΘΝΙΚΗΣ ΑΜΥΝΑΣ",
        "ΔΙΠΛΩΜΑΤΑ ΕΥΡΕΣΙΤΕΧΝΙΑΣ",
        "ΣΩΜΑΤΕΙΑ ΓΕΩΡΓΙΚΩΝ ΕΡΓΑΤΩΝ",
        "ΚΩΔΙΚΑΣ ΠΕΡΙ ΕΙΣΠΡΑΞΕΩΣ ΔΗΜΟΣΙΩΝ ΕΣΟΔΩΝ",
        "ΤΡΑΠΕΖΟΓΡΑΜΜΑΤΙΑ",
        "ΠΡΟΜΗΘΕΥΤΙΚΟΣ ΟΡΓΑΝΙΣΜΟΣ Ε.Β.Α",
        "ΕΛΕΓΧΟΣ ΑΣΦΑΛΕΙΑΣ ΑΥΤΟΚΙΝΗΤΩΝΚΕΝΤΡΑ ΤΕΧΝΙΚΟΥ ΕΛΕΓΧΟΥ ΟΧΗΜΑΤΩΝ (Κ.Τ.Ε.Ο.)",
        "ΕΞΑΓΩΓΗ ΤΥΡΟΥ",
        "ΝΑΥΤΙΛΙΑΚΟ ΣΥΝΑΛΛΑΓΜΑ",
        "ΤΑΜΕΙΟ ΕΠΙΚΟΥΡΙΚΗΣ ΑΣΦΑΛΙΣΕΩΣ ΗΛΕΤΡΟΤΕΧΝΙΤΩΝ ΕΛΛΑΔΟΣ (T.E.A.H.E.)",
        "ΜΙΣΘΟΙ ΣΤΡΑΤΙΩΤΙΚΩΝ ΚΑΙ ΠΡΟΣΑΥΞΗΣΕΙΣ",
        "ΑΣΤΙΚΟΣ ΚΩΔΙΚΑΣ",
        "ΜΕ ΤΙΣ ΗΝΩΜΕΝΕΣ ΠΟΛΙΤΕΙΕΣ ΑΜΕΡΙΚΗΣ",
        "ΤΑΜΕΙΟ ΑΣΦΑΛΙΣΕΩΣ ΠΡΟΣΩΠΙΚΟΥ Ο.Τ.Ε. (Τ.Α.Π.-Ο.Τ.Ε.)",
        "ΜΑΙΕΣ",
        "ΦΥΓΟΔΙΚΙΑ",
        "ΟΡΓΑΝΙΣΜΟΣ ΞΕΝΟΔΟΧΕΙΑΚΗΣ ΠΙΣΤΗΣ",
        "ΔΗΜΟΤΙΚΟΙ ΣΤΡΑΤΟΛΟΓΟΙ",
        "ΑΝΩΤΑΤΟ ΔΙΚΑΣΤΙΚΟ ΣΥΜΒΟΥΛΙΟ",
        "ΙΣΤΟΡΙΚΟ ΑΡΧΕΙΟ ΚΡΗΤΗΣ",
        "ΕΛΛΗΝΙΚΗ ΘΑΛΑΣΣΙΑ ΄ΕΝΩΣΗ",
        "ΕΚΠΟΙΗΣΕΙΣ ΚΑΙ ΕΚΜΙΣΘΩΣΕΙΣ",
        "ΤΑΧΥΔΡΟΜΙΚΕΣ ΕΠΙΤΑΓΕΣ",
        "ΥΠΗΡΕΣΙΑ ΜΗΤΡΩΟΥ",
        "ΔΙΑΦΟΡΑ ΟΙΚΟΝΟΜΙΚΑ ΘΕΜΑΤΑ",
        "ΕΝΔΙΚΑ ΜΕΣΑ",
        "ΤΕΛΗ ΑΕΡΟΠΟΡΙΚΩΝ ΤΑΞΙΔΙΩΝ",
        "ΜΕ ΤΗΝ ΑΙΓΥΠΤΟ",
        "ΔΙΑΦΟΡΕΣ ΒΙΒΛΙΟΘΗΚΕΣ",
        "ΚΕΝΤΡΙΚΗ ΥΠΗΡΕΣΙΑ",
    ],
}

_HOMEPAGE = "https://doi.org/10.5281/zenodo.5528002"

_LICENSE = "CC BY-SA (Creative Commons / Attribution-ShareAlike)"

URL = "https://zenodo.org/record/5528002/files/greek_legislation.tar.xz"


class GreekLegalCodeConfig(datasets.BuilderConfig):
    """BuilderConfig for Greek_Legal_Code."""

    def __init__(self, label_type="volume", **kwargs):
        """BuilderConfig for Greek_Legal_Code.

        Args:
        label_type: One of following: [volume, chapter, subject]
          **kwargs: keyword arguments forwarded to super.
        """
        super(GreekLegalCodeConfig, self).__init__(**kwargs)
        self.label_type = label_type
        self.label_names = _LABEL_NAMES[self.label_type]


class GreekLegalCode(datasets.GeneratorBasedBuilder):
    """\
    Greek_Legal_Code contains 47k classified legal resources from Greek Legislation. Its origin is “Permanent Greek Legislation Code - Raptarchis”,
    a collection of Greek legislative documents classified into multi-level (from broader to more specialized) categories.
    """

    VERSION = datasets.Version("1.0.0")

    BUILDER_CONFIGS = [
        GreekLegalCodeConfig(
            name=label_type,
            label_type=label_type,
            version=datasets.Version("1.0.0", ""),
            description=f"Plain text import of Greek_Legal_Code for the {label_type} level",
        )
        for label_type in ["volume", "chapter", "subject"]
    ]

    DEFAULT_CONFIG_NAME = "volume"

    def _info(self):
        features = datasets.Features(
            {"text": datasets.Value("string"), "label": datasets.ClassLabel(names=self.config.label_names)}
        )
        return datasets.DatasetInfo(
            # This is the description that will appear on the datasets page.
            description=_DESCRIPTION,
            # This defines the different columns of the dataset and their types
            features=features,  # Here we define them above because they are different between the two configurations
            # If there's a common (input, target) tuple from the features,
            # specify them here. They'll be used if as_supervised=True in
            # builder.as_dataset.
            supervised_keys=None,
            # Homepage of the dataset for documentation
            homepage=_HOMEPAGE,
            # License for the dataset if available
            license=_LICENSE,
            # Citation for the dataset
            citation=_CITATION,
            # Task template
            task_templates=[datasets.tasks.TextClassification(text_column="text", label_column="label")],
        )

    def _split_generators(self, dl_manager):
        """Returns SplitGenerators."""
        data_dir = dl_manager.download_and_extract(URL)
        return [
            datasets.SplitGenerator(
                name=datasets.Split.TRAIN,
                # These kwargs will be passed to _generate_examples
                gen_kwargs={
                    "filepath": os.path.join(data_dir, "train.jsonl"),
                    "split": "train",
                },
            ),
            datasets.SplitGenerator(
                name=datasets.Split.TEST,
                # These kwargs will be passed to _generate_examples
                gen_kwargs={"filepath": os.path.join(data_dir, "test.jsonl"), "split": "test"},
            ),
            datasets.SplitGenerator(
                name=datasets.Split.VALIDATION,
                # These kwargs will be passed to _generate_examples
                gen_kwargs={
                    "filepath": os.path.join(data_dir, "dev.jsonl"),
                    "split": "dev",
                },
            ),
        ]

    def _generate_examples(
        self, filepath, split  # method parameters are unpacked from `gen_kwargs` as given in `_split_generators`
    ):
        """Yields examples as (key, example) tuples."""

        with open(filepath, encoding="utf-8") as f:
            for id_, row in enumerate(f):
                data = json.loads(row)
                yield id_, {
                    "text": data["text"],
                    "label": data[self.config.label_type],
                }
