####Script for RDA analysis of diatoms and insects####


# Prepared by Fabiana Schneck (fabiana.schneck@gmail.com) in March/2021
# Revised by FS in October/2021
#===============================================================================.

#loading required packages
library(vegan)
library(usdm) 
library(adespatial)
library(ggplot2)
library(ggrepel)
library(cowplot)

#install ggvegan for RDA plots
install.packages("remotes")
remotes::install_github("gavinsimpson/ggvegan")
library(ggvegan)

#loading data from scripts 1 and 2
source("script_1-Regression_models_Diatoms.R")
source("script_2-Regression_models_Insects.R")



####Diatoms####
diat #biological matrix
env.s.diat #instream standardized data
land.diat #land-use data

pred.diat<-data.frame(env.s.diat,forest=decostand(land.diat$Forest,method="standardize"))

#Variable boulder excluded bc. substrates sum 1; all VIF<3
vifstep(pred.diat[,-8]) 

#Hellinger transformation on abundance data 
diat.hell<-decostand(diat,"hellinger")

#RDA - global test
rda.diat<-rda(diat.hell~.,data=pred.diat[,-8])
set.seed(1)
anova(rda.diat)
Radj.diat<-RsquareAdj(rda.diat)$adj.r.squared

#Variable selection (Blanchet et al. 2008)
set.seed(1)
forward.sel(diat.hell,pred.diat[,-8],adjR2thresh=Radj.diat)

#RDA reduced model 
rda.diat.simplify<-rda(diat.hell~cond+cover_site+forest+pH+depth_cm+nitro,data=pred.diat)

set.seed(1)
anova(rda.diat.simplify)

RsquareAdj(rda.diat.simplify)$adj.r.squared

head(summary(rda.diat.simplify))


####Figure S7 - RDA diatoms####
My_Theme <- theme(
  axis.title.x = element_text(size = 16,colour="black"),
  axis.text.x = element_text(size = 14,colour="black"),
  axis.title.y = element_text(size = 16,colour="black"),
  axis.text.y = element_text(size = 14,colour="black"),
  legend.text = element_text(size = 14,colour="black"),
  legend.title = element_text(size = 14,colour="black"))



data.rda.diat<-fortify(rda.diat.simplify,axes=1:2)

#Selecting only species scores
score.sp.diat<-subset(data.rda.diat,Score=="species") 

#Ordering by species names
score.sp.diat<-score.sp.diat[order(score.sp.diat$Label),] 

#Adding SCBD values
score.sp.diat<-data.frame(score.sp.diat,SCBD=SCBD.diat$SCBD) 

#Checking
all.equal(score.sp.diat$Label,rownames(SCBD.diat))

#Change labels for plotted species
score.sp.diat$Label[score.sp.diat$Label=="Achnanthidium_modestiforme"]<-"Acmo"
score.sp.diat$Label[score.sp.diat$Label=="Nupela_sp6"]<-"Nu6"
score.sp.diat$Label[score.sp.diat$Label=="Nupela_sp4_BR14.1"]<-"Nu4"
score.sp.diat$Label[score.sp.diat$Label=="Karayeva_oblongela"]<-"Kaob"
score.sp.diat$Label[score.sp.diat$Label=="Eunotia_botulitropica"]<-"Eubo"
score.sp.diat$Label[score.sp.diat$Label=="Eunotia_pseudosudetica"]<-"Eusu"
score.sp.diat$Label[score.sp.diat$Label=="Eunotia_pseudoimplicata"]<-"Euim"
score.sp.diat$Label[score.sp.diat$Label=="Encyonema_sp12"]<-"En12"
score.sp.diat$Label[score.sp.diat$Label=="Sellaphora_nigri"]<-"Seni"
score.sp.diat$Label[score.sp.diat$Label=="Platessa_hustedii"]<-"Plhu"

#Scale vector arrows to fill proportion of the data
take <- c('RDA1', 'RDA2')  # which columns contain the scores we want
arrows.diat <- subset(data.rda.diat, Score == 'biplot')  

#Multiplier for arrows to scale them to the plot range
mul.diat<-ggvegan:::arrowMul(arrows.diat[, take],subset(data.rda.diat, select = take, Score == 'sites'))
arrows.diat[, take] <- arrows.diat[, take] * mul.diat  # scale biplot arrows

#Change labels for plotted variables
arrows.diat$Label
arrows.diat$Label[2]<-"shading"
arrows.diat$Label[5]<-"depth"
arrows.diat$Label[6]<-"N"


#Visualizing SCBD values in order
SCBD.diat[order(SCBD.diat$SCBD,decreasing=T),][1:11,1]


plot.rda.diat<-  ggplot() +
  geom_point(data=subset(score.sp.diat,SCBD<0.022354), #plot all other species
             aes(x=RDA1,y=RDA2),shape=3,size=2,colour="orange",alpha=0.6)+
  geom_point(data=subset(score.sp.diat,SCBD>=0.022354), #plot species with highest SCBD
               aes(x=RDA1,y=RDA2),shape=3,size=2,colour="darkorange",stroke=1.5)+
  #geom_point(data=subset(data.rda.diat, Score == "sites"),aes(x=RDA1,y=RDA2),size=1,colour="grey",alpha=0.5)+
  geom_segment(data = arrows.diat,aes(x = 0, y = 0, xend = RDA1, yend = RDA2), 
               arrow = arrow(length = unit(0.01, "npc")),colour="black")+
  geom_text(data = arrows.diat,aes(x=RDA1*1.1, y=RDA2*1.1,label=Label,size=0.025))+
  geom_text_repel(data=subset(score.sp.diat,SCBD>=0.022354), #plot 10 spp with higher SCBD values
                  aes(x=RDA1,y=RDA2,label=Label,size=SCBD),colour="orange",seed=TRUE)+
  scale_size_continuous(range = c(5, 10))+
  coord_fixed()+
  geom_vline(xintercept = 0, linetype = 'dotted',alpha=0.4)+
  geom_hline(yintercept = 0, linetype = 'dotted',alpha=0.4)+
  xlab("RDA1 (6.8%)")+
  ylab("RDA2 (4.2%)")+
  theme_bw()+      
  theme(panel.grid=element_blank())+
  My_Theme+
  theme(legend.position = "none")


ggsave("ESM_FigS7.png",plot.rda.diat,height = 25, width =25,units="cm",dpi=600)



####Insects####
inv #biological matrix
env.s.inv #instream standardized data
land.inv #land-use data


pred.inv<-data.frame(env.s.inv,forest=decostand(land.inv$Forest,method="standardize"))
vifstep(pred.inv[,-c(8)], th=3) #boulder excluded because substrates sum 1; all VIF < 3

#RDA global model
inv.hell<-decostand(inv,method="hellinger")
rda.inv<-rda(inv.hell~.,data=pred.inv[,-8])
set.seed(11)
anova(rda.inv) #significant global model
Radj.inv<-RsquareAdj(rda.inv)$adj.r.squared
set.seed(11)
forward.sel(inv.hell,pred.inv[,-8],adjR2thresh=Radj.inv)

#rda using only selected predictors
rda.inv.simplify<-rda(inv.hell~pH+depth_cm+forest+cobble+width_m+mud,data=pred.inv)
set.seed(11)
anova(rda.inv.simplify) #significance
RsquareAdj(rda.inv.simplify)$adj.r.squared
head(summary(rda.inv))

####Figure S8 - RDA insects####
data.rda.inv<-fortify(rda.inv.simplify,axes=1:2)

score.sp.inv<-subset(data.rda.inv,Score=="species")#selecting only species scores
score.sp.inv<-score.sp.inv[order(score.sp.inv$Label),] #ordering by species names
score.sp.inv<-data.frame(score.sp.inv,SCBD=SCBD.inv$SCBD) #adding SCBD values
all.equal(score.sp.inv$Label,rownames(SCBD.inv))

score.sp.inv$Label[score.sp.inv$Label=="Anacroneuria"]<-"Anac"
score.sp.inv$Label[score.sp.inv$Label=="Callibaetis"]<-"Call"
score.sp.inv$Label[score.sp.inv$Label=="Cloeodes"]<-"Cloe"
score.sp.inv$Label[score.sp.inv$Label=="Farrodes"]<-"Farr"
score.sp.inv$Label[score.sp.inv$Label=="Gripopteryx"]<-"Grip"
score.sp.inv$Label[score.sp.inv$Label=="Heterelmis"]<-"Hete"
score.sp.inv$Label[score.sp.inv$Label=="Hexacylloepus"]<-"Hexa"
score.sp.inv$Label[score.sp.inv$Label=="Leptonema"]<-"Lept"
score.sp.inv$Label[score.sp.inv$Label=="Smicridea"]<-"Smic"
score.sp.inv$Label[score.sp.inv$Label=="Xenelmis"]<-"Xene"

#Scale vector arrows to fill proportion of the data
take <- c('RDA1', 'RDA2')  # which columns contain the scores we want
arrows <- subset(data.rda.inv, Score == 'biplot')  

## multiplier for arrows to scale them to the plot range
mul<-ggvegan:::arrowMul(arrows[, take],subset(data.rda.inv, select = take, Score == 'sites'))
arrows[, take] <- arrows[, take] * mul  # scale biplot arrows
arrows$Label
arrows$Label[2]<-"depth"
arrows$Label[5]<-"width"


#visualizing SCBV values in order
SCBD.inv[order(SCBD.inv$SCBD,decreasing=T),][1:11,]

plot.rda.inv<- ggplot() +
  geom_point(data=subset(score.sp.inv,SCBD<0.0297),
             aes(x=RDA1,y=RDA2),shape=3,size=2,colour="steelblue",alpha=0.6)+
  geom_point(data=subset(score.sp.inv,SCBD>=0.0297),
             aes(x=RDA1,y=RDA2),shape=3,size=2,stroke=1.5,colour="steelblue")+
  #geom_point(data=subset(data.rda.inv, Score == "sites"),aes(x=RDA1,y=RDA2),size=2,colour="black",alpha=0.5)+
  geom_segment(data = arrows,aes(x = 0, y = 0, xend = RDA1, yend = RDA2), 
               arrow = arrow(length = unit(0.01, "npc")),colour="black")+
  geom_text(data = arrows,aes(x=RDA1*1.1, y=RDA2*1.1,label=Label,size=0.03))+
  geom_text_repel(data=subset(score.sp.inv,SCBD>=0.0297),
                  aes(x=RDA1,y=RDA2,label=Label,size=SCBD),colour="steelblue",seed=TRUE)+
  scale_size_continuous(range = c(5, 10))+
  coord_fixed()+
  geom_vline(xintercept = 0, linetype = 'dotted',alpha=0.4)+
  geom_hline(yintercept = 0, linetype = 'dotted',alpha=0.4)+
  xlab("RDA1 (5.4%)")+
  ylab("RDA2 (4.6%)")+
  theme_bw()+      
  theme(panel.grid=element_blank())+
  My_Theme+
  theme(legend.position = "none")


ggsave("ESM_FigS8.png",plot.rda.inv,height = 25, width =25,units="cm",dpi=600)




