# R-script with all analyses for:

####################################################################################
# Hampus Petrn, Gabriele Gloder, Diana Posledovich, Christer Wiklund, Magne Friberg

# Innate preference hierarchies coupled with adult experience,
# rather than larval imprinting or transgenerational acclimation,
# determine host plant use in Pieris rapae

# Ecology and Evolution
####################################################################################



# R packages ############

library(car)
library(effects)
library(nlme)




# Data #############

setwd("")
getwd()


leafs <- read.csv("Petren_et_al_leaf_nitrogen.csv") # Nitrogen data
pref <- read.csv("Petren_et_al_preference.csv") # Preference data
perf <- read.csv("Petren_et_al_performance.csv") # Performance data



# Leaf nitrogen analyses #################

leveneTest((PropNitrogen) ~ Cohort * PlantSpecies, data=leafs, center="mean") # No normal distirubtion
leveneTest(log10(PropNitrogen) ~ Cohort * PlantSpecies, data=leafs, center="mean") # Ok

mod1 <- lm(PropNitrogen~Cohort * PlantSpecies, data=leafs) # For figure

mod2 <- lm(log10(PropNitrogen)~Cohort * PlantSpecies, data=leafs)
Anova(mod2) # Anova for leaf nitrogen

# Mean and CI for Figure S1
ef1 <- effect("Cohort:PlantSpecies", mod1)
summary(ef1)

# Mean and SE for proportion nitrogen
tapply(leafs$PropNitrogen, leafs$Plant, mean)
tapply(leafs$PropNitrogen, leafs$Plant, FUN = function(x) sd(x)/sqrt(length(x)))

# Sample sizes
table(leafs$Plant)





# Oviposition generation 1 #############
# Testing effect of female larval host, male larval host and
# no-choice host on female oviposition acceptance and preference in generation 1.

# Acceptance ###
# Table S1 and Results

mod3a <- lm(EggsNoChoiceHost~MaleLarvHost+FemaleLarvHost*NoChoiceHost,
            data=pref[pref$Generation == "1",])
Anova(mod3a, Type="II", white.adjust=TRUE)

mod3b <- lm(EggsNoChoiceHost~MaleLarvHost+FemaleLarvHost+NoChoiceHost,
            data=pref[pref$Generation == "1",])
Anova(mod3b, Type="II", white.adjust=TRUE)

mod3c <- lm(EggsNoChoiceHost~FemaleLarvHost*NoChoiceHost,
            data=pref[pref$Generation == "1",])
Anova(mod3c, Type="II", white.adjust=TRUE)

mod3d <- lm(EggsNoChoiceHost~FemaleLarvHost+NoChoiceHost,
            data=pref[pref$Generation == "1",])
Anova(mod3d, Type="II", white.adjust=TRUE)

mod3e <- lm(EggsNoChoiceHost~NoChoiceHost,
            data=pref[pref$Generation == "1",])
Anova(mod3e, Type="II", white.adjust=TRUE)

mod3f <- lm(EggsNoChoiceHost~1,
            data=pref[pref$Generation == "1",])


AIC(mod3a)
AIC(mod3b)
AIC(mod3c)
AIC(mod3d)
AIC(mod3e)
AIC(mod3f)


# Mean and CI for Figure 1a
ef2 <- effect("FemaleLarvHost:NoChoiceHost", mod3c)
summary(ef2)


# Sample size Figure 1a
table(pref[pref$Generation == "1",]$FemaleLarvHost,
      pref[pref$Generation == "1",]$NoChoiceHost) 




# Binary preference ###
# Table S3 and Results

mod4a <- glm(EggsPrefBerteroaBinary ~ MaleLarvHost+FemaleLarvHost*NoChoiceHost, 
           family=binomial(logit), data=pref[pref$Generation == "1",])
Anova(mod4a)

mod4b <- glm(EggsPrefBerteroaBinary ~ MaleLarvHost+FemaleLarvHost+NoChoiceHost, 
            family=binomial(logit), data=pref[pref$Generation == "1",])
Anova(mod4b)

mod4c <- glm(EggsPrefBerteroaBinary ~ FemaleLarvHost*NoChoiceHost, 
             family=binomial(logit), data=pref[pref$Generation == "1",])
Anova(mod4c)

mod4d <- glm(EggsPrefBerteroaBinary ~ FemaleLarvHost+NoChoiceHost, 
             family=binomial(logit), data=pref[pref$Generation == "1",])
Anova(mod4d)

mod4e <- glm(EggsPrefBerteroaBinary ~ FemaleLarvHost, 
             family=binomial(logit), data=pref[pref$Generation == "1",])
Anova(mod4e)

mod4f <- glm(EggsPrefBerteroaBinary ~ NoChoiceHost, 
             family=binomial(logit), data=pref[pref$Generation == "1",])
Anova(mod4f)

mod4g <- glm(EggsPrefBerteroaBinary ~ 1, 
             family=binomial(logit), data=pref[pref$Generation == "1",])


AIC(mod4a)
AIC(mod4b)
AIC(mod4c)
AIC(mod4d)
AIC(mod4e)
AIC(mod4f)
AIC(mod4g)




# Percentage of females ovipositing on Berteroa incana 
# in preference experiment in generation 1
sum(pref[pref$Generation == "1",]$EggsPrefBerteroaBinary)/
  length(pref[pref$Generation == "1",]$EggsPrefBerteroaBinary)


# Table 1
table(pref[pref$Generation == "1",]$EggsPrefBerteroaBinary, 
      pref[pref$Generation == "1",]$NoChoiceHost)




# Oviposition generation 2 #############
# Testing effect of female larval host, male larval host,
# no-choice host and maternal larval host on female oviposition
# acceptance and preference in generation 2.

# Acceptance ###
# Table S2 and Results

mod5a <- lm(EggsNoChoiceHost ~ MaleLarvHost+FemaleLarvHost+MaternalLarvHost*NoChoiceHost,
            data = pref[pref$Generation == "2",])
Anova(mod5a, Type="II", white.adjust=TRUE)
mod5b <- lm(EggsNoChoiceHost ~ MaleLarvHost+MaternalLarvHost+FemaleLarvHost*NoChoiceHost,
            data = pref[pref$Generation == "2",])
Anova(mod5b, Type="II", white.adjust=TRUE)
mod5c <- lm(EggsNoChoiceHost ~ MaleLarvHost+MaternalLarvHost+FemaleLarvHost+NoChoiceHost,
            data = pref[pref$Generation == "2",])
Anova(mod5c, Type="II", white.adjust=TRUE)
mod5d <- lm(EggsNoChoiceHost ~ MaternalLarvHost+FemaleLarvHost+NoChoiceHost,
            data = pref[pref$Generation == "2",])
Anova(mod5d, Type="II", white.adjust=TRUE)
mod5e <- lm(EggsNoChoiceHost ~ FemaleLarvHost+MaternalLarvHost*NoChoiceHost,
            data = pref[pref$Generation == "2",])
Anova(mod5e, Type="II", white.adjust=TRUE)
mod5f <- lm(EggsNoChoiceHost ~ MaternalLarvHost+FemaleLarvHost*NoChoiceHost,
            data = pref[pref$Generation == "2",])
Anova(mod5f, Type="II", white.adjust=TRUE)
mod5g <- lm(EggsNoChoiceHost ~ NoChoiceHost,
            data = pref[pref$Generation == "2",])
Anova(mod5g, Type="II", white.adjust=TRUE)
mod5h <- lm(EggsNoChoiceHost ~ 1,
            data = pref[pref$Generation == "2",])

AIC(mod5a)
AIC(mod5b)
AIC(mod5c)
AIC(mod5d)
AIC(mod5e)
AIC(mod5f)
AIC(mod5g)
AIC(mod5h)


# Mean and CI for Figure 1c
ef3 <- effect("MaternalLarvHost:FemaleLarvHost:NoChoiceHost", mod5f)
summary(ef3)


# Figure 1c sample size
table(pref[pref$Generation == "2",]$MaternalLarvHost,
      pref[pref$Generation == "2",]$FemaleLarvHost,
      pref[pref$Generation == "2",]$NoChoiceHost)



# Binary preference ###
# Table S4 and Results

mod6a <- glm(EggsPrefBerteroaBinary ~ NoChoiceHost+MaleLarvHost+MaternalLarvHost*FemaleLarvHost, 
            family=binomial(logit), data=pref[pref$Generation == "2",])
Anova(mod6a)

mod6b <- glm(EggsPrefBerteroaBinary ~ MaleLarvHost+FemaleLarvHost+MaternalLarvHost*NoChoiceHost, 
             family=binomial(logit), data=pref[pref$Generation == "2",])
Anova(mod6b)

mod6c <- glm(EggsPrefBerteroaBinary ~ MaleLarvHost+MaternalLarvHost+FemaleLarvHost*NoChoiceHost, 
             family=binomial(logit), data=pref[pref$Generation == "2",])
Anova(mod6c)

mod6d <- glm(EggsPrefBerteroaBinary ~ MaleLarvHost+MaternalLarvHost +FemaleLarvHost+NoChoiceHost, 
             family=binomial(logit), data=pref[pref$Generation == "2",])
Anova(mod6d)

mod6e <- glm(EggsPrefBerteroaBinary ~ MaternalLarvHost+FemaleLarvHost+NoChoiceHost, 
             family=binomial(logit), data=pref[pref$Generation == "2",])
Anova(mod6e)

mod6f <- glm(EggsPrefBerteroaBinary ~ MaternalLarvHost, 
             family=binomial(logit), data=pref[pref$Generation == "2",])
Anova(mod6f)

mod6g <- glm(EggsPrefBerteroaBinary ~ FemaleLarvHost, 
             family=binomial(logit), data=pref[pref$Generation == "2",])
Anova(mod6g)

mod6h <- glm(EggsPrefBerteroaBinary ~ NoChoiceHost, 
             family=binomial(logit), data=pref[pref$Generation == "2",])
Anova(mod6h)

mod6i <- glm(EggsPrefBerteroaBinary ~ 1, 
             family=binomial(logit), data=pref[pref$Generation == "2",])


AIC(mod6a)
AIC(mod6b)
AIC(mod6c)
AIC(mod6d)
AIC(mod6e)
AIC(mod6f)
AIC(mod6g)
AIC(mod6h)
AIC(mod6i)



# Percentage of females ovipositing on Berteroa incana 
# in preference experiment in generation 2
sum(pref[pref$Generation == "2",]$EggsPrefBerteroaBinary)/
  length(pref[pref$Generation == "2",]$EggsPrefBerteroaBinary)



# Table 1
table(pref[pref$Generation == "2",]$EggsPrefBerteroaBinary,
      pref[pref$Generation == "2",]$NoChoiceHost)





# Larval performance gen 2 ####################
# Testing effect of sex, maternal host and larval host on
# development time, adult weight and growth rate.

mod7 <- lme(DevelopmentTime ~ Sex*MaternalHost*LarvalHost, random=~1|MotherID,
           weights=varIdent(form=~1|Sex*MaternalHost*LarvalHost),
           data=perf)
Anova(mod7) # No effect of sex
mod8 <- lme(DevelopmentTime ~ MaternalHost*LarvalHost, random=~1|MotherID,
           weights=varIdent(form=~1|MaternalHost*LarvalHost),
           data=perf)
Anova(mod8) # Table 2
summary(effect("MaternalHost:LarvalHost", mod8)) # Figure 2a



mod9 <- lme(AdultWeight ~ Sex*MaternalHost*LarvalHost, random=~1|MotherID,
            weights=varIdent(form=~1|Sex*MaternalHost*LarvalHost),
            data=perf)
Anova(mod9) # No effect of sex
mod10 <- lme(AdultWeight ~ MaternalHost*LarvalHost, random=~1|MotherID,
            weights=varIdent(form=~1|MaternalHost*LarvalHost),
            data=perf)
Anova(mod10) # Table 2
summary(effect("MaternalHost:LarvalHost", mod10)) # Figure 2b



mod11 <- lme(GrowthRate ~ Sex*MaternalHost*LarvalHost, random=~1|MotherID,
           weights=varIdent(form=~1|Sex*MaternalHost*LarvalHost),
           data=perf)
Anova(mod11) # No effect of sex
mod12 <- lme(GrowthRate ~ MaternalHost*LarvalHost, random=~1|MotherID,
            weights=varIdent(form=~1|MaternalHost*LarvalHost),
            data=perf)
Anova(mod12) # Table 2
summary(effect("MaternalHost:LarvalHost", mod12)) # Figure 2c

