package backend.analysis.machineLearning;

import backend.analysis.CodeChunksLinker;
import backend.analysis.GerritExtractor;
import backend.analysis.ImportFilter;
import backend.base.Change;
import backend.base.gerrit_data.CodeChunk;
import backend.base.gerrit_data.FeaturesList;
import backend.base.gerrit_data.Modification;
import backend.base.taxonomy.Type;
import backend.db_connectors.Dataset;
import backend.db_connectors.FilesIndexer;
import backend.main.CodeChunksExtractor;
import backend.support.ComparableModification;
import backend.support.ModificationDate;
import com.google.common.collect.Multimap;
import com.opencsv.CSVReader;
import com.opencsv.CSVWriter;
import org.junit.Test;
import weka.core.Instances;

import java.io.*;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.*;

public class ClassifierTest {

    private static Multimap<String, Change> changes;


    /*
     * Leave-one-out Android
     * All tests filter imports statements
     */


    @Test
    public void androidClassifyCategory(){
        List<Modification> trainingModifications = new ArrayList<>();
        trainingModifications.addAll(init(FilesIndexer.JGitDatasetPath, FilesIndexer.JGitGerritUrl, FilesIndexer.JGitFile, true));
        trainingModifications.addAll(init(FilesIndexer.JavaClientDatasetPath, FilesIndexer.JavaClientGerritUrl, FilesIndexer.JavaClientFile, true));
        List<Modification> testModifications = init(FilesIndexer.AndroidDatasetPath, FilesIndexer.AndroidGerritUrl, FilesIndexer.AndroidFile, true);
        classifyCategory(trainingModifications, testModifications);
    }

    @Test
    public void androidClassifyCategorySMOTE(){
        List<Modification> trainingModifications = new ArrayList<>();
        trainingModifications.addAll(init(FilesIndexer.JGitDatasetPath, FilesIndexer.JGitGerritUrl, FilesIndexer.JGitFile, true));
        trainingModifications.addAll(init(FilesIndexer.JavaClientDatasetPath, FilesIndexer.JavaClientGerritUrl, FilesIndexer.JavaClientFile, true));
        List<Modification> testModifications = init(FilesIndexer.AndroidDatasetPath, FilesIndexer.AndroidGerritUrl, FilesIndexer.AndroidFile, true);
        classifyCategoryWithSMOTE(trainingModifications, testModifications);
    }

    @Test
    public void androidClassifyGroup(){
        List<Modification> trainingModifications = new ArrayList<>();
        trainingModifications.addAll(init(FilesIndexer.JGitDatasetPath, FilesIndexer.JGitGerritUrl, FilesIndexer.JGitFile, true));
        trainingModifications.addAll(init(FilesIndexer.JavaClientDatasetPath, FilesIndexer.JavaClientGerritUrl, FilesIndexer.JavaClientFile, true));
        List<Modification> testModifications = init(FilesIndexer.AndroidDatasetPath, FilesIndexer.AndroidGerritUrl, FilesIndexer.AndroidFile, true);
        classifyGroup(trainingModifications, testModifications);
    }

    @Test
    public void androidClassifyGroupSMOTE(){
        List<Modification> trainingModifications = new ArrayList<>();
        trainingModifications.addAll(init(FilesIndexer.JGitDatasetPath, FilesIndexer.JGitGerritUrl, FilesIndexer.JGitFile, true));
        trainingModifications.addAll(init(FilesIndexer.JavaClientDatasetPath, FilesIndexer.JavaClientGerritUrl, FilesIndexer.JavaClientFile, true));
        List<Modification> testModifications = init(FilesIndexer.AndroidDatasetPath, FilesIndexer.AndroidGerritUrl, FilesIndexer.AndroidFile, true);
        classifyGroupWithSMOTE(trainingModifications, testModifications);
    }


    /*
     * Leave-one-out JGit
     * All tests filter imports statements
     */


    @Test
    public void JGitClassifyCategory(){
        List<Modification> trainingModifications = new ArrayList<>();
        trainingModifications.addAll(init(FilesIndexer.AndroidDatasetPath, FilesIndexer.AndroidGerritUrl, FilesIndexer.AndroidFile, true));
        trainingModifications.addAll(init(FilesIndexer.JavaClientDatasetPath, FilesIndexer.JavaClientGerritUrl, FilesIndexer.JavaClientFile, true));
        List<Modification> testModifications = init(FilesIndexer.JGitDatasetPath, FilesIndexer.JGitGerritUrl, FilesIndexer.JGitFile, true);
        classifyCategory(trainingModifications, testModifications);
    }

    @Test
    public void JGitClassifyCategorySMOTE(){
        List<Modification> trainingModifications = new ArrayList<>();
        trainingModifications.addAll(init(FilesIndexer.AndroidDatasetPath, FilesIndexer.AndroidGerritUrl, FilesIndexer.AndroidFile, true));
        trainingModifications.addAll(init(FilesIndexer.JavaClientDatasetPath, FilesIndexer.JavaClientGerritUrl, FilesIndexer.JavaClientFile, true));
        List<Modification> testModifications = init(FilesIndexer.JGitDatasetPath, FilesIndexer.JGitGerritUrl, FilesIndexer.JGitFile, true);
        classifyCategoryWithSMOTE(trainingModifications, testModifications);
    }

    @Test
    public void JGitClassifyGroup(){
        List<Modification> trainingModifications = new ArrayList<>();
        trainingModifications.addAll(init(FilesIndexer.AndroidDatasetPath, FilesIndexer.AndroidGerritUrl, FilesIndexer.AndroidFile, true));
        trainingModifications.addAll(init(FilesIndexer.JavaClientDatasetPath, FilesIndexer.JavaClientGerritUrl, FilesIndexer.JavaClientFile, true));
        List<Modification> testModifications = init(FilesIndexer.JGitDatasetPath, FilesIndexer.JGitGerritUrl, FilesIndexer.JGitFile, true);
        classifyGroup(trainingModifications, testModifications);
    }

    @Test
    public void JGitClassifyGroupSMOTE(){
        List<Modification> trainingModifications = new ArrayList<>();
        trainingModifications.addAll(init(FilesIndexer.AndroidDatasetPath, FilesIndexer.AndroidGerritUrl, FilesIndexer.AndroidFile, true));
        trainingModifications.addAll(init(FilesIndexer.JavaClientDatasetPath, FilesIndexer.JavaClientGerritUrl, FilesIndexer.JavaClientFile, true));
        List<Modification> testModifications = init(FilesIndexer.JGitDatasetPath, FilesIndexer.JGitGerritUrl, FilesIndexer.JGitFile, true);
        classifyGroupWithSMOTE(trainingModifications, testModifications);
    }


    /*
     * Leave-one-out Java-client
     * All tests filter imports statements
     */

    @Test
    public void JavaClientClassifyCategory(){
        List<Modification> trainingModifications = new ArrayList<>();
        trainingModifications.addAll(init(FilesIndexer.AndroidDatasetPath, FilesIndexer.AndroidGerritUrl, FilesIndexer.AndroidFile, true));
        trainingModifications.addAll(init(FilesIndexer.JGitDatasetPath, FilesIndexer.JGitGerritUrl, FilesIndexer.JGitFile, true));
        List<Modification> testModifications = init(FilesIndexer.JavaClientDatasetPath, FilesIndexer.JavaClientGerritUrl, FilesIndexer.JavaClientFile, true);
        classifyCategory(trainingModifications, testModifications);
    }

    @Test
    public void JavaClientClassifyCategorySMOTE(){
        List<Modification> trainingModifications = new ArrayList<>();
        trainingModifications.addAll(init(FilesIndexer.AndroidDatasetPath, FilesIndexer.AndroidGerritUrl, FilesIndexer.AndroidFile, true));
        trainingModifications.addAll(init(FilesIndexer.JGitDatasetPath, FilesIndexer.JGitGerritUrl, FilesIndexer.JGitFile, true));
        List<Modification> testModifications = init(FilesIndexer.JavaClientDatasetPath, FilesIndexer.JavaClientGerritUrl, FilesIndexer.JavaClientFile, true);
        classifyCategoryWithSMOTE(trainingModifications, testModifications);
    }

    @Test
    public void JavaClientClassifyGroup(){
        List<Modification> trainingModifications = new ArrayList<>();
        trainingModifications.addAll(init(FilesIndexer.AndroidDatasetPath, FilesIndexer.AndroidGerritUrl, FilesIndexer.AndroidFile, true));
        trainingModifications.addAll(init(FilesIndexer.JGitDatasetPath, FilesIndexer.JGitGerritUrl, FilesIndexer.JGitFile, true));
        List<Modification> testModifications = init(FilesIndexer.JavaClientDatasetPath, FilesIndexer.JavaClientGerritUrl, FilesIndexer.JavaClientFile, true);
        classifyGroup(trainingModifications, testModifications);
    }

    @Test
    public void JavaClientClassifyGroupSMOTE(){
        List<Modification> trainingModifications = new ArrayList<>();
        trainingModifications.addAll(init(FilesIndexer.AndroidDatasetPath, FilesIndexer.AndroidGerritUrl, FilesIndexer.AndroidFile, true));
        trainingModifications.addAll(init(FilesIndexer.JGitDatasetPath, FilesIndexer.JGitGerritUrl, FilesIndexer.JGitFile, true));
        List<Modification> testModifications = init(FilesIndexer.JavaClientDatasetPath, FilesIndexer.JavaClientGerritUrl, FilesIndexer.JavaClientFile, true);
        classifyGroupWithSMOTE(trainingModifications, testModifications);
    }

    /**
     * Tests with 10 Times 10-fold cross validation
     */

    @Test
    public void classifyCategory10Times(){
        List<Modification> modifications = extractModificationFromAllProjectsFilterImports();
        classifyCategory10Times(modifications);
    }

    @Test
    public void classifyCategory10TimesSMOTE(){
        List<Modification> modifications = extractModificationFromAllProjectsFilterImports();
        classifyCategory10TimesSMOTE(modifications);
    }

    @Test
    public void classifyCategory10TimesUndersamplingMajorityClass(){
        List<Modification> modifications = extractModificationFromAllProjectsFilterImports();
        classifyCategory10TimesUndersample(modifications);
    }

    @Test
    public void classifyGroup10Times(){
        List<Modification> modifications = extractModificationFromAllProjectsFilterImports();
        classifyGroup10Times(modifications);
    }

    @Test
    public void classifyGroup10TimesSMOTE(){
        List<Modification> modifications = extractModificationFromAllProjectsFilterImports();
        classifyGroup10TimesSMOTE(modifications);
    }

    /*
     *
     * === Leave one out in chronological order
     *
     */

    @Test
    public void AllClassifyCategoryTime(){
        List<Modification> allModifications = new ArrayList<>();
        allModifications.addAll(initExtractDate(FilesIndexer.AndroidDatasetPath, FilesIndexer.AndroidGerritUrl, FilesIndexer.AndroidFile, FilesIndexer.AndroidDatePath, true));
        allModifications.addAll(initExtractDate(FilesIndexer.JGitDatasetPath, FilesIndexer.JGitGerritUrl, FilesIndexer.JGitFile, FilesIndexer.JGitDatePath, true));
        allModifications.addAll(initExtractDate(FilesIndexer.JavaClientDatasetPath, FilesIndexer.JavaClientGerritUrl, FilesIndexer.JavaClientFile, FilesIndexer.JavaClientDatePath, true));
        List<Modification> orderedModifications = orderModifications(allModifications);
        int split = orderedModifications.size()/2;
        ClassifierLeaveOneOut classifier = new ClassifierLeaveOneOut();
        List<Modification> modificationsWithFeatures = new ModificationConverter().extractFeatures(orderedModifications, true);
        for(int index = 0; index < split-1; index++) {
            List<Modification> trainingModifications = new ArrayList<>();
            List<Modification> testModifications = new ArrayList<>();
            for (int i = 0; i<modificationsWithFeatures.size() - split + index; i++) {
                trainingModifications.add(modificationsWithFeatures.get(i));
            }
            testModifications.add(modificationsWithFeatures.get(modificationsWithFeatures.size() - split + index));
            System.out.println("===== training set size "+trainingModifications.size());
            System.out.println("===== test set size "+testModifications.size());
            classifyCategoryLeaveOneOut(trainingModifications, testModifications, classifier);
        }
        List<ClassResults> classResults = classifier.classResults;
        System.out.println("\n\n Evolvability: ===== \n\n");

        System.out.println("TP: "+classResults.get(0).truePositive);
        System.out.println("FP: "+classResults.get(0).falsePositive);
        System.out.println("FN: "+classResults.get(0).falseNegative);

        System.out.println("Precision: "+classResults.get(0).truePositive/(classResults.get(0).truePositive + classResults.get(0).falsePositive));
        System.out.println("Recall: "+classResults.get(0).truePositive/(classResults.get(0).truePositive + classResults.get(0).falseNegative));
        System.out.println("\n\n Functional: ===== \n\n");

        System.out.println("TP: "+classResults.get(1).truePositive);
        System.out.println("FP: "+classResults.get(1).falsePositive);
        System.out.println("FN: "+classResults.get(1).falseNegative);

        System.out.println("Precision: "+classResults.get(1).truePositive/(classResults.get(1).truePositive + classResults.get(1).falsePositive));
        System.out.println("Recall: "+classResults.get(1).truePositive/(classResults.get(1).truePositive + classResults.get(1).falseNegative));
    }


    @Test
    public void AllClassifyCategoryUnordered(){
        List<Modification> allModifications = new ArrayList<>();
        allModifications.addAll(initExtractDate(FilesIndexer.AndroidDatasetPath, FilesIndexer.AndroidGerritUrl, FilesIndexer.AndroidFile, FilesIndexer.AndroidDatePath, true));
        allModifications.addAll(initExtractDate(FilesIndexer.JGitDatasetPath, FilesIndexer.JGitGerritUrl, FilesIndexer.JGitFile, FilesIndexer.JGitDatePath, true));
        allModifications.addAll(initExtractDate(FilesIndexer.JavaClientDatasetPath, FilesIndexer.JavaClientGerritUrl, FilesIndexer.JavaClientFile, FilesIndexer.JavaClientDatePath, true));
        ClassifierLeaveOneOut classifier = new ClassifierLeaveOneOut();
        List<Modification> modificationsWithFeatures = new ModificationConverter().extractFeatures(allModifications, true);
        for(int index = 0; index < allModifications.size(); index++) {
            System.out.println("\n\n Progress: " +index + "\n\n");
            List<Modification> trainingModifications = new ArrayList<>();
            List<Modification> testModifications = new ArrayList<>();
            for (int i = 0; i<allModifications.size(); i++) {
                if(i != index) {
                    trainingModifications.add(modificationsWithFeatures.get(i));
                }
            }
            testModifications.add(modificationsWithFeatures.get(index));
            System.out.println("===== training set size "+trainingModifications.size());
            System.out.println("===== test set size "+testModifications.size());
            classifyCategoryLeaveOneOut(trainingModifications, testModifications, classifier);
        }
        List<ClassResults> classResults = classifier.classResults;
        System.out.println("\n\n Evolvability: ===== \n\n");

        System.out.println("TP: "+classResults.get(0).truePositive);
        System.out.println("FP: "+classResults.get(0).falsePositive);
        System.out.println("FN: "+classResults.get(0).falseNegative);

        System.out.println("Precision: "+classResults.get(0).truePositive/(classResults.get(0).truePositive + classResults.get(0).falsePositive));
        System.out.println("Recall: "+classResults.get(0).truePositive/(classResults.get(0).truePositive + classResults.get(0).falseNegative));
        System.out.println("\n\n Functional: ===== \n\n");

        System.out.println("TP: "+classResults.get(1).truePositive);
        System.out.println("FP: "+classResults.get(1).falsePositive);
        System.out.println("FN: "+classResults.get(1).falseNegative);

        System.out.println("Precision: "+classResults.get(1).truePositive/(classResults.get(1).truePositive + classResults.get(1).falsePositive));
        System.out.println("Recall: "+classResults.get(1).truePositive/(classResults.get(1).truePositive + classResults.get(1).falseNegative));
    }


    //Unordered data with SMOTE

    @Test
    public void AllClassifyCategoryUnorderedSMOTE(){
        List<Modification> allModifications = new ArrayList<>();
        allModifications.addAll(initExtractDate(FilesIndexer.AndroidDatasetPath, FilesIndexer.AndroidGerritUrl, FilesIndexer.AndroidFile, FilesIndexer.AndroidDatePath, true));
        allModifications.addAll(initExtractDate(FilesIndexer.JGitDatasetPath, FilesIndexer.JGitGerritUrl, FilesIndexer.JGitFile, FilesIndexer.JGitDatePath, true));
        allModifications.addAll(initExtractDate(FilesIndexer.JavaClientDatasetPath, FilesIndexer.JavaClientGerritUrl, FilesIndexer.JavaClientFile, FilesIndexer.JavaClientDatePath, true));
        int split = allModifications.size()/10;
        Collections.shuffle(allModifications, new Random());
        List<Modification> trainingModifications = new ArrayList<>();
        List<Modification> testModifications = new ArrayList<>();
        for (int i = 0; i<allModifications.size() - split; i++) {
            trainingModifications.add(allModifications.get(i));
        }
        for(int i = allModifications.size() - split; i<allModifications.size(); i++) {
            testModifications.add(allModifications.get(i));
        }
        System.out.println("===== training set size "+trainingModifications.size());
        System.out.println("===== test set size "+testModifications.size());
        classifyCategoryWithSMOTE(trainingModifications, testModifications);
    }


    @Test
    public void AllClassifyCategoryTimeSMOTE(){
        List<Modification> allModifications = new ArrayList<>();
        allModifications.addAll(initExtractDate(FilesIndexer.AndroidDatasetPath, FilesIndexer.AndroidGerritUrl, FilesIndexer.AndroidFile, FilesIndexer.AndroidDatePath, true));
        allModifications.addAll(initExtractDate(FilesIndexer.JGitDatasetPath, FilesIndexer.JGitGerritUrl, FilesIndexer.JGitFile, FilesIndexer.JGitDatePath, true));
        allModifications.addAll(initExtractDate(FilesIndexer.JavaClientDatasetPath, FilesIndexer.JavaClientGerritUrl, FilesIndexer.JavaClientFile, FilesIndexer.JavaClientDatePath, true));
        List<Modification> orderedModifications = orderModifications(allModifications);
        int split = orderedModifications.size()/10;
        List<Modification> trainingModifications = new ArrayList<>();
        List<Modification> testModifications = new ArrayList<>();
        for(Modification m : orderedModifications) {
            System.out.println("ordered date " + m.getDate());
        }
        for (int i = 0; i<orderedModifications.size() - split; i++) {
            trainingModifications.add(orderedModifications.get(i));
        }
        for(int i = orderedModifications.size() - split; i<orderedModifications.size(); i++) {
            testModifications.add(orderedModifications.get(i));
        }
        System.out.println("===== training set size "+trainingModifications.size());
        System.out.println("===== test set size "+testModifications.size());
        classifyCategoryWithSMOTE(trainingModifications, testModifications);
    }


    /*
    ===============
    Common methods
    ===============
     */


    private List<Modification> orderModifications(List<Modification> modifications) {
        int count = 0;
        List<ComparableModification> comparableModifications = new ArrayList<>();
        List<Modification> orderedModifications = new ArrayList<>();
        for(Modification m : modifications) {
            if(m.getDate() == null) {
                count++;
                System.out.println("HEREEEEEEE" + m.getReviewCode());
            }
            else {
                System.out.println("OK");
                comparableModifications.add(new ComparableModification(m.getDate(), m));
            }
        }
        System.out.println("Count null "+count);
        Collections.sort(comparableModifications);
        for(ComparableModification cm : comparableModifications) {
            orderedModifications.add(cm.getModification());
        }
        return orderedModifications;
    }


    private List<Modification> init(String datasetPath, String gerritUrl, String csvName, boolean filterImports){
        Dataset dataset = new Dataset(datasetPath);
        changes =  dataset.getChanges();
        GerritExtractor ge = new GerritExtractor(changes);
        ge.set(gerritUrl);
        CodeChunksExtractor cce = new CodeChunksExtractor();
        List<CodeChunk> codeChunkList = cce.extractCodeChunks(ge, csvName);

        if(gerritUrl.equals(FilesIndexer.AndroidGerritUrl)){
            codeChunkList = removeNonJavaFiles(codeChunkList);
        }

        if(filterImports){
            ImportFilter importFilter = new ImportFilter();
            codeChunkList = importFilter.filterImports(codeChunkList);
        }

        CodeChunksLinker cl = new CodeChunksLinker();
        return cl.splitChangeBased(codeChunkList);
    }



    private List<Modification> initExtractDate(String datasetPath, String gerritUrl, String csvName, String dateFilePath, boolean filterImports) {
        Dataset dataset = new Dataset(datasetPath);
        changes =  dataset.getChanges();
        GerritExtractor ge = new GerritExtractor(changes);
        ge.set(gerritUrl);
        CodeChunksExtractor cce = new CodeChunksExtractor();
        List<CodeChunk> codeChunkList = cce.extractCodeChunks(ge, csvName);

        if(gerritUrl.equals(FilesIndexer.AndroidGerritUrl)){
            codeChunkList = removeNonJavaFiles(codeChunkList);
        }

        if(filterImports){
            ImportFilter importFilter = new ImportFilter();
            codeChunkList = importFilter.filterImports(codeChunkList);
        }

        CodeChunksLinker cl = new CodeChunksLinker();
        List<Modification> modifications = cl.splitChangeBased(codeChunkList);

        modifications = setIDs(modifications, changes);
        File datesFile = new File(dateFilePath);

        if(!datesFile.exists()) {
            for (Modification m : modifications) {
                if (m.getReviewCode() != null) {
                    Date date = ge.mineRevisionDate(m.getReviewCode(), m.getRevisionCode());
                    m.setDate(date);
                    CSVWriter writer = null;
                    try {
                        writer = new CSVWriter(new FileWriter(datesFile, true));
                        writer.writeNext(new String[]{m.getReviewCode(), m.getRevisionCode(), m.getDate().toString()});
                        writer.close();
                    } catch (IOException e) {
                        e.printStackTrace();
                    }
                }
            }
        }
        else {
            for (Modification m : modifications) {
                List<ModificationDate> dates = new ArrayList<>();
                DateFormat dateFormat = new SimpleDateFormat(
                        "EEE MMM dd HH:mm:ss zzz yyyy", Locale.US);
                if (m.getReviewCode() != null) {
                    try (CSVReader reader = new CSVReader(new FileReader(datesFile))) {
                        String[] lineInArray;
                        while ((lineInArray = reader.readNext()) != null) {
                            dates.add(new ModificationDate(lineInArray[0], lineInArray[1], dateFormat.parse(lineInArray[2])));
                        }
                    } catch (FileNotFoundException e) {
                        e.printStackTrace();
                    } catch (IOException e) {
                        e.printStackTrace();
                    } catch (ParseException e) {
                        e.printStackTrace();
                    }
                }
                for(ModificationDate md : dates) {
                    if(md.reviewID.equals(m.getReviewCode()) && md.revisionID.equals(m.getRevisionCode())){
                        m.setDate(md.date);
                        break;
                    }
                }
            }
        }
        return modifications;
    }


    private List<Modification> setIDs(List<Modification> modifications, Multimap<String, Change> changes) {
        int count = 0;
        for(Modification m : modifications) {
            String changeID = null;
            if(m.getOldCodeChunk() == null) {
                changeID = m.getNewCodeChunk().getChangeID();
            }
            if(m.getNewCodeChunk() == null) {
                changeID = m.getOldCodeChunk().getChangeID();
            }
            if(m.getOldCodeChunk()!= null && m.getNewCodeChunk()!=null) {
                if(!m.getOldCodeChunk().getChangeID().equals(m.getNewCodeChunk().getChangeID())) {
                    count++;
                    System.out.println("Error on change id!");
                }
                else {
                    changeID = m.getNewCodeChunk().getChangeID();
                }
            }
            for(Change ch: changes.get(changeID)) {
                m.setReviewCode(ch.getReviewID());
                m.setRevisionCode(ch.getRevisionID());
            }
        }
        return modifications;
    }


    private List<CodeChunk> removeNonJavaFiles(List<CodeChunk> codeChunkList){
        List<CodeChunk> filteredChunksList = new ArrayList<>();
        for(CodeChunk codeChunk : codeChunkList) {
            String fileName = codeChunk.getFileName();
            if(fileName!=null) {
                String[] fileNameSplit = fileName.split("\\.");
                if (fileNameSplit[fileNameSplit.length - 1].equals("java")) {
                    filteredChunksList.add(codeChunk);
                }
            }
        }
        return filteredChunksList;
    }


    private void classifyCategory(List<Modification> modifications){
        List<Modification> classifierEntries = new ModificationConverter().extractFeatures(modifications, true);
        Instances data = new ModificationConverter().createCategoryInstances(classifierEntries);
        Classifier classifier = new Classifier(data);
        classifier.showFeaturesRank();
        classifier.showFeaturesRank();
        classifier.runJ48();
        classifier.runRandomForest();
    }


    private void classifyCategory10Times(List<Modification> modifications){
        List<Modification> classifierEntries = new ModificationConverter().extractFeatures(modifications, true);
        Instances data = new ModificationConverter().createCategoryInstances(classifierEntries);
        Classifier classifier = new Classifier(data);
        List<ClassifierResult> results = new ArrayList<>();
        results.addAll(classifier.runNaiveBayes10T(false));
        results.addAll(classifier.runJ4810T(false));
        results.addAll(classifier.runRandomForest10T(false));
        printResults(results);
    }


    private void classifyGroup10Times(List<Modification> modifications){
        List<Modification> classifierEntries = new ModificationConverter().extractFeatures(modifications, true);
        Instances data = new ModificationConverter().createGroupInstances(classifierEntries);
        Classifier classifier = new Classifier(data);
        //classifier.showFeaturesRank();
        List<ClassifierResult> results = new ArrayList<>();
        results.addAll(classifier.runNaiveBayes10T(true));
        results.addAll(classifier.runJ4810T(true));
        results.addAll(classifier.runRandomForest10T(true));
        printResults(results);
    }


    private void classifyCategory10TimesSMOTE(List<Modification> modifications){
        List<Modification> trainingEntries = new ModificationConverter().extractFeatures(modifications, true);
        Instances data = new ModificationConverter().createCategoryInstances(trainingEntries);
        Classifier classifier = new Classifier(data);
        //classifier.showFeaturesRank();
        List<ClassifierResult> results = new ArrayList<>();
        results.addAll(classifier.runNaiveBayesWithSMOTE10Times(false));
        results.addAll(classifier.runJ48WithSMOTE10Times(false));
        results.addAll(classifier.runRandomForestWithSMOTE10Times(false));
        printResults(results);
    }

    private void classifyCategory10TimesUndersample(List<Modification> modifications){
        List<Modification> trainingEntries = new ModificationConverter().extractFeatures(modifications, true);
        Instances data = new ModificationConverter().createCategoryInstances(trainingEntries);
        Classifier classifier = new Classifier(data);
        //classifier.showFeaturesRank();
        List<ClassifierResult> results = new ArrayList<>();
        results.addAll(classifier.runNaiveBayesWithUndersample10Times(false));
        results.addAll(classifier.runJ48WithUndersample10Times(false));
        results.addAll(classifier.runRandomForestWithUndersample10Times(false));
        printResults(results);
    }

    private void classifyGroup10TimesSMOTE(List<Modification> modifications){
        List<Modification> trainingEntries = new ModificationConverter().extractFeatures(modifications, true);
        Instances data = new ModificationConverter().createGroupInstances(trainingEntries);
        Classifier classifier = new Classifier(data);
        classifier.showFeaturesRank();
        List<ClassifierResult> results = new ArrayList<>();
        results.addAll(classifier.runNaiveBayesWithSMOTE10Times(true));
        results.addAll(classifier.runJ48WithSMOTE10Times(true));
        results.addAll(classifier.runRandomForestWithSMOTE10Times(true));
        printResults(results);
    }


    private void classifyGroup10TimesSMOTEAttrSelection(List<Modification> modifications){
        List<Modification> trainingEntries = new ModificationConverter().extractFeatures(modifications, true);
        Instances data = new ModificationConverter().createGroupInstances(trainingEntries);
        Classifier classifier = new Classifier(data);
        classifier.showFeaturesRank();
        List<ClassifierResult> results = new ArrayList<>();
        results.addAll(classifier.runNaiveBayesAttributeSelectionSMOTE10Times(true));
        results.addAll(classifier.runJ48AttributeSelectionSMOTE10Times(true));
        results.addAll(classifier.runRandomForestAttributeSelection10TimesSMOTE(true));
        printResults(results);
    }


    private void classifyCategory10TimesAttrSelection(List<Modification> modifications){
        List<Modification> trainingEntries = new ModificationConverter().extractFeatures(modifications, true);
        Instances data = new ModificationConverter().createCategoryInstances(trainingEntries);
        Classifier classifier = new Classifier(data);
        classifier.showFeaturesRank();
        classifier.showFeaturesRank();
        List<ClassifierResult> results = new ArrayList<>();
        results.addAll(classifier.runNaiveBayesAttributeSelection10Times(false));
        results.addAll(classifier.runJ48AttributeSelection10Times(false));
        results.addAll(classifier.runRandomForestAttributeSelection10Times(false));
        printResults(results);
    }

    private void classifyGroup10TimesAttrSelection(List<Modification> modifications){
        List<Modification> trainingEntries = new ModificationConverter().extractFeatures(modifications, true);
        Instances data = new ModificationConverter().createGroupInstances(trainingEntries);
        Classifier classifier = new Classifier(data);
        classifier.showFeaturesRank();
        classifier.showFeaturesRank();
        List<ClassifierResult> results = new ArrayList<>();
        results.addAll(classifier.runNaiveBayesAttributeSelection10Times(true));
        results.addAll(classifier.runJ48AttributeSelection10Times(true));
        results.addAll(classifier.runRandomForestAttributeSelection10Times(true));
        printResults(results);
    }


    private void classifyCategory10TimesSMOTEAttrSelection(List<Modification> modifications){
        List<Modification> trainingEntries = new ModificationConverter().extractFeatures(modifications, true);
        Instances data = new ModificationConverter().createCategoryInstances(trainingEntries);
        Classifier classifier = new Classifier(data);
        classifier.showFeaturesRank();
        classifier.showFeaturesRank();
        List<ClassifierResult> results = new ArrayList<>();
        results.addAll(classifier.runNaiveBayesAttributeSelectionSMOTE10Times(false));
        results.addAll(classifier.runJ48AttributeSelectionSMOTE10Times(false));
        results.addAll(classifier.runRandomForestAttributeSelection10TimesSMOTE(false));
        printResults(results);
    }


    private void classifyGroup10TimesSMOTELessFeatures(List<Modification> modifications){
        List<Modification> trainingEntries = new ModificationConverter().extractFeatures(modifications, true);
        Instances data = new ModificationConverter().createGroupInstancesLessFeatures(trainingEntries);
        Classifier classifier = new Classifier(data);
        classifier.showFeaturesRank();
        classifier.showFeaturesRank();
        List<ClassifierResult> results = new ArrayList<>();
        results.addAll(classifier.runNaiveBayesWithSMOTE10Times(true));
        results.addAll(classifier.runJ48WithSMOTE10Times(true));
        results.addAll(classifier.runRandomForestWithSMOTE10Times(true));
        printResults(results);
    }


    private void printResults(List<ClassifierResult> results) {
        System.out.println("==== SUMMARY ===");
        String classifierName = null;
        for(ClassifierResult result : results) {
            if(classifierName != null) {
                if(!classifierName.equals(result.classifierName)) {
                    System.out.println("\n\n");
                }
            }
            classifierName = result.classifierName;
            System.out.println(result.classifierName + " class: "+ result.className);
            System.out.println(" Precision: "+result.precision);
            System.out.println(" Recall: "+result.recall);
            System.out.println(" F-Measure: "+result.fmeasure);
            System.out.println(" AUC: "+result.auc);
            System.out.println(" MCC: "+result.mcc);
            System.out.println(" \n ");
        }
    }


    private void classifyCategory(List<Modification> trainingModifications, List<Modification> testModifications){
        List<Modification> trainingEntries = new ModificationConverter().extractFeatures(trainingModifications, true);
        Instances trainingData = new ModificationConverter().createCategoryInstancesLeaveOneOut(trainingEntries);
        List<Modification> testEntries = new ModificationConverter().extractFeatures(testModifications, true);
        Instances testData = new ModificationConverter().createCategoryInstancesLeaveOneOut(testEntries);
        Classifier classifier = new Classifier(trainingData, testData);
        classifier.filterTrainingData();
        classifier.filterTestData();
        List<ClassifierResult> classifierResults = new ArrayList<>();
        classifierResults.addAll(classifier.runNaiveBayesWithTestsSet(false));
        classifierResults.addAll(classifier.runJ48WithTestsSet(false));
        classifierResults.addAll(classifier.runRandomForestWithTestsSet(false));
        printResults(classifierResults);
    }


    private void classifyCategoryLeaveOneOut(List<Modification> trainingModifications, List<Modification> testModifications, ClassifierLeaveOneOut classifier){
        Instances trainingData = new ModificationConverter().createCategoryInstances(trainingModifications);
        Instances testData = new ModificationConverter().createCategoryInstances(testModifications);
        classifier.setData(trainingData, testData);
        classifier.filterTrainingData(new String[]{"-R","47"});
        classifier.filterTestData(new String[]{"-R","47"});
        List<ClassifierResult> classifierResults = new ArrayList<>();
        //classifierResults.addAll(classifier.runNaiveBayesWithTestsSetLeaveOneOut(false));
        //classifierResults.addAll(classifier.runJ48WithTestsSet(false));
        classifierResults.addAll(classifier.runRandomForestWithTestsSetLeaveOneOut(false));
        printResults(classifierResults);
    }


    private void classifyCategoryWithSMOTE(List<Modification> modifications){
        List<Modification> trainingEntries = new ModificationConverter().extractFeatures(modifications, true);
        Instances data = new ModificationConverter().createCategoryInstances(trainingEntries);
        Classifier classifier = new Classifier(data);
        classifier.showFeaturesRank();
        //classifier.filterTestData(new String[]{"-R","19"});
        classifier.showFeaturesRank();
        classifier.runRandomForestWithSMOTE();
    }


    private void classifyCategoryWithSMOTE(List<Modification> trainingModifications, List<Modification> testModifications){
        List<Modification> trainingEntries = new ModificationConverter().extractFeatures(trainingModifications, true);
        Instances trainingData = new ModificationConverter().createCategoryInstancesLeaveOneOut(trainingEntries);
        List<Modification> testEntries = new ModificationConverter().extractFeatures(testModifications, true);
        Instances testData = new ModificationConverter().createCategoryInstancesLeaveOneOut(testEntries);
        Classifier classifier = new Classifier(trainingData, testData);
        classifier.showFeaturesRank();
        classifier.filterTrainingDataWithSMOTE();
        //classifier.filterTestData(new String[]{"-R","19"});
        classifier.showFeaturesRank();
        List<ClassifierResult> results = new ArrayList<>();
        results.addAll(classifier.runNaiveBayesWithTestsSetSMOTE(false));
        results.addAll(classifier.runJ48WithTestsSetSMOTE(false));
        results.addAll(classifier.runRandomForestWithTestsSetSMOTE(false));
        printResults(results);
    }


    private void classifyGroupWithSMOTE(List<Modification> trainingModifications, List<Modification> testModifications){
        List<Modification> trainingEntries = new ModificationConverter().extractFeatures(trainingModifications, true);
        Instances trainingData = new ModificationConverter().createGroupInstancesLeaveOneOut(trainingEntries);
        List<Modification> testEntries = new ModificationConverter().extractFeatures(testModifications, true);
        Instances testData = new ModificationConverter().createGroupInstancesLeaveOneOut(testEntries);
        Classifier classifier = new Classifier(trainingData, testData);
        classifier.showFeaturesRank();
        classifier.filterTrainingDataWithSMOTE();
        //classifier.filterTestData(new String[]{"-R","19"});
        classifier.showFeaturesRank();
        List<ClassifierResult> results = new ArrayList<>();
        results.addAll(classifier.runNaiveBayesWithTestsSetSMOTE(true));
        results.addAll(classifier.runJ48WithTestsSetSMOTE(true));
        results.addAll(classifier.runRandomForestWithTestsSetSMOTE(true));
        printResults(results);
    }


    private void classifyCategoryWithUndersampling(List<Modification> modifications){
        List<Modification> trainingEntries = new ModificationConverter().extractFeatures(modifications, true);
        Instances data = new ModificationConverter().createCategoryInstances(trainingEntries);
        Classifier classifier = new Classifier(data);
        classifier.showFeaturesRank();
        //classifier.filterTestData(new String[]{"-R","19"});
        classifier.showFeaturesRank();
        classifier.runRandomForestWithUndersampling();
    }


    private void classifyCategoryWithSMOTEAndUndersampling(List<Modification> modifications){
        List<Modification> trainingEntries = new ModificationConverter().extractFeatures(modifications, true);
        Instances data = new ModificationConverter().createCategoryInstances(trainingEntries);
        Classifier classifier = new Classifier(data);
        classifier.showFeaturesRank();
        //classifier.filterTestData(new String[]{"-R","19"});
        classifier.showFeaturesRank();
        classifier.runRandomForestWithSMOTEAndUndersampling();
    }


    private void classifyGroup(List<Modification> modifications){
        List<Modification> classifierEntries = new ModificationConverter().extractFeatures(modifications, true);
        Instances data = new ModificationConverter().createGroupInstances(classifierEntries);
        Classifier classifier = new Classifier(data);
        classifier.showFeaturesRank();
        //classifier.filterData(new String[]{"-R","23"});
        classifier.showFeaturesRank();
        classifier.computeAttributeCorrelation(data);
        classifier.runJ48();
        classifier.runRandomForest();
        //classifier.runGridSearch();
    }

    private void classifyGroup(List<Modification> trainingModifications, List<Modification> testModifications){
        List<Modification> trainingEntries = new ModificationConverter().extractFeatures(trainingModifications, true);
        Instances trainingData = new ModificationConverter().createGroupInstancesLeaveOneOut(trainingEntries);
        List<Modification> testEntries = new ModificationConverter().extractFeatures(testModifications, true);
        Instances testData = new ModificationConverter().createGroupInstancesLeaveOneOut(testEntries);
        Classifier classifier = new Classifier(trainingData, testData);
        //classifier.showFeaturesRank();
        classifier.filterTrainingData();
        classifier.filterTestData();
        //classifier.showFeaturesRank();
        List<ClassifierResult> results = new ArrayList<>();
        results.addAll(classifier.runNaiveBayesWithTestsSet(true));
        results.addAll(classifier.runJ48WithTestsSet(true));
        results.addAll(classifier.runRandomForestWithTestsSet(true));
        printResults(results);
    }


    public void classifyCategoryGridSearch(List<Modification> modifications) {
        List<Modification> classifierEntries = new ModificationConverter().extractFeatures(modifications, true);
        Instances data = new ModificationConverter().createGroupInstances(classifierEntries);
        Classifier classifier = new Classifier(data);
        classifier.filterData(new String[]{"-R", "47"});
        classifier.runGridSearch();
    }


    private void classifyGroupWithSearchGrid(List<Modification> modifications){
        List<Modification> classifierEntries = new ModificationConverter().extractFeatures(modifications, true);
        Instances data = new ModificationConverter().createGroupInstances(classifierEntries);
        Classifier classifier = new Classifier(data);
        classifier.filterData(new String[]{"-R","30"});
        classifier.runGridSearch();
    }


    private void classifyGroupWithSmote(List<Modification> modifications){
        List<Modification> classifierEntries = new ModificationConverter().extractFeatures(modifications, true);
        Instances data = new ModificationConverter().createGroupInstances(classifierEntries);
        Classifier classifier = new Classifier(data);
        classifier.showFeaturesRank();
        //classifier.filterData(new String[]{"-R","23"});
        classifier.showFeaturesRank();
        classifier.runRandomForestWithSMOTE();
    }

    private void classifyGroupWithCost(List<Modification> modifications){
        List<Modification> classifierEntries = new ModificationConverter().extractFeatures(modifications, true);
        Instances data = new ModificationConverter().createGroupInstances(classifierEntries);
        Classifier classifier = new Classifier(data);
        classifier.showFeaturesRank();
        //classifier.filterData(new String[]{"-R","23"});
        classifier.showFeaturesRank();
        classifier.computeAttributeCorrelation(data);
        classifier.runJ48();
        classifier.runRandomForestWithCostMatrix();
        //classifier.runGridSearch();
    }


    private void classifyDecomposedGroup(List<Modification> modifications){
        List<Modification> classifierEntries = new ModificationConverter().extractFeatures(modifications, true);
        Instances data = new ModificationConverter().createDecomposedGroupInstances(classifierEntries);
        Classifier classifier = new Classifier(data);
        classifier.showFeaturesRank();
        //classifier.filterData(new String[]{"-R","23"});
        classifier.showFeaturesRank();
        classifier.runJ48();
        classifier.runRandomForest();
    }


    private void classifyType(List<Modification> modifications){
        List<Modification> classifierEntries = new ModificationConverter().extractFeatures(modifications, true);
        Instances data = new ModificationConverter().createTypeInstances(classifierEntries);
        Classifier classifier = new Classifier(data);
        classifier.showFeaturesRank();
        classifier.filterData(new String[]{"-R","22"});
        classifier.showFeaturesRank();
        classifier.runJ48();
        classifier.runRandomForest();
        //classifier.runGridSearch();
    }


    private void classifyType(List<Modification> trainingModifications, List<Modification> testModifications){
        List<Modification> trainingEntries = new ModificationConverter().extractFeatures(trainingModifications, true);
        Instances trainingData = new ModificationConverter().createTypeInstances(trainingEntries);
        List<Modification> testEntries = new ModificationConverter().extractFeatures(testModifications, true);
        Instances testData = new ModificationConverter().createTypeInstances(testEntries);
        Classifier classifier = new Classifier(trainingData, testData);
        classifier.showFeaturesRank();
        classifier.filterTrainingData();
        classifier.filterTestData();
        classifier.showFeaturesRank();
        classifier.runJ48WithTestsSet(false); //the method need to be changed to handle types
        classifier.runRandomForestWithTestsSet(false);
    }


    private void classifySubtype(List<Modification> modifications){
        List<Modification> classifierEntries = new ModificationConverter().extractFeatures(modifications, true);
        Instances data = new ModificationConverter().createSubtypeInstances(classifierEntries);
        Classifier classifier = new Classifier(data);
        classifier.showFeaturesRank();
        //classifier.filterData(new String[]{"-R","23"});
        classifier.showFeaturesRank();
        classifier.runJ48();
        classifier.runRandomForest();
    }

    /*
    Test only functional vs structure to understand the most important features.
    */

    @Test
    public void testClassifyFunctionalVsStructure10Times(){
        List<Modification> modifications = extractModificationFromAllProjectsFilterImports();
        List<Modification> classifierEntries = new ModificationConverter().extractFeatures(modifications, true);
        Instances data = new ModificationConverter().createStructVsFunctInstances(classifierEntries);
        Classifier classifier = new Classifier(data);
        //classifier.showFeaturesRank();
        List<ClassifierResult> results = new ArrayList<>();
        results.addAll(classifier.runNaiveBayes10T(false));
        results.addAll(classifier.runJ4810T(false));
        results.addAll(classifier.runRandomForest10T(false));
        printResults(results);
    }


    /*
    Look at feature distribution
     */

    @Test
    public void testPrintFeaturesDistribution(){
        List<Modification> modifications = extractModificationFromAllProjectsFilterImports();
        List<Modification> classifierEntries = new ModificationConverter().extractFeatures(modifications, true);
        String path = "./FeaturesToPlot.csv";
        CSVWriter writer = null;
        for(Modification c : classifierEntries) {
            String[] stringArray = new String[7];
            FeaturesList featuresList = c.getFeatures();
            stringArray[0] = Integer.toString(featuresList.ifAdded);
            stringArray[1] = Integer.toString(featuresList.countFunctionsAdded);
            stringArray[2] = Integer.toString(featuresList.countNewAdded);
            stringArray[3] = Integer.toString(featuresList.newLOCBlank);
            stringArray[4] = Integer.toString(featuresList.newLOCExec);
            stringArray[5] = Integer.toString(featuresList.newLOCComments);
            stringArray[6] = featuresList.group;
            if (featuresList.group == null || featuresList.group.equals("UNKNOWN")){
                continue;
            }
            try {
                File file = new File(path);
                if(!file.exists()) {
                    writer = new CSVWriter(new FileWriter(path));
                    String[] header = {"ifAdded", "methodsAdded", "newAdded", "new_blank", "new_exec", "new_comments", "Group"};
                    writer.writeNext(header);
                    writer.close();
                }
                writer = new CSVWriter(new FileWriter(path, true));
                writer.writeNext(stringArray);
                writer.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        return;
    }


    /*
    ===========
    Common methods: 10-FOLD CROSS VALIDATION ON ALL PROJECTS
    ===========
     */

    private List<Modification> extractModificationFromAllProjects(){
        List<Modification> modifications = new ArrayList<>();
        modifications.addAll(init(FilesIndexer.AndroidDatasetPath, FilesIndexer.AndroidGerritUrl, FilesIndexer.AndroidFile, false));
        modifications.addAll(init(FilesIndexer.JGitDatasetPath, FilesIndexer.JGitGerritUrl, FilesIndexer.JGitFile, false));
        modifications.addAll(init(FilesIndexer.JavaClientDatasetPath, FilesIndexer.JavaClientGerritUrl, FilesIndexer.JavaClientFile, false));
        return modifications;
    }


    private List<Modification> extractModificationFromAllProjectsFilterImports(){
        List<Modification> modifications = new ArrayList<>();
        modifications.addAll(init(FilesIndexer.AndroidDatasetPath, FilesIndexer.AndroidGerritUrl, FilesIndexer.AndroidFile, true));
        modifications.addAll(init(FilesIndexer.JGitDatasetPath, FilesIndexer.JGitGerritUrl, FilesIndexer.JGitFile, true));
        modifications.addAll(init(FilesIndexer.JavaClientDatasetPath, FilesIndexer.JavaClientGerritUrl, FilesIndexer.JavaClientFile, true));

        int checkCount = 0;
        int functionalCount = 0;
        for (Modification m : modifications) {
            if(m.getNewCodeChunk() != null) {
                if(m.getNewCodeChunk().getCategory().equals(Change.Category.FUNCTIONAL)) {
                    functionalCount++;
                    if(m.getNewCodeChunk().getType().equals(Type.LOGIC)) {
                        checkCount++;
                    }
                }
            }
            else {
                if(m.getOldCodeChunk() != null) {
                    if(m.getOldCodeChunk().getCategory().equals(Change.Category.FUNCTIONAL)) {
                        functionalCount++;
                        if(m.getOldCodeChunk().getType().equals(Type.LOGIC)) {
                            checkCount++;
                        }
                    }
                }
            }
        }
        System.out.println("Functional "+functionalCount);
        System.out.println("Check "+checkCount);


        return modifications;
    }


    @Test
    public void classifierCategoryTest(){
        List<Modification> modifications = extractModificationFromAllProjects();
        classifyCategory(modifications);
    }

    @Test
    public void classifierCategoryWithGridSearchTest(){
        List<Modification> modifications = extractModificationFromAllProjects();
        classifyCategoryGridSearch(modifications);
    }

    @Test
    public void classifierGroupTest(){
        List<Modification> modifications = extractModificationFromAllProjects();
        classifyGroup(modifications);
    }

    @Test
    public void classifierGroupWithSmoteTest(){
        List<Modification> modifications = extractModificationFromAllProjects();
        classifyGroupWithSmote(modifications);
    }

    @Test
    public void classifierGroupWithSearchGridTest(){
        List<Modification> modifications = extractModificationFromAllProjects();
        classifyGroupWithSearchGrid(modifications);
    }

    @Test
    public void classifierDecomposedGroupTest(){
        List<Modification> modifications = extractModificationFromAllProjects();
        classifyDecomposedGroup(modifications);
    }

    @Test
    public void classifierTypeTest(){
        List<Modification> modifications = extractModificationFromAllProjects();
        classifyType(modifications);
    }

    @Test
    public void classifierSubTypeTest(){
        List<Modification> modifications = extractModificationFromAllProjects();
        classifySubtype(modifications);
    }


    @Test
    public void classifierCategoryFilterImportsTest(){
        List<Modification> modifications = extractModificationFromAllProjectsFilterImports();
        classifyCategory(modifications);
    }


    /**
     *
     *10-times 10-fold cross validation
     */


    @Test
    public void classifierCategoryFilterImportsTest10Times10Fold(){
        List<Modification> modifications = extractModificationFromAllProjectsFilterImports();
        classifyCategory10Times(modifications);
    }

    @Test
    public void classifierGroupFilterImportsTest10Times10Fold(){
        List<Modification> modifications = extractModificationFromAllProjectsFilterImports();
        classifyGroup10Times(modifications);
    }

    @Test
    public void classifierCategoryFilterImportsTest10Times10FoldSMOTE(){
        List<Modification> modifications = extractModificationFromAllProjectsFilterImports();
        classifyCategory10TimesSMOTE(modifications);
    }

    @Test
    public void classifierGroupFilterImportsTest10Times10FoldSMOTE(){
        List<Modification> modifications = extractModificationFromAllProjectsFilterImports();
        classifyGroup10TimesSMOTE(modifications);
    }

    @Test
    public void classifierCategoryFilterImportsTestSMOTE(){
        List<Modification> modifications = extractModificationFromAllProjectsFilterImports();
        classifyCategoryWithSMOTE(modifications);
    }

    @Test
    public void classifierGroupFilterImportsTest(){
        List<Modification> modifications = extractModificationFromAllProjectsFilterImports();
        classifyGroup(modifications);
    }

    @Test
    public void classifierGroupFilterImportsSMOTETest(){
        List<Modification> modifications = extractModificationFromAllProjectsFilterImports();
        classifyGroupWithSmote(modifications);
    }

    @Test
    public void classifierGroupFilterImportsAndCostTest(){
        List<Modification> modifications = extractModificationFromAllProjectsFilterImports();
        classifyGroupWithCost(modifications);
    }

    @Test
    public void classifierDecomposedGroupFilterImportsTest(){
        List<Modification> modifications = extractModificationFromAllProjectsFilterImports();
        classifyDecomposedGroup(modifications);
    }

    @Test
    public void classifierTypeFilterImportsTest(){
        List<Modification> modifications = extractModificationFromAllProjectsFilterImports();
        classifyType(modifications);
    }

    @Test
    public void classifierSubTypeFilterImportTest(){
        List<Modification> modifications = extractModificationFromAllProjectsFilterImports();
        classifySubtype(modifications);
    }


    @Test
    public void classifierCategoryWithSMOTETest(){
        List<Modification> modifications = extractModificationFromAllProjects();
        classifyCategoryWithSMOTE(modifications);
    }


    @Test
    public void classifierCategoryWithSMOTEWithoutImportsTest(){
        List<Modification> modifications = extractModificationFromAllProjectsFilterImports();
        classifyCategoryWithSMOTE(modifications);
    }


    @Test
    public void classifierCategoryWithUndersamplingTest(){
        List<Modification> modifications = extractModificationFromAllProjects();
        classifyCategoryWithUndersampling(modifications);
    }

    @Test
    public void classifierCategoryWithSMOTEAndUndersamplingTest(){
        List<Modification> modifications = extractModificationFromAllProjects();
        classifyCategoryWithSMOTEAndUndersampling(modifications);
    }



    /*
    =========
    Test only on functional type
    =========
     */


    @Test
    public void classifierOnlyFunctionalTypeTest(){
        List<Modification> modifications = extractModificationFromAllProjects();
        List<Modification> classifierEntries = new ModificationConverter().extractFeatures(modifications, true);
        List<Modification> modificationsOnlyFunctional = new ArrayList<>();
        for(Modification m : classifierEntries){
            if(m.getFeatures().category!=null && m.getFeatures().category.equals("FUNCTIONAL")){
                modificationsOnlyFunctional.add(m);
            }
        }
        Instances data = new ModificationConverter().createTypeInstances(modificationsOnlyFunctional);
        Classifier classifier = new Classifier(data);
        classifier.showFeaturesRank();
        classifier.filterData(new String[]{"-R","21"});
        classifier.showFeaturesRank();
        classifier.runJ48();
        classifier.runRandomForest();
    }


    /*
    Tests with attribute selection
     */

    @Test
    public void classifyGroup10TimesAttributeSelection(){
        List<Modification> modifications = extractModificationFromAllProjectsFilterImports();
        classifyGroup10TimesAttrSelection(modifications);
    }

    @Test
    public void classifyGroup10TimesSMOTEAttributeSelection(){
        List<Modification> modifications = extractModificationFromAllProjectsFilterImports();
        classifyGroup10TimesSMOTEAttrSelection(modifications);
    }

    @Test
    public void classifyCategory10TimesAttributeSelection(){
        List<Modification> modifications = extractModificationFromAllProjectsFilterImports();
        classifyCategory10TimesAttrSelection(modifications);
    }

    @Test
    public void classifyCategory10TimesSMOTEAttributeSelection(){
        List<Modification> modifications = extractModificationFromAllProjectsFilterImports();
        classifyCategory10TimesSMOTEAttrSelection(modifications);
    }

    @Test
    public void classifyGroup10TimesSMOTELessFeatures(){
        List<Modification> modifications = extractModificationFromAllProjectsFilterImports();
        classifyGroup10TimesSMOTELessFeatures(modifications);
    }

}