package backend.main;

import backend.analysis.CodeChunksLinker;
import backend.analysis.GerritExtractor;
import backend.analysis.ImportFilter;
import backend.analysis.machineLearning.Classifier;
import backend.analysis.machineLearning.ModificationConverter;
import backend.base.Change;
import backend.base.gerrit_data.*;
import backend.db_connectors.Dataset;
import backend.db_connectors.FilesIndexer;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Iterables;
import com.google.common.collect.Multimap;
import com.opencsv.CSVWriter;
import weka.core.Instances;

import java.io.*;
import java.util.*;

public class RepositoryAnalyzer {


    private final static String infoFile = "./classifiedChangeForReport.csv";


    private static File file = new File(infoFile);
    private static FileWriter fileWriter;
    private static CSVWriter writer;

    /*
    Group codes
     */

    private final static int DOCUMENTATIONCODE = 0;
    private final static int VISUALREPCODE = 1;
    private final static int STRUCTURALCODE = 2;
    private final static int FUNCTIONALCODE = 3;


    /*
    Category codes
     */

    private final static int EVOLVABILITYCATCODE = 0;
    private final static int FUNCTIONALCATCODE = 1;




    private static List<ProjectToExtract> initializeProjects() {
        List<ProjectToExtract> projectsToExtract = new ArrayList<>();

        //projectsToExtract.add(new ProjectToExtract("http://review.couchbase.com", "./JavaClientRecordExtract",
          //    "couchbase-java-client", 120000, 70000));

        //projectsToExtract.add(new ProjectToExtract("https://git.eclipse.org/r", "./JGitRecordExtract",
          //      "jgit/jgit", 150000, 110000));

        /*
        projectsToExtract.add(new ProjectToExtract("https://git.eclipse.org/r", "./EGitRecordExtract",
                "egit/egit", 150000, 110000));

        projectsToExtract.add(new ProjectToExtract("https://git.eclipse.org/r", "./LinuxtoolsRecordExtract",
                "linuxtools/org.eclipse.linuxtools", 150000, 110000));

        projectsToExtract.add(new ProjectToExtract("https://git.eclipse.org/r", "./PlatformUIRecordExtract",
                "platform/eclipse.platform.ui", 150000, 110000));

        projectsToExtract.add(new ProjectToExtract("http://review.couchbase.com", "./JvmCoreRecordExtract",
                "couchbase-jvm-core", 120000, 80000));

        projectsToExtract.add(new ProjectToExtract("http://review.couchbase.com", "./SpymemcachedRecordExtract",
                "spymemcached", 80000, 40000));
        */

        projectsToExtract.add(new ProjectToExtract("https://git.eclipse.org/r", "./TestExtract",
                "jgit/jgit", 150000, 130000));



        return projectsToExtract;
    }


    public static void main(String args[]) {

        List<ProjectToExtract> projectsToExtract = initializeProjects();


        for(ProjectToExtract p : projectsToExtract) {

            if (!file.exists()) {
                try {
                    fileWriter = new FileWriter(file, true);
                    writer = new CSVWriter(fileWriter);
                    String[] header = {"Info", "Documentation/Evolvability", "Visual Representation", "Structure", "Function"};
                    writer.writeNext(header);
                    writer.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }

            try {
                fileWriter = new FileWriter(file, true);
                writer = new CSVWriter(fileWriter);
                String projectName = "";
                if(p.getProjectName()!=null){
                    projectName = p.getProjectName();
                }
                String[] data = {p.getGerritUrl(), projectName};
                writer.writeNext(data);
                writer.close();
            } catch (IOException e) {
                e.printStackTrace();
            }


            GerritExtractor ge = new GerritExtractor();
            ge.setProjectName(p.getProjectName());
            ge.set(p.getGerritUrl());

            File temporaryFile = new File(p.getTemporaryFilePath());
            List<Modification> modificationsToClassify;

            Map<String, Integer> reviews = null;
            Map<String, Integer> revisions = null;

            if (!temporaryFile.exists()) {
                List<ReviewDiff> reviewDiffList = ge.mineRepositoryData(p.getStart(), p.getEnd());

                int numberOfReviews = reviewDiffList.size();
                int numberOfRevisions = countNumberOfRevisions(reviewDiffList);
                String[] lineToWrite = {"Total Reviews", Integer.toString(numberOfReviews), "Total Revision", Integer.toString(numberOfRevisions)};

                reviews = countReviewsPerMonth(reviewDiffList);
                revisions = countRevisionPerMonth(reviewDiffList);

                writeToFile(lineToWrite);

                modificationsToClassify = createListOfModificationsToClassify(reviewDiffList);
                try (FileOutputStream fileOutputStream = new FileOutputStream(p.getTemporaryFilePath())) {
                    ObjectOutputStream objectOutputStream = new ObjectOutputStream(fileOutputStream);
                    for (Modification c : modificationsToClassify) {
                        objectOutputStream.writeObject(c);
                    }
                } catch (IOException e) {
                    e.printStackTrace();
                }
            } else {
                modificationsToClassify = new ArrayList<>();
                try (FileInputStream fileInputStream = new FileInputStream(temporaryFile)) {
                    ObjectInputStream objectInputStream = new ObjectInputStream(fileInputStream);
                    for (; ; ) {
                        modificationsToClassify.add((Modification) objectInputStream.readObject());
                    }
                } catch (EOFException e) {
                    //ignored
                } catch (IOException | ClassNotFoundException e) {
                    e.printStackTrace();
                }
            }


            for(Modification m : modificationsToClassify){
                System.out.println("=============");
                if(m.getOldCodeChunk()!=null){
                    for(DiffLine diffLine : m.getOldCodeChunk().getLines()) {
                        System.out.println("-"+diffLine.getLineNumber()+ "  :   "+diffLine.getDiffLine());
                    }
                }
                if(m.getNewCodeChunk()!=null){
                    for(DiffLine diffLine : m.getNewCodeChunk().getLines()) {
                        System.out.println(+diffLine.getLineNumber()+ "  :   "+diffLine.getDiffLine());
                    }
                }
            }


            System.out.println("MODIFICATIONS TO CLASSIFY:  " + modificationsToClassify.size());

            List<Modification> trainingModifications = loadTestProjectsModifications();
            List<Modification> trainingEntries = new ModificationConverter().extractFeatures(trainingModifications, true);
            Instances trainingDataGroup = new ModificationConverter().createGroupInstances(trainingEntries);

            //classifyModificationsCategory(modificationsToClassify);
            classifyModificationsGroup(modificationsToClassify, trainingDataGroup, "Total group");

            Multimap<String, Modification> modificationsPerMonth = groupModificationsPerMonth(modificationsToClassify);

            List<String> orderedKeys = orderKeys(modificationsPerMonth);

            for (String key : orderedKeys) {
                System.out.println("KEY  "+key);
                if(reviews!=null && revisions!=null) {
                    String[] generalInfo = {"number reviews per month", Integer.toString(reviews.get(key)), "number revisions per month", Integer.toString(revisions.get(key))};
                    writeToFile(generalInfo);
                }
                classifyModificationsGroup(new ArrayList<>(modificationsPerMonth.get(key)), trainingDataGroup, key);
            }

            Instances trainingDataCategory = new ModificationConverter().createCategoryInstances(trainingEntries);
            classifyModificationsCategory(modificationsToClassify, trainingDataCategory, "Total category");

            for (String key : orderedKeys) {
                classifyModificationsCategory(new ArrayList<>(modificationsPerMonth.get(key)), trainingDataCategory, key);
            }
        }

    }


    private static List<String> orderKeys (Multimap<String, Modification> modificationMultimap){
        Set<String> keys = modificationMultimap.keySet();
        List<String> orderedKeys = new ArrayList<>();
        for(String k : keys){
            if(orderedKeys.isEmpty()){
                orderedKeys.add(k);
                System.out.println("\n\n1:  " +k );
            }
            else{
                boolean found = false;
                int index = 0;

                while(index<orderedKeys.size() && found==false){
                    if(Iterables.get(modificationMultimap.get(k), 0).getDate().getYear()>
                    Iterables.get(modificationMultimap.get(orderedKeys.get(index)), 0).getDate().getYear()) {
                        found = true;
                        orderedKeys.add(index, k);
                    } else{
                        if(Iterables.get(modificationMultimap.get(k), 0).getDate().getYear()==
                                Iterables.get(modificationMultimap.get(orderedKeys.get(index)), 0).getDate().getYear()
                                && Iterables.get(modificationMultimap.get(k), 0).getDate().getMonth()>
                                Iterables.get(modificationMultimap.get(orderedKeys.get(index)), 0).getDate().getMonth()){
                            System.out.println("\n\n2:  " +k + "  at   " +index );
                            orderedKeys.add(index, k);
                            found = true;
                        }
                        else {
                            index++;
                        }
                    }
                }
                if(found==false){
                    System.out.println("\n\n3:  " +k + "  at   " +index );
                    orderedKeys.add(k);
                }
            }
        }
        return orderedKeys;
    }


    private static Map<String, Integer> countReviewsPerMonth(List<ReviewDiff> reviews) {
        Map<String, Integer> reviewsMap = new HashMap<>();
        for(ReviewDiff review : reviews){
            int month = review.getDate().getMonth();
            String monthString = Integer.toString(month);
            String separator = "_";
            int year = review.getDate().getYear() + 1900;
            String yearString = Integer.toString(year);
            String key = monthString.concat(separator).concat(yearString);
            if(!reviewsMap.containsKey(key)){
                reviewsMap.put(key, 1);
            }
            else {
                int value = reviewsMap.get(key);
                reviewsMap.put(key, value+1);
            }
        }
        return reviewsMap;
    }


    private static Map<String, Integer> countRevisionPerMonth(List<ReviewDiff> reviews) {
        Map<String, Integer> revisionsMap = new HashMap<>();
        for(ReviewDiff review : reviews){
            int month = review.getDate().getMonth();
            String monthString = Integer.toString(month);
            String separator = "_";
            int year = review.getDate().getYear() + 1900;

            String yearString = Integer.toString(year);
            String key = monthString.concat(separator).concat(yearString);
            if(!revisionsMap.containsKey(key)){
                revisionsMap.put(key, review.getRevisions().size());
            }
            else {
                int value = revisionsMap.get(key);
                revisionsMap.put(key, value+review.getRevisions().size());
            }
        }
        return revisionsMap;
    }


    private static void writeToFile(String[] stringToWrite){
        if (!file.exists()) {
            try {
                fileWriter = new FileWriter(file, true);
                writer = new CSVWriter(fileWriter);
                String[] header = {"Info", "Documentation/Evolvability", "Visual Representation", "Structure", "Function"};
                writer.writeNext(header);
                writer.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }

        try {
            fileWriter = new FileWriter(file, true);
            writer = new CSVWriter(fileWriter);
            writer.writeNext(stringToWrite);
            writer.close();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }


    private static int countNumberOfRevisions(List<ReviewDiff> reviews){
        int count = 0;
        for(ReviewDiff r : reviews){
            count = count + r.getRevisions().size();
        }
        return count;
    }


    private static Multimap<String, Modification> groupModificationsPerMonth(List<Modification> modifications){
        Multimap<String, Modification> modificationsPerMonth = ArrayListMultimap.create();
        for(Modification m : modifications) {
            int month = m.getDate().getMonth();
            int year = m.getDate().getYear() + 1900;
            String monthString = Integer.toString(month);
            String separator = "_";
            String yearString = Integer.toString(year);

            String key = monthString.concat(separator).concat(yearString);

            modificationsPerMonth.put(key, m);
        }
        return modificationsPerMonth;
    }


    private static List<Modification> createListOfModificationsToClassify(List<ReviewDiff> reviewDiffList) {
        List<Modification> modifications = new ArrayList<>();
        for(ReviewDiff reviewDiff : reviewDiffList) {
            for(RevisionDiff revisionDiff : reviewDiff.getRevisions()) {
                for(FileDiff fileDiff : revisionDiff.getModificationsPerFile()) {
                    for(Modification m: fileDiff.getModifications()){
                        m.setDate(reviewDiff.getDate());
                        modifications.add(m);
                    }
                }
            }
        }
        return modifications;
    }


    private  static void classifyModificationsCategory(List<Modification> testModifications, Instances trainingData, String info) {
        List<Modification> testEntries = new ModificationConverter().extractFeatures(testModifications, false);
        System.out.println("Test entries: "+testEntries.size() +   " out  of  "+ testModifications.size());
        Instances testData = new ModificationConverter().createCategoryInstances(testEntries);
        System.out.println("test data "+testData.size());
        Classifier classifier = new Classifier(trainingData, testData);
        List<Double> classifiedValues = classifier.runRandomForestWithoutLabels();
        countCategoryOccurrences(classifiedValues, info);
    }


    private  static void classifyModificationsGroup(List<Modification> testModifications, Instances trainingData, String info) {
        List<Modification> testEntries = new ModificationConverter().extractFeatures(testModifications, false);
        System.out.println("Test entries: "+testEntries.size() +   " out  of  "+ testModifications.size());
        Instances testData = new ModificationConverter().createGroupInstances(testEntries);
        System.out.println("test data "+testData.size());
        Classifier classifier = new Classifier(trainingData, testData);
        List<Double> classifiedValues = classifier.runRandomForestWithoutLabels();
        countGroupOccurrences(classifiedValues, info);
    }


    private static List<Modification> loadTestProjectsModifications() {
        List<Modification> modifications = new ArrayList<>();
        modifications.addAll(init(FilesIndexer.AndroidDatasetPath, FilesIndexer.AndroidGerritUrl, FilesIndexer.AndroidFile, true));
        modifications.addAll(init(FilesIndexer.JGitDatasetPath, FilesIndexer.JGitGerritUrl, FilesIndexer.JGitFile, true));
        modifications.addAll(init(FilesIndexer.JavaClientDatasetPath, FilesIndexer.JavaClientGerritUrl, FilesIndexer.JavaClientFile, true));
        return modifications;
    }


    private static List<Modification> init(String datasetPath, String gerritUrl, String csvName, boolean filterImports){
        Dataset dataset = new Dataset(datasetPath);
        Multimap<String, Change>  changes =  dataset.getChanges();
        GerritExtractor ge = new GerritExtractor(changes);
        ge.set(gerritUrl);
        CodeChunksExtractor cce = new CodeChunksExtractor();
        List<CodeChunk> codeChunkList = cce.extractCodeChunks(ge, csvName);

        if(gerritUrl.equals(FilesIndexer.AndroidGerritUrl)){
            codeChunkList = removeNonJavaFiles(codeChunkList);
        }

        if(filterImports){
            ImportFilter importFilter = new ImportFilter();
            codeChunkList = importFilter.filterImports(codeChunkList);
        }

        CodeChunksLinker cl = new CodeChunksLinker();
        return cl.splitChangeBased(codeChunkList);
    }


    private static List<CodeChunk> removeNonJavaFiles(List<CodeChunk> codeChunkList){
        List<CodeChunk> filteredChunksList = new ArrayList<>();
        for(CodeChunk codeChunk : codeChunkList) {
            String fileName = codeChunk.getFileName();
            if(fileName!=null) {
                String[] fileNameSplit = fileName.split("\\.");
                if (fileNameSplit[fileNameSplit.length - 1].equals("java")) {
                    filteredChunksList.add(codeChunk);
                }
            }
        }
        return filteredChunksList;
    }


    private static void countCategoryOccurrences(List<Double> classifiedValues, String info) {
        int evolvabilityCount = 0;
        int functionalCount = 0;

        for (Double value : classifiedValues) {
            if (value == EVOLVABILITYCATCODE) {
                evolvabilityCount++;
            }
            if (value == FUNCTIONALCATCODE) {
                functionalCount++;
            }
        }


        if (!file.exists()) {
            try {
                fileWriter = new FileWriter(file, true);
                writer = new CSVWriter(fileWriter);
                String[] header = {"Info", "Documentation/Evolvability", "Visual Representation", "Structure", "Function"};
                writer.writeNext(header);
                writer.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }

            try {
                fileWriter = new FileWriter(file, true);
                writer = new CSVWriter(fileWriter);
                String[] data = {info, Integer.toString(evolvabilityCount), "", "", Integer.toString(functionalCount)};
                writer.writeNext(data);
                writer.close();
            } catch (IOException e) {
                e.printStackTrace();
            }

        }


    private static void countGroupOccurrences(List<Double> classifiedValues, String info){
        int documentationCount = 0;
        int visualRepCount = 0;
        int structural = 0;
        int functional = 0;

        for(Double value : classifiedValues){
            if(value==DOCUMENTATIONCODE){
                documentationCount++;
            }
            if(value==VISUALREPCODE){
                visualRepCount++;
            }
            if(value==STRUCTURALCODE){
                structural++;
            }
            if(value==FUNCTIONALCODE){
                functional++;
            }
        }

        System.out.println("\n\nResults:  \nDocumentation: " +documentationCount+ "\nVisual representation: "+visualRepCount+
        "\nStructural: "+structural+  "\nFunctional: "+functional);

        if(!file.exists()) {
            try {
                fileWriter = new FileWriter(file, true);
                writer = new CSVWriter(fileWriter);
                String[] header = {"Info", "Documentation", "Visual Representation", "Structure", "Function"};
                writer.writeNext(header);
                writer.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        try {
                fileWriter = new FileWriter(file, true);
                writer = new CSVWriter(fileWriter);
                String[] data = {info, Integer.toString(documentationCount), Integer.toString(visualRepCount), Integer.toString(structural), Integer.toString(functional)};
                writer.writeNext(data);
                writer.close();
            } catch (IOException e) {
                e.printStackTrace();
            }




    }


}
