package backend.db_connectors;

import backend.base.Change;
import backend.base.LineBlock;
import backend.base.taxonomy.Subtype;
import backend.base.taxonomy.Type;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import com.opencsv.CSVReader;

import java.io.IOException;
import java.io.Reader;
import java.nio.file.Files;
import java.nio.file.Paths;

public class DatasetReader {

    private String datasetPath;
    private Multimap<String, Change> changesInDataset;

    public DatasetReader(String datasetPath) {
        this.datasetPath = datasetPath;
        changesInDataset = ArrayListMultimap.create();
    }

    public void extractDefects() throws IOException {
        Reader reader = Files.newBufferedReader(Paths.get(datasetPath));
        try (CSVReader csvReader = new CSVReader(reader, ',', '\'', 1)) {
            String[] nextRecord;
            int count = 0;
            while ((nextRecord = csvReader.readNext()) != null) {
                count++;
                LineBlock b = createLineBlock(nextRecord);
                addChange(nextRecord, b);
            }
        }
        catch(IOException e){
            e.printStackTrace();
        }
    }

    private LineBlock createLineBlock(String[] line){
        int beginLine = Integer.parseInt(line[4]);
        int endLine = Integer.parseInt(line[5]);
        char beginChar = '\0';
        char endChar = '\0';
        if(!line[6].equals("")){
            beginChar = line[6].charAt(0);
        }
        if(!line[7].equals("")){
            endChar = line[7].charAt(0);
        }
        return new LineBlock(beginLine, endLine, beginChar, endChar);
    }

    private void addChange(String[] record, LineBlock lineBlock){
        Change.Category category = initializeCategory(record[8]);
        Type type = initializeType(record[9]);
        Subtype subtype = initializeSubType(record[10]);
        Change.Triggered triggered = initializeTriggered(record[11]);
        changesInDataset.put(record[0], new Change(record[0], record[1], record[2],
                record[3], lineBlock, category, type, subtype, triggered));
    }

    private Change.Category initializeCategory(String category){
        if(category.equals("Evolvability changes")) {
            return Change.Category.EVOLVABILITY;
        }
        if(category.equals("Functional changes")) {
            return  Change.Category.FUNCTIONAL;
        }
        return Change.Category.UNKNOWN;
    }

    private Change.Triggered initializeTriggered(String triggered){
        if(triggered.equals("Triggered by comment")){
            return Change.Triggered.TRIGGERED;
        }
        if(triggered.equals("Undocumented")){
            return Change.Triggered.UNDOCUMENTED;
        }
        return Change.Triggered.UNKNOWN;
    }

    private Type initializeType(String type){
        if(type.equals("Textual")) {
            return Type.TEXTUAL;
        }
        if(type.equals("Supported by language")) {
            return Type.SUPPORTEDBYLANGUAGE;
        }
        if(type.equals("Visual representation")) {
            return Type.VISUALREPRESENTATION;
        }
        if(type.equals("Organization")){
            return Type.ORGANIZATION;
        }
        if(type.equals("Solution approach")) {
            return Type.SOLUTIONAPPROACH;
        }
        if(type.equals("Resource")) {
            return Type.RESOURCE;
        }
        if(type.equals("Check")) {
            return Type.CHECK;
        }
        if(type.equals("Interface")) {
            return Type.INTERFACE;
        }
        if(type.equals("Logic")) {
            return Type.LOGIC;
        }
        if(type.equals("Larger defects")) {
            return Type.LARGER;
        }
        if(type.equals("Support")){
            return Type.SUPPORT;
        }
        return Type.UNKNOWN;
    }


    private Subtype initializeSubType(String subtype) {
        if (subtype.equals("Naming")) {
            return Subtype.NAMING;
        }
        if (subtype.equals("Comments")) {
            return Subtype.COMMENTS;
        }
        if (subtype.equals("Debug Info")) {
            return Subtype.DEBUGINFO;
        }
        if (subtype.equals("Element Type")) {
            return Subtype.ELEMENTTYPE;
        }
        if (subtype.equals("Immutable")) {
            return Subtype.IMMUTABLE;
        }
        if (subtype.equals("Visibility")) {
            return Subtype.VISIBILITY;
        }
        if (subtype.equals("Void Parameter")) {
            return Subtype.VOIDPARAMETER;
        }
        if (subtype.equals("Element Reference")) {
            return Subtype.ELEMENTREFERENCE;
        }
        if (subtype.equals("Bracket Usage")) {
            return Subtype.BRACKETUSAGE;
        }
        if (subtype.equals("Indentation")) {
            return Subtype.INDENTATION;
        }
        if (subtype.equals("The Blank Line Usage")) {
            return Subtype.BLANKLINE;
        }
        if (subtype.equals("Long Line")) {
            return Subtype.LONGLINE;
        }
        if (subtype.equals("Space Usage")) {
            return Subtype.SPACEUSAGE;
        }
        if (subtype.equals("Grouping")) {
            return Subtype.GROUPING;
        }
        if (subtype.equals("Move Functionality")) {
            return Subtype.MOVEFUNCTIONALITY;
        }
        if(subtype.equals("Long Sub-routine")){
            return Subtype.LONGSUBROUTINE;
        }
        if(subtype.equals("Dead Code")){
            return Subtype.DEADCODE;
        }
        if(subtype.equals("Duplication")){
            return Subtype.DUPLICATION;
        }
        if(subtype.equals("Complex Code")){
            return Subtype.COMPLEXCODE;
        }
        if(subtype.equals("Statement Issues")){
            return Subtype.STATEMENTISSUES;
        }
        if(subtype.equals("Consistency")){
            return Subtype.CONSISTENCY;
        }
        if(subtype.equals("Semantic Duplication")){
            return Subtype.SEMANTICDUPLICATION;
        }
        if(subtype.equals("Semantic Dead Code")){
            return Subtype.SEMANTICDEADCODE;
        }
        if(subtype.equals("Change Function")){
            return Subtype.CHANGEFUNCTION;
        }
        if(subtype.equals("Use Standard Method")){
            return Subtype.USESTANDARDMETHOD;
        }
        if(subtype.equals("New Functionality")){
            return Subtype.NEWFUNCTIONALITY;
        }
        if(subtype.equals("Minor")){
            return Subtype.MINOR;
        }
        if(subtype.equals("Variable Initialization")){
            return Subtype.VARIABLEINITIALIZATION;
        }
        if(subtype.equals("Memory management")){
            return Subtype.MEMORYMANAGEMENT;
        }
        if(subtype.equals("Data and Resource Manipulation")){
            return Subtype.DATA_RESOURCEMANIPULATION;
        }
        if(subtype.equals("Check Function")){
            return Subtype.CHECKFUNCTION;
        }
        if(subtype.equals("Check Variable")){
            return Subtype.CHECKVARIABLE;
        }
        if(subtype.equals("Check User Input")){
            return Subtype.CHECKUSERINPUT;
        }
        if(subtype.equals("Function Call")){
            return Subtype.FUNCTIONCALL;
        }
        if(subtype.equals("Parameter")){
            return Subtype.PARAMETER;
        }
        if(subtype.equals("Compare")){
            return Subtype.COMPARE;
        }
        if(subtype.equals("Compute")){
            return Subtype.COMPUTE;
        }
        if(subtype.equals("Wrong Location")){
            return Subtype.WRONGLOCATION;
        }
        if(subtype.equals("Algorithm")){
            return Subtype.ALGORITHM_PERFORMANCE;
        }
        if(subtype.equals("Completeness")){
            return Subtype.COMPLETENESS;
        }
        if(subtype.equals("GUI")){
            return Subtype.GUI;
        }
        if(subtype.equals("Check outside code")){
            return Subtype.CHECKOUTCODE;
        }
        if(subtype.equals("")){
            return Subtype.UNKNOWN;
        }
        return Subtype.OTHER;
    }

    public Multimap<String, Change> getChangesInDataset(){
        return changesInDataset;
    }

}
