package backend.base.graphs;

import backend.analysis.CodeChunksLinker;
import backend.analysis.GerritExtractor;
import backend.analysis.ImportFilter;
import backend.analysis.machineLearning.ModificationConverter;
import backend.base.Change;
import backend.base.gerrit_data.CodeChunk;
import backend.base.gerrit_data.Modification;
import backend.db_connectors.Dataset;
import backend.db_connectors.FilesIndexer;
import backend.main.CodeChunksExtractor;
import com.google.common.collect.Multimap;
import org.jfree.chart.ChartFactory;
import org.jfree.chart.ChartPanel;
import org.jfree.chart.JFreeChart;
import org.jfree.chart.axis.CategoryAxis;
import org.jfree.chart.axis.NumberAxis;
import org.jfree.chart.plot.CategoryPlot;
import org.jfree.chart.plot.DatasetRenderingOrder;
import org.jfree.chart.plot.PlotOrientation;
import org.jfree.chart.renderer.category.LineAndShapeRenderer;
import org.jfree.chart.title.TextTitle;
import org.jfree.data.DataUtilities;
import org.jfree.data.DefaultKeyedValues;
import org.jfree.data.KeyedValues;
import org.jfree.data.category.CategoryDataset;
import org.jfree.data.general.DatasetUtilities;
import org.jfree.ui.ApplicationFrame;
import org.jfree.ui.RefineryUtilities;
import org.jfree.util.SortOrder;

import java.awt.*;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.List;

public class ParetoPlotter extends ApplicationFrame {

    /**
     * Constructs a new application frame.
     *
     * @param title the frame title.
     */
    public ParetoPlotter(String title, List<Modification> dataToPlot, String granularity) {
        super(title);

        final DefaultKeyedValues data = new DefaultKeyedValues();

        if(granularity.equals("Type")) {
            int[] dataTypes = countTypeOccurrences(dataToPlot);
            data.addValue("Textual", dataTypes[0]);
            data.addValue("Supported by language", dataTypes[1]);
            data.addValue("Visual representation", dataTypes[2]);
            data.addValue("Organization", dataTypes[3]);
            data.addValue("Solution Approach", dataTypes[4]);
            data.addValue("Resource", dataTypes[5]);
            data.addValue("Check", dataTypes[6]);
            data.addValue("Interface", dataTypes[7]);
            data.addValue("Logic", dataTypes[8]);
            data.addValue("Support", dataTypes[9]);
            data.addValue("Larger", dataTypes[10]);
        }

        if(granularity.equals("Group")){
            int[] dataGroups = countGroupOccurrences(dataToPlot);

            System.out.println("docum  "+dataGroups[0]);
            System.out.println("visual  "+dataGroups[1]);
            System.out.println("structure "+dataGroups[2]);
            System.out.println("func  "+dataGroups[3]);

            data.addValue("Documentation", dataGroups[0]);
            data.addValue("Visual", dataGroups[1]);
            data.addValue("Structure", dataGroups[2]);
            data.addValue("Functional", dataGroups[3]);
        }

        if(granularity.equals("Decomp")){
            int[] dataGroups = countDecompGroupOccurrences(dataToPlot);
            data.addValue("Documentation", dataGroups[0]);
            data.addValue("Visual", dataGroups[1]);
            data.addValue("Structure", dataGroups[2]);
            data.addValue("Resource", dataGroups[3]);
            data.addValue("Check", dataGroups[4]);
            data.addValue("Interface", dataGroups[5]);
            data.addValue("Logic", dataGroups[6]);
            data.addValue("Support", dataGroups[7]);
            data.addValue("Larger", dataGroups[8]);
        }

        data.sortByValues(SortOrder.DESCENDING);
        final KeyedValues cumulative = DataUtilities.getCumulativePercentages(data);
        final CategoryDataset dataset = DatasetUtilities.createCategoryDataset("Groups", data);

        // create the chart...
        final JFreeChart chart = ChartFactory.createBarChart(
                "Dataset modifications",  // chart title
                "Group",                     // domain axis label
                "Amount",                     // range axis label
                dataset,                        // data
                PlotOrientation.VERTICAL,
                true,                           // include legend
                true,
                false
        );

        // NOW DO SOME OPTIONAL CUSTOMISATION OF THE CHART...
        chart.addSubtitle(new TextTitle("By groups"));

        // set the background color for the chart...
        chart.setBackgroundPaint(new Color(0xBBBBDD));

        // get a reference to the plot for further customisation...
        final CategoryPlot plot = chart.getCategoryPlot();

        final CategoryAxis domainAxis = plot.getDomainAxis();
        domainAxis.setLowerMargin(0.02);
        domainAxis.setUpperMargin(0.02);

        // set the range axis to display integers only...
        final NumberAxis rangeAxis = (NumberAxis) plot.getRangeAxis();
        rangeAxis.setStandardTickUnits(NumberAxis.createIntegerTickUnits());

        final LineAndShapeRenderer renderer2 = new LineAndShapeRenderer();

        final CategoryDataset dataset2 = DatasetUtilities.createCategoryDataset(
                "Cumulative", cumulative
        );
        final NumberAxis axis2 = new NumberAxis("Percent");
        axis2.setNumberFormatOverride(NumberFormat.getPercentInstance());
        plot.setRangeAxis(1, axis2);
        plot.setDataset(1, dataset2);
        plot.setRenderer(1, renderer2);
        plot.mapDatasetToRangeAxis(1, 1);

        plot.setDatasetRenderingOrder(DatasetRenderingOrder.REVERSE);
        // OPTIONAL CUSTOMISATION COMPLETED.

        // add the chart to a panel...
        final ChartPanel chartPanel = new ChartPanel(chart);
        chartPanel.setPreferredSize(new java.awt.Dimension(550, 270));
        setContentPane(chartPanel);


    }


    private int[] countGroupOccurrences(List<Modification> modifications){
        int[] typesCounter = new int[11];
        for(int i =0; i<4; i++){
            typesCounter[i] = 0;
        }
        for(Modification m : modifications) {
            if (m.getFeatures().group != null) {
                if (m.getFeatures().group.equals("DOCUMENTATION")) {
                    typesCounter[0] = typesCounter[0] + 1;
                }
                if (m.getFeatures().group.equals("VISUALREPRESENTATION")) {
                    typesCounter[1] = typesCounter[1] + 1;
                }
                if (m.getFeatures().group.equals("STRUCTURE")) {
                    typesCounter[2] = typesCounter[2] + 1;
                }
                if (m.getFeatures().group.equals("FUNCTIONAL")) {
                    typesCounter[3] = typesCounter[3] + 1;
                }
            }
        }
        return typesCounter;
    }


    private int[] countDecompGroupOccurrences(List<Modification> modifications){
        int[] typesCounter = new int[9];
        for(int i =0; i<9; i++){
            typesCounter[i] = 0;
        }
        for(Modification m : modifications) {
            if (m.getFeatures().decomposedGroup != null) {
                if (m.getFeatures().decomposedGroup.equals("DOCUMENTATION")) {
                    typesCounter[0] = typesCounter[0] + 1;
                }
                if (m.getFeatures().decomposedGroup.equals("VISUALREPRESENTATION")) {
                    typesCounter[1] = typesCounter[1] + 1;
                }
                if (m.getFeatures().decomposedGroup.equals("STRUCTURE")) {
                    typesCounter[2] = typesCounter[2] + 1;
                }
                if (m.getFeatures().decomposedGroup.equals("RESOURCE")) {
                    typesCounter[3] = typesCounter[3] + 1;
                }
                if (m.getFeatures().decomposedGroup.equals("CHECK")) {
                    typesCounter[4] = typesCounter[4] + 1;
                }
                if (m.getFeatures().decomposedGroup.equals("INTERFACE")) {
                    typesCounter[5] = typesCounter[5] + 1;
                }
                if (m.getFeatures().decomposedGroup.equals("LOGIC")) {
                    typesCounter[6] = typesCounter[6] + 1;
                }
                if (m.getFeatures().decomposedGroup.equals("SUPPORT")) {
                    typesCounter[7] = typesCounter[7] + 1;
                }
                if (m.getFeatures().decomposedGroup.equals("LARGER")) {
                    typesCounter[8] = typesCounter[8] + 1;
                }
            }
        }
        return typesCounter;
    }


    private int[] countTypeOccurrences(List<Modification> modifications){
        int[] typesCounter = new int[11];
        for(int i =0; i<typesCounter.length; i++){
            typesCounter[i] = 0;
        }
        for(Modification m : modifications) {
            if (m.getFeatures().type != null) {
                if (m.getFeatures().type.equals(backend.base.taxonomy.Type.TEXTUAL.toString())) {
                    typesCounter[0] = typesCounter[0] + 1;
                }
                if (m.getFeatures().type.equals(backend.base.taxonomy.Type.SUPPORTEDBYLANGUAGE.toString())) {
                    typesCounter[1] = typesCounter[1] + 1;
                }
                if (m.getFeatures().type.equals(backend.base.taxonomy.Type.VISUALREPRESENTATION.toString())) {
                    typesCounter[2] = typesCounter[2] + 1;
                }
                if (m.getFeatures().type.equals(backend.base.taxonomy.Type.ORGANIZATION.toString())) {
                    typesCounter[3] = typesCounter[3] + 1;
                }
                if (m.getFeatures().type.equals(backend.base.taxonomy.Type.SOLUTIONAPPROACH.toString())) {
                    typesCounter[4] = typesCounter[4] + 1;
                }
                if (m.getFeatures().type.equals(backend.base.taxonomy.Type.RESOURCE.toString())) {
                    typesCounter[5] = typesCounter[5] + 1;
                }
                if (m.getFeatures().type.equals(backend.base.taxonomy.Type.CHECK.toString())) {
                    typesCounter[6] = typesCounter[6] + 1;
                }
                if (m.getFeatures().type.equals(backend.base.taxonomy.Type.INTERFACE.toString())) {
                    typesCounter[7] = typesCounter[7] + 1;
                }
                if (m.getFeatures().type.equals(backend.base.taxonomy.Type.LOGIC.toString())) {
                    typesCounter[8] = typesCounter[8] + 1;
                }
                if (m.getFeatures().type.equals(backend.base.taxonomy.Type.SUPPORT.toString())) {
                    typesCounter[9] = typesCounter[9] + 1;
                }
                if (m.getFeatures().type.equals(backend.base.taxonomy.Type.LARGER.toString())) {
                    typesCounter[10] = typesCounter[10] + 1;
                }
            }
        }
        return typesCounter;
    }


    public static void main(String args[]) {
        List<Modification> modificationsToPlot = prepareModifications();
        ParetoPlotter paretoPlotter = new ParetoPlotter("Review changes groups", modificationsToPlot, "Group");
        paretoPlotter.pack();
        RefineryUtilities.centerFrameOnScreen(paretoPlotter);
        paretoPlotter.setVisible(true);
    }


    private static List<Modification> init(String datasetPath, String gerritUrl, String csvName, boolean filterImports){

        Dataset dataset = new Dataset(datasetPath);
        Multimap<String, Change> changes; changes =  dataset.getChanges();
        GerritExtractor ge = new GerritExtractor(changes);
        ge.set(gerritUrl);
        CodeChunksExtractor cce = new CodeChunksExtractor();
        List<CodeChunk> codeChunkList = cce.extractCodeChunks(ge, csvName);

        if(gerritUrl.equals(FilesIndexer.AndroidGerritUrl)){
            codeChunkList = removeNonJavaFiles(codeChunkList);
        }

        if(filterImports){
            ImportFilter importFilter = new ImportFilter();
            codeChunkList = importFilter.filterImports(codeChunkList);
        }

        CodeChunksLinker cl = new CodeChunksLinker();
        return cl.splitChangeBased(codeChunkList);
    }


    private static List<CodeChunk> removeNonJavaFiles(List<CodeChunk> codeChunkList){
        List<CodeChunk> filteredChunksList = new ArrayList<>();
        for(CodeChunk codeChunk : codeChunkList) {
            String fileName = codeChunk.getFileName();
            if(fileName!=null) {
                String[] fileNameSplit = fileName.split("\\.");
                if (fileNameSplit[fileNameSplit.length - 1].equals("java")) {
                    filteredChunksList.add(codeChunk);
                }
            }
        }
        return filteredChunksList;
    }


    private static List<Modification> prepareModifications() {
        List<Modification> modifications = new ArrayList<>();
        modifications.addAll(init(FilesIndexer.AndroidDatasetPath, FilesIndexer.AndroidGerritUrl, FilesIndexer.AndroidFile, true));
        modifications.addAll(init(FilesIndexer.JGitDatasetPath, FilesIndexer.JGitGerritUrl, FilesIndexer.JGitFile, true));
        modifications.addAll(init(FilesIndexer.JavaClientDatasetPath, FilesIndexer.JavaClientGerritUrl, FilesIndexer.JavaClientFile, true));
        System.out.println("Modifications: "+modifications.size());
        return new ModificationConverter().extractFeatures(modifications, true);
    }


}
