package backend.base.graphs;

import backend.analysis.CodeChunksLinker;
import backend.analysis.GerritExtractor;
import backend.analysis.ImportFilter;
import backend.analysis.machineLearning.ModificationConverter;
import backend.base.Change;
import backend.base.gerrit_data.CodeChunk;
import backend.base.gerrit_data.Modification;
import backend.db_connectors.Dataset;
import backend.db_connectors.FilesIndexer;
import backend.main.CodeChunksExtractor;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import org.jfree.chart.ChartFactory;
import org.jfree.chart.ChartPanel;
import org.jfree.chart.JFreeChart;
import org.jfree.data.statistics.*;
import org.jfree.ui.ApplicationFrame;
import org.jfree.ui.RefineryUtilities;

import java.awt.*;
import java.util.ArrayList;
import java.util.List;

public class BoxPlotPlotter extends ApplicationFrame {

    public enum FeaturesGroupKey {

        NEWLOC,
        OLDLOC,
        NEWCHAR,
        OLDCHAR,
        OTHER,
    }


    private Multimap<String, Modification> modificationMultimap;

    public BoxPlotPlotter(String title, Multimap<String, Modification> modificationMultimap, FeaturesGroupKey featuresGroupKey){
        super(title);
        this.modificationMultimap = modificationMultimap;
        final BoxAndWhiskerCategoryDataset dataset = createDataset(featuresGroupKey);
        final JFreeChart chart = createChart(dataset);

        final ChartPanel chartPanel = new ChartPanel(chart);
        chartPanel.setPreferredSize(new java.awt.Dimension(700, 400));
        setContentPane(chartPanel);
    }


    private BoxAndWhiskerCategoryDataset createDataset(FeaturesGroupKey featuresGroupKey) {
        DefaultBoxAndWhiskerCategoryDataset dataset = new
                DefaultBoxAndWhiskerCategoryDataset();

        if(featuresGroupKey.equals(FeaturesGroupKey.NEWLOC)) {
            for (String key : modificationMultimap.keySet()) {
                List newLOCValues = new ArrayList();
                List newLOCBlankValues = new ArrayList();
                List newLOCCommentsValues = new ArrayList();
                List newLOCExecValues = new ArrayList();
                for (Modification m : modificationMultimap.get(key)) {
                    newLOCValues.add(m.getFeatures().newLOC);
                    newLOCBlankValues.add(m.getFeatures().newLOCBlank);
                    newLOCCommentsValues.add(m.getFeatures().newLOCComments);
                    newLOCExecValues.add(m.getFeatures().newLOCExec);

                }
                dataset.add(newLOCValues, key, "new LOC");
                dataset.add(newLOCBlankValues, key, "new blank LOC");
                dataset.add(newLOCCommentsValues, key, "new com LOC");
                dataset.add(newLOCExecValues, key, "new exec LOC");
            }
        }

        if(featuresGroupKey.equals(FeaturesGroupKey.OLDLOC)) {
            for (String key : modificationMultimap.keySet()) {
                List oldLOCValues = new ArrayList();
                List oldLOCBlankValues = new ArrayList();
                List oldLOCCommentsValues = new ArrayList();
                List oldLOCExecValues = new ArrayList();
                for (Modification m : modificationMultimap.get(key)) {
                    oldLOCValues.add(m.getFeatures().oldLOC);
                    oldLOCBlankValues.add(m.getFeatures().oldLOCBlank);
                    oldLOCCommentsValues.add(m.getFeatures().oldLOCComments);
                    oldLOCExecValues.add(m.getFeatures().oldLOCExec);

                }
                dataset.add(oldLOCValues, key, "old LOC");
                dataset.add(oldLOCBlankValues, key, "old blank LOC");
                dataset.add(oldLOCCommentsValues, key, "old com LOC");
                dataset.add(oldLOCExecValues, key, "old exec LOC");
            }
        }

        if(featuresGroupKey.equals(FeaturesGroupKey.NEWCHAR)) {
            for (String key : modificationMultimap.keySet()) {
                List addedWordsValues = new ArrayList();
                List delWordsValues = new ArrayList();
                List addedCharValues = new ArrayList();
                List delCharValues = new ArrayList();
                for (Modification m : modificationMultimap.get(key)) {
                    addedWordsValues.add(m.getFeatures().numberOfAddedWords);
                    delWordsValues.add(m.getFeatures().numberOfDeletedWords);
                    addedCharValues.add(m.getFeatures().numberOfAddedCharacters);
                    delCharValues.add(m.getFeatures().numberOfDeletedCharacters);
                }
                dataset.add(addedWordsValues, key, "added words");
                dataset.add(delWordsValues, key, "deleted words");
                dataset.add(addedCharValues, key, "added chars");
                dataset.add(delCharValues, key, "deleted chars");
            }
        }

        if(featuresGroupKey.equals(FeaturesGroupKey.OLDCHAR)) {
            for (String key : modificationMultimap.keySet()) {
                List oldCycValues = new ArrayList();
                List newCycValues = new ArrayList();
                List numKeywordsValues = new ArrayList();
                List levDistanceValues = new ArrayList();
                List oneIfValues = new ArrayList();
                for (Modification m : modificationMultimap.get(key)) {
                    oldCycValues.add(m.getFeatures().oldCyclomaticComplexity);
                    newCycValues.add(m.getFeatures().newCyclomaticComplexity);
                    numKeywordsValues.add(m.getFeatures().numKeywords);
                    levDistanceValues.add(m.getFeatures().levenshteinDistance);
                    oneIfValues.add(m.getFeatures().ifAdded);
                }
                dataset.add(oldCycValues, key, "old Cyclomatic");
                dataset.add(newCycValues, key, "new Cyclomatic");
                dataset.add(numKeywordsValues, key, "number of keywords");
                dataset.add(levDistanceValues, key, "Levenshtein distance");
                dataset.add(oneIfValues, key, "if values");
            }
        }

        return dataset;
    }


    private JFreeChart createChart(final BoxAndWhiskerCategoryDataset dataset) {
        JFreeChart chart = ChartFactory.createBoxAndWhiskerChart(
                "Box and Whisker Chart", "Time", "Value", dataset, true);
        chart.setBackgroundPaint(new Color(249, 231, 236));
        return chart;
    }


    public static void main(final String[] args) {
        Multimap<String, Modification> modificationMultimap = prepareModificationsPerProject();
        final BoxPlotPlotter demo = new BoxPlotPlotter("", modificationMultimap, FeaturesGroupKey.NEWLOC);
        demo.pack();
        RefineryUtilities.centerFrameOnScreen(demo);
        demo.setVisible(true);

        final BoxPlotPlotter demo2 = new BoxPlotPlotter("", modificationMultimap, FeaturesGroupKey.OLDLOC);
        demo2.pack();
        RefineryUtilities.centerFrameOnScreen(demo2);
        demo2.setVisible(true);

        final BoxPlotPlotter demo3 = new BoxPlotPlotter("", modificationMultimap, FeaturesGroupKey.NEWCHAR);
        demo3.pack();
        RefineryUtilities.centerFrameOnScreen(demo3);
        demo3.setVisible(true);

        final BoxPlotPlotter demo4 = new BoxPlotPlotter("", modificationMultimap, FeaturesGroupKey.OLDCHAR);
        demo4.pack();
        RefineryUtilities.centerFrameOnScreen(demo4);
        demo4.setVisible(true);
    }

    private static List<Modification> init(String datasetPath, String gerritUrl, String csvName, boolean filterImports){

        Dataset dataset = new Dataset(datasetPath);
        Multimap<String, Change> changes; changes =  dataset.getChanges();
        GerritExtractor ge = new GerritExtractor(changes);
        ge.set(gerritUrl);
        CodeChunksExtractor cce = new CodeChunksExtractor();
        List<CodeChunk> codeChunkList = cce.extractCodeChunks(ge, csvName);

        if(gerritUrl.equals(FilesIndexer.AndroidGerritUrl)){
            codeChunkList = removeNonJavaFiles(codeChunkList);
        }

        if(filterImports){
            ImportFilter importFilter = new ImportFilter();
            codeChunkList = importFilter.filterImports(codeChunkList);
        }

        CodeChunksLinker cl = new CodeChunksLinker();
        return cl.splitChangeBased(codeChunkList);
    }


    private static List<CodeChunk> removeNonJavaFiles(List<CodeChunk> codeChunkList){
        List<CodeChunk> filteredChunksList = new ArrayList<>();
        for(CodeChunk codeChunk : codeChunkList) {
            String fileName = codeChunk.getFileName();
            if(fileName!=null) {
                String[] fileNameSplit = fileName.split("\\.");
                if (fileNameSplit[fileNameSplit.length - 1].equals("java")) {
                    filteredChunksList.add(codeChunk);
                }
            }
        }
        return filteredChunksList;
    }


    private static Multimap<String, Modification> prepareModificationsPerProject() {
        Multimap<String, Modification> modifications = ArrayListMultimap.create();

        List<Modification> androidModifications = init(FilesIndexer.AndroidDatasetPath, FilesIndexer.AndroidGerritUrl, FilesIndexer.AndroidFile, false);
        List<Modification> androidClassifierEntries = new ModificationConverter().extractFeatures(androidModifications, true);
        for(Modification m : androidClassifierEntries) {
            modifications.put("Android", m);
        }

        List<Modification> jgitModifications = init(FilesIndexer.JGitDatasetPath, FilesIndexer.JGitGerritUrl, FilesIndexer.JGitFile, false);
        List<Modification> jgitClassifierEntries = new ModificationConverter().extractFeatures(jgitModifications, true);
        for(Modification m : jgitClassifierEntries) {
            modifications.put("JGit", m);
        }

        List<Modification> javaClientModifications = init(FilesIndexer.JavaClientDatasetPath, FilesIndexer.JavaClientGerritUrl, FilesIndexer.JavaClientFile, false);
        List<Modification> javaClientClassifierEntries = new ModificationConverter().extractFeatures(javaClientModifications, true);
        for(Modification m : javaClientClassifierEntries){
            modifications.put("JavaClient", m);
        }

        return modifications;
    }

}
