package backend.analysis.machineLearning;

import backend.analysis.CodeChunksLinker;
import backend.base.gerrit_data.CodeChunk;
import backend.base.gerrit_data.Modification;
import weka.attributeSelection.*;
import weka.classifiers.CostMatrix;
import weka.classifiers.Evaluation;
import weka.classifiers.bayes.NaiveBayes;
import weka.classifiers.evaluation.NominalPrediction;
import weka.classifiers.meta.CostSensitiveClassifier;
import weka.classifiers.meta.FilteredClassifier;
import weka.classifiers.meta.GridSearch;
import weka.classifiers.trees.J48;
import weka.classifiers.trees.RandomForest;
import weka.core.*;
import weka.core.converters.ConverterUtils.DataSource;
import weka.filters.Filter;
import weka.filters.MultiFilter;
import weka.filters.supervised.instance.Resample;
import weka.filters.supervised.instance.SMOTE;
import weka.filters.supervised.instance.SpreadSubsample;
import weka.filters.unsupervised.attribute.Normalize;
import weka.filters.unsupervised.attribute.Remove;
import weka.gui.treevisualizer.PlaceNode2;
import weka.gui.treevisualizer.TreeVisualizer;

import javax.swing.*;
import java.io.FileInputStream;
import java.io.ObjectInputStream;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;

public class Classifier {

    private Instances data;

    private Instances trainingData;
    private Instances testData;

    private AttributeSelection attSelect;


    public Classifier(String path){
        try{DataSource dataSource = new DataSource(path);
            data = dataSource.getDataSet();
            System.out.println(data.numInstances() + " instances loaded.");
            this.data = setAttributeSelection(data);
        }catch (Exception e){
            System.out.println("Problem in Initialization. Check the path.");
            e.printStackTrace();
        }

    }

    public Classifier(Instances data){
        Normalize normalize = new Normalize();
        try {
            normalize.setInputFormat(data);
            this.data = Filter.useFilter(data, normalize);
        } catch (Exception e) {
            e.printStackTrace();
        }
        System.out.println(this.data.numInstances() + " instances loaded.");
        System.out.println(this.data.get(1).toString(1));
        System.out.println(this.data.get(1).toString(2));
        //showFeaturesRank();
        this.data = setAttributeSelection(data);
    }


    public Classifier(Instances trainingData, Instances testData){
        Normalize normalize = new Normalize();
        try {
            normalize.setInputFormat(trainingData);
            this.trainingData = Filter.useFilter(trainingData, normalize);
            this.testData = Filter.useFilter(testData, normalize);

        } catch (Exception e) {
            e.printStackTrace();
       }

        //this.trainingData = setAttributeSelection(trainingData);
    }

    /**
     * Creates and sets the parameters of a Resample filter.
     * @return instance of Resample
     */

    private Resample createResampleFilter() {
        Resample resample = new Resample();
        resample.setNoReplacement(true);
        resample.setBiasToUniformClass(1);
        resample.setSampleSizePercent(10);
        return resample;
    }


    public void filterData(String[] opt){
        try{
            Remove remove = new Remove();
            remove.setOptions(opt);
            remove.setInputFormat(data);
            data = Filter.useFilter(data, remove);
            setAttributeSelection(data);
        }catch (Exception e){
            System.out.println("Problem in Data Filtering. Check option for filtering");
            e.printStackTrace();
        }
    }

    public void filterTrainingData(){
        try{
            trainingData.setClassIndex(testData.numAttributes()-1);
            //setAttributeSelection(trainingData);
        }catch (Exception e){
            System.out.println("Problem in Data Filtering. Check option for filtering");
            e.printStackTrace();
        }
    }

    public void filterTrainingDataWithSMOTE(){
        try{
            System.out.println("Before smote " +trainingData.numInstances());
            trainingData.setClassIndex(trainingData.numAttributes()-1);
            testData.setClassIndex(testData.numAttributes()-1);
            SMOTE smote = new SMOTE();
            smote.setOptions(new String[]{"-P","100"}); // change this parameter
            smote.setInputFormat(trainingData);
            trainingData = Filter.useFilter(trainingData, smote);
            System.out.println("After smote: "+ trainingData.numInstances());
        }catch (Exception e){
            System.out.println("Problem in Data Filtering. Check option for filtering");
            e.printStackTrace();
        }
    }

    public void filterTestData(){
        try{
            testData.setClassIndex(testData.numAttributes()-1);
            //setAttributeSelection(testData);
        }catch (Exception e){
            System.out.println("Problem in Data Filtering. Check option for filtering");
            e.printStackTrace();
        }
    }

    public Instances setAttributeSelection(Instances data){
        try {
            //GainRatioAttributeEval evalGain = new GainRatioAttributeEval();
            CorrelationAttributeEval corrEval = new CorrelationAttributeEval();
            //WrapperSubsetEval wrapper = new WrapperSubsetEval();
            //System.out.println("num attrib  "+data.numAttributes());
            data.setClassIndex(data.numAttributes() - 1);
            //wrapper.setClassifier(new RandomForest());
            //wrapper.buildEvaluator(data);
            Ranker search = new Ranker();
            //search.setOptions(new String[] { "-T", "0.001" });
            search.setOptions(new String[] { "-T", "0.005" });

            attSelect = new AttributeSelection();
            //attSelect.setEvaluator(wrapper);
            //attSelect.setEvaluator(evalGain);
            attSelect.setEvaluator(corrEval);
            attSelect.setSearch(search);

            //attSelect.setSearch(new BestFirst());

            attSelect.SelectAttributes(data);

            System.out.println(attSelect.toResultsString());

            System.out.println("Ending test!");

            int[] indices = attSelect.selectedAttributes();
            System.out.println("\nFeature Rank:\n");
            System.out.println(Utils.arrayToString(indices));
            System.out.println("\nFeature and score Rank:\n");
            double[][] scores = attSelect.rankedAttributes();
            for(int i=0; i<scores.length; i++) {
                for(int j=0; j<scores[i].length; j++) {
                   System.out.println("Variable: "+scores[i]+ "  score: "+scores[i][j]);
                }
            }
            return attSelect.reduceDimensionality(data);
        } catch (Exception e) {
            System.out.println("Problem in Attribute Selection");
            e.printStackTrace();
        }
        return null;
    }

    public void showFeaturesRank(){
        try{
            GainRatioAttributeEval evalGain = new GainRatioAttributeEval();
            InfoGainAttributeEval eval = new InfoGainAttributeEval();
            Ranker search = new Ranker();
            search.setOptions(new String[] { "-T", "0.01" });
            attSelect.setEvaluator(evalGain);
            attSelect.setSearch(search);

            int[] indices = attSelect.selectedAttributes();
            System.out.println("\nFeature Rank:\n");
            System.out.println(Utils.arrayToString(indices));
            System.out.println("\nFeature and score Rank:\n");
            double[][] scores = attSelect.rankedAttributes();
            for(int i=0; i<scores.length; i++) {
                for(int j=0; j<scores[i].length; j++) {
                    System.out.println("Variable: "+scores[i]+ "  score: "+scores[i][j]);
                }
            }
        }catch (Exception e){
            System.out.println("Problem in Feature ranking.");
            e.printStackTrace();
        }
    }


    public void computeAttributeCorrelation(Instances data) {
        CorrelationAttributeEval correlationAttributeEval = new CorrelationAttributeEval();
        try {
            correlationAttributeEval.buildEvaluator(data);
            for(int i=0; i< data.numAttributes(); i++){
                System.out.println(i+"  Eval attribute: " +correlationAttributeEval.evaluateAttribute(i));
            }
        } catch (Exception e) {
            e.printStackTrace();
        }

    }


    public void visualizeTree(Drawable tree){
        try{
            TreeVisualizer tv = new TreeVisualizer(null, tree.graph(), new PlaceNode2());
            JFrame frame = new javax.swing.JFrame("Tree Visualizer");
            frame.setSize(800, 500);
            frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
            frame.getContentPane().add(tv);
            frame.setVisible(true);
            tv.fitToScreen();
        }catch(Exception e){
            e.printStackTrace();
        }

    }

    public void runJ48(){
        String[] options = new String[1];
        options[0] = "-U";
        J48 tree = new J48();
        Random random = new Random(1);
        try {
            tree.setOptions(options);
            tree.buildClassifier(data);
            System.out.println(tree);
            Evaluation eval = new Evaluation(data);
            eval.crossValidateModel(tree, data, 10, random);
            visualizeTree(tree);
            System.out.println(eval.toSummaryString());
            System.out.println(eval.toMatrixString());

            System.out.println("Weighted MCC:   " +eval.weightedMatthewsCorrelation());

        } catch (Exception e) {
            e.printStackTrace();
        }
    }


    public List<ClassifierResult> runJ48WithTestsSet(boolean isGroup) {
        List<ClassifierResult> results = new ArrayList<>();
        String[] options = new String[1];
        options[0] = "-U";
        J48 tree = new J48();
        try {
            tree.setOptions(options);
            tree.buildClassifier(trainingData);
            Evaluation eval = new Evaluation(trainingData);
            eval.evaluateModel(tree, testData);

            if(isGroup) {
                results.add(new ClassifierResult("J48","Documentation", eval.precision(0), eval.recall(0), eval.fMeasure(0),
                        eval.areaUnderROC(0), eval.matthewsCorrelationCoefficient(0)));
                results.add(new ClassifierResult("J48","Visual representation", eval.precision(1), eval.recall(1), eval.fMeasure(1),
                        eval.areaUnderROC(1), eval.matthewsCorrelationCoefficient(1)));
                results.add(new ClassifierResult("J48","Structure", eval.precision(2), eval.recall(2), eval.fMeasure(2),
                        eval.areaUnderROC(2), eval.matthewsCorrelationCoefficient(2)));
                results.add(new ClassifierResult("J48","Functional", eval.precision(3), eval.recall(3), eval.fMeasure(3),
                        eval.areaUnderROC(3), eval.matthewsCorrelationCoefficient(3)));
            }
            else {
                results.add(new ClassifierResult("J48","Evolvability", eval.precision(0), eval.recall(0), eval.fMeasure(0),
                        eval.areaUnderROC(0), eval.matthewsCorrelationCoefficient(0)));
                results.add(new ClassifierResult("J48","Functional", eval.precision(1), eval.recall(1), eval.fMeasure(1),
                        eval.areaUnderROC(1), eval.matthewsCorrelationCoefficient(1)));
            }

            System.out.println("\n J48 tree  \n");

            System.out.println("Precision 0: "+ eval.precision(0));
            System.out.println("Precision 1: "+eval.precision(1));
            if(isGroup) {
                System.out.println("Precision 2: "+eval.precision(2));
                System.out.println("Precision 3: "+eval.precision(3));
            }

            System.out.println("\n");
            System.out.println("Recall 0: "+eval.recall(0));
            System.out.println("Recall 1: "+eval.recall(1));
            if(isGroup) {
                System.out.println("Recall 2: "+eval.recall(2));
                System.out.println("Recall 3: "+eval.recall(3));
            }

            System.out.println("\n");
            System.out.println("F-Measure 0:  "+eval.fMeasure(0));
            System.out.println("F-Measure 1:  "+eval.fMeasure(1));
            if(isGroup) {
                System.out.println("F-Measure 2:  "+eval.fMeasure(2));
                System.out.println("F-Measure 3:  "+eval.fMeasure(3));
            }

            System.out.println("\n");
            System.out.println("AUC 0 : "+ eval.areaUnderROC(0));
            System.out.println("AUC 1 : "+ eval.areaUnderROC(1));
            if(isGroup) {
                System.out.println("AUC 2 : "+ eval.areaUnderROC(2));
                System.out.println("AUC 3 : "+ eval.areaUnderROC(3));
            }

            System.out.println("\n");
            System.out.println("MCC 0 : "+ eval.matthewsCorrelationCoefficient(0));
            System.out.println("MCC 1 : "+ eval.matthewsCorrelationCoefficient(1));
            if(isGroup) {
                System.out.println("MCC 2 : "+ eval.matthewsCorrelationCoefficient(2));
                System.out.println("MCC 3 : "+ eval.matthewsCorrelationCoefficient(3));
            }

        } catch (Exception e) {
            e.printStackTrace();
        }
        return results;
    }

    public List<ClassifierResult> runJ48WithTestsSetSMOTE(boolean isGroup) {
        List<ClassifierResult> results = new ArrayList<>();
        String[] options = new String[1];
        options[0] = "-U";
        J48 tree = new J48();
        try {

            SMOTE smote = new SMOTE();
            smote.setOptions(new String[]{"-P","150"});

            tree.setOptions(options);
            tree.buildClassifier(trainingData);
            FilteredClassifier fc = new FilteredClassifier();
            fc.setFilter(smote);
            fc.setClassifier(tree);
            Evaluation eval = new Evaluation(trainingData);
            eval.evaluateModel(tree, testData);

            if(isGroup) {
                results.add(new ClassifierResult("J48","Documentation", eval.precision(0), eval.recall(0), eval.fMeasure(0),
                        eval.areaUnderROC(0), eval.matthewsCorrelationCoefficient(0)));
                results.add(new ClassifierResult("J48","Visual representation", eval.precision(1), eval.recall(1), eval.fMeasure(1),
                        eval.areaUnderROC(1), eval.matthewsCorrelationCoefficient(1)));
                results.add(new ClassifierResult("J48","Structure", eval.precision(2), eval.recall(2), eval.fMeasure(2),
                        eval.areaUnderROC(2), eval.matthewsCorrelationCoefficient(2)));
                results.add(new ClassifierResult("J48","Functional", eval.precision(3), eval.recall(3), eval.fMeasure(3),
                        eval.areaUnderROC(3), eval.matthewsCorrelationCoefficient(3)));
            }
            else {
                results.add(new ClassifierResult("J48","Evolvability", eval.precision(0), eval.recall(0), eval.fMeasure(0),
                        eval.areaUnderROC(0), eval.matthewsCorrelationCoefficient(0)));
                results.add(new ClassifierResult("J48","Functional", eval.precision(1), eval.recall(1), eval.fMeasure(1),
                        eval.areaUnderROC(1), eval.matthewsCorrelationCoefficient(1)));
            }

            System.out.println("\n J48 tree  \n");

            System.out.println("Precision 0: "+ eval.precision(0));
            System.out.println("Precision 1: "+eval.precision(1));
            if(isGroup) {
                System.out.println("Precision 2: "+eval.precision(2));
                System.out.println("Precision 3: "+eval.precision(3));
            }

            System.out.println("\n");
            System.out.println("Recall 0: "+eval.recall(0));
            System.out.println("Recall 1: "+eval.recall(1));
            if(isGroup) {
                System.out.println("Recall 2: "+eval.recall(2));
                System.out.println("Recall 3: "+eval.recall(3));
            }

            System.out.println("\n");
            System.out.println("F-Measure 0:  "+eval.fMeasure(0));
            System.out.println("F-Measure 1:  "+eval.fMeasure(1));
            if(isGroup) {
                System.out.println("F-Measure 2:  "+eval.fMeasure(2));
                System.out.println("F-Measure 3:  "+eval.fMeasure(3));
            }

            System.out.println("\n");
            System.out.println("AUC 0 : "+ eval.areaUnderROC(0));
            System.out.println("AUC 1 : "+ eval.areaUnderROC(1));
            if(isGroup) {
                System.out.println("AUC 2 : "+ eval.areaUnderROC(2));
                System.out.println("AUC 3 : "+ eval.areaUnderROC(3));
            }

            System.out.println("\n");
            System.out.println("MCC 0 : "+ eval.matthewsCorrelationCoefficient(0));
            System.out.println("MCC 1 : "+ eval.matthewsCorrelationCoefficient(1));
            if(isGroup) {
                System.out.println("MCC 2 : "+ eval.matthewsCorrelationCoefficient(2));
                System.out.println("MCC 3 : "+ eval.matthewsCorrelationCoefficient(3));
            }

        } catch (Exception e) {
            e.printStackTrace();
        }
        return results;
    }

    public void runJ48WithoutLabels() {
        String[] options = new String[1];
        options[0] = "-U";
        J48 tree = new J48();
        try {
            trainingData.setClassIndex(trainingData.numAttributes()-1);
            testData.setClassIndex(testData.numAttributes()-1);
            tree.setOptions(options);
            tree.buildClassifier(trainingData);
            for(Instance instance : testData){
                System.out.println(tree.classifyInstance(instance));
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


    public void runRandomForest(){
        String[] options = new String[1];
        options[0] = "-attribute-importance";
        RandomForest tree = new RandomForest();
        data.setClassIndex(data.numAttributes()-1);
        Random random = new Random(1);
        try {
            tree.setOptions(options);
            tree.buildClassifier(data);
            Evaluation eval = new Evaluation(data);
            eval.crossValidateModel(tree, data, 10, random);
            System.out.println(eval.toSummaryString());
            System.out.println(eval.toMatrixString());

            System.out.println("MCC class 0:   " +eval.matthewsCorrelationCoefficient(0));
            System.out.println("MCC class 1:   " +eval.matthewsCorrelationCoefficient(1));
            System.out.println("MCC class 2:   " +eval.matthewsCorrelationCoefficient(2));
            System.out.println("MCC class 3:   " +eval.matthewsCorrelationCoefficient(3));
            System.out.println("MCC class 4:   " +eval.matthewsCorrelationCoefficient(4));
            System.out.println("MCC class 5:   " +eval.matthewsCorrelationCoefficient(5));
            System.out.println("MCC class 6:   " +eval.matthewsCorrelationCoefficient(6));
            System.out.println("MCC class 7:   " +eval.matthewsCorrelationCoefficient(7));
            System.out.println("MCC class 8:   " +eval.matthewsCorrelationCoefficient(8));
            System.out.println("MCC class 9:   " +eval.matthewsCorrelationCoefficient(9));
            System.out.println("MCC class 10:   " +eval.matthewsCorrelationCoefficient(10));

            System.out.println("Weighted MCC:   " +eval.weightedMatthewsCorrelation());

            System.out.println("Precision 0: "+ eval.precision(0));
            System.out.println("Recall 0: "+eval.recall(0));
            System.out.println("Precision 1: "+eval.precision(1));
            System.out.println("Recall 1: "+eval.recall(1));

            for(int i=0; i<4; i++) {
                System.out.println("Precision"+ i+ ": " +eval.precision(i));
                System.out.println("Recall: "+eval.recall(i));
                System.out.println("F-Measure " + eval.fMeasure(i));
            }

            System.out.println("AUC 0 : "+ eval.areaUnderROC(0));
            System.out.println("AUC 1 : "+ eval.areaUnderROC(1));

            printInfoGain(data);


        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    public List<ClassifierResult> runRandomForest10T(boolean isGroup){

        double precision0 = 0;
        double precision1 = 0;
        double precision2 = 0;
        double precision3 = 0;


        double recall0 = 0;
        double recall1 = 0;
        double recall2 = 0;
        double recall3 = 0;

        double fmeasure0 = 0;
        double fmeasure1 = 0;
        double fmeasure2 = 0;
        double fmeasure3 = 0;

        double auc0 =0;
        double auc1 =0;
        double auc2 =0;
        double auc3 =0;

        double mcc0 = 0;
        double mcc1 = 0;
        double mcc2 = 0;
        double mcc3 = 0;

        List<ClassifierResult> results = new ArrayList<>();

        String[] options = new String[1];
        Random random = new Random(1);

        try {
            for(int i=0; i<10; i++) {
                RandomForest tree = new RandomForest();
                data.setClassIndex(data.numAttributes()-1);
                tree.buildClassifier(data);
                Evaluation eval = new Evaluation(data);
                eval.crossValidateModel(tree, data, 10, random);
                System.out.println(eval.toSummaryString());
                System.out.println(eval.toMatrixString());

                precision0 = eval.precision(0) + precision0;
                precision1 = eval.precision(1) +  precision1;

                recall0 = eval.recall(0) + recall0;
                recall1 = eval.recall(1) + recall1;

                fmeasure0 = eval.fMeasure(0) + fmeasure0;
                fmeasure1 = eval.fMeasure(1) + fmeasure1;

                auc0 = eval.areaUnderROC(0) + auc0;
                auc1 = eval.areaUnderROC(1) + auc1;


                mcc0 = eval.matthewsCorrelationCoefficient(0) + mcc0;
                mcc1 = eval.matthewsCorrelationCoefficient(1) + mcc1;

                if(isGroup) {
                    precision2 = eval.precision(2) +  precision2;
                    precision3 = eval.precision(3) +  precision3;
                    recall2 = eval.recall(2) + recall2;
                    recall3 = eval.recall(3) + recall3;
                    fmeasure2 = eval.fMeasure(2) + fmeasure2;
                    fmeasure3 = eval.fMeasure(3) + fmeasure3;
                    auc2 = eval.areaUnderROC(2) + auc2;
                    auc3 = eval.areaUnderROC(3) + auc3;
                    mcc2 = eval.matthewsCorrelationCoefficient(2) + mcc2;
                    mcc3 = eval.matthewsCorrelationCoefficient(3) + mcc3;
                }

            }
        } catch (Exception e) {
            e.printStackTrace();
        }

        if(isGroup) {
            results.add(new ClassifierResult("Random forest","Documentation", precision0/10, recall0/10, fmeasure0/10,
                    auc0/10, mcc0/10));
            results.add(new ClassifierResult("Random forest","Visual representation", precision1/10, recall1/10, fmeasure1/10,
                    auc1/10, mcc1/10));
            results.add(new ClassifierResult("Random forest","Structure", precision2/10, recall2/10, fmeasure2/10,
                    auc2/10, mcc2/10));
            results.add(new ClassifierResult("Random forest","Functional", precision3/10, recall3/10, fmeasure3/10,
                    auc3/10, mcc3/10));
        }
        else {
            results.add(new ClassifierResult("Random forest","Evolvability", precision0/10, recall0/10, fmeasure0/10,
                    auc0/10, mcc0/10));
            results.add(new ClassifierResult("Random forest","Functional", precision1/10, recall1/10, fmeasure1/10,
                    auc1/10, mcc1/10));
        }

        System.out.println("\n Random forest \n");

        System.out.println("Precision 0: "+ precision0/10);
        System.out.println("Precision 1: "+precision1/10);
        if(isGroup) {
            System.out.println("Precision 2: " + precision2 / 10);
            System.out.println("Precision 3: " + precision3 / 10);
        }

        System.out.println("\n");
        System.out.println("Recall 0: "+recall0/10);
        System.out.println("Recall 1: "+recall1/10);
        if(isGroup) {
            System.out.println("Recall 2: " + recall2 / 10);
            System.out.println("Recall 3: " + recall3 / 10);
        }

        System.out.println("\n");
        System.out.println("F-Measure 0:  "+fmeasure0/10);
        System.out.println("F-Measure 1:  "+fmeasure1/10);
        if(isGroup) {
            System.out.println("F-Measure 2:  " + fmeasure2 / 10);
            System.out.println("F-Measure 3:  " + fmeasure3 / 10);
        }

        System.out.println("\n");
        System.out.println("AUC 0 : "+ auc0/10);
        System.out.println("AUC 1 : "+ auc1/10);
        if(isGroup) {
            System.out.println("AUC 2: " + auc2 / 10);
            System.out.println("AUC 3: " + auc3 / 10);
        }

        System.out.println("\n");
        System.out.println("MCC 0 : "+ mcc0/10);
        System.out.println("MCC 1 : "+ mcc1/10);
        if(isGroup) {
            System.out.println("MCC 2 : " + mcc2 / 10);
            System.out.println("MCC 3 : " + mcc3 / 10);
        }
        return results;
    }

    public List<ClassifierResult> runRandomForestWithSMOTE10Times(boolean isGroup) {

        double precision0 = 0;
        double precision1 = 0;
        double precision2 = 0;
        double precision3 = 0;


        double recall0 = 0;
        double recall1 = 0;
        double recall2 = 0;
        double recall3 = 0;

        double fmeasure0 = 0;
        double fmeasure1 = 0;
        double fmeasure2 = 0;
        double fmeasure3 = 0;

        double auc0 =0;
        double auc1 =0;
        double auc2 =0;
        double auc3 =0;

        double mcc0 = 0;
        double mcc1 = 0;
        double mcc2 = 0;
        double mcc3 = 0;

        List<ClassifierResult> results = new ArrayList<>();
        Random random = new Random(1);
        for(int i=0; i<10; i++) {
            try {
                data.setClassIndex(data.numAttributes() - 1);

                SMOTE smote = new SMOTE();
                smote.setOptions(new String[]{"-P", "100"});

                String[] options = new String[1];
                RandomForest tree = new RandomForest();
                tree.buildClassifier(data);
                FilteredClassifier fc = new FilteredClassifier();
                fc.setFilter(smote);
                fc.setClassifier(tree);
                Evaluation eval = new Evaluation(data);
                eval.crossValidateModel(fc, data, 10, random);

                System.out.println(eval.toSummaryString());
                System.out.println(eval.toMatrixString());

                precision0 = eval.precision(0) + precision0;
                precision1 = eval.precision(1) +  precision1;

                recall0 = eval.recall(0) + recall0;
                recall1 = eval.recall(1) + recall1;

                fmeasure0 = eval.fMeasure(0) + fmeasure0;
                fmeasure1 = eval.fMeasure(1) + fmeasure1;

                auc0 = eval.areaUnderROC(0) + auc0;
                auc1 = eval.areaUnderROC(1) + auc1;

                mcc0 = eval.matthewsCorrelationCoefficient(0) + mcc0;
                mcc1 = eval.matthewsCorrelationCoefficient(1) + mcc1;

                if(isGroup) {
                    precision2 = eval.precision(2) +  precision2;
                    precision3 = eval.precision(3) +  precision3;
                    recall2 = eval.recall(2) + recall2;
                    recall3 = eval.recall(3) + recall3;
                    fmeasure2 = eval.fMeasure(2) + fmeasure2;
                    fmeasure3 = eval.fMeasure(3) + fmeasure3;
                    auc2 = eval.areaUnderROC(2) + auc2;
                    auc3 = eval.areaUnderROC(3) + auc3;
                    mcc2 = eval.matthewsCorrelationCoefficient(2) + mcc2;
                    mcc3 = eval.matthewsCorrelationCoefficient(3) + mcc3;
                }



            } catch (Exception e) {
                e.printStackTrace();
            }
        }

        if(isGroup) {
            results.add(new ClassifierResult("Random forest","Documentation", precision0/10, recall0/10, fmeasure0/10,
                    auc0/10, mcc0/10));
            results.add(new ClassifierResult("Random forest","Visual representation", precision1/10, recall1/10, fmeasure1/10,
                    auc1/10, mcc1/10));
            results.add(new ClassifierResult("Random forest","Structure", precision2/10, recall2/10, fmeasure2/10,
                    auc2/10, mcc2/10));
            results.add(new ClassifierResult("Random forest","Functional", precision3/10, recall3/10, fmeasure3/10,
                    auc3/10, mcc3/10));
        }
        else {
            results.add(new ClassifierResult("Random forest","Evolvability", precision0/10, recall0/10, fmeasure0/10,
                    auc0/10, mcc0/10));
            results.add(new ClassifierResult("Random forest","Functional", precision1/10, recall1/10, fmeasure1/10,
                    auc1/10, mcc1/10));
        }

        System.out.println("\n Random forest SMOTE \n");

        System.out.println("Precision 0: "+ precision0/10);
        System.out.println("Precision 1: "+precision1/10);
        if(isGroup) {
            System.out.println("Precision 2: " + precision2 / 10);
            System.out.println("Precision 3: " + precision3 / 10);
        }
        System.out.println("\n");
        System.out.println("Recall 0: "+recall0/10);
        System.out.println("Recall 1: "+recall1/10);
        if(isGroup) {
            System.out.println("Recall 2: " + recall2 / 10);
            System.out.println("Recall 3: " + recall3 / 10);
        }
        System.out.println("\n");
        System.out.println("F-Measure 0:  "+fmeasure0/10);
        System.out.println("F-Measure 1:  "+fmeasure1/10);
        if(isGroup) {
            System.out.println("F-Measure 2:  " + fmeasure2 / 10);
            System.out.println("F-Measure 3:  " + fmeasure3 / 10);
        }
        System.out.println("\n");
        System.out.println("AUC 0 : "+ auc0/10);
        System.out.println("AUC 1 : "+ auc1/10);
        if(isGroup) {
            System.out.println("AUC 2: " + auc2 / 10);
            System.out.println("AUC 3: " + auc3 / 10);
        }
        System.out.println("\n");
        System.out.println("MCC 0 : "+ mcc0/10);
        System.out.println("MCC 1 : "+ mcc1/10);
        if(isGroup) {
            System.out.println("MCC 2 : " + mcc2 / 10);
            System.out.println("MCC 3 : " + mcc3 / 10);
        }
        return results;
    }


    public List<ClassifierResult> runRandomForestWithUndersample10Times(boolean isGroup) {

        double precision0 = 0;
        double precision1 = 0;
        double precision2 = 0;
        double precision3 = 0;


        double recall0 = 0;
        double recall1 = 0;
        double recall2 = 0;
        double recall3 = 0;

        double fmeasure0 = 0;
        double fmeasure1 = 0;
        double fmeasure2 = 0;
        double fmeasure3 = 0;

        double auc0 =0;
        double auc1 =0;
        double auc2 =0;
        double auc3 =0;

        double mcc0 = 0;
        double mcc1 = 0;
        double mcc2 = 0;
        double mcc3 = 0;

        List<ClassifierResult> results = new ArrayList<>();
        Random random = new Random(1);
        for(int i=0; i<10; i++) {
            try {
                data.setClassIndex(data.numAttributes() - 1);

                SpreadSubsample ff = new SpreadSubsample();
                ff.setOptions(new String[]{"-M", "4"});

                String[] options = new String[1];
                RandomForest tree = new RandomForest();
                tree.buildClassifier(data);
                FilteredClassifier fc = new FilteredClassifier();
                fc.setFilter(ff);
                fc.setClassifier(tree);
                Evaluation eval = new Evaluation(data);
                eval.crossValidateModel(fc, data, 10, random);

                System.out.println(eval.toSummaryString());
                System.out.println(eval.toMatrixString());

                precision0 = eval.precision(0) + precision0;
                precision1 = eval.precision(1) +  precision1;

                recall0 = eval.recall(0) + recall0;
                recall1 = eval.recall(1) + recall1;

                fmeasure0 = eval.fMeasure(0) + fmeasure0;
                fmeasure1 = eval.fMeasure(1) + fmeasure1;

                auc0 = eval.areaUnderROC(0) + auc0;
                auc1 = eval.areaUnderROC(1) + auc1;

                mcc0 = eval.matthewsCorrelationCoefficient(0) + mcc0;
                mcc1 = eval.matthewsCorrelationCoefficient(1) + mcc1;

                if(isGroup) {
                    precision2 = eval.precision(2) +  precision2;
                    precision3 = eval.precision(3) +  precision3;
                    recall2 = eval.recall(2) + recall2;
                    recall3 = eval.recall(3) + recall3;
                    fmeasure2 = eval.fMeasure(2) + fmeasure2;
                    fmeasure3 = eval.fMeasure(3) + fmeasure3;
                    auc2 = eval.areaUnderROC(2) + auc2;
                    auc3 = eval.areaUnderROC(3) + auc3;
                    mcc2 = eval.matthewsCorrelationCoefficient(2) + mcc2;
                    mcc3 = eval.matthewsCorrelationCoefficient(3) + mcc3;
                }



            } catch (Exception e) {
                e.printStackTrace();
            }
        }

        if(isGroup) {
            results.add(new ClassifierResult("Random forest","Documentation", precision0/10, recall0/10, fmeasure0/10,
                    auc0/10, mcc0/10));
            results.add(new ClassifierResult("Random forest","Visual representation", precision1/10, recall1/10, fmeasure1/10,
                    auc1/10, mcc1/10));
            results.add(new ClassifierResult("Random forest","Structure", precision2/10, recall2/10, fmeasure2/10,
                    auc2/10, mcc2/10));
            results.add(new ClassifierResult("Random forest","Functional", precision3/10, recall3/10, fmeasure3/10,
                    auc3/10, mcc3/10));
        }
        else {
            results.add(new ClassifierResult("Random forest","Evolvability", precision0/10, recall0/10, fmeasure0/10,
                    auc0/10, mcc0/10));
            results.add(new ClassifierResult("Random forest","Functional", precision1/10, recall1/10, fmeasure1/10,
                    auc1/10, mcc1/10));
        }

        System.out.println("\n Random forest Undersampling \n");

        System.out.println("Precision 0: "+ precision0/10);
        System.out.println("Precision 1: "+precision1/10);
        if(isGroup) {
            System.out.println("Precision 2: " + precision2 / 10);
            System.out.println("Precision 3: " + precision3 / 10);
        }
        System.out.println("\n");
        System.out.println("Recall 0: "+recall0/10);
        System.out.println("Recall 1: "+recall1/10);
        if(isGroup) {
            System.out.println("Recall 2: " + recall2 / 10);
            System.out.println("Recall 3: " + recall3 / 10);
        }
        System.out.println("\n");
        System.out.println("F-Measure 0:  "+fmeasure0/10);
        System.out.println("F-Measure 1:  "+fmeasure1/10);
        if(isGroup) {
            System.out.println("F-Measure 2:  " + fmeasure2 / 10);
            System.out.println("F-Measure 3:  " + fmeasure3 / 10);
        }
        System.out.println("\n");
        System.out.println("AUC 0 : "+ auc0/10);
        System.out.println("AUC 1 : "+ auc1/10);
        if(isGroup) {
            System.out.println("AUC 2: " + auc2 / 10);
            System.out.println("AUC 3: " + auc3 / 10);
        }
        System.out.println("\n");
        System.out.println("MCC 0 : "+ mcc0/10);
        System.out.println("MCC 1 : "+ mcc1/10);
        if(isGroup) {
            System.out.println("MCC 2 : " + mcc2 / 10);
            System.out.println("MCC 3 : " + mcc3 / 10);
        }
        return results;
    }


    public List<ClassifierResult> runNaiveBayesWithSMOTE10Times(boolean isGroup) {

        double precision0 = 0;
        double precision1 = 0;
        double precision2 = 0;
        double precision3 = 0;

        double recall0 = 0;
        double recall1 = 0;
        double recall2 = 0;
        double recall3 = 0;

        double fmeasure0 = 0;
        double fmeasure1 = 0;
        double fmeasure2 = 0;
        double fmeasure3 = 0;

        double auc0 =0;
        double auc1 =0;
        double auc2 =0;
        double auc3 =0;

        double mcc0 = 0;
        double mcc1 = 0;
        double mcc2 = 0;
        double mcc3 = 0;

        List<ClassifierResult> results = new ArrayList<>();
        Random random = new Random(1);
        for(int i=0; i<10; i++) {

            try {
                System.out.println("Before smote " + data.numInstances());
                data.setClassIndex(data.numAttributes() - 1);

                SMOTE smote = new SMOTE();
                smote.setOptions(new String[]{"-P", "100"});

                String[] options = new String[1];
                NaiveBayes tree = new NaiveBayes();
                tree.buildClassifier(data);
                FilteredClassifier fc = new FilteredClassifier();
                fc.setFilter(smote);
                fc.setClassifier(tree);
                Evaluation eval = new Evaluation(data);
                eval.crossValidateModel(fc, data, 10, random);
                System.out.println(eval.toSummaryString());
                System.out.println(eval.toMatrixString());

                precision0 = eval.precision(0) + precision0;
                precision1 = eval.precision(1) +  precision1;
                if(isGroup) {
                    precision2 = eval.precision(2) +  precision2;
                    precision3 = eval.precision(3) +  precision3;
                }


                recall0 = eval.recall(0) + recall0;
                recall1 = eval.recall(1) + recall1;
                if(isGroup) {
                    recall2 = eval.recall(2) + recall2;
                    recall3 = eval.recall(3) + recall3;
                }

                fmeasure0 = eval.fMeasure(0) + fmeasure0;
                fmeasure1 = eval.fMeasure(1) + fmeasure1;
                if(isGroup) {
                    fmeasure2 = eval.fMeasure(2) + fmeasure2;
                    fmeasure3 = eval.fMeasure(3) + fmeasure3;
                }

                auc0 = eval.areaUnderROC(0) + auc0;
                auc1 = eval.areaUnderROC(1) + auc1;
                if(isGroup) {
                    auc2 = eval.areaUnderROC(2) + auc2;
                    auc3 = eval.areaUnderROC(3) + auc3;
                }


                mcc0 = eval.matthewsCorrelationCoefficient(0) + mcc0;
                mcc1 = eval.matthewsCorrelationCoefficient(1) + mcc1;
                if(isGroup) {
                    mcc2 = eval.matthewsCorrelationCoefficient(2) + mcc2;
                    mcc3 = eval.matthewsCorrelationCoefficient(3) + mcc3;
                }


            } catch (Exception e) {
                e.printStackTrace();
            }
        }

        if(isGroup) {
            results.add(new ClassifierResult("Naive Bayes","Documentation", precision0/10, recall0/10, fmeasure0/10,
                    auc0/10, mcc0/10));
            results.add(new ClassifierResult("Naive Bayes","Visual representation", precision1/10, recall1/10, fmeasure1/10,
                    auc1/10, mcc1/10));
            results.add(new ClassifierResult("Naive Bayes","Structure", precision2/10, recall2/10, fmeasure2/10,
                   auc2/10, mcc2/10));
            results.add(new ClassifierResult("Naive Bayes","Functional", precision3/10, recall3/10, fmeasure3/10,
                    auc3/10, mcc3/10));
        }
        else {
            results.add(new ClassifierResult("Naive Bayes","Evolvability", precision0/10, recall0/10, fmeasure0/10,
                    auc0/10, mcc0/10));
            results.add(new ClassifierResult("Naive Bayes","Functional", precision1/10, recall1/10, fmeasure1/10,
                    auc1/10, mcc1/10));
        }

        System.out.println("\n Naive Bayes SMOTE \n");

        System.out.println("Precision 0: "+ precision0/10);
        System.out.println("Precision 1: "+precision1/10);
        System.out.println("Precision 2: "+ precision2/10);
        System.out.println("Precision 3: "+precision3/10);
        System.out.println("\n");
        System.out.println("Recall 0: "+recall0/10);
        System.out.println("Recall 1: "+recall1/10);
        System.out.println("Recall 2: "+recall2/10);
        System.out.println("Recall 3: "+recall3/10);
        System.out.println("\n");
        System.out.println("F-Measure 0:  "+fmeasure0/10);
        System.out.println("F-Measure 1:  "+fmeasure1/10);
        System.out.println("F-Measure 2:  "+fmeasure2/10);
        System.out.println("F-Measure 3:  "+fmeasure3/10);
        System.out.println("\n");
        System.out.println("AUC 0 : "+ auc0/10);
        System.out.println("AUC 1 : "+ auc1/10);
        System.out.println("AUC 2: "+ auc2/10);
        System.out.println("AUC 3: "+ auc3/10);
        System.out.println("\n");
        System.out.println("MCC 0 : "+ mcc0/10);
        System.out.println("MCC 1 : "+ mcc1/10);
        System.out.println("MCC 2 : "+ mcc2/10);
        System.out.println("MCC 3 : "+ mcc3/10);
        return results;
    }


    public List<ClassifierResult> runNaiveBayesWithUndersample10Times(boolean isGroup) {

        double precision0 = 0;
        double precision1 = 0;
        double precision2 = 0;
        double precision3 = 0;

        double recall0 = 0;
        double recall1 = 0;
        double recall2 = 0;
        double recall3 = 0;

        double fmeasure0 = 0;
        double fmeasure1 = 0;
        double fmeasure2 = 0;
        double fmeasure3 = 0;

        double auc0 =0;
        double auc1 =0;
        double auc2 =0;
        double auc3 =0;

        double mcc0 = 0;
        double mcc1 = 0;
        double mcc2 = 0;
        double mcc3 = 0;

        List<ClassifierResult> results = new ArrayList<>();
        Random random = new Random(1);
        for(int i=0; i<10; i++) {

            try {
                System.out.println("Before smote " + data.numInstances());
                data.setClassIndex(data.numAttributes() - 1);

                SpreadSubsample ff = new SpreadSubsample();

                String[] options = new String[1];

                ff.setOptions(new String[]{"-M", "1.0"});
                NaiveBayes tree = new NaiveBayes();
                tree.buildClassifier(data);
                FilteredClassifier fc = new FilteredClassifier();
                fc.setFilter(ff);
                fc.setClassifier(tree);
                Evaluation eval = new Evaluation(data);
                eval.crossValidateModel(fc, data, 10, random);
                System.out.println(eval.toSummaryString());
                System.out.println(eval.toMatrixString());

                precision0 = eval.precision(0) + precision0;
                precision1 = eval.precision(1) +  precision1;
                if(isGroup) {
                    precision2 = eval.precision(2) +  precision2;
                    precision3 = eval.precision(3) +  precision3;
                }


                recall0 = eval.recall(0) + recall0;
                recall1 = eval.recall(1) + recall1;
                if(isGroup) {
                    recall2 = eval.recall(2) + recall2;
                    recall3 = eval.recall(3) + recall3;
                }

                fmeasure0 = eval.fMeasure(0) + fmeasure0;
                fmeasure1 = eval.fMeasure(1) + fmeasure1;
                if(isGroup) {
                    fmeasure2 = eval.fMeasure(2) + fmeasure2;
                    fmeasure3 = eval.fMeasure(3) + fmeasure3;
                }

                auc0 = eval.areaUnderROC(0) + auc0;
                auc1 = eval.areaUnderROC(1) + auc1;
                if(isGroup) {
                    auc2 = eval.areaUnderROC(2) + auc2;
                    auc3 = eval.areaUnderROC(3) + auc3;
                }


                mcc0 = eval.matthewsCorrelationCoefficient(0) + mcc0;
                mcc1 = eval.matthewsCorrelationCoefficient(1) + mcc1;
                if(isGroup) {
                    mcc2 = eval.matthewsCorrelationCoefficient(2) + mcc2;
                    mcc3 = eval.matthewsCorrelationCoefficient(3) + mcc3;
                }


            } catch (Exception e) {
                e.printStackTrace();
            }
        }

        if(isGroup) {
            results.add(new ClassifierResult("Naive Bayes","Documentation", precision0/10, recall0/10, fmeasure0/10,
                    auc0/10, mcc0/10));
            results.add(new ClassifierResult("Naive Bayes","Visual representation", precision1/10, recall1/10, fmeasure1/10,
                    auc1/10, mcc1/10));
            results.add(new ClassifierResult("Naive Bayes","Structure", precision2/10, recall2/10, fmeasure2/10,
                    auc2/10, mcc2/10));
            results.add(new ClassifierResult("Naive Bayes","Functional", precision3/10, recall3/10, fmeasure3/10,
                    auc3/10, mcc3/10));
        }
        else {
            results.add(new ClassifierResult("Naive Bayes","Evolvability", precision0/10, recall0/10, fmeasure0/10,
                    auc0/10, mcc0/10));
            results.add(new ClassifierResult("Naive Bayes","Functional", precision1/10, recall1/10, fmeasure1/10,
                    auc1/10, mcc1/10));
        }

        System.out.println("\n Naive Bayes SMOTE \n");

        System.out.println("Precision 0: "+ precision0/10);
        System.out.println("Precision 1: "+precision1/10);
        System.out.println("Precision 2: "+ precision2/10);
        System.out.println("Precision 3: "+precision3/10);
        System.out.println("\n");
        System.out.println("Recall 0: "+recall0/10);
        System.out.println("Recall 1: "+recall1/10);
        System.out.println("Recall 2: "+recall2/10);
        System.out.println("Recall 3: "+recall3/10);
        System.out.println("\n");
        System.out.println("F-Measure 0:  "+fmeasure0/10);
        System.out.println("F-Measure 1:  "+fmeasure1/10);
        System.out.println("F-Measure 2:  "+fmeasure2/10);
        System.out.println("F-Measure 3:  "+fmeasure3/10);
        System.out.println("\n");
        System.out.println("AUC 0 : "+ auc0/10);
        System.out.println("AUC 1 : "+ auc1/10);
        System.out.println("AUC 2: "+ auc2/10);
        System.out.println("AUC 3: "+ auc3/10);
        System.out.println("\n");
        System.out.println("MCC 0 : "+ mcc0/10);
        System.out.println("MCC 1 : "+ mcc1/10);
        System.out.println("MCC 2 : "+ mcc2/10);
        System.out.println("MCC 3 : "+ mcc3/10);
        return results;
    }


    public List<ClassifierResult> runNaiveBayesAttributeSelectionSMOTE10Times(boolean isGroup) {

        double precision0 = 0;
        double precision1 = 0;
        double precision2 = 0;
        double precision3 = 0;

        double recall0 = 0;
        double recall1 = 0;
        double recall2 = 0;
        double recall3 = 0;

        double fmeasure0 = 0;
        double fmeasure1 = 0;
        double fmeasure2 = 0;
        double fmeasure3 = 0;

        double auc0 =0;
        double auc1 =0;
        double auc2 =0;
        double auc3 =0;

        double mcc0 = 0;
        double mcc1 = 0;
        double mcc2 = 0;
        double mcc3 = 0;

        List<ClassifierResult> results = new ArrayList<>();
        Random random = new Random(1);
        for(int i=0; i<10; i++) {

            try {

                System.out.println("Before smote " + data.numInstances());
                data.setClassIndex(data.numAttributes() - 1);

                SMOTE smote = new SMOTE();
                smote.setOptions(new String[]{"-P", "100"});

                String[] options = new String[1];
                NaiveBayes tree = new NaiveBayes();
                tree.buildClassifier(data);
                FilteredClassifier fc = new FilteredClassifier();
                fc.setFilter(smote);
                fc.setClassifier(tree);
                Evaluation eval = new Evaluation(data);
                eval.crossValidateModel(fc, data, 10, random);
                System.out.println(eval.toSummaryString());
                System.out.println(eval.toMatrixString());

                precision0 = eval.precision(0) + precision0;
                precision1 = eval.precision(1) +  precision1;
                if(isGroup) {
                    precision2 = eval.precision(2) +  precision2;
                    precision3 = eval.precision(3) +  precision3;
                }


                recall0 = eval.recall(0) + recall0;
                recall1 = eval.recall(1) + recall1;
                if(isGroup) {
                    recall2 = eval.recall(2) + recall2;
                    recall3 = eval.recall(3) + recall3;
                }

                fmeasure0 = eval.fMeasure(0) + fmeasure0;
                fmeasure1 = eval.fMeasure(1) + fmeasure1;
                if(isGroup) {
                    fmeasure2 = eval.fMeasure(2) + fmeasure2;
                    fmeasure3 = eval.fMeasure(3) + fmeasure3;
                }

                auc0 = eval.areaUnderROC(0) + auc0;
                auc1 = eval.areaUnderROC(1) + auc1;
                if(isGroup) {
                    auc2 = eval.areaUnderROC(2) + auc2;
                    auc3 = eval.areaUnderROC(3) + auc3;
                }


                mcc0 = eval.matthewsCorrelationCoefficient(0) + mcc0;
                mcc1 = eval.matthewsCorrelationCoefficient(1) + mcc1;
                if(isGroup) {
                    mcc2 = eval.matthewsCorrelationCoefficient(2) + mcc2;
                    mcc3 = eval.matthewsCorrelationCoefficient(3) + mcc3;
                }


            } catch (Exception e) {
                e.printStackTrace();
            }
        }

        if(isGroup) {
            results.add(new ClassifierResult("Naive Bayes","Documentation", precision0/10, recall0/10, fmeasure0/10,
                    auc0/10, mcc0/10));
            results.add(new ClassifierResult("Naive Bayes","Visual representation", precision1/10, recall1/10, fmeasure1/10,
                    auc1/10, mcc1/10));
            results.add(new ClassifierResult("Naive Bayes","Structure", precision2/10, recall2/10, fmeasure2/10,
                    auc2/10, mcc2/10));
            results.add(new ClassifierResult("Naive Bayes","Functional", precision3/10, recall3/10, fmeasure3/10,
                    auc3/10, mcc3/10));
        }
        else {
            results.add(new ClassifierResult("Naive Bayes","Evolvability", precision0/10, recall0/10, fmeasure0/10,
                    auc0/10, mcc0/10));
            results.add(new ClassifierResult("Naive Bayes","Functional", precision1/10, recall1/10, fmeasure1/10,
                    auc1/10, mcc1/10));
        }

        System.out.println("\n Naive Bayes SMOTE \n");

        System.out.println("Precision 0: "+ precision0/10);
        System.out.println("Precision 1: "+precision1/10);
        System.out.println("Precision 2: "+ precision2/10);
        System.out.println("Precision 3: "+precision3/10);
        System.out.println("\n");
        System.out.println("Recall 0: "+recall0/10);
        System.out.println("Recall 1: "+recall1/10);
        System.out.println("Recall 2: "+recall2/10);
        System.out.println("Recall 3: "+recall3/10);
        System.out.println("\n");
        System.out.println("F-Measure 0:  "+fmeasure0/10);
        System.out.println("F-Measure 1:  "+fmeasure1/10);
        System.out.println("F-Measure 2:  "+fmeasure2/10);
        System.out.println("F-Measure 3:  "+fmeasure3/10);
        System.out.println("\n");
        System.out.println("AUC 0 : "+ auc0/10);
        System.out.println("AUC 1 : "+ auc1/10);
        System.out.println("AUC 2: "+ auc2/10);
        System.out.println("AUC 3: "+ auc3/10);
        System.out.println("\n");
        System.out.println("MCC 0 : "+ mcc0/10);
        System.out.println("MCC 1 : "+ mcc1/10);
        System.out.println("MCC 2 : "+ mcc2/10);
        System.out.println("MCC 3 : "+ mcc3/10);
        return results;
    }


    public List<ClassifierResult> runNaiveBayesAttributeSelection10Times(boolean isGroup) {

        double precision0 = 0;
        double precision1 = 0;
        double precision2 = 0;
        double precision3 = 0;

        double recall0 = 0;
        double recall1 = 0;
        double recall2 = 0;
        double recall3 = 0;

        double fmeasure0 = 0;
        double fmeasure1 = 0;
        double fmeasure2 = 0;
        double fmeasure3 = 0;

        double auc0 =0;
        double auc1 =0;
        double auc2 =0;
        double auc3 =0;

        double mcc0 = 0;
        double mcc1 = 0;
        double mcc2 = 0;
        double mcc3 = 0;

        List<ClassifierResult> results = new ArrayList<>();
        Random random = new Random(1);

        for(int i=0; i<10; i++) {
            try {
                data.setClassIndex(data.numAttributes() - 1);

                String[] options = new String[1];
                NaiveBayes tree = new NaiveBayes();
                tree.buildClassifier(data);
                Evaluation eval = new Evaluation(data);
                eval.crossValidateModel(tree, data, 10, random);
                System.out.println(eval.toSummaryString());

                System.out.println(eval.toMatrixString());

                printInfoGain(data);

                precision0 = eval.precision(0) + precision0;
                precision1 = eval.precision(1) +  precision1;
                if(isGroup) {
                    precision2 = eval.precision(2) +  precision2;
                    precision3 = eval.precision(3) +  precision3;
                }


                recall0 = eval.recall(0) + recall0;
                recall1 = eval.recall(1) + recall1;
                if(isGroup) {
                    recall2 = eval.recall(2) + recall2;
                    recall3 = eval.recall(3) + recall3;
                }

                fmeasure0 = eval.fMeasure(0) + fmeasure0;
                fmeasure1 = eval.fMeasure(1) + fmeasure1;
                if(isGroup) {
                    fmeasure2 = eval.fMeasure(2) + fmeasure2;
                    fmeasure3 = eval.fMeasure(3) + fmeasure3;
                }

                auc0 = eval.areaUnderROC(0) + auc0;
                auc1 = eval.areaUnderROC(1) + auc1;
                if(isGroup) {
                    auc2 = eval.areaUnderROC(2) + auc2;
                    auc3 = eval.areaUnderROC(3) + auc3;
                }


                mcc0 = eval.matthewsCorrelationCoefficient(0) + mcc0;
                mcc1 = eval.matthewsCorrelationCoefficient(1) + mcc1;
                if(isGroup) {
                    mcc2 = eval.matthewsCorrelationCoefficient(2) + mcc2;
                    mcc3 = eval.matthewsCorrelationCoefficient(3) + mcc3;
                }


            } catch (Exception e) {
                e.printStackTrace();
            }
        }

        if(isGroup) {
            results.add(new ClassifierResult("Naive Bayes","Documentation", precision0/10, recall0/10, fmeasure0/10,
                    auc0/10, mcc0/10));
            results.add(new ClassifierResult("Naive Bayes","Visual representation", precision1/10, recall1/10, fmeasure1/10,
                    auc1/10, mcc1/10));
            results.add(new ClassifierResult("Naive Bayes","Structure", precision2/10, recall2/10, fmeasure2/10,
                    auc2/10, mcc2/10));
            results.add(new ClassifierResult("Naive Bayes","Functional", precision3/10, recall3/10, fmeasure3/10,
                    auc3/10, mcc3/10));
        }
        else {
            results.add(new ClassifierResult("Naive Bayes","Evolvability", precision0/10, recall0/10, fmeasure0/10,
                    auc0/10, mcc0/10));
            results.add(new ClassifierResult("Naive Bayes","Functional", precision1/10, recall1/10, fmeasure1/10,
                    auc1/10, mcc1/10));
        }

        System.out.println("\n Naive Bayes SMOTE \n");

        System.out.println("Precision 0: "+ precision0/10);
        System.out.println("Precision 1: "+precision1/10);
        System.out.println("Precision 2: "+ precision2/10);
        System.out.println("Precision 3: "+precision3/10);
        System.out.println("\n");
        System.out.println("Recall 0: "+recall0/10);
        System.out.println("Recall 1: "+recall1/10);
        System.out.println("Recall 2: "+recall2/10);
        System.out.println("Recall 3: "+recall3/10);
        System.out.println("\n");
        System.out.println("F-Measure 0:  "+fmeasure0/10);
        System.out.println("F-Measure 1:  "+fmeasure1/10);
        System.out.println("F-Measure 2:  "+fmeasure2/10);
        System.out.println("F-Measure 3:  "+fmeasure3/10);
        System.out.println("\n");
        System.out.println("AUC 0 : "+ auc0/10);
        System.out.println("AUC 1 : "+ auc1/10);
        System.out.println("AUC 2: "+ auc2/10);
        System.out.println("AUC 3: "+ auc3/10);
        System.out.println("\n");
        System.out.println("MCC 0 : "+ mcc0/10);
        System.out.println("MCC 1 : "+ mcc1/10);
        System.out.println("MCC 2 : "+ mcc2/10);
        System.out.println("MCC 3 : "+ mcc3/10);
        return results;
    }


    public List<ClassifierResult> runJ48WithSMOTE10Times(boolean isGroup) {

        double precision0 = 0;
        double precision1 = 0;
        double precision2 = 0;
        double precision3 = 0;


        double recall0 = 0;
        double recall1 = 0;
        double recall2 = 0;
        double recall3 = 0;

        double fmeasure0 = 0;
        double fmeasure1 = 0;
        double fmeasure2 = 0;
        double fmeasure3 = 0;

        double auc0 =0;
        double auc1 =0;
        double auc2 =0;
        double auc3 =0;

        double mcc0 = 0;
        double mcc1 = 0;
        double mcc2 = 0;
        double mcc3 = 0;

        List<ClassifierResult> results = new ArrayList<>();
        Random random = new Random(1);

        for(int i=0; i<10; i++) {
            try {
                System.out.println("Before smote " + data.numInstances());
                data.setClassIndex(data.numAttributes() - 1);

                SMOTE smote = new SMOTE();
                smote.setOptions(new String[]{"-P", "100"});

                String[] options = new String[1];
                options[0] = "-U";
                J48 tree = new J48();
                tree.buildClassifier(data);
                FilteredClassifier fc = new FilteredClassifier();
                fc.setFilter(smote);
                fc.setClassifier(tree);
                Evaluation eval = new Evaluation(data);
                eval.crossValidateModel(fc, data, 10, random);
                System.out.println(eval.toSummaryString());
                System.out.println(eval.toMatrixString());

                precision0 = eval.precision(0) + precision0;
                precision1 = eval.precision(1) +  precision1;
                if(isGroup) {
                    precision2 = eval.precision(2) +  precision2;
                    precision3 = eval.precision(3) +  precision3;
                }


                recall0 = eval.recall(0) + recall0;
                recall1 = eval.recall(1) + recall1;
                if(isGroup) {
                    recall2 = eval.recall(2) + recall2;
                    recall3 = eval.recall(3) + recall3;
                }

                fmeasure0 = eval.fMeasure(0) + fmeasure0;
                fmeasure1 = eval.fMeasure(1) + fmeasure1;
                if(isGroup) {
                    fmeasure2 = eval.fMeasure(2) + fmeasure2;
                    fmeasure3 = eval.fMeasure(3) + fmeasure3;
                }

                auc0 = eval.areaUnderROC(0) + auc0;
                auc1 = eval.areaUnderROC(1) + auc1;
                if(isGroup) {
                    auc2 = eval.areaUnderROC(2) + auc2;
                    auc3 = eval.areaUnderROC(3) + auc3;
                }


                mcc0 = eval.matthewsCorrelationCoefficient(0) + mcc0;
                mcc1 = eval.matthewsCorrelationCoefficient(1) + mcc1;
                if(isGroup) {
                    mcc2 = eval.matthewsCorrelationCoefficient(2) + mcc2;
                    mcc3 = eval.matthewsCorrelationCoefficient(3) + mcc3;
                }


            } catch (Exception e) {
                e.printStackTrace();
            }
        }

        if(isGroup) {
            results.add(new ClassifierResult("J48","Documentation", precision0/10, recall0/10, fmeasure0/10,
                    auc0/10, mcc0/10));
            results.add(new ClassifierResult("J48","Visual representation", precision1/10, recall1/10, fmeasure1/10,
                    auc1/10, mcc1/10));
            results.add(new ClassifierResult("J48","Structure", precision2/10, recall2/10, fmeasure2/10,
                    auc2/10, mcc2/10));
            results.add(new ClassifierResult("J48","Functional", precision3/10, recall3/10, fmeasure3/10,
                    auc3/10, mcc3/10));
        }
        else {
            results.add(new ClassifierResult("J48","Evolvability", precision0/10, recall0/10, fmeasure0/10,
                    auc0/10, mcc0/10));
            results.add(new ClassifierResult("J48","Functional", precision1/10, recall1/10, fmeasure1/10,
                    auc1/10, mcc1/10));
        }

        System.out.println("\n j48 SMOTE \n");

        System.out.println("Precision 0: "+ precision0/10);
        System.out.println("Precision 1: "+precision1/10);
        System.out.println("Precision 2: "+ precision2/10);
        System.out.println("Precision 3: "+precision3/10);
        System.out.println("\n");
        System.out.println("Recall 0: "+recall0/10);
        System.out.println("Recall 1: "+recall1/10);
        System.out.println("Recall 2: "+recall2/10);
        System.out.println("Recall 3: "+recall3/10);
        System.out.println("\n");
        System.out.println("F-Measure 0:  "+fmeasure0/10);
        System.out.println("F-Measure 1:  "+fmeasure1/10);
        System.out.println("F-Measure 2:  "+fmeasure2/10);
        System.out.println("F-Measure 3:  "+fmeasure3/10);
        System.out.println("\n");
        System.out.println("AUC 0 : "+ auc0/10);
        System.out.println("AUC 1 : "+ auc1/10);
        System.out.println("AUC 2: "+ auc2/10);
        System.out.println("AUC 3: "+ auc3/10);
        System.out.println("\n");
        System.out.println("MCC 0 : "+ mcc0/10);
        System.out.println("MCC 1 : "+ mcc1/10);
        System.out.println("MCC 2 : "+ mcc2/10);
        System.out.println("MCC 3 : "+ mcc3/10);
        return results;
    }


    public List<ClassifierResult> runJ48WithUndersample10Times(boolean isGroup) {

        double precision0 = 0;
        double precision1 = 0;
        double precision2 = 0;
        double precision3 = 0;


        double recall0 = 0;
        double recall1 = 0;
        double recall2 = 0;
        double recall3 = 0;

        double fmeasure0 = 0;
        double fmeasure1 = 0;
        double fmeasure2 = 0;
        double fmeasure3 = 0;

        double auc0 =0;
        double auc1 =0;
        double auc2 =0;
        double auc3 =0;

        double mcc0 = 0;
        double mcc1 = 0;
        double mcc2 = 0;
        double mcc3 = 0;

        List<ClassifierResult> results = new ArrayList<>();
        Random random = new Random(1);

        for(int i=0; i<10; i++) {
            try {
                System.out.println("Before undersampling " + data.numInstances());
                data.setClassIndex(data.numAttributes() - 1);

                SpreadSubsample ff = new SpreadSubsample();
                ff.setOptions(new String[]{"-M", "1.0"});

                String[] options = new String[1];
                options[0] = "-U";
                J48 tree = new J48();
                tree.buildClassifier(data);
                FilteredClassifier fc = new FilteredClassifier();
                fc.setFilter(ff);
                fc.setClassifier(tree);
                Evaluation eval = new Evaluation(data);
                eval.crossValidateModel(fc, data, 10, random);
                System.out.println(eval.toSummaryString());
                System.out.println(eval.toMatrixString());

                precision0 = eval.precision(0) + precision0;
                precision1 = eval.precision(1) +  precision1;
                if(isGroup) {
                    precision2 = eval.precision(2) +  precision2;
                    precision3 = eval.precision(3) +  precision3;
                }


                recall0 = eval.recall(0) + recall0;
                recall1 = eval.recall(1) + recall1;
                if(isGroup) {
                    recall2 = eval.recall(2) + recall2;
                    recall3 = eval.recall(3) + recall3;
                }

                fmeasure0 = eval.fMeasure(0) + fmeasure0;
                fmeasure1 = eval.fMeasure(1) + fmeasure1;
                if(isGroup) {
                    fmeasure2 = eval.fMeasure(2) + fmeasure2;
                    fmeasure3 = eval.fMeasure(3) + fmeasure3;
                }

                auc0 = eval.areaUnderROC(0) + auc0;
                auc1 = eval.areaUnderROC(1) + auc1;
                if(isGroup) {
                    auc2 = eval.areaUnderROC(2) + auc2;
                    auc3 = eval.areaUnderROC(3) + auc3;
                }


                mcc0 = eval.matthewsCorrelationCoefficient(0) + mcc0;
                mcc1 = eval.matthewsCorrelationCoefficient(1) + mcc1;
                if(isGroup) {
                    mcc2 = eval.matthewsCorrelationCoefficient(2) + mcc2;
                    mcc3 = eval.matthewsCorrelationCoefficient(3) + mcc3;
                }


            } catch (Exception e) {
                e.printStackTrace();
            }
        }

        if(isGroup) {
            results.add(new ClassifierResult("J48","Documentation", precision0/10, recall0/10, fmeasure0/10,
                    auc0/10, mcc0/10));
            results.add(new ClassifierResult("J48","Visual representation", precision1/10, recall1/10, fmeasure1/10,
                    auc1/10, mcc1/10));
            results.add(new ClassifierResult("J48","Structure", precision2/10, recall2/10, fmeasure2/10,
                    auc2/10, mcc2/10));
            results.add(new ClassifierResult("J48","Functional", precision3/10, recall3/10, fmeasure3/10,
                    auc3/10, mcc3/10));
        }
        else {
            results.add(new ClassifierResult("J48","Evolvability", precision0/10, recall0/10, fmeasure0/10,
                    auc0/10, mcc0/10));
            results.add(new ClassifierResult("J48","Functional", precision1/10, recall1/10, fmeasure1/10,
                    auc1/10, mcc1/10));
        }

        System.out.println("\n j48 undersampling \n");

        System.out.println("Precision 0: "+ precision0/10);
        System.out.println("Precision 1: "+precision1/10);
        System.out.println("Precision 2: "+ precision2/10);
        System.out.println("Precision 3: "+precision3/10);
        System.out.println("\n");
        System.out.println("Recall 0: "+recall0/10);
        System.out.println("Recall 1: "+recall1/10);
        System.out.println("Recall 2: "+recall2/10);
        System.out.println("Recall 3: "+recall3/10);
        System.out.println("\n");
        System.out.println("F-Measure 0:  "+fmeasure0/10);
        System.out.println("F-Measure 1:  "+fmeasure1/10);
        System.out.println("F-Measure 2:  "+fmeasure2/10);
        System.out.println("F-Measure 3:  "+fmeasure3/10);
        System.out.println("\n");
        System.out.println("AUC 0 : "+ auc0/10);
        System.out.println("AUC 1 : "+ auc1/10);
        System.out.println("AUC 2: "+ auc2/10);
        System.out.println("AUC 3: "+ auc3/10);
        System.out.println("\n");
        System.out.println("MCC 0 : "+ mcc0/10);
        System.out.println("MCC 1 : "+ mcc1/10);
        System.out.println("MCC 2 : "+ mcc2/10);
        System.out.println("MCC 3 : "+ mcc3/10);
        return results;
    }


    public List<ClassifierResult> runJ48AttributeSelection10Times(boolean isGroup) {

        double precision0 = 0;
        double precision1 = 0;
        double precision2 = 0;
        double precision3 = 0;


        double recall0 = 0;
        double recall1 = 0;
        double recall2 = 0;
        double recall3 = 0;

        double fmeasure0 = 0;
        double fmeasure1 = 0;
        double fmeasure2 = 0;
        double fmeasure3 = 0;

        double auc0 =0;
        double auc1 =0;
        double auc2 =0;
        double auc3 =0;

        double mcc0 = 0;
        double mcc1 = 0;
        double mcc2 = 0;
        double mcc3 = 0;

        List<ClassifierResult> results = new ArrayList<>();
        Random random = new Random(1);

        for(int i=0; i<10; i++) {
            try {

                InfoGainAttributeEval evalGain = new InfoGainAttributeEval();
                Ranker search = new Ranker();
                search.setOptions(new String[] { "-T", "0.001" });

                AttributeSelection attSelect = new AttributeSelection();
                attSelect.setEvaluator(evalGain);
                attSelect.setSearch(search);
                attSelect.SelectAttributes(data);

                data = attSelect.reduceDimensionality(data);


                data.setClassIndex(data.numAttributes() - 1);

                data.setClassIndex(data.numAttributes() - 1);

                String[] options = new String[1];
                options[0] = "-U";
                J48 tree = new J48();
                tree.buildClassifier(data);
                Evaluation eval = new Evaluation(data);
                eval.crossValidateModel(tree, data, 10, random);
                System.out.println(eval.toSummaryString());

                System.out.println(eval.toMatrixString());

                precision0 = eval.precision(0) + precision0;
                precision1 = eval.precision(1) +  precision1;
                if(isGroup) {
                    precision2 = eval.precision(2) +  precision2;
                    precision3 = eval.precision(3) +  precision3;
                }


                recall0 = eval.recall(0) + recall0;
                recall1 = eval.recall(1) + recall1;
                if(isGroup) {
                    recall2 = eval.recall(2) + recall2;
                    recall3 = eval.recall(3) + recall3;
                }

                fmeasure0 = eval.fMeasure(0) + fmeasure0;
                fmeasure1 = eval.fMeasure(1) + fmeasure1;
                if(isGroup) {
                    fmeasure2 = eval.fMeasure(2) + fmeasure2;
                    fmeasure3 = eval.fMeasure(3) + fmeasure3;
                }

                auc0 = eval.areaUnderROC(0) + auc0;
                auc1 = eval.areaUnderROC(1) + auc1;
                if(isGroup) {
                    auc2 = eval.areaUnderROC(2) + auc2;
                    auc3 = eval.areaUnderROC(3) + auc3;
                }


                mcc0 = eval.matthewsCorrelationCoefficient(0) + mcc0;
                mcc1 = eval.matthewsCorrelationCoefficient(1) + mcc1;
                if(isGroup) {
                    mcc2 = eval.matthewsCorrelationCoefficient(2) + mcc2;
                    mcc3 = eval.matthewsCorrelationCoefficient(3) + mcc3;
                }


            } catch (Exception e) {
                e.printStackTrace();
            }
        }

        if(isGroup) {
            results.add(new ClassifierResult("J48","Documentation", precision0/10, recall0/10, fmeasure0/10,
                    auc0/10, mcc0/10));
            results.add(new ClassifierResult("J48","Visual representation", precision1/10, recall1/10, fmeasure1/10,
                    auc1/10, mcc1/10));
            results.add(new ClassifierResult("J48","Structure", precision2/10, recall2/10, fmeasure2/10,
                    auc2/10, mcc2/10));
            results.add(new ClassifierResult("J48","Functional", precision3/10, recall3/10, fmeasure3/10,
                    auc3/10, mcc3/10));
        }
        else {
            results.add(new ClassifierResult("J48","Evolvability", precision0/10, recall0/10, fmeasure0/10,
                    auc0/10, mcc0/10));
            results.add(new ClassifierResult("J48","Functional", precision1/10, recall1/10, fmeasure1/10,
                    auc1/10, mcc1/10));
        }

        System.out.println("\n j48 SMOTE \n");

        System.out.println("Precision 0: "+ precision0/10);
        System.out.println("Precision 1: "+precision1/10);
        System.out.println("Precision 2: "+ precision2/10);
        System.out.println("Precision 3: "+precision3/10);
        System.out.println("\n");
        System.out.println("Recall 0: "+recall0/10);
        System.out.println("Recall 1: "+recall1/10);
        System.out.println("Recall 2: "+recall2/10);
        System.out.println("Recall 3: "+recall3/10);
        System.out.println("\n");
        System.out.println("F-Measure 0:  "+fmeasure0/10);
        System.out.println("F-Measure 1:  "+fmeasure1/10);
        System.out.println("F-Measure 2:  "+fmeasure2/10);
        System.out.println("F-Measure 3:  "+fmeasure3/10);
        System.out.println("\n");
        System.out.println("AUC 0 : "+ auc0/10);
        System.out.println("AUC 1 : "+ auc1/10);
        System.out.println("AUC 2: "+ auc2/10);
        System.out.println("AUC 3: "+ auc3/10);
        System.out.println("\n");
        System.out.println("MCC 0 : "+ mcc0/10);
        System.out.println("MCC 1 : "+ mcc1/10);
        System.out.println("MCC 2 : "+ mcc2/10);
        System.out.println("MCC 3 : "+ mcc3/10);
        return results;
    }

    public List<ClassifierResult> runJ48AttributeSelectionSMOTE10Times(boolean isGroup) {

        double precision0 = 0;
        double precision1 = 0;
        double precision2 = 0;
        double precision3 = 0;


        double recall0 = 0;
        double recall1 = 0;
        double recall2 = 0;
        double recall3 = 0;

        double fmeasure0 = 0;
        double fmeasure1 = 0;
        double fmeasure2 = 0;
        double fmeasure3 = 0;

        double auc0 =0;
        double auc1 =0;
        double auc2 =0;
        double auc3 =0;

        double mcc0 = 0;
        double mcc1 = 0;
        double mcc2 = 0;
        double mcc3 = 0;

        List<ClassifierResult> results = new ArrayList<>();
        Random random = new Random(1);

        for(int i=0; i<10; i++) {
            try {

                InfoGainAttributeEval evalGain = new InfoGainAttributeEval();
                Ranker search = new Ranker();
                search.setOptions(new String[] { "-T", "0.001" });

                AttributeSelection attSelect = new AttributeSelection();
                attSelect.setEvaluator(evalGain);
                attSelect.setSearch(search);
                attSelect.SelectAttributes(data);

                data = attSelect.reduceDimensionality(data);


                data.setClassIndex(data.numAttributes() - 1);

                System.out.println("Before smote " + data.numInstances());
                data.setClassIndex(data.numAttributes() - 1);

                SMOTE smote = new SMOTE();
                smote.setOptions(new String[]{"-P", "100"}); // change this parameter

                String[] options = new String[1];
                options[0] = "-U";
                J48 tree = new J48();
                tree.buildClassifier(data);
                FilteredClassifier fc = new FilteredClassifier();
                fc.setFilter(smote);
                fc.setClassifier(tree);
                Evaluation eval = new Evaluation(data);
                eval.crossValidateModel(fc, data, 10, random);
                System.out.println(eval.toSummaryString());

                System.out.println(eval.toMatrixString());

                precision0 = eval.precision(0) + precision0;
                precision1 = eval.precision(1) +  precision1;
                if(isGroup) {
                    precision2 = eval.precision(2) +  precision2;
                    precision3 = eval.precision(3) +  precision3;
                }


                recall0 = eval.recall(0) + recall0;
                recall1 = eval.recall(1) + recall1;
                if(isGroup) {
                    recall2 = eval.recall(2) + recall2;
                    recall3 = eval.recall(3) + recall3;
                }

                fmeasure0 = eval.fMeasure(0) + fmeasure0;
                fmeasure1 = eval.fMeasure(1) + fmeasure1;
                if(isGroup) {
                    fmeasure2 = eval.fMeasure(2) + fmeasure2;
                    fmeasure3 = eval.fMeasure(3) + fmeasure3;
                }

                auc0 = eval.areaUnderROC(0) + auc0;
                auc1 = eval.areaUnderROC(1) + auc1;
                if(isGroup) {
                    auc2 = eval.areaUnderROC(2) + auc2;
                    auc3 = eval.areaUnderROC(3) + auc3;
                }


                mcc0 = eval.matthewsCorrelationCoefficient(0) + mcc0;
                mcc1 = eval.matthewsCorrelationCoefficient(1) + mcc1;
                if(isGroup) {
                    mcc2 = eval.matthewsCorrelationCoefficient(2) + mcc2;
                    mcc3 = eval.matthewsCorrelationCoefficient(3) + mcc3;
                }


            } catch (Exception e) {
                e.printStackTrace();
            }
        }

        if(isGroup) {
            results.add(new ClassifierResult("J48","Documentation", precision0/10, recall0/10, fmeasure0/10,
                    auc0/10, mcc0/10));
            results.add(new ClassifierResult("J48","Visual representation", precision1/10, recall1/10, fmeasure1/10,
                    auc1/10, mcc1/10));
            results.add(new ClassifierResult("J48","Structure", precision2/10, recall2/10, fmeasure2/10,
                    auc2/10, mcc2/10));
            results.add(new ClassifierResult("J48","Functional", precision3/10, recall3/10, fmeasure3/10,
                    auc3/10, mcc3/10));
        }
        else {
            results.add(new ClassifierResult("J48","Evolvability", precision0/10, recall0/10, fmeasure0/10,
                    auc0/10, mcc0/10));
            results.add(new ClassifierResult("J48","Functional", precision1/10, recall1/10, fmeasure1/10,
                    auc1/10, mcc1/10));
        }

        System.out.println("\n j48 SMOTE \n");

        System.out.println("Precision 0: "+ precision0/10);
        System.out.println("Precision 1: "+precision1/10);
        System.out.println("Precision 2: "+ precision2/10);
        System.out.println("Precision 3: "+precision3/10);
        System.out.println("\n");
        System.out.println("Recall 0: "+recall0/10);
        System.out.println("Recall 1: "+recall1/10);
        System.out.println("Recall 2: "+recall2/10);
        System.out.println("Recall 3: "+recall3/10);
        System.out.println("\n");
        System.out.println("F-Measure 0:  "+fmeasure0/10);
        System.out.println("F-Measure 1:  "+fmeasure1/10);
        System.out.println("F-Measure 2:  "+fmeasure2/10);
        System.out.println("F-Measure 3:  "+fmeasure3/10);
        System.out.println("\n");
        System.out.println("AUC 0 : "+ auc0/10);
        System.out.println("AUC 1 : "+ auc1/10);
        System.out.println("AUC 2: "+ auc2/10);
        System.out.println("AUC 3: "+ auc3/10);
        System.out.println("\n");
        System.out.println("MCC 0 : "+ mcc0/10);
        System.out.println("MCC 1 : "+ mcc1/10);
        System.out.println("MCC 2 : "+ mcc2/10);
        System.out.println("MCC 3 : "+ mcc3/10);
        return results;
    }

    public List<ClassifierResult> runJ4810T(boolean isGroup){

        double precision0 = 0;
        double precision1 = 0;
        double precision2 = 0;
        double precision3 = 0;


        double recall0 = 0;
        double recall1 = 0;
        double recall2 = 0;
        double recall3 = 0;

        double fmeasure0 = 0;
        double fmeasure1 = 0;
        double fmeasure2 = 0;
        double fmeasure3 = 0;

        double auc0 =0;
        double auc1 =0;
        double auc2 =0;
        double auc3 =0;

        double mcc0 = 0;
        double mcc1 = 0;
        double mcc2 = 0;
        double mcc3 = 0;

        List<ClassifierResult> results = new ArrayList<>();

        String[] options = new String[1];
        options[0] = "-U";
        Random random = new Random(1);
        try {
            for(int i=0; i<10; i++) {
                J48 tree = new J48();
                data.setClassIndex(data.numAttributes()-1);
                tree.setOptions(options);
                tree.buildClassifier(data);
                System.out.println(tree);
                Evaluation eval = new Evaluation(data);
                eval.crossValidateModel(tree, data, 10, random);
                System.out.println(eval.toSummaryString());
                System.out.println(eval.toMatrixString());

                precision0 = eval.precision(0) + precision0;
                precision1 = eval.precision(1) +  precision1;
                if(isGroup) {
                    precision2 = eval.precision(2) +  precision2;
                    precision3 = eval.precision(3) +  precision3;
                }


                recall0 = eval.recall(0) + recall0;
                recall1 = eval.recall(1) + recall1;
                if(isGroup) {
                    recall2 = eval.recall(2) + recall2;
                    recall3 = eval.recall(3) + recall3;
                }

                fmeasure0 = eval.fMeasure(0) + fmeasure0;
                fmeasure1 = eval.fMeasure(1) + fmeasure1;
                if(isGroup) {
                    fmeasure2 = eval.fMeasure(2) + fmeasure2;
                    fmeasure3 = eval.fMeasure(3) + fmeasure3;
                }

                auc0 = eval.areaUnderROC(0) + auc0;
                auc1 = eval.areaUnderROC(1) + auc1;
                if(isGroup) {
                    auc2 = eval.areaUnderROC(2) + auc2;
                    auc3 = eval.areaUnderROC(3) + auc3;
                }


                mcc0 = eval.matthewsCorrelationCoefficient(0) + mcc0;
                mcc1 = eval.matthewsCorrelationCoefficient(1) + mcc1;
                if(isGroup) {
                    mcc2 = eval.matthewsCorrelationCoefficient(2) + mcc2;
                    mcc3 = eval.matthewsCorrelationCoefficient(3) + mcc3;
                }

            }
        } catch (Exception e) {
            e.printStackTrace();
        }

        if(isGroup) {
            results.add(new ClassifierResult("J48","Documentation", precision0/10, recall0/10, fmeasure0/10,
                    auc0/10, mcc0/10));
            results.add(new ClassifierResult("J48","Visual representation", precision1/10, recall1/10, fmeasure1/10,
                    auc1/10, mcc1/10));
            results.add(new ClassifierResult("J48","Structure", precision2/10, recall2/10, fmeasure2/10,
                    auc2/10, mcc2/10));
            results.add(new ClassifierResult("J48","Functional", precision3/10, recall3/10, fmeasure3/10,
                    auc3/10, mcc3/10));
        }
        else {
            results.add(new ClassifierResult("J48","Evolvability", precision0/10, recall0/10, fmeasure0/10,
                    auc0/10, mcc0/10));
            results.add(new ClassifierResult("J48","Functional", precision1/10, recall1/10, fmeasure1/10,
                    auc1/10, mcc1/10));
        }

        System.out.println("\n J48 \n");

        System.out.println("Precision 0: "+ precision0/10);
        System.out.println("Precision 1: "+precision1/10);
        System.out.println("Precision 2: "+ precision2/10);
        System.out.println("Precision 3: "+precision3/10);
        System.out.println("\n");
        System.out.println("Recall 0: "+recall0/10);
        System.out.println("Recall 1: "+recall1/10);
        System.out.println("Recall 2: "+recall2/10);
        System.out.println("Recall 3: "+recall3/10);
        System.out.println("\n");
        System.out.println("F-Measure 0:  "+fmeasure0/10);
        System.out.println("F-Measure 1:  "+fmeasure1/10);
        System.out.println("F-Measure 2:  "+fmeasure2/10);
        System.out.println("F-Measure 3:  "+fmeasure3/10);
        System.out.println("\n");
        System.out.println("AUC 0 : "+ auc0/10);
        System.out.println("AUC 1 : "+ auc1/10);
        System.out.println("AUC 2: "+ auc2/10);
        System.out.println("AUC 3: "+ auc3/10);
        System.out.println("\n");
        System.out.println("MCC 0 : "+ mcc0/10);
        System.out.println("MCC 1 : "+ mcc1/10);
        System.out.println("MCC 2 : "+ mcc2/10);
        System.out.println("MCC 3 : "+ mcc3/10);
        return results;
    }


    public List<ClassifierResult> runNaiveBayes10T(boolean isGroup){

        double precision0 = 0;
        double precision1 = 0;
        double precision2 = 0;
        double precision3 = 0;


        double recall0 = 0;
        double recall1 = 0;
        double recall2 = 0;
        double recall3 = 0;

        double fmeasure0 = 0;
        double fmeasure1 = 0;
        double fmeasure2 = 0;
        double fmeasure3 = 0;

        double auc0 =0;
        double auc1 =0;
        double auc2 =0;
        double auc3 =0;

        double mcc0 = 0;
        double mcc1 = 0;
        double mcc2 = 0;
        double mcc3 = 0;

        List<ClassifierResult> results = new ArrayList<>();

        String[] options = new String[1];
        NaiveBayes tree = new NaiveBayes();
        data.setClassIndex(data.numAttributes()-1);
        Random random = new Random(1);
        try {
            for(int i=0; i<10; i++) {
                tree.buildClassifier(data);
                System.out.println(tree);
                Evaluation eval = new Evaluation(data);
                eval.crossValidateModel(tree, data, 10, random);
                System.out.println(eval.toSummaryString());
                System.out.println(eval.toMatrixString());

                precision0 = eval.precision(0) + precision0;
                precision1 = eval.precision(1) +  precision1;
                if(isGroup) {
                    precision2 = eval.precision(2) +  precision2;
                    precision3 = eval.precision(3) +  precision3;
                }


                recall0 = eval.recall(0) + recall0;
                recall1 = eval.recall(1) + recall1;
                if(isGroup) {
                    recall2 = eval.recall(2) + recall2;
                    recall3 = eval.recall(3) + recall3;
                }

                fmeasure0 = eval.fMeasure(0) + fmeasure0;
                fmeasure1 = eval.fMeasure(1) + fmeasure1;
                if(isGroup) {
                    fmeasure2 = eval.fMeasure(2) + fmeasure2;
                    fmeasure3 = eval.fMeasure(3) + fmeasure3;
                }

                auc0 = eval.areaUnderROC(0) + auc0;
                auc1 = eval.areaUnderROC(1) + auc1;
                if(isGroup) {
                    auc2 = eval.areaUnderROC(2) + auc2;
                    auc3 = eval.areaUnderROC(3) + auc3;
                }


                mcc0 = eval.matthewsCorrelationCoefficient(0) + mcc0;
                mcc1 = eval.matthewsCorrelationCoefficient(1) + mcc1;
                if(isGroup) {
                    mcc2 = eval.matthewsCorrelationCoefficient(2) + mcc2;
                    mcc3 = eval.matthewsCorrelationCoefficient(3) + mcc3;
                }

            }
        } catch (Exception e) {
            e.printStackTrace();
        }

        if(isGroup) {
            results.add(new ClassifierResult("Naive Bayes","Documentation", precision0/10, recall0/10, fmeasure0/10,
                    auc0/10, mcc0/10));
            results.add(new ClassifierResult("Naive Bayes","Visual representation", precision1/10, recall1/10, fmeasure1/10,
                    auc1/10, mcc1/10));
            results.add(new ClassifierResult("Naive Bayes","Structure", precision2/10, recall2/10, fmeasure2/10,
                    auc2/10, mcc2/10));
            results.add(new ClassifierResult("Naive Bayes","Functional", precision3/10, recall3/10, fmeasure3/10,
                    auc3/10, mcc3/10));
        }
        else {
            results.add(new ClassifierResult("Naive Bayes","Evolvability", precision0/10, recall0/10, fmeasure0/10,
                    auc0/10, mcc0/10));
            results.add(new ClassifierResult("Naive Bayes","Functional", precision1/10, recall1/10, fmeasure1/10,
                    auc1/10, mcc1/10));
        }

        System.out.println("\n Naive Bayes \n");

        System.out.println("Precision 0: "+ precision0/10);
        System.out.println("Precision 1: "+precision1/10);
        System.out.println("Precision 2: "+ precision2/10);
        System.out.println("Precision 3: "+precision3/10);
        System.out.println("\n");
        System.out.println("Recall 0: "+recall0/10);
        System.out.println("Recall 1: "+recall1/10);
        System.out.println("Recall 2: "+recall2/10);
        System.out.println("Recall 3: "+recall3/10);
        System.out.println("\n");
        System.out.println("F-Measure 0:  "+fmeasure0/10);
        System.out.println("F-Measure 1:  "+fmeasure1/10);
        System.out.println("F-Measure 2:  "+fmeasure2/10);
        System.out.println("F-Measure 3:  "+fmeasure3/10);
        System.out.println("\n");
        System.out.println("AUC 0 : "+ auc0/10);
        System.out.println("AUC 1 : "+ auc1/10);
        System.out.println("AUC 2: "+ auc2/10);
        System.out.println("AUC 3: "+ auc3/10);
        System.out.println("\n");
        System.out.println("MCC 0 : "+ mcc0/10);
        System.out.println("MCC 1 : "+ mcc1/10);
        System.out.println("MCC 2 : "+ mcc2/10);
        System.out.println("MCC 3 : "+ mcc3/10);
        return results;
    }


    public void runRandomForestWithCostMatrix(){

        //String weightString = "[1.0 1.0 1.0 1.0; 1.0 1.0 1.0 1.0; 1.0 1.0 1.0 1.0; 5.0 5.0 5.0 1.0]";

        CostSensitiveClassifier costSensitiveClassifier = new CostSensitiveClassifier();
        RandomForest tree = new RandomForest();
        Random random = new Random(1);

        try {
            tree.buildClassifier(data);
            Evaluation eval = new Evaluation(data);

            CostMatrix costMatrix = new CostMatrix(4);

            costMatrix.setCell(0, 0, 0.0);
            costMatrix.setCell(0, 1, 0.0);
            costMatrix.setCell(0, 2, 0.0);
            costMatrix.setCell(0, 3, 200.0);

            costMatrix.setCell(1, 0, 0.0);
            costMatrix.setCell(1, 1, 0.0);
            costMatrix.setCell(1, 2, 0.0);
            costMatrix.setCell(1, 3, 200.0);

            costMatrix.setCell(2, 0, 0.0);
            costMatrix.setCell(2, 1, 0.0);
            costMatrix.setCell(2, 2, 0.0);
            costMatrix.setCell(2, 3, 200.0);

            costMatrix.setCell(3, 0, 250.0);
            costMatrix.setCell(3, 1, 250.0);
            costMatrix.setCell(3, 2, 250.0);
            costMatrix.setCell(3, 3, 0.0);

            costSensitiveClassifier.setClassifier(tree);
            costSensitiveClassifier.setCostMatrix(costMatrix);

            eval.crossValidateModel(costSensitiveClassifier, data, 10, random);

            System.out.println("MCC class 0:   " +eval.matthewsCorrelationCoefficient(0));
            System.out.println("MCC class 1:   " +eval.matthewsCorrelationCoefficient(1));
            System.out.println("MCC class 2:   " +eval.matthewsCorrelationCoefficient(2));
            System.out.println("MCC class 3:   " +eval.matthewsCorrelationCoefficient(3));
            System.out.println("MCC class 4:   " +eval.matthewsCorrelationCoefficient(4));
            System.out.println("MCC class 5:   " +eval.matthewsCorrelationCoefficient(5));
            System.out.println("MCC class 6:   " +eval.matthewsCorrelationCoefficient(6));
            System.out.println("MCC class 7:   " +eval.matthewsCorrelationCoefficient(7));
            System.out.println("MCC class 8:   " +eval.matthewsCorrelationCoefficient(8));
            System.out.println("MCC class 9:   " +eval.matthewsCorrelationCoefficient(9));
            System.out.println("MCC class 10:   " +eval.matthewsCorrelationCoefficient(10));

            System.out.println("Weighted MCC:   " +eval.weightedMatthewsCorrelation());

            System.out.println("Precision 0: "+ eval.precision(0));
            System.out.println("Recall 0: "+eval.recall(0));
            System.out.println("Precision 1: "+eval.precision(1));
            System.out.println("Recall 1: "+eval.recall(1));

            for(int i=0; i<4; i++) {
                System.out.println("Precision"+ i+ ": " +eval.precision(i));
                System.out.println("Recall: "+eval.recall(i));
            }

            for(int i=0; i<4; i++) {
                System.out.println("F-Measure " + eval.fMeasure(i));
            }

            System.out.println("AUC 0 : "+ eval.areaUnderROC(0));
            System.out.println("AUC 1 : "+ eval.areaUnderROC(1));

            printInfoGain(data);


        } catch (Exception e) {
            e.printStackTrace();
        }
    }


    public List<Double> runRandomForestWithoutLabels(){
        RandomForest tree = new RandomForest();
        List<Double> classifiedValues = new ArrayList<>();
        try {
            trainingData.setClassIndex(trainingData.numAttributes()-1);
            testData.setClassIndex(testData.numAttributes()-1);
            tree.buildClassifier(trainingData);


            for(Instance instance : testData) {
                double value = tree.classifyInstance(instance);
                classifiedValues.add(value);
                System.out.println(value);
            }

        } catch (Exception e) {
            e.printStackTrace();
        }
        return classifiedValues;
    }


    public void runGridSearch(){
        String[] options = new String[1];
        GridSearch tree = new GridSearch();
        GridSearch gs = new GridSearch();
        Random random = new Random(1);
        try {

            gs.buildClassifier(data);
            int requiredIndex = 6; // for accuracy
            SelectedTag st=new SelectedTag(requiredIndex , weka.classifiers.meta.GridSearch.TAGS_EVALUATION);
            gs.setEvaluation(st);

            tree.buildClassifier(data);
            Evaluation eval = new Evaluation(data);
            eval.crossValidateModel(tree, data, 10, random);
            System.out.println(eval.toSummaryString());
            System.out.println(eval.toMatrixString());

            System.out.println("MCC class 0:   " +eval.matthewsCorrelationCoefficient(0));
            System.out.println("MCC class 1:   " +eval.matthewsCorrelationCoefficient(1));
            System.out.println("MCC class 2:   " +eval.matthewsCorrelationCoefficient(2));
            System.out.println("MCC class 3:   " +eval.matthewsCorrelationCoefficient(3));
            System.out.println("MCC class 4:   " +eval.matthewsCorrelationCoefficient(4));
            System.out.println("MCC class 5:   " +eval.matthewsCorrelationCoefficient(5));
            System.out.println("MCC class 6:   " +eval.matthewsCorrelationCoefficient(6));
            System.out.println("MCC class 7:   " +eval.matthewsCorrelationCoefficient(7));
            System.out.println("MCC class 8:   " +eval.matthewsCorrelationCoefficient(8));
            System.out.println("MCC class 9:   " +eval.matthewsCorrelationCoefficient(9));
            System.out.println("MCC class 10:   " +eval.matthewsCorrelationCoefficient(10));

            System.out.println("Weighted MCC:   " +eval.weightedMatthewsCorrelation());

            System.out.println("Precision 0: "+ eval.precision(0));
            System.out.println("Recall 0: "+eval.recall(0));
            System.out.println("Precision 1: "+eval.precision(1));
            System.out.println("Recall 1: "+eval.recall(1));

            for(int i=0; i<2; i++) {
                System.out.println("F-Measure " + eval.fMeasure(i));
            }

            System.out.println("AUC 0 : "+ eval.areaUnderROC(0));
            System.out.println("AUC 1 : "+ eval.areaUnderROC(1));

            printInfoGain(data);


        } catch (Exception e) {
            e.printStackTrace();
        }
    }


    private void printInfoGain(Instances data){
        InfoGainAttributeEval infoGainAttributeEval = new InfoGainAttributeEval();
        try {
            infoGainAttributeEval.buildEvaluator(data);
            for(int i=0; i<35; i++){
                System.out.println(i + "  info gain:  "+infoGainAttributeEval.evaluateAttribute(i));
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }


    public List<ClassifierResult> runRandomForestWithTestsSet(boolean isGroup) {
        List<ClassifierResult> results = new ArrayList<>();
        String[] options = new String[1];
        RandomForest tree = new RandomForest();
        try {
            tree.buildClassifier(trainingData);
            Evaluation eval = new Evaluation(trainingData);
            eval.evaluateModel(tree, testData);
            System.out.println(eval.toSummaryString());
            System.out.println(eval.toMatrixString());

            if(isGroup) {
                results.add(new ClassifierResult("Random forest","Documentation", eval.precision(0), eval.recall(0), eval.fMeasure(0),
                        eval.areaUnderROC(0), eval.matthewsCorrelationCoefficient(0)));
                results.add(new ClassifierResult("Random forest","Visual representation", eval.precision(1), eval.recall(1), eval.fMeasure(1),
                        eval.areaUnderROC(1), eval.matthewsCorrelationCoefficient(1)));
                results.add(new ClassifierResult("Random forest","Structure", eval.precision(2), eval.recall(2), eval.fMeasure(2),
                        eval.areaUnderROC(2), eval.matthewsCorrelationCoefficient(2)));
                results.add(new ClassifierResult("Random forest","Functional", eval.precision(3), eval.recall(3), eval.fMeasure(3),
                        eval.areaUnderROC(3), eval.matthewsCorrelationCoefficient(3)));
            }
            else {
                results.add(new ClassifierResult("Random forest","Evolvability", eval.precision(0), eval.recall(0), eval.fMeasure(0),
                        eval.areaUnderROC(0), eval.matthewsCorrelationCoefficient(0)));
                results.add(new ClassifierResult("Random forest","Functional", eval.precision(1), eval.recall(1), eval.fMeasure(1),
                        eval.areaUnderROC(1), eval.matthewsCorrelationCoefficient(1)));
            }

            System.out.println("\n Random forest \n");

            System.out.println("Precision 0: "+ eval.precision(0));
            System.out.println("Precision 1: "+eval.precision(1));
            if(isGroup) {
                System.out.println("Precision 2: "+eval.precision(2));
                System.out.println("Precision 3: "+eval.precision(3));
            }

            System.out.println("\n");
            System.out.println("Recall 0: "+eval.recall(0));
            System.out.println("Recall 1: "+eval.recall(1));
            if(isGroup) {
                System.out.println("Recall 2: "+eval.recall(2));
                System.out.println("Recall 3: "+eval.recall(3));
            }

            System.out.println("\n");
            System.out.println("F-Measure 0:  "+eval.fMeasure(0));
            System.out.println("F-Measure 1:  "+eval.fMeasure(1));
            if(isGroup) {
                System.out.println("F-Measure 2:  "+eval.fMeasure(2));
                System.out.println("F-Measure 3:  "+eval.fMeasure(3));
            }

            System.out.println("\n");
            System.out.println("AUC 0 : "+ eval.areaUnderROC(0));
            System.out.println("AUC 1 : "+ eval.areaUnderROC(1));
            if(isGroup) {
                System.out.println("AUC 2 : "+ eval.areaUnderROC(2));
                System.out.println("AUC 3 : "+ eval.areaUnderROC(3));
            }

            System.out.println("\n");
            System.out.println("MCC 0 : "+ eval.matthewsCorrelationCoefficient(0));
            System.out.println("MCC 1 : "+ eval.matthewsCorrelationCoefficient(1));
            if(isGroup) {
                System.out.println("MCC 2 : "+ eval.matthewsCorrelationCoefficient(2));
                System.out.println("MCC 3 : "+ eval.matthewsCorrelationCoefficient(3));
            }

        } catch (Exception e) {
            e.printStackTrace();
        }
        return results;
    }


    public List<ClassifierResult> runRandomForestWithTestsSetLeaveOneOut(boolean isGroup) {
        List<ClassifierResult> results = new ArrayList<>();
        RandomForest tree = new RandomForest();
        try {
            tree.buildClassifier(trainingData);
            Evaluation eval = new Evaluation(trainingData);
            eval.evaluateModel(tree, testData);
            System.out.println(eval.toSummaryString());
            System.out.println(eval.toMatrixString());

            if(isGroup) {
                results.add(new ClassifierResult("Random forest","Documentation", eval.precision(0), eval.recall(0), eval.fMeasure(0),
                        eval.areaUnderROC(0), eval.matthewsCorrelationCoefficient(0)));
                results.add(new ClassifierResult("Random forest","Visual representation", eval.precision(1), eval.recall(1), eval.fMeasure(1),
                        eval.areaUnderROC(1), eval.matthewsCorrelationCoefficient(1)));
                results.add(new ClassifierResult("Random forest","Structure", eval.precision(2), eval.recall(2), eval.fMeasure(2),
                        eval.areaUnderROC(2), eval.matthewsCorrelationCoefficient(2)));
                results.add(new ClassifierResult("Random forest","Functional", eval.precision(3), eval.recall(3), eval.fMeasure(3),
                        eval.areaUnderROC(3), eval.matthewsCorrelationCoefficient(3)));
            }
            else {
                results.add(new ClassifierResult("Random forest","Evolvability", eval.precision(0), eval.recall(0), eval.fMeasure(0),
                        eval.areaUnderROC(0), eval.matthewsCorrelationCoefficient(0)));
                results.add(new ClassifierResult("Random forest","Functional", eval.precision(1), eval.recall(1), eval.fMeasure(1),
                        eval.areaUnderROC(1), eval.matthewsCorrelationCoefficient(1)));
            }

            System.out.println("\n Random forest \n");

            System.out.println("Precision 0: "+ eval.precision(0));
            System.out.println("Precision 1: "+eval.precision(1));
            if(isGroup) {
                System.out.println("Precision 2: "+eval.precision(2));
                System.out.println("Precision 3: "+eval.precision(3));
            }

            System.out.println("\n");
            System.out.println("Recall 0: "+eval.recall(0));
            System.out.println("Recall 1: "+eval.recall(1));
            if(isGroup) {
                System.out.println("Recall 2: "+eval.recall(2));
                System.out.println("Recall 3: "+eval.recall(3));
            }

            System.out.println("\n");
            System.out.println("F-Measure 0:  "+eval.fMeasure(0));
            System.out.println("F-Measure 1:  "+eval.fMeasure(1));
            if(isGroup) {
                System.out.println("F-Measure 2:  "+eval.fMeasure(2));
                System.out.println("F-Measure 3:  "+eval.fMeasure(3));
            }

            System.out.println("\n");
            System.out.println("AUC 0 : "+ eval.areaUnderROC(0));
            System.out.println("AUC 1 : "+ eval.areaUnderROC(1));
            if(isGroup) {
                System.out.println("AUC 2 : "+ eval.areaUnderROC(2));
                System.out.println("AUC 3 : "+ eval.areaUnderROC(3));
            }

            System.out.println("\n");
            System.out.println("MCC 0 : "+ eval.matthewsCorrelationCoefficient(0));
            System.out.println("MCC 1 : "+ eval.matthewsCorrelationCoefficient(1));
            if(isGroup) {
                System.out.println("MCC 2 : "+ eval.matthewsCorrelationCoefficient(2));
                System.out.println("MCC 3 : "+ eval.matthewsCorrelationCoefficient(3));
            }

            System.out.println("Class Attribute" + testData.classAttribute());
            FastVector predictions = new FastVector();
            predictions.appendElements(eval.predictions());
            for (int i = 0; i < predictions.size(); i++) {
                NominalPrediction np = (NominalPrediction) predictions.elementAt(i);
                System.out.println("np actual: "+np.actual());
                System.out.println("np predicited " + np.predicted());
                if (np.predicted() == np.actual()) {

                }
            }

        } catch (Exception e) {
            e.printStackTrace();
        }
        return results;
    }


    public List<ClassifierResult> runRandomForestWithTestsSetSMOTE(boolean isGroup) {
        List<ClassifierResult> results = new ArrayList<>();
        String[] options = new String[1];
        RandomForest tree = new RandomForest();
        try {
            SMOTE smote = new SMOTE();
            smote.setOptions(new String[]{"-P", "150"});

            tree.buildClassifier(trainingData);

            FilteredClassifier fc = new FilteredClassifier();
            fc.setFilter(smote);
            fc.setClassifier(tree);

            Evaluation eval = new Evaluation(trainingData);
            eval.evaluateModel(tree, testData);
            System.out.println(eval.toSummaryString());
            System.out.println(eval.toMatrixString());

            if(isGroup) {
                results.add(new ClassifierResult("Random forest","Documentation", eval.precision(0), eval.recall(0), eval.fMeasure(0),
                        eval.areaUnderROC(0), eval.matthewsCorrelationCoefficient(0)));
                results.add(new ClassifierResult("Random forest","Visual representation", eval.precision(1), eval.recall(1), eval.fMeasure(1),
                        eval.areaUnderROC(1), eval.matthewsCorrelationCoefficient(1)));
                results.add(new ClassifierResult("Random forest","Structure", eval.precision(2), eval.recall(2), eval.fMeasure(2),
                        eval.areaUnderROC(2), eval.matthewsCorrelationCoefficient(2)));
                results.add(new ClassifierResult("Random forest","Functional", eval.precision(3), eval.recall(3), eval.fMeasure(3),
                        eval.areaUnderROC(3), eval.matthewsCorrelationCoefficient(3)));
            }
            else {
                results.add(new ClassifierResult("Random forest","Evolvability", eval.precision(0), eval.recall(0), eval.fMeasure(0),
                        eval.areaUnderROC(0), eval.matthewsCorrelationCoefficient(0)));
                results.add(new ClassifierResult("Random forest","Functional", eval.precision(1), eval.recall(1), eval.fMeasure(1),
                        eval.areaUnderROC(1), eval.matthewsCorrelationCoefficient(1)));
            }

            System.out.println("\n Random forest \n");

            System.out.println("Precision 0: "+ eval.precision(0));
            System.out.println("Precision 1: "+eval.precision(1));
            if(isGroup) {
                System.out.println("Precision 2: "+eval.precision(2));
                System.out.println("Precision 3: "+eval.precision(3));
            }

            System.out.println("\n");
            System.out.println("Recall 0: "+eval.recall(0));
            System.out.println("Recall 1: "+eval.recall(1));
            if(isGroup) {
                System.out.println("Recall 2: "+eval.recall(2));
                System.out.println("Recall 3: "+eval.recall(3));
            }

            System.out.println("\n");
            System.out.println("F-Measure 0:  "+eval.fMeasure(0));
            System.out.println("F-Measure 1:  "+eval.fMeasure(1));
            if(isGroup) {
                System.out.println("F-Measure 2:  "+eval.fMeasure(2));
                System.out.println("F-Measure 3:  "+eval.fMeasure(3));
            }

            System.out.println("\n");
            System.out.println("AUC 0 : "+ eval.areaUnderROC(0));
            System.out.println("AUC 1 : "+ eval.areaUnderROC(1));
            if(isGroup) {
                System.out.println("AUC 2 : "+ eval.areaUnderROC(2));
                System.out.println("AUC 3 : "+ eval.areaUnderROC(3));
            }

            System.out.println("\n");
            System.out.println("MCC 0 : "+ eval.matthewsCorrelationCoefficient(0));
            System.out.println("MCC 1 : "+ eval.matthewsCorrelationCoefficient(1));
            if(isGroup) {
                System.out.println("MCC 2 : "+ eval.matthewsCorrelationCoefficient(2));
                System.out.println("MCC 3 : "+ eval.matthewsCorrelationCoefficient(3));
            }

        } catch (Exception e) {
            e.printStackTrace();
        }
        return results;
    }


    public List<ClassifierResult> runNaiveBayesWithTestsSet(boolean isGroup) {
        List<ClassifierResult> results = new ArrayList<>();
        String[] options = new String[1];
        NaiveBayes tree = new NaiveBayes();
        try {
            tree.buildClassifier(trainingData);
            Evaluation eval = new Evaluation(trainingData);
            eval.evaluateModel(tree, testData);
            System.out.println(eval.toSummaryString());
            System.out.println(eval.toMatrixString());

            if(isGroup) {
                results.add(new ClassifierResult("Naive Bayes","Documentation", eval.precision(0), eval.recall(0), eval.fMeasure(0),
                        eval.areaUnderROC(0), eval.matthewsCorrelationCoefficient(0)));
                results.add(new ClassifierResult("Naive Bayes","Visual representation", eval.precision(1), eval.recall(1), eval.fMeasure(1),
                        eval.areaUnderROC(1), eval.matthewsCorrelationCoefficient(1)));
                results.add(new ClassifierResult("Naive Bayes", "Structure", eval.precision(2), eval.recall(2), eval.fMeasure(2),
                        eval.areaUnderROC(2), eval.matthewsCorrelationCoefficient(2)));
                results.add(new ClassifierResult("Naive Bayes","Functional", eval.precision(3), eval.recall(3), eval.fMeasure(3),
                        eval.areaUnderROC(3), eval.matthewsCorrelationCoefficient(3)));
            }
            else {
                results.add(new ClassifierResult("Naive Bayes","Evolvability", eval.precision(0), eval.recall(0), eval.fMeasure(0),
                        eval.areaUnderROC(0), eval.matthewsCorrelationCoefficient(0)));
                results.add(new ClassifierResult("Naive Bayes","Functional", eval.precision(1), eval.recall(1), eval.fMeasure(1),
                        eval.areaUnderROC(1), eval.matthewsCorrelationCoefficient(1)));
            }

            System.out.println("\n Naive Bayes \n");

            System.out.println("Precision 0: "+ eval.precision(0));
            System.out.println("Precision 1: "+eval.precision(1));
            if(isGroup) {
                System.out.println("Precision 2: "+eval.precision(2));
                System.out.println("Precision 3: "+eval.precision(3));
            }

            System.out.println("\n");
            System.out.println("Recall 0: "+eval.recall(0));
            System.out.println("Recall 1: "+eval.recall(1));
            if(isGroup) {
                System.out.println("Recall 2: " + eval.recall(2));
                System.out.println("Recall 3: " + eval.recall(3));
            }

            System.out.println("\n");
            System.out.println("F-Measure 0:  "+eval.fMeasure(0));
            System.out.println("F-Measure 1:  "+eval.fMeasure(1));
            if(isGroup) {
                System.out.println("F-Measure 2:  "+eval.fMeasure(2));
                System.out.println("F-Measure 3:  "+eval.fMeasure(3));
            }

            System.out.println("\n");
            System.out.println("AUC 0 : "+ eval.areaUnderROC(0));
            System.out.println("AUC 1 : "+ eval.areaUnderROC(1));
            if(isGroup) {
                System.out.println("AUC 2 : " + eval.areaUnderROC(2));
                System.out.println("AUC 3 : " + eval.areaUnderROC(3));
            }

            System.out.println("\n");
            System.out.println("MCC 0 : "+ eval.matthewsCorrelationCoefficient(0));
            System.out.println("MCC 1 : "+ eval.matthewsCorrelationCoefficient(1));
            if(isGroup) {
                System.out.println("MCC 2 : "+ eval.matthewsCorrelationCoefficient(2));
                System.out.println("MCC 3 : "+ eval.matthewsCorrelationCoefficient(3));
            }

        } catch (Exception e) {
            e.printStackTrace();
        }
        return results;
    }



    public List<ClassifierResult> runNaiveBayesWithTestsSetLeaveOneOut(boolean isGroup) {
        List<ClassifierResult> results = new ArrayList<>();
        NaiveBayes tree = new NaiveBayes();
        try {
            tree.buildClassifier(trainingData);
            Evaluation eval = new Evaluation(trainingData);
            eval.evaluateModel(tree, testData);

            if(isGroup) {
                results.add(new ClassifierResult("Naive Bayes","Documentation", eval.precision(0), eval.recall(0), eval.fMeasure(0),
                        eval.areaUnderROC(0), eval.matthewsCorrelationCoefficient(0)));
                results.add(new ClassifierResult("Naive Bayes","Visual representation", eval.precision(1), eval.recall(1), eval.fMeasure(1),
                        eval.areaUnderROC(1), eval.matthewsCorrelationCoefficient(1)));
                results.add(new ClassifierResult("Naive Bayes", "Structure", eval.precision(2), eval.recall(2), eval.fMeasure(2),
                        eval.areaUnderROC(2), eval.matthewsCorrelationCoefficient(2)));
                results.add(new ClassifierResult("Naive Bayes","Functional", eval.precision(3), eval.recall(3), eval.fMeasure(3),
                        eval.areaUnderROC(3), eval.matthewsCorrelationCoefficient(3)));
            }
            else {
                results.add(new ClassifierResult("Naive Bayes","Evolvability", eval.precision(0), eval.recall(0), eval.fMeasure(0),
                        eval.areaUnderROC(0), eval.matthewsCorrelationCoefficient(0)));
                results.add(new ClassifierResult("Naive Bayes","Functional", eval.precision(1), eval.recall(1), eval.fMeasure(1),
                        eval.areaUnderROC(1), eval.matthewsCorrelationCoefficient(1)));
            }

            System.out.println("Class Attribute" + testData.classAttribute());
            FastVector predictions = new FastVector();
            predictions.appendElements(eval.predictions());
            for (int i = 0; i < predictions.size(); i++) {
                NominalPrediction np = (NominalPrediction) predictions.elementAt(i);
                System.out.println("np actual: "+np.actual());
                System.out.println("np predicited " + np.predicted());
                if (np.predicted() == np.actual()) {

                }
            }

        } catch (Exception e) {
            e.printStackTrace();
        }
        return results;
    }


    public List<ClassifierResult> runNaiveBayesWithTestsSetSMOTE(boolean isGroup) {
        List<ClassifierResult> results = new ArrayList<>();
        String[] options = new String[1];

        NaiveBayes tree = new NaiveBayes();
        try {

            SMOTE smote = new SMOTE();
            smote.setOptions(new String[]{"-P","150"});

            tree.buildClassifier(trainingData);
            System.out.println(tree);

            FilteredClassifier fc = new FilteredClassifier();
            fc.setFilter(smote);
            fc.setClassifier(tree);

            Evaluation eval = new Evaluation(trainingData);
            eval.evaluateModel(tree, testData);
            System.out.println(eval.toSummaryString());
            System.out.println(eval.toMatrixString());

            if(isGroup) {
                results.add(new ClassifierResult("Naive Bayes","Documentation", eval.precision(0), eval.recall(0), eval.fMeasure(0),
                        eval.areaUnderROC(0), eval.matthewsCorrelationCoefficient(0)));
                results.add(new ClassifierResult("Naive Bayes","Visual representation", eval.precision(1), eval.recall(1), eval.fMeasure(1),
                        eval.areaUnderROC(1), eval.matthewsCorrelationCoefficient(1)));
                results.add(new ClassifierResult("Naive Bayes", "Structure", eval.precision(2), eval.recall(2), eval.fMeasure(2),
                        eval.areaUnderROC(2), eval.matthewsCorrelationCoefficient(2)));
                results.add(new ClassifierResult("Naive Bayes","Functional", eval.precision(3), eval.recall(3), eval.fMeasure(3),
                        eval.areaUnderROC(3), eval.matthewsCorrelationCoefficient(3)));
            }
            else {
                results.add(new ClassifierResult("Naive Bayes","Evolvability", eval.precision(0), eval.recall(0), eval.fMeasure(0),
                        eval.areaUnderROC(0), eval.matthewsCorrelationCoefficient(0)));
                results.add(new ClassifierResult("Naive Bayes","Functional", eval.precision(1), eval.recall(1), eval.fMeasure(1),
                        eval.areaUnderROC(1), eval.matthewsCorrelationCoefficient(1)));
            }

            System.out.println("\n Naive Bayes \n");

            System.out.println("Precision 0: "+ eval.precision(0));
            System.out.println("Precision 1: "+eval.precision(1));
            if(isGroup) {
                System.out.println("Precision 2: "+eval.precision(2));
                System.out.println("Precision 3: "+eval.precision(3));
            }

            System.out.println("\n");
            System.out.println("Recall 0: "+eval.recall(0));
            System.out.println("Recall 1: "+eval.recall(1));
            if(isGroup) {
                System.out.println("Recall 2: " + eval.recall(2));
                System.out.println("Recall 3: " + eval.recall(3));
            }

            System.out.println("\n");
            System.out.println("F-Measure 0:  "+eval.fMeasure(0));
            System.out.println("F-Measure 1:  "+eval.fMeasure(1));
            if(isGroup) {
                System.out.println("F-Measure 2:  "+eval.fMeasure(2));
                System.out.println("F-Measure 3:  "+eval.fMeasure(3));
            }

            System.out.println("\n");
            System.out.println("AUC 0 : "+ eval.areaUnderROC(0));
            System.out.println("AUC 1 : "+ eval.areaUnderROC(1));
            if(isGroup) {
                System.out.println("AUC 2 : " + eval.areaUnderROC(2));
                System.out.println("AUC 3 : " + eval.areaUnderROC(3));
            }

            System.out.println("\n");
            System.out.println("MCC 0 : "+ eval.matthewsCorrelationCoefficient(0));
            System.out.println("MCC 1 : "+ eval.matthewsCorrelationCoefficient(1));
            if(isGroup) {
                System.out.println("MCC 2 : "+ eval.matthewsCorrelationCoefficient(2));
                System.out.println("MCC 3 : "+ eval.matthewsCorrelationCoefficient(3));
            }

        } catch (Exception e) {
            e.printStackTrace();
        }
        return results;
    }


    public void runRandomForestWithSMOTE() {
        try {
            System.out.println("Before smote " +data.numInstances());
            data.setClassIndex(data.numAttributes()-1);

            Random random = new Random(1);

            SMOTE smote = new SMOTE();
            smote.setOptions(new String[]{"-P","150"});

            String[] options = new String[1];
            RandomForest tree = new RandomForest();
            tree.buildClassifier(data);
            FilteredClassifier fc = new FilteredClassifier();
            fc.setFilter(smote);
            fc.setClassifier(tree);
            Evaluation eval = new Evaluation(data);
            eval.crossValidateModel(fc, data, 10, random);
            System.out.println(eval.toSummaryString());

            System.out.println("MCC class 0:   " +eval.matthewsCorrelationCoefficient(0));
            System.out.println("MCC class 1:   " +eval.matthewsCorrelationCoefficient(1));
            System.out.println("MCC class 2:   " +eval.matthewsCorrelationCoefficient(2));
            System.out.println("MCC class 3:   " +eval.matthewsCorrelationCoefficient(3));
            System.out.println("MCC class 4:   " +eval.matthewsCorrelationCoefficient(4));
            System.out.println("MCC class 5:   " +eval.matthewsCorrelationCoefficient(5));
            System.out.println("MCC class 6:   " +eval.matthewsCorrelationCoefficient(6));
            System.out.println("MCC class 7:   " +eval.matthewsCorrelationCoefficient(7));
            System.out.println("MCC class 8:   " +eval.matthewsCorrelationCoefficient(8));
            System.out.println("MCC class 9:   " +eval.matthewsCorrelationCoefficient(9));
            System.out.println("MCC class 10:   " +eval.matthewsCorrelationCoefficient(10));

            for(int i=0; i<4; i++) {
                System.out.println("F-Measure " + eval.fMeasure(i));
            }

            System.out.println("Weighted MCC:   " +eval.weightedMatthewsCorrelation());

            System.out.println("Precision 0: "+ eval.precision(0));
            System.out.println("Recall 0: "+eval.recall(0));
            System.out.println("Precision 1: "+eval.precision(1));
            System.out.println("Recall 1: "+eval.recall(1));
            System.out.println("AUC 0 : "+ eval.areaUnderROC(0));
            System.out.println("AUC 1 : "+ eval.areaUnderROC(1));
            System.out.println(eval.toMatrixString());

            printInfoGain(data);


        } catch (Exception e) {
            e.printStackTrace();
        }

    }


    public void runRandomForestWithSMOTEAndUndersampling() {
        try {
            System.out.println("Before smote " +data.numInstances());
            data.setClassIndex(data.numAttributes()-1);

            Random random = new Random(1);

            SMOTE smote = new SMOTE();
            smote.setOptions(new String[]{"-P","150"});

            String[] options = new String[1];
            RandomForest tree = new RandomForest();
            tree.buildClassifier(data);
            Resample resample = createResampleFilter();
            MultiFilter mf = new MultiFilter();
            Filter[] filters = new Filter[2];
            filters[0] = resample;
            filters[1] = smote;
            mf.setFilters(filters);
            Evaluation eval = new Evaluation(data);

            FilteredClassifier fc = new FilteredClassifier();
            fc.setFilter(mf);
            fc.setClassifier(tree);

            eval.crossValidateModel(fc, data, 10, random);
            System.out.println(eval.toSummaryString());

            System.out.println("Weighted MCC:   " +eval.weightedMatthewsCorrelation());

            System.out.println("Precision 0: "+ eval.precision(0));
            System.out.println("Recall 0: "+eval.recall(0));
            System.out.println("Precision 1: "+eval.precision(1));
            System.out.println("Recall 1: "+eval.recall(1));
            System.out.println("AUC 0 : "+ eval.areaUnderROC(0));
            System.out.println("AUC 1 : "+ eval.areaUnderROC(1));
            System.out.println(eval.toMatrixString());

            printInfoGain(data);


        } catch (Exception e) {
            e.printStackTrace();
        }

    }


    public void runRandomForestWithUndersampling() {
        try {
            System.out.println("Before resample " +data.numInstances());
            data.setClassIndex(data.numAttributes()-1);

            Resample resample = createResampleFilter();

            String[] options = new String[1];
            RandomForest tree = new RandomForest();
            tree.buildClassifier(data);
            FilteredClassifier fc = new FilteredClassifier();
            fc.setFilter(resample);
            fc.setClassifier(tree);
            Evaluation eval = new Evaluation(data);
            eval.crossValidateModel(fc, data, 10, new Random(1));
            System.out.println(eval.toSummaryString());

            System.out.println("Weighted MCC:   " +eval.weightedMatthewsCorrelation());

            System.out.println("Precision 0: "+ eval.precision(0));
            System.out.println("Recall 0: "+eval.recall(0));
            System.out.println("Precision 1: "+eval.precision(1));
            System.out.println("Recall 1: "+eval.recall(1));
            System.out.println("AUC 0 : "+ eval.areaUnderROC(0));
            System.out.println("AUC 1 : "+ eval.areaUnderROC(1));
            System.out.println(eval.toMatrixString());
        } catch (Exception e) {
            e.printStackTrace();
        }

    }


    public static void main(String args[]) {

        ArrayList<CodeChunk> chunks = new ArrayList<CodeChunk>();
        boolean flag = true;
        try {
            FileInputStream fis = new FileInputStream("JGitRecord");
            ObjectInputStream ois = new ObjectInputStream(fis);
            while (flag) {
                CodeChunk obj = null;
                obj = (CodeChunk) ois.readObject();
                if (obj != null)
                    chunks.add(obj);
                else
                    flag = false;
            }
            fis.close();
            ois.close();
        }catch (Exception e) {
            e.printStackTrace();
        }


        CodeChunksLinker linker = new CodeChunksLinker();
        List<Modification> modifications = linker.splitChangeBased(chunks);

        List<Modification> classifierEntries = new ModificationConverter().extractFeatures(modifications, true);
        Instances data = new ModificationConverter().createTypeInstances(classifierEntries);
        Classifier classifier = new Classifier(data);
        classifier.showFeaturesRank();
        classifier.filterData(new String[]{"-R","22"});
        classifier.showFeaturesRank();
        classifier.runJ48();
        classifier.runRandomForest();
    }


    public List<ClassifierResult> runRandomForestAttributeSelection10TimesSMOTE(boolean isGroup) {

        double precision0 = 0;
        double precision1 = 0;
        double precision2 = 0;
        double precision3 = 0;


        double recall0 = 0;
        double recall1 = 0;
        double recall2 = 0;
        double recall3 = 0;

        double fmeasure0 = 0;
        double fmeasure1 = 0;
        double fmeasure2 = 0;
        double fmeasure3 = 0;

        double auc0 =0;
        double auc1 =0;
        double auc2 =0;
        double auc3 =0;

        double mcc0 = 0;
        double mcc1 = 0;
        double mcc2 = 0;
        double mcc3 = 0;

        List<ClassifierResult> results = new ArrayList<>();
        Random random = new Random(1);

        try {
            InfoGainAttributeEval evalGain = new InfoGainAttributeEval();
            Ranker search = new Ranker();
            search.setOptions(new String[] { "-T", "0.001" });

            AttributeSelection attSelect = new AttributeSelection();
            attSelect.setEvaluator(evalGain);
            attSelect.setSearch(search);
            attSelect.SelectAttributes(data);
            data = attSelect.reduceDimensionality(data);


            for (int i = 0; i < 10; i++) {

                try {
                    System.out.println("Before smote " + data.numInstances());


                    data.setClassIndex(data.numAttributes() - 1);

                    SMOTE smote = new SMOTE();
                    smote.setOptions(new String[]{"-P", "150"});

                    String[] options = new String[1];
                    RandomForest tree = new RandomForest();
                    tree.buildClassifier(data);
                    FilteredClassifier fc = new FilteredClassifier();
                    fc.setFilter(smote);
                    fc.setClassifier(tree);
                    Evaluation eval = new Evaluation(data);
                    eval.crossValidateModel(fc, data, 10, random);
                    System.out.println(eval.toSummaryString());
                    System.out.println(eval.toMatrixString());

                    printInfoGain(data);

                    precision0 = eval.precision(0) + precision0;
                    precision1 = eval.precision(1) + precision1;

                    recall0 = eval.recall(0) + recall0;
                    recall1 = eval.recall(1) + recall1;

                    fmeasure0 = eval.fMeasure(0) + fmeasure0;
                    fmeasure1 = eval.fMeasure(1) + fmeasure1;

                    auc0 = eval.areaUnderROC(0) + auc0;
                    auc1 = eval.areaUnderROC(1) + auc1;

                    mcc0 = eval.matthewsCorrelationCoefficient(0) + mcc0;
                    mcc1 = eval.matthewsCorrelationCoefficient(1) + mcc1;

                    if (isGroup) {
                        precision2 = eval.precision(2) + precision2;
                        precision3 = eval.precision(3) + precision3;
                        recall2 = eval.recall(2) + recall2;
                        recall3 = eval.recall(3) + recall3;
                        fmeasure2 = eval.fMeasure(2) + fmeasure2;
                        fmeasure3 = eval.fMeasure(3) + fmeasure3;
                        auc2 = eval.areaUnderROC(2) + auc2;
                        auc3 = eval.areaUnderROC(3) + auc3;
                        mcc2 = eval.matthewsCorrelationCoefficient(2) + mcc2;
                        mcc3 = eval.matthewsCorrelationCoefficient(3) + mcc3;
                    }


                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }
        catch (Exception e) {
            e.printStackTrace();
        }


        if(isGroup) {
            results.add(new ClassifierResult("Random forest","Documentation", precision0/10, recall0/10, fmeasure0/10,
                    auc0/10, mcc0/10));
            results.add(new ClassifierResult("Random forest","Visual representation", precision1/10, recall1/10, fmeasure1/10,
                    auc1/10, mcc1/10));
            results.add(new ClassifierResult("Random forest","Structure", precision2/10, recall2/10, fmeasure2/10,
                    auc2/10, mcc2/10));
            results.add(new ClassifierResult("Random forest","Functional", precision3/10, recall3/10, fmeasure3/10,
                    auc3/10, mcc3/10));
        }
        else {
            results.add(new ClassifierResult("Random forest","Evolvability", precision0/10, recall0/10, fmeasure0/10,
                    auc0/10, mcc0/10));
            results.add(new ClassifierResult("Random forest","Functional", precision1/10, recall1/10, fmeasure1/10,
                    auc1/10, mcc1/10));
        }

        System.out.println("\n Random forest SMOTE \n");

        System.out.println("Precision 0: "+ precision0/10);
        System.out.println("Precision 1: "+precision1/10);
        if(isGroup) {
            System.out.println("Precision 2: " + precision2 / 10);
            System.out.println("Precision 3: " + precision3 / 10);
        }
        System.out.println("\n");
        System.out.println("Recall 0: "+recall0/10);
        System.out.println("Recall 1: "+recall1/10);
        if(isGroup) {
            System.out.println("Recall 2: " + recall2 / 10);
            System.out.println("Recall 3: " + recall3 / 10);
        }
        System.out.println("\n");
        System.out.println("F-Measure 0:  "+fmeasure0/10);
        System.out.println("F-Measure 1:  "+fmeasure1/10);
        if(isGroup) {
            System.out.println("F-Measure 2:  " + fmeasure2 / 10);
            System.out.println("F-Measure 3:  " + fmeasure3 / 10);
        }
        System.out.println("\n");
        System.out.println("AUC 0 : "+ auc0/10);
        System.out.println("AUC 1 : "+ auc1/10);
        if(isGroup) {
            System.out.println("AUC 2: " + auc2 / 10);
            System.out.println("AUC 3: " + auc3 / 10);
        }
        System.out.println("\n");
        System.out.println("MCC 0 : "+ mcc0/10);
        System.out.println("MCC 1 : "+ mcc1/10);
        if(isGroup) {
            System.out.println("MCC 2 : " + mcc2 / 10);
            System.out.println("MCC 3 : " + mcc3 / 10);
        }
        return results;
    }

    public List<ClassifierResult> runRandomForestAttributeSelection10Times(boolean isGroup) {

        double precision0 = 0;
        double precision1 = 0;
        double precision2 = 0;
        double precision3 = 0;


        double recall0 = 0;
        double recall1 = 0;
        double recall2 = 0;
        double recall3 = 0;

        double fmeasure0 = 0;
        double fmeasure1 = 0;
        double fmeasure2 = 0;
        double fmeasure3 = 0;

        double auc0 =0;
        double auc1 =0;
        double auc2 =0;
        double auc3 =0;

        double mcc0 = 0;
        double mcc1 = 0;
        double mcc2 = 0;
        double mcc3 = 0;

        List<ClassifierResult> results = new ArrayList<>();


        try {
            InfoGainAttributeEval evalGain = new InfoGainAttributeEval();
            Ranker search = new Ranker();
            search.setOptions(new String[] { "-T", "0.001" });

            AttributeSelection attSelect = new AttributeSelection();
            attSelect.setEvaluator(evalGain);
            attSelect.setSearch(search);
            attSelect.SelectAttributes(data);
            data = attSelect.reduceDimensionality(data);
            Random random = new Random(1);

            for (int i = 0; i < 10; i++) {

                try {

                    data.setClassIndex(data.numAttributes() - 1);

                    String[] options = new String[1];
                    RandomForest tree = new RandomForest();
                    tree.buildClassifier(data);
                    Evaluation eval = new Evaluation(data);
                    eval.crossValidateModel(tree, data, 10, random);
                    System.out.println(eval.toSummaryString());
                    System.out.println(eval.toMatrixString());

                    precision0 = eval.precision(0) + precision0;
                    precision1 = eval.precision(1) + precision1;

                    recall0 = eval.recall(0) + recall0;
                    recall1 = eval.recall(1) + recall1;

                    fmeasure0 = eval.fMeasure(0) + fmeasure0;
                    fmeasure1 = eval.fMeasure(1) + fmeasure1;

                    auc0 = eval.areaUnderROC(0) + auc0;
                    auc1 = eval.areaUnderROC(1) + auc1;

                    mcc0 = eval.matthewsCorrelationCoefficient(0) + mcc0;
                    mcc1 = eval.matthewsCorrelationCoefficient(1) + mcc1;

                    if (isGroup) {
                        precision2 = eval.precision(2) + precision2;
                        precision3 = eval.precision(3) + precision3;
                        recall2 = eval.recall(2) + recall2;
                        recall3 = eval.recall(3) + recall3;
                        fmeasure2 = eval.fMeasure(2) + fmeasure2;
                        fmeasure3 = eval.fMeasure(3) + fmeasure3;
                        auc2 = eval.areaUnderROC(2) + auc2;
                        auc3 = eval.areaUnderROC(3) + auc3;
                        mcc2 = eval.matthewsCorrelationCoefficient(2) + mcc2;
                        mcc3 = eval.matthewsCorrelationCoefficient(3) + mcc3;
                    }


                } catch (Exception e) {
                    e.printStackTrace();
                }
            }
        }
        catch (Exception e) {
            e.printStackTrace();
        }


        if(isGroup) {
            results.add(new ClassifierResult("Random forest","Documentation", precision0/10, recall0/10, fmeasure0/10,
                    auc0/10, mcc0/10));
            results.add(new ClassifierResult("Random forest","Visual representation", precision1/10, recall1/10, fmeasure1/10,
                    auc1/10, mcc1/10));
            results.add(new ClassifierResult("Random forest","Structure", precision2/10, recall2/10, fmeasure2/10,
                    auc2/10, mcc2/10));
            results.add(new ClassifierResult("Random forest","Functional", precision3/10, recall3/10, fmeasure3/10,
                    auc3/10, mcc3/10));
        }
        else {
            results.add(new ClassifierResult("Random forest","Evolvability", precision0/10, recall0/10, fmeasure0/10,
                    auc0/10, mcc0/10));
            results.add(new ClassifierResult("Random forest","Functional", precision1/10, recall1/10, fmeasure1/10,
                    auc1/10, mcc1/10));
        }

        System.out.println("\n Random forest SMOTE \n");

        System.out.println("Precision 0: "+ precision0/10);
        System.out.println("Precision 1: "+precision1/10);
        if(isGroup) {
            System.out.println("Precision 2: " + precision2 / 10);
            System.out.println("Precision 3: " + precision3 / 10);
        }
        System.out.println("\n");
        System.out.println("Recall 0: "+recall0/10);
        System.out.println("Recall 1: "+recall1/10);
        if(isGroup) {
            System.out.println("Recall 2: " + recall2 / 10);
            System.out.println("Recall 3: " + recall3 / 10);
        }
        System.out.println("\n");
        System.out.println("F-Measure 0:  "+fmeasure0/10);
        System.out.println("F-Measure 1:  "+fmeasure1/10);
        if(isGroup) {
            System.out.println("F-Measure 2:  " + fmeasure2 / 10);
            System.out.println("F-Measure 3:  " + fmeasure3 / 10);
        }
        System.out.println("\n");
        System.out.println("AUC 0 : "+ auc0/10);
        System.out.println("AUC 1 : "+ auc1/10);
        if(isGroup) {
            System.out.println("AUC 2: " + auc2 / 10);
            System.out.println("AUC 3: " + auc3 / 10);
        }
        System.out.println("\n");
        System.out.println("MCC 0 : "+ mcc0/10);
        System.out.println("MCC 1 : "+ mcc1/10);
        if(isGroup) {
            System.out.println("MCC 2 : " + mcc2 / 10);
            System.out.println("MCC 3 : " + mcc3 / 10);
        }
        return results;
    }

}


