package backend.analysis;

import backend.base.Change;
import backend.base.gerrit_data.*;
import com.google.common.collect.Multimap;
import com.google.gerrit.extensions.api.changes.*;
import com.google.gerrit.extensions.common.ChangeInfo;
import com.google.gerrit.extensions.client.ListChangesOption;
import com.google.gerrit.extensions.common.CommentInfo;
import com.google.gerrit.extensions.common.DiffInfo;
import com.google.gerrit.extensions.common.RevisionInfo;
import com.google.gerrit.extensions.restapi.RestApiException;
import com.urswolfer.gerrit.client.rest.GerritAuthData;
import com.urswolfer.gerrit.client.rest.GerritRestApi;
import com.urswolfer.gerrit.client.rest.GerritRestApiFactory;

import java.sql.Timestamp;
import java.text.SimpleDateFormat;
import java.util.*;

import static java.lang.Math.abs;

public class GerritExtractor {

    private class Revision {

        private String revisionID;
        private RevisionInfo revisionInfo;

        public Revision(String revisionID, RevisionInfo revisionInfo){
            this.revisionID = revisionID;
            this.revisionInfo = revisionInfo;
        }

    }

    private class ReviewRound {

        private Revision newRevision;
        private Revision oldRevision;

        private ReviewRound(Revision newRevision, Revision oldRevision){
            this.newRevision = newRevision;
            this.oldRevision = oldRevision;
        }

    }



    private Multimap<String, Change> changesInDataset;
    private Changes changes;

    private String projectName;


    public GerritExtractor() {}


    public GerritExtractor(Multimap<String, Change> changesInDataset) {
        this.changesInDataset = changesInDataset;
    }


    public void setProjectName(String projectName) {
        this.projectName = projectName;
    }


    public void set(String connectionUrl){
        GerritRestApiFactory gerritRestApiFactory = new GerritRestApiFactory();
        GerritAuthData.Basic authData = new GerritAuthData.Basic(connectionUrl);
        GerritRestApi gerritRestApi = gerritRestApiFactory.create(authData);
        changes = gerritRestApi.changes();
    }


    public List<ReviewDiff> mineRepositoryData(int startpoint, int endpoint) {
        List<ReviewDiff> reviewDiffs = new ArrayList<>();
        while (startpoint >= endpoint) {
            System.out.println("PROGRESS:  "+startpoint);
            List<ReviewDiff> reviewDiffList = mine(changes, startpoint);
            if(reviewDiffList!=null){
                reviewDiffs.addAll(reviewDiffList);
            }
            startpoint--;
        }
        return reviewDiffs;
    }

    private List<ReviewDiff> mine(Changes changes, int id) {
        List<ReviewDiff> reviewDiffs = new ArrayList<>();
        List<ChangeInfo> reviews = mineChange(changes, Integer.toString(id));
        if (reviews.isEmpty()){
            return null;
        }
        if(!reviews.get(0).status.toString().equals("MERGED")){
            return null;
        }
        if(projectName!=null) {
            if (!reviews.get(0).project.equals(projectName)) {
                return null;
            }
        }
        for (ChangeInfo c : reviews) {
            System.out.println("here in mining");
            List<RevisionDiff> revisionDiff = mineAllRevisions(c);
            List<ReviewerInfo> revs = getReviewers(changes, c);
            ReviewDiff reviewDiff = new ReviewDiff(c.changeId, revisionDiff, revs, c.updated);
            if(!revisionDiff.isEmpty()) {
                reviewDiffs.add(reviewDiff);
            }
        }
        return reviewDiffs;
    }


    public Date mineRevisionDate(String reviewId, String revisionID) {
        List<ChangeInfo> reviews = mineChange(changes, reviewId);
        if (reviews.isEmpty()){
            return null;
        }
        for (ChangeInfo c : reviews) {
            Map<String, RevisionInfo> map = c.revisions;
            for(String key : map.keySet()){
                if(key.equals(revisionID)) {
                    return new Date(map.get(key).created.getTime());
                }
            }
        }
        return null;
    }


    private List<ReviewerInfo> getReviewers(Changes changes, ChangeInfo c) {
        List<ReviewerInfo> res = new ArrayList<>();
        int numTries = 1;
        while (true){
            try{
                // Getting the list of comments for each file
                res = changes.id(c._number).listReviewers();
                break;
            } catch(com.urswolfer.gerrit.client.rest.http.HttpStatusException e){
                System.out.println("Unable to get the list of the reviewers");
                break;
            } catch (com.google.gerrit.extensions.restapi.RestApiException e){
                if (numTries > 100){
                    System.out.println("Too many tries! Quitting...");
                    System.out.println("Sort key of the last element: " + c._sortkey);
                    System.exit(-1);
                }
                numTries++;
                System.out.println("--------REQUEST FAILED: doing a new request. Request number " + numTries);
            }
        }
        return res;
    }

    private List<RevisionDiff> mineAllRevisions(ChangeInfo c) {
        List<RevisionDiff> revisionDiffs = new ArrayList<>();
        List<Revision> revisions = orderBaseOnTimestamp(c.revisions);
        String oldRevisionID = revisions.get(0).revisionID;
        revisions.remove(0);
        for (Revision r : revisions) {
            String revisionID = r.revisionID;
            List<FileDiff> files = new ArrayList<>();
            for(String fileName : r.revisionInfo.files.keySet()) {
                System.out.println("==== File name ==== "+fileName);
                if(isJavaFile(fileName)) {
                    DiffInfo fileDiff = computeDiffInfo(fileName, c.changeId, revisionID, oldRevisionID);
                    DiffLineCounter diffLineCounter = new DiffLineCounter();

                    ImportFilter importFilter = new ImportFilter();

                    List<CodeChunk> newCodeChunks = diffLineCounter.countNewDiffLines(fileDiff);
                    List<CodeChunk> oldCodeChunks = diffLineCounter.countOldDiffLines(fileDiff);

                    newCodeChunks = importFilter.filterImports(newCodeChunks);
                    oldCodeChunks = importFilter.filterImports(oldCodeChunks);

                    CodeChunksLinker codeChunksLinker = new CodeChunksLinker();
                    List<Modification> modifications = codeChunksLinker.linkCodeChunks(oldCodeChunks, newCodeChunks);
                    FileDiff fileD = new FileDiff(fileName, modifications, fileDiff);
                    files.add(fileD);
                }
            }
            if(!files.isEmpty()) {
                RevisionDiff revisionDiff = new RevisionDiff(revisionID, files);
                revisionDiffs.add(revisionDiff);
            }
            oldRevisionID = revisionID;
        }
        return revisionDiffs;
    }


    private boolean isJavaFile(String fileName){
        String[] fileNameSplit = fileName.split("\\.");
        if (fileNameSplit[fileNameSplit.length - 1].equals("java")) {
            return true;
        }
        return false;
    }


    public List<CodeChunk> mineDatasetRevisionData(){
        List<CodeChunk> chunks = new ArrayList<>();
        int keyCount = 1;
        for(String key : changesInDataset.keySet()) {
            System.out.println("KEY:  "+keyCount);
            for(Change ch : changesInDataset.get(key)) {
                System.out.println("Here");
                List<ChangeInfo> minedReviews = mineChange(changes, ch.getReviewID());
                System.out.println("size "+minedReviews.size());
                ReviewRound reviewRound = mineRevision(minedReviews, ch.getRevisionID());
                if (reviewRound != null) {
                    System.out.println("not null! ");
                    String f = ch.getFileName();
                    int beginLine = ch.getLineBlock().getBeginLine();
                    int endLine = ch.getLineBlock().getEndLine();
                    if (reviewRound.newRevision.revisionInfo.files.containsKey(f) && reviewRound.oldRevision != null) {

                        System.out.println("Revision "+reviewRound.newRevision.revisionID);
                        System.out.println("file name "+f);


                        DiffInfo fileDiff = computeDiffInfo(f, ch.getReviewID(), reviewRound.oldRevision.revisionID, reviewRound.newRevision.revisionID);
                        DiffLineCounter diffLineCounter = new DiffLineCounter();
                        List<CodeChunk> codeChunks;
                        if(beginLine >= 0) {
                            codeChunks = diffLineCounter.countNewDiffLines(fileDiff);
                        }
                        else {
                            codeChunks = diffLineCounter.countOldDiffLines(fileDiff);
                        }

                        try {
                            List<Comment> comments = getComments(reviewRound.oldRevision.revisionID, ch.getReviewID());

                            for(CodeChunk c : codeChunks){
                                List<Comment> commentsToInsert = new ArrayList<>();
                                if(beginLine<0) {
                                    for (Comment comment : comments) {
                                        if (f.equals(comment.getFileName()) && comment.getLine() <= c.getBeginLine() + 2 && comment.getLine() >= c.getBeginLine() - 2) {
                                            System.out.println(" \n Life is good... sort of..." + comment.getLine() + "\n\n" + comment.getMessage());
                                            commentsToInsert.add(comment);
                                        }
                                    }
                                }
                                c.setComments(commentsToInsert);
                            }
                        } catch (RestApiException e) {
                            e.printStackTrace();
                        }


                        CodeChunk filteredOldCodeChunk = removeUnrelatedCodeChunks(codeChunks, beginLine, endLine);
                        if(filteredOldCodeChunk!=null) {
                            filteredOldCodeChunk.setRelatedChange(ch.getChangeID(), ch.getFileName(), ch.getLineBlock().getBeginLine(), ch.getLineBlock().getEndLine(), ch.getCategory(),  ch.getType(), ch.getSubtype());
                            chunks.add(filteredOldCodeChunk);
                        }
                    } else if (reviewRound.oldRevision != null) {
                        if (reviewRound.oldRevision.revisionInfo.files.containsKey(f)) {
                            DiffInfo fileDiff = computeDiffInfo(f, ch.getReviewID(), reviewRound.oldRevision.revisionID, reviewRound.newRevision.revisionID);
                            DiffLineCounter diffLineCounter = new DiffLineCounter();
                            List<CodeChunk> codeChunks;
                            if(beginLine >= 0) {
                                codeChunks = diffLineCounter.countNewDiffLines(fileDiff);
                            }
                            else {
                                codeChunks = diffLineCounter.countOldDiffLines(fileDiff);
                            }
                            CodeChunk filteredOldCodeChunk = removeUnrelatedCodeChunks(codeChunks, beginLine, endLine);
                            if(filteredOldCodeChunk!=null) {
                                filteredOldCodeChunk.setRelatedChange(ch.getChangeID(), ch.getFileName(), ch.getLineBlock().getBeginLine(), ch.getLineBlock().getEndLine(), ch.getCategory(), ch.getType(), ch.getSubtype());
                                chunks.add(filteredOldCodeChunk);
                            }
                        }
                    }
                }
            }
            keyCount++;
        }
        return chunks;
    }

    public Multimap<String, ReviewDiff> relateCodeChunks(List<CodeChunk> relatedCodeChunks) {
        return null;
    }

    private CodeChunk removeUnrelatedCodeChunks(List<CodeChunk> modificationsToFilter, int begin, int end) {
        int beginLine = abs(begin);
        int endLine = abs(end);
        int index = 0;
        while(index<modificationsToFilter.size()){
                if ((modificationsToFilter.get(index).getBeginLine()-2 <= beginLine) && (endLine <= modificationsToFilter.get(index).getEndLine()+2)) {
                    return modificationsToFilter.get(index);
                } else {
                    index++;
                }
        }
        return null;
    }

    private List<ChangeInfo> mineChange(Changes changes, String ID) {
        List<ChangeInfo> reviews;
        int numTries = 1;
        Date date = new GregorianCalendar(2014, Calendar.FEBRUARY, 11).getTime();
        while (true) {
            try {
                System.out.println("Mining:  "+ID);
                reviews = changes.query(ID).withOptions(ListChangesOption.ALL_FILES, ListChangesOption.ALL_REVISIONS, ListChangesOption.DETAILED_ACCOUNTS, ListChangesOption.ALL_COMMITS).get();
                break;
            } catch (com.google.gerrit.extensions.restapi.RestApiException e) {
                if (numTries > 100) {
                    System.out.println("Too many tries! Quitting...");
                    System.exit(-1);
                }
                numTries++;
                System.out.println("--------REQUEST FAILED: doing a new request. Request number " + numTries);
            }
        }
        return reviews;
    }


    private ReviewRound mineRevision(List<ChangeInfo> minedReviews, String revisionID) {
        int reviewIndex = 0;
        if(!minedReviews.isEmpty()) {
            if (minedReviews.size() > 1) {
                for (int i=0; i<minedReviews.size(); i++) {
                    if(minedReviews.get(i).revisions.containsKey(revisionID)) {
                        reviewIndex = i;
                    }
                }
            }
            if (minedReviews.get(reviewIndex).revisions.containsKey(revisionID)) {
                List<Revision> revisions = orderBaseOnTimestamp(minedReviews.get(reviewIndex).revisions);
                int index = -1;
                int counter = 0;
                for (Revision r : revisions) {
                    if (r.revisionID.equals(revisionID)) {
                        index = counter;
                    } else {
                        counter++;
                    }
                }
                if (index > 0) {
                    return new ReviewRound(revisions.get(index), revisions.get(index - 1));
                } else {
                    return new ReviewRound(revisions.get(index), null);
                }
            }
        }
        return null;
    }


    private DiffInfo computeDiffInfo(String fileName, String reviewID, String oldRevisionID, String newRevisionID){
        DiffInfo diffInfoPrevious = null;
        try {
            ChangeApi changeApi = setChangeApi(reviewID);
            RevisionApi revisionApi = changeApi.revision(newRevisionID);
            FileApi fileApi = revisionApi.file(fileName);
            diffInfoPrevious = fileApi.diff(oldRevisionID);
        } catch (RestApiException e) {
            e.printStackTrace();
        }
        return diffInfoPrevious;
    }


    private ChangeApi setChangeApi(String reviewId){
        ChangeApi changeApi=null;
        try{
            changeApi=changes.id(reviewId);
        }catch(RestApiException e){
            e.printStackTrace();
        }
        return changeApi;
    }


    private List<Revision> orderBaseOnTimestamp(Map<String, RevisionInfo> map){
        List<Revision> orderedRevisions = new ArrayList<>();
        for(Map.Entry<String, RevisionInfo> entry : map.entrySet()){
            if(orderedRevisions.isEmpty()){
                orderedRevisions.add(new Revision(entry.getKey(), entry.getValue()));
            }
            else{
                Timestamp timeStamp = entry.getValue().created;
                boolean elementInserted = false;
                int counter = 0;
                while((elementInserted==false)&&(counter<orderedRevisions.size())){
                    Timestamp currentTimeStamp = orderedRevisions.get(counter).revisionInfo.created;
                    if(timeStamp.before(currentTimeStamp)){
                        orderedRevisions.add(counter, new Revision(entry.getKey(), entry.getValue()));
                        elementInserted = true;
                    }
                    else{
                        counter++;
                    }
                }
                if(elementInserted==false){
                    orderedRevisions.add(new Revision(entry.getKey(), entry.getValue()));
                }
            }
        }
        return orderedRevisions;
    }


    private List<Comment> getComments(String revisionId, String reviewId)
            throws RestApiException {
        ChangeApi changeApi = setChangeApi(reviewId);
        Map<String, List<CommentInfo>> commentInfos = changeApi.revision(revisionId).comments();
        Set<String> keys = commentInfos.keySet();
        List<Comment> comments = new ArrayList<>();
        for(String s : keys){
            for(CommentInfo i : commentInfos.get(s)){
                if((i!=null) && (i.message != null) && (i.line != null)) {
                    comments.add(new Comment(i.line, i.message, s));
                }
                else if ( (i!=null) && i.line == null && (i.message != null)){
                    System.out.println("\n\n\n======NULL:  " + i.message);
                }
            }
        }
        return comments;
    }
}
