package backend.analysis;

import backend.base.gerrit_data.CodeChunk;
import backend.base.gerrit_data.DiffLine;
import backend.base.gerrit_data.Modification;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.Multimap;
import org.apache.commons.text.similarity.LevenshteinDistance;

import java.util.ArrayList;
import java.util.List;

/**
 * This class links code chunks together to originate Modification objects.
 * The link is done based on the Levenshtein Distance between their code.
 */

public class CodeChunksLinker {

    private class ChunksDistance {

        private double distance;
        private CodeChunk destinationChunk;

        private ChunksDistance(double distance, CodeChunk destinationChunk) {
            this.distance = distance;
            this.destinationChunk = destinationChunk;
        }
    }


    private class ChunkToRemove {

        private ChunksDistance chunkToRemove;
        private CodeChunk keyChunk;

        private ChunkToRemove(ChunksDistance chunkToRemove, CodeChunk keyChunk){
            this.chunkToRemove = chunkToRemove;
            this.keyChunk = keyChunk;
        }
    }

    /**
     * Splits the list of CodeChunk object received as paramenter in chunks belonging to the old diff and to the new diff,
     * using their line number
     * Then, it call the method linkCodeChunks to generate the modifications
     * @param chunks - List of chunks to separate
     * @return modifications - List of modifications generated
     */

    public List<Modification> splitChangeBased(List<CodeChunk> chunks) {
        List<Modification> modifications = new ArrayList<>();
        Multimap<String, CodeChunk> chunksByChange = ArrayListMultimap.create();
        for (CodeChunk ck : chunks) {
            chunksByChange.put(ck.getChangeID(), ck);
        }
        for (String changeID : chunksByChange.keySet()) {
            List<CodeChunk> oldChunks = new ArrayList<>();
            List<CodeChunk> newChunks = new ArrayList<>();
            for (CodeChunk chunk : chunksByChange.get(changeID)) {
                if (chunk.getChangeBeginLine() >= 0) {
                    newChunks.add(chunk);
                } else {
                    oldChunks.add(chunk);
                }
            }
            modifications.addAll(linkCodeChunks(oldChunks, newChunks));
        }
        return modifications;
    }

    public List<Modification> linkCodeChunks(List<CodeChunk> oldCodeChunks, List<CodeChunk> newCodeChunks) {
        List<Modification> modifications = new ArrayList<>();

        if (oldCodeChunks.size() == 0) {
            for (int i = 0; i < newCodeChunks.size(); i++) {
                modifications.add(new Modification(null, newCodeChunks.get(i)));
            }
            return modifications;
        }
        if (newCodeChunks.size() == 0) {
            for (int i = 0; i < oldCodeChunks.size(); i++) {
                modifications.add(new Modification(oldCodeChunks.get(i), null));
            }
            return modifications;
        }


        if (oldCodeChunks.size() == newCodeChunks.size()) {
            modifications.addAll(linkOddCodeChunks(oldCodeChunks, newCodeChunks, true));
            return modifications;
        }


        if (oldCodeChunks.size() > newCodeChunks.size()) {
            modifications.addAll(linkOddCodeChunks(oldCodeChunks, newCodeChunks, true));
            return modifications;
        } else {
            modifications.addAll(linkOddCodeChunks(newCodeChunks, oldCodeChunks, false));
            return modifications;
        }
    }

    private String concatenateString(List<DiffLine> diffLines) {
        String sumLine = "";
        for(DiffLine d : diffLines) {
            sumLine = sumLine.concat(d.getDiffLine());
        }
        return sumLine;
    }

    /**
     * Links old and new code chunks to generate the modification.
     * It constructs a graph of old chunks and new chunks and then it computes the similarity of each one of them.
     * Then, it takes the couple with the highest similarity, match them, remove then from the graph and iteratively proceeds.
     * Finally, once no pair can be formed anymore, it takes the remaining new or old nodes and creates modifications linked to a null chunk.
     *
     * @param biggerCodeChunksList
     * @param smallerCodeChunksList
     * @param oldFirst
     * @return modifications - list of created modifications.
     */

    private List<Modification> linkOddCodeChunks(List<CodeChunk> biggerCodeChunksList, List<CodeChunk> smallerCodeChunksList, boolean oldFirst) {
        List<Modification> modifications = new ArrayList<>();
        Multimap<CodeChunk, ChunksDistance> chunksGraph = ArrayListMultimap.create();
        List<CodeChunk> remainingNodes = new ArrayList<>();


        for (CodeChunk bigCodeChunk : biggerCodeChunksList) {
            String bigChunkLine = concatenateString(bigCodeChunk.getLines());
            for (CodeChunk smallCodeChunk : smallerCodeChunksList) {
                String smallChunkLine = concatenateString(smallCodeChunk.getLines());
                double distance = computeChangeRelatedSimilarity(bigChunkLine, smallChunkLine);
                chunksGraph.put(bigCodeChunk, new ChunksDistance(distance, smallCodeChunk));
            }
        }


        for(int index = 0; index<smallerCodeChunksList.size(); index++) {
            double maxDistance = -1;
            CodeChunk priorityChunk = null;
            CodeChunk removeChunk = null;
            int lineDistance = -1;
            double finalDistance = 0;
            int finalLineDistance = 0;
            for (CodeChunk key : chunksGraph.keySet()) {
                //System.out.println("key: " + key.getBeginLine());
                for (ChunksDistance cd : chunksGraph.get(key)) {
                    //System.out.println("cd: " + cd.destinationChunk.getBeginLine());
                    //System.out.println("cd distance: " + cd.distance);
                    if(maxDistance == -1){
                        maxDistance = cd.distance;
                    }
                    if(cd.distance == maxDistance){
                        if(priorityChunk != null && removeChunk!=null) {
                            int lineDistanceAmongChunks = Math.abs(priorityChunk.getBeginLine()-removeChunk.getBeginLine());
                            if(lineDistance == -1){
                                lineDistance = lineDistanceAmongChunks;
                            }
                            else {
                                if(lineDistanceAmongChunks<lineDistance) {
                                    lineDistance = lineDistanceAmongChunks;
                                    priorityChunk = key;
                                    removeChunk = cd.destinationChunk;
                                }
                            }
                        }
                        else {
                            maxDistance = cd.distance;
                            priorityChunk = key;
                            removeChunk = cd.destinationChunk;
                        }
                    }
                    else if(cd.distance>maxDistance) { //controllare
                        maxDistance = cd.distance;
                        priorityChunk = key;
                        removeChunk = cd.destinationChunk;
                    }
                    finalDistance = cd.distance;
                    finalLineDistance = lineDistance;
                }
            }

            //System.out.println("line distance" + finalLineDistance);
            //System.out.println("Distance:  " + finalDistance);
            if (oldFirst) {
                modifications.add(new Modification(priorityChunk, removeChunk));
            } else {
                modifications.add(new Modification(removeChunk, priorityChunk));
            }

            chunksGraph.removeAll(priorityChunk);
            List<ChunkToRemove> chunksToRemove = new ArrayList<>();
            for(CodeChunk key : chunksGraph.keySet()) {
                for(ChunksDistance chunksDistance : chunksGraph.get(key)) {
                    if(chunksDistance.destinationChunk == removeChunk) {
                        chunksToRemove.add(new ChunkToRemove(chunksDistance, key));
                    }
                }
            }


           for(ChunkToRemove chunkToRemove : chunksToRemove){
               if(chunksGraph.get(chunkToRemove.keyChunk).size() == 1) {
                   remainingNodes.add(chunkToRemove.keyChunk);
               }
               chunksGraph.remove(chunkToRemove.keyChunk, chunkToRemove.chunkToRemove);
           }
        }

        for(CodeChunk remainingChunk : remainingNodes) {
            if (oldFirst) {
                modifications.add(new Modification(remainingChunk, null));
            } else {
                modifications.add(new Modification(null, remainingChunk));
            }
        }


        return modifications;
    }

    public static double computeChangeRelatedSimilarity(String s1, String s2) {
        String longer = s1, shorter = s2;
        LevenshteinDistance distance = new LevenshteinDistance();
        if (s1.length() < s2.length()) { // longer should always have greater length
            longer = s2; shorter = s1;
        }
        int longerLength = longer.length();
        if (longerLength == 0) { return 1.0; /* both strings are zero length */ }

        if(longer.length()>500000 && shorter.length()>500000){
            return 10000;
        }

        return (longerLength - distance.apply(longer, shorter)) / (double) longerLength;
    }
}
