/**
 * 
 */
package core;

import java.util.HashMap;

/**
 * @author Charles Winterhalter
 * Class computing melting temperatures (Tm)
 * for a given pair of oligonucleotides
 */
public class TmCalculator {

	// Attributes
	public HashMap<String, double[]> deltaValues; // deltaH=values[0] and deltaS=values[1]
	public double saltConcentration = 50.0; // in mM
	public double primerConcentration = 1000.0; // in nM
	
	/**
	 * Constructor
	 * Initiates the calculator by loading SantaLucia (1998) data
	 * (consecutive dinucleotide energy profiles)
	 */
	public TmCalculator (double salts, double primers) {
		if (salts != 50.0) this.saltConcentration = salts;
		if (primers != 1000.0) this.primerConcentration = primers;
		initiate();
	}
	
	// Loads SantaLucia energy profiles
	public void initiate () {
		this.deltaValues = new HashMap<String, double[]>();
		double[] tmp = new double[2];
		tmp[0] = -7.9; tmp[1] = -22.2;
		deltaValues.put("AA", tmp);
		tmp = new double[2];
		tmp[0] = -7.2; tmp[1] = -20.4;
		deltaValues.put("AT", tmp);
		tmp = new double[2];
		tmp[0] = -8.2; tmp[1] = -22.2;
		deltaValues.put("AG", tmp);
		tmp = new double[2];
		tmp[0] = -8.5; tmp[1] = -22.7;
		deltaValues.put("AC", tmp);
		tmp = new double[2];
		tmp[0] = -7.9; tmp[1] = -22.2;
		deltaValues.put("TT", tmp);
		tmp = new double[2];
		tmp[0] = -7.2; tmp[1] = -21.3;
		deltaValues.put("TA", tmp);
		tmp = new double[2];
		tmp[0] = -8.4; tmp[1] = -22.4;
		deltaValues.put("TG", tmp);
		tmp = new double[2];
		tmp[0] = -7.8; tmp[1] = -21.0;
		deltaValues.put("TC", tmp);
		tmp = new double[2];
		tmp[0] = -8.0; tmp[1] = -19.9;
		deltaValues.put("GG", tmp);
		tmp = new double[2];
		tmp[0] = -8.2; tmp[1] = -22.2;
		deltaValues.put("GA", tmp);
		tmp = new double[2];
		tmp[0] = -8.4; tmp[1] = -22.4;
		deltaValues.put("GT", tmp);
		tmp = new double[2];
		tmp[0] = -9.8; tmp[1] = -24.4;
		deltaValues.put("GC", tmp);
		tmp = new double[2];
		tmp[0] = -8.0; tmp[1] = -19.9;
		deltaValues.put("CC", tmp);
		tmp = new double[2];
		tmp[0] = -8.5; tmp[1] = -22.7;
		deltaValues.put("CA", tmp);
		tmp = new double[2];
		tmp[0] = -7.8; tmp[1] = -21.0;
		deltaValues.put("CT", tmp);
		tmp = new double[2];
		tmp[0] = -10.6; tmp[1] = -27.2;
		deltaValues.put("CG", tmp);
		tmp = new double[2];
		tmp[0] = 0.1; tmp[1] = -2.8;
		deltaValues.put("initGC", tmp);		
		tmp = new double[2];
		tmp[0] = 2.3; tmp[1] = 4.1;
		deltaValues.put("initAT", tmp);
	}
	
	// Calculates the sum of deltaH/deltaS for a given primer
	// from an index to the end of the sequence
	public double[] deltaSum (String primer) {
		double[] delta = new double[2];
		double deltaH = 0.0;
		double deltaS = 0.0;
		for (int i = 0; i < primer.length()-1; i++) {
			if (i == 0) {
				if ((primer.charAt(i) == 'G')|(primer.charAt(i) == 'C')) {
					deltaH += deltaValues.get("initGC")[0];
					deltaS += deltaValues.get("initGC")[1];
				} else if ((primer.charAt(i) == 'A')|(primer.charAt(i) == 'T')) {
					deltaH += deltaValues.get("initAT")[0];
					deltaS += deltaValues.get("initAT")[1];
				}
			}
			try{
			deltaH += deltaValues.get(primer.substring(i, i+2))[0];
			deltaS += deltaValues.get(primer.substring(i, i+2))[1];
			} catch (Exception e) {}
		}
		delta[0] = deltaH; delta[1] = deltaS;
		return delta;
	}
	
	// Calculates the Gibbs free energy for a specific primer
	// computes at 37deg (very little difference for primers)
	public double deltaG (String primer) {
		double dG = -1000.0;
		double[] delta = deltaSum(primer);
		dG = (delta[0]*1000.0) - ((273.15+37)*delta[1]);
		return (dG/1000);
	}
	
	// Calculates the Tm without salt correction for a given primer
	public double getInitialTm (String primer) {
		double tm = 0.0;
		double[] deltaVals = deltaSum(primer);
		//System.out.println(deltaVals[0]+"\t"+deltaVals[1]);
		tm = ((deltaVals[0]*1000.0) / (deltaVals[1]+(1.987*Math.log(primerConcentration/1000000000)))) - 273.15;
		return tm;
	}
	
	// Calculates the fractional GC content for a primer
	public double getFractionalGC(String primer) {
		double gc = 0.5;
		double total = primer.length();
		double numberG = total - primer.replace("G", "").length();
		numberG += total - primer.replace("g", "").length();
		double numberC = total - primer.replace("C", "").length();
		numberC += total - primer.replace("c", "").length();
		gc = (numberG + numberC) / total;
		return gc;
	}
	
	// Calculates the Tm with salt correction for a given primer
	public double getTm (String primer) {
			String primerUp = primer.toUpperCase();
			double tm = getInitialTm(primerUp);
			double fgc = getFractionalGC(primerUp);
			double tmcorr = (1 / ((1/tm) + ((((4.29*fgc)-3.95) * Math.log(saltConcentration/1000) + (0.94 * (Math.log(saltConcentration/1000) * Math.log(saltConcentration/1000)))) * 0.00001)));
			return tmcorr;
	}
	
	/**
	 * For testing only
	 * @param args
	 */
	public static void main(String[] args) {
		TmCalculator tmc = new TmCalculator(50.0, 1000.0);
		/*String primerF1 = "CACGGCAGGGCTCTAAATC";
		String primerF2 = "tcaaatctgtcactattgcgTGGACATACATAGTATACTCTGGTG";
		String primerF3 = "ctacaGGATTCATTAAAGAGGAGAAAcctaggatg";
		System.out.println(primerF1+"\tdG="+tmc.deltaG(primerF1));
		System.out.println(primerF2+"\tdG="+tmc.deltaG(primerF2));
		System.out.println(primerF3+"\tdG="+tmc.deltaG(primerF3));
		**/
	}

}
