/**
 * 
 */
package core;

import java.util.HashMap;

/**
 * @author Charles Winterhalter
 * Class generating mutagenic forward/reverse primers
 * for a given gene coding sequence
 */
public class PrimerPair {

	// Attributes
	private String fPrimer = "";
	private String rPrimer = "";
	private String prefixF;
	private String prefixR;
	private String suffixF;
	private String suffixR;
	private String template;
	private String targetCodon;
	private int targetPosition;
	private String overlap;
	private int minSuffix;
	
	
	public PrimerPair(String templateSequence, String codingRegion, String targetAA, int aaToChange, int min1, int max1, int min2, int max2) {
		this.template = templateSequence;
		this.targetCodon = targetAA;
		this.targetPosition = aaToChange;
		this.minSuffix = min2+11;
		makePrimers();	// make default size primers
		// adjust length for suffix and overlap regions
		checkSuffixes(min2+11, max2+11);
		checkOverlap(min1+11, max1+11);
		setForward(this.prefixF+targetCodon+this.suffixF);
		setReverse(reverseComplement(new StringBuffer(targetCodon+this.prefixR).reverse().toString())+this.suffixR); // suffixR already reverse complemented for Tm calculations
		correctTmDiff();	// greedy primer pair adjustment	
		if (!isStable()) System.out.println("Primer pair at "+targetPosition+" unstable"); // verify primers energy profile
	}
	
	// 
	
	// Check for suitable deltaG
	public boolean isStable () {
		if ((MutateProtein.computeTm.deltaG(getReverse()) < -9) | (MutateProtein.computeTm.deltaG(getForward()) < -9)) {
			return false;
		}
		return true;
	}
	
	// Check for satisfying prefix lengths
	// should be between 3 and 6bp for prefixes (overlap~9-15bp)
	// should be between 15 and 25bp for suffixes
	public boolean isNormalSize(String s, int min, int max) {
		if (s.length() > min && s.length() < max) return true;
		return false;
	}
	
	// Check the algorithm isn't stuck in optimisation
	public boolean isAlgoBlocked (String[] t) {
		HashMap<String, Integer> traceRepo = new HashMap<String, Integer>();
		for (String s : t) {
			if (!traceRepo.containsKey(s)) traceRepo.put(s, 0); // initiate a new solution
			traceRepo.put(s, traceRepo.get(s)+1); // increase count for solution
		}
		for (int count : traceRepo.values()) if (count >= 5) return true;
		return false;
	}
	
	// Adjust a primair pair overlap to meet temperature requirements
	public void checkOverlap (int min, int max) {
		String[] trace = new String[15];
		int counter = 0;
		String ovlp = this.overlap;
		String preF = this.prefixF;
		String preR = this.prefixR;
		trace[0] = ovlp;
		int affixIndex = isCompatible(MutateProtein.computeTm.getTm(ovlp), min, max);
		EXIT: while (affixIndex != 0) {
			int initF = preF.length();
			int initR = preR.length();
			if (initF > initR) { //preF is longer
				if (affixIndex == 1) { //too short
					if (isNormalSize(preR, 3, 6))	preR = this.template.substring(targetPosition+3, targetPosition+3+initR+1);
				} else if (affixIndex == -1) { //too long
					if (isNormalSize(preF, 3, 6))	preF = this.template.substring(targetPosition-initF+1, targetPosition);
				}
			} else if (initF < initR) { //preR is longer
				if (affixIndex == 1) { //too short
					if (isNormalSize(preF, 3, 6))	preF = this.template.substring(targetPosition-initF-1, targetPosition);
				} else if (affixIndex == -1) { //too long
					if (isNormalSize(preR, 3, 6))	preR = this.template.substring(targetPosition+3, targetPosition+3+initR-1);
				}
			} else { // preF/preR same size, so look at suffixes
				if (affixIndex == 1) { // overlap too short
					if (this.suffixF.length() >= this.suffixR.length()) { // extend towards 5'end
						if (isNormalSize(preF, 3, 6))	preF = this.template.substring(targetPosition-initF-1, targetPosition);
					} else { // extend towards 3' end
						if (isNormalSize(preR, 3, 6))	preR = this.template.substring(targetPosition+3, targetPosition+3+initR+1);
					}
				} else if (affixIndex == -1) { // overlap too long
					if (this.suffixF.length() >= this.suffixR.length()) { // shrink towards 5'end
						if (isNormalSize(preF, 3, 6))	preF = this.template.substring(targetPosition-initF+1, targetPosition);
					} else { // shrink towards 3' end
						if (isNormalSize(preR, 3, 6))	preR = this.template.substring(targetPosition+3, targetPosition+3+initR-1);
					}
				} else {
					System.out.println("error in primer overlap compatibility");
					System.exit(0);
				}
			}
			ovlp = preF + targetCodon + preR;
			affixIndex = isCompatible(MutateProtein.computeTm.getTm(ovlp), min, max);
			if (!isNormalSize(preF, 3, 6) & !isNormalSize(preR, 3, 6)) { // exit loop based on size
				break;
			} else if (MutateProtein.computeTm.getFractionalGC(ovlp) <= 0.4) break;
			counter++;
			try {
				trace[counter] = ovlp;
			} catch (Exception e) {
				if (isAlgoBlocked(trace)) break EXIT;
			}
		}
		setOverlap(preF, preR);
	}
	
	// Adjust a primer length to meet temperature requirements
	public void checkSuffixes (int min, int max) {
		String[] pair = new String[2];
		pair[0] = this.suffixF; pair[1] = this.suffixR;
		// Adjust based on Tm
		for (int i = 0; i < 2; i++) {
			int affixIndex = isCompatible(MutateProtein.computeTm.getTm(pair[i]), min, max);
			String newAffix = pair[i];
			INNER: while (affixIndex != 0) {
				int init = newAffix.length();
				if (affixIndex == 1) {
					if (i == 0) { // suffixF too short
						if (isNormalSize(newAffix, 12, 25))	newAffix = this.template.substring(targetPosition+3, targetPosition+3+init+1);
					} else { // suffixR too short
						if (isNormalSize(newAffix, 12, 25))	newAffix = this.template.substring(targetPosition-init-1, targetPosition);
					}
				} else if (affixIndex == -1) {
					if (i == 0) { // suffixF too long
						if (isNormalSize(newAffix, 12, 25))	newAffix = this.template.substring(targetPosition+3, targetPosition+3+init-1);
					} else { // suffixR too long
						if (isNormalSize(newAffix, 12, 25))	newAffix = this.template.substring(targetPosition-init+1, targetPosition);
					}
				} else {
					System.out.println("error in primer compatibility");
					System.exit(0);
				}
				if (i == 1) newAffix = reverseComplement(new StringBuffer(newAffix).reverse().toString()); // reverse complement if dealing with reverse primer
				affixIndex = isCompatible(MutateProtein.computeTm.getTm(newAffix), min, max);
				if (!isNormalSize(newAffix, 12, 25)) break INNER;	
			}
			pair[i] = newAffix;
		}
		// Adjust based on GC-clamp
		// this code corrects suffixes length based on their last 3'end nucleotide
		// Generate a warning if this check produces any suffixes out of bounds (15<=length<=25)
		for (int i = 0; i < 2; i++) {
			String newAffix = pair[i];
			double tmInit = MutateProtein.computeTm.getTm(newAffix);
			int neighbour = 1;
			while (newAffix.endsWith("A") | newAffix.endsWith("T") | newAffix.endsWith("a") | newAffix.endsWith("t")) {
				int init = newAffix.length();
				double tmMinus = 0.0;
				double tmPlus = 0.0;
				if (i == 0) { // suffixF
					if (this.template.substring(targetPosition+3, targetPosition+3+init-neighbour).endsWith("G") | this.template.substring(targetPosition+3, targetPosition+3+init-neighbour).endsWith("C")
							| this.template.substring(targetPosition+3, targetPosition+3+init-neighbour).endsWith("g") | this.template.substring(targetPosition+3, targetPosition+3+init-neighbour).endsWith("c")) {
						tmMinus = MutateProtein.computeTm.getTm(this.template.substring(targetPosition+3, targetPosition+3+init-neighbour));						
					}
					if (this.template.substring(targetPosition+3, targetPosition+3+init+neighbour).endsWith("G") | this.template.substring(targetPosition+3, targetPosition+3+init+neighbour).endsWith("C")
							| this.template.substring(targetPosition+3, targetPosition+3+init+neighbour).endsWith("g") | this.template.substring(targetPosition+3, targetPosition+3+init+neighbour).endsWith("c")) {
						tmPlus = MutateProtein.computeTm.getTm(this.template.substring(targetPosition+3, targetPosition+3+init+neighbour));						
					}
				} else { // suffixR
					if (this.template.substring(targetPosition-init+neighbour, targetPosition).startsWith("G") | this.template.substring(targetPosition-init+neighbour, targetPosition).startsWith("C")
							| this.template.substring(targetPosition-init+neighbour, targetPosition).startsWith("g") | this.template.substring(targetPosition-init+neighbour, targetPosition).startsWith("c")) {
						tmMinus = MutateProtein.computeTm.getTm(reverseComplement(new StringBuffer(this.template.substring(targetPosition-init+neighbour, targetPosition)).reverse().toString()));						
					}
					if (this.template.substring(targetPosition-init-neighbour, targetPosition).startsWith("G") | this.template.substring(targetPosition-init-neighbour, targetPosition).startsWith("C")
							| this.template.substring(targetPosition-init-neighbour, targetPosition).startsWith("g") | this.template.substring(targetPosition-init-neighbour, targetPosition).startsWith("c")) {
						tmPlus = MutateProtein.computeTm.getTm(reverseComplement(new StringBuffer(this.template.substring(targetPosition-init-neighbour, targetPosition)).reverse().toString()));						
					}
				}
				if ((tmMinus != 0.0) | (tmPlus != 0.0)) {
					double dif1 = Math.abs(tmInit - tmMinus);
					double dif2 = Math.abs(tmInit - tmPlus);
					if (dif1 <= dif2) { // tmMinus closer to initial primer
						if (i == 0) { // suffixF
							newAffix = this.template.substring(targetPosition+3, targetPosition+3+init-neighbour);				
						} else { // suffixR
							newAffix = reverseComplement(new StringBuffer(this.template.substring(targetPosition-init+neighbour, targetPosition)).reverse().toString());
						}
					} else { // tmPlus closer to initial primer
						if (i == 0) { // suffixF
							newAffix = this.template.substring(targetPosition+3, targetPosition+3+init+neighbour);				
						} else { // suffixR
							newAffix = reverseComplement(new StringBuffer(this.template.substring(targetPosition-init-neighbour, targetPosition)).reverse().toString());
						}
					}
				}
				neighbour += 1;
			}
			pair[i] = newAffix;
		}
		setSuffix(pair[0], pair[1]);
	}
	
	// Make a pair of forward/reverse primers
	// to mutate a specific aminoacid in a target sequence
	public void makePrimers () {
		String prefixF = this.template.substring(targetPosition-5, targetPosition);
		String prefixR = this.template.substring(targetPosition+3, targetPosition+8);
		setOverlap(prefixF, prefixR);
		String suffixF = this.template.substring(targetPosition+3, targetPosition+23);
		String suffixR = this.template.substring(targetPosition-20, targetPosition);
		setSuffix(suffixF, suffixR);
		String primerF = prefixF+targetCodon+suffixF;
		String primerReverse = suffixR+targetCodon+prefixR;
		String buffer = new StringBuffer(primerReverse).reverse().toString();
		String primerR = reverseComplement(buffer);
		setForward(primerF);
		setReverse(primerR);
	}
	
	// Greedy adjustment lower Tm primer
	public void correctTmDiff () {
		String newSuffixF = getSuffixForward();
		String newSuffixR = getSuffixReverse();
		double tmF = MutateProtein.computeTm.getTm(newSuffixF);
		double tmR = MutateProtein.computeTm.getTm(newSuffixR);
		while (Math.abs(tmF-tmR) > 3.0 | (tmF < minSuffix & tmR < minSuffix)) { // adjust for a Tm difference greater that 5deg
			if (tmF-tmR < 0.0) { // extend forward suffix
				int currentSize = newSuffixF.length();
				newSuffixF = findNextGCClamp(targetPosition+3, targetPosition+3+currentSize);
				tmF = MutateProtein.computeTm.getTm(newSuffixF);
			} else { // extend reverse suffix
				int currentSize = newSuffixR.length();
				newSuffixR = findNextGCClamp(targetPosition, targetPosition-currentSize);
				tmR = MutateProtein.computeTm.getTm(newSuffixR);
			}
		}
		setSuffix(newSuffixF, newSuffixR);
	}
	
	// Find next GCclamp for primer extension
	public String findNextGCClamp (int start, int stop) {
		String initSeq = "";
		if (start < stop) {
			initSeq = this.template.substring(start, stop);
		} else {
			initSeq = reverseComplement(new StringBuffer(this.template.substring(stop, start)).reverse().toString());
		}
		String newSeq = initSeq;
		int extension = 1;
		while (newSeq.equals(initSeq) | newSeq.endsWith("A") | newSeq.endsWith("T") | newSeq.endsWith("a") | newSeq.endsWith("t")) {
			if (start < stop) { // forward strand
				newSeq = this.template.substring(start, stop+extension);				
			} else { // reverse strand
				newSeq = reverseComplement(new StringBuffer(this.template.substring(stop-extension, start)).reverse().toString());
			}
			extension++;
		}
		return newSeq;
	}
	
	public String getOverlap () {
		return this.overlap;
	}
	
	public void setOverlap (String p1, String p2) {
		this.prefixF = p1;
		this.prefixR = p2;
		this.overlap = p1+targetCodon+p2;
	}
	
	public void setSuffix (String s1, String s2) {
		this.suffixF = s1;
		this.suffixR = s2;
	}
	
	public String getSuffixForward () {
		return this.suffixF;
	}
	
	public String getSuffixReverse () {
		return this.suffixR;
	}
	
	private String reverseComplement(String input){
		String output = "";
	    for (int i = 0; i < input.length(); i++) {
	      if (input.charAt(i) == 'A') {
	        output = output.concat("T");
	      } else if (input.charAt(i) == 'C') {
	        output = output.concat("G");
	      } else if (input.charAt(i) == 'G') {
	        output = output.concat("C");
	      } else if (input.charAt(i) == 'T') {
	        output = output.concat("A");
	      } else if (input.charAt(i) == 'a') {
		    output = output.concat("t");
	      } else if (input.charAt(i) == 'c') {
		    output = output.concat("g");
	      } else if (input.charAt(i) == 'g') {
		    output = output.concat("c");
	      } else if (input.charAt(i) == 't') {
		    output = output.concat("a");
		  }
	    }
	    return output;
	}
	
	// Get/Set forward primer
	public String getForward() {
		return fPrimer;
	}
	public void setForward (String s) {
		this.fPrimer = s;
	}
	
	// Get/Set reverse primer
	public String getReverse() {
		return rPrimer;
	}
	public void setReverse (String s) {
		this.rPrimer = s;
	}
	
	// Check for a primer compatibility
	// to a specific temprature range
	public int isCompatible(double t, int min, int max) {
		if (t <= min) {
			return 1; // extend primer
		} else if (t >= max){
			return -1; // shrink primer
		} else {
			return 0; // satisfactory primer
		}
	}
}
