/**
 * 
 */
package core;

import java.io.FileWriter;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.Map;

/**
 * @author Charles Winterhalter
 * Class aimed at mutating AA in a target protein
 */
public class MutateProtein extends DnaProteinOperator {

	// Attributes
	public String template;
	public String startSeq;
	public String targetCodon;
	public Map<String, String> codonMap;
	public String targetCDS;
	public int startPosition;
	public String[][] outputPrimers;
	public static TmCalculator computeTm;
	
	/**
	 * Basic constructor instantiating main information to carry out full protein mutagenesis
	 */
	public MutateProtein (String templateSequence, String start, String targetAA, boolean isHighExpression, double saltConcentration, double primersConcentration) {
		this.template = templateSequence.toLowerCase();
		this.startSeq = start.toLowerCase();
		this.targetCodon = DnaProteinOperator.CodonOptimise(isHighExpression, targetAA);
		this.codonMap = DnaProteinOperator.getCodonTable();
		String params[] = DnaProteinOperator.findCodingSequence(template, startSeq);
		this.targetCDS = params[0];
		this.startPosition = Integer.parseInt(params[1]);
		this.outputPrimers = new String[getNumberCodons()][11];
		computeTm = new TmCalculator(saltConcentration, primersConcentration);
	}
	
	/**
	 * Loop over the coding sequence and generate pairs of mutagenic primers
	 * for each codon
	 */
	public void getMutagenicPrimers (int min1, int max1, int min2, int max2) {
		int j = 0;
		for (int i = 3; i < targetCDS.length()-3; i+=3) {
			PrimerPair newPrimerPair = new PrimerPair(template, targetCDS, targetCodon, startPosition+i, min1, max1, min2, max2);
			outputPrimers[j][0] = codonMap.get(template.substring(startPosition+i, startPosition+i+3).toUpperCase())+(j+2)+codonMap.get(targetCodon);	// mutation name
			outputPrimers[j][1] = newPrimerPair.getForward();	// forward primer
			outputPrimers[j][2] = String.valueOf(newPrimerPair.getForward().length()); // F primer size
			outputPrimers[j][3] = String.valueOf(computeTm.getTm(newPrimerPair.getSuffixForward())-11); // NEB Tm correction
			outputPrimers[j][4] = newPrimerPair.getReverse();	// reverse primer
			outputPrimers[j][5] = String.valueOf(newPrimerPair.getReverse().length()); // R primer size
			outputPrimers[j][6] = String.valueOf(computeTm.getTm(newPrimerPair.getSuffixReverse())-11); // NEB Tm correction
			outputPrimers[j][7] = String.valueOf(Math.abs(Double.parseDouble(outputPrimers[j][3])-Double.parseDouble(outputPrimers[j][6])));	// TmDiff
			outputPrimers[j][8] = newPrimerPair.getOverlap();	// overlap forward strand
			outputPrimers[j][9] = String.valueOf(computeTm.getTm(newPrimerPair.getOverlap())-11);	// NEB Tm correction
			outputPrimers[j][10] = String.valueOf(getTa(Double.parseDouble(outputPrimers[j][3]), Double.parseDouble(outputPrimers[j][6])));	// NEB recommended Ta
			j++;
		}
	}
	
	// Find annealing temperature
	public double getTa (double tm1, double tm2) {
		if (tm1 <= tm2) return (tm1+1.0);
		return (tm2+1.0);
	}
	
	/**
	 * Finds the overall number of codons to be mutated in the target sequence
	 */
	public int getNumberCodons () {
		int fullLength = targetCDS.length() / 3;
		return (fullLength-2);
	}
	
	// Writes output data
	public void writeData (String fileName) {
		try {
			FileWriter fw = new FileWriter(fileName);
			PrintWriter pw = new PrintWriter(fw);
			for (int i=0; i<outputPrimers.length; i++) {
				if (i==0) pw.print("Mutation\tF-primer\tF-size\tTm(F)\tR-primer\tR-size\tTm(R)\tTm(diff)\tOverlap\tTm overlap\tTa\n");
				for (int j = 0; j<outputPrimers[0].length; j++) {
					pw.print(outputPrimers[i][j]);
					if (j==outputPrimers[0].length-1) {
						pw.print("\n");
					} else {
						pw.print("\t");
					}
				}
			}
			pw.close();
			fw.close();
		} catch (IOException e) {
			System.out.println("Error Printing Tab Delimited File");
		}
		
	}
	
	/**
	 * @param args
	 */
	public static void main(String[] args) {
		
		// DnaD screen
		// Provide about 40 bases flanking the open reading frame on each side, the start sequence, amino-acid to substitute to, boolean value for high expression, final salt concentration (mM) in PCR buffer, final primer concentration (nM) in PCR reaction
		MutateProtein dnaD = new MutateProtein("gcgcctcacgttcgtgaaacgattccgttcccaagactgttaaaccgtctgtatccgtaatacattcaaatgaaacaaaaaagagtctcctgcaagggagactttttacagtaaaggggcaatcacaatcctttacagtaagaggtgtgacgatgaaaaaacagcaatttattgatatgcaggagcagggaacatcaaccatccccaatcttctgctcacgcattataaacagcttgggcttaatgaaacagaacttattctgctgttaaaaattaaaatgcatttagaaaaaggatcatattttcctacaccgaatcagctgcaggaaggtatgtcaatttctgttgaagaatgtacaaacagattgcggatgtttattcaaaaaggctttctgtttattgaagaatgcgaggatcaaaacggcatcaaatttgagaaatattctcttcagcctttatggggcaagctgtacgagtatattcagcttgcacagaatcaaacacaggaaagaaaagcagaaggggaacaaaaaagcctttataccatttttgaggaagagttcgcaagaccgttatcgcctttggagtgtgaaacgctggcgatctggcaggaccaggatcagcatgacgcacaactgatcaaacacgcgttaaaagaggctgtactatcaggaaaactcagtttccgctacattgaccggattttgtttgaatggaagaaaaatgggcttaaaactgtggagcaggcaaaaatacacagccaaaaattccggcgtgtacaagcaaagcagaatgaaccgcaaaaagagtataaaaggcaggttcctttttacaattggcttgaacaataaagtgaaaaggtgacaatcgtgttaaatctaaaacaaattgaattctgtttagacaagataggtgacatgtttcctcatgcggagtgtgaactggttcattccaatccttttgaattagtggtggctgttgctttatctgcgcaatgtacagatgcacttgtaaacag",
				"ATGAAAAAACAGC", "A", false, 50, 1000);
		
		// Provide min and max for overlap temperature, min and max for annealing temperature
		// Does not need changing
		dnaD.getMutagenicPrimers(40, 45, 60, 65);
		
		// Name of the output file
		dnaD.writeData("./dnaD_primers.tab");
	}

}
