#### INFO ####

# This file is a supplement to:
#
# Bajnai, D., & Herwartz, D.
# Kinetic oxygen isotope fractionation between water
# and aqueous OH- during hydroxylation of CO2.
# ACS Earth and Space Chemistry.
# https://doi.org/10.1021/acsearthspacechem.1c00194

# Use this script to:
#  1) create the plots for Figures 1, 3, 4, S1, and S2
#  2) calculate Equation 9
#  3) compile Table 2

# Data is imported from from an Excel file (OH_data.xlsx) that is
# deposited along with this code.


#### Housekeeping ####

# Clean up
# rm(list = ls())
# cat("\014")

# Set working directory 
setwd ("~/Documents/R/OH R codes")

# Used for importing data from an excel file
if (!require("openxlsx"))
  install.packages("openxlsx")

# Used for data manipulation
if (!require("dplyr"))
  install.packages("dplyr")

# Used for plotting
if (!require("viridisLite"))
  install.packages("viridisLite")
library(viridisLite)

# Used for modeling DIC speciation and calculating the regression.
# For the manuscript v1.0.9 was used.
# The source code of 'isogeochem' is available at:
# https://doi.org/10.5281/zenodo.5665966
install.packages("isogeochem")
library(isogeochem)

#### Data import ####

# data from this study
dat = openxlsx::read.xlsx("OH_data.xlsx", sheet = "samples")

# Data from Green and Taube (1964)
GT_temp = c(14.8, 15.2)
GT_epsilon = c(epsilon(1.044), epsilon(1.046))
GT_epsilon_err = c((epsilon(1.044 + 0.004) - epsilon(1.044 - 0.004)) / 2,
                   (epsilon(1.046 + 0.004) - epsilon(1.046 - 0.004)) / 2)
GT_reg = coefficients(lm(c(epsilon(1.045), epsilon(1.040)) ~ c(15, 25)))

# Data from Beck et al. (2005)
BCK_temp = c(15, 25, 40)
BCK_epsilon = c(42.8, 44.8, 43.3)
BCK_epsilon_err = c(0.6, 0.6, 0.3)

# Data from Clark et al. (1992)
CLK_temp = 22
CLK_temp_err = 1.5
CLK_epsilon = 43.5

#### Calculations ####

# Differences in d13C between the carbonate and the CO2
Dd13C = dat$d13C_VPDB - dat$d13C_CO2_VPDB

# Differences in d18O between the carbonate and the CO2
Dd18O = dat$d18O_VSMOW - dat$d18O_CO2_VSMOW

# Oxygen isotope fractionation H2O/OH-
d18O_OH  = (dat$d18O_VSMOW - (2/3) * dat$d18O_CO2_VSMOW) * 3
e_H2O_OH = epsilon((dat$d18O_sol_VSMOW + 1000) / (d18O_OH + 1000))

# Error propagation for e_H2O_OH
## Error from d18O_CO2 and d18O_BaCO3
d18O_OH_E1 = ((dat$d18O_VSMOW - dat$d18O_SD) - (2/3) * (dat$d18O_CO2_VSMOW + 0.03))*3
d18O_OH_E2 = ((dat$d18O_VSMOW + dat$d18O_SD) - (2/3) * (dat$d18O_CO2_VSMOW - 0.03))*3
## Error from d18O_H2O
e_H2O_OH_E1 = epsilon((dat$d18O_sol_VSMOW + 0.1 + 1000) / (d18O_OH_E1 + 1000))
e_H2O_OH_E2 = epsilon((dat$d18O_sol_VSMOW - 0.1 + 1000) / (d18O_OH_E2 + 1000))
## Final error
e_H2O_OH_err = ((e_H2O_OH-e_H2O_OH_E2)+(e_H2O_OH_E1-e_H2O_OH))/2

# Add calculated values to data frame
dat = cbind(dat, d18O_OH)
dat = cbind(dat, Dd13C)
dat = cbind(dat, Dd18O)
dat = cbind(dat, e_H2O_OH)
dat = cbind(dat, e_H2O_OH_err)

##### Exclude data #####

# Select data points based on Dd13C
dat$Comment[dat$Dd13C< -0.7] = "exclude"

##### Equation 9 #####

# Error considering linear regression on selected data points
ECLR = york_fit(x = dat$Temp[dat$Comment == 0],
                x_err = rep(2, length(dat$Temp[dat$Comment == 0])), 
                y = dat$e_H2O_OH[dat$Comment==0],
                y_err = dat$e_H2O_OH_err[dat$Comment==0])


#### Figure 1A ####

pdf("OH Figure 1A.pdf", 5, 5, encoding = "MacRoman")
par(mar = c(4.5, 4.5, 0.3, 0.3))
plot(0, type = "l", las = 1,
     xlim = c(0, 14),
     ylim = c(0, 100),
     xlab = "pH",
     ylab = "Relative rates of CO2 absoprtion reactions (%)")

pH = seq(0, 14, 0.1)
S = 0

temp = 80
lines(pH,X_absorption(temp, pH, S)$X_hydration, col = "#31688E", lty = 2, lwd = 3)
lines(pH,X_absorption(temp, pH, S)$X_hydroxylation, col = "#35B779", lty = 2, lwd = 3)

temp = 1
lines(pH, X_absorption(temp, pH, S)$X_hydration, col = "#31688E", lty = 1, lwd = 3)
lines(pH, X_absorption(temp, pH, S)$X_hydroxylation, col = "#35B779", lty = 1, lwd = 3)

# mark the pH of the precipitation experiments
abline(v = c(13), col = "gray50")

legend(0, 70, bty="n", adj = c(0, NA),
       lty = c(1, 2, 1, 1),
       lwd = c(3, 3, 3, 3),
       col = c("gray40", "gray40", "#31688E", "#35B779"),
       legend = c("1°C", "80°C", "hydration", "hydroxylation"))

dev.off()


#### Figure 1B ####

pdf("OH Figure 1B.pdf", 5, 5, encoding = "MacRoman")
par(mar = c(4.5, 4.5, 0.3, 0.3))
plot(0, type = "l", las = 1,
          xlim = c(0, 14),
          ylim = c(0, 100),
          xlab = "pH",
          ylab = "Relative abundance of DIC species (%)")

pH = seq(0, 14, 0.1)
S = 0

temp = 1
lines(pH, X_DIC(temp, pH, S)$X_CO2, col = "#FDE725", lty = 1, lwd = 3)
lines(pH, X_DIC(temp, pH, S)$X_HCO3, col = "#31688E", lty = 1, lwd = 3)
lines(pH, X_DIC(temp, pH, S)$X_CO3, col = "#35B779", lty = 1, lwd = 3)

temp = 80
lines(pH, X_DIC(temp, pH, S)$X_CO2, col = "#FDE725", lty = 3, lwd = 3)
lines(pH, X_DIC(temp, pH, S)$X_HCO3, col = "#31688E", lty = 3, lwd = 3)
lines(pH, X_DIC(temp, pH, S)$X_CO3, col = "#35B779", lty = 3, lwd = 3)

# mark the pH of the precipitation experiments
abline(v = c(13), col = "gray50")

legend(0, 70, bty = "n", adj = c(0, NA),
       lty = c(1, 1, 1, 1, 3),
       lwd = c(3, 3, 3, 3, 3),
       col = c("#FDE725", "#31688E", "#35B779", "gray40", "gray40"),
       legend = c("[CO2]", "[HCO3-]", "[CO32-]", "1°C", "80°C"))

dev.off()


#### Figure 3A ####

# Specify the colors of the data points
color = data.frame(unique(dat$Temp),
                   viridisLite::viridis(
                     n = length(unique(dat$Temp)),
                     option = "D"
                   ))
names(color)[1] = c("Temp")
names(color)[2] = c("color")

pdf("OH Figure 3A.pdf", 5, 5, encoding = "MacRoman")
par(mar = c(4.5, 4.5, 0.3, 0.3))
plot(0, type = "l", las = 1, xaxt = "n",
     ylim = c(0, 10),
     xlim = c(-6.0, -4.5),
     ylab = expression(delta ^ 18 * "O"[BaCO3] * " (‰ VSMOW)"),
     xlab = expression(delta ^ 13 * "C"[BaCO3] * " (‰ VPDB)"))
axis(1, at = seq(-6.0, -4.5, by = 0.25))
axis(2, at = seq(0, 10, by = 0.25), labels = NA)

# CO2 gas composition
rect(xleft = -4.55, xright = -4.57, ybottom = -1, ytop = 11, col = "gray79", border = NA)
abline(v = -4.56, lty = 2, lwd = 1.5, col = "gray40")

# CO2 absorption slope
abline(b = 0.5, a = 9.5, col = "magenta")

# Add the points with the error bars to the plot
segments(dat$d13C_VPDB - dat$d13C_SD, dat$d18O_VSMOW,
         dat$d13C_VPDB + dat$d13C_SD, dat$d18O_VSMOW,
         col = "gray1")
segments(dat$d13C_VPDB, dat$d18O_VSMOW - dat$d18O_SD,
         dat$d13C_VPDB, dat$d18O_VSMOW + dat$d18O_SD,
         col = "gray1")

for (i in unique(dat$Temp)) {
  points(
    dat$d13C_VPDB[dat$Temp == i],
    dat$d18O_VSMOW[dat$Temp == i],
    pch = ifelse(dat$d18O_sol_VSMOW[dat$Temp == i] < 0, 15, 16),
    col = color$color[color$Temp == i]
  )
}

# mark the excluded replicates
points(dat$d13C_VPDB[dat$Comment != 0],
       dat$d18O_VSMOW[dat$Comment != 0],
       pch = 4,
       col = "red")

dev.off()


#### Figure 3B ####

pdf("OH Figure 3B.pdf", 5, 5, encoding = "MacRoman")
par(mar = c(4.5, 4.5, 0.3, 0.3))
plot(0, type = "l", xaxt = "n", las = 1,
     xlim = c(-1.4, 0.2),
     ylim = c(38, 46),
     xlab = expression(delta ^ 13 * "C"[carbonate] * -delta ^ 13 * "C"[CO2] * " (‰ VPDB)"),
     ylab = expression(epsilon ^ 18 * ""[H2O / OH] * " (‰)"))
axis(1, at = seq(-1.4, 0.2, by = 0.1), labels = seq(-1.4, 0.2, by = 0.1))

# CO2 gas composition
rect(xleft = -0.01, xright = 0.01, ybottom = 37, ytop = 47, col = "gray79", border = NA)
abline(v = 0, lty = 2, lwd = 1.5, col = "gray40")

# CO2 absorption slope
abline(b = -1.6, a = 41, col = "magenta")

# Add the points with the error bars to the plot.
segments(dat$Dd13C - dat$d13C_SD, dat$e_H2O_OH,
         dat$Dd13C + dat$d13C_SD, dat$e_H2O_OH,
         col = "gray1")

segments(dat$Dd13C, dat$e_H2O_OH - dat$e_H2O_OH_err,
         dat$Dd13C, dat$e_H2O_OH + dat$e_H2O_OH_err,
         col = "gray1")

for (i in unique(dat$Temp)) {
  points(
    dat$Dd13C[dat$Temp == i],
    dat$e_H2O_OH[dat$Temp == i],
    pch = ifelse(dat$d18O_sol_VSMOW[dat$Temp == i] < 0, 15, 16),
    col = color$color[color$Temp == i]
  )
}

# mark the excluded replicates
points(dat$Dd13C[dat$Comment != 0],
       dat$e_H2O_OH[dat$Comment != 0],
       pch = 4,
       col = "red")

dev.off()


#### Figure 4 ####

pdf("OH Figure 4.pdf", 5, 5, encoding = "MacRoman")
par(mar = c(4.5, 4.5, 0.3, 0.3))
plot(0, type = "l", las = 1,
     xlim = c(0, 80),
     ylim = c(10, 50),
     xlab = "Temperature (°C)",
     ylab = expression(epsilon ^ 18 * ""[H2O / OH] * " (‰)"))

# Add the error considering linear regression to the plot.
york_plot(
  x = dat$Temp[dat$Comment == 0],
  intercept = ECLR$intercept,
  intercept_se = ECLR$intercept_se,
  slope = ECLR$slope,
  slope_se = ECLR$slope_se,
  add = TRUE,
  col = "#31688E"
)

# Add Green and Taube (1963)
segments(GT_temp, GT_epsilon-GT_epsilon_err,
         GT_temp, GT_epsilon + GT_epsilon_err,
         col = "#35B779")
segments(5, GT_reg[1] + GT_reg[2] * 5,
         30, GT_reg[1] + GT_reg[2] * 30,
         col = "#35B779", lwd = 1.5)
points(GT_temp, GT_epsilon, pch = 16, col = "#35B779")

# Add Zeebe (2020)
temp_range = seq(1, 80, 1)
lines(temp_range, epsilon(a18_H2O_OH(temp_range, "Z20-X3LYP")),
      col = "black", lwd = 2, lty = 2)
lines(temp_range, epsilon(a18_H2O_OH(temp_range, "Z20-MP2")),
      col = "black", lwd = 2, lty = 2)

# Add Beck et al. (2005)
segments(BCK_temp, BCK_epsilon - BCK_epsilon_err,
         BCK_temp, BCK_epsilon + BCK_epsilon_err,
         col = "#FDE725")
points(BCK_temp, BCK_epsilon, pch = 18, col = "#FDE725")

# Add Clark et al. (2005)
segments(CLK_temp - CLK_temp_err,
         CLK_epsilon, CLK_temp + CLK_temp_err, CLK_epsilon,
         col = "#482878")
points(CLK_temp, CLK_epsilon, pch = 18, col = "#482878")

# Add the points with the error bars to the plot.
segments(dat$Temp, dat$e_H2O_OH - dat$e_H2O_OH_err,
         dat$Temp, dat$e_H2O_OH + dat$e_H2O_OH_err,
         col = "#31688E")
segments(dat$Temp - rep(2, length(dat$Temp)), dat$e_H2O_OH,
         dat$Temp + rep(2, length(dat$Temp)), dat$e_H2O_OH,
         col = "#31688E")
points(
  dat$Temp,
  dat$e_H2O_OH,
  pch = ifelse(dat$d18O_sol_VSMOW < 0, 15, 16),
  col = "#31688E"
)

# mark the excluded replicates
points(dat$Temp[dat$Comment != 0],
       dat$e_H2O_OH[dat$Comment != 0],
       pch = 4,
       col = "red")

dev.off()


#### Figure S1 ####

pdf("OH Figure S1.pdf", 5, 5, encoding = "MacRoman")
par(mar = c(4.5, 4.5, 0.3, 0.3))
plot(0, type = "l", xaxt = "n", yaxt = "n",
     xlim = c(-1.4, 0.2),
     ylim = c(-25, -17),
     xlab = expression(delta ^ 13 * "C"[carbonate] * -delta ^ 13 * "C"[CO2] * " (‰ VPDB)"),
     ylab = expression(delta ^ 18 * "O"[carbonate] * -delta ^ 18 * "O"[CO2] * " (‰ VSMOW)"))
axis(1, at = seq(-1.4, 0.2, by = 0.1), labels = seq(-1.4, 0.2, by = 0.1))
axis(2, at = seq(-24.5, -17.5, by = 1), labels = NA)
axis(2, at = seq(-25, -17, by = 1), labels = seq(-25, -17, by = 1), las = 1)

# CO2 gas composition
rect(xleft = -0.01, xright = 0.01, ybottom = -26, ytop = -16, col = "gray79", border = NA)
abline(v = 0, lty = 2, lwd = 1.5, col = "gray40")

# Add the points with the error bars to the plot.
segments(dat$Dd13C - dat$d13C_SD, dat$Dd18O,
         dat$Dd13C + dat$d13C_SD, dat$Dd18O,
         col = "gray1")

segments(Dd13C, dat$Dd18O - dat$d18O_SD,
         Dd13C, dat$Dd18O + dat$d18O_SD,
         col = "gray1")

for (i in unique(dat$Temp)) {
  points(
    dat$Dd13C[dat$Temp == i],
    dat$Dd18O[dat$Temp == i],
    pch = ifelse(dat$d18O_sol_VSMOW[dat$Temp == i] < 0, 15, 16),
    col = color$color[color$Temp == i]
  )
}

# Mark the excluded replicates
points(dat$Dd13C[dat$Comment != 0],
       dat$Dd18O[dat$Comment != 0],
       pch = 4,
       col = "red")

dev.off()


#### Figure S2 ####

pdf("OH Figure S2.pdf", 5, 5, encoding = "MacRoman")
par(mar = c(4.5, 4.5, 0.3, 0.3))
plot(0, type = "l", axes = T, xlim = c(0, 80), ylim = c(35, 50),
     xlab = "Temperature (°C)",
     ylab = expression(epsilon ^ 18 * ""[H2O / OH] * " (‰)"),
     lty = 0, las = 1)

# Exclude further samples by commenting in the "OH_data_v1.xlsx" file
dat$Comment[dat$Dd13C< -0.4] = "exclude_FigS2"

ECLR_FigS2 = york_fit(
  x = dat$Temp[dat$Comment == 0],
  x_err = rep(2, length(dat$Temp[dat$Comment == 0])),
  y = dat$e_H2O_OH[dat$Comment == 0],
  y_err = dat$e_H2O_OH_err[dat$Comment == 0]
)

# Add the error considering linear regression to the plot.
york_plot(
  x = dat$Temp[dat$Comment == 0],
  intercept = ECLR_FigS2$intercept,
  intercept_se = ECLR_FigS2$intercept_se,
  slope = ECLR_FigS2$slope,
  slope_se = ECLR_FigS2$slope_se,
  add = TRUE,
  col = "#31688E")

# Add the points with the error bars to the plot.
segments(dat$Temp, dat$e_H2O_OH - dat$e_H2O_OH_err,
         dat$Temp, dat$e_H2O_OH + dat$e_H2O_OH_err,
         col = "#31688E")
segments(dat$Temp - rep(2, length(dat$Temp)), dat$e_H2O_OH,
         dat$Temp + rep(2, length(dat$Temp)), dat$e_H2O_OH,
         col = "#31688E")
points(
  dat$Temp,
  dat$e_H2O_OH,
  pch = ifelse(dat$d18O_sol_VSMOW < 0, 15, 16),
  col = "#31688E"
)

# mark the excluded replicates
points(dat$Temp[dat$Comment != 0],
       dat$e_H2O_OH[dat$Comment != 0],
       pch = 4,
       col = "red")

dev.off()


#### Manuscript parts ####

# Save all data (Table 2) in an Excel file.
wb = openxlsx::createWorkbook()
openxlsx::addWorksheet(wb, "Table 2")
openxlsx::writeData(
  wb = wb,
  sheet = 1,
  x = dplyr::select(
    dat,
    Sample,
    Replicates,
    Temp,
    d18O_sol_VSMOW,
    d13C_VPDB,
    d18O_SD,
    d18O_VSMOW,
    d18O_SD,
    e_H2O_OH,
    e_H2O_OH_err
  )
)
openxlsx::saveWorkbook(wb = wb, file = "OH Table 2.xlsx", overwrite = TRUE)

# Bulk oxygen isotope KIE for this study
d18O_OH_Z20a = B_from_a(a18_H2O_OH(dat$Temp, "Z20-X3LYP"), dat$d18O_sol_VSMOW)
reactants_Z20a = 2 / 3 * dat$d18O_CO2_VSMOW + 1 / 3 * d18O_OH_Z20a
KIE_Z20a = (((dat$d18O_VSMOW + 1000) / (reactants_Z20a + 1000)) - 1) * 10 ^ 3
d18O_OH_Z20b = B_from_a(a18_H2O_OH(dat$Temp, "Z20-MP2"), dat$d18O_sol_VSMOW)
reactants_Z20b = 2 / 3 * dat$d18O_CO2_VSMOW + 1 / 3 * d18O_OH_Z20b
KIE_Z20b = (((dat$d18O_VSMOW + 1000) / (reactants_Z20b + 1000)) - 1) * 10 ^ 3
round(mean(KIE_Z20a[1:20]), 1)
round(mean(KIE_Z20b[1:20]), 1)

# Bulk oxygen isotope KIE for high-pH pools: Table 7 of Christensen et al. (2021)
## Data from: Clark et al. (1992), Mervine et al. (2014),
## Falk et al. (2016), and Christensen et al. (2021)
temps = c(28, 28, 28, 27, 17)
d18O_c = c(14.3, 13.5, 14.4, 14.0, 11.4)
d18O_H2O = c(-0.5, -0.9, -0.5, 0.1, -5.7)
d18O_CO2 = c(31.1, 31.1, 31.1, 31.1, 30.9)

d18O_OH_Z20a = B_from_a(a18_H2O_OH(temps, "Z20-X3LYP"), d18O_H2O)
reactants_Z20a = 2 / 3 * d18O_CO2 + 1 / 3 * d18O_OH_Z20a
KFF_Z20a = (((d18O_c + 1000) / (reactants_Z20a + 1000)) - 1) * 10 ^ 3
d18O_OH_Z20b = B_from_a(a18_H2O_OH(temps, "Z20-MP2"), d18O_H2O)
reactants_Z20b = 2 / 3 * d18O_CO2 + 1 / 3 * d18O_OH_Z20b
KFF_Z20b = (((d18O_c + 1000) / (reactants_Z20b + 1000)) - 1) * 10 ^ 3
round(KFF_Z20a, 1)
round(KFF_Z20b, 1)

# Equation 9 in the manuscript and the abstract
round(ECLR$slope, 3)
round(ECLR$slope_se, 3)
round(ECLR$intercept, 1)
round(ECLR$intercept_se, 1)
round(ECLR$R2, 2)
