function r = ddm_bound_mn(covfunc)
% 
% Compute the common prediction values at bpts boundary points
% by taking the weighted average of predictions from two neighboring local
% Gaussian process regression models. In order to understand the rationale 
% behind the weighted average, please refer to the following paper:
%
%   Park et al. (2011), 'Domain Decomposition Approach for Fast Gaussian 
%     Process Regression of Large Spatial Datasets', Journal of Machine 
%     Learning Research, vol 12.
%
% Copyright (c) by Chiwoo Park, 2011-06-07

global subdomains
global interfaces
global vertices
global M
global V
global dof
global bpts
global iT_q2
global my_rank
global comm

int_pts= dof-1;
rowIdx = @(m) [interfaces{m}.x(1), V+int_pts*(m-1)+(1:int_pts), interfaces{m}.x(2)];
wVert  = [((bpts-1):-1:1)'/bpts, (1:(bpts-1))'/bpts];

r = zeros(V+int_pts*M, 1);
if my_rank == 0  % for master node
    %centerized logic execution for obtaining boundary values
    for m = 1:M
        interface = interfaces{m};
        o_i       = subdomains{interface.i};  
        o_j       = subdomains{interface.j};
        interface.rowI = rowIdx(m); interfaces{m} = interface;
        x0        = [vertices(interfaces{m}.x(1), :); wVert*vertices(interface.x, :); vertices(interfaces{m}.x(2), :)];
        me_i = localGP_pred(x0, o_i, covfunc); 
        me_j = localGP_pred(x0, o_j, covfunc); 
        r(rowIdx(m), :) = iT_q2 * (o_j.yKy * me_i + o_i.yKy * me_j)/(o_i.yKy + o_j.yKy);
    end
    %broadcast the boundary values (r) to local nodes
    MPI_Bcast(0, '33000', comm, r);
    
else   % for slave nodes
    for m = 1:M
        interface = interfaces{m};
        interface.rowI = rowIdx(m); interfaces{m} = interface;
    end
    % wait for the updates from the root node
    r = MPI_Recv(0, '33000', comm);
end
end