function [model, train, test, mu, s2, mse, nlpd] = ddmGP_mc(x, y, dd_param, cv_param, xs, ys)
% Domain Decomposition Method for 2-d Gaussian Process regression. The
% ddmGP function is used to train the regression model based on the domain 
% decomposition method (Park et al. 2011) with a very large spatial data,
% or it is used to predict the regression function values at test cases.
% This function only works when the training data is two dimensional. 
%
% Two modes are possible: training or prediction: if no test cases are
% supplied, then the trained GP regression model is provided; If test cases 
% are given, then the predictions on the test cases are returned. Usage:
%
%   training: [model               ] = main_ddm(x,y, dd_param, cv_param);
% prediction: [model mu s2         ] = main_ddm(x,y, dd_param, cv_param, xs);
%         or: [model mu s2 mse nlpd] = main_ddm(x,y, dd_param, cv_param, xs, ys);
%
% where:
%
%   x                 n by 2 matrix of training inputs
%   y                 column vector of length n of training targets
%   dd_param.meshfunc the string name of the function used to decompose the
%                     domain into subdomains.
%   dd_param.mparam   input parameters for meshfunc (e.g. the size of
%                     subdomains)
%   dd_param.p        degrees of freedom for representing each boundary
%                     function
%   dd_param.q        the number of locations to check the continuity of 
%                     predictions on a boundary
%   
%   cv_param.covfunc  prior covariance function 
%   cv_param.local    (optional) 1 (use the local hyperparameters; default), 
%                                0 (use the global one)
%   cv_param.frachyper (optional) the fraction of the training dataset used
%                      used for learning the hyperparameters (range: 0.0 - 1.0, 
%                      default: 1.0)
%   cv_param.nIter    (optional) the maximum number of iterations for optimizing the
%                     hyperparameters (default: 50)
%   cv_param.logtheta  (optional) used as hyperparameters of the prior
%                      covariance function. If it is specified, the
%                      hyperparameter learning step is skipped.
%   cv_param.logtheta0 (optional) initial guess of hyperparameters of the
%                      prior covariance function
%   xs             (optional) ns by 2 matrix of test inputs
%   ys             (optional) column vector of length ns of test targets
%
%   model    trained Gaussian process model
%   train    the training time in seconds
%   test     the testing time in seconds
%   mu       column vector (of length ns) of predictive output means
%   s2       column vector (of length ns) of predictive output variances
%   mse      mean squared error computed with the test data (xs, ys)
%   nlpd     negative log predictive density
% 
% Refer to:
%     Park et al. (2011), 'Domain Decomposition Approach for Fast Gaussian 
%     Process Regression of Large Spatial Datasets', Journal of Machine 
%     Learning Research, vol 12.
%
% Copyright (c) by Chiwoo Park, 2011-06-07

global N
global M
global V

% check if the inputs are valid
if check_field(dd_param, {'meshfunc', 'mparam', 'p', 'q'}) < 4
     error('ddmGP_mc:argChk', 'Some mandatory fields in dd_param are missing.');
end

% check if the inputs are valid
if ~isfield(cv_param, 'covfunc')
     error('ddmGP_mc:argChk', 'Some mandatory fields in cv_param are missing.');
end

% default parameter setting
if ~isfield(cv_param, 'local') 
    cv_param.local = 1;
end
if ~isfield(cv_param, 'nIter')
    cv_param.nIter = 50;
end
if ~isfield(cv_param, 'frachyper')
    cv_param.frachyper = 1;
end

% check the dimensions of training and test dataset 
if size(x, 2) ~= 2
    error('ddmGP:argChk', 'x must have two columns');
end
if size(x, 1) ~= size(y, 1)
    error('ddmGP:argChk', 'x and y must have the same number of rows');
end
if nargin > 4 
    if size(xs, 2) ~= 2
        error('ddmGP:argChk', 'xs must have two columns');
    end
    if nargin == 6
        if size(xs, 1) ~= size(ys, 1)
            error('ddmGP:argChk', 'xs and ys must have the same number of rows');
        end
    end
end
dof       = dd_param.p;
bpts      = dd_param.q; 
covfunc   = cv_param.covfunc;

% subsampling the training dataset for efficient hyperparameter learning
[dum, I]   = sort(rand(size(x,1),1)); clear dum;
idx        = zeros(size(x,1),1); 
idx(I(1:round(size(x,1)*cv_param.frachyper))) = 1; 
idx        = logical(idx);

% decompose the training dataset following the inputs specified in dd_param
[subdomains, cp_subdomains, interfaces, vertices, memberFunc, affine] = ...
     meshFunc_mc(x, y, idx, dd_param);

N = size(subdomains,1);
M = size(interfaces,1);
V = size(vertices,1);

% T_q  : the global basis functions of Largrange finite element with degree
%        of freedom dof; evaluated at bpts points
% iT_q2: the inverse of T_q 
% rowIdx: the row indices (in 'vertices' matrix) of the bpts point coordinates
%         where the prediction continuity is checked over the mth interface
T_q     = LargrangeFE(((0:bpts)/bpts)', dof);
iT_q2   = pinv(T_q);
tic;
% train the global hyperparameter if necessary
disp('learning hyperparamters....');
if cv_param.local == 0
    cv_param.logtheta = cv_param.logtheta0;
else
    if ~isfield(cv_param, 'logtheta')
       cv_param.logtheta = minimize(cv_param.logtheta0, 'loglikelihood_ddm_mc', ...
                                    -cv_param.nIter, covfunc, cp_subdomains); 
    end
end

parfor i = 1:N
    subdomains{i} = ddm_local_mc(subdomains{i}, cv_param, bpts, dof, V, affine);
end 
disp('learning boundary values');
r = ddm_bound_mc(subdomains, interfaces, dof, bpts, iT_q2, covfunc, affine);

% learning local models
disp('learning local models');
for i = 1:N
    sd     = subdomains{i};
    n_intf = sd.int_idx(length(sd.int_idx));
    qvalue = zeros(1, sd.q);
    for j = 1:n_intf
        qvalue(sd.gx_block{j}) = r(sd.rx_block{j}, 1)' * T_q';
    end
    sd.qvalue = qvalue;
    sd.S  = (qvalue - sd.h_i' * sd.K_iq) * sd.G_i;
    sd.v  = sd.h_i * sd.S / sd.yKy;
    subdomains{i} = sd;
end 
train = toc;

% compile the trained model
model.subdomains = subdomains;
model.dd_param   = dd_param;
model.cv_param   = cv_param;
model.affine     = affine;
model.memberFunc = memberFunc;

% prediction at test inputs
if nargin > 4
    disp('testing....');
    [mu, s2, test] = ddm_pred_mc(xs, model);
    if nargin > 5  %compute MSE & NLPD
        ns = size(xs,1);
    	se   = (ys - mu)' * (ys - mu);
        nlpd = sum(0.5*(log(2*pi*s2) + ((ys - mu).^2)./s2));
        mse  = se / ns;
        nlpd = nlpd / ns;
    end
end