function [subdomains, interfaces, vertices, memberFunc, affine] = rectGrid(sx, x, y, idx)
%
% uniform grid (retangular mesh)
% -----------------> x1 (n1 meshes)
% (i)----(i+n2)
% |        |
% (i+1)--(i+n2+1)
% |
% x2 (n2 meshes)
%
% where 
%     sx        : 1 by 2 array containing the number of grid points along each
%                 dimesion within a subdomain
%     x         : n by 2 matrix of training inputs
%     y         : column vector of length n of training targets
%     idx       : logical mask to indicate the subset of training data
%                 used for learning the hyperparameters.
%     subdomains: N by 1 cell array of structures containing information
%                 regarding all the subdomains obtained by the decomposition
%     interfaces: M by 1 cell array of structures containing information
%                 regarding all the polygonal boundaries of the subdomains 
%     vertices  : V by 2 matrix of the vertex coordinates of the
%                 polygons representing the subdomain boundaries.
%     memberFunc: a membership function to check if data x belongs to 
%                 subdomain sd (usage: memberFunc(x, sd))
%     affine    : affine transformation that transforms the global coordinates
%                 into the local coordinates
%
% Copyright (c) by Chiwoo Park, 2011-06-07

xmesh1 = unique(x(:,1)); n1 = size(xmesh1, 1);
xmesh2 = unique(x(:,2)); n2 = size(xmesh2, 1);
g_idx1 = 1:sx(1):n1;
if g_idx1(end) ~= n1
    sp = round((n1 - g_idx1(end))/2);
    g_idx1 = sp:sx(1):n1;
    g_idx1(1) = 1;
    g_idx1(end) = n1;
end
gridx1 = xmesh1(g_idx1);
g_idx2 = 1:sx(2):n2;
if g_idx2(end) ~= n2
    sp = round((n2 - g_idx2(end))/2);
    g_idx2 = sp:sx(2):n2;
    g_idx2(1) = 1;
    g_idx2(end) = n2;
end
gridx2 = xmesh2(g_idx2);

% generate meshes and obtain the vertex cooridates of the rectangular grids
[mx1, mx2] = meshgrid(gridx1, gridx2); [n2, n1] = size(mx1);
vertices   = [reshape(mx1, n1*n2, 1) reshape(mx2, n1*n2, 1)];

% create a membership function & an affine transformation function
memberFunc = @(x, sd) logical(x(:,1) >= vertices(sd.T(1), 1) & ...
                      x(:,1) <= vertices(sd.T(3), 1) &  ...
                      x(:,2) >= vertices(sd.T(1), 2) &  ...
                      x(:,2) <= vertices(sd.T(2), 2));
affine  = @(x, sd) (x - repmat(vertices(sd.T(1), :), size(x,1), 1))./ repmat(sd.range, size(x,1), 1);

% construct structures: subdomains, interfaces
N          = (n1-1)*(n2-1);  % total number of subdomains
subdomains = cell(N, 1);
M1         = (n2-2) * (n1-1); M2 = (n1-2) * (n2-1);
M          = M1 + M2;
interfaces = cell(M, 1);    % total number of subdomain boundaries
intc       = 0;
for i = 1:N
    rowI    = mod(i, n2-1); if(rowI == 0); rowI = n2-1; end
    colI    = floor((i-1)/(n2-1))+1;
    s       = i + colI - 1;
    sd.rowI = rowI; sd.colI = colI;
    sd.T    = [s s+1 s+n2 s+n2+1];
    sd.segI = memberFunc(x, sd);
    sd.x    = x(sd.segI, :);
    sd.range= [vertices(s+n2, 1) - vertices(s, 1), vertices(s+1, 2) - vertices(s, 2)];
    sd.hx   = x(sd.segI & idx, :); sd.hy = y(sd.segI & idx, :);
    sd.y    = y(sd.segI, :);
    sd.n    = size(sd.x, 1);
    
    neighbors = zeros(1, 5);  cnt = 0;
    int_idx   = zeros(1, 5); 
    if rowI > 1
        cnt = cnt+1; neighbors(cnt) = i-1;
        intc = intc+1;
        interface.x = [s s+n2];
        interface.i  = i-1;
        interface.j  = i;
        interfaces{intc} = interface;
        int_idx(cnt) = intc;
        subdomains{interface.i}.int_idx(logical(subdomains{interface.i}.neighbors(1:4) == i)) = intc;
    end
    if colI > 1  
        cnt = cnt+1; neighbors(cnt) = i-n2+1;
        intc = intc+1;
        interface.x = [s s+1];
        interface.i  = i-n2+1;
        interface.j  = i;
        interfaces{intc} = interface;
        int_idx(cnt) = intc;
        subdomains{interface.i}.int_idx(logical(subdomains{interface.i}.neighbors(1:4) == i)) = intc;
    end
    if rowI < (n2-1)
        cnt = cnt+1; neighbors(cnt) = i+1;
    end
    if colI < (n1-1)
        cnt = cnt+1; neighbors(cnt) = i+n2-1;
    end
    neighbors(5) = cnt;
    int_idx(5) = cnt;
    sd.neighbors = neighbors;
    sd.int_idx = int_idx;
    subdomains{i} = sd;
end