function [nll] = pic_lik(w, x, subdomains, param)

% pic_lik: neg. log likelihood for Local and Global Gaussina Process with 
% respect to pseudo-inputs and hyperparameters. Gaussian covariance with one
% lengthscale per dimension.
%
% w               [reshape(xb,n*dim,1); hyp] where xb is pseudo-inputs
% x               training inputs
% param.M         number of pseudo-inputs
% param.covfunc   prior covariance function of Gaussian process
% subdomains      N-by-1 cell array, containing the information about 
%                 a partition of training inputs by k-means clustering
% idx             logical mask to indicate the subset of training data
%                 used for learning the hyperparameters.
% nll -- negative log likelihood
%
% Copyright (c) by Chiwoo Park, 2011-06-07
fprintf('.');
covfunc = param.covfunc;
M       = param.M; 

[N,dim] = size(x); 
xb       = reshape(w(1:end-dim-2),M,dim);
logtheta = w((end-dim-1):end);

K_M  = feval(covfunc{:}, logtheta, xb);
[K_xx, K_NM] = feval(covfunc{:}, logtheta, x, xb);               
L_M  = chol(K_M, 'lower');
iLK  = L_M \ K_NM';

nGrp    = size(subdomains, 1);
Q_M     = K_M;
X       = zeros(M, 1);
nll     = 0;
logdet  = 0;
for i = 1:nGrp
    sd    = subdomains{i};
    K_B   = feval(covfunc{:}, logtheta, sd.hx); %C_B
    lidx  = sd.segI & param.idx;
    K_BM  = K_NM(lidx, :);  %UB
    D_B   = K_B - iLK(:,lidx)'*iLK(:,lidx); %AB
    
    L_B   = chol(D_B, 'lower');
    LK_BM = L_B \ K_BM; 
    Ly    = L_B \ sd.hy;
    
    Q_M   = Q_M + LK_BM' * LK_BM;
    X     = X   + LK_BM' * Ly;
    nll   = nll  + Ly' * Ly;
    
    % use logdet(A) = tr(log(A))
    logdet = logdet + 2*sum(log(diag(L_B)));
    sd.L_B = L_B;
    subdomains{i} = sd;
end

% Negative Log Likelihood   
C_L    = chol(Q_M, 'lower');
b      = C_L \ X;
nll    = nll - b'*b;
logdet = logdet - 2*sum(log(diag(L_M))) + 2*sum(log(diag(C_L)));
nll    = nll + logdet;
end