function [model, train, test, mu, s2, mse, nlpd] = baggingGP(x, y, param, xs, ys)
% This is the implementation of 'Bagging for Gaussian Process', which is
% published at 
%
% Tao Chen and Jianghong Ren. Bagging for Gaussian process regression. 
% Neurocomputing, 72(7-9):16051610, 2009.

% Two modes are possible: training or prediction: if no test cases are
% supplied, then the trained GP regression model is provided; If test cases 
% are given, then the predictions on the test cases are returned. Usage:
%
%   training: [model               ] = main_ddm(x,y, dd_param, cv_param);
% prediction: [model mu s2         ] = main_ddm(x,y, dd_param, cv_param, xs);
%         or: [model mu s2 mse nlpd] = main_ddm(x,y, dd_param, cv_param, xs, ys);
%
% where:
%
%   x                 n by d matrix of training inputs
%   y                 column vector of length n of training targets
%
%   param.M         the size of each bagging sample
%   param.K         the total number of bagging samples
%   param.covfunc   prior covariance function 
%   param.local     (optional) 1 (use the localized sampling), 
%                   0 (use the standard bagging; default)
%   param.frachyper (optional) the fraction of the training dataset used
%                   used for learning the hyperparameters (range: 0.0 - 1.0, 
%                   default: 1.0)
%   param.nIter     (optional) the maximum number of iterations for optimizing the
%                   hyperparameters (default: 50)
%   param.logtheta0 initial guess of hyperparameters of the
%                   prior covariance function
%   xs             (optional) ns by d matrix of test inputs
%   ys             (optional) column vector of length ns of test targets
%
%   model    trained Gaussian process model
%   train    the training time in seconds
%   test     the testing time in seconds
%   mu       column vector (of length ns) of predictive output means
%   s2       column vector (of length ns) of predictive output variances
%   mse      mean squared error computed with the test data (xs, ys)
%   nlpd     negative log predictive density
% 
% Copyright (c) by Chiwoo Park, 2011-06-07

% check if the inputs are valid
if sum(check_field(param, {'M', 'K', 'covfunc', 'logtheta0'})) < 4
     error('bgp_global:argChk', 'Some mandatory fields in param are missing.');
end

% default parameter setting
if ~isfield(param, 'frachyper')
    param.frachyper = 1;
end

if ~isfield(param, 'nIter')
    param.nIter = 50;
end

if ~isfield(param, 'local')
    param.local = 0;
end

M = param.M;
K = param.K;
logtheta0 = param.logtheta0;
covfunc   = param.covfunc;
frachyper = param.frachyper;

N = size(x, 1); 
experts = cell(K, 1);
if param.local == 0 
    % bootstrapping (random sampling with replacement)
    for i = 1:K
        [dum, idx] = sort(rand(N,1)); clear dum;
        idx = idx(1:M);
        ex.idx = idx;
        ex.x   = x(idx, :);
        ex.y   = y(idx, :);
        experts{i} = ex;
    end
else
    % construct a KD Tree & perform the localized sampling based on the
    % KD tree
    [IDX, cpt] = kmeans(x, K); 
    Tree_x  = kdtree_build(x);
    for i = 1:K
        idx  = kdtree_k_nearest_neighbors(Tree_x, cpt(i, :)', M);
        ex.x =  x(idx, :);
        ex.y =  y(idx, :);
        experts{i} = ex;
    end
end

% training
tic;
for i = 1:K
    ex = experts{i};
    [dum, idx0] = sort(rand(M,1)); clear dum;
    idx0 = idx0(1:round(M*frachyper));
    ex.logtheta = minimize(logtheta0, 'loglikelihood', -param.nIter, ...
        covfunc, ex.x(idx0, :), ex.y(idx0, :));
    ex.sigK     = feval(covfunc{:}, ex.logtheta, ex.x);
    L           = chol(ex.sigK, 'lower');
    ex.h_i      = L \ ex.y;
    ex.L_i      = L \ eye(size(ex.x, 1));
    experts{i}  = ex;
end
train = toc;
model.experts = experts;
model.param   = param;

% testing 
if nargin > 3
    tic;
    [mu, s2] = bgp_pred(xs, model);
    ns = size(xs, 1);
    test = toc;
    if nargin > 4
        se   = (ys - mu)' * (ys - mu);
        nlpd = sum(0.5*(log(2*pi*s2) + ((ys - mu).^2)./s2));
        mse  = se / ns;
        nlpd = nlpd / ns;
    end
end