/****************************************************************************
	Main source file for the paper: Shrivastava, and Ding (2009) ``Graph based isomorph-free generation 
	of two-level regular fractional factorial designs,'' Journal of Statistical Inference and Planning, 
	under second revision.
	The code includes the source files for nauty 2.2 (copyright, Brendan McKay)

	Code written by: Abhishek K. Shrivastava (Last Update on June 30, 2009)
	Copyright: Abhishek K. Shrivastava and Yu Ding (Industrial & Systems Engineering, Texas A&M)

	The code is free to use for academic research only, unless it violates the copyright held by Brendan 
	McKay for nauty. Additionally, if any use of the code may lead to any benefits, personal or otherwise, 
	directly or indirectly, then the permission of the authors of the above mentioned paper SHOULD be 
	sought. In case of any confusions with the use of the code, please seek the permission of the authors.
	Please cite the above mentioned paper and, if necessary, nauty, when using this code.

	Comment: As of June 30, 2009, the commenting on the code is clearly inadequate. If anyone has any 
		trouble using the code, then please contact the first author of the paper. In future, we will 
		try to add more comments to improve readability
*******************************************************************************/

/* -----------------------------------------------------------------
	Gist of the algorithm/code:
	This extends the algorithm given in Lin & Sitter (2007)
	We improve their algo by two ways:
	1. use graph-based isomorphism check
	2. reduce the candidate generators considered to get the new design

	------------------------------
	* TO DOS:
	 1. Replace mallocs by new
	 2. Check for memory leaks
	 3. Optimize memory usage --
		a. exploit the knowledge of bipartite graph by keeping only a quarter of the graph in storage(??)

	----------------------
	The BASIC ALGO is here...

	// input N (# factors), K (fraction)
	// create ordered list of generators g_list
	// create initial design D0
	// D_current = {D0}; D_new = null;
	// for k=k_of_D0:K
		// for each d \in D_current
			// last_gen_used = last_gen(d)
			// get g_list_uniq = subset of g_list unique under the automorphism group of d
			// for each generator 'g' in g_list_uniq
				// d' = d \+ g
				// w' = word_length_pattern(d')
				// if(w' is unique among all ds in D_new) D_new = D_new \+ {d'}
				// else 
						// cg' = canonical_graph(d')
						// if(cg' is unique among all ds in D_new) D_new = D_new \+ {d'}
						// else discard d'.. construct a new d'
					//endif
				// endif
			// endfor
		// endfor
	// D_current = D_new
	// D_new = null
	// endfor
	// output D_current 
--------------------------------------------------------------------*/

#define GET_STATS 1 //=> computation times will be recorded
#define USE_BAGS 1

//#define USE_AUTG 1 -- try switching only through the preprocessor options

//#define EXPLOIT_DCS_TREATMENT_SUBGROUP_1TO1_RELATION 1 -- only switch through project properties; 
	//see Bailey '77, the treatment combinations (design matrix) & defining contrast subgroup have a 
	//1-1 correspondence, it is more efficient to work with the treatment comb group when the 
	//size of latter group < size of DCS i.e. 2^{n-k} < 2^k =>n<2k!
//#define WORDSIZE 64 //32 => // change for 32-bit & 64-bit; only through preprocessor options

//-------------------------

#define LEVELS 2

#include <cmath>
#include <vector>
#include <iostream>
#include <fstream>
#include <algorithm>
#include <ctime>
#include <string>

#include "nauty/nauty.h"



#include "FF2Design.h"

using namespace std;

unsigned int MAX_FACTORS, NUM_FACTORS, FRAC, RES;
int numxs;
ofstream fout;
FF2Design *d_ptr_nauty; // a global reference to the current design called
				  // by current nauty call for catching the automorphism
				  // group of the current design *d_ptr

// --- NOTE SOME STATS ------
#ifdef GET_STATS
	ofstream fstats;

	int NUM_WLP_COMPUTATIONS = 0;
	int NUM_CG_COMPUTATIONS = 0; //= NUM_NAUTY_CALLS;

	int NUM_WLP_COMPARISONS = 0;
	int NUM_CG_COMPARISONS = 0;

	clock_t START_TIME;
	clock_t INC_TIME;
	clock_t END_TIME;
#endif
//----------------------------


static int nextOrderedGen(const int & last_gen);
static int vpopcount(int *vec, const int & vlen);
static void dec2basel(const int & n, const int & l, int* nb, const int & nb_sz);
static int basel2dec(int* nb, const int & nb_sz, const int & l);
static vector<bool> getUniqueGensUnderAutG(FF2Design & d);
static vector<bool> computeNewCSWords(const vector<bool> & oldCS, const int & gen, const int & gen_no);
static void computeWordLengthPattern(FF2Design & dx); // no constant 'cause add wlp to dx
#ifdef USE_BAGS
static bool isUniqueWLP(const vector<int> & wx, vector<FF2Design> & D_new, vector<int> & wlp_bag, int & xwlp_bag);
#else
static bool isUniqueWLP(const vector<int> & wx, vector<FF2Design> & D_new);
#endif


static void computeCanonicalGraph(FF2Design & dx, const int & gen_no);
#ifdef USE_BAGS
static bool isUniqueCanonicalGraph(const vector <vector<bool> > & cgx, vector<FF2Design> & D_new, vector<int> & wlp_bag, int & xwlp_bag);
#else
static bool isUniqueCanonicalGraph(const vector <vector<bool> > & cgx, vector<FF2Design> & D_new);
#endif

#ifdef USE_BAGS
static void outputDesigns(vector<FF2Design> & D_current, const int & gen_no, const vector<int> & wlp_bag);
#else
static void outputDesigns(vector<FF2Design> & D_current, const int & gen_no);
#endif
static UPROC userautomproc(int, permutation *, int *, int, int, int);
static void freeSomeMemory(vector<FF2Design> & Dvec);
static void writegraph(const graph *g, const unsigned int & n, const unsigned int & m);
static void writeset(const set *s, const unsigned int & slen);
static void writegraph(const vector <vector<bool> > & g, const unsigned int & n);
static void reduceCandidateGensByWordLength(vector<bool> & gens);

int main(int argc, char *argv[])
{
	register int k;
	int last_gen_used, g;
	vector<bool> unique_gens;
	string fname="out";

	// input N (# factors), K (fraction)

	if (argc>1)
	{
		MAX_FACTORS = atoi(argv[1]);
		FRAC = atoi(argv[2]);
		if (argc>=4) RES = atoi(argv[3]);
		else RES = 3;
		if (argc==5) fname = argv[4];
	}
	else
	{
		cout << "Input Number of Factors: ";
		cin >> MAX_FACTORS;
		cout << endl << "Input Design Fraction: ";
		cin >> FRAC;
		cout << endl << "Input Design Resolution (>=3): ";
		cin >> RES;
		cout << endl;
	}
	fout.open((fname+".txt").c_str());
#ifdef GET_STATS
	fstats.open((fname+"_stats.csv").c_str());
	fstats << "NUM_RUNS,NUM_FACTORS,FRACTION,RESOLUTION,NUM_UNIQUE_DESIGNS,INCREMENTAL_TIME,TOTAL_TIME,NUM_WLP_COMPUTATIONS,NUM_CAN_GRAPH_COMPUTATIONS,NUM_WLP_COMPARISONS,NUM_GRAPH_COMPARISONS" << endl;
#endif

#ifdef GET_STATS
	START_TIME = clock();
	END_TIME = START_TIME;
#endif

	// create ordered list of generators g_list
	numxs = (int) (1 << (MAX_FACTORS-FRAC));

	// create initial design D0
	FF2Design d0;

	// D_current = {D0}; D_new = null;
	vector<FF2Design> D_current;
	D_current.push_back(d0);

	// for k=k_of_D0:K
	for (k=1; k<= (int) FRAC; k++)
	{
#ifdef GET_STATS
		INC_TIME = END_TIME;
#endif
		NUM_FACTORS = MAX_FACTORS-FRAC+k;
		vector<FF2Design> D_new;
		vector<FF2Design>::iterator d = D_current.begin();
		vector<FF2Design>::iterator dlast = D_current.end();
#ifdef USE_BAGS
		vector<int> wlp_bag;
		int xwlp_bag;
#endif
		// for each d \in D_current
		for (;d!=dlast; d++)
		{
			// last_gen_used = last_gen(d)
			if (d->isNullDesign())
				last_gen_used = 0;
			else
				last_gen_used = d->getLastCSGenerator();

			// get g_list_uniq = subset of g_list unique under the automorphism group of d
			unique_gens = getUniqueGensUnderAutG(*d);
//			// for each generator 'g' in g_list_uniq
//			g = nextOrderedGen(last_gen_used);
			g = last_gen_used;
//			while (g>=0)
			do {
//				if (!unique_gens[g])
				do {
					g = nextOrderedGen(g);
					if (g<0) break;
//					continue;
				} while (!unique_gens[g]);
				if (g<0) break;
				// d' = d \+ g
				FF2Design dx;
#ifdef USE_BAGS
				xwlp_bag = (int)D_new.size();
#endif
				dx.setCSGenerators(*(d->getCSGenerators()));
				dx.addCSGenerator(g);
				// compute the contrast subgroup
				dx.setContrastSubgroup(computeNewCSWords(*d->getContrastSubgroup(),g,k));
				// w' = word_length_pattern(d')
				computeWordLengthPattern(dx);
				if (dx.getWordLengthPattern()->empty())
					continue;
				// if(w' is unique among all ds in D_new) D_new = D_new \+ {d'}
#ifdef USE_BAGS
				if (isUniqueWLP(*dx.getWordLengthPattern(),D_new, wlp_bag, xwlp_bag))
#else
				if (isUniqueWLP(*dx.getWordLengthPattern(),D_new))
#endif
				{
#ifdef USE_BAGS
					wlp_bag.push_back(xwlp_bag);
#endif
					D_new.push_back(dx);
				}
				// else 
				else
				{

						// cg' = canonical_graph(d')
						computeCanonicalGraph(dx,k);
						// if(cg' is unique among all ds in D_new) D_new = D_new \+ {d'}
#ifdef USE_BAGS
						if (isUniqueCanonicalGraph(*dx.getCanonicalGraph(),D_new,wlp_bag,xwlp_bag))
#else
						if (isUniqueCanonicalGraph(*dx.getCanonicalGraph(),D_new))
#endif
						{ // this happens only if eigs are identical but the canonical graph signatures are not!
#ifdef USE_BAGS
							wlp_bag.push_back(xwlp_bag);
							dx.freeWLPMemory();
#endif
							D_new.push_back(dx);
						}
						// else discard d'.. construct a new d'
						else continue;

				// endif
				}
			// endfor
			} while(g >=0);
			// d is no more needed.. let's just free it's memory (#4 (b) of TODOs)
			d->~FF2Design();
		// endfor
		}
		// D_current = D_new
		D_current.swap(D_new);
		// D_new = null
		D_new.clear();
		// output D_current 
#ifdef GET_STATS
		END_TIME = clock();
		INC_TIME = END_TIME-INC_TIME;
#endif
#ifdef USE_BAGS
		outputDesigns(D_current,k,wlp_bag);
#else
		outputDesigns(D_current,k);
#endif
		freeSomeMemory(D_current);
	// endfor
	}

#ifdef GET_STATS
	END_TIME = clock();
	fout << "Total Time: " << (double) (END_TIME-START_TIME)/CLOCKS_PER_SEC << endl;
#endif

	fout.close();
#ifdef GET_STATS
	fstats.close();
#endif
	return 0;
}


// Returns the next generator (base-10 equivalent) after lastgen in the list 
// of Lin-Sitter ordered generators, given the maximum generator is numxs
static int 
nextOrderedGen(const int & last_gen)
{
	int lastgen = last_gen;
	int last_popcount, i;
	int *last_vec, *next_vec;

	{
		last_vec = (int *)ALLOCS(MAX_FACTORS-FRAC,sizeof(int));
		dec2basel(last_gen,LEVELS,last_vec,MAX_FACTORS-FRAC);
		last_popcount = vpopcount(last_vec,MAX_FACTORS-FRAC);
		FREES(last_vec);
	}
	if (lastgen <= 0) 
	{
		lastgen = 0;
		last_popcount = 1;
	}

	next_vec = (int *)ALLOCS(MAX_FACTORS-FRAC,sizeof(int));

	while (last_popcount <= (int) (MAX_FACTORS-FRAC))
	{
		for (i=lastgen+1; i<numxs; i++)
		{
			dec2basel(i,LEVELS,next_vec,MAX_FACTORS-FRAC);
			if (vpopcount(next_vec,MAX_FACTORS-FRAC) == last_popcount)
			{
				FREES(next_vec);
				return i;
			}
		}
		lastgen = 0;
		last_popcount++;
	}

	FREES(next_vec);
	return -1;
}



static int 
vpopcount(int *vec, const int & vlen)
{
	register int i;
	int count=0;

	for (i=0; i<vlen; i++)
		if (vec[i] > 0) count++;

	return count;
}


// converts base-10 num 'n' to base-l num array 'nb'
static void
dec2basel(const int & ndec, const int & l, int* nbasel, const int & nb_sz)
{
	register int i;
	int quot, rem;
	quot = ndec;
	for (i=0; i<nb_sz; i++)
	{
		rem = quot%l;
		quot = (int)(quot/l);
		nbasel[i] = rem;
	}
	return;
}


// converts base-l num array 'nb' to base-10 num 'n'
static int
basel2dec(int* nbasel, const int & nb_sz, const int & l)
{
	register int i;
	int n;
	n = 0;
	for (i=0; i<nb_sz; i++)
		n += nbasel[i]*(1<<i);
//		n += nbasel[i]*((int) pow(LEVELS,i));

	return n;
}

// Returns a vector<bool> bv with bv[i]=1 if the generator i is
// unique under the automorphism group of design d, where
// the aut(d) is given by a set of generators of the aut group
static vector<bool> 
getUniqueGensUnderAutG(FF2Design & d)
{
	// trivial implementation: currently not discarding isomorphs,
	// i.e., setting all bits to one
	vector<bool> bv(numxs,true);
	reduceCandidateGensByWordLength(bv);
	vector<int> *csg;
	csg = d.getCSGenerators();

	register int i;
	int pj, vppc;
	// remove the contrast subgroup generators from the candidate list
	for (i=0; i < (int)csg->size(); i++)
		bv[(*csg)[i]] = false;

#ifdef USE_AUTG
	// remove the candidates that fall before the last generator of d
	pj = d.getLastCSGenerator();
	int *tmp = new int[MAX_FACTORS-FRAC];
	dec2basel(pj,LEVELS,tmp,MAX_FACTORS-FRAC);
	vppc = vpopcount(tmp, MAX_FACTORS-FRAC);
	for (i=0; i <= pj; i++)
	{
		dec2basel(i,LEVELS,tmp,MAX_FACTORS-FRAC);
		if (vpopcount(tmp,MAX_FACTORS-FRAC) <= vppc)
			bv[i] = false;
	}
	delete[] tmp;



	register int j,k;
	vector< vector<permutation> > *perms;
	vector<permutation> *p;
//	vector<bool> bv_copy(bv);

	perms = d.getAutGroup();

	// loop for each generator of the permutation automorphism group
	for (i=0; i < (int)perms->size(); i++)
	{
		p = &(*perms)[i]; // p -> pointer to ith permutation in aut group
		// loop for each candidate generator j
		for (j=0; j < (int)bv.size(); j++)
		{
			if (!bv[j]) continue; // skip generators which are already rejected
			pj = 0; // pj -> stores the permuted generator, i.e., j^*p (image of j under perm *p)
			// loop for computing pj.. each bit/factor_label is permuted under *p
			for (k=0; k < (int)(MAX_FACTORS-FRAC); k++)
			{
				if (j & (1<<k)) // if gen j has bit k (or factor k) then permute it
													// here, MAX_FACTORS-FRAC complementation is needed 'cause binary format (j) is in reverse ascending order of factor labels
				{
					if ((*p)[k] >= (int)(MAX_FACTORS-FRAC)) // if the gen j's kth factor is getting permuted to a label > MAX_FACTORS-FRAC, then skip this generators permutation under *p
						break;
					pj = pj | (1<<(*p)[k]); // permute kth factor to *p[k] with the obvious (from above) complementation
				}
			}
			if (k == (int) (MAX_FACTORS-FRAC) && pj < (int)bv.size() && pj>j)
				bv[pj] = false;
		}
	}

#endif

	return bv;
}


// computes the additional new words of the contrast subgroup by
// the addition of the new generator gen
static vector<bool> 
computeNewCSWords(const vector<bool> & oldCS, const int & gen, const int & gen_no)
{
	vector<bool> newCS(1<<NUM_FACTORS);
//	int *tmp, *tmp2;
	newCS[0] = true;

	newCS[gen + (1<<(NUM_FACTORS-1))] = true;

	if (oldCS.empty()) return newCS;

	//tmp = (int *) ALLOCS(NUM_FACTORS,sizeof(int));
	//tmp2 = (int *) ALLOCS(NUM_FACTORS,sizeof(int));
	//dec2basel(g,LEVELS,tmp,NUM_FACTORS);
	//tmp[NUM_FACTORS-FRAC+gen_no-1] = 1;

	int i;//, j;

	for (i=1; i< (int) oldCS.size(); i++)
		if (oldCS[i])
		{
					//int j, *tmp, *tmp2;
					//tmp = (int *) ALLOCS(NUM_FACTORS,sizeof(int));
					//tmp2 = (int *) ALLOCS(NUM_FACTORS,sizeof(int));
					//dec2basel(gen,LEVELS,tmp,NUM_FACTORS);
					//tmp[NUM_FACTORS-FRAC+gen_no-1] = 1;
					//for (j=0; j<NUM_FACTORS; j++) tmp2[j]=0;
					//dec2basel(i,LEVELS,tmp2,NUM_FACTORS);
					//for (j=0; j<NUM_FACTORS; j++) tmp2[j] = (tmp[j]+tmp2[j])%LEVELS;
					//basel2dec(tmp2,NUM_FACTORS,LEVELS);
			newCS[i] = true;
			newCS[i^(gen+ (1<<(NUM_FACTORS-1)))] = true;
			//for (j=0; j<NUM_FACTORS; j++) tmp2[j]=0;
			//dec2basel(i,LEVELS,tmp2,NUM_FACTORS);
			//for (j=0; j<NUM_FACTORS; j++) tmp2[j] = (tmp[j]+tmp2[j])%LEVELS;
			//newCS[basel2dec(tmp2,NUM_FACTORS,LEVELS)] = true;
		}

	return newCS;
}


static void
computeWordLengthPattern(FF2Design & dx)
{
	vector<int> wlp(NUM_FACTORS+1-RES,0);
	vector<bool> *csg;
	csg = dx.getContrastSubgroup();
	int i, wl;

	if (!dx.getWordLengthPattern()->empty())
		return;

	int *tmp;
	tmp = (int *) ALLOCS(NUM_FACTORS,sizeof(int));
	for (i=1; i< (int) csg->size(); i++)
		if ((*csg)[i])
		{
			dec2basel(i,LEVELS,tmp,NUM_FACTORS);
			wl = vpopcount(tmp,NUM_FACTORS);
			if (wl < (int) RES)
			{
				FREES(tmp);
				wlp.clear();
				dx.setWordLengthPattern(wlp);
				return;
			}
			wlp[wl-RES]++;
		}

	FREES(tmp);

#ifdef GET_STATS
	++NUM_WLP_COMPUTATIONS;
#endif

	dx.setWordLengthPattern(wlp);
	return;

}


static bool 
#ifdef USE_BAGS
isUniqueWLP(const vector<int> & wx, vector<FF2Design> & D_new, vector<int> & wlp_bag, int & xwlp_bag)
#else
isUniqueWLP(const vector<int> & wx, vector<FF2Design> & D_new)
#endif
{
	if (D_new.empty())
		return true;

	register int i=0;

	for (; i < (int)D_new.size(); i++)
	{
#ifdef USE_BAGS
		if (wlp_bag[i]!=i)
			continue;
#endif

		// the proper WLP comparison

#ifdef GET_STATS
		++NUM_WLP_COMPARISONS;
#endif

		if (*D_new[i].getWordLengthPattern() == wx)
			break;
	}

	if (i!=D_new.size())
	{
#ifdef USE_BAGS
		xwlp_bag = i;
#endif
		return false;
	}

	return true;
}




// use nauty to compute the canonical graph and store the graph
// in dx
static void 
computeCanonicalGraph(FF2Design & dx, const int & gen_no)
{
	graph *gx;
	size_t gx_sz = 0;
	unsigned int gm,gn;
	register int i,j,j2;
	set *gv, *gv2;

#ifdef EXPLOIT_DCS_TREATMENT_SUBGROUP_1TO1_RELATION
	if ((int)NUM_FACTORS < 2*gen_no) 
	{
		int *tmp;
		vector<bool> *csg;
		csg = dx.getContrastSubgroup();

//		gn = (unsigned int)(NUM_FACTORS + (1<<gen_no)-1);
		gn = (unsigned int)(NUM_FACTORS + (1<<(NUM_FACTORS-gen_no))-1);
		gm = (unsigned int)((gn + WORDSIZE -1) / WORDSIZE);
		gx = NULL;
		DYNALLOC2(graph,gx,gx_sz,gn,gm,"malloc");

		nauty_check(WORDSIZE,gm,gn,NAUTYVERSIONID);

		// create the graph gx
		// first the empty graph
		for (i=0; i < (int) gn; i++)
		{
			gv = GRAPHROW(gx,i,gm);
			EMPTYSET(gv,gm);
		}

		tmp = (int *) ALLOCS(NUM_FACTORS,sizeof(int));
		// construct the design table: first the full factorial, then the added columns
		//full factorial
		for (i=0; i<(int)(gn-NUM_FACTORS); i++)
		{
			dec2basel(i+1,LEVELS,tmp,NUM_FACTORS-gen_no);
			gv = GRAPHROW(gx,i+NUM_FACTORS,gm);
			for (j=0; j<(int)NUM_FACTORS-gen_no; j++)
				if (tmp[j] > 0)
				{
					ADDELEMENT(gv,j);
					gv2 = GRAPHROW(gx,j,gm);
					ADDELEMENT(gv2,i+NUM_FACTORS);
				}
		}
		//other columns
		vector<int> *gens = dx.getCSGenerators();
		for (int k=0; k<gen_no; k++)
		{
			dec2basel((*gens)[k],LEVELS,tmp,NUM_FACTORS-gen_no);
			for (i=0; i<(int)(gn-NUM_FACTORS); i++)
			{
				gv = GRAPHROW(gx,i+NUM_FACTORS,gm);
				int flag=tmp[0]*ISELEMENT(gv,0);
				for (j=1; j<(int)NUM_FACTORS-gen_no; j++)
				{
					flag = flag^(tmp[j]*ISELEMENT(gv,j));
				}
				if (flag) 
				{
					ADDELEMENT(gv,NUM_FACTORS-gen_no+k);
					gv2 = GRAPHROW(gx,NUM_FACTORS-gen_no+k,gm);
					ADDELEMENT(gv2,i+NUM_FACTORS);
				}
			}
		}

		// fill the graph with words from the contrast subgroup
		// graph ready!

//		writegraph(gx,gn,gm);

		FREES(tmp);
	}
	else
	{
		int *tmp;
		vector<bool> *csg;
		csg = dx.getContrastSubgroup();

		gn = (unsigned int)(NUM_FACTORS + (1<<gen_no)-1);
		gm = (unsigned int)((gn + WORDSIZE -1) / WORDSIZE);
		gx = NULL;
		DYNALLOC2(graph,gx,gx_sz,gn,gm,"malloc");

		nauty_check(WORDSIZE,gm,gn,NAUTYVERSIONID);

		// create the graph gx
		// first the empty graph
		for (i=0; i < (int) gn; i++)
		{
			gv = GRAPHROW(gx,i,gm);
			EMPTYSET(gv,gm);
		}

		tmp = (int *) ALLOCS(NUM_FACTORS,sizeof(int));

		// fill the graph with words from the contrast subgroup
		for (i=1,j=0; i<(int)csg->size() && j<(int)(gn-NUM_FACTORS); i++)
			if ((*csg)[i])
			{
				dec2basel(i,LEVELS,tmp,NUM_FACTORS);
				gv = GRAPHROW(gx,j+NUM_FACTORS,gm);
				for (j2=0; j2<(int)NUM_FACTORS; j2++)
					if (tmp[j2] > 0)
					{
						ADDELEMENT(gv,j2);
						gv2 = GRAPHROW(gx,j2,gm);
						ADDELEMENT(gv2,j+NUM_FACTORS);
					}

				++j;
			}
		// graph ready!

		FREES(tmp);
	}
#else
	{
		int *tmp;
		vector<bool> *csg;
		csg = dx.getContrastSubgroup();

		gn = (unsigned int)(NUM_FACTORS + (1<<gen_no)-1);
		gm = (unsigned int)((gn + WORDSIZE -1) / WORDSIZE);
		gx = NULL;
		DYNALLOC2(graph,gx,gx_sz,gn,gm,"malloc");

		nauty_check(WORDSIZE,gm,gn,NAUTYVERSIONID);

		// create the graph gx
		// first the empty graph
		for (i=0; i < (int) gn; i++)
		{
			gv = GRAPHROW(gx,i,gm);
			EMPTYSET(gv,gm);
		}

		tmp = (int *) ALLOCS(NUM_FACTORS,sizeof(int));

		// fill the graph with words from the contrast subgroup
		for (i=1,j=0; i<(int)csg->size() && j<(int)(gn-NUM_FACTORS); i++)
			if ((*csg)[i])
			{
				dec2basel(i,LEVELS,tmp,NUM_FACTORS);
				gv = GRAPHROW(gx,j+NUM_FACTORS,gm);
				for (j2=0; j2<(int)NUM_FACTORS; j2++)
					if (tmp[j2] > 0)
					{
						ADDELEMENT(gv,j2);
						gv2 = GRAPHROW(gx,j2,gm);
						ADDELEMENT(gv2,j+NUM_FACTORS);
					}

				++j;
			}
		// graph ready!

		FREES(tmp);
	}
#endif
//	writegraph(gx,gn,gm);

	// now set up the partitions for nauty call
	graph *cgx;
	size_t cgx_sz = 0;
	{
		int *lab, *ptn, *orbits;
		statsblk stats;
		static DEFAULTOPTIONS_GRAPH(options);
		setword *workspace;
		size_t workspace_sz=0;

		lab = (int *) ALLOCS(gn,sizeof(int));
		ptn = (int *) ALLOCS(gn,sizeof(int));
		orbits = (int *) ALLOCS(gn,sizeof(int));

		workspace = NULL;
		DYNALLOC1(setword,workspace,workspace_sz,50*gm,"malloc");

		for (i=0; i<(int)gn; i++)
		{
			lab[i] = i;
			ptn[i] = 1;
		}
		ptn[NUM_FACTORS-1] = 0; // the factors make up one group
		ptn[gn-1] = 0; // all the words in the CS-group form one group

		options.writemarkers = FALSE;
		options.writeautoms = FALSE;
		options.getcanon = TRUE; // cause i need the canonical graph in cgx
		options.defaultptn = FALSE;
		options.userautomproc = userautomproc; // cause i want to get the aut group to reduce candidate xs when growing design

		cgx = NULL;
		DYNALLOC2(graph,cgx,cgx_sz,gn,gm,"malloc");

		d_ptr_nauty = &dx;
		nauty(gx,lab,ptn,NULL,orbits,&options,&stats,workspace,(int)workspace_sz,gm,gn,cgx);

#ifdef GET_STATS
		++NUM_CG_COMPUTATIONS;
#endif


		FREES(lab);
		FREES(ptn);
		FREES(orbits);
		DYNFREE(workspace,workspace_sz);
	}
	d_ptr_nauty = NULL;

//	writegraph(cgx,gn,gm);
	DYNFREE(gx,gx_sz);

	// convert cgx from graph* to vec<bit_vec> format
	vector < vector<bool> > cg(gn, vector<bool>(gn,false));

	for (i=0; i<(int)gn; i++)
	{
//		(cg[i])(gn,false);
		gv = GRAPHROW(cgx,i,gm);
		for (j=-1; (j=nextelement(gv,gm,j)) >=0;)
			cg[i][j] = true;
	}

//	writegraph(cg,gn);

	DYNFREE(cgx,cgx_sz);

	dx.setCanonicalGraph(cg);
	return;

}



static bool 
#ifdef USE_BAGS
isUniqueCanonicalGraph(const vector< vector<bool> > & cgx, vector<FF2Design> & D_new, vector<int> & wlp_bag, int & xwlp_bag)
#else
isUniqueCanonicalGraph(const vector< vector<bool> > & cgx, vector<FF2Design> & D_new)
#endif
{
	if (D_new.empty())
		return true;

	register int i, j=0;

	for (; j < (int) D_new.size(); j++)
	{
#ifdef USE_BAGS
		if (xwlp_bag!=wlp_bag[j])
		continue;
#endif
		// if the canonical graph of *itr1 was not computed before then
		// compute it now
		if (D_new[j].getCanonicalGraph()->empty())
			computeCanonicalGraph(D_new[j],(int)D_new[j].getCSGenerators()->size());

#ifdef GET_STATS
		++NUM_CG_COMPARISONS;
#endif

		// compare the canonical graphs
		for (i=0; i<(int)cgx.size(); i++)
		{
			vector< vector<bool> > *cg;
			cg = D_new[j].getCanonicalGraph();
			// if not all graph rows are same then compare with next design in list
			if (!((*cg)[i]==cgx[i]))
				break;
		}
		// if all rows were same then the designs are isomorphic
		if (i==cgx.size())
		{
			return false;
		}
	}

	return true;

}




static UPROC 
userautomproc(int count, permutation *p, int *orbits, int numorbits, int stabvertex, int n)
{
	vector<permutation> pvec(n);
	register int i;

	for (i=0; i<n; i++)
		pvec[i] = p[i];

	d_ptr_nauty->addAutGroupGenerator(pvec);

	return;
}




static void 
#ifdef USE_BAGS
outputDesigns(vector<FF2Design> & D_current, const int & gen_no, const vector<int> & wlp_bag)
#else
outputDesigns(vector<FF2Design> & D_current, const int & gen_no)
#endif
{
	register int i,j, k=0;
	int dcount=0;
	int *tmp;
	tmp = (int *) ALLOCS(MAX_FACTORS-FRAC,sizeof(int));

	fout << NUM_FACTORS << "-" << gen_no << ". # Designs = " << D_current.size() << endl;

	for (; k < (int)D_current.size(); k++)
	{
		fout << NUM_FACTORS << "-" << gen_no << "." << ++dcount << ": ";

		//print the generators
		vector<int> *dgens;
		dgens = D_current[k].getCSGenerators();

		for (i=0; i<gen_no; i++)
		{
			dec2basel((*dgens)[i],LEVELS,tmp,MAX_FACTORS-FRAC);
			fout << (char)(65+MAX_FACTORS-FRAC+i) <<"=";
			for (j=0; j<(int)(MAX_FACTORS-FRAC); j++)
				if (tmp[j]>0) fout << (char)(65+j);
			fout << " ";
		}

		//print the word length pattern
		fout <<"\t";
		
		vector<int> *dwlp;
#ifdef USE_BAGS
		dwlp = D_current[wlp_bag[k]].getWordLengthPattern();
#else
		dwlp = D_current[k].getWordLengthPattern();
#endif
		fout << "[";
		if ((int)(NUM_FACTORS+1-RES)>=0) fout << (*dwlp)[0];
		for (i=1; i<(int)(NUM_FACTORS+1-RES); i++)
			fout <<"," << (*dwlp)[i];
		fout <<"]";

		fout << endl;
	}
	fout << endl;

	FREES(tmp);

#ifdef GET_STATS
	fstats << (1<<(MAX_FACTORS-FRAC)) << ",";
	fstats << NUM_FACTORS << "," << gen_no << "," << RES << ",";
	fstats << D_current.size() << ",";

	fstats << (double) INC_TIME/CLOCKS_PER_SEC << "," << (double) (END_TIME-START_TIME)/CLOCKS_PER_SEC << ",";

	fstats << NUM_WLP_COMPUTATIONS << "," << NUM_CG_COMPUTATIONS << ",";

	fstats << NUM_WLP_COMPARISONS << "," << NUM_CG_COMPARISONS;

	fstats << endl;
	
#endif


	return;
}

static void 
freeSomeMemory(vector<FF2Design> & Dvec)
{
	vector<FF2Design>::iterator itr1 = Dvec.begin();
	vector<FF2Design>::iterator itr2 = Dvec.end();

	for (; itr1!= itr2; itr1++)
		itr1->freeSomeMemory();

	return;
}


static void
writegraph(const graph *g, const unsigned int & n, const unsigned int & m)
/* write graph g (n vertices) to file f in nauty format (my way!!)*/
{
	register unsigned int i;
	static unsigned int gcount=0;
	set *gv;

	fout << "Graph " << ++gcount << ": " << n << " vertices" << endl;
	for (i=0; i<n; i++)
	{
		gv = GRAPHROW(g,i,m);
		writeset(gv,n);
	}
	fout <<endl;

	return;
}

static void
writegraph(const vector <vector<bool> > & g, const unsigned int & n)
/* write graph g (n vertices) to file f in nauty format (my way!!)*/
{
	register unsigned int i, j;
	static unsigned int gcount=0;

	fout << "Graph " << ++gcount << ": " << n << " vertices" << endl;
	for (i=0; i<n; i++)
	{
		for (j=0; j<n; j++)
			if (g[i][j]) fout << "1 ";
			else fout << "0 ";
		fout << endl;
	}
	fout <<endl;

	return;
}

static void
writeset(const set *s, const unsigned int & slen)
{
	register unsigned int i=0;
	for (i=0; i<slen; i++)
		fout << ISELEMENT(s,i) << " ";
	fout << endl;
	return;
}


static void 
reduceCandidateGensByWordLength(vector<bool> & gens)
{
	register int i;
	int *tmp;
	tmp = new int[MAX_FACTORS-FRAC];

	for (i=0; i < (int)(MAX_FACTORS-FRAC); i++) tmp[i] = 0;

	for (i=0; i < (int)gens.size(); i++)
	{
		dec2basel(i,LEVELS,tmp,MAX_FACTORS-FRAC);
		if (vpopcount(tmp,MAX_FACTORS-FRAC) < (int) RES-1)
			gens[i] = false;
	}
	delete[] tmp;

	return;
}
