library(KernSmooth)
library(caTools)

data.process<-function(data.TR,data.TS,c.add,rho.mean,rho.sd){
	n.TR<-dim(data.TR)[1]
	n.TS<-dim(data.TS)[1]
	c.par<-c(1:(dim(data.TR)[2]-1))	# except for the y column
	n.par<-length(c.par)
	c.TR<-dim(data.TR)[2]
	c.TS<-dim(data.TS)[2]

	Y.TR<-as.matrix(data.TR[,c.TR])
	Y.TS<-as.matrix(data.TS[,c.TS])
	V.TR<-data.TR[,1]		# wind speed
	D.TR<-data.TR[,2]		# wind direction
	A.TR<-data.TR[,3]		# air density
	V.TS<-data.TS[,1]		# wind speed
	D.TS<-data.TS[,2]		# wind direction
	A.TS<-data.TS[,3]		# air density

	# variable adjustment
	V_adj.TR<-V.TR*(((A.TR*rho.sd+rho.mean)/rho.mean)^(1/3))	# rho/rho0?
	V_adj.TS<-V.TS*(((A.TS*rho.sd+rho.mean)/rho.mean)^(1/3))
	VD.TR<-cbind(V.TR*sin(D.TR/360*2*pi),V.TR*cos(D.TR/360*2*pi))
	VD.TS<-cbind(V.TS*sin(D.TS/360*2*pi),V.TS*cos(D.TS/360*2*pi))
	VD_adj.TR<-cbind(V_adj.TR*sin(D.TR/360*2*pi),V_adj.TR*cos(D.TR/360*2*pi))
	VD_adj.TS<-cbind(V_adj.TS*sin(D.TS/360*2*pi),V_adj.TS*cos(D.TS/360*2*pi))

	# explanatory variables
	X.TR<-as.matrix(data.TR[,1:(c.TR-1)])[,c.par]
	X.TS<-as.matrix(data.TS[,1:(c.TS-1)])[,c.par]
	X_VD.TR<-X.TR
	X_VD.TS<-X.TS
	X.TR[,1:2]<-VD.TR
	X.TS[,1:2]<-VD.TS

	list(X.TR,X.TS,Y.TR,Y.TS,V_adj.TR,V_adj.TS,VD_adj.TR,VD_adj.TS,
		X_VD.TR,X_VD.TS)
}

### Bandwidth selection algorithm1 (DPI)
bandwidth.sim<-function(X.TR,Y.TR){
	X.TR<-as.matrix(X.TR)
	q.TR<-dim(X.TR)[2]
	h.sim<-matrix(NA,1,q.TR)

	for(i in 1:q.TR){
		h.sim[1,i]<-dpill(X.TR[,i],Y.TR)
	}

	return(h.sim)
}

### Self calibration
calibration<-function(X.TR,X.TS,Y.TR,h.sim){
	X.TR<-as.matrix(X.TR)
	X.TS<-as.matrix(X.TS)
	n.TR<-dim(X.TR)[1]
	n.TS<-dim(X.TS)[1]

	id.Xcal<-rep(0,n.TS)
	for(i in 1:n.TS){
		distance<-mahalanobis(X.TR[,1:4],X.TS[i,1:4],
			cov=diag(c(h.sim[1:4])))
		id.Xcal[i]<-which(distance==min(distance))
	}

	X.TR_new<-X.TR
	Y.TR_new<-Y.TR
	X.TS_new<-X.TR[id.Xcal,]
	Y.TS_new<-Y.TR[id.Xcal,]

	return(list(X.TR_new,X.TS_new,Y.TR_new,Y.TS_new))
}

### Estimation by binning method
bin.est<-function(V.TR,V.TS,Y.TR,cutIn,cutOut,bin.size){
	V.TR[which(V.TR>cutOut)]<-cutOut
	V.TS[which(V.TS>cutOut)]<-cutOut

	n.bin<-ceiling((cutOut-cutIn)/bin.size)
	id.TR<-trunc((V.TR-cutIn)/bin.size)+1
	id.TS<-trunc((V.TS-cutIn)/bin.size)+1

	est.bin<-rep(0,n.bin)
	for(i in 1:n.bin){
		ref<-which(id.TR==i)
		if(length(ref)>0){
			est.bin[i]<-mean(Y.TR[ref])
		}
		else{
			est.bin[i]<-NA
		}
	}
	est<-est.bin[id.TS]

	return(est)
}

### Get the difference between vector x and point x_i
get.diff<-function(X.TR,x_i.TS){
	X.TR<-as.matrix(X.TR)
	n.TR<-dim(X.TR)[1]
	q.TR<-dim(X.TR)[2]
	x_i.TS<-matrix(x_i.TS,1,q.TR)
	oneV<-matrix(1,n.TR,1)
	diff<-X.TR-(oneV%*%x_i.TS)
	return(diff)
}

### Nadaraya-Watson (NW) regression estimator (point estimator)
NW.est<-function(X.TR,X.TS,Y.TR,h){
	X.TR<-as.matrix(X.TR)
	X.TS<-as.matrix(X.TS)
	q.TR<-dim(X.TR)[2]
	n.TR<-dim(X.TR)[1]
	n.TS<-dim(X.TS)[1]

	est.nw<-matrix(NA,n.TS,1)
	for(i in 1:n.TS){
		diff<-get.diff(X.TR,X.TS[i,])

		if(q.TR==1){	# Univariate kernel regression
			kappa<-dnorm(diff/c(h))/c(h)
			yhat<-sum(Y.TR*kappa/sum(kappa))
		}
		else if(q.TR==2){	# Bivariate kernel regression (Jeon&Taylor)
			dir<-diff[,2]
			id.adj<-which(abs(dir)<180)
			diff[-id.adj,2]<-sign(dir[-id.adj])*(360-abs(dir[-id.adj]))
			kappa.j<-matrix(NA,n.TR,2)
			for(j in 1:2){
				kappa.j[,j]<-dnorm(diff[,j]/h[j])/h[j]
			}
			kappa<-kappa.j[,1]*kappa.j[,2]
			yhat<-sum(Y.TR*kappa/sum(kappa))
		}
		else if(q.TR>2){	# Additive multivariate kernel method
			dir<-diff[,2]
			id.adj<-which(abs(dir)<180)
			diff[-id.adj,2]<-sign(dir[-id.adj])*(360-abs(dir[-id.adj]))
			kappa<-rep(0,n.TR)
			yhat<-rep(NA,(q.TR-2))
			for(j in 3:q.TR){
				kappa.v<-dnorm(diff[,1]/h[1])/h[1]
				kappa.d<-dnorm(diff[,2]/h[2])/h[2]
				kappa.j<-dnorm(diff[,j]/h[j])/h[j]
				kappa<-(kappa.v*kappa.d*kappa.j)
				yhat[j-2]<-sum(Y.TR*kappa/sum(kappa))
			}

		}
		est.nw[i]<-mean(yhat)
	}
	
	return(est.nw)
}

### Conditional Kernel Density (CKD) estimation (density estimation)
CKD.est<-function(X.TR,X.TS,Y.TR,Y.TS,Y,h.x,h.y){
	X.TR<-as.matrix(X.TR)
	X.TS<-as.matrix(X.TS)
	Y.TR<-as.matrix(Y.TR)
	Y.TS<-as.matrix(Y.TS)

	q.TR<-dim(X.TR)[2]
	n.TR<-dim(X.TR)[1]
	n.TS<-dim(X.TS)[1]

	Y<-as.matrix(Y)
	h.x<-as.matrix(h.x)

	est.nw<-matrix(NA,n.TS,1)
	distY<-matrix(NA,n.TS,length(Y))
	distY.TS<-matrix(NA,n.TS,1)

	for(i in 1:n.TS){
		diff<-get.diff(X.TR,X.TS[i,])

		if(q.TR==1){	# Univariate kernel regression
			kappa<-dnorm(diff/c(h.x))/c(h.x)
			est.nw[i]<-sum(Y.TR*kappa/sum(kappa))
			
			for(j in 1:length(Y)){
				diffY<-get.diff(Y.TR,Y[j])
				kappaY<-dnorm(diffY/h.y)/h.y
				distY[i,j]<-sum(kappa*kappaY)/sum(kappa)
			}
			
			diffY.TS<-get.diff(Y.TR,Y.TS[i])
			kappaY.TS<-dnorm(diffY.TS/h.y)/h.y
			distY.TS[i]<-sum(kappa*kappaY.TS)/sum(kappa)
		}
		else if(q.TR==2){	# Bivariate kernel regression (Jeon&Taylor)
			dir<-diff[,2]
			id.adj<-which(abs(dir)<180)
			diff[-id.adj,2]<-sign(dir[-id.adj])*(360-abs(dir[-id.adj]))

			kappa.j<-matrix(NA,n.TR,2)
			for(j in 1:2){
				kappa.j[,j]<-dnorm(diff[,j]/h.x[j])/h.x[j]
			}
			kappa<-kappa.j[,1]*kappa.j[,2]
			est.nw[i]<-sum(Y.TR*kappa/sum(kappa))

			for(j in 1:length(Y)){
				diffY<-get.diff(Y.TR,Y[j])
				kappaY<-dnorm(diffY/h.y)/h.y
				distY[i,j]<-sum(kappa*kappaY)/sum(kappa)				
			}

			diffY.TS<-get.diff(Y.TR,Y.TS[i])
			kappaY.TS<-dnorm(diffY.TS/h.y)/h.y
			distY.TS[i]<-sum(kappa*kappaY.TS)/sum(kappa)			
		}
		else if(q.TR>2){	# Additive multivariate kernel method
			dir<-diff[,2]
			id.adj<-which(abs(dir)<180)
			diff[-id.adj,2]<-sign(dir[-id.adj])*(360-abs(dir[-id.adj]))

			weight<-matrix(NA,n.TR,(q.TR-2))
			for(j in 3:q.TR){
				kappa.v<-dnorm(diff[,1]/h.x[1])/h.x[1]
				kappa.d<-dnorm(diff[,2]/h.x[2])/h.x[2]
				kappa.j<-dnorm(diff[,j]/h.x[j])/h.x[j]
				weight[,(j-2)]<-(kappa.v*kappa.d*kappa.j)/sum(kappa.v*kappa.d*kappa.j)
			}			
			est.nw[i]<-mean(apply(weight*Y.TR[,1],2,sum))

			kappaY<-matrix(NA,n.TR,length(Y))
			for(j in 1:length(Y)){
				diffY<-get.diff(Y.TR,Y[j])
				kappaY[,j]<-dnorm(diffY/h.y)/h.y
				distY[i,j]<-mean(apply(weight*kappaY[,j],2,sum))				
			}

			diffY.TS<-get.diff(Y.TR,Y.TS[i])
			kappaY.TS<-dnorm(diffY.TS/h.y)/h.y
			distY.TS[i]<-mean(apply(weight*kappaY.TS[,1],2,sum))
		}		
	}

	return(list(est.nw,distY,distY.TS))
}

IMSE<-function(Y,distY,Y.TS,distY.TS){
	n.TS<-length(Y.TS)
	I1.est<-rep(NA,n.TS)
	I2.est<-rep(NA,n.TS)

	for(i in 1:n.TS){
		I1.est[i]<-trapz(Y,(distY.TS[i,]/trapz(Y,distY.TS[i,]))^2)
		I2.est[i]<-distY.TS[i]/trapz(Y,distY.TS[i,])
	}

	est<-mean(I1.est)-2*mean(I2.est)
	return(est)
}

CRPS<-function(Y,distY.i,y.i){
	cdf<-rep(0,length(Y))
	cdf[1]<-distY.i[1]	# trapz fn not working for a single value
	for(i in 2:length(Y)){
		cdf[i]<-trapz(Y[1:i],distY.i[1:i])
	}

	if(y.i<0.0001){
		y.i<-0.0001
	}
	id.obs<-max(which(Y<y.i))
	R1<-cdf[1:id.obs]^2
	R2<-(1-cdf[(id.obs+1):length(Y)])^2
	value<-trapz(Y,c(R1,R2))

	return(value)
}
