#######################################################################
#                                                                     #
# R Script for computation of posterior predictions of the manuscript #
# "Exploring cocoa bean fermentation processes by kinetic modelling"  #
#                                                                     #
# The following script contains data from 28 fermentation trials. For #
# a more detailed description of datasets, see Supplementary Material #
# Section 1, and Supplementary Table S1.                              #
#                                                                     #
# This code works in prompt shell of Linux OS. For its execution it   #
# needs seven arguments:                                              #
# d = dataset (integer from 1 to 28):                                 #
#     1:  ghhp1           15: ghhp7                                   #
#     2:  dowb1           16: brwb3                                   #
#     3:  cipb1           17: brwb4                                   #
#     4:  brwb1           18: brst2                                   #
#     5:  brwb2           19: brwb5                                   #
#     6:  ecpt1           20: brwb6                                   #
#     7:  ecpt2           21: brwb7                                   #
#     8:  ecwb1           22: mywb3                                   #
#     9:  ecwb2           23: brwb8                                   #
#     10: brpb1           24: hnwb1                                   #
#     11: brst1           25: ecpb1                                   #
#     12: mywb1           26: niwb1                                   #
#     13: mywb2           27: niwb2                                   #
#     14: ghhp6           28: ecpb2                                   #
#                                                                     #
# mi = model iteration (integer), refers to model iteration code, for #
#      instance, MI(1,2,3), here is called as 123                     #
#                                                                     #
# The following are Stan parameters. For further understanding, see   #
# reference manual for Stan. Between parenthesis are noted the values #
# used in our paper                                                   #
# ad = adapt_delta parameter (0.995 to 0.999)                         #
#                                                                     #
# st = step_size parameter (0.1)                                      #
#                                                                     #
# mt = max_treedepth parameter (10)                                   #
#                                                                     #
# s  = seed parameter (141085)                                        #
#                                                                     #
# ni = number of iterations (3000)                                    #
#                                                                     #
# Hence, for running this script you must follow the following form:  #
#                                                                     #
# $ Rscript post_sims1.R d mi ad st mt s ni                           #
#                                                                     #
# Once the script is done running, an .Rsave file is saved with the   #
# dataset coding as name containing the Stan output.                  #
#                                                                     #
# Code written by Mauricio Moreno-Zambrano                            #
# 2021                                                                #
#                                                                     #
#######################################################################


setwd('/home') # directory name, change accordingly if needed
rm(list = ls())
args = commandArgs(TRUE)

library(rstan)

rstan_options(auto_write = TRUE)

# datasets ordered as mentioned above

if (args[1]==1){
  T = 16
  x0 = c(51.963,57.741,0,0,0,0.029180401,0.007868827,3.36634E-06)
  t0 = 0
  ts = c(6,12,18,24,30,36,42,48,54,60,66,72,84,96,120,144)
  x = structure(c(43.884,24.25,26.098,11.871,4.607,1.714,4.821,5.409,3.701,3.696,1.544,4.428,1.086,2.337,2.837,3.486,
                  49.588,35.584,33.876,19.871,8.607,2.825,4.821,7.409,4.812,7.548,1.544,5.539,1.086,2.337,2.837,3.486,
                  2.97,9.081,9.62,6.067,11.211,16.213,10.215,20.106,22.492,5.637,5.636,13.85,4.782,9.3,4.267,7.448,
                  0.267,0.801,1.256,1.745,5.944,3.246,7.787,7.991,6.943,8.593,8.296,6.85,8.021,7.667,7.209,8.117,
                  0.152,1.616,0.979,4.033,5.121,2.759,3.541,4.278,4.141,6.139,3.219,5.319,4.328,3.621,4.208,5.93,
                  0.026187332,0.955193281,0.238831309,0.045299276,0.01021154,0.06608323,0.008262115,0.000554742,0.000267973,0.006013001,0.002203389,0.000375916,0.000889388,0.000948618,0.000110177,0.000105461,
                  0.067128975,0.07959944,0.248834167,0.20179482,0.416783016,0.401707567,0.566121975,0.386286929,0.372314554,0.199484893,0.167459586,0.161402409,0.14484717,0.008748025,0.006575216,0.00181514,
                  9.82105E-06,1.70278E-05,2.16351E-05,5.49741E-05,0.00010572,0.000177484,0.000259512,0.000681017,0.000364887,0.00118075,0.001914953,0.000676329,0.001127608,0.000237224,6.32642E-05,2.64337E-05),
                .Dim=c(16,8))
  lbl<-'ghhp1_'
}

if (args[1]==2){
  T=6
  x0 = c(56.909,88.795,0,0,0,0.022808213,1.25E-08,0.00028)
  t0 = 0
  ts = c(24,48,72,96,120,144)
  x = structure(c(50.266,20.771,15.19,13.95,1.284,1.815,
                  70.992,40.877,28.034,29.008,12.621,12.09,
                  1.003,9.702,0.972,0,0,0,
                  1.178,0.486,1.551,1.411,2.25,0,
                  0,6.488,22.472,11.306,5.842,5.375,
                  0.957395229,0.120528918,0.022084688,0.007604861,0.00000015,0.00000015,
                  0.001096251,0.092877392,0.053692053,0.011295618,0.006381312,0.001621474,
                  0.017871378,0.045098078,0.020613799,0.002530219,0.000982105,0.000156012),
                .Dim=c(6,8))
  lbl<-'dowb1_'
}

if (args[1]==3){
  T=6
  x0 = c(95.6,182,0,0,0,4.30617E-05,5.25908E-06,2.8E-09)
  t0 = 0
  ts = c(24,48,72,96,120,144)
  x = structure(c(44.5,30.7,7.77,0.36,0.51,0,
                  70.2,9.11,12.4,1.11,1.76,0,
                  2.3,8.99,16.6,14.3,5.95,3.68,
                  3.08,0.35,12.4,16.3,3.52,3.74,
                  4.34,0.73,6.89,38.5,18.8,10.6,
                  0.015034579,0.15103975,0.070484116,0.071629391,0.016869075,0.000673118,
                  6.8851E-05,0.470879749,0.837355762,0.560931737,0.16784562,0.04633509,
                  7.7653E-06,0.085734976,0.131267747,0.147624361,0.056644537,0.007501671),
                .Dim=c(6,8))
  lbl<-'cipb1_'
}

if (args[1]==4){
  T = 13
  x0 = c(55.482, 49.669, 0, 0, 1.379, 0.175424909,0.003047264,2.86522E-06)
  t0 = 0
  ts = c(6,12,24,30,36,48,54,60,72,84,96,120,144)
  
  x = structure(c(26.802,16.311,13.706,13.248,7.54,1.278,2.694,3.08,0.757,2.465,2.673,3.183,7.631,
                  31.959,18.937,19.238,15.217,10.351,0.809,1.007,1.205,0.757,1.996,2.673,3.183,2.287,
                  0.201,2.324,2.46,1.996,3.173,4.025,6.033,5.685,3.605,1.617,0.576,0.964,1.56,
                  0.148,0.342,0.548,1.033,0.658,4.873,3.194,2.694,2.72,1.928,2.482,1.883,1.63,
                  0.963,1.216,1.467,1.996,1.211,3.333,6.033,4.323,8.685,5.658,11.935,11.354,11.511,
                  0.119149235,0.212369067,0.068721283,0.045508368,0.052250597,0.035488795,0.020468747,0.03750518,0.004509114,0.00048539,0.000190586,2.91133E-06,1.45912E-06,
                  0.006575216,0.02528774,0.37403308,0.364678377,0.36383964,0.073945204,0.019182712,0.027410062,0.001068833,0.00101137,0.023873166,0.00458047,0.000150977,
                  7.71184E-06,7.09836E-06,1.0427E-05,7.90966E-06,1.10194E-05,6.05561E-05,0.000127104,0.000700097,0.000130665,0.000127104,0.005347589,0.008514478,3.38188E-05),
                .Dim=c(13,8))
  lbl<-'brwb1_'
}

if (args[1]==5){
  T = 13
  x0 = c(42.936,67.249,0.318,0,0.942,0.004754351,0.000221264,3.86508E-06)
  t0 = 0
  ts = c(6,12,24,30,36,48,54,60,72,84,96,120,144)
  
  x = structure(c(23.452,8.264,4.85,1.27,1.62,2.32,0.294,1.467,1.892,5.975,7.863,6.521,7.19,
                  21.349,14.478,10.974,2.641,0.889,4.057,1.756,1.467,3.812,8.351,9.416,9.081,9.109,
                  0.386,2.211,4.496,4.912,6.714,4.307,7.057,4.097,1.044,1.642,0.9,0.341,0.29,
                  0.165,0.679,1.75,0.393,0.823,0.992,1.161,0.968,1.609,4.123,2.476,2.565,1.642,
                  1.079,1.194,2.532,1.329,2.183,1.718,2.272,2.733,6.568,11.767,9.083,18.625,17.187,
                  0.069998907,0.243271515,0.037852212,0.022340416,0.038823194,0.015491421,0.055474227,0.000444725,0.001542024,7.90845E-05,0.00016676,0.000144574,9.48618E-06,
                  0.002753658,0.026786133,0.016669018,0.051749959,0.125866459,0.54564479,0.699697002,0.05151219,0.11800761,0.013928682,0.024149604,0.002923547,0.001451811,
                  8.37834E-06,1.41957E-05,2.74889E-05,4.53062E-05,9.06062E-06,3.61541E-05,0.002723692,0.001033137,0.001436012,6.57897E-05,0.026987213,0.00468984,9.33594E-06),
                .Dim=c(13,8))
  lbl<-'brwb2_'
}

if (args[1]==6){
  T = 14
  x0 = c(45.93,53.933,0.321,0,1.048,0.000944259,0.0008628,9.29304E-06)
  t0 = 0
  ts = c(6,12,18,24,30,36,42,48,54,60,66,72,84,96)
  
  x = structure(c(49.119,47.058,36.219,18.238,6.712,14.375,4.655,3.541,7.934,3.378,0,0.805,2.018,1.769,
                  43.353,36.559,25.893,30.888,6.97,14.805,4.655,3.541,7.934,10.004,2.263,2.87,0.813,2.371,
                  0.575,0.871,2.517,3.562,4.073,10.187,12.537,12.406,2.632,6.073,5.044,5.256,0.932,0.757,
                  0.276,0.744,1.17,5.829,3.346,2.361,5.224,6.034,3.937,4.319,2.948,1.792,2.515,1.826,
                  1.046,1.642,2.217,2.921,2.021,1.334,1.717,1.116,2.312,2.523,2.222,2.048,3.413,3.922,
                  0.001577943,0.021042206,0.185392115,0.030980702,0.051771561,0.153847789,0.342839821,0.159254334,0.9270246,0.7657575,1.05460848,1.094186265,0.046141452,0.408405196,
                  0.002264175,0.212769814,0.131798362,0.489677346,0.55707031,0.084705188,0.004334211,0.032501995,0.156282379,0.061788836,0.047963406,0.093952862,0.007077991,0.038628693,
                  0.00017067,7.45003E-05,8.2065E-05,4.33669E-05,0.000189303,0.00096863,0.00042185,0.000436675,0.008038186,0.006639846,0.01734435,0.016832865,0.001268113,0.007873322),
                .Dim=c(14,8))
  lbl <- 'ecpt1_'
}

if (args[1]==7){
  T = 13
  x0 = c(35.801,36.408,0.554,0,2.987,7.38059E-05,8.37356E-05,4.17021E-07)
  t0 = 0
  ts = c(6,12,18,24,30,36,42,48,54,60,72,84,96)
  
  x = structure(c(42.651,27.468,18.88,7.776,10.549,9.333,7.429,2.737,2.474,1.952,1.167,0,0,
                  38.921,37.444,19.573,6.475,9.595,6.471,11.499,3.777,3.515,2.992,4.549,0.815,2.806,
                  0.639,1.332,7.369,4.76,4.019,8.014,7.099,9.617,3.576,1.879,0.224,1.565,0.302,
                  0.378,0,1.634,0.718,0.934,1.193,0.495,1.145,1.316,0.793,0.224,1.262,0.823,
                  3.81,3.2,5.37,3.717,3.628,4.19,4.797,9.617,11.613,15.869,17.298,17.337,7.905,
                  0.011697452,0.654773748,0.338915366,0.871146626,0.218318862,0.046783344,0.239933704,0.033043897,0.048316032,0.060130008,0.001425907,0.240486809,0.034283982,
                  0.000608009,0.169398677,0.174948415,0.043843984,0.007531995,0.048073973,0.06207404,0.046763824,0.099519919,0.154849573,0.031110716,0.299854115,0.082396737,
                  0.000218352,0.002307587,0.002168493,0.003476626,0.000929304,0.000239971,0.001122427,0.000700097,0.003484641,0.006747735,0.000362375,0.019686025,0.005783064),
                .Dim=c(13,8))
  lbl <- 'ecpt2_'
}

if (args[1]==8){
  T = 14
  x0 = c(42.083,53.899,0.889,0,0.889,0.005047674,0.000990627,3.10569E-05)
  t0 = 0
  ts = c(6,12,18,24,30,36,42,48,54,60,66,72,84,96)
  
  x = structure(c(40.032,30.959,22.229,28.397,26.687,1.175,1.607,5.976,0.757,3.072,1.791,0.681,0.773,1.121,
                  43.799,36.867,26.082,30.623,29.513,3.401,4.432,12.484,3.839,8.209,9.754,2.393,1.971,2.749,
                  0.889,1.257,1.912,2.362,4.282,17.847,13.354,14.744,12.129,12.538,11.763,6.737,3.552,5.146,
                  0,0.604,0.645,2.688,1.055,3.261,0.729,2.527,1.71,3.631,5.062,3.387,2.04,1.796,
                  0.889,1.257,1.912,3.587,2.24,3.588,1.587,2.527,2.282,4.367,6.982,6.737,7.842,6.29,
                  0.002080134,0.01049763,0.103297844,0.058491298,0.172223043,0.841571964,0.189710452,0.269210044,0.02831987,0.025008708,0.003076743,0.009979097,1.52087E-06,6.87213E-05,
                  0.002192351,0.090138435,0.248834167,0.331825695,0.64700854,0.283080538,0.332590632,0.242610735,0.113477566,0.026357852,0.090972476,0.088289694,0.066513532,0.024940779,
                  0.000147285,0.002723692,0.000899825,3.02105E-05,0.001127608,0.000464684,6.09759E-05,2.85206E-05,0.000280645,0.000262517,9.25035E-05,0.000531078,6.87318E-06,7.34781E-06),
                .Dim=c(14,8))
  lbl <- 'ecwb1_'
} 

if (args[1]==9){
  T = 14
  x0 = c(36.674,32.884,0,0,3.345,0.000290463,0.000479634,1.22789E-05)
  t0 = 0
  ts = c(6,12,18,24,30,36,42,48,54,60,66,72,84,96)
  
  x = structure(c(14.472,16.459,8.373,1.213,5.758,0.956,2.554,6.846,1.792,4.988,12.314,0.69,2.286,5.229,
                  16.377,14.943,7.531,4.918,10.726,1.882,4.07,5.667,4.234,6.926,6.503,2.038,5.318,6.408,
                  0.199,0.616,0.909,2.609,4.021,6.633,12.517,7.798,8.339,4.488,4.492,3.127,1.972,1.605,
                  0,0.202,0.619,2.03,2.861,4.106,3.943,4.567,4.445,5.152,3.58,2.63,1.849,3.303,
                  2.726,2.936,3.601,5.302,5.263,7.378,8.209,11.69,10.492,11.779,13.232,17.418,20.447,24.097,
                  0.001233364,0.114048942,0.188404495,0.165993568,0.227557555,0.515336922,0.662355671,0.013870473,0.004614145,1.61098E-05,7.55251E-06,0.000339697,4.13134E-05,3.64831E-05,
                  0.0014385,0.13737573,0.200405674,0.375759538,0.498781128,0.64107673,0.439450551,0.200405674,0.257578739,0.017820095,0.053445361,0.035555764,0.022177369,0.001119206,
                  0.001368227,0.003405321,0.00075363,0.00080197,0.000853411,0.00096863,0.005459565,0.00096863,0.000228117,1.78714E-05,8.95691E-06,0.004967731,3.35087E-05,7.58854E-05),
                .Dim=c(14,8))
  lbl <- 'ecwb2_'
}

if (args[1]==10){
  T = 12
  x0 = c(40.008,35.371,0,2.021,0,0.002851617,0.001570037,2.8E-10)
  t0 = 0
  ts = c(12,24,36,48,60,72,84,96,108,120,132,144)
  
  x = structure(c(102.342,23.407,1.09,5.923,2.81,9.85,11.593,13.556,10.443,11.303,10.948,1.986,
                  95.058,24.952,10.14,11.331,6.121,6.98,6.516,13.556,15.189,7.992,6.866,1.324,
                  43.93,45.732,65.326,20.521,26.127,27.8,33.277,19.801,19.218,25.082,21.597,23.529,
                  23.545,22.156,31.425,20.922,18.4,16.24,15.313,12.533,9.445,6.819,6.306,6.153,
                  2.874,3.091,12.608,42.164,55.501,68.726,60.363,21.663,19.943,18.665,18.383,19.984,
                  0.548392187,0.088123403,0.002046875,0.013806744,0.004809404,0.000553466,2.61873E-05,0.000110431,0.000218319,0.000000015,0.000000015,0.000000015,
                  0.149592566,0.335668056,0.070943076,0.086279975,0.039986189,0.020319359,0.008768191,0.005739975,0.039710926,0.097478764,0.123853993,0.074115666,
                  0.000604168,0.000324458,0.00024955,4.76604E-05,9.73101E-06,7.83715E-06,1.13281E-06,8.51448E-07,2.8E-10,2.8E-10,2.8E-10,2.8E-10),
                .Dim=c(12,8))
  lbl<-'brpb1_'
} 

if (args[1]==11){
  T = 12
  x0 = c(35.371,30.269,0,2.021,0,0.001718269,0.000176161,2.8E-10)
  t0 = 0
  ts = c(12,24,36,48,60,72,84,96,108,120,132,144)
  
  x = structure(c(132.472,15.902,1.09,0.184,6.121,6.98,6.516,10.687,10.443,10.2,9.734,1.986,
                  122.539,21.199,1.09,5.923,2.81,2.565,5.192,8.038,5.587,4.902,3.113,3.532,
                  41.931,33.29,56.753,78.152,50.043,51.845,60.48,50.035,64.859,64.726,53.637,46.156,
                  8.124,8.794,14.742,18.734,15.98,9.571,14.437,17.039,20,23.476,18.843,19.206,
                  0,0,0,0.369,11.215,11.099,10.43,11.09,12.801,12.963,16.557,18.71,
                  0.242712006,0.732978539,0.055602108,0.016447173,0.005985374,0.015384779,0.000000015,0.000000015,0.000000015,0.000000015,0.000000015,0.000000015,
                  0.065299524,0.076897109,0.065299524,0.111406367,0.070292666,0.062648404,0.01729458,0.106637514,0.004455639,0.004569935,0.006164673,0.003801106,
                  3.76841E-05,8.26339E-05,0.000202376,0.000198225,2.33431E-05,4.56203E-07,3.6321E-07,7.67641E-07,2.8E-10,2.8E-10,2.8E-10,2.8E-10),
                .Dim=c(12,8))
  lbl<-'brst1_'
}

if (args[1]==12){
  T=4
  x0 = c(46.171,55.783,0.515,0,0,0.097744259,0.035229787,0.00085735)
  t0 = 0
  ts = c(24,48,72,96)
  x = structure(c(0,0,0,0,
                  0,0,0,0,
                  17.211,10.062,9.462,5.658,
                  4.528,5.676,5.646,5.615,
                  1.996,2.799,6.131,6.135,
                  0.439633987,0.028846376,0.000153494,3.11955E-05,
                  0.812662113,0.230626927,0.01644031,0.036383964,
                  0.001538714,0.001531645,0.001094355,0.000195956),
                .Dim=c(4,8))
  lbl<-'mywb1_'
}

if (args[1]==13){
  T=4
  x0 = c(28.406,29.475,0,0,0,0.004046609,0.014720075,0.000467905)
  t0 = 0
  ts = c(24,48,72,96)
  x = structure(c(7.613,0,0,0,
                  14.559,0,0,0,
                  13.384,12.146,11.621,11.879,
                  5.082,10.91,9.458,10.26,
                  3.107,7.686,5.992,4.09,
                  0.196377288,0.008832655,0.000105948,3.01364E-05,
                  0.040356177,0.032802732,0.009747876,0.00531998,
                  0.000616819,0.002133821,0.004843486,0.000191495),
                .Dim=c(4,8))
  lbl<-'mywb2_'
}

if (args[1]==14){
  T=4
  x0 = c(42.538,78.288,0,0,0,0.137749889,0.020697125,0.004636156)
  t0 = 0
  ts = c(24,48,96,144)
  x = structure(c(17.573,5.46,5.41,3.204,
                  48.77,20.433,8.446,5.734,
                  9.181,2.4,0,0,
                  3.462,2.585,0,0,
                  2.207,5.994,6.797,6.143,
                  0.018369243,0.03918242,0.002070576,9.55193E-06,
                  0.248261865,0.807067786,0.80892827,0.036133499,
                  0.001841442,0.051186806,0.01227886,0.018756769),
                .Dim=c(4,8))
  lbl<-'ghhp6_'
}

if (args[1]==15){
  T=4
  x0 = c(51.082,59.34,0.413,0,0,0.073976071,0.025171553,0.003413171)
  t0 = 0
  ts = c(24,48,72,96)
  x = structure(c(9.111,2.152,0,0,
                  36.04,12.027,11.713,14.809,
                  18.124,10.377,8.088,5.188,
                  2.846,2.595,2.621,2.474,
                  1.565,5.467,5.908,8.287,
                  1.208067662,0.001153696,0.000000015,0.000000015,
                  1.199250789,1.823517825,0.004874275,0.006529952,
                  0.013066463,0.00096418,0.000336634,0.001462709),
                .Dim=c(4,8))
  lbl<-'ghhp7_'
}

if (args[1]==16){
  T = 12
  x0 = c(146.589,162.153,0,0,0,0.001706441,0.001610312,5.07175E-05)
  t0 = 0
  ts = c(12,24,36,48,60,72,84,96,108,120,132,144)
  
  x = structure(c(49.233,26.151,38.968,23.108,23.34,27.285,24.629,33.114,23.237,13.565,13.798,13.617,
                  26.664,56.027,43.591,40.548,41.262,56.771,55.137,44.11,37.309,34.031,31.223,30.059,
                  63.627,24.995,26.887,36.165,26.28,12.701,34.656,18.482,18.079,15.479,14.577,13.475,
                  14.478,14.172,15.188,16.944,12.729,10.146,9.809,6.579,6.98,6.951,6.768,6.462,
                  0,16.799,11.709,14.24,14.076,22.226,23.217,25.593,26.276,31.27,32.107,36.177,
                  1.892741302,0.469992859,0.003476092,0.020705764,0.004966967,0.000604076,4.08405E-05,0.000636929,0.001445744,0.001145754,0.001343047,0.001318534,
                  0.102780331,0.093952862,0.08748025,0.124712508,0.06900968,0.01119206,0.008277706,0.016289585,0.008239674,0.07143483,0.015132477,0.117195251,
                  0.00084949,0.000401013,0.000335087,0.000169886,1.34945E-05,3.28971E-05,9.77593E-07,1.99599E-06,1.47285E-06,4.29693E-07,1.72647E-06,0),
                .Dim=c(12,8))
  lbl <- 'brwb3_'
}

if (args[1]==17){
  T = 12
  x0 = c(132.972,149.002,0,0,0,0.002773903,0.001509767,1.68717E-05)
  t0 = 0
  ts = c(12,24,36,48,60,72,84,96,108,120,132,144)
  
  x = structure(c(93.798,70.922,23.494,5.777,17.769,21.921,18.646,23.417,17.047,16.248,16.48,16.093,
                  98.053,39.823,2.964,52.994,46.664,53.014,51.849,50.215,49.756,44.834,48.601,47.202,
                  23.901,24.995,28.085,40.058,47.839,34.061,25.871,18.28,18.079,15.479,14.577,15.571,
                  10.847,13.68,22.05,19.374,12.083,12.054,11.194,10.609,9.565,8.613,8.214,7.754,
                  2.184,16.799,21.793,24.554,24.851,26.921,23.217,22.976,23.428,20.186,21.1,22.014,
                  0.540867965,0.088734245,0.002032784,0.013554742,0.004571842,0.000533447,2.47224E-05,0.000108415,0.00021582,0.001241913,0.00128852,0.001054608,
                  0.142859792,0.340337664,0.068062832,0.086081537,0.038806995,0.019902609,0.008509617,0.00522288,0.037317283,0.090346225,0.116120798,0.075319948,
                  0.000613985,0.000332015,0.00024387,4.54107E-05,9.44404E-06,7.45003E-06,1.11727E-06,8.83401E-07,0,0,0,0),
                .Dim=c(12,8))
  lbl <- 'brwb4_'
}

if (args[1]==18){
  T = 12
  x0 = c(32.082,36.517,0,0,0,0.000141283,1.33632E-05,2.8E-10)
  t0 = 0
  ts = c(12,24,36,48,60,72,84,96,108,120,132,144)
  
  x = structure(c(122.888,56.893,1.624,3.301,28.498,7.891,7.092,6.705,6.73,8.408,9.259,8.046,
                  136.565,59.55,2.964,2.74,4.628,19.667,11.223,12.172,14.531,13.601,11.262,10.333,
                  36.877,64.62,71.004,65.411,53.429,53.225,46.833,28.863,12.888,15.479,18.27,18.865,
                  12.201,19.464,23.773,21.313,22.914,19.592,20.147,13.441,15.042,14.367,9.753,8.277,
                  0,0,2.242,21.937,23.005,25.305,36.071,37.139,36.514,46.665,56.968,57.421,
                  0.071464648,0.308383589,0.011616927,0.002254713,0.00067157,0.001884045,3.63154E-06,5.70284E-06,9.20643E-06,0.000000015,0.000000015,0.000000015,
                  0.028438718,0.036551905,0.028308054,0.057399752,0.031326366,0.026909772,0.004964894,0.051749959,0.000916031,0.000891066,0.0012999,0.000641077,
                  2.8E-10,7.81912E-06,2.67398E-05,2.41634E-05,1.66785E-06,1.02367E-08,2.8E-10,2.8E-10,2.8E-10,2.8E-10,2.8E-10,2.8E-10),
                .Dim=c(12,8))
  lbl<-'brst2_'
}

if (args[1]==19){
  T = 6
  x0 = c(10.032,12.612,0,0,0,0.000319399,2.94546E-06,5.96212E-06)
  t0 = 0
  ts = c(24,48,72,96,120,144)
  
  x = structure(c(9.952,2.133,0,0,0,0,
                  10.89,2.858,0.358,0.881,0.855,0,
                  0.281,3.074,2.078,0.76,0.346,0.121,
                  0.33,0.847,0.124,0,0,0,
                  0.281,0.378,0.579,1.593,0.983,1.017,
                  0.006016359,0.000102172,5.67981E-05,0.000388449,1.56435E-05,6.12821E-06,
                  1.2307E-05,1.67939E-05,3.69107E-05,4.9906E-06,0.000704941,0.002366751,
                  2.68164E-05,3.59253E-05,0.000107746,2.42885E-05,0.000204362,7.30131E-05),
                .Dim=c(6,8))
  lbl <- 'brwb5_'
}

if (args[1]==20){
  T = 6
  x0 = c(13.428,16.076,0.663,0.187,0.14,0.002095722,2.14858E-05,1.29241E-05)
  t0 = 0
  ts = c(24,48,72,96,120,144)
  
  x = structure(c(9.984,7.431,0,0,0,0,
                  10.986,10.613,0.562,0.79,0.951,1.113,
                  1.931,3.939,3.976,2.581,0.472,0.464,
                  0.43,0.912,0.421,0.347,0.181,0.06119,
                  0.305,0.508,0.62,0.948,1.694,1.529,
                  0.012059624,0.000384002,3.83119E-05,5.24003E-05,7.16694E-06,2.63836E-05,
                  0.000144595,0.000235588,6.45881E-05,9.20773E-05,4.23792E-06,0.000634092,
                  6.89288E-05,1.64589E-05,3.39939E-05,2.21514E-06,2.98126E-06,1.17869E-06),
                .Dim=c(6,8))
  lbl <- 'brwb6_'
}

if (args[1]==21){
  T = 6
  x0 = c(15.694,18.579,0.954,0.177,0.154,0.00015358,5.03677E-06,1.12824E-06)
  t0 = 0
  ts = c(24,48,72,96,120,144)
  
  x = structure(c(8.316,2.48,0,0,0,0,
                  10.814,7.022,0.255,0.369,0.778,1.05,
                  2.359,9.774,6.964,4.552,1.175,0.92,
                  0.479,0.978,1.001,0.595,0.142,0.142,
                  0.305,0.351,0.479,1.117,1.999,3.438,
                  0.0101703,0.000148025,0.00164943,0.000120874,3.80482E-06,1.8044E-06,
                  0.000126227,0.00022759,0.001409784,0.00017545,2.30756E-06,7.62347E-06,
                  6.49233E-05,2.04833E-05,0.000303669,3.93007E-05,1.57544E-06,1.70766E-06),
                .Dim=c(6,8))
  lbl <- 'brwb7_'
} 

if (args[1]==22){
  T = 12
  x0 = c(39.826,51.835,0.396,0,0,0.002911329,0.000677501,2.12891E-05)
  t0 = 0
  ts = c(6,12,24,30,36,48,54,60,72,84,96,120)
  
  x = structure(c(39.926,34.367,31.39,1.613,1.117,0.422,0.918,0.223,1.414,0.72,1.315,1.712,
                  45.98,38.734,18.983,4.591,3.002,1.017,1.911,1.216,1.414,0.72,2.407,2.903,
                  4.511,6.952,12.576,16.649,14.294,11.367,17.225,13.792,11.034,6.693,10.532,10.569,
                  0.776,1.456,5.347,7.529,7.301,9.132,9.184,8.519,6.702,8.202,8.374,11.932,
                  0.76,0.92,3.73,4.463,3.64,4.394,5.354,5.24,7.046,8.554,7.16,9.926,
                  0.053099601,0.084935893,0.161842008,0.157431364,0.258280286,0.007692921,0.028780031,0.001285557,0.000333496,0.000123336,4.61415E-06,1.56348E-06,
                  0.010663751,0.02883434,0.044968667,1.348683403,0.948221969,0.149592566,0.311824341,0.16364774,0.173744079,0.173744079,0.064107673,0.776086293,
                  0.000186276,0.000147285,5.59961E-05,1.68717E-05,0.002193603,0.000604168,0.003405321,0.005783064,0.004711487,0.013157035,0.001026025,0.003524991),
                .Dim=c(12,8))
  lbl <- 'mywb3_'
}

if (args[1]==23){
  T=6
  x0 = c(81.742,86.484,0,0,0,0.022755756,1.25E-09,0.028194087)
  t0 = 0
  ts = c(24,48,72,96,120,144)
  x = structure(c(40.694,20.746,12.656,8.121,6.076,7.113,
                  31.683,24.422,15.62,12.744,9.632,9.839,
                  7.199,10.829,7.136,1.738,0,0,
                  1.256,6.395,9.039,11.154,7.749,5.988,
                  0,1.199,0.847,0.555,0.762,0.881,
                  0.182008328,0.166759759,14.32488879,0.187971176,0.302058637,0.065326781,
                  0.213751914,0.000226418,0.003916607,0.001451811,0.005068857,0.002197405,
                  0.023235821,0.017666806,0.001858481,0.044479309,0.032074362,0.040472314),
                .Dim=c(6,8))
  lbl<-'brwb8_'
}

if (args[1]==24){
  T=5
  x0 = c(43.555,53.603,6.28,0.525,0,0.258875684,0.00739452,2.88508E-05)
  t0 = 0
  ts = c(24,48,72,96,120)
  x = structure(c(22.277,6.959,3.22,3.23,2.726,
                  27.385,8.831,6.286,4.846,5.96,
                  21.863,13.726,6.226,6.102,0,
                  2.849,4.887,2.728,3.621,2.576,
                  2.467,10.355,14.302,13.924,11.002,
                  0.144574354,0.029722905,0.009399208,0.000729611,0.000110431,
                  0.222284926,0.339554909,0.100440765,0.014754008,0.021033426,
                  0.000227069,0.001368227,0.010096202,0.00279356,0.001968602),
                .Dim=c(5,8))
  lbl<-'hnwb1_'
}

if (args[1]==25){
  T = 6
  x0 = c(29.894,36.631,0,0.718,0,5.67981E-05,1.25E-09,2.8E-10)
  t0 = 0
  ts = c(24,48,72,96,120,144)
  
  x = structure(c(28.244,0.638,0,0,0,0,
                  36.401,0.638,0,0,0,0,
                  1.264,19.898,5.967,3.571,0,0,
                  0.776,0.621,3.767,6.698,8.78,11.819,
                  0.776,1.1,8.292,16.335,20.333,24.702,
                  0.998467121,1.353112071,4.095753099,73.0018336,0.000000015,0.000000015,
                  7.19702E-07,0.00349263,3.615368106,27.0490815,0.040378718,0.001079326,
                  1.61213E-07,0.000782349,0.809842456,6.058994256,0.009044833,0.000241769),
                .Dim=c(6,8))
  lbl<-'ecpb1_'
}

if (args[1]==26){
  T = 5
  x0 = c(20.293,27.875,0,0,0,0.670025388,6.26484E-06,5.85003E-07)
  t0 = 0
  ts = c(24,48,72,96,120)
  
  x = structure(c(24.351,8.802,6.819,8.028,5.133,
                  31.25,9.999,11.037,12.359,8.667,
                  13.738,17.412,12.924,7.619,1.592,
                  3.188,3.943,4.98,5.672,2.911,
                  1.932,1.871,7.021,9.063,15.846,
                  0.903839379,0.023773398,0.071794514,0.000248938,1.43249E-05,
                  0.016103119,0.002494078,0.008849322,0.00013706,0.002863585,
                  4.86584E-06,0.000585003,0.002028421,0.00024387,0.004755082),
                .Dim=c(5,8))
  lbl <- 'niwb1_'
}

if (args[1]==27){
  T = 6
  x0 = c(20.684,23.663,0,0,0,0.030626069,9.48222E-05,1.27985E-05)
  t0 = 0
  ts = c(24,48,72,96,120,144)
  
  x = structure(c(21.418,8.099,2.021,2.73,3.367,1.952,
                  25.533,14.058,2.021,2.73,4.076,1.952,
                  6.451,17.029,18.745,9.088,6.255,2.051,
                  2.412,5.069,7.765,5.833,4.333,5.776,
                  0,1.382,2.275,4.226,6.882,8.443,
                  0.328164244,1.164370675,0.065477375,0.001191492,7.17945E-05,4.52993E-06,                
                  0.001435192,0.119374073,0.202726262,0.004044921,0.057136024,0.001891952,
                  7.03328E-05,0.000140332,0.007197108,0.003070139,0.027362642,0.000906062),
                .Dim=c(6,8))
  lbl <- 'niwb2_'
}

if (args[1]==28){
  T=7
  x0 = c(29.504,34.78,0,0.13,0.196,1.81172E-06,8.47052E-09,3.14165E-08)
  t0 = 0
  ts = c(24,48,72,96,120,144,168)
  x = structure(c(29.784,0,0,0,0,0,0,
                  35.29,0,0,0,0,0,0,
                  0,36.306,38.599,31.526,33.192,30.297,28.238,
                  0.147,0.867,2.361,3.371,4.501,4.831,5.255,
                  0.196,0.196,7.02,6.555,7.335,7.44,8.025,
                  0.023125507,2.958634104,3.09094487,0.312673632,0.001875389,1.35547E-06,0.000000015,
                  2.9779E-05,0.000665135,0.839286066,1.202015348,0.219740452,0.012850204,0.000430437,
                  0.000248404,9.82105E-06,0.10142804,0.193267145,0.040659125,0.025477572,0.000139046),
                .Dim=c(7,8))
  
  lbl<-'ecpb2_'
}


x1 = rbind(x0,x)
scl = c(max(x1[,1]),max(x1[,2]),max(x1[,3]),max(x1[,4]),max(x1[,5]),max(x1[,6]),max(x1[,7]),max(x1[,8]))

#depending on chosen MI, initial values are randomly generated for MCMC-NUTS sampler

if (args[2]==0){
  ini = function(){
    list(mu1=abs(rnorm(1,0.5,0.2)),mu2=abs(rnorm(1,0.5,0.2)),
         mu3=abs(rnorm(1,0.5,0.2)),mu4=abs(rnorm(1,0.5,0.2)),
         mu5=abs(rnorm(1,0.5,0.2)),ks1=abs(rnorm(1,0.5,0.2)),
         ks2=abs(rnorm(1,0.5,0.2)),ks3=abs(rnorm(1,0.5,0.2)),
         ks4=abs(rnorm(1,0.5,0.2)),ks5=abs(rnorm(1,0.5,0.2)),
         k1=abs(rnorm(1,0.5,0.2)),k2=abs(rnorm(1,0.5,0.2)),
         k3=abs(rnorm(1,0.5,0.2)),yc1=abs(rnorm(1,0.5,0.2)),
         yc2=abs(rnorm(1,0.5,0.2)),yc3=abs(rnorm(1,0.5,0.2)),
         yc4=abs(rnorm(1,0.5,0.2)),yc5=abs(rnorm(1,0.5,0.2)),
         yc6=abs(rnorm(1,0.5,0.2)),yc7=abs(rnorm(1,0.5,0.2)),
         yc8=abs(rnorm(1,0.5,0.2)),yc9=abs(rnorm(1,0.5,0.2)),
         yc10=abs(rnorm(1,0.5,0.2)),yc11=abs(rnorm(1,0.5,0.2)),
         sigma=abs(rnorm(1,0.5,0.2)))
  }
}
if (args[2]==1){
  ini = function(){
    list(mu1=abs(rnorm(1,0.5,0.2)),mu2=abs(rnorm(1,0.5,0.2)),
         mu3=abs(rnorm(1,0.5,0.2)),mu4=abs(rnorm(1,0.5,0.2)),
         mu5=abs(rnorm(1,0.5,0.2)),ks1=abs(rnorm(1,0.5,0.2)),
         ks2=abs(rnorm(1,0.5,0.2)),ks3=abs(rnorm(1,0.5,0.2)),
         ks4=abs(rnorm(1,0.5,0.2)),ks5=abs(rnorm(1,0.5,0.2)),
         k1=abs(rnorm(1,0.5,0.2)),k2=abs(rnorm(1,0.5,0.2)),
         k3=abs(rnorm(1,0.5,0.2)),yc1=abs(rnorm(1,0.5,0.2)),
         yc2=abs(rnorm(1,0.5,0.2)),yc3=abs(rnorm(1,0.5,0.2)),
         yc4=abs(rnorm(1,0.5,0.2)),yc5=abs(rnorm(1,0.5,0.2)),
         yc6=abs(rnorm(1,0.5,0.2)),yc7=abs(rnorm(1,0.5,0.2)),
         yc8=abs(rnorm(1,0.5,0.2)),yc9=abs(rnorm(1,0.5,0.2)),
         yc10=abs(rnorm(1,0.5,0.2)),yc11=abs(rnorm(1,0.5,0.2)),
         ev1=abs(rnorm(1,0.5,0.2)),ev2=abs(rnorm(1,0.5,0.2)),
         ev3=abs(rnorm(1,0.5,0.2)),sigma=abs(rnorm(1,0.5,0.2)))
  }
}
if (args[2]==2){
  ini = function(){
    list(mu1=abs(rnorm(1,0.5,0.2)),mu2=abs(rnorm(1,0.5,0.2)),
         mu3=abs(rnorm(1,0.5,0.2)),mu4=abs(rnorm(1,0.5,0.2)),
         mu5=abs(rnorm(1,0.5,0.2)),mu6=abs(rnorm(1,0.5,0.2)),
         ks1=abs(rnorm(1,0.5,0.2)),ks2=abs(rnorm(1,0.5,0.2)),
         ks3=abs(rnorm(1,0.5,0.2)),ks4=abs(rnorm(1,0.5,0.2)),
         ks5=abs(rnorm(1,0.5,0.2)),ks6=abs(rnorm(1,0.5,0.2)),
         k1=abs(rnorm(1,0.5,0.2)),k2=abs(rnorm(1,0.5,0.2)),
         k3=abs(rnorm(1,0.5,0.2)),yc1=abs(rnorm(1,0.5,0.2)),
         yc2=abs(rnorm(1,0.5,0.2)),yc3=abs(rnorm(1,0.5,0.2)),
         yc4=abs(rnorm(1,0.5,0.2)),yc5=abs(rnorm(1,0.5,0.2)),
         yc6=abs(rnorm(1,0.5,0.2)),yc7=abs(rnorm(1,0.5,0.2)),
         yc8=abs(rnorm(1,0.5,0.2)),yc9=abs(rnorm(1,0.5,0.2)),
         yc10=abs(rnorm(1,0.5,0.2)),yc11=abs(rnorm(1,0.5,0.2)),
         yc12=abs(rnorm(1,0.5,0.2)),yc13=abs(rnorm(1,0.5,0.2)),
         yc14=abs(rnorm(1,0.5,0.2)),yc15=abs(rnorm(1,0.5,0.2)),
         yc16=abs(rnorm(1,0.5,0.2)),sigma=abs(rnorm(1,0.5,0.2)))
  }
}
if (args[2]==3){
  ini = function(){
    list(mu1=abs(rnorm(1,0.5,0.2)),mu2=abs(rnorm(1,0.5,0.2)),
         mu3=abs(rnorm(1,0.5,0.2)),mu4=abs(rnorm(1,0.5,0.2)),
         mu5=abs(rnorm(1,0.5,0.2)),ks1=abs(rnorm(1,0.5,0.2)),
         ks2=abs(rnorm(1,0.5,0.2)),ks3=abs(rnorm(1,0.5,0.2)),
         ks4=abs(rnorm(1,0.5,0.2)),ks5=abs(rnorm(1,0.5,0.2)),
         k1=abs(rnorm(1,0.5,0.2)),k2=abs(rnorm(1,0.5,0.2)),
         k3=abs(rnorm(1,0.5,0.2)),yc1=abs(rnorm(1,0.5,0.2)),
         yc2=abs(rnorm(1,0.5,0.2)),yc3=abs(rnorm(1,0.5,0.2)),
         yc4=abs(rnorm(1,0.5,0.2)),yc5=abs(rnorm(1,0.5,0.2)),
         yc6=abs(rnorm(1,0.5,0.2)),yc7=abs(rnorm(1,0.5,0.2)),
         yc8=abs(rnorm(1,0.5,0.2)),yc9=abs(rnorm(1,0.5,0.2)),
         yc10=abs(rnorm(1,0.5,0.2)),yc11=abs(rnorm(1,0.5,0.2)),
         yc12=abs(rnorm(1,0.5,0.2)),yc13=abs(rnorm(1,0.5,0.2)),
         sigma=abs(rnorm(1,0.5,0.2)))
  }
}

if (args[2]==4){
  ini = function(){
    list(mu1=abs(rnorm(1,0.5,0.2)),mu2=abs(rnorm(1,0.5,0.2)),
         mu3=abs(rnorm(1,0.5,0.2)),mu4=abs(rnorm(1,0.5,0.2)),
         mu5=abs(rnorm(1,0.5,0.2)),mu6=abs(rnorm(1,0.5,0.2)),
         ks1=abs(rnorm(1,0.5,0.2)),ks2=abs(rnorm(1,0.5,0.2)),
         ks3=abs(rnorm(1,0.5,0.2)),ks4=abs(rnorm(1,0.5,0.2)),
         ks5=abs(rnorm(1,0.5,0.2)),ks6=abs(rnorm(1,0.5,0.2)),
         k1=abs(rnorm(1,0.5,0.2)),k2=abs(rnorm(1,0.5,0.2)),
         k3=abs(rnorm(1,0.5,0.2)),yc1=abs(rnorm(1,0.5,0.2)),
         yc2=abs(rnorm(1,0.5,0.2)),yc3=abs(rnorm(1,0.5,0.2)),
         yc4=abs(rnorm(1,0.5,0.2)),yc5=abs(rnorm(1,0.5,0.2)),
         yc6=abs(rnorm(1,0.5,0.2)),yc7=abs(rnorm(1,0.5,0.2)),
         yc8=abs(rnorm(1,0.5,0.2)),yc9=abs(rnorm(1,0.5,0.2)),
         yc10=abs(rnorm(1,0.5,0.2)),yc11=abs(rnorm(1,0.5,0.2)),
         yc12=abs(rnorm(1,0.5,0.2)),yc13=abs(rnorm(1,0.5,0.2)),
         sigma=abs(rnorm(1,0.5,0.2)))
  }
}

if (args[2]==5){
  ini = function(){
    list(mu1=abs(rnorm(1,0.5,0.2)),mu2=abs(rnorm(1,0.5,0.2)),
         mu3=abs(rnorm(1,0.5,0.2)),mu4=abs(rnorm(1,0.5,0.2)),
         mu5=abs(rnorm(1,0.5,0.2)),mu6=abs(rnorm(1,0.5,0.2)),
         ks1=abs(rnorm(1,0.5,0.2)),ks2=abs(rnorm(1,0.5,0.2)),
         ks3=abs(rnorm(1,0.5,0.2)),ks4=abs(rnorm(1,0.5,0.2)),
         ks5=abs(rnorm(1,0.5,0.2)),ks6=abs(rnorm(1,0.5,0.2)),
         k1=abs(rnorm(1,0.5,0.2)),k2=abs(rnorm(1,0.5,0.2)),
         k3=abs(rnorm(1,0.5,0.2)),yc1=abs(rnorm(1,0.5,0.2)),
         yc2=abs(rnorm(1,0.5,0.2)),yc3=abs(rnorm(1,0.5,0.2)),
         yc4=abs(rnorm(1,0.5,0.2)),yc5=abs(rnorm(1,0.5,0.2)),
         yc6=abs(rnorm(1,0.5,0.2)),yc7=abs(rnorm(1,0.5,0.2)),
         yc8=abs(rnorm(1,0.5,0.2)),yc9=abs(rnorm(1,0.5,0.2)),
         yc10=abs(rnorm(1,0.5,0.2)),yc11=abs(rnorm(1,0.5,0.2)),
         yc12=abs(rnorm(1,0.5,0.2)),sigma=abs(rnorm(1,0.5,0.2)))
  }
}
if (args[2]==12){
  ini = function(){
    list(mu1=abs(rnorm(1,0.5,0.2)),mu2=abs(rnorm(1,0.5,0.2)),
         mu3=abs(rnorm(1,0.5,0.2)),mu4=abs(rnorm(1,0.5,0.2)),
         mu5=abs(rnorm(1,0.5,0.2)),mu6=abs(rnorm(1,0.5,0.2)),
         ks1=abs(rnorm(1,0.5,0.2)),ks2=abs(rnorm(1,0.5,0.2)),
         ks3=abs(rnorm(1,0.5,0.2)),ks4=abs(rnorm(1,0.5,0.2)),
         ks5=abs(rnorm(1,0.5,0.2)),ks6=abs(rnorm(1,0.5,0.2)),
         k1=abs(rnorm(1,0.5,0.2)),k2=abs(rnorm(1,0.5,0.2)),
         k3=abs(rnorm(1,0.5,0.2)),yc1=abs(rnorm(1,0.5,0.2)),
         yc2=abs(rnorm(1,0.5,0.2)),yc3=abs(rnorm(1,0.5,0.2)),
         yc4=abs(rnorm(1,0.5,0.2)),yc5=abs(rnorm(1,0.5,0.2)),
         yc6=abs(rnorm(1,0.5,0.2)),yc7=abs(rnorm(1,0.5,0.2)),
         yc8=abs(rnorm(1,0.5,0.2)),yc9=abs(rnorm(1,0.5,0.2)),
         yc10=abs(rnorm(1,0.5,0.2)),yc11=abs(rnorm(1,0.5,0.2)),
         yc12=abs(rnorm(1,0.5,0.2)),yc13=abs(rnorm(1,0.5,0.2)),
         yc14=abs(rnorm(1,0.5,0.2)),yc15=abs(rnorm(1,0.5,0.2)),
         yc16=abs(rnorm(1,0.5,0.2)),sigma=abs(rnorm(1,0.5,0.2)),
         ev1=abs(rnorm(1,0.5,0.2)),ev2=abs(rnorm(1,0.5,0.2)),
         ev3=abs(rnorm(1,0.5,0.2)))
  }
}
if (args[2]==13){
  ini = function(){
    list(mu1=abs(rnorm(1,0.5,0.2)),mu2=abs(rnorm(1,0.5,0.2)),
         mu3=abs(rnorm(1,0.5,0.2)),mu4=abs(rnorm(1,0.5,0.2)),
         mu5=abs(rnorm(1,0.5,0.2)),ks1=abs(rnorm(1,0.5,0.2)),
         ks2=abs(rnorm(1,0.5,0.2)),ks3=abs(rnorm(1,0.5,0.2)),
         ks4=abs(rnorm(1,0.5,0.2)),ks5=abs(rnorm(1,0.5,0.2)),
         k1=abs(rnorm(1,0.5,0.2)),k2=abs(rnorm(1,0.5,0.2)),
         k3=abs(rnorm(1,0.5,0.2)),yc1=abs(rnorm(1,0.5,0.2)),
         yc2=abs(rnorm(1,0.5,0.2)),yc3=abs(rnorm(1,0.5,0.2)),
         yc4=abs(rnorm(1,0.5,0.2)),yc5=abs(rnorm(1,0.5,0.2)),
         yc6=abs(rnorm(1,0.5,0.2)),yc7=abs(rnorm(1,0.5,0.2)),
         yc8=abs(rnorm(1,0.5,0.2)),yc9=abs(rnorm(1,0.5,0.2)),
         yc10=abs(rnorm(1,0.5,0.2)),yc11=abs(rnorm(1,0.5,0.2)),
         yc12=abs(rnorm(1,0.5,0.2)),yc13=abs(rnorm(1,0.5,0.2)),
         ev1=abs(rnorm(1,0.5,0.2)),ev2=abs(rnorm(1,0.5,0.2)),
         ev3=abs(rnorm(1,0.5,0.2)),sigma=abs(rnorm(1,0.5,0.2)))
  }
}
if (args[2]==14){
  ini = function(){
    list(mu1=abs(rnorm(1,0.5,0.2)),mu2=abs(rnorm(1,0.5,0.2)),
         mu3=abs(rnorm(1,0.5,0.2)),mu4=abs(rnorm(1,0.5,0.2)),
         mu5=abs(rnorm(1,0.5,0.2)),mu6=abs(rnorm(1,0.5,0.2)),
         ks1=abs(rnorm(1,0.5,0.2)),ks2=abs(rnorm(1,0.5,0.2)),
         ks3=abs(rnorm(1,0.5,0.2)),ks4=abs(rnorm(1,0.5,0.2)),
         ks5=abs(rnorm(1,0.5,0.2)),ks6=abs(rnorm(1,0.5,0.2)),
         k1=abs(rnorm(1,0.5,0.2)),k2=abs(rnorm(1,0.5,0.2)),
         k3=abs(rnorm(1,0.5,0.2)),yc1=abs(rnorm(1,0.5,0.2)),
         yc2=abs(rnorm(1,0.5,0.2)),yc3=abs(rnorm(1,0.5,0.2)),
         yc4=abs(rnorm(1,0.5,0.2)),yc5=abs(rnorm(1,0.5,0.2)),
         yc6=abs(rnorm(1,0.5,0.2)),yc7=abs(rnorm(1,0.5,0.2)),
         yc8=abs(rnorm(1,0.5,0.2)),yc9=abs(rnorm(1,0.5,0.2)),
         yc10=abs(rnorm(1,0.5,0.2)),yc11=abs(rnorm(1,0.5,0.2)),
         yc12=abs(rnorm(1,0.5,0.2)),yc13=abs(rnorm(1,0.5,0.2)),
         ev1=abs(rnorm(1,0.5,0.2)),ev2=abs(rnorm(1,0.5,0.2)),
         ev3=abs(rnorm(1,0.5,0.2)),sigma=abs(rnorm(1,0.5,0.2)))
  }
}
if (args[2]==15){
  ini = function(){
    list(mu1=abs(rnorm(1,0.5,0.2)),mu2=abs(rnorm(1,0.5,0.2)),
         mu3=abs(rnorm(1,0.5,0.2)),mu4=abs(rnorm(1,0.5,0.2)),
         mu5=abs(rnorm(1,0.5,0.2)),mu6=abs(rnorm(1,0.5,0.2)),
         ks1=abs(rnorm(1,0.5,0.2)),ks2=abs(rnorm(1,0.5,0.2)),
         ks3=abs(rnorm(1,0.5,0.2)),ks4=abs(rnorm(1,0.5,0.2)),
         ks5=abs(rnorm(1,0.5,0.2)),ks6=abs(rnorm(1,0.5,0.2)),
         k1=abs(rnorm(1,0.5,0.2)),k2=abs(rnorm(1,0.5,0.2)),
         k3=abs(rnorm(1,0.5,0.2)),yc1=abs(rnorm(1,0.5,0.2)),
         yc2=abs(rnorm(1,0.5,0.2)),yc3=abs(rnorm(1,0.5,0.2)),
         yc4=abs(rnorm(1,0.5,0.2)),yc5=abs(rnorm(1,0.5,0.2)),
         yc6=abs(rnorm(1,0.5,0.2)),yc7=abs(rnorm(1,0.5,0.2)),
         yc8=abs(rnorm(1,0.5,0.2)),yc9=abs(rnorm(1,0.5,0.2)),
         yc10=abs(rnorm(1,0.5,0.2)),yc11=abs(rnorm(1,0.5,0.2)),
         yc12=abs(rnorm(1,0.5,0.2)),ev1=abs(rnorm(1,0.5,0.2)),
         ev2=abs(rnorm(1,0.5,0.2)),ev3=abs(rnorm(1,0.5,0.2)),
         sigma=abs(rnorm(1,0.5,0.2)))
  }
}
if (args[2]==23){
  ini = function(){
    list(mu1=abs(rnorm(1,0.5,0.2)),mu2=abs(rnorm(1,0.5,0.2)),
         mu3=abs(rnorm(1,0.5,0.2)),mu4=abs(rnorm(1,0.5,0.2)),
         mu5=abs(rnorm(1,0.5,0.2)),mu6=abs(rnorm(1,0.5,0.2)),
         ks1=abs(rnorm(1,0.5,0.2)),ks2=abs(rnorm(1,0.5,0.2)),
         ks3=abs(rnorm(1,0.5,0.2)),ks4=abs(rnorm(1,0.5,0.2)),
         ks5=abs(rnorm(1,0.5,0.2)),ks6=abs(rnorm(1,0.5,0.2)),
         k1=abs(rnorm(1,0.5,0.2)),k2=abs(rnorm(1,0.5,0.2)),
         k3=abs(rnorm(1,0.5,0.2)),yc1=abs(rnorm(1,0.5,0.2)),
         yc2=abs(rnorm(1,0.5,0.2)),yc3=abs(rnorm(1,0.5,0.2)),
         yc4=abs(rnorm(1,0.5,0.2)),yc5=abs(rnorm(1,0.5,0.2)),
         yc6=abs(rnorm(1,0.5,0.2)),yc7=abs(rnorm(1,0.5,0.2)),
         yc8=abs(rnorm(1,0.5,0.2)),yc9=abs(rnorm(1,0.5,0.2)),
         yc10=abs(rnorm(1,0.5,0.2)),yc11=abs(rnorm(1,0.5,0.2)),
         yc12=abs(rnorm(1,0.5,0.2)),yc13=abs(rnorm(1,0.5,0.2)),
         yc14=abs(rnorm(1,0.5,0.2)),yc15=abs(rnorm(1,0.5,0.2)),
         yc16=abs(rnorm(1,0.5,0.2)),yc17=abs(rnorm(1,0.5,0.2)),
         yc18=abs(rnorm(1,0.5,0.2)),sigma=abs(rnorm(1,0.5,0.2)))
  }
}
if (args[2]==24){
  ini = function(){
    list(mu1=abs(rnorm(1,0.5,0.2)),mu2=abs(rnorm(1,0.5,0.2)),
         mu3=abs(rnorm(1,0.5,0.2)),mu4=abs(rnorm(1,0.5,0.2)),
         mu5=abs(rnorm(1,0.5,0.2)),mu6=abs(rnorm(1,0.5,0.2)),
         mu7=abs(rnorm(1,0.5,0.2)),ks1=abs(rnorm(1,0.5,0.2)),
         ks2=abs(rnorm(1,0.5,0.2)),ks3=abs(rnorm(1,0.5,0.2)),
         ks4=abs(rnorm(1,0.5,0.2)),ks5=abs(rnorm(1,0.5,0.2)),
         ks6=abs(rnorm(1,0.5,0.2)),ks7=abs(rnorm(1,0.5,0.2)),
         k1=abs(rnorm(1,0.5,0.2)),k2=abs(rnorm(1,0.5,0.2)),
         k3=abs(rnorm(1,0.5,0.2)),yc1=abs(rnorm(1,0.5,0.2)),
         yc2=abs(rnorm(1,0.5,0.2)),yc3=abs(rnorm(1,0.5,0.2)),
         yc4=abs(rnorm(1,0.5,0.2)),yc5=abs(rnorm(1,0.5,0.2)),
         yc6=abs(rnorm(1,0.5,0.2)),yc7=abs(rnorm(1,0.5,0.2)),
         yc8=abs(rnorm(1,0.5,0.2)),yc9=abs(rnorm(1,0.5,0.2)),
         yc10=abs(rnorm(1,0.5,0.2)),yc11=abs(rnorm(1,0.5,0.2)),
         yc12=abs(rnorm(1,0.5,0.2)),yc13=abs(rnorm(1,0.5,0.2)),
         yc14=abs(rnorm(1,0.5,0.2)),yc15=abs(rnorm(1,0.5,0.2)),
         yc16=abs(rnorm(1,0.5,0.2)),yc17=abs(rnorm(1,0.5,0.2)),
         yc18=abs(rnorm(1,0.5,0.2)),sigma=abs(rnorm(1,0.5,0.2)))
  }
}
if (args[2]==25){
  ini = function(){
    list(mu1=abs(rnorm(1,0.5,0.2)),mu2=abs(rnorm(1,0.5,0.2)),
         mu3=abs(rnorm(1,0.5,0.2)),mu4=abs(rnorm(1,0.5,0.2)),
         mu5=abs(rnorm(1,0.5,0.2)),mu6=abs(rnorm(1,0.5,0.2)),
         mu7=abs(rnorm(1,0.5,0.2)),ks1=abs(rnorm(1,0.5,0.2)),
         ks2=abs(rnorm(1,0.5,0.2)),ks3=abs(rnorm(1,0.5,0.2)),
         ks4=abs(rnorm(1,0.5,0.2)),ks5=abs(rnorm(1,0.5,0.2)),
         ks6=abs(rnorm(1,0.5,0.2)),ks7=abs(rnorm(1,0.5,0.2)),
         k1=abs(rnorm(1,0.5,0.2)),k2=abs(rnorm(1,0.5,0.2)),
         k3=abs(rnorm(1,0.5,0.2)),yc1=abs(rnorm(1,0.5,0.2)),
         yc2=abs(rnorm(1,0.5,0.2)),yc3=abs(rnorm(1,0.5,0.2)),
         yc4=abs(rnorm(1,0.5,0.2)),yc5=abs(rnorm(1,0.5,0.2)),
         yc6=abs(rnorm(1,0.5,0.2)),yc7=abs(rnorm(1,0.5,0.2)),
         yc8=abs(rnorm(1,0.5,0.2)),yc9=abs(rnorm(1,0.5,0.2)),
         yc10=abs(rnorm(1,0.5,0.2)),yc11=abs(rnorm(1,0.5,0.2)),
         yc12=abs(rnorm(1,0.5,0.2)),yc13=abs(rnorm(1,0.5,0.2)),
         yc14=abs(rnorm(1,0.5,0.2)),yc15=abs(rnorm(1,0.5,0.2)),
         yc16=abs(rnorm(1,0.5,0.2)),yc17=abs(rnorm(1,0.5,0.2)),
         sigma=abs(rnorm(1,0.5,0.2)))
  }
}
if (args[2]==34){
  ini = function(){
    list(mu1=abs(rnorm(1,0.5,0.2)),mu2=abs(rnorm(1,0.5,0.2)),
         mu3=abs(rnorm(1,0.5,0.2)),mu4=abs(rnorm(1,0.5,0.2)),
         mu5=abs(rnorm(1,0.5,0.2)),mu6=abs(rnorm(1,0.5,0.2)),
         ks1=abs(rnorm(1,0.5,0.2)),ks2=abs(rnorm(1,0.5,0.2)),
         ks3=abs(rnorm(1,0.5,0.2)),ks4=abs(rnorm(1,0.5,0.2)),
         ks5=abs(rnorm(1,0.5,0.2)),ks6=abs(rnorm(1,0.5,0.2)),
         k1=abs(rnorm(1,0.5,0.2)),k2=abs(rnorm(1,0.5,0.2)),
         k3=abs(rnorm(1,0.5,0.2)),yc1=abs(rnorm(1,0.5,0.2)),
         yc2=abs(rnorm(1,0.5,0.2)),yc3=abs(rnorm(1,0.5,0.2)),
         yc4=abs(rnorm(1,0.5,0.2)),yc5=abs(rnorm(1,0.5,0.2)),
         yc6=abs(rnorm(1,0.5,0.2)),yc7=abs(rnorm(1,0.5,0.2)),
         yc8=abs(rnorm(1,0.5,0.2)),yc9=abs(rnorm(1,0.5,0.2)),
         yc10=abs(rnorm(1,0.5,0.2)),yc11=abs(rnorm(1,0.5,0.2)),
         yc12=abs(rnorm(1,0.5,0.2)),yc13=abs(rnorm(1,0.5,0.2)),
         yc14=abs(rnorm(1,0.5,0.2)),yc15=abs(rnorm(1,0.5,0.2)),
         sigma=abs(rnorm(1,0.5,0.2)))
  }
}
if (args[2]==35){
  ini = function(){
    list(mu1=abs(rnorm(1,0.5,0.2)),mu2=abs(rnorm(1,0.5,0.2)),
         mu3=abs(rnorm(1,0.5,0.2)),mu4=abs(rnorm(1,0.5,0.2)),
         mu5=abs(rnorm(1,0.5,0.2)),mu6=abs(rnorm(1,0.5,0.2)),
         ks1=abs(rnorm(1,0.5,0.2)),ks2=abs(rnorm(1,0.5,0.2)),
         ks3=abs(rnorm(1,0.5,0.2)),ks4=abs(rnorm(1,0.5,0.2)),
         ks5=abs(rnorm(1,0.5,0.2)),ks6=abs(rnorm(1,0.5,0.2)),
         k1=abs(rnorm(1,0.5,0.2)),k2=abs(rnorm(1,0.5,0.2)),
         k3=abs(rnorm(1,0.5,0.2)),yc1=abs(rnorm(1,0.5,0.2)),
         yc2=abs(rnorm(1,0.5,0.2)),yc3=abs(rnorm(1,0.5,0.2)),
         yc4=abs(rnorm(1,0.5,0.2)),yc5=abs(rnorm(1,0.5,0.2)),
         yc6=abs(rnorm(1,0.5,0.2)),yc7=abs(rnorm(1,0.5,0.2)),
         yc8=abs(rnorm(1,0.5,0.2)),yc9=abs(rnorm(1,0.5,0.2)),
         yc10=abs(rnorm(1,0.5,0.2)),yc11=abs(rnorm(1,0.5,0.2)),
         yc12=abs(rnorm(1,0.5,0.2)),yc13=abs(rnorm(1,0.5,0.2)),
         yc14=abs(rnorm(1,0.5,0.2)),sigma=abs(rnorm(1,0.5,0.2)))
  }
}
if (args[2]==45){
  ini = function(){
    list(mu1=abs(rnorm(1,0.5,0.2)),mu2=abs(rnorm(1,0.5,0.2)),
         mu3=abs(rnorm(1,0.5,0.2)),mu4=abs(rnorm(1,0.5,0.2)),
         mu5=abs(rnorm(1,0.5,0.2)),mu6=abs(rnorm(1,0.5,0.2)),
         mu7=abs(rnorm(1,0.5,0.2)),ks1=abs(rnorm(1,0.5,0.2)),
         ks2=abs(rnorm(1,0.5,0.2)),ks3=abs(rnorm(1,0.5,0.2)),
         ks4=abs(rnorm(1,0.5,0.2)),ks5=abs(rnorm(1,0.5,0.2)),
         ks6=abs(rnorm(1,0.5,0.2)),ks7=abs(rnorm(1,0.5,0.2)),
         k1=abs(rnorm(1,0.5,0.2)),k2=abs(rnorm(1,0.5,0.2)),
         k3=abs(rnorm(1,0.5,0.2)),yc1=abs(rnorm(1,0.5,0.2)),
         yc2=abs(rnorm(1,0.5,0.2)),yc3=abs(rnorm(1,0.5,0.2)),
         yc4=abs(rnorm(1,0.5,0.2)),yc5=abs(rnorm(1,0.5,0.2)),
         yc6=abs(rnorm(1,0.5,0.2)),yc7=abs(rnorm(1,0.5,0.2)),
         yc8=abs(rnorm(1,0.5,0.2)),yc9=abs(rnorm(1,0.5,0.2)),
         yc10=abs(rnorm(1,0.5,0.2)),yc11=abs(rnorm(1,0.5,0.2)),
         yc12=abs(rnorm(1,0.5,0.2)),yc13=abs(rnorm(1,0.5,0.2)),
         yc14=abs(rnorm(1,0.5,0.2)),sigma=abs(rnorm(1,0.5,0.2)))
  }
}
if (args[2]==123){
  ini = function(){
    list(mu1=abs(rnorm(1,0.5,0.2)),mu2=abs(rnorm(1,0.5,0.2)),
         mu3=abs(rnorm(1,0.5,0.2)),mu4=abs(rnorm(1,0.5,0.2)),
         mu5=abs(rnorm(1,0.5,0.2)),mu6=abs(rnorm(1,0.5,0.2)),
         ks1=abs(rnorm(1,0.5,0.2)),ks2=abs(rnorm(1,0.5,0.2)),
         ks3=abs(rnorm(1,0.5,0.2)),ks4=abs(rnorm(1,0.5,0.2)),
         ks5=abs(rnorm(1,0.5,0.2)),ks6=abs(rnorm(1,0.5,0.2)),
         k1=abs(rnorm(1,0.5,0.2)),k2=abs(rnorm(1,0.5,0.2)),
         k3=abs(rnorm(1,0.5,0.2)),yc1=abs(rnorm(1,0.5,0.2)),
         yc2=abs(rnorm(1,0.5,0.2)),yc3=abs(rnorm(1,0.5,0.2)),
         yc4=abs(rnorm(1,0.5,0.2)),yc5=abs(rnorm(1,0.5,0.2)),
         yc6=abs(rnorm(1,0.5,0.2)),yc7=abs(rnorm(1,0.5,0.2)),
         yc8=abs(rnorm(1,0.5,0.2)),yc9=abs(rnorm(1,0.5,0.2)),
         yc10=abs(rnorm(1,0.5,0.2)),yc11=abs(rnorm(1,0.5,0.2)),
         yc12=abs(rnorm(1,0.5,0.2)),yc13=abs(rnorm(1,0.5,0.2)),
         yc14=abs(rnorm(1,0.5,0.2)),yc15=abs(rnorm(1,0.5,0.2)),
         yc16=abs(rnorm(1,0.5,0.2)),y17=abs(rnorm(1,0.5,0.2)),
         yc18=abs(rnorm(1,0.5,0.2)),ev1=abs(rnorm(1,0.5,0.2)),
         ev2=abs(rnorm(1,0.5,0.2)),ev3=abs(rnorm(1,0.5,0.2)),
         sigma=abs(rnorm(1,0.5,0.2)))
  }
}
if (args[2]==124){
  ini = function(){
    list(mu1=abs(rnorm(1,0.5,0.2)),mu2=abs(rnorm(1,0.5,0.2)),
         mu3=abs(rnorm(1,0.5,0.2)),mu4=abs(rnorm(1,0.5,0.2)),
         mu5=abs(rnorm(1,0.5,0.2)),mu6=abs(rnorm(1,0.5,0.2)),
         mu7=abs(rnorm(1,0.5,0.2)),ks1=abs(rnorm(1,0.5,0.2)),
         ks2=abs(rnorm(1,0.5,0.2)),ks3=abs(rnorm(1,0.5,0.2)),
         ks4=abs(rnorm(1,0.5,0.2)),ks5=abs(rnorm(1,0.5,0.2)),
         ks6=abs(rnorm(1,0.5,0.2)),ks7=abs(rnorm(1,0.5,0.2)),
         k1=abs(rnorm(1,0.5,0.2)),k2=abs(rnorm(1,0.5,0.2)),
         k3=abs(rnorm(1,0.5,0.2)),yc1=abs(rnorm(1,0.5,0.2)),
         yc2=abs(rnorm(1,0.5,0.2)),yc3=abs(rnorm(1,0.5,0.2)),
         yc4=abs(rnorm(1,0.5,0.2)),yc5=abs(rnorm(1,0.5,0.2)),
         yc6=abs(rnorm(1,0.5,0.2)),yc7=abs(rnorm(1,0.5,0.2)),
         yc8=abs(rnorm(1,0.5,0.2)),yc9=abs(rnorm(1,0.5,0.2)),
         yc10=abs(rnorm(1,0.5,0.2)),yc11=abs(rnorm(1,0.5,0.2)),
         yc12=abs(rnorm(1,0.5,0.2)),yc13=abs(rnorm(1,0.5,0.2)),
         yc14=abs(rnorm(1,0.5,0.2)),yc15=abs(rnorm(1,0.5,0.2)),
         yc16=abs(rnorm(1,0.5,0.2)),yc17=abs(rnorm(1,0.5,0.2)),
         yc18=abs(rnorm(1,0.5,0.2)),ev1=abs(rnorm(1,0.5,0.2)),
         ev2=abs(rnorm(1,0.5,0.2)),ev3=abs(rnorm(1,0.5,0.2)),
         sigma=abs(rnorm(1,0.5,0.2)))
  }
}

if (args[2]==125){
  ini = function(){
    list(mu1=abs(rnorm(1,0.5,0.2)),mu2=abs(rnorm(1,0.5,0.2)),
         mu3=abs(rnorm(1,0.5,0.2)),mu4=abs(rnorm(1,0.5,0.2)),
         mu5=abs(rnorm(1,0.5,0.2)),mu6=abs(rnorm(1,0.5,0.2)),
         mu7=abs(rnorm(1,0.5,0.2)),ks1=abs(rnorm(1,0.5,0.2)),
         ks2=abs(rnorm(1,0.5,0.2)),ks3=abs(rnorm(1,0.5,0.2)),
         ks4=abs(rnorm(1,0.5,0.2)),ks5=abs(rnorm(1,0.5,0.2)),
         ks6=abs(rnorm(1,0.5,0.2)),ks7=abs(rnorm(1,0.5,0.2)),
         k1=abs(rnorm(1,0.5,0.2)),k2=abs(rnorm(1,0.5,0.2)),
         k3=abs(rnorm(1,0.5,0.2)),yc1=abs(rnorm(1,0.5,0.2)),
         yc2=abs(rnorm(1,0.5,0.2)),yc3=abs(rnorm(1,0.5,0.2)),
         yc4=abs(rnorm(1,0.5,0.2)),yc5=abs(rnorm(1,0.5,0.2)),
         yc6=abs(rnorm(1,0.5,0.2)),yc7=abs(rnorm(1,0.5,0.2)),
         yc8=abs(rnorm(1,0.5,0.2)),yc9=abs(rnorm(1,0.5,0.2)),
         yc10=abs(rnorm(1,0.5,0.2)),yc11=abs(rnorm(1,0.5,0.2)),
         yc12=abs(rnorm(1,0.5,0.2)),yc13=abs(rnorm(1,0.5,0.2)),
         yc14=abs(rnorm(1,0.5,0.2)),yc15=abs(rnorm(1,0.5,0.2)),
         yc16=abs(rnorm(1,0.5,0.2)),yc17=abs(rnorm(1,0.5,0.2)),
         ev1=abs(rnorm(1,0.5,0.2)),ev2=abs(rnorm(1,0.5,0.2)),
         ev3=abs(rnorm(1,0.5,0.2)),sigma=abs(rnorm(1,0.5,0.2)))
  }
}
if (args[2]==134){
  ini = function(){
    list(mu1=abs(rnorm(1,0.5,0.2)),mu2=abs(rnorm(1,0.5,0.2)),
         mu3=abs(rnorm(1,0.5,0.2)),mu4=abs(rnorm(1,0.5,0.2)),
         mu5=abs(rnorm(1,0.5,0.2)),mu6=abs(rnorm(1,0.5,0.2)),
         ks1=abs(rnorm(1,0.5,0.2)),ks2=abs(rnorm(1,0.5,0.2)),
         ks3=abs(rnorm(1,0.5,0.2)),ks4=abs(rnorm(1,0.5,0.2)),
         ks5=abs(rnorm(1,0.5,0.2)),ks6=abs(rnorm(1,0.5,0.2)),
         k1=abs(rnorm(1,0.5,0.2)),k2=abs(rnorm(1,0.5,0.2)),
         k3=abs(rnorm(1,0.5,0.2)),yc1=abs(rnorm(1,0.5,0.2)),
         yc2=abs(rnorm(1,0.5,0.2)),yc3=abs(rnorm(1,0.5,0.2)),
         yc4=abs(rnorm(1,0.5,0.2)),yc5=abs(rnorm(1,0.5,0.2)),
         yc6=abs(rnorm(1,0.5,0.2)),yc7=abs(rnorm(1,0.5,0.2)),
         yc8=abs(rnorm(1,0.5,0.2)),yc9=abs(rnorm(1,0.5,0.2)),
         yc10=abs(rnorm(1,0.5,0.2)),yc11=abs(rnorm(1,0.5,0.2)),
         yc12=abs(rnorm(1,0.5,0.2)),yc13=abs(rnorm(1,0.5,0.2)),
         yc14=abs(rnorm(1,0.5,0.2)),yc15=abs(rnorm(1,0.5,0.2)),
         ev1=abs(rnorm(1,0.5,0.2)),ev2=abs(rnorm(1,0.5,0.2)),
         ev3=abs(rnorm(1,0.5,0.2)),sigma=abs(rnorm(1,0.5,0.2)))
  }
}
if (args[2]==135){
  ini = function(){
    list(mu1=abs(rnorm(1,0.5,0.2)),mu2=abs(rnorm(1,0.5,0.2)),
         mu3=abs(rnorm(1,0.5,0.2)),mu4=abs(rnorm(1,0.5,0.2)),
         mu5=abs(rnorm(1,0.5,0.2)),mu6=abs(rnorm(1,0.5,0.2)),
         ks1=abs(rnorm(1,0.5,0.2)),ks2=abs(rnorm(1,0.5,0.2)),
         ks3=abs(rnorm(1,0.5,0.2)),ks4=abs(rnorm(1,0.5,0.2)),
         ks5=abs(rnorm(1,0.5,0.2)),ks6=abs(rnorm(1,0.5,0.2)),
         k1=abs(rnorm(1,0.5,0.2)),k2=abs(rnorm(1,0.5,0.2)),
         k3=abs(rnorm(1,0.5,0.2)),yc1=abs(rnorm(1,0.5,0.2)),
         yc2=abs(rnorm(1,0.5,0.2)),yc3=abs(rnorm(1,0.5,0.2)),
         yc4=abs(rnorm(1,0.5,0.2)),yc5=abs(rnorm(1,0.5,0.2)),
         yc6=abs(rnorm(1,0.5,0.2)),yc7=abs(rnorm(1,0.5,0.2)),
         yc8=abs(rnorm(1,0.5,0.2)),yc9=abs(rnorm(1,0.5,0.2)),
         yc10=abs(rnorm(1,0.5,0.2)),yc11=abs(rnorm(1,0.5,0.2)),
         yc12=abs(rnorm(1,0.5,0.2)),yc13=abs(rnorm(1,0.5,0.2)),
         yc14=abs(rnorm(1,0.5,0.2)),ev1=abs(rnorm(1,0.5,0.2)),
         ev2=abs(rnorm(1,0.5,0.2)),ev3=abs(rnorm(1,0.5,0.2)),
         sigma=abs(rnorm(1,0.5,0.2)))
  }
}
if (args[2]==145){
  ini = function(){
    list(mu1=abs(rnorm(1,0.5,0.2)),mu2=abs(rnorm(1,0.5,0.2)),
         mu3=abs(rnorm(1,0.5,0.2)),mu4=abs(rnorm(1,0.5,0.2)),
         mu5=abs(rnorm(1,0.5,0.2)),mu6=abs(rnorm(1,0.5,0.2)),
         mu7=abs(rnorm(1,0.5,0.2)),ks1=abs(rnorm(1,0.5,0.2)),
         ks2=abs(rnorm(1,0.5,0.2)),ks3=abs(rnorm(1,0.5,0.2)),
         ks4=abs(rnorm(1,0.5,0.2)),ks5=abs(rnorm(1,0.5,0.2)),
         ks6=abs(rnorm(1,0.5,0.2)),ks7=abs(rnorm(1,0.5,0.2)),
         k1=abs(rnorm(1,0.5,0.2)),k2=abs(rnorm(1,0.5,0.2)),
         k3=abs(rnorm(1,0.5,0.2)),yc1=abs(rnorm(1,0.5,0.2)),
         yc2=abs(rnorm(1,0.5,0.2)),yc3=abs(rnorm(1,0.5,0.2)),
         yc4=abs(rnorm(1,0.5,0.2)),yc5=abs(rnorm(1,0.5,0.2)),
         yc6=abs(rnorm(1,0.5,0.2)),yc7=abs(rnorm(1,0.5,0.2)),
         yc8=abs(rnorm(1,0.5,0.2)),yc9=abs(rnorm(1,0.5,0.2)),
         yc10=abs(rnorm(1,0.5,0.2)),yc11=abs(rnorm(1,0.5,0.2)),
         yc13=abs(rnorm(1,0.5,0.2)),yc14=abs(rnorm(1,0.5,0.2)),
         ev1=abs(rnorm(1,0.5,0.2)),ev2=abs(rnorm(1,0.5,0.2)),
         ev3=abs(rnorm(1,0.5,0.2)),sigma=abs(rnorm(1,0.5,0.2)))
  }
}
if (args[2]==234){
  ini = function(){
    list(mu1=abs(rnorm(1,0.5,0.2)),mu2=abs(rnorm(1,0.5,0.2)),
         mu3=abs(rnorm(1,0.5,0.2)),mu4=abs(rnorm(1,0.5,0.2)),
         mu5=abs(rnorm(1,0.5,0.2)),mu6=abs(rnorm(1,0.5,0.2)),
         mu7=abs(rnorm(1,0.5,0.2)),ks1=abs(rnorm(1,0.5,0.2)),
         ks2=abs(rnorm(1,0.5,0.2)),ks3=abs(rnorm(1,0.5,0.2)),
         ks4=abs(rnorm(1,0.5,0.2)),ks5=abs(rnorm(1,0.5,0.2)),
         ks6=abs(rnorm(1,0.5,0.2)),ks7=abs(rnorm(1,0.5,0.2)),
         k1=abs(rnorm(1,0.5,0.2)),k2=abs(rnorm(1,0.5,0.2)),
         k3=abs(rnorm(1,0.5,0.2)),yc1=abs(rnorm(1,0.5,0.2)),
         yc2=abs(rnorm(1,0.5,0.2)),yc3=abs(rnorm(1,0.5,0.2)),
         yc4=abs(rnorm(1,0.5,0.2)),yc5=abs(rnorm(1,0.5,0.2)),
         yc6=abs(rnorm(1,0.5,0.2)),yc7=abs(rnorm(1,0.5,0.2)),
         yc8=abs(rnorm(1,0.5,0.2)),yc9=abs(rnorm(1,0.5,0.2)),
         yc10=abs(rnorm(1,0.5,0.2)),yc11=abs(rnorm(1,0.5,0.2)),
         yc12=abs(rnorm(1,0.5,0.2)),yc13=abs(rnorm(1,0.5,0.2)),
         yc14=abs(rnorm(1,0.5,0.2)),yc15=abs(rnorm(1,0.5,0.2)),
         yc16=abs(rnorm(1,0.5,0.2)),yc17=abs(rnorm(1,0.5,0.2)),
         yc18=abs(rnorm(1,0.5,0.2)),yc19=abs(rnorm(1,0.5,0.2)),
         yc20=abs(rnorm(1,0.5,0.2)),sigma=abs(rnorm(1,0.5,0.2)))
  }
}
if (args[2]==235){
  ini = function(){
    list(mu1=abs(rnorm(1,0.5,0.2)),mu2=abs(rnorm(1,0.5,0.2)),
         mu3=abs(rnorm(1,0.5,0.2)),mu4=abs(rnorm(1,0.5,0.2)),
         mu5=abs(rnorm(1,0.5,0.2)),mu6=abs(rnorm(1,0.5,0.2)),
         mu7=abs(rnorm(1,0.5,0.2)),ks1=abs(rnorm(1,0.5,0.2)),
         ks2=abs(rnorm(1,0.5,0.2)),ks3=abs(rnorm(1,0.5,0.2)),
         ks4=abs(rnorm(1,0.5,0.2)),ks5=abs(rnorm(1,0.5,0.2)),
         ks6=abs(rnorm(1,0.5,0.2)),ks7=abs(rnorm(1,0.5,0.2)),
         k1=abs(rnorm(1,0.5,0.2)),k2=abs(rnorm(1,0.5,0.2)),
         k3=abs(rnorm(1,0.5,0.2)),yc1=abs(rnorm(1,0.5,0.2)),
         yc2=abs(rnorm(1,0.5,0.2)),yc3=abs(rnorm(1,0.5,0.2)),
         yc4=abs(rnorm(1,0.5,0.2)),yc5=abs(rnorm(1,0.5,0.2)),
         yc6=abs(rnorm(1,0.5,0.2)),yc7=abs(rnorm(1,0.5,0.2)),
         yc8=abs(rnorm(1,0.5,0.2)),yc9=abs(rnorm(1,0.5,0.2)),
         yc10=abs(rnorm(1,0.5,0.2)),yc11=abs(rnorm(1,0.5,0.2)),
         yc12=abs(rnorm(1,0.5,0.2)),yc13=abs(rnorm(1,0.5,0.2)),
         yc14=abs(rnorm(1,0.5,0.2)),yc15=abs(rnorm(1,0.5,0.2)),
         yc16=abs(rnorm(1,0.5,0.2)),yc17=abs(rnorm(1,0.5,0.2)),
         yc18=abs(rnorm(1,0.5,0.2)),yc19=abs(rnorm(1,0.5,0.2)),
         sigma=abs(rnorm(1,0.5,0.2)))
  }
}
if (args[2]==245){
  ini = function(){
    list(mu1=abs(rnorm(1,0.5,0.2)),mu2=abs(rnorm(1,0.5,0.2)),
         mu3=abs(rnorm(1,0.5,0.2)),mu4=abs(rnorm(1,0.5,0.2)),
         mu5=abs(rnorm(1,0.5,0.2)),mu6=abs(rnorm(1,0.5,0.2)),
         mu7=abs(rnorm(1,0.5,0.2)),mu8=abs(rnorm(1,0.5,0.2)),
         ks1=abs(rnorm(1,0.5,0.2)),ks2=abs(rnorm(1,0.5,0.2)),
         ks3=abs(rnorm(1,0.5,0.2)),ks4=abs(rnorm(1,0.5,0.2)),
         ks5=abs(rnorm(1,0.5,0.2)),ks6=abs(rnorm(1,0.5,0.2)),
         ks7=abs(rnorm(1,0.5,0.2)),ks8=abs(rnorm(1,0.5,0.2)),
         k1=abs(rnorm(1,0.5,0.2)),k2=abs(rnorm(1,0.5,0.2)),
         k3=abs(rnorm(1,0.5,0.2)),yc1=abs(rnorm(1,0.5,0.2)),
         yc2=abs(rnorm(1,0.5,0.2)),yc3=abs(rnorm(1,0.5,0.2)),
         yc4=abs(rnorm(1,0.5,0.2)),yc5=abs(rnorm(1,0.5,0.2)),
         yc6=abs(rnorm(1,0.5,0.2)),yc7=abs(rnorm(1,0.5,0.2)),
         yc8=abs(rnorm(1,0.5,0.2)),yc9=abs(rnorm(1,0.5,0.2)),
         yc10=abs(rnorm(1,0.5,0.2)),yc11=abs(rnorm(1,0.5,0.2)),
         yc12=abs(rnorm(1,0.5,0.2)),yc13=abs(rnorm(1,0.5,0.2)),
         yc14=abs(rnorm(1,0.5,0.2)),yc15=abs(rnorm(1,0.5,0.2)),
         yc16=abs(rnorm(1,0.5,0.2)),yc17=abs(rnorm(1,0.5,0.2)),
         yc18=abs(rnorm(1,0.5,0.2)),yc19=abs(rnorm(1,0.5,0.2)),
         sigma=abs(rnorm(1,0.5,0.2)))
  }
}
if (args[2]==345){
  ini = function(){
    list(mu1=abs(rnorm(1,0.5,0.2)),mu2=abs(rnorm(1,0.5,0.2)),
         mu3=abs(rnorm(1,0.5,0.2)),mu4=abs(rnorm(1,0.5,0.2)),
         mu5=abs(rnorm(1,0.5,0.2)),mu6=abs(rnorm(1,0.5,0.2)),
         mu7=abs(rnorm(1,0.5,0.2)),ks1=abs(rnorm(1,0.5,0.2)),
         ks2=abs(rnorm(1,0.5,0.2)),ks3=abs(rnorm(1,0.5,0.2)),
         ks4=abs(rnorm(1,0.5,0.2)),ks5=abs(rnorm(1,0.5,0.2)),
         ks6=abs(rnorm(1,0.5,0.2)),ks7=abs(rnorm(1,0.5,0.2)),
         k1=abs(rnorm(1,0.5,0.2)),k2=abs(rnorm(1,0.5,0.2)),
         k3=abs(rnorm(1,0.5,0.2)),yc1=abs(rnorm(1,0.5,0.2)),
         yc2=abs(rnorm(1,0.5,0.2)),yc3=abs(rnorm(1,0.5,0.2)),
         yc4=abs(rnorm(1,0.5,0.2)),yc5=abs(rnorm(1,0.5,0.2)),
         yc6=abs(rnorm(1,0.5,0.2)),yc7=abs(rnorm(1,0.5,0.2)),
         yc8=abs(rnorm(1,0.5,0.2)),yc9=abs(rnorm(1,0.5,0.2)),
         yc10=abs(rnorm(1,0.5,0.2)),yc11=abs(rnorm(1,0.5,0.2)),
         yc12=abs(rnorm(1,0.5,0.2)),yc13=abs(rnorm(1,0.5,0.2)),
         yc14=abs(rnorm(1,0.5,0.2)),yc15=abs(rnorm(1,0.5,0.2)),
         yc16=abs(rnorm(1,0.5,0.2)),sigma=abs(rnorm(1,0.5,0.2)))
  }
}
if (args[2]==1234){
  ini = function(){
    list(mu1=abs(rnorm(1,0.5,0.2)),mu2=abs(rnorm(1,0.5,0.2)),
         mu3=abs(rnorm(1,0.5,0.2)),mu4=abs(rnorm(1,0.5,0.2)),
         mu5=abs(rnorm(1,0.5,0.2)),mu6=abs(rnorm(1,0.5,0.2)),
         mu7=abs(rnorm(1,0.5,0.2)),ks1=abs(rnorm(1,0.5,0.2)),
         ks2=abs(rnorm(1,0.5,0.2)),ks3=abs(rnorm(1,0.5,0.2)),
         ks4=abs(rnorm(1,0.5,0.2)),ks5=abs(rnorm(1,0.5,0.2)),
         ks6=abs(rnorm(1,0.5,0.2)),ks7=abs(rnorm(1,0.5,0.2)),
         k1=abs(rnorm(1,0.5,0.2)),k2=abs(rnorm(1,0.5,0.2)),
         k3=abs(rnorm(1,0.5,0.2)),yc1=abs(rnorm(1,0.5,0.2)),
         yc2=abs(rnorm(1,0.5,0.2)),yc3=abs(rnorm(1,0.5,0.2)),
         yc4=abs(rnorm(1,0.5,0.2)),yc5=abs(rnorm(1,0.5,0.2)),
         yc6=abs(rnorm(1,0.5,0.2)),yc7=abs(rnorm(1,0.5,0.2)),
         yc8=abs(rnorm(1,0.5,0.2)),yc9=abs(rnorm(1,0.5,0.2)),
         yc10=abs(rnorm(1,0.5,0.2)),yc11=abs(rnorm(1,0.5,0.2)),
         yc12=abs(rnorm(1,0.5,0.2)),yc13=abs(rnorm(1,0.5,0.2)),
         yc14=abs(rnorm(1,0.5,0.2)),yc15=abs(rnorm(1,0.5,0.2)),
         yc16=abs(rnorm(1,0.5,0.2)),yc17=abs(rnorm(1,0.5,0.2)),
         yc18=abs(rnorm(1,0.5,0.2)),yc19=abs(rnorm(1,0.5,0.2)),
         yc20=abs(rnorm(1,0.5,0.2)),ev1=abs(rnorm(1,0.5,0.2)),
         ev2=abs(rnorm(1,0.5,0.2)),ev3=abs(rnorm(1,0.5,0.2)),
         sigma=abs(rnorm(1,0.5,0.2)))
  }
}
if (args[2]==1235){
  ini = function(){
    list(mu1=abs(rnorm(1,0.5,0.2)),mu2=abs(rnorm(1,0.5,0.2)),
         mu3=abs(rnorm(1,0.5,0.2)),mu4=abs(rnorm(1,0.5,0.2)),
         mu5=abs(rnorm(1,0.5,0.2)),mu6=abs(rnorm(1,0.5,0.2)),
         mu7=abs(rnorm(1,0.5,0.2)),ks1=abs(rnorm(1,0.5,0.2)),
         ks2=abs(rnorm(1,0.5,0.2)),ks3=abs(rnorm(1,0.5,0.2)),
         ks4=abs(rnorm(1,0.5,0.2)),ks5=abs(rnorm(1,0.5,0.2)),
         ks6=abs(rnorm(1,0.5,0.2)),ks7=abs(rnorm(1,0.5,0.2)),
         k1=abs(rnorm(1,0.5,0.2)),k2=abs(rnorm(1,0.5,0.2)),
         k3=abs(rnorm(1,0.5,0.2)),yc1=abs(rnorm(1,0.5,0.2)),
         yc2=abs(rnorm(1,0.5,0.2)),yc3=abs(rnorm(1,0.5,0.2)),
         yc4=abs(rnorm(1,0.5,0.2)),yc5=abs(rnorm(1,0.5,0.2)),
         yc6=abs(rnorm(1,0.5,0.2)),yc7=abs(rnorm(1,0.5,0.2)),
         yc8=abs(rnorm(1,0.5,0.2)),yc9=abs(rnorm(1,0.5,0.2)),
         yc10=abs(rnorm(1,0.5,0.2)),yc11=abs(rnorm(1,0.5,0.2)),
         yc12=abs(rnorm(1,0.5,0.2)),yc13=abs(rnorm(1,0.5,0.2)),
         yc14=abs(rnorm(1,0.5,0.2)),yc15=abs(rnorm(1,0.5,0.2)),
         yc16=abs(rnorm(1,0.5,0.2)),yc17=abs(rnorm(1,0.5,0.2)),
         yc18=abs(rnorm(1,0.5,0.2)),yc19=abs(rnorm(1,0.5,0.2)),
         ev1=abs(rnorm(1,0.5,0.2)),ev2=abs(rnorm(1,0.5,0.2)),
         ev3=abs(rnorm(1,0.5,0.2)),sigma=abs(rnorm(1,0.5,0.2)))
  }
}
if (args[2]==1245){
  ini = function(){
    list(mu1=abs(rnorm(1,0.5,0.2)),mu2=abs(rnorm(1,0.5,0.2)),
         mu3=abs(rnorm(1,0.5,0.2)),mu4=abs(rnorm(1,0.5,0.2)),
         mu5=abs(rnorm(1,0.5,0.2)),mu6=abs(rnorm(1,0.5,0.2)),
         mu7=abs(rnorm(1,0.5,0.2)),mu8=abs(rnorm(1,0.5,0.2)),
         ks1=abs(rnorm(1,0.5,0.2)),ks2=abs(rnorm(1,0.5,0.2)),
         ks3=abs(rnorm(1,0.5,0.2)),ks4=abs(rnorm(1,0.5,0.2)),
         ks5=abs(rnorm(1,0.5,0.2)),ks6=abs(rnorm(1,0.5,0.2)),
         ks7=abs(rnorm(1,0.5,0.2)),ks8=abs(rnorm(1,0.5,0.2)),
         k1=abs(rnorm(1,0.5,0.2)),k2=abs(rnorm(1,0.5,0.2)),
         k3=abs(rnorm(1,0.5,0.2)),yc1=abs(rnorm(1,0.5,0.2)),
         yc2=abs(rnorm(1,0.5,0.2)),yc3=abs(rnorm(1,0.5,0.2)),
         yc4=abs(rnorm(1,0.5,0.2)),yc5=abs(rnorm(1,0.5,0.2)),
         yc6=abs(rnorm(1,0.5,0.2)),yc7=abs(rnorm(1,0.5,0.2)),
         yc8=abs(rnorm(1,0.5,0.2)),yc9=abs(rnorm(1,0.5,0.2)),
         yc10=abs(rnorm(1,0.5,0.2)),yc11=abs(rnorm(1,0.5,0.2)),
         yc12=abs(rnorm(1,0.5,0.2)),yc13=abs(rnorm(1,0.5,0.2)),
         yc14=abs(rnorm(1,0.5,0.2)),yc15=abs(rnorm(1,0.5,0.2)),
         yc16=abs(rnorm(1,0.5,0.2)),yc17=abs(rnorm(1,0.5,0.2)),
         yc18=abs(rnorm(1,0.5,0.2)),yc19=abs(rnorm(1,0.5,0.2)),
         ev1=abs(rnorm(1,0.5,0.2)),ev2=abs(rnorm(1,0.5,0.2)),
         ev3=abs(rnorm(1,0.5,0.2)),sigma=abs(rnorm(1,0.5,0.2)))
  }
}

if (args[2]==1345){
  ini = function(){
    list(mu1=abs(rnorm(1,0.5,0.2)),mu2=abs(rnorm(1,0.5,0.2)),
         mu3=abs(rnorm(1,0.5,0.2)),mu4=abs(rnorm(1,0.5,0.2)),
         mu5=abs(rnorm(1,0.5,0.2)),mu6=abs(rnorm(1,0.5,0.2)),
         mu7=abs(rnorm(1,0.5,0.2)),ks1=abs(rnorm(1,0.5,0.2)),
         ks2=abs(rnorm(1,0.5,0.2)),ks3=abs(rnorm(1,0.5,0.2)),
         ks4=abs(rnorm(1,0.5,0.2)),ks5=abs(rnorm(1,0.5,0.2)),
         ks6=abs(rnorm(1,0.5,0.2)),ks7=abs(rnorm(1,0.5,0.2)),
         k1=abs(rnorm(1,0.5,0.2)),k2=abs(rnorm(1,0.5,0.2)),
         k3=abs(rnorm(1,0.5,0.2)),yc1=abs(rnorm(1,0.5,0.2)),
         yc2=abs(rnorm(1,0.5,0.2)),yc3=abs(rnorm(1,0.5,0.2)),
         yc4=abs(rnorm(1,0.5,0.2)),yc5=abs(rnorm(1,0.5,0.2)),
         yc6=abs(rnorm(1,0.5,0.2)),yc7=abs(rnorm(1,0.5,0.2)),
         yc8=abs(rnorm(1,0.5,0.2)),yc9=abs(rnorm(1,0.5,0.2)),
         yc10=abs(rnorm(1,0.5,0.2)),yc11=abs(rnorm(1,0.5,0.2)),
         yc12=abs(rnorm(1,0.5,0.2)),yc13=abs(rnorm(1,0.5,0.2)),
         yc14=abs(rnorm(1,0.5,0.2)),yc15=abs(rnorm(1,0.5,0.2)),
         yc16=abs(rnorm(1,0.5,0.2)),ev1=abs(rnorm(1,0.5,0.2)),
         ev2=abs(rnorm(1,0.5,0.2)),ev3=abs(rnorm(1,0.5,0.2)),
         sigma=abs(rnorm(1,0.5,0.2)))
  }
}

if (args[2]==2345){
  ini = function(){
    list(mu1=abs(rnorm(1,0.5,0.2)),mu2=abs(rnorm(1,0.5,0.2)),
         mu3=abs(rnorm(1,0.5,0.2)),mu4=abs(rnorm(1,0.5,0.2)),
         mu5=abs(rnorm(1,0.5,0.2)),mu6=abs(rnorm(1,0.5,0.2)),
         mu7=abs(rnorm(1,0.5,0.2)),mu8=abs(rnorm(1,0.5,0.2)),
         ks1=abs(rnorm(1,0.5,0.2)),ks2=abs(rnorm(1,0.5,0.2)),
         ks3=abs(rnorm(1,0.5,0.2)),ks4=abs(rnorm(1,0.5,0.2)),
         ks5=abs(rnorm(1,0.5,0.2)),ks6=abs(rnorm(1,0.5,0.2)),
         ks7=abs(rnorm(1,0.5,0.2)),ks8=abs(rnorm(1,0.5,0.2)),
         k1=abs(rnorm(1,0.5,0.2)),k2=abs(rnorm(1,0.5,0.2)),
         k3=abs(rnorm(1,0.5,0.2)),yc1=abs(rnorm(1,0.5,0.2)),
         yc2=abs(rnorm(1,0.5,0.2)),yc3=abs(rnorm(1,0.5,0.2)),
         yc4=abs(rnorm(1,0.5,0.2)),yc5=abs(rnorm(1,0.5,0.2)),
         yc6=abs(rnorm(1,0.5,0.2)),yc7=abs(rnorm(1,0.5,0.2)),
         yc8=abs(rnorm(1,0.5,0.2)),yc9=abs(rnorm(1,0.5,0.2)),
         yc10=abs(rnorm(1,0.5,0.2)),yc11=abs(rnorm(1,0.5,0.2)),
         yc12=abs(rnorm(1,0.5,0.2)),yc13=abs(rnorm(1,0.5,0.2)),
         yc14=abs(rnorm(1,0.5,0.2)),yc15=abs(rnorm(1,0.5,0.2)),
         yc16=abs(rnorm(1,0.5,0.2)),yc17=abs(rnorm(1,0.5,0.2)),
         yc18=abs(rnorm(1,0.5,0.2)),yc19=abs(rnorm(1,0.5,0.2)),
         yc20=abs(rnorm(1,0.5,0.2)),yc21=abs(rnorm(1,0.5,0.2)),
         sigma=abs(rnorm(1,0.5,0.2)))
  }
}

if (args[2]==12345){
  ini = function(){
    list(mu1=abs(rnorm(1,0.5,0.2)),mu2=abs(rnorm(1,0.5,0.2)),
         mu3=abs(rnorm(1,0.5,0.2)),mu4=abs(rnorm(1,0.5,0.2)),
         mu5=abs(rnorm(1,0.5,0.2)),mu6=abs(rnorm(1,0.5,0.2)),
         mu7=abs(rnorm(1,0.5,0.2)),mu8=abs(rnorm(1,0.5,0.2)),
         ks1=abs(rnorm(1,0.5,0.2)),ks2=abs(rnorm(1,0.5,0.2)),
         ks3=abs(rnorm(1,0.5,0.2)),ks4=abs(rnorm(1,0.5,0.2)),
         ks5=abs(rnorm(1,0.5,0.2)),ks6=abs(rnorm(1,0.5,0.2)),
         ks7=abs(rnorm(1,0.5,0.2)),ks8=abs(rnorm(1,0.5,0.2)),
         k1=abs(rnorm(1,0.5,0.2)),k2=abs(rnorm(1,0.5,0.2)),
         k3=abs(rnorm(1,0.5,0.2)),yc1=abs(rnorm(1,0.5,0.2)),
         yc2=abs(rnorm(1,0.5,0.2)),yc3=abs(rnorm(1,0.5,0.2)),
         yc4=abs(rnorm(1,0.5,0.2)),yc5=abs(rnorm(1,0.5,0.2)),
         yc6=abs(rnorm(1,0.5,0.2)),yc7=abs(rnorm(1,0.5,0.2)),
         yc8=abs(rnorm(1,0.5,0.2)),yc9=abs(rnorm(1,0.5,0.2)),
         yc10=abs(rnorm(1,0.5,0.2)),yc11=abs(rnorm(1,0.5,0.2)),
         yc12=abs(rnorm(1,0.5,0.2)),yc13=abs(rnorm(1,0.5,0.2)),
         yc14=abs(rnorm(1,0.5,0.2)),yc15=abs(rnorm(1,0.5,0.2)),
         yc16=abs(rnorm(1,0.5,0.2)),yc17=abs(rnorm(1,0.5,0.2)),
         yc18=abs(rnorm(1,0.5,0.2)),yc19=abs(rnorm(1,0.5,0.2)),
         yc20=abs(rnorm(1,0.5,0.2)),yc21=abs(rnorm(1,0.5,0.2)),
         ev1=abs(rnorm(1,0.5,0.2)),ev2=abs(rnorm(1,0.5,0.2)),
         ev3=abs(rnorm(1,0.5,0.2)),sigma=abs(rnorm(1,0.5,0.2))
    )
  }
}

#Stan call:

fit = stan(paste0("mm",args[2],".stan"),
           data=c("x0", "t0", "ts", "x", 'T','scl'),
           control=list(adapt_delta=as.numeric(args[3]),
                        stepsize=as.numeric(args[4]),
                        max_treedepth=as.numeric(args[5])),
           warmup = round(as.numeric(args[7])*1/3,0),
           init = ini,
           refresh = 5,
           cores = min(4, parallel::detectCores()),
           chains=4, iter=as.numeric(args[7]), seed=as.numeric(args[6]))

save(fit, file=paste0(lbl,"mm",args[2],".Rsave"))# saving output