#######################################################################
#                                                                     #
# R Script for plotting posterior predictions in main manuscript and  #
# supplementary material of:                                          #
#                                                                     #
# "Exploring cocoa bean fermentation processes by kinetic modelling"  #
#                                                                     #
# Code made by Mauricio Moreno-Zambrano                               #
# 2021                                                                #
#                                                                     #
#######################################################################

#loading necessary R packages

library(cowplot)
library(ggplot2)
library(dplyr)
library(tidyr)
library(stringr)

setwd("/home")      #set working directory accordingly
dpath<- '/home/'    #character string with the directory where Stan outputs are saved
outpth <- '/home/'  #character string with the directory where plots are going to be saved  


########################################################################
# FIGURES S1 TO S23 OF SUPPLEMENTARY MATERIAL AND FIGURE 2 OF MAIN     #
# MANUSCRIPT                                                           #
#                                                                      #
# explanation of functions:                                            #
#                                                                      #
# swf1 = plotting function for datasets that did not require LA's time #
#        series to be generated. Arguments:                            #
#        dsn = dataset assigned order (order is same as shown in Table #
#             1 of main manuscript).                                   #
#        fitnam = name of Rsave file where Stan output is stored.      #
#        mm = model iteration code (without commas).                   #
#        bh = base hight of the plot                                   #
#        bw = base width of the plot                                   #
#        lp = right plot margin                                        #
#                                                                      #
# swf2 = plotting function for datasets that required LA's time ser-   #
#        ies to be generated. Arguments are the same as for swf        #
#                                                                      # 
# Example:                                                             #
# swf1(1,"mywb3_mm23.Rsave",'23',8.3,17.2,1) #generates Figure 2 of    #
#                                            #main manuscript          #
#                                                                      #
########################################################################

swf1 <- function(dsn,fitnam,mm,bh,bw,lp){
  if (dsn==1){
    T = 16
    x0 = c(51.963,57.741,0,0,0,0.029180401,0.007868827,3.36634E-06)
    t0 = 0
    ts = c(6,12,18,24,30,36,42,48,54,60,66,72,84,96,120,144)
    x = structure(c(43.884,24.25,26.098,11.871,4.607,1.714,4.821,5.409,3.701,3.696,1.544,4.428,1.086,2.337,2.837,3.486,
                    49.588,35.584,33.876,19.871,8.607,2.825,4.821,7.409,4.812,7.548,1.544,5.539,1.086,2.337,2.837,3.486,
                    2.97,9.081,9.62,6.067,11.211,16.213,10.215,20.106,22.492,5.637,5.636,13.85,4.782,9.3,4.267,7.448,
                    0.267,0.801,1.256,1.745,5.944,3.246,7.787,7.991,6.943,8.593,8.296,6.85,8.021,7.667,7.209,8.117,
                    0.152,1.616,0.979,4.033,5.121,2.759,3.541,4.278,4.141,6.139,3.219,5.319,4.328,3.621,4.208,5.93,
                    0.026187332,0.955193281,0.238831309,0.045299276,0.01021154,0.06608323,0.008262115,0.000554742,0.000267973,0.006013001,0.002203389,0.000375916,0.000889388,0.000948618,0.000110177,0.000105461,
                    0.067128975,0.07959944,0.248834167,0.20179482,0.416783016,0.401707567,0.566121975,0.386286929,0.372314554,0.199484893,0.167459586,0.161402409,0.14484717,0.008748025,0.006575216,0.00181514,
                    9.82105E-06,1.70278E-05,2.16351E-05,5.49741E-05,0.00010572,0.000177484,0.000259512,0.000681017,0.000364887,0.00118075,0.001914953,0.000676329,0.001127608,0.000237224,6.32642E-05,2.64337E-05),
                  .Dim=c(16,8))
    lbl <- 'ghhp1'
  } 
  
  if (dsn==2){
    T=6
    x0 = c(56.909,88.795,0,0,0,0.022808213,1.25E-08,0.00028)
    t0 = 0
    ts = c(24,48,72,96,120,144)
    x = structure(c(50.266,20.771,15.19,13.95,1.284,1.815,
                    70.992,40.877,28.034,29.008,12.621,12.09,
                    1.003,9.702,0.972,0,0,0,
                    1.178,0.486,1.551,1.411,2.25,0,
                    0,6.488,22.472,11.306,5.842,5.375,
                    0.957395229,0.120528918,0.022084688,0.007604861,0.00000015,0.00000015,
                    0.001096251,0.092877392,0.053692053,0.011295618,0.006381312,0.001621474,
                    0.017871378,0.045098078,0.020613799,0.002530219,0.000982105,0.000156012),
                  .Dim=c(6,8))
    lbl <- 'dowb1'
  } 
  
  if (dsn==3){
    T = 13
    x0 = c(55.482, 49.669, 0, 0, 1.379, 0.175424909,0.003047264,2.86522E-06)
    t0 = 0
    ts = c(6,12,24,30,36,48,54,60,72,84,96,120,144)
    
    x = structure(c(26.802,16.311,13.706,13.248,7.54,1.278,2.694,3.08,0.757,2.465,2.673,3.183,7.631,
                    31.959,18.937,19.238,15.217,10.351,0.809,1.007,1.205,0.757,1.996,2.673,3.183,2.287,
                    0.201,2.324,2.46,1.996,3.173,4.025,6.033,5.685,3.605,1.617,0.576,0.964,1.56,
                    0.148,0.342,0.548,1.033,0.658,4.873,3.194,2.694,2.72,1.928,2.482,1.883,1.63,
                    0.963,1.216,1.467,1.996,1.211,3.333,6.033,4.323,8.685,5.658,11.935,11.354,11.511,
                    0.119149235,0.212369067,0.068721283,0.045508368,0.052250597,0.035488795,0.020468747,0.03750518,0.004509114,0.00048539,0.000190586,2.91133E-06,1.45912E-06,
                    0.006575216,0.02528774,0.37403308,0.364678377,0.36383964,0.073945204,0.019182712,0.027410062,0.001068833,0.00101137,0.023873166,0.00458047,0.000150977,
                    7.71184E-06,7.09836E-06,1.0427E-05,7.90966E-06,1.10194E-05,6.05561E-05,0.000127104,0.000700097,0.000130665,0.000127104,0.005347589,0.008514478,3.38188E-05),
                  .Dim=c(13,8))
    lbl <- 'brwb1'
  } 
  
  if (dsn==4){
    T = 13
    x0 = c(42.936,67.249,0.318,0,0.942,0.004754351,0.000221264,3.86508E-06)
    t0 = 0
    ts = c(6,12,24,30,36,48,54,60,72,84,96,120,144)
    
    x = structure(c(23.452,8.264,4.85,1.27,1.62,2.32,0.294,1.467,1.892,5.975,7.863,6.521,7.19,
                    21.349,14.478,10.974,2.641,0.889,4.057,1.756,1.467,3.812,8.351,9.416,9.081,9.109,
                    0.386,2.211,4.496,4.912,6.714,4.307,7.057,4.097,1.044,1.642,0.9,0.341,0.29,
                    0.165,0.679,1.75,0.393,0.823,0.992,1.161,0.968,1.609,4.123,2.476,2.565,1.642,
                    1.079,1.194,2.532,1.329,2.183,1.718,2.272,2.733,6.568,11.767,9.083,18.625,17.187,
                    0.069998907,0.243271515,0.037852212,0.022340416,0.038823194,0.015491421,0.055474227,0.000444725,0.001542024,7.90845E-05,0.00016676,0.000144574,9.48618E-06,
                    0.002753658,0.026786133,0.016669018,0.051749959,0.125866459,0.54564479,0.699697002,0.05151219,0.11800761,0.013928682,0.024149604,0.002923547,0.001451811,
                    8.37834E-06,1.41957E-05,2.74889E-05,4.53062E-05,9.06062E-06,3.61541E-05,0.002723692,0.001033137,0.001436012,6.57897E-05,0.026987213,0.00468984,9.33594E-06),
                  .Dim=c(13,8))
    lbl <- 'brwb2'
  } 
  
  if (dsn==5){
    T = 14
    x0 = c(45.93,53.933,0.321,0,1.048,0.000944259,0.0008628,9.29304E-06)
    t0 = 0
    ts = c(6,12,18,24,30,36,42,48,54,60,66,72,84,96)
    
    x = structure(c(49.119,47.058,36.219,18.238,6.712,14.375,4.655,3.541,7.934,3.378,0,0.805,2.018,1.769,
                    43.353,36.559,25.893,30.888,6.97,14.805,4.655,3.541,7.934,10.004,2.263,2.87,0.813,2.371,
                    0.575,0.871,2.517,3.562,4.073,10.187,12.537,12.406,2.632,6.073,5.044,5.256,0.932,0.757,
                    0.276,0.744,1.17,5.829,3.346,2.361,5.224,6.034,3.937,4.319,2.948,1.792,2.515,1.826,
                    1.046,1.642,2.217,2.921,2.021,1.334,1.717,1.116,2.312,2.523,2.222,2.048,3.413,3.922,
                    0.001577943,0.021042206,0.185392115,0.030980702,0.051771561,0.153847789,0.342839821,0.159254334,0.9270246,0.7657575,1.05460848,1.094186265,0.046141452,0.408405196,
                    0.002264175,0.212769814,0.131798362,0.489677346,0.55707031,0.084705188,0.004334211,0.032501995,0.156282379,0.061788836,0.047963406,0.093952862,0.007077991,0.038628693,
                    0.00017067,7.45003E-05,8.2065E-05,4.33669E-05,0.000189303,0.00096863,0.00042185,0.000436675,0.008038186,0.006639846,0.01734435,0.016832865,0.001268113,0.007873322),
                  .Dim=c(14,8))
    lbl <- 'ecpt1'
  } 
  
  if (dsn==6){
    T = 13
    x0 = c(35.801,36.408,0.554,0,2.987,7.38059E-05,8.37356E-05,4.17021E-07)
    t0 = 0
    ts = c(6,12,18,24,30,36,42,48,54,60,72,84,96)
    
    x = structure(c(42.651,27.468,18.88,7.776,10.549,9.333,7.429,2.737,2.474,1.952,1.167,0,0,
                    38.921,37.444,19.573,6.475,9.595,6.471,11.499,3.777,3.515,2.992,4.549,0.815,2.806,
                    0.639,1.332,7.369,4.76,4.019,8.014,7.099,9.617,3.576,1.879,0.224,1.565,0.302,
                    0.378,0,1.634,0.718,0.934,1.193,0.495,1.145,1.316,0.793,0.224,1.262,0.823,
                    3.81,3.2,5.37,3.717,3.628,4.19,4.797,9.617,11.613,15.869,17.298,17.337,7.905,
                    0.011697452,0.654773748,0.338915366,0.871146626,0.218318862,0.046783344,0.239933704,0.033043897,0.048316032,0.060130008,0.001425907,0.240486809,0.034283982,
                    0.000608009,0.169398677,0.174948415,0.043843984,0.007531995,0.048073973,0.06207404,0.046763824,0.099519919,0.154849573,0.031110716,0.299854115,0.082396737,
                    0.000218352,0.002307587,0.002168493,0.003476626,0.000929304,0.000239971,0.001122427,0.000700097,0.003484641,0.006747735,0.000362375,0.019686025,0.005783064),
                  .Dim=c(13,8))
    lbl <- 'ecpt2'
  } 
  
  if (dsn==7){
    T = 14
    x0 = c(42.083,53.899,0.889,0,0.889,0.005047674,0.000990627,3.10569E-05)
    t0 = 0
    ts = c(6,12,18,24,30,36,42,48,54,60,66,72,84,96)
    
    x = structure(c(40.032,30.959,22.229,28.397,26.687,1.175,1.607,5.976,0.757,3.072,1.791,0.681,0.773,1.121,
                    43.799,36.867,26.082,30.623,29.513,3.401,4.432,12.484,3.839,8.209,9.754,2.393,1.971,2.749,
                    0.889,1.257,1.912,2.362,4.282,17.847,13.354,14.744,12.129,12.538,11.763,6.737,3.552,5.146,
                    0,0.604,0.645,2.688,1.055,3.261,0.729,2.527,1.71,3.631,5.062,3.387,2.04,1.796,
                    0.889,1.257,1.912,3.587,2.24,3.588,1.587,2.527,2.282,4.367,6.982,6.737,7.842,6.29,
                    0.002080134,0.01049763,0.103297844,0.058491298,0.172223043,0.841571964,0.189710452,0.269210044,0.02831987,0.025008708,0.003076743,0.009979097,1.52087E-06,6.87213E-05,
                    0.002192351,0.090138435,0.248834167,0.331825695,0.64700854,0.283080538,0.332590632,0.242610735,0.113477566,0.026357852,0.090972476,0.088289694,0.066513532,0.024940779,
                    0.000147285,0.002723692,0.000899825,3.02105E-05,0.001127608,0.000464684,6.09759E-05,2.85206E-05,0.000280645,0.000262517,9.25035E-05,0.000531078,6.87318E-06,7.34781E-06),
                  .Dim=c(14,8))
    lbl <- 'ecwb1'
  } 
  
  if (dsn==8){
    T = 14
    x0 = c(36.674,32.884,0,0,3.345,0.000290463,0.000479634,1.22789E-05)
    t0 = 0
    ts = c(6,12,18,24,30,36,42,48,54,60,66,72,84,96)
    
    x = structure(c(14.472,16.459,8.373,1.213,5.758,0.956,2.554,6.846,1.792,4.988,12.314,0.69,2.286,5.229,
                    16.377,14.943,7.531,4.918,10.726,1.882,4.07,5.667,4.234,6.926,6.503,2.038,5.318,6.408,
                    0.199,0.616,0.909,2.609,4.021,6.633,12.517,7.798,8.339,4.488,4.492,3.127,1.972,1.605,
                    0,0.202,0.619,2.03,2.861,4.106,3.943,4.567,4.445,5.152,3.58,2.63,1.849,3.303,
                    2.726,2.936,3.601,5.302,5.263,7.378,8.209,11.69,10.492,11.779,13.232,17.418,20.447,24.097,
                    0.001233364,0.114048942,0.188404495,0.165993568,0.227557555,0.515336922,0.662355671,0.013870473,0.004614145,1.61098E-05,7.55251E-06,0.000339697,4.13134E-05,3.64831E-05,
                    0.0014385,0.13737573,0.200405674,0.375759538,0.498781128,0.64107673,0.439450551,0.200405674,0.257578739,0.017820095,0.053445361,0.035555764,0.022177369,0.001119206,
                    0.001368227,0.003405321,0.00075363,0.00080197,0.000853411,0.00096863,0.005459565,0.00096863,0.000228117,1.78714E-05,8.95691E-06,0.004967731,3.35087E-05,7.58854E-05),
                  .Dim=c(14,8))
    lbl <- 'ecwb2'
  } 
  
  if (dsn==9){
    T = 12
    x0 = c(40.008,35.371,0,2.021,0,0.002851617,0.001570037,2.8E-10)
    t0 = 0
    ts = c(12,24,36,48,60,72,84,96,108,120,132,144)
    
    x = structure(c(102.342,23.407,1.09,5.923,2.81,9.85,11.593,13.556,10.443,11.303,10.948,1.986,
                    95.058,24.952,10.14,11.331,6.121,6.98,6.516,13.556,15.189,7.992,6.866,1.324,
                    43.93,45.732,65.326,20.521,26.127,27.8,33.277,19.801,19.218,25.082,21.597,23.529,
                    23.545,22.156,31.425,20.922,18.4,16.24,15.313,12.533,9.445,6.819,6.306,6.153,
                    2.874,3.091,12.608,42.164,55.501,68.726,60.363,21.663,19.943,18.665,18.383,19.984,
                    0.548392187,0.088123403,0.002046875,0.013806744,0.004809404,0.000553466,2.61873E-05,0.000110431,0.000218319,0.000000015,0.000000015,0.000000015,
                    0.149592566,0.335668056,0.070943076,0.086279975,0.039986189,0.020319359,0.008768191,0.005739975,0.039710926,0.097478764,0.123853993,0.074115666,
                    0.000604168,0.000324458,0.00024955,4.76604E-05,9.73101E-06,7.83715E-06,1.13281E-06,8.51448E-07,2.8E-10,2.8E-10,2.8E-10,2.8E-10),
                  .Dim=c(12,8))
    lbl <- 'brpb1'
  } 
  
  if (dsn==10){
    T = 12
    x0 = c(35.371,30.269,0,2.021,0,0.001718269,0.000176161,2.8E-10)
    t0 = 0
    ts = c(12,24,36,48,60,72,84,96,108,120,132,144)
    
    x = structure(c(132.472,15.902,1.09,0.184,6.121,6.98,6.516,10.687,10.443,10.2,9.734,1.986,
                    122.539,21.199,1.09,5.923,2.81,2.565,5.192,8.038,5.587,4.902,3.113,3.532,
                    41.931,33.29,56.753,78.152,50.043,51.845,60.48,50.035,64.859,64.726,53.637,46.156,
                    8.124,8.794,14.742,18.734,15.98,9.571,14.437,17.039,20,23.476,18.843,19.206,
                    0,0,0,0.369,11.215,11.099,10.43,11.09,12.801,12.963,16.557,18.71,
                    0.242712006,0.732978539,0.055602108,0.016447173,0.005985374,0.015384779,0.000000015,0.000000015,0.000000015,0.000000015,0.000000015,0.000000015,
                    0.065299524,0.076897109,0.065299524,0.111406367,0.070292666,0.062648404,0.01729458,0.106637514,0.004455639,0.004569935,0.006164673,0.003801106,
                    3.76841E-05,8.26339E-05,0.000202376,0.000198225,2.33431E-05,4.56203E-07,3.6321E-07,7.67641E-07,2.8E-10,2.8E-10,2.8E-10,2.8E-10),
                  .Dim=c(12,8))
    lbl <- 'brst1'
  } 
  
  if (dsn==11){
    T = 12
    x0 = c(146.589,162.153,0,0,0,0.001706441,0.001610312,5.07175E-05)
    t0 = 0
    ts = c(12,24,36,48,60,72,84,96,108,120,132,144)
    
    x = structure(c(49.233,26.151,38.968,23.108,23.34,27.285,24.629,33.114,23.237,13.565,13.798,13.617,
                    26.664,56.027,43.591,40.548,41.262,56.771,55.137,44.11,37.309,34.031,31.223,30.059,
                    63.627,24.995,26.887,36.165,26.28,12.701,34.656,18.482,18.079,15.479,14.577,13.475,
                    14.478,14.172,15.188,16.944,12.729,10.146,9.809,6.579,6.98,6.951,6.768,6.462,
                    0,16.799,11.709,14.24,14.076,22.226,23.217,25.593,26.276,31.27,32.107,36.177,
                    1.892741302,0.469992859,0.003476092,0.020705764,0.004966967,0.000604076,4.08405E-05,0.000636929,0.001445744,0.001145754,0.001343047,0.001318534,
                    0.102780331,0.093952862,0.08748025,0.124712508,0.06900968,0.01119206,0.008277706,0.016289585,0.008239674,0.07143483,0.015132477,0.117195251,
                    0.00084949,0.000401013,0.000335087,0.000169886,1.34945E-05,3.28971E-05,9.77593E-07,1.99599E-06,1.47285E-06,4.29693E-07,1.72647E-06,0),
                  .Dim=c(12,8))
    lbl <- 'brwb3'
  } 
  
  if (dsn==12){
    T = 12
    x0 = c(132.972,149.002,0,0,0,0.002773903,0.001509767,1.68717E-05)
    t0 = 0
    ts = c(12,24,36,48,60,72,84,96,108,120,132,144)
    
    x = structure(c(93.798,70.922,23.494,5.777,17.769,21.921,18.646,23.417,17.047,16.248,16.48,16.093,
                    98.053,39.823,2.964,52.994,46.664,53.014,51.849,50.215,49.756,44.834,48.601,47.202,
                    23.901,24.995,28.085,40.058,47.839,34.061,25.871,18.28,18.079,15.479,14.577,15.571,
                    10.847,13.68,22.05,19.374,12.083,12.054,11.194,10.609,9.565,8.613,8.214,7.754,
                    2.184,16.799,21.793,24.554,24.851,26.921,23.217,22.976,23.428,20.186,21.1,22.014,
                    0.540867965,0.088734245,0.002032784,0.013554742,0.004571842,0.000533447,2.47224E-05,0.000108415,0.00021582,0.001241913,0.00128852,0.001054608,
                    0.142859792,0.340337664,0.068062832,0.086081537,0.038806995,0.019902609,0.008509617,0.00522288,0.037317283,0.090346225,0.116120798,0.075319948,
                    0.000613985,0.000332015,0.00024387,4.54107E-05,9.44404E-06,7.45003E-06,1.11727E-06,8.83401E-07,0,0,0,0),
                  .Dim=c(12,8))
    lbl <- 'brwb4'
  } 
  
  if (dsn==13){
    T = 12
    x0 = c(32.082,36.517,0,0,0,0.000141283,1.33632E-05,2.8E-10)
    t0 = 0
    ts = c(12,24,36,48,60,72,84,96,108,120,132,144)
    
    x = structure(c(122.888,56.893,1.624,3.301,28.498,7.891,7.092,6.705,6.73,8.408,9.259,8.046,
                    136.565,59.55,2.964,2.74,4.628,19.667,11.223,12.172,14.531,13.601,11.262,10.333,
                    36.877,64.62,71.004,65.411,53.429,53.225,46.833,28.863,12.888,15.479,18.27,18.865,
                    12.201,19.464,23.773,21.313,22.914,19.592,20.147,13.441,15.042,14.367,9.753,8.277,
                    0,0,2.242,21.937,23.005,25.305,36.071,37.139,36.514,46.665,56.968,57.421,
                    0.071464648,0.308383589,0.011616927,0.002254713,0.00067157,0.001884045,3.63154E-06,5.70284E-06,9.20643E-06,0.000000015,0.000000015,0.000000015,
                    0.028438718,0.036551905,0.028308054,0.057399752,0.031326366,0.026909772,0.004964894,0.051749959,0.000916031,0.000891066,0.0012999,0.000641077,
                    2.8E-10,7.81912E-06,2.67398E-05,2.41634E-05,1.66785E-06,1.02367E-08,2.8E-10,2.8E-10,2.8E-10,2.8E-10,2.8E-10,2.8E-10),
                  .Dim=c(12,8))
    lbl <- 'brst2'
  } 
  
  if (dsn==14){
    T = 6
    x0 = c(15.694,18.579,0.954,0.177,0.154,0.00015358,5.03677E-06,1.12824E-06)
    t0 = 0
    ts = c(24,48,72,96,120,144)
    
    x = structure(c(8.316,2.48,0,0,0,0,
                    10.814,7.022,0.255,0.369,0.778,1.05,
                    2.359,9.774,6.964,4.552,1.175,0.92,
                    0.479,0.978,1.001,0.595,0.142,0.142,
                    0.305,0.351,0.479,1.117,1.999,3.438,
                    0.0101703,0.000148025,0.00164943,0.000120874,3.80482E-06,1.8044E-06,
                    0.000126227,0.00022759,0.001409784,0.00017545,2.30756E-06,7.62347E-06,
                    6.49233E-05,2.04833E-05,0.000303669,3.93007E-05,1.57544E-06,1.70766E-06),
                  .Dim=c(6,8))
    lbl <- 'brwb7'
  } 
  
  if (dsn==15){
    T = 12
    x0 = c(39.826,51.835,0.396,0,0,0.002911329,0.000677501,2.12891E-05)
    t0 = 0
    ts = c(6,12,24,30,36,48,54,60,72,84,96,120)
    
    x = structure(c(39.926,34.367,31.39,1.613,1.117,0.422,0.918,0.223,1.414,0.72,1.315,1.712,
                    45.98,38.734,18.983,4.591,3.002,1.017,1.911,1.216,1.414,0.72,2.407,2.903,
                    4.511,6.952,12.576,16.649,14.294,11.367,17.225,13.792,11.034,6.693,10.532,10.569,
                    0.776,1.456,5.347,7.529,7.301,9.132,9.184,8.519,6.702,8.202,8.374,11.932,
                    0.76,0.92,3.73,4.463,3.64,4.394,5.354,5.24,7.046,8.554,7.16,9.926,
                    0.053099601,0.084935893,0.161842008,0.157431364,0.258280286,0.007692921,0.028780031,0.001285557,0.000333496,0.000123336,4.61415E-06,1.56348E-06,
                    0.010663751,0.02883434,0.044968667,1.348683403,0.948221969,0.149592566,0.311824341,0.16364774,0.173744079,0.173744079,0.064107673,0.776086293,
                    0.000186276,0.000147285,5.59961E-05,1.68717E-05,0.002193603,0.000604168,0.003405321,0.005783064,0.004711487,0.013157035,0.001026025,0.003524991),
                  .Dim=c(12,8))
    lbl <- 'mywb3'
  } 
  
  if (dsn==16){
    T=5
    x0 = c(43.555,53.603,6.28,0.525,0,0.258875684,0.00739452,2.88508E-05)
    t0 = 0
    ts = c(24,48,72,96,120)
    x = structure(c(22.277,6.959,3.22,3.23,2.726,
                    27.385,8.831,6.286,4.846,5.96,
                    21.863,13.726,6.226,6.102,0,
                    2.849,4.887,2.728,3.621,2.576,
                    2.467,10.355,14.302,13.924,11.002,
                    0.144574354,0.029722905,0.009399208,0.000729611,0.000110431,
                    0.222284926,0.339554909,0.100440765,0.014754008,0.021033426,
                    0.000227069,0.001368227,0.010096202,0.00279356,0.001968602),
                  .Dim=c(5,8))
    
    lbl<-'hnwb1'
  } 
  
  if (dsn==17){
    T = 6
    x0 = c(29.894,36.631,0,0.718,0,5.67981E-05,1.25E-09,2.8E-10)
    t0 = 0
    ts = c(24,48,72,96,120,144)
    
    x = structure(c(28.244,0.638,0,0,0,0,
                    36.401,0.638,0,0,0,0,
                    1.264,19.898,5.967,3.571,0,0,
                    0.776,0.621,3.767,6.698,8.78,11.819,
                    0.776,1.1,8.292,16.335,20.333,24.702,
                    0.998467121,1.353112071,4.095753099,73.0018336,0.000000015,0.000000015,
                    7.19702E-07,0.00349263,3.615368106,27.0490815,0.040378718,0.001079326,
                    1.61213E-07,0.000782349,0.809842456,6.058994256,0.009044833,0.000241769),
                  .Dim=c(6,8))
    lbl <- 'ecpb1'
  } 
  
  if (dsn==18){
    T = 5
    x0 = c(20.293,27.875,0,0,0,0.670025388,6.26484E-06,5.85003E-07)
    t0 = 0
    ts = c(24,48,72,96,120)
    
    x = structure(c(24.351,8.802,6.819,8.028,5.133,
                    31.25,9.999,11.037,12.359,8.667,
                    13.738,17.412,12.924,7.619,1.592,
                    3.188,3.943,4.98,5.672,2.911,
                    1.932,1.871,7.021,9.063,15.846,
                    0.903839379,0.023773398,0.071794514,0.000248938,1.43249E-05,
                    0.016103119,0.002494078,0.008849322,0.00013706,0.002863585,
                    4.86584E-06,0.000585003,0.002028421,0.00024387,0.004755082),
                  .Dim=c(5,8))
    lbl <- 'niwb1'
  } 
  
  if (dsn==19){
    T = 6
    x0 = c(20.684,23.663,0,0,0,0.030626069,9.48222E-05,1.27985E-05)
    t0 = 0
    ts = c(24,48,72,96,120,144)
    
    x = structure(c(21.418,8.099,2.021,2.73,3.367,1.952,
                    25.533,14.058,2.021,2.73,4.076,1.952,
                    6.451,17.029,18.745,9.088,6.255,2.051,
                    2.412,5.069,7.765,5.833,4.333,5.776,
                    0,1.382,2.275,4.226,6.882,8.443,
                    0.328164244,1.164370675,0.065477375,0.001191492,7.17945E-05,4.52993E-06,                
                    0.001435192,0.119374073,0.202726262,0.004044921,0.057136024,0.001891952,
                    7.03328E-05,0.000140332,0.007197108,0.003070139,0.027362642,0.000906062),
                  .Dim=c(6,8))
    lbl <- 'niwb2'
  } 
  
  x1 = rbind(x0,x)
  scl = c(max(x1[,1]),max(x1[,2]),max(x1[,3]),max(x1[,4]),max(x1[,5]),max(x1[,6]),max(x1[,7]),max(x1[,8]))
  
  load(paste0(dpath,fitnam))
  
  pred = as.data.frame(fit, pars = "x_hat") %>%
    gather(factor_key = TRUE) %>%
    group_by(key) %>%
    summarize(lb = quantile(value, probs = 0.025),
              median = quantile(value, probs = 0.5),
              ub = quantile(value, probs = 0.975)) 
  
  sp = split(pred,rep(1:8,each=T))
  
  metb <- c('(a)','(b)','(c)','(d)','(e)','(f)','(g)','(h)')
  
  for (i in 1:8){
    sp[i] = lapply(sp[i],"*",scl[i])
    sp[i] = mapply(cbind, sp[i], "obs"=list(x[,i]), SIMPLIFY=F)
    sp[i] = mapply(cbind, sp[i], "time"=list(ts), SIMPLIFY=F)
    sp[i] = mapply(rbind,list(c(NA,x0[i],x0[i],x0[i],x0[i],0)),sp[i],SIMPLIFY=F)
    sp[i] = mapply(cbind, sp[i], "met"=metb[i],SIMPLIFY = F)
    sp[i] = mapply(cbind, sp[i], "data"=lbl,SIMPLIFY= F)
    sp[i] = mapply(cbind, sp[i], "model"=mm, SIMPLIFY = F)
  }
  
  dfout <- do.call(rbind.data.frame,sp)
  
  p1 <- ggplot(data=dfout,aes(x=time,y=median))+
    geom_line(color="#D43F3AFF",size=0.7)+
    geom_point(aes(x=time,y=obs))+
    geom_line(aes(x = time, y = ub),color='#EEA2367F',linetype='dashed',size=1) +
    geom_line(aes(x = time, y = lb),color='#EEA2367F',linetype='dashed',size=1)+
    geom_ribbon(aes(ymin = lb, ymax = ub), fill = "#EEA2367F",alpha=0.3)+
    facet_wrap(~met,scales = 'free_y',ncol=4)+
    theme_bw()+
    theme(strip.text.x = element_text(hjust = -0.01),
          panel.grid.major = element_line(colour = "grey90",linetype='dashed'),
          panel.grid.minor = element_blank(),
          strip.background = element_blank(),
          plot.margin = unit(c(0,lp,0,0),'lines'),
          strip.text = element_text(size=16),
          axis.text=element_text(size=14),
          axis.title=element_text(size=14)
    )+
    xlab('time (h)')+
    ylab(expression(paste('mg g(pulp) '^{-1})))

  return(save_plot(paste0(outpth,lbl,'.pdf'),p1,base_height = bh,base_width = bw))
  
} 

swf2 <- function(dsf,fitnam,mm,bh,bw,lp){
  if (dsf==1){
    T = 15
    xi_r = c(47.956,51.386,0.286,0,0.048539049,0.004044921,0.000906062)
    t0 = 0
    ts = c(6,12,18,24,30,36,42,48,54,60,72,84,96,120,144)
    
    x = structure(c(34.158,24.028,12.064,7.996,7.438,4.885,1.854,1.615,2.094,2.094,3.131,2.333,3.21,5.922,0.259,
                    41.575,23.789,15.972,7.916,12.303,6.64,3.529,3.45,3.45,3.529,5.444,4.088,4.806,7.278,2.014,
                    1.883,5.389,3.098,10.491,4.833,4.799,1.501,3.896,2.751,1.883,1.467,0.564,1.189,1.501,0.148,
                    0,1.536,1.605,1.64,3.827,4.07,4.833,11.359,9.623,6.465,15.802,11.289,9.207,11.845,5.632,
                    0.278670668,0.394540199,0.508266234,0.189710452,0.013934496,0.021090713,0.013934496,0.000478731,0.000244958,0.000000015,0.000000015,0.000000015,0.000000015,0.000000015,0.000000015,
                    0.032954142,0.03287835,0.024712121,0.015809204,0.141550045,0.333357333,0.402633599,0.061646725,0.322040145,0.003740331,0.014220341,0.04177438,0.00020602,0.00018068,0.000126739,
                    0.001880001,0.00736475,0.001816176,0.002495503,0.00136194,0.000432671,0.000260111,0.000228643,0.000925035,0.000313443,0.001349454,0.000628287,0.000364887,0.000483235,0.000450981#aab
    ),
    .Dim=c(15,7))
    lbl <- 'ghhp2'
  } 
  
  
  if (dsf==2){
    T = 15
    xi_r = c(40.246,50.851,0,0,0.059304993,0.014618742,0.001107027)
    t0 = 0
    ts = c(6,12,18,24,30,36,42,48,54,60,72,84,96,120,144)
    
    x = structure(c(29.88,35.62,31.234,18.236,8.827,8.108,1.968,4.597,2.126,13.288,6.509,11.291,4.034,1.32,8.971,
                    47.9,47.66,43.353,27.964,15.604,12.813,5.636,6.99,3.162,8.025,8.263,9.537,5.867,4.031,8.413,
                    0.254,7.115,13.419,11.193,6.668,5.068,5.835,11.2,8.625,6.677,5.67,6.892,7.105,9.863,9.242,
                    0.185,0.708,1.233,1.826,0.573,0.158,2.284,4.305,3.889,1.976,1.213,1.774,1.708,1.68,3.289,
                    0.32143359,0.0539624,0.358997363,0.229663119,0.005117894,0.001599894,0.004721622,0.007483267,0.022860791,0.002198322,0.000790845,0.000508266,0.000721259,0.000363992,0.00016147,
                    0.039166072,0.060243475,0.044556392,0.028967433,0.074115666,0.212280457,0.178200949,0.142531223,0.795994401,0.23708824,0.218227769,0.053077445,0.057532072,0.049763396,0.02937041,
                    0.002128914,0.010142804,0.006701284,0.00416062,0.000558673,0.000446846,0.001500231,0.013936638,0.001449299,0.000393693,0.000277433,0.000419912,0.001023665,0.00079279,0.000767641 #aab
    ),
    .Dim=c(15,7))
    lbl <- 'ghhp3'
  } 
  
  if (dsf==3){
    T = 15
    xi_r = c(41.573,44.215,0.617,0,0.02761158,0.001314952,8.4949E-05)
    t0 = 0
    ts = c(6,12,18,24,30,36,42,48,54,60,72,84,96,120,144)
    
    x = structure(c(34.776,24.698,7.016,13.984,13.188,3.67,7.276,12.482,1.604,3.289,3.459,3.948,5.479,3.496,4.077,
                    37.737,30.14,12.698,18.465,17.35,6.391,13.758,17.204,4.005,6.01,6.66,6.988,7.959,7.979,7.678,
                    1.871,9.427,13.117,14.546,12.351,18.271,9.358,5.632,8.417,4.309,2.812,2.046,2.846,0.514,0.339,
                    0.339,0.653,2.429,1.001,0.827,3.16,3.96,6.119,8.522,10.367,11.064,10.89,11.029,12.073,9.218,
                    0.055602108,0.41695699,0.641344329,0.338915366,0.124191325,0.014193557,0.155629262,0.15103975,0.050244816,0.00052371,0.001130033,0.00104494,0.000354072,0.000133996,0.000000015,
                    0.071106616,0.247121205,0.831591445,0.251715531,0.734361691,0.64700854,0.560931737,0.494208275,0.485187957,0.184039063,0.15202325,0.147540079,0.005443898,0.002830805,0.003862869,
                    0.006624575,0.009123428,0.001202702,0.008168796,0.002637291,0.001858481,0.001227886,0.003200059,0.004407152,0.011485715,0.009821052,0.002265469,0.000944404,0.000781912,0.001462709 #aab
    ),
    .Dim=c(15,7))
    lbl <- 'ghhp4'
  } 
  
  if (dsf==4){
    T = 15
    xi_r = c(49.309,51.391,0,0,0.025240111,0.007061712,0.000861307)
    t0 = 0
    ts = c(6,12,18,24,30,36,42,48,54,60,72,84,96,120,144)
    
    x = structure(c(45.631,15.927,13.69,7.45,5.934,10.343,3.783,5.55,3.314,4.761,6.694,2.861,5.596,11.384,12.127,
                    48.514,21.853,18.335,10.972,10.418,15.709,7.787,11.717,5.876,8.765,12.94,6.465,9.679,17.55,17.572,
                    2.876,2.527,13.479,17.859,20.257,13.024,11.215,10.762,13.682,5.683,11.696,10.164,8.597,6.438,2.713,
                    0,1.276,0.649,0.577,2.281,1.863,6.938,6.728,3.737,4.64,7.141,6.617,4.773,11.758,10.05,
                    0.054211479,0.227557555,0.333496484,0.074660563,0.171037468,0.026127103,0.025298295,0.019014778,0.023019255,0.003090945,0.000952996,0.000716294,0.000366515,0.000213841,0.000000015,
                    0.029642171,0.444539148,0.503396293,1.693986765,0.632280828,0.632280828,0.652995236,0.741156656,0.895179263,0.76543799,0.039619593,0.128798265,0.061788836,0.058198262,0.002055465,
                    0.003192699,0.004679054,0.005850029,0.000760603,0.005484765,0.003000254,0.003865076,0.003000254,0.00290508,0.000628287,0.003304898,0.004131978,0.009775929,0.00164497,0.000188 #aab
    ),
    .Dim=c(15,7))
    lbl <- 'ghhp5'
  } 
  
  x_d <- rbind(xi_r,x)
  x <- cbind(x,NA)
  
  scl <- c()
  for (i in 1:7){
    scl[i] <- max(x_d[,i])
  }
  
  scl <- c(scl,1)
  x0<-xi_r
  
  x0 = c(x0,0)
  
  load(paste0(dpath,fitnam))
  
  pred = as.data.frame(fit, pars = "x_hat") %>%
    gather(factor_key = TRUE) %>%
    group_by(key) %>%
    summarize(lb = quantile(value, probs = 0.025),
              median = quantile(value, probs = 0.5),
              ub = quantile(value, probs = 0.975))
  
  sp = split(pred,rep(1:8,each=T))
  
  metb <- c('(a)','(b)','(c)','(e)','(f)','(g)','(h)','(d)')
  
  for (i in 1:8){
    sp[i] = lapply(sp[i],"*",scl[i])
    sp[i] = mapply(cbind, sp[i], "obs"=list(x[,i]), SIMPLIFY=F)
    sp[i] = mapply(cbind, sp[i], "time"=list(ts), SIMPLIFY=F)
    sp[i] = mapply(rbind,list(c(NA,x0[i],x0[i],x0[i],x0[i],0)),sp[i],SIMPLIFY=F)
    sp[i] = mapply(cbind, sp[i], "met"=metb[i],SIMPLIFY = F)
    sp[i] = mapply(cbind, sp[i], "data"=lbl,SIMPLIFY= F)
    sp[i] = mapply(cbind, sp[i], "model"=mm, SIMPLIFY = F)
  }
  
  dfout <- do.call(rbind.data.frame,sp)
  
  p1 <- ggplot(data=dfout,aes(x=time,y=median))+
    geom_line(color="#D43F3AFF",size=0.7)+
    geom_point(aes(x=time,y=obs))+
    geom_line(aes(x = time, y = ub),color='#EEA2367F',linetype='dashed',size=1) +
    geom_line(aes(x = time, y = lb),color='#EEA2367F',linetype='dashed',size=1)+
    geom_ribbon(aes(ymin = lb, ymax = ub), fill = "#EEA2367F",alpha=0.3)+
    facet_wrap(~met,scales = 'free_y',ncol=4)+
    theme_bw()+
    theme(strip.text.x = element_text(hjust = -0.01),
          panel.grid.major = element_line(colour = "grey90",linetype='dashed'),
          panel.grid.minor = element_blank(),
          strip.background = element_blank(),
          plot.margin = unit(c(0,lp,0,0),'lines'),
          strip.text = element_text(size=16),
          axis.text=element_text(size=14),
          axis.title=element_text(size=14)
    )+
    xlab('time (h)')+
    ylab(expression(paste('mg g(pulp) '^{-1})))
  
  return(save_plot(paste0(outpth,str_sub(fitnam,0,-7),'.pdf'),p1,base_height = bh,base_width = bw))
  
}

########################################################################
# FIGURES S24 TO S26 OF SUPPLEMENTARY MATERIAL                         #
#                                                                      #
# explanation of functions:                                            #
#                                                                      #
# swf3 = data preparation function. Returns a dataframe containing     #
#        posterior predictions for datasets ecwb1, brpb1 and niwb2     #
#        series to be generated. Arguments:                            #
#        dsn = dataset to extract posterior predictions:               #
#              1: ecwb1                                                #
#              2: bspb1                                                #
#              3: niwb2                                                #
#        fitnam = name of Rsave file where Stan output is stored.      #
#        mm = model iteration code (without commas).                   #
#        fi = binary argument for labelling.                           #
#             1: return labels "Glc", "Fru", ... , "AAB"               #
#             2: return labels "(a)", "(b)", ... , "(h)"               #
#                                                                      #
# cplt1 = plotting function for Figure S24. Arguments:                 #
#         idf = dataframe containing posterior predictions             #
#         lbl = name to be given to resulting pdf file                 #
#         bh = base hight of the plot                                  #
#         bw = base width of the plot                                  #
#         lp = right plot margin                                       #
#                                                                      #
# cplt2 = plotting function for Figures S25 and S26. Arguments:        #
#         Same as for cplt1                                            #
#                                                                      #
########################################################################

swf3 <- function(dsn,fitnam,mm,fi){
  if (dsn==1){
    T = 14
    x0 = c(42.083,53.899,0.889,0,0.889,0.005047674,0.000990627,3.10569E-05)
    t0 = 0
    ts = c(6,12,18,24,30,36,42,48,54,60,66,72,84,96)
    
    x = structure(c(40.032,30.959,22.229,28.397,26.687,1.175,1.607,5.976,0.757,3.072,1.791,0.681,0.773,1.121,
                    43.799,36.867,26.082,30.623,29.513,3.401,4.432,12.484,3.839,8.209,9.754,2.393,1.971,2.749,
                    0.889,1.257,1.912,2.362,4.282,17.847,13.354,14.744,12.129,12.538,11.763,6.737,3.552,5.146,
                    0,0.604,0.645,2.688,1.055,3.261,0.729,2.527,1.71,3.631,5.062,3.387,2.04,1.796,
                    0.889,1.257,1.912,3.587,2.24,3.588,1.587,2.527,2.282,4.367,6.982,6.737,7.842,6.29,
                    0.002080134,0.01049763,0.103297844,0.058491298,0.172223043,0.841571964,0.189710452,0.269210044,0.02831987,0.025008708,0.003076743,0.009979097,1.52087E-06,6.87213E-05,
                    0.002192351,0.090138435,0.248834167,0.331825695,0.64700854,0.283080538,0.332590632,0.242610735,0.113477566,0.026357852,0.090972476,0.088289694,0.066513532,0.024940779,
                    0.000147285,0.002723692,0.000899825,3.02105E-05,0.001127608,0.000464684,6.09759E-05,2.85206E-05,0.000280645,0.000262517,9.25035E-05,0.000531078,6.87318E-06,7.34781E-06),
                  .Dim=c(14,8))
    lbl <- 'ecwb1'
  } 
  if (dsn==2){
    T = 12
    x0 = c(40.008,35.371,0,2.021,0,0.002851617,0.001570037,2.8E-10)
    t0 = 0
    ts = c(12,24,36,48,60,72,84,96,108,120,132,144)
    
    x = structure(c(102.342,23.407,1.09,5.923,2.81,9.85,11.593,13.556,10.443,11.303,10.948,1.986,
                    95.058,24.952,10.14,11.331,6.121,6.98,6.516,13.556,15.189,7.992,6.866,1.324,
                    43.93,45.732,65.326,20.521,26.127,27.8,33.277,19.801,19.218,25.082,21.597,23.529,
                    23.545,22.156,31.425,20.922,18.4,16.24,15.313,12.533,9.445,6.819,6.306,6.153,
                    2.874,3.091,12.608,42.164,55.501,68.726,60.363,21.663,19.943,18.665,18.383,19.984,
                    0.548392187,0.088123403,0.002046875,0.013806744,0.004809404,0.000553466,2.61873E-05,0.000110431,0.000218319,0.000000015,0.000000015,0.000000015,
                    0.149592566,0.335668056,0.070943076,0.086279975,0.039986189,0.020319359,0.008768191,0.005739975,0.039710926,0.097478764,0.123853993,0.074115666,
                    0.000604168,0.000324458,0.00024955,4.76604E-05,9.73101E-06,7.83715E-06,1.13281E-06,8.51448E-07,2.8E-10,2.8E-10,2.8E-10,2.8E-10),
                  .Dim=c(12,8))
    lbl <- 'brpb1'
  } 
  if (dsn==3){
    T = 6
    x0 = c(20.684,23.663,0,0,0,0.030626069,9.48222E-05,1.27985E-05)
    t0 = 0
    ts = c(24,48,72,96,120,144)
    
    x = structure(c(21.418,8.099,2.021,2.73,3.367,1.952,
                    25.533,14.058,2.021,2.73,4.076,1.952,
                    6.451,17.029,18.745,9.088,6.255,2.051,
                    2.412,5.069,7.765,5.833,4.333,5.776,
                    0,1.382,2.275,4.226,6.882,8.443,
                    0.328164244,1.164370675,0.065477375,0.001191492,7.17945E-05,4.52993E-06,                
                    0.001435192,0.119374073,0.202726262,0.004044921,0.057136024,0.001891952,
                    7.03328E-05,0.000140332,0.007197108,0.003070139,0.027362642,0.000906062),
                  .Dim=c(6,8))
    lbl <- 'niwb2'
  } 
  
  x1 = rbind(x0,x)
  scl = c(max(x1[,1]),max(x1[,2]),max(x1[,3]),max(x1[,4]),max(x1[,5]),max(x1[,6]),max(x1[,7]),max(x1[,8]))
  
  load(paste0(dpath,fitnam))
  
  pred = as.data.frame(fit, pars = "x_hat") %>%
    gather(factor_key = TRUE) %>%
    group_by(key) %>%
    summarize(lb = quantile(value, probs = 0.025),
              median = quantile(value, probs = 0.5),
              ub = quantile(value, probs = 0.975)) 
  
  sp = split(pred,rep(1:8,each=T))
  
  rnams <- c('0','1','2','3','4','5','1,2','1,3','1,4','1,5','2,3','2,4','2,5','3,4','3,5','1,2,3','1,2,4','1,2,5',
             '1,3,4','1,3,5','1,4,5','2,3,4','2,3,5','1,2,3,4','1,2,3,5')
  
  if (fi==1){
    metb <- c('Glc','Fru','EtOH','LA','Ac','Y','LAB','AAB')
  }
  
  if (fi==2){
    metb <- c('(a)','(b)','(c)','(d)','(e)','(f)','(g)','(h)')
  }
  
  for (i in 1:8){
    sp[i] = lapply(sp[i],"*",scl[i])
    sp[i] = mapply(cbind, sp[i], "obs"=list(x[,i]), SIMPLIFY=F)
    sp[i] = mapply(cbind, sp[i], "time"=list(ts), SIMPLIFY=F)
    sp[i] = mapply(rbind,list(c(NA,x0[i],x0[i],x0[i],x0[i],0)),sp[i],SIMPLIFY=F)
    sp[i] = mapply(cbind, sp[i], "met"=metb[i],SIMPLIFY = F)
    sp[i] = mapply(cbind, sp[i], "data"=lbl,SIMPLIFY= F)
  }
  
  dfout <- do.call(rbind.data.frame,sp)
  dfout$model <- mm
  dfout$model <- factor(dfout$model,levels=rnams)
  dfout$met <- factor(dfout$met,levels=metb)
  
  return(dfout)
} 

cplt1 <- function(idf,lbl,bh,bw,lp){
  plt <- ggplot(data=idf,aes(x=time,y=median,group=model))+
    geom_line(color="#D43F3AFF",size=0.7)+
    geom_point(aes(x=time,y=obs),colour='black')+
    geom_line(aes(x = time, y = ub),color='#EEA2367F',linetype='dashed',size=1) +
    geom_line(aes(x = time, y = lb),color='#EEA2367F',linetype='dashed',size=1)+
    facet_grid(met~model,scales = 'free_y')+
    geom_ribbon(aes(ymin = lb, ymax = ub),fill = "#EEA2367F",alpha=0.3,linetype=2)+
    theme_bw()+
    theme(panel.grid.major = element_line(colour = "grey90",linetype='dashed'),
          panel.grid.minor = element_blank(),
          strip.background = element_blank(),
          plot.margin = unit(c(0,lp,0,0),'lines'),
          strip.text = element_text(size=16),
          axis.text=element_text(size=14),
          axis.title=element_text(size=14),
          legend.position = 'bottom',
          panel.spacing = unit(0.75, "lines"))+
    xlab('time (h)')+
    ylab(expression(paste('mg g(pulp) '^{-1})))+
    
    scale_fill_d3()+
    scale_color_d3()+
    guides(fill = guide_legend(override.aes = list(shape = NA)))+
    labs(fill= 'MI ( )', color = 'MI ( )')
  
  return(save_plot(paste0(outpth,lbl,'.pdf'),plt,base_height = bh,base_width = bw))
}

cplt2 <- function(idf,lbl,bh,bw,lp){
  idf1 <- idf
  idf1$model1 <- idf1$model
  plt <- ggplot(data=idf,aes(x=time,y=median,group=model,fill=model,colour=model))+
    geom_line(data=subset(idf1,model=='2,3'),aes(x = time, y = ub),color='#EEA2367F',linetype='dashed',size=1)+
    geom_line(data=subset(idf1,model=='2,3'),aes(x = time, y = lb),color='#EEA2367F',linetype='dashed',size=1)+
    geom_line(data=subset(idf1,model!='2,3'),aes(x = time, y = ub),color='#46B8DAFF',linetype='dashed',size=1)+
    geom_line(data=subset(idf1,model!='2,3'),aes(x = time, y = lb),color='#46B8DAFF',linetype='dashed',size=1)+
    geom_line(size=1)+
    geom_point(aes(x=time,y=obs),colour='black')+
    facet_wrap(~met,scales = 'free_y',ncol=4)+ 
    geom_ribbon(aes(ymin = lb, ymax = ub),alpha=0.3,linetype=0)+
    theme_bw()+
    theme(strip.text.x = element_text(hjust = -0.01),
          panel.grid.major = element_line(colour = "grey90",linetype='dashed'),
          panel.grid.minor = element_blank(),
          strip.background = element_blank(),
          plot.margin = unit(c(0,lp,0,0),'lines'),
          strip.text = element_text(size=16),
          axis.text=element_text(size=14),
          axis.title=element_text(size=14),
          legend.position = 'bottom')+
    xlab('time (h)')+
    ylab(expression(paste('mg g(pulp) '^{-1})))+
    
    scale_colour_manual(values=c('#357EBDFF','#D43F3AFF'))+
    scale_fill_manual(values=pal_locuszoom('default')(5)[c(4,2)])+
    guides(fill = guide_legend(override.aes = list(shape = NA)))+
    labs(fill= 'MI ( )', color = 'MI ( )')
  
  return(save_plot(paste0(outpth,lbl,'.pdf'),plt,base_height = bh,base_width = bw))

}



#Figure S24 plotting: This figure is composed by three subplots as follows
cplt1(rbind(swf3(1,'ecwb1_mm0.Rsave','0',1),
            swf3(1,'ecwb1_mm1.Rsave','1',1),
            swf3(1,'ecwb1_mm3.Rsave','3',1),
            swf3(1,'ecwb1_mm4.Rsave','4',1),
            swf3(1,'ecwb1_mm12.Rsave','1,2',1)),'figS24a',17.5,5*3,1)

cplt1(rbind(swf3(1,'ecwb1_mm13.Rsave','1,3',1),
            swf3(1,'ecwb1_mm14.Rsave','1,4',1),
            swf3(1,'ecwb1_mm23.Rsave','2,3',1),
            swf3(1,'ecwb1_mm24.Rsave','2,4',1),
            swf3(1,'ecwb1_mm34.Rsave','3,4',1)),'figS24b',17.5,5*3,1)

cplt1(rbind(swf3(1,'ecwb1_mm123.Rsave','1,2,3',1),
            swf3(1,'ecwb1_mm124.Rsave','1,2,4',1),
            swf3(1,'ecwb1_mm134.Rsave','1,3,4',1),
            swf3(1,'ecwb1_mm234.Rsave','2,3,4',1),
            swf3(1,'ecwb1_mm1234.Rsave','1,2,3,4',1)),'figS24c',17.5,5*3,1)

#Figure S25 plotting:
cplt2(rbind(swf3(2,'brpb1_mm23.Rsave','2,3',2),
            swf3(2,'brpb1_mm1.Rsave','1',2)),'brpb1_23_1.1',8.3,17.2,1)


#Figure S26 plotting:
cplt2(rbind(swf3(3,'niwb2_mm23.Rsave','2,3',2),
            swf3(3,'niwb2_mm1.Rsave','1',2)),'niwb2_23_1.1',8.3,17.2,1)
