!-------------------------------------------------------------------------------
!> module ADMIN TIME
!!
!! @par Description
!!          Time management for SCALE-RM
!!
!! @author Team SCALE
!!
!<
#include "scalelib.h"
module mod_admin_time
  !-----------------------------------------------------------------------------
  !
  !++ used modules
  !
  use scale_precision
  use scale_io
  use scale_prof
  !-----------------------------------------------------------------------------
  implicit none
  private
  !-----------------------------------------------------------------------------
  !
  !++ Public procedure
  !
  public :: ADMIN_TIME_setup
  public :: ADMIN_TIME_checkstate
  public :: ADMIN_TIME_advance

  !-----------------------------------------------------------------------------
  !
  !++ Public parameters & variables
  !
  real(DP), public :: TIME_DTSEC_ATMOS_RESTART  !< time interval of atmosphere restart [sec]
  real(DP), public :: TIME_DTSEC_OCEAN_RESTART  !< time interval of ocean restart      [sec]
  real(DP), public :: TIME_DTSEC_LAND_RESTART   !< time interval of land restart       [sec]
  real(DP), public :: TIME_DTSEC_URBAN_RESTART  !< time interval of urban restart      [sec]
  real(DP), public :: TIME_DTSEC_RESUME         !< time interval for resume            [sec]
  real(DP), public :: TIME_DTSEC_ATMOS_DA       !< time interval of atmosphere DA      [sec]

  integer,  public :: TIME_DSTEP_ATMOS_RESTART  !< interval of atmosphere restart [step]
  integer,  public :: TIME_DSTEP_OCEAN_RESTART  !< interval of ocean restart      [step]
  integer,  public :: TIME_DSTEP_LAND_RESTART   !< interval of land restart       [step]
  integer,  public :: TIME_DSTEP_URBAN_RESTART  !< interval of urban restart      [step]
  integer,  public :: TIME_DSTEP_RESUME         !< interval for resume            [step]
  integer,  public :: TIME_DSTEP_ATMOS_DA  !< interval of atmosphere DA [step]

  logical,  public :: TIME_DOATMOS_step         !< execute atmosphere component in this step?
  logical,  public :: TIME_DOATMOS_DYN          !< execute dynamics in this step?
  logical,  public :: TIME_DOATMOS_PHY_CP       !< execute physics  in this step? (cumulus     )
  logical,  public :: TIME_DOATMOS_PHY_MP       !< execute physics  in this step? (microphysics)
  logical,  public :: TIME_DOATMOS_PHY_RD       !< execute physics  in this step? (radiation   )
  logical,  public :: TIME_DOATMOS_PHY_SF       !< execute physics  in this step? (surface flux)
  logical,  public :: TIME_DOATMOS_PHY_TB       !< execute physics  in this step? (turbulence  )
  logical,  public :: TIME_DOATMOS_PHY_BL       !< execute physics  in this step? (boudary layer  )
  logical,  public :: TIME_DOATMOS_PHY_CH       !< execute physics  in this step? (chemistry   )
  logical,  public :: TIME_DOATMOS_PHY_AE       !< execute physics  in this step? (aerosol     )
  logical,  public :: TIME_DOATMOS_restart      !< execute atmosphere restart output in this step?
  logical,  public :: TIME_DOATMOS_DA           !< execute DA in this step?
  logical,  public :: TIME_DOOCEAN_step         !< execute ocean      component      in this step?
  logical,  public :: TIME_DOOCEAN_restart      !< execute ocean      restart output in this step?
  logical,  public :: TIME_DOLAND_step          !< execute land       component      in this step?
  logical,  public :: TIME_DOLAND_restart       !< execute land       restart output in this step?
  logical,  public :: TIME_DOURBAN_step         !< execute urban      component      in this step?
  logical,  public :: TIME_DOURBAN_restart      !< execute urban      restart output in this step?
  logical,  public :: TIME_DOresume             !< resume in this step?
  logical,  public :: TIME_DOend                !< finish program in this step?

  logical,  public :: TIME_END_RESTART_OUT = .true. !< always output restart file at the end?

  !-----------------------------------------------------------------------------
  !
  !++ Private procedure
  !
  !-----------------------------------------------------------------------------
  !
  !++ Private parameters & variables
  !
  integer,  private :: TIME_STARTDATE(6) = (/ -999, 1, 1, 0, 0, 0 /)
  real(DP), private :: TIME_STARTMS      = 0.0_DP !< [millisec]
  integer,  private :: TIME_STARTDAY
  real(DP), private :: TIME_STARTSEC

  integer,  private :: TIME_ENDDATE(6)
  real(DP), private :: TIME_ENDMS
  integer,  private :: TIME_ENDDAY
  real(DP), private :: TIME_ENDSEC

  integer,  private :: TIME_RES_ATMOS_DYN     = 0
  integer,  private :: TIME_RES_ATMOS_PHY_CP  = 0
  integer,  private :: TIME_RES_ATMOS_PHY_MP  = 0
  integer,  private :: TIME_RES_ATMOS_PHY_RD  = 0
  integer,  private :: TIME_RES_ATMOS_PHY_SF  = 0
  integer,  private :: TIME_RES_ATMOS_PHY_TB  = 0
  integer,  private :: TIME_RES_ATMOS_PHY_BL  = 0
  integer,  private :: TIME_RES_ATMOS_PHY_CH  = 0
  integer,  private :: TIME_RES_ATMOS_PHY_AE  = 0
  integer,  private :: TIME_RES_ATMOS_RESTART = 0
  integer,  private :: TIME_RES_OCEAN         = 0
  integer,  private :: TIME_RES_OCEAN_RESTART = 0
  integer,  private :: TIME_RES_LAND          = 0
  integer,  private :: TIME_RES_LAND_RESTART  = 0
  integer,  private :: TIME_RES_URBAN         = 0
  integer,  private :: TIME_RES_URBAN_RESTART = 0
  integer,  private :: TIME_RES_RESUME
  integer,  private :: TIME_RES_ATMOS_DA = 0

  real(DP), private :: TIME_WALLCLOCK_START             ! Start time of wall clock             [sec]
  real(DP), private :: TIME_WALLCLOCK_LIMIT   = -1.0_DP ! Elapse time limit of wall clock time [sec]
  real(DP), private :: TIME_WALLCLOCK_SAFE    =  0.9_DP ! Safety coefficient for elapse time limit
  real(DP), private :: TIME_WALLCLOCK_safelim           ! TIME_WALLCLOCK_LIMIT * TIME_WALLCLOCK_SAFE

  logical,  private :: debug = .false.

  real(DP), private, parameter :: eps = 1.E-6_DP !> epsilon for timesec

  !-----------------------------------------------------------------------------
contains
  !-----------------------------------------------------------------------------
  !> Setup
  subroutine ADMIN_TIME_setup( &
       setup_TimeIntegration )
    use scale_file, only: &
       FILE_Get_Attribute
    use scale_prc, only: &
       PRC_myrank,  &
       PRC_abort, &
       PRC_MPItime
    use scale_const, only: &
       UNDEF8 => CONST_UNDEF8
    use scale_calendar, only: &
       CALENDAR_date2daysec,    &
       CALENDAR_daysec2date,    &
       CALENDAR_adjust_daysec,  &
       CALENDAR_combine_daysec, &
       CALENDAR_unit2sec,       &
       CALENDAR_CFunits2sec,    &
       CALENDAR_date2char
    use scale_time, only: &
       TIME_DTSEC,                 &
       TIME_NOWDATE,               &
       TIME_NOWMS,                 &
       TIME_NOWDAY,                &
       TIME_NOWSEC,                &
       TIME_NOWDAYSEC,             &
       TIME_NOWSTEP,               &
       TIME_NSTEP,                 &
       TIME_DTSEC_ATMOS_DYN,       &
       NSTEP_DYN => TIME_NSTEP_ATMOS_DYN, &
       TIME_DTSEC_ATMOS_PHY_CP,    &
       TIME_DTSEC_ATMOS_PHY_MP,    &
       TIME_DTSEC_ATMOS_PHY_RD,    &
       TIME_DTSEC_ATMOS_PHY_SF,    &
       TIME_DTSEC_ATMOS_PHY_TB,    &
       TIME_DTSEC_ATMOS_PHY_BL,    &
       TIME_DTSEC_ATMOS_PHY_CH,    &
       TIME_DTSEC_ATMOS_PHY_AE,    &
       TIME_DTSEC_OCEAN,           &
       TIME_DTSEC_LAND,            &
       TIME_DTSEC_URBAN,           &
       TIME_DTSEC_WALLCLOCK_CHECK, &
       TIME_DSTEP_ATMOS_DYN,       &
       TIME_DSTEP_ATMOS_PHY_CP,    &
       TIME_DSTEP_ATMOS_PHY_MP,    &
       TIME_DSTEP_ATMOS_PHY_RD,    &
       TIME_DSTEP_ATMOS_PHY_SF,    &
       TIME_DSTEP_ATMOS_PHY_TB,    &
       TIME_DSTEP_ATMOS_PHY_BL,    &
       TIME_DSTEP_ATMOS_PHY_CH,    &
       TIME_DSTEP_ATMOS_PHY_AE,    &
       TIME_DSTEP_OCEAN,           &
       TIME_DSTEP_LAND,            &
       TIME_DSTEP_URBAN,           &
       TIME_DSTEP_WALLCLOCK_CHECK, &
       TIME_OFFSET_YEAR,           &
       TIME_STARTDAYSEC
    use mod_atmos_vars, only: &
       RESTART_IN_BASENAME => ATMOS_RESTART_IN_BASENAME
    implicit none

    logical, intent(in) :: setup_TimeIntegration

    real(DP)               :: TIME_DURATION                = UNDEF8
    character(len=H_SHORT) :: TIME_DURATION_UNIT           = "SEC"
    real(DP)               :: TIME_DT                      = UNDEF8
    character(len=H_SHORT) :: TIME_DT_UNIT                 = "SEC"

    real(DP)               :: TIME_DT_ATMOS_DYN            = UNDEF8
    character(len=H_SHORT) :: TIME_DT_ATMOS_DYN_UNIT       = "SEC"
    integer                :: TIME_NSTEP_ATMOS_DYN         = -1
    real(DP)               :: TIME_DT_ATMOS_PHY_CP         = UNDEF8
    character(len=H_SHORT) :: TIME_DT_ATMOS_PHY_CP_UNIT    = ""
    real(DP)               :: TIME_DT_ATMOS_PHY_MP         = UNDEF8
    character(len=H_SHORT) :: TIME_DT_ATMOS_PHY_MP_UNIT    = ""
    real(DP)               :: TIME_DT_ATMOS_PHY_RD         = UNDEF8
    character(len=H_SHORT) :: TIME_DT_ATMOS_PHY_RD_UNIT    = ""
    real(DP)               :: TIME_DT_ATMOS_PHY_SF         = UNDEF8
    character(len=H_SHORT) :: TIME_DT_ATMOS_PHY_SF_UNIT    = ""
    real(DP)               :: TIME_DT_ATMOS_PHY_TB         = UNDEF8
    character(len=H_SHORT) :: TIME_DT_ATMOS_PHY_TB_UNIT    = ""
    real(DP)               :: TIME_DT_ATMOS_PHY_BL         = UNDEF8
    character(len=H_SHORT) :: TIME_DT_ATMOS_PHY_BL_UNIT    = ""
    real(DP)               :: TIME_DT_ATMOS_PHY_CH         = UNDEF8
    character(len=H_SHORT) :: TIME_DT_ATMOS_PHY_CH_UNIT    = ""
    real(DP)               :: TIME_DT_ATMOS_PHY_AE         = UNDEF8
    character(len=H_SHORT) :: TIME_DT_ATMOS_PHY_AE_UNIT    = ""
    real(DP)               :: TIME_DT_ATMOS_RESTART        = UNDEF8
    character(len=H_SHORT) :: TIME_DT_ATMOS_RESTART_UNIT   = ""
    real(DP)               :: TIME_DT_ATMOS_DA             = UNDEF8
    character(len=H_SHORT) :: TIME_DT_ATMOS_DA_UNIT        = ""

    real(DP)               :: TIME_DT_OCEAN                = UNDEF8
    character(len=H_SHORT) :: TIME_DT_OCEAN_UNIT           = ""
    real(DP)               :: TIME_DT_OCEAN_RESTART        = UNDEF8
    character(len=H_SHORT) :: TIME_DT_OCEAN_RESTART_UNIT   = ""

    real(DP)               :: TIME_DT_LAND                 = UNDEF8
    character(len=H_SHORT) :: TIME_DT_LAND_UNIT            = ""
    real(DP)               :: TIME_DT_LAND_RESTART         = UNDEF8
    character(len=H_SHORT) :: TIME_DT_LAND_RESTART_UNIT    = ""

    real(DP)               :: TIME_DT_URBAN                = UNDEF8
    character(len=H_SHORT) :: TIME_DT_URBAN_UNIT           = ""
    real(DP)               :: TIME_DT_URBAN_RESTART        = UNDEF8
    character(len=H_SHORT) :: TIME_DT_URBAN_RESTART_UNIT   = ""

    real(DP)               :: TIME_DT_RESUME               = UNDEF8
    character(len=H_SHORT) :: TIME_DT_RESUME_UNIT          = ""

    real(DP)               :: TIME_DT_WALLCLOCK_CHECK      = UNDEF8
    character(len=H_SHORT) :: TIME_DT_WALLCLOCK_CHECK_UNIT = ""

    namelist / PARAM_TIME / &
       TIME_STARTDATE,               &
       TIME_STARTMS,                 &
       TIME_DURATION,                &
       TIME_DURATION_UNIT,           &
       TIME_DT,                      &
       TIME_DT_UNIT,                 &
       TIME_DT_ATMOS_DYN,            &
       TIME_DT_ATMOS_DYN_UNIT,       &
       TIME_NSTEP_ATMOS_DYN,         &
       TIME_DT_ATMOS_PHY_CP,         &
       TIME_DT_ATMOS_PHY_CP_UNIT,    &
       TIME_DT_ATMOS_PHY_MP,         &
       TIME_DT_ATMOS_PHY_MP_UNIT,    &
       TIME_DT_ATMOS_PHY_RD,         &
       TIME_DT_ATMOS_PHY_RD_UNIT,    &
       TIME_DT_ATMOS_PHY_SF,         &
       TIME_DT_ATMOS_PHY_SF_UNIT,    &
       TIME_DT_ATMOS_PHY_TB,         &
       TIME_DT_ATMOS_PHY_TB_UNIT,    &
       TIME_DT_ATMOS_PHY_BL,         &
       TIME_DT_ATMOS_PHY_BL_UNIT,    &
       TIME_DT_ATMOS_PHY_CH,         &
       TIME_DT_ATMOS_PHY_CH_UNIT,    &
       TIME_DT_ATMOS_PHY_AE,         &
       TIME_DT_ATMOS_PHY_AE_UNIT,    &
       TIME_DT_ATMOS_RESTART,        &
       TIME_DT_ATMOS_RESTART_UNIT,   &
       TIME_DT_ATMOS_DA,             &
       TIME_DT_ATMOS_DA_UNIT,        &
       TIME_DT_OCEAN,                &
       TIME_DT_OCEAN_UNIT,           &
       TIME_DT_OCEAN_RESTART,        &
       TIME_DT_OCEAN_RESTART_UNIT,   &
       TIME_DT_LAND,                 &
       TIME_DT_LAND_UNIT,            &
       TIME_DT_LAND_RESTART,         &
       TIME_DT_LAND_RESTART_UNIT,    &
       TIME_DT_URBAN,                &
       TIME_DT_URBAN_UNIT,           &
       TIME_DT_URBAN_RESTART,        &
       TIME_DT_URBAN_RESTART_UNIT,   &
       TIME_DT_WALLCLOCK_CHECK,      &
       TIME_DT_WALLCLOCK_CHECK_UNIT, &
       TIME_DT_RESUME,               &
       TIME_DT_RESUME_UNIT,          &
       TIME_WALLCLOCK_LIMIT,         &
       TIME_WALLCLOCK_SAFE,          &
       TIME_END_RESTART_OUT,         &
       debug

    integer              :: dateday
    real(DP)             :: datesec
    real(DP)             :: cftime(1)
    character(len=H_MID) :: cfunits

    real(DP)          :: TIME_DURATIONSEC
    character(len=27) :: startchardate
    character(len=27) :: endchardate

    integer :: ierr
    !---------------------------------------------------------------------------

    LOG_NEWLINE
    LOG_INFO("ADMIN_TIME_setup",*) 'Setup'

    TIME_NSTEP_ATMOS_DYN = -1

    !--- read namelist
    rewind(IO_FID_CONF)
    read(IO_FID_CONF,nml=PARAM_TIME,iostat=ierr)
    if( ierr < 0 ) then !--- missing
       LOG_INFO("ADMIN_TIME_setup",*) 'Not found namelist. Default used.'
    elseif( ierr > 0 ) then !--- fatal error
       LOG_ERROR("ADMIN_TIME_setup",*) 'Not appropriate names in namelist PARAM_TIME. Check!'
       call PRC_abort
    endif
    LOG_NML(PARAM_TIME)

    ! check time setting
    if ( setup_TimeIntegration ) then

       LOG_NEWLINE
       LOG_INFO("ADMIN_TIME_setup",*) 'Check time interval and unit for each component '

       if ( TIME_DT == UNDEF8 ) then
          LOG_ERROR("ADMIN_TIME_setup",*) 'Not found TIME_DT. STOP.'
          call PRC_abort
       endif
       if ( TIME_DURATION == UNDEF8 ) then
          LOG_ERROR("ADMIN_TIME_setup",*) 'Not found TIME_DURATION. STOP.'
          call PRC_abort
       endif

       ! DYN
       if ( TIME_DT_ATMOS_DYN == UNDEF8 ) then
          if ( TIME_NSTEP_ATMOS_DYN < 0 ) then
          LOG_INFO_CONT(*) 'Not found TIME_DT_ATMOS_DYN.          TIME_DT is used.'
             TIME_DT_ATMOS_DYN = TIME_DT
          endif
       endif
       if ( TIME_DT_ATMOS_DYN_UNIT == '' ) then
          LOG_INFO_CONT(*) 'Not found TIME_DT_ATMOS_DYN_UNIT.     TIME_DT_UNIT is used.'
          TIME_DT_ATMOS_DYN_UNIT = TIME_DT_UNIT
       endif
       ! PHY_CP
       if ( TIME_DT_ATMOS_PHY_CP == UNDEF8 ) then
          LOG_INFO_CONT(*) 'Not found TIME_DT_ATMOS_PHY_CP.       TIME_DT is used.'
          TIME_DT_ATMOS_PHY_CP = TIME_DT
       endif
       if ( TIME_DT_ATMOS_PHY_CP_UNIT == '' ) then
          LOG_INFO_CONT(*) 'Not found TIME_DT_ATMOS_PHY_CP_UNIT.  TIME_DT_UNIT is used.'
          TIME_DT_ATMOS_PHY_CP_UNIT = TIME_DT_UNIT
       endif
       ! PHY_MP
       if ( TIME_DT_ATMOS_PHY_MP == UNDEF8 ) then
          LOG_INFO_CONT(*) 'Not found TIME_DT_ATMOS_PHY_MP.       TIME_DT is used.'
          TIME_DT_ATMOS_PHY_MP = TIME_DT
       endif
       if ( TIME_DT_ATMOS_PHY_MP_UNIT == '' ) then
          LOG_INFO_CONT(*) 'Not found TIME_DT_ATMOS_PHY_MP_UNIT.  TIME_DT_UNIT is used.'
          TIME_DT_ATMOS_PHY_MP_UNIT = TIME_DT_UNIT
       endif
       ! PHY_RD
       if ( TIME_DT_ATMOS_PHY_RD == UNDEF8 ) then
          LOG_INFO_CONT(*) 'Not found TIME_DT_ATMOS_PHY_RD.       TIME_DT is used.'
          TIME_DT_ATMOS_PHY_RD = TIME_DT
       endif
       if ( TIME_DT_ATMOS_PHY_RD_UNIT == '' ) then
          LOG_INFO_CONT(*) 'Not found TIME_DT_ATMOS_PHY_RD_UNIT.  TIME_DT_UNIT is used.'
          TIME_DT_ATMOS_PHY_RD_UNIT = TIME_DT_UNIT
       endif
       ! PHY_SF
       if ( TIME_DT_ATMOS_PHY_SF == UNDEF8 ) then
          LOG_INFO_CONT(*) 'Not found TIME_DT_ATMOS_PHY_SF.       TIME_DT is used.'
          TIME_DT_ATMOS_PHY_SF = TIME_DT
       endif
       if ( TIME_DT_ATMOS_PHY_SF_UNIT == '' ) then
          LOG_INFO_CONT(*) 'Not found TIME_DT_ATMOS_PHY_SF_UNIT.  TIME_DT_UNIT is used.'
          TIME_DT_ATMOS_PHY_SF_UNIT = TIME_DT_UNIT
       endif
       ! PHY_TB
       if ( TIME_DT_ATMOS_PHY_TB == UNDEF8 ) then
          LOG_INFO_CONT(*) 'Not found TIME_DT_ATMOS_PHY_TB.       TIME_DT is used.'
          TIME_DT_ATMOS_PHY_TB = TIME_DT
       endif
       if ( TIME_DT_ATMOS_PHY_TB_UNIT == '' ) then
          LOG_INFO_CONT(*) 'Not found TIME_DT_ATMOS_PHY_TB_UNIT.  TIME_DT_UNIT is used.'
          TIME_DT_ATMOS_PHY_TB_UNIT = TIME_DT_UNIT
       endif
       ! PHY_BL
       if ( TIME_DT_ATMOS_PHY_BL == UNDEF8 ) then
          LOG_INFO_CONT(*) 'Not found TIME_DT_ATMOS_PHY_BL.       TIME_DT is used.'
          TIME_DT_ATMOS_PHY_BL = TIME_DT
       endif
       if ( TIME_DT_ATMOS_PHY_BL_UNIT == '' ) then
          LOG_INFO_CONT(*) 'Not found TIME_DT_ATMOS_PHY_BL_UNIT.  TIME_DT_UNIT is used.'
          TIME_DT_ATMOS_PHY_BL_UNIT = TIME_DT_UNIT
       endif
       ! PHY_CH
       if ( TIME_DT_ATMOS_PHY_CH == UNDEF8 ) then
          LOG_INFO_CONT(*) 'Not found TIME_DT_ATMOS_PHY_CH.       TIME_DT is used.'
          TIME_DT_ATMOS_PHY_CH = TIME_DT
       endif
       if ( TIME_DT_ATMOS_PHY_CH_UNIT == '' ) then
          LOG_INFO_CONT(*) 'Not found TIME_DT_ATMOS_PHY_CH_UNIT.  TIME_DT_UNIT is used.'
          TIME_DT_ATMOS_PHY_CH_UNIT = TIME_DT_UNIT
       endif
       ! PHY_AE
       if ( TIME_DT_ATMOS_PHY_AE == UNDEF8 ) then
          LOG_INFO_CONT(*) 'Not found TIME_DT_ATMOS_PHY_AE.       TIME_DT is used.'
          TIME_DT_ATMOS_PHY_AE = TIME_DT
       endif
       if ( TIME_DT_ATMOS_PHY_AE_UNIT == '' ) then
          LOG_INFO_CONT(*) 'Not found TIME_DT_ATMOS_PHY_AE_UNIT.  TIME_DT_UNIT is used.'
          TIME_DT_ATMOS_PHY_AE_UNIT = TIME_DT_UNIT
       endif
       ! ATMOS RESTART
       if ( TIME_DT_ATMOS_RESTART == UNDEF8 ) then
          LOG_INFO_CONT(*) 'Not found TIME_DT_ATMOS_RESTART.      TIME_DURATION is used.'
          TIME_DT_ATMOS_RESTART = TIME_DURATION
       endif
       if ( TIME_DT_ATMOS_RESTART_UNIT == '' ) then
          LOG_INFO_CONT(*) 'Not found TIME_DT_ATMOS_RESTART_UNIT. TIME_DURATION_UNIT is used.'
          TIME_DT_ATMOS_RESTART_UNIT = TIME_DURATION_UNIT
       endif
       ! ATMOS DA
       if ( TIME_DT_ATMOS_DA == UNDEF8 ) then
          LOG_INFO_CONT(*) 'Not found TIME_DT_ATMOS_DA.      TIME_DT_ATMOS_RESTART is used.'
          TIME_DT_ATMOS_DA = TIME_DT_ATMOS_RESTART
       endif
       if ( TIME_DT_ATMOS_DA_UNIT == '' ) then
          LOG_INFO_CONT(*) 'Not found TIME_DT_ATMOS_DA_UNIT. TIME_DT_ATMOS_RESTART_UNIT is used.'
          TIME_DT_ATMOS_DA_UNIT = TIME_DT_ATMOS_RESTART_UNIT
       endif
       ! OCEAN
       if ( TIME_DT_OCEAN == UNDEF8 ) then
          LOG_INFO_CONT(*) 'Not found TIME_DT_OCEAN.              TIME_DT is used.'
          TIME_DT_OCEAN = TIME_DT
       endif
       if ( TIME_DT_OCEAN_UNIT == '' ) then
          LOG_INFO_CONT(*) 'Not found TIME_DT_OCEAN_UNIT.         TIME_DT_UNIT is used.'
          TIME_DT_OCEAN_UNIT = TIME_DT_UNIT
       endif
       ! OCEAN RESTART
       if ( TIME_DT_OCEAN_RESTART == UNDEF8 ) then
          LOG_INFO_CONT(*) 'Not found TIME_DT_OCEAN_RESTART.      TIME_DURATION is used.'
          TIME_DT_OCEAN_RESTART = TIME_DURATION
       endif
       if ( TIME_DT_OCEAN_RESTART_UNIT == '' ) then
          LOG_INFO_CONT(*) 'Not found TIME_DT_OCEAN_RESTART_UNIT. TIME_DURATION_UNIT is used.'
          TIME_DT_OCEAN_RESTART_UNIT = TIME_DURATION_UNIT
       endif
       ! LAND
       if ( TIME_DT_LAND == UNDEF8 ) then
          LOG_INFO_CONT(*) 'Not found TIME_DT_LAND.               TIME_DT is used.'
          TIME_DT_LAND = TIME_DT
       endif
       if ( TIME_DT_LAND_UNIT == '' ) then
          LOG_INFO_CONT(*) 'Not found TIME_DT_LAND_UNIT.          TIME_DT_UNIT is used.'
          TIME_DT_LAND_UNIT = TIME_DT_UNIT
       endif
       ! LAND RESTART
       if ( TIME_DT_LAND_RESTART == UNDEF8 ) then
          LOG_INFO_CONT(*) 'Not found TIME_DT_LAND_RESTART.       TIME_DURATION is used.'
          TIME_DT_LAND_RESTART = TIME_DURATION
       endif
       if ( TIME_DT_LAND_RESTART_UNIT == '' ) then
          LOG_INFO_CONT(*) 'Not found TIME_DT_LAND_RESTART_UNIT.  TIME_DURATION_UNIT is used.'
          TIME_DT_LAND_RESTART_UNIT = TIME_DURATION_UNIT
       endif
       ! URBAN
       if ( TIME_DT_URBAN == UNDEF8 ) then
          LOG_INFO_CONT(*) 'Not found TIME_DT_URBAN.              TIME_DT is used.'
          TIME_DT_URBAN = TIME_DT
       endif
       if ( TIME_DT_URBAN_UNIT == '' ) then
          LOG_INFO_CONT(*) 'Not found TIME_DT_URBAN_UNIT.         TIME_DT_UNIT is used.'
          TIME_DT_URBAN_UNIT = TIME_DT_UNIT
       endif
       ! URBAN RESTART
       if ( TIME_DT_URBAN_RESTART == UNDEF8 ) then
          LOG_INFO_CONT(*) 'Not found TIME_DT_URBAN_RESTART.      TIME_DURATION is used.'
          TIME_DT_URBAN_RESTART = TIME_DURATION
       endif
       if ( TIME_DT_URBAN_RESTART_UNIT == '' ) then
          LOG_INFO_CONT(*) 'Not found TIME_DT_URBAN_RESTART_UNIT. TIME_DURATION_UNIT is used.'
          TIME_DT_URBAN_RESTART_UNIT = TIME_DURATION_UNIT
       endif
       ! Resume
       if ( TIME_DT_RESUME == UNDEF8 ) then
          TIME_DT_RESUME = TIME_DURATION
       endif
       if ( TIME_DT_RESUME_UNIT == '' ) then
          LOG_INFO_CONT(*) 'Not found TIME_DT_RESUME_UNIT.        TIME_DURATION_UNIT is used.'
          TIME_DT_RESUME_UNIT = TIME_DURATION_UNIT
       endif
    endif

    !--- calculate time
    if ( TIME_STARTDATE(1) == -999 ) then
       if ( RESTART_IN_BASENAME /= '' ) then ! read start time from the restart data
          call FILE_Get_Attribute( RESTART_IN_BASENAME, & ! [IN]
                                   "global",            & ! [IN]
                                   'time_start',        & ! [IN]
                                   cftime(:),           & ! [OUT]
                                   rankid = PRC_myrank, & ! [IN]
                                   single = .false.     ) ! [IN]

          call FILE_Get_Attribute( RESTART_IN_BASENAME, & ! [IN]
                                   "global",            & ! [IN]
                                   'time_units',        & ! [IN]
                                   cfunits,             & ! [OUT]
                                   rankid = PRC_myrank, & ! [IN]
                                   single = .false.     ) ! [IN]

          dateday = 0
          datesec = CALENDAR_CFunits2sec( cftime(1), cfunits, 0 )

          call CALENDAR_adjust_daysec( dateday, datesec )

          call CALENDAR_daysec2date( TIME_STARTDATE, & ! [OUT]
                                     TIME_STARTMS,   & ! [OUT]
                                     dateday,        & ! [IN]
                                     datesec,        & ! [IN]
                                     0               ) ! [IN]
       else
          TIME_STARTDATE = (/ 0, 1, 1, 0, 0, 0 /)
          TIME_STARTMS = 0.0_DP
       endif
    else
       TIME_STARTMS = TIME_STARTMS * 1.E-3_DP
    endif

    TIME_OFFSET_YEAR = TIME_STARTDATE(1)

    call CALENDAR_date2daysec( TIME_STARTDAY,     & ! [OUT]
                               TIME_STARTSEC,     & ! [OUT]
                               TIME_STARTDATE(:), & ! [IN]
                               TIME_STARTMS,      & ! [IN]
                               TIME_OFFSET_YEAR   ) ! [IN]

    call CALENDAR_date2char( startchardate,     & ! [OUT]
                             TIME_STARTDATE(:), & ! [IN]
                             TIME_STARTMS       ) ! [IN]

    TIME_STARTDAYSEC  = CALENDAR_combine_daysec( TIME_STARTDAY, TIME_STARTSEC )

    TIME_NOWDATE(:)   = TIME_STARTDATE(:)
    TIME_NOWMS        = TIME_STARTMS
    TIME_NOWDAY       = TIME_STARTDAY
    TIME_NOWSEC       = TIME_STARTSEC
    TIME_NOWDAYSEC    = CALENDAR_combine_daysec( TIME_NOWDAY, TIME_NOWSEC )

    TIME_ENDDAY       = TIME_STARTDAY

    if ( setup_TimeIntegration ) then
       call CALENDAR_unit2sec( TIME_DURATIONSEC, TIME_DURATION, TIME_DURATION_UNIT )
       TIME_ENDSEC = TIME_STARTSEC + TIME_DURATIONSEC
    else
       TIME_ENDSEC = TIME_STARTSEC
    endif

    call CALENDAR_adjust_daysec( TIME_ENDDAY, TIME_ENDSEC ) ! [INOUT]

    call CALENDAR_daysec2date( TIME_ENDDATE(:), & ! [OUT]
                               TIME_ENDMS,      & ! [OUT]
                               TIME_ENDDAY,     & ! [IN]
                               TIME_ENDSEC,     & ! [IN]
                               TIME_OFFSET_YEAR ) ! [IN]

    call CALENDAR_date2char( endchardate,     & ! [OUT]
                             TIME_ENDDATE(:), & ! [IN]
                             TIME_ENDMS       ) ! [IN]

    LOG_NEWLINE
    LOG_INFO("ADMIN_TIME_setup",*) 'Global date / time setting '
    LOG_INFO_CONT('(1x,A,A)') 'START Date     : ', startchardate
    LOG_INFO_CONT('(1x,A,A)') 'END   Date     : ', endchardate

    if ( setup_TimeIntegration ) then

       call CALENDAR_unit2sec( TIME_DTSEC, TIME_DT, TIME_DT_UNIT )

       TIME_NSTEP   = int( TIME_DURATIONSEC / TIME_DTSEC )
       TIME_NOWSTEP = 1

       LOG_INFO_CONT('(1x,A,F12.3)') 'delta t (sec.) :', TIME_DTSEC
       LOG_INFO_CONT('(1x,A,I12)'  ) 'No. of steps   :', TIME_NSTEP

       !--- calculate intervals for atmosphere
       if ( TIME_DT_ATMOS_DYN /= UNDEF8 ) then
          call CALENDAR_unit2sec( TIME_DTSEC_ATMOS_DYN,     TIME_DT_ATMOS_DYN,     TIME_DT_ATMOS_DYN_UNIT     )
       else
          TIME_DTSEC_ATMOS_DYN = TIME_DTSEC / TIME_NSTEP_ATMOS_DYN
       endif
       call CALENDAR_unit2sec( TIME_DTSEC_ATMOS_PHY_CP,  TIME_DT_ATMOS_PHY_CP,  TIME_DT_ATMOS_PHY_CP_UNIT  )
       call CALENDAR_unit2sec( TIME_DTSEC_ATMOS_PHY_MP,  TIME_DT_ATMOS_PHY_MP,  TIME_DT_ATMOS_PHY_MP_UNIT  )
       call CALENDAR_unit2sec( TIME_DTSEC_ATMOS_PHY_RD,  TIME_DT_ATMOS_PHY_RD,  TIME_DT_ATMOS_PHY_RD_UNIT  )
       call CALENDAR_unit2sec( TIME_DTSEC_ATMOS_PHY_SF,  TIME_DT_ATMOS_PHY_SF,  TIME_DT_ATMOS_PHY_SF_UNIT  )
       call CALENDAR_unit2sec( TIME_DTSEC_ATMOS_PHY_TB,  TIME_DT_ATMOS_PHY_TB,  TIME_DT_ATMOS_PHY_TB_UNIT  )
       call CALENDAR_unit2sec( TIME_DTSEC_ATMOS_PHY_BL,  TIME_DT_ATMOS_PHY_BL,  TIME_DT_ATMOS_PHY_BL_UNIT  )
       call CALENDAR_unit2sec( TIME_DTSEC_ATMOS_PHY_CH,  TIME_DT_ATMOS_PHY_CH,  TIME_DT_ATMOS_PHY_CH_UNIT  )
       call CALENDAR_unit2sec( TIME_DTSEC_ATMOS_PHY_AE,  TIME_DT_ATMOS_PHY_AE,  TIME_DT_ATMOS_PHY_AE_UNIT  )
       call CALENDAR_unit2sec( TIME_DTSEC_ATMOS_RESTART, TIME_DT_ATMOS_RESTART, TIME_DT_ATMOS_RESTART_UNIT )
       call CALENDAR_unit2sec( TIME_DTSEC_ATMOS_DA,      TIME_DT_ATMOS_DA,      TIME_DT_ATMOS_DA_UNIT      )
       call CALENDAR_unit2sec( TIME_DTSEC_OCEAN,         TIME_DT_OCEAN,         TIME_DT_OCEAN_UNIT         )
       call CALENDAR_unit2sec( TIME_DTSEC_OCEAN_RESTART, TIME_DT_OCEAN_RESTART, TIME_DT_OCEAN_RESTART_UNIT )
       call CALENDAR_unit2sec( TIME_DTSEC_LAND,          TIME_DT_LAND,          TIME_DT_LAND_UNIT          )
       call CALENDAR_unit2sec( TIME_DTSEC_LAND_RESTART,  TIME_DT_LAND_RESTART,  TIME_DT_LAND_RESTART_UNIT  )
       call CALENDAR_unit2sec( TIME_DTSEC_URBAN,         TIME_DT_URBAN,         TIME_DT_URBAN_UNIT         )
       call CALENDAR_unit2sec( TIME_DTSEC_URBAN_RESTART, TIME_DT_URBAN_RESTART, TIME_DT_URBAN_RESTART_UNIT )
       call CALENDAR_unit2sec( TIME_DTSEC_RESUME,        TIME_DT_RESUME,        TIME_DT_RESUME_UNIT        )

       TIME_NSTEP_ATMOS_DYN = max( nint( TIME_DTSEC / TIME_DTSEC_ATMOS_DYN ), 1 )
       NSTEP_DYN = TIME_NSTEP_ATMOS_DYN

       TIME_DTSEC_ATMOS_DYN     = max( TIME_DTSEC_ATMOS_DYN,     TIME_DTSEC          /TIME_NSTEP_ATMOS_DYN )
       TIME_DTSEC_ATMOS_PHY_CP  = max( TIME_DTSEC_ATMOS_PHY_CP,  TIME_DTSEC_ATMOS_DYN*TIME_NSTEP_ATMOS_DYN )
       TIME_DTSEC_ATMOS_PHY_MP  = max( TIME_DTSEC_ATMOS_PHY_MP,  TIME_DTSEC_ATMOS_DYN*TIME_NSTEP_ATMOS_DYN )
       TIME_DTSEC_ATMOS_PHY_RD  = max( TIME_DTSEC_ATMOS_PHY_RD,  TIME_DTSEC_ATMOS_DYN*TIME_NSTEP_ATMOS_DYN )
       TIME_DTSEC_ATMOS_PHY_SF  = max( TIME_DTSEC_ATMOS_PHY_SF,  TIME_DTSEC_ATMOS_DYN*TIME_NSTEP_ATMOS_DYN )
       TIME_DTSEC_ATMOS_PHY_TB  = max( TIME_DTSEC_ATMOS_PHY_TB,  TIME_DTSEC_ATMOS_DYN*TIME_NSTEP_ATMOS_DYN )
       TIME_DTSEC_ATMOS_PHY_BL  = max( TIME_DTSEC_ATMOS_PHY_BL,  TIME_DTSEC_ATMOS_DYN*TIME_NSTEP_ATMOS_DYN )
       TIME_DTSEC_ATMOS_PHY_CH  = max( TIME_DTSEC_ATMOS_PHY_CH,  TIME_DTSEC_ATMOS_DYN*TIME_NSTEP_ATMOS_DYN )
       TIME_DTSEC_ATMOS_PHY_AE  = max( TIME_DTSEC_ATMOS_PHY_AE,  TIME_DTSEC_ATMOS_DYN*TIME_NSTEP_ATMOS_DYN )
       TIME_DTSEC_ATMOS_RESTART = max( TIME_DTSEC_ATMOS_RESTART, TIME_DTSEC_ATMOS_DYN*TIME_NSTEP_ATMOS_DYN )
       TIME_DTSEC_ATMOS_DA      = max( TIME_DTSEC_ATMOS_DA,      TIME_DTSEC_ATMOS_DYN*TIME_NSTEP_ATMOS_DYN )
       TIME_DTSEC_OCEAN         = max( TIME_DTSEC_OCEAN,         TIME_DTSEC_ATMOS_DYN*TIME_NSTEP_ATMOS_DYN )
       TIME_DTSEC_OCEAN_RESTART = max( TIME_DTSEC_OCEAN_RESTART, TIME_DTSEC_ATMOS_DYN*TIME_NSTEP_ATMOS_DYN )
       TIME_DTSEC_LAND          = max( TIME_DTSEC_LAND,          TIME_DTSEC_ATMOS_DYN*TIME_NSTEP_ATMOS_DYN )
       TIME_DTSEC_LAND_RESTART  = max( TIME_DTSEC_LAND_RESTART,  TIME_DTSEC_ATMOS_DYN*TIME_NSTEP_ATMOS_DYN )
       TIME_DTSEC_URBAN         = max( TIME_DTSEC_URBAN,         TIME_DTSEC_ATMOS_DYN*TIME_NSTEP_ATMOS_DYN )
       TIME_DTSEC_URBAN_RESTART = max( TIME_DTSEC_URBAN_RESTART, TIME_DTSEC_ATMOS_DYN*TIME_NSTEP_ATMOS_DYN )
       TIME_DTSEC_RESUME        = max( TIME_DTSEC_RESUME,        TIME_DTSEC_ATMOS_DYN*TIME_NSTEP_ATMOS_DYN )

       TIME_DSTEP_ATMOS_DYN     = nint( TIME_DTSEC_ATMOS_DYN     / TIME_DTSEC * TIME_NSTEP_ATMOS_DYN )
       TIME_DSTEP_ATMOS_PHY_CP  = nint( TIME_DTSEC_ATMOS_PHY_CP  / TIME_DTSEC )
       TIME_DSTEP_ATMOS_PHY_MP  = nint( TIME_DTSEC_ATMOS_PHY_MP  / TIME_DTSEC )
       TIME_DSTEP_ATMOS_PHY_RD  = nint( TIME_DTSEC_ATMOS_PHY_RD  / TIME_DTSEC )
       TIME_DSTEP_ATMOS_PHY_SF  = nint( TIME_DTSEC_ATMOS_PHY_SF  / TIME_DTSEC )
       TIME_DSTEP_ATMOS_PHY_TB  = nint( TIME_DTSEC_ATMOS_PHY_TB  / TIME_DTSEC )
       TIME_DSTEP_ATMOS_PHY_BL  = nint( TIME_DTSEC_ATMOS_PHY_BL  / TIME_DTSEC )
       TIME_DSTEP_ATMOS_PHY_CH  = nint( TIME_DTSEC_ATMOS_PHY_CH  / TIME_DTSEC )
       TIME_DSTEP_ATMOS_PHY_AE  = nint( TIME_DTSEC_ATMOS_PHY_AE  / TIME_DTSEC )
       TIME_DSTEP_OCEAN         = nint( TIME_DTSEC_OCEAN         / TIME_DTSEC )
       TIME_DSTEP_LAND          = nint( TIME_DTSEC_LAND          / TIME_DTSEC )
       TIME_DSTEP_URBAN         = nint( TIME_DTSEC_URBAN         / TIME_DTSEC )
       TIME_DSTEP_ATMOS_RESTART = nint( TIME_DTSEC_ATMOS_RESTART / TIME_DTSEC )
       TIME_DSTEP_ATMOS_DA      = nint( TIME_DTSEC_ATMOS_DA      / TIME_DTSEC )
       TIME_DSTEP_OCEAN_RESTART = nint( TIME_DTSEC_OCEAN_RESTART / TIME_DTSEC )
       TIME_DSTEP_LAND_RESTART  = nint( TIME_DTSEC_LAND_RESTART  / TIME_DTSEC )
       TIME_DSTEP_URBAN_RESTART = nint( TIME_DTSEC_URBAN_RESTART / TIME_DTSEC )
       TIME_DSTEP_RESUME        = nint( TIME_DTSEC_RESUME        / TIME_DTSEC )

       TIME_RES_RESUME = TIME_DSTEP_RESUME - 1

       if ( abs( real(TIME_NSTEP_ATMOS_DYN,kind=DP)*TIME_DTSEC_ATMOS_DYN &
               - real(TIME_DSTEP_ATMOS_DYN,kind=DP)*TIME_DTSEC            ) > eps ) then
          LOG_ERROR("ADMIN_TIME_setup",*) 'delta t(ATMOS_DYN) must be a multiple of delta t ', &
                     TIME_DTSEC_ATMOS_DYN, real(TIME_DSTEP_ATMOS_DYN,kind=DP)*TIME_DTSEC
          call PRC_abort
       endif
       if ( abs(TIME_DTSEC_ATMOS_PHY_CP-real(TIME_DSTEP_ATMOS_PHY_CP,kind=DP)*TIME_DTSEC) > eps ) then
          LOG_ERROR("ADMIN_TIME_setup",*) 'delta t(ATMOS_PHY_CP) must be a multiple of delta t ', &
                     TIME_DTSEC_ATMOS_PHY_CP, real(TIME_DSTEP_ATMOS_PHY_CP,kind=DP)*TIME_DTSEC
          call PRC_abort
       endif
       if ( abs(TIME_DTSEC_ATMOS_PHY_MP-real(TIME_DSTEP_ATMOS_PHY_MP,kind=DP)*TIME_DTSEC) > eps ) then
          LOG_ERROR("ADMIN_TIME_setup",*) 'delta t(ATMOS_PHY_MP) must be a multiple of delta t ', &
                     TIME_DTSEC_ATMOS_PHY_MP, real(TIME_DSTEP_ATMOS_PHY_MP,kind=DP)*TIME_DTSEC
          call PRC_abort
       endif
       if ( abs(TIME_DTSEC_ATMOS_PHY_RD-real(TIME_DSTEP_ATMOS_PHY_RD,kind=DP)*TIME_DTSEC) > eps ) then
          LOG_ERROR("ADMIN_TIME_setup",*) 'delta t(ATMOS_PHY_RD) must be a multiple of delta t ', &
                     TIME_DTSEC_ATMOS_PHY_RD, real(TIME_DSTEP_ATMOS_PHY_RD,kind=DP)*TIME_DTSEC
          call PRC_abort
       endif
       if ( abs(TIME_DTSEC_ATMOS_PHY_SF-real(TIME_DSTEP_ATMOS_PHY_SF,kind=DP)*TIME_DTSEC) > eps ) then
          LOG_ERROR("ADMIN_TIME_setup",*) 'delta t(ATMOS_PHY_SF) must be a multiple of delta t ', &
                     TIME_DTSEC_ATMOS_PHY_SF, real(TIME_DSTEP_ATMOS_PHY_SF,kind=DP)*TIME_DTSEC
          call PRC_abort
       endif
       if ( abs(TIME_DTSEC_ATMOS_PHY_TB-real(TIME_DSTEP_ATMOS_PHY_TB,kind=DP)*TIME_DTSEC) > eps ) then
          LOG_ERROR("ADMIN_TIME_setup",*) 'delta t(ATMOS_PHY_TB) must be a multiple of delta t ', &
                     TIME_DTSEC_ATMOS_PHY_TB, real(TIME_DSTEP_ATMOS_PHY_TB,kind=DP)*TIME_DTSEC
          call PRC_abort
       endif
       if ( abs(TIME_DTSEC_ATMOS_PHY_BL-real(TIME_DSTEP_ATMOS_PHY_BL,kind=DP)*TIME_DTSEC) > eps ) then
          LOG_ERROR("ADMIN_TIME_setup",*) 'delta t(ATMOS_PHY_BL) must be a multiple of delta t ', &
                     TIME_DTSEC_ATMOS_PHY_BL, real(TIME_DSTEP_ATMOS_PHY_BL,kind=DP)*TIME_DTSEC
          call PRC_abort
       endif
       if ( abs(TIME_DTSEC_ATMOS_PHY_CH-real(TIME_DSTEP_ATMOS_PHY_CH,kind=DP)*TIME_DTSEC) > eps ) then
          LOG_ERROR("ADMIN_TIME_setup",*) 'delta t(ATMOS_PHY_CH) must be a multiple of delta t ', &
                     TIME_DTSEC_ATMOS_PHY_CH, real(TIME_DSTEP_ATMOS_PHY_CH,kind=DP)*TIME_DTSEC
          call PRC_abort
       endif
       if ( abs(TIME_DTSEC_ATMOS_PHY_AE-real(TIME_DSTEP_ATMOS_PHY_AE,kind=DP)*TIME_DTSEC) > eps ) then
          LOG_ERROR("ADMIN_TIME_setup",*) 'delta t(ATMOS_PHY_AE) must be a multiple of delta t ', &
                     TIME_DTSEC_ATMOS_PHY_AE, real(TIME_DSTEP_ATMOS_PHY_AE,kind=DP)*TIME_DTSEC
          call PRC_abort
       endif
       if ( abs(TIME_DTSEC_OCEAN-real(TIME_DSTEP_OCEAN,kind=DP)*TIME_DTSEC) > eps ) then
          LOG_ERROR("ADMIN_TIME_setup",*) 'delta t(OCEAN) must be a multiple of delta t ', &
                     TIME_DTSEC_OCEAN, real(TIME_DSTEP_OCEAN,kind=DP)*TIME_DTSEC
          call PRC_abort
       endif
       if ( abs(TIME_DTSEC_LAND-real(TIME_DSTEP_LAND,kind=DP)*TIME_DTSEC) > eps ) then
          LOG_ERROR("ADMIN_TIME_setup",*) 'delta t(LAND) must be a multiple of delta t ', &
                     TIME_DTSEC_LAND, real(TIME_DSTEP_LAND,kind=DP)*TIME_DTSEC
          call PRC_abort
       endif
       if ( abs(TIME_DTSEC_URBAN-real(TIME_DSTEP_URBAN,kind=DP)*TIME_DTSEC) > eps ) then
          LOG_ERROR("ADMIN_TIME_setup",*) 'delta t(URBAN) must be a multiple of delta t ', &
                     TIME_DTSEC_URBAN, real(TIME_DSTEP_URBAN,kind=DP)*TIME_DTSEC
          call PRC_abort
       endif
       if ( abs(TIME_DTSEC_ATMOS_RESTART-real(TIME_DSTEP_ATMOS_RESTART,kind=DP)*TIME_DTSEC) > eps ) then
          LOG_ERROR("ADMIN_TIME_setup",*) 'delta t(ATMOS_RESTART) must be a multiple of delta t ', &
                     TIME_DTSEC_ATMOS_RESTART, real(TIME_DSTEP_ATMOS_RESTART,kind=DP)*TIME_DTSEC
          call PRC_abort
       endif
       if ( abs(TIME_DTSEC_OCEAN_RESTART-real(TIME_DSTEP_OCEAN_RESTART,kind=DP)*TIME_DTSEC) > eps ) then
          LOG_ERROR("ADMIN_TIME_setup",*) 'delta t(OCEAN_RESTART) must be a multiple of delta t ', &
                     TIME_DTSEC_OCEAN_RESTART, real(TIME_DSTEP_OCEAN_RESTART,kind=DP)*TIME_DTSEC
          call PRC_abort
       endif
       if ( abs(TIME_DTSEC_LAND_RESTART-real(TIME_DSTEP_LAND_RESTART,kind=DP)*TIME_DTSEC) > eps ) then
          LOG_ERROR("ADMIN_TIME_setup",*) 'delta t(LAND_RESTART) must be a multiple of delta t ', &
                     TIME_DTSEC_LAND_RESTART, real(TIME_DSTEP_LAND_RESTART,kind=DP)*TIME_DTSEC
          call PRC_abort
       endif
       if ( abs(TIME_DTSEC_URBAN_RESTART-real(TIME_DSTEP_URBAN_RESTART,kind=DP)*TIME_DTSEC) > eps ) then
          LOG_ERROR("ADMIN_TIME_setup",*) 'delta t(URBAN_RESTART) must be a multiple of delta t ', &
                     TIME_DTSEC_URBAN_RESTART, real(TIME_DSTEP_URBAN_RESTART,kind=DP)*TIME_DTSEC
          call PRC_abort
       endif
       if ( abs(TIME_DTSEC_URBAN_RESTART-real(TIME_DSTEP_URBAN_RESTART,kind=DP)*TIME_DTSEC) > eps ) then
          LOG_ERROR("ADMIN_TIME_setup",*) 'delta t(URBAN_RESTART) must be a multiple of delta t ', &
                     TIME_DTSEC_URBAN_RESTART, real(TIME_DSTEP_URBAN_RESTART,kind=DP)*TIME_DTSEC
          call PRC_abort
       endif
       if ( abs(TIME_DTSEC_RESUME-real(TIME_DSTEP_RESUME,kind=DP)*TIME_DTSEC) > eps ) then
          LOG_ERROR("ADMIN_TIME_setup",*) 'delta t(RESUME) must be a multiple of delta t ', &
                     TIME_DTSEC_RESUME, real(TIME_DSTEP_RESUME,kind=DP)*TIME_DTSEC
          call PRC_abort
       endif
       if ( abs(TIME_DTSEC_ATMOS_DA-real(TIME_DSTEP_ATMOS_DA,kind=DP)*TIME_DTSEC) > eps ) then
          LOG_ERROR("ADMIN_TIME_setup",*) 'delta t(ATMOS_DA) must be a multiple of delta t ', &
                     TIME_DTSEC_ATMOS_DA, real(TIME_DSTEP_ATMOS_DA,kind=DP)*TIME_DTSEC
          call PRC_abort
       endif

       LOG_NEWLINE
       LOG_INFO("ADMIN_TIME_setup",*)                     'Time interval for each component (sec.)'
       LOG_INFO_CONT(*)                     'Atmosphere'
       LOG_INFO_CONT('(1x,A,F10.3)')        'Dynamics (time)             : ', TIME_DTSEC_ATMOS_DYN
       LOG_INFO_CONT('(1x,A,I10,A,I8,A)')   '(step)             : ', TIME_NSTEP_ATMOS_DYN, &
                                       ' (step interval=', TIME_DSTEP_ATMOS_DYN,    ')'
       LOG_INFO_CONT('(1x,A,F10.3,A,I8,A)') 'Physics, Cumulus            : ', TIME_DTSEC_ATMOS_PHY_CP, &
                                       ' (step interval=', TIME_DSTEP_ATMOS_PHY_CP, ')'
       LOG_INFO_CONT('(1x,A,F10.3,A,I8,A)') 'Physics, Cloud Microphysics : ', TIME_DTSEC_ATMOS_PHY_MP, &
                                       ' (step interval=', TIME_DSTEP_ATMOS_PHY_MP, ')'
       LOG_INFO_CONT('(1x,A,F10.3,A,I8,A)') 'Physics, Radiation          : ', TIME_DTSEC_ATMOS_PHY_RD, &
                                       ' (step interval=', TIME_DSTEP_ATMOS_PHY_RD, ')'
       LOG_INFO_CONT('(1x,A,F10.3,A,I8,A)') 'Physics, Surface Flux       : ', TIME_DTSEC_ATMOS_PHY_SF, &
                                       ' (step interval=', TIME_DSTEP_ATMOS_PHY_SF, ')'
       LOG_INFO_CONT('(1x,A,F10.3,A,I8,A)') 'Physics, Turbulence         : ', TIME_DTSEC_ATMOS_PHY_TB, &
                                       ' (step interval=', TIME_DSTEP_ATMOS_PHY_TB, ')'
       LOG_INFO_CONT('(1x,A,F10.3,A,I8,A)') 'Physics, Boundary layer     : ', TIME_DTSEC_ATMOS_PHY_BL, &
                                       ' (step interval=', TIME_DSTEP_ATMOS_PHY_BL, ')'
       LOG_INFO_CONT('(1x,A,F10.3,A,I8,A)') 'Physics, Chemistry          : ', TIME_DTSEC_ATMOS_PHY_CH, &
                                       ' (step interval=', TIME_DSTEP_ATMOS_PHY_CH, ')'
       LOG_INFO_CONT('(1x,A,F10.3,A,I8,A)') 'Physics, Aerosol            : ', TIME_DTSEC_ATMOS_PHY_AE, &
                                       ' (step interval=', TIME_DSTEP_ATMOS_PHY_AE, ')'
       LOG_INFO_CONT('(1x,A,F10.3,A,I8,A)') 'Ocean                       : ', TIME_DTSEC_OCEAN, &
                                       ' (step interval=', TIME_DSTEP_OCEAN,        ')'
       LOG_INFO_CONT('(1x,A,F10.3,A,I8,A)') 'Land                        : ', TIME_DTSEC_LAND, &
                                       ' (step interval=', TIME_DSTEP_LAND,         ')'
       LOG_INFO_CONT('(1x,A,F10.3,A,I8,A)') 'Urban                       : ', TIME_DTSEC_URBAN, &
                                       ' (step interval=', TIME_DSTEP_URBAN,        ')'
       LOG_NEWLINE
       LOG_INFO_CONT(*)                     'Time interval for restart (sec.)'
       LOG_INFO_CONT('(1x,A,F10.3,A,I8,A)') 'Atmospheric Variables       : ', TIME_DTSEC_ATMOS_RESTART, &
                                       ' (step interval=', TIME_DSTEP_ATMOS_RESTART, ')'
       LOG_INFO_CONT('(1x,A,F10.3,A,I8,A)') 'Ocean       Variables       : ', TIME_DTSEC_OCEAN_RESTART, &
                                       ' (step interval=', TIME_DSTEP_OCEAN_RESTART, ')'
       LOG_INFO_CONT('(1x,A,F10.3,A,I8,A)') 'Land        Variables       : ', TIME_DTSEC_LAND_RESTART,  &
                                       ' (step interval=', TIME_DSTEP_LAND_RESTART,  ')'
       LOG_INFO_CONT('(1x,A,F10.3,A,I8,A)') 'Urban       Variables       : ', TIME_DTSEC_URBAN_RESTART, &
                                       ' (step interval=', TIME_DSTEP_URBAN_RESTART, ')'
       LOG_INFO_CONT('(1x,A,F10.3,A,I8,A)') 'Resume                      : ', TIME_DTSEC_RESUME, &
                                                          ' (step interval=', TIME_DSTEP_RESUME,        ')'
       LOG_INFO_CONT('(1x,A,F10.3,A,I8,A)') 'Atmospheric DA       : ', TIME_DTSEC_ATMOS_DA, &
                                       ' (step interval=', TIME_DSTEP_ATMOS_DA, ')'
    else
       TIME_DTSEC = 1.0_RP
    endif

    TIME_WALLCLOCK_START = PRC_MPItime()

    ! WALLCLOCK TERMINATOR SETUP
    if ( TIME_WALLCLOCK_LIMIT > 0.0_DP ) then
       LOG_NEWLINE
       LOG_INFO("ADMIN_TIME_setup",*) 'Wall clock time limit of execution is specified.'

       if ( TIME_DT_WALLCLOCK_CHECK == UNDEF8 ) then
          LOG_INFO_CONT(*) 'Not found TIME_DT_WALLCLOCK_CHECK. largest time step interval is used.'
          TIME_DTSEC_WALLCLOCK_CHECK = max( TIME_DTSEC_ATMOS_DYN*TIME_NSTEP_ATMOS_DYN, &
                                            TIME_DTSEC_ATMOS_PHY_CP,                   &
                                            TIME_DTSEC_ATMOS_PHY_MP,                   &
                                            TIME_DTSEC_ATMOS_PHY_RD,                   &
                                            TIME_DTSEC_ATMOS_PHY_SF,                   &
                                            TIME_DTSEC_ATMOS_PHY_TB,                   &
                                            TIME_DTSEC_ATMOS_PHY_BL,                   &
                                            TIME_DTSEC_ATMOS_PHY_CH,                   &
                                            TIME_DTSEC_ATMOS_PHY_AE,                   &
                                            TIME_DTSEC_OCEAN,                          &
                                            TIME_DTSEC_LAND,                           &
                                            TIME_DTSEC_URBAN                           )
       else
          if ( TIME_DT_WALLCLOCK_CHECK_UNIT == '' ) then
             LOG_INFO_CONT(*) 'Not found TIME_DT_WALLCLOCK_CHECK_UNIT. TIME_DURATION_UNIT is used.'
             TIME_DT_WALLCLOCK_CHECK_UNIT = TIME_DURATION_UNIT
          endif
          call CALENDAR_unit2sec( TIME_DTSEC_WALLCLOCK_CHECK, TIME_DT_WALLCLOCK_CHECK, TIME_DT_WALLCLOCK_CHECK_UNIT )
          TIME_DTSEC_WALLCLOCK_CHECK = max( TIME_DTSEC_WALLCLOCK_CHECK, TIME_DTSEC_ATMOS_DYN*TIME_NSTEP_ATMOS_DYN )
       endif

       TIME_DSTEP_WALLCLOCK_CHECK = int( TIME_DTSEC_WALLCLOCK_CHECK / TIME_DTSEC )

       TIME_WALLCLOCK_SAFE    = max( min( TIME_WALLCLOCK_SAFE, 1.0_DP ), 0.0_DP )
       TIME_WALLCLOCK_safelim = TIME_WALLCLOCK_LIMIT * TIME_WALLCLOCK_SAFE

       LOG_INFO_CONT('(1x,A,F10.1,A)')      'This job stops after ', TIME_WALLCLOCK_safelim, ' seconds.'
       LOG_INFO_CONT('(1x,A,F10.3,A,I8,A)') 'Time interval for check     : ', TIME_DTSEC_WALLCLOCK_CHECK, &
                                            ' (step interval=', TIME_DSTEP_WALLCLOCK_CHECK, ')'
    endif

    if ( debug ) then
       LOG_INFO("ADMIN_TIME_setup",*) TIME_NOWDAY, TIME_NOWSEC, TIME_NOWDATE(:), TIME_NOWMS
     endif

    return
  end subroutine ADMIN_TIME_setup

  !-----------------------------------------------------------------------------
  !> Evaluate component execution
  subroutine ADMIN_TIME_checkstate
    use scale_prc, only: &
       PRC_UNIVERSAL_IsMaster, &
       PRC_MPItime
    use scale_calendar, only: &
       CALENDAR_date2char
    use scale_time, only: &
       TIME_NOWDATE,            &
       TIME_NOWMS,              &
       TIME_NOWSTEP,            &
       TIME_NSTEP,              &
       TIME_DSTEP_ATMOS_DYN,    &
       TIME_DSTEP_ATMOS_PHY_CP, &
       TIME_DSTEP_ATMOS_PHY_MP, &
       TIME_DSTEP_ATMOS_PHY_RD, &
       TIME_DSTEP_ATMOS_PHY_SF, &
       TIME_DSTEP_ATMOS_PHY_TB, &
       TIME_DSTEP_ATMOS_PHY_BL, &
       TIME_DSTEP_ATMOS_PHY_CH, &
       TIME_DSTEP_ATMOS_PHY_AE, &
       TIME_DSTEP_OCEAN,        &
       TIME_DSTEP_LAND,         &
       TIME_DSTEP_URBAN
    implicit none

    real(DP)          :: WALLCLOCK_elapse
    character(len=27) :: nowchardate
    logical           :: TO_STDOUT
    !---------------------------------------------------------------------------

    TIME_DOATMOS_step     = .false.
    TIME_DOATMOS_DYN      = .false.
    TIME_DOATMOS_PHY_CP   = .false.
    TIME_DOATMOS_PHY_MP   = .false.
    TIME_DOATMOS_PHY_RD   = .false.
    TIME_DOATMOS_PHY_SF   = .false.
    TIME_DOATMOS_PHY_TB   = .false.
    TIME_DOATMOS_PHY_BL   = .false.
    TIME_DOATMOS_PHY_CH   = .false.
    TIME_DOATMOS_PHY_AE   = .false.
    TIME_DOOCEAN_step     = .false.
    TIME_DOLAND_step      = .false.
    TIME_DOURBAN_step     = .false.
    TIME_DOresume         = .false.

    TIME_RES_ATMOS_DYN    = TIME_RES_ATMOS_DYN    + 1
    TIME_RES_ATMOS_PHY_CP = TIME_RES_ATMOS_PHY_CP + 1
    TIME_RES_ATMOS_PHY_MP = TIME_RES_ATMOS_PHY_MP + 1
    TIME_RES_ATMOS_PHY_RD = TIME_RES_ATMOS_PHY_RD + 1
    TIME_RES_ATMOS_PHY_SF = TIME_RES_ATMOS_PHY_SF + 1
    TIME_RES_ATMOS_PHY_TB = TIME_RES_ATMOS_PHY_TB + 1
    TIME_RES_ATMOS_PHY_BL = TIME_RES_ATMOS_PHY_BL + 1
    TIME_RES_ATMOS_PHY_CH = TIME_RES_ATMOS_PHY_CH + 1
    TIME_RES_ATMOS_PHY_AE = TIME_RES_ATMOS_PHY_AE + 1
    TIME_RES_OCEAN        = TIME_RES_OCEAN        + 1
    TIME_RES_LAND         = TIME_RES_LAND         + 1
    TIME_RES_URBAN        = TIME_RES_URBAN        + 1
    TIME_RES_RESUME       = TIME_RES_RESUME       + 1

    if ( TIME_RES_ATMOS_DYN    == TIME_DSTEP_ATMOS_DYN ) then
       TIME_DOATMOS_step     = .true.
       TIME_DOATMOS_DYN      = .true.
       TIME_RES_ATMOS_DYN    = 0
    endif
    if ( TIME_RES_ATMOS_PHY_CP == TIME_DSTEP_ATMOS_PHY_CP ) then
       TIME_DOATMOS_step     = .true.
       TIME_DOATMOS_PHY_CP   = .true.
       TIME_RES_ATMOS_PHY_CP = 0
    endif
    if ( TIME_RES_ATMOS_PHY_MP == TIME_DSTEP_ATMOS_PHY_MP ) then
       TIME_DOATMOS_step     = .true.
       TIME_DOATMOS_PHY_MP   = .true.
       TIME_RES_ATMOS_PHY_MP = 0
    endif
    if ( TIME_RES_ATMOS_PHY_RD == TIME_DSTEP_ATMOS_PHY_RD ) then
       TIME_DOATMOS_step     = .true.
       TIME_DOATMOS_PHY_RD   = .true.
       TIME_RES_ATMOS_PHY_RD = 0
    endif
    if ( TIME_RES_ATMOS_PHY_SF == TIME_DSTEP_ATMOS_PHY_SF ) then
       TIME_DOATMOS_step     = .true.
       TIME_DOATMOS_PHY_SF   = .true.
       TIME_RES_ATMOS_PHY_SF = 0
    endif
    if ( TIME_RES_ATMOS_PHY_TB == TIME_DSTEP_ATMOS_PHY_TB ) then
       TIME_DOATMOS_step     = .true.
       TIME_DOATMOS_PHY_TB   = .true.
       TIME_RES_ATMOS_PHY_TB = 0
    endif
    if ( TIME_RES_ATMOS_PHY_BL == TIME_DSTEP_ATMOS_PHY_BL ) then
       TIME_DOATMOS_step     = .true.
       TIME_DOATMOS_PHY_BL   = .true.
       TIME_RES_ATMOS_PHY_BL = 0
    endif
    if ( TIME_RES_ATMOS_PHY_CH == TIME_DSTEP_ATMOS_PHY_CH ) then
       TIME_DOATMOS_step     = .true.
       TIME_DOATMOS_PHY_CH   = .true.
       TIME_RES_ATMOS_PHY_CH = 0
    endif
    if ( TIME_RES_ATMOS_PHY_AE == TIME_DSTEP_ATMOS_PHY_AE ) then
       TIME_DOATMOS_step     = .true.
       TIME_DOATMOS_PHY_AE   = .true.
       TIME_RES_ATMOS_PHY_AE = 0
    endif

    if ( TIME_RES_OCEAN  == TIME_DSTEP_OCEAN  ) then
       TIME_DOOCEAN_step     = .true.
       TIME_RES_OCEAN        = 0
    endif
    if ( TIME_RES_LAND   == TIME_DSTEP_LAND   ) then
       TIME_DOLAND_step      = .true.
       TIME_RES_LAND         = 0
    endif
    if ( TIME_RES_URBAN  == TIME_DSTEP_URBAN  ) then
       TIME_DOURBAN_step     = .true.
       TIME_RES_URBAN        = 0
    endif
    if ( TIME_RES_RESUME == TIME_DSTEP_RESUME ) then
       TIME_DOresume         = .true.
       TIME_RES_RESUME       = 0
    endif

    TO_STDOUT = .false.
    if ( IO_STEP_TO_STDOUT > 0 ) then
       if( mod(TIME_NOWSTEP-1,IO_STEP_TO_STDOUT) == 0 ) TO_STDOUT = .true.
    endif

    call CALENDAR_date2char( nowchardate,     & ! [OUT]
                             TIME_NOWDATE(:), & ! [IN]
                             TIME_NOWMS       ) ! [IN]

    WALLCLOCK_elapse = PRC_MPItime() - TIME_WALLCLOCK_START

    LOG_NEWLINE
    if ( TIME_WALLCLOCK_LIMIT > 0.0_DP ) then
       LOG_PROGRESS('(1x,2A,2(A,I7),2(A,F10.1))') 'TIME: ', nowchardate,' STEP:',TIME_NOWSTEP, '/', TIME_NSTEP, &
                                                                 ' WCLOCK:', WALLCLOCK_elapse, '/', TIME_WALLCLOCK_safelim
       if ( PRC_UNIVERSAL_IsMaster .AND. TO_STDOUT ) then ! universal master node
          write(*,'(1x,2A,2(A,I7),2(A,F10.1))') 'TIME: ', nowchardate,' STEP:',TIME_NOWSTEP, '/', TIME_NSTEP, &
                                                ' WCLOCK:', WALLCLOCK_elapse, '/', TIME_WALLCLOCK_safelim
       endif
    else
       LOG_PROGRESS('(1x,2A,2(A,I7),A,F10.1)') 'TIME: ', nowchardate,' STEP:',TIME_NOWSTEP, '/', TIME_NSTEP, &
                                                              ' WCLOCK:', WALLCLOCK_elapse
       if ( PRC_UNIVERSAL_IsMaster .AND. TO_STDOUT ) then ! universal master node
          write(*,'(1x,2A,2(A,I7),A,F10.1)') 'TIME: ', nowchardate,' STEP:',TIME_NOWSTEP, '/', TIME_NSTEP, &
                                             ' WCLOCK:', WALLCLOCK_elapse
       endif
    endif

    return
  end subroutine ADMIN_TIME_checkstate

  !-----------------------------------------------------------------------------
  !> Advance the time & evaluate restart & stop
  subroutine ADMIN_TIME_advance
    use scale_prc, only: &
       PRC_IsMaster, &
       PRC_MPItime
    use scale_calendar, only: &
       CALENDAR_daysec2date,   &
       CALENDAR_adjust_daysec, &
       CALENDAR_combine_daysec
    use scale_comm_cartesC, only: &
       COMM_bcast
    use scale_time, only: &
       TIME_DTSEC,                &
       TIME_NOWDATE,              &
       TIME_NOWMS,                &
       TIME_NOWDAY,               &
       TIME_NOWSEC,               &
       TIME_NOWDAYSEC,            &
       TIME_NOWSTEP,              &
       TIME_NSTEP,                &
       TIME_OFFSET_YEAR,          &
       TIME_DSTEP_WALLCLOCK_CHECK
    implicit none

    real(DP) :: WALLCLOCK_elapse
    logical  :: WALLCLOCK_DOend
    !---------------------------------------------------------------------------

    TIME_DOend = .false.

    TIME_NOWSTEP = TIME_NOWSTEP + 1
    TIME_NOWDAY  = TIME_STARTDAY
    TIME_NOWSEC  = TIME_STARTSEC + real(TIME_NOWSTEP-1,kind=DP) * TIME_DTSEC

    ! reallocate day & sub-day
    call CALENDAR_adjust_daysec( TIME_NOWDAY, TIME_NOWSEC ) ! [INOUT]

    call CALENDAR_daysec2date( TIME_NOWDATE(:), & ! [OUT]
                               TIME_NOWMS,      & ! [OUT]
                               TIME_NOWDAY,     & ! [IN]
                               TIME_NOWSEC,     & ! [IN]
                               TIME_OFFSET_YEAR ) ! [IN]

    TIME_NOWDAYSEC = CALENDAR_combine_daysec( TIME_NOWDAY, TIME_NOWSEC )

    if ( debug ) then
       LOG_INFO("ADMIN_TIME_advance",*) TIME_NOWDAY, TIME_NOWSEC, TIME_NOWDATE(:), TIME_NOWMS
    endif

    if ( TIME_NOWSTEP > TIME_NSTEP ) then
       TIME_DOend = .true.
    endif

    if ( TIME_WALLCLOCK_LIMIT > 0.0_DP ) then ! use wallclock limiter
       WALLCLOCK_DOend = .false.

       if (       PRC_IsMaster                                        &      ! master node
            .AND. mod(TIME_NOWSTEP-1,TIME_DSTEP_WALLCLOCK_CHECK) == 0 ) then ! step to check

          WALLCLOCK_elapse = PRC_MPItime() - TIME_WALLCLOCK_START

          if( WALLCLOCK_elapse > TIME_WALLCLOCK_safelim ) WALLCLOCK_DOend = .true.

       endif

       call COMM_bcast( WALLCLOCK_DOend ) ! [INOUT]

       if ( WALLCLOCK_DOend ) then
          LOG_NEWLINE
          LOG_INFO("ADMIN_TIME_advance",*) '**************************************************************'
          LOG_INFO("ADMIN_TIME_advance",*) 'Elapse time limit is detected. Termination operation starts. '
          LOG_INFO("ADMIN_TIME_advance",*) '**************************************************************'
          LOG_NEWLINE
          TIME_DOend = .true.
       endif
    endif

    TIME_DOATMOS_restart = .false.
    TIME_DOOCEAN_restart = .false.
    TIME_DOLAND_restart  = .false.
    TIME_DOURBAN_restart = .false.
    TIME_DOATMOS_DA      = .false.

    TIME_RES_ATMOS_RESTART = TIME_RES_ATMOS_RESTART + 1
    TIME_RES_OCEAN_RESTART = TIME_RES_OCEAN_RESTART + 1
    TIME_RES_LAND_RESTART  = TIME_RES_LAND_RESTART  + 1
    TIME_RES_URBAN_RESTART = TIME_RES_URBAN_RESTART + 1
    TIME_RES_ATMOS_DA = TIME_RES_ATMOS_DA + 1

    if ( TIME_RES_ATMOS_RESTART == TIME_DSTEP_ATMOS_RESTART ) then
       TIME_DOATMOS_restart   = .true.
       TIME_RES_ATMOS_RESTART = 0
    elseif( TIME_DOend .and. TIME_END_RESTART_OUT ) then
       TIME_DOATMOS_restart   = .true.
    endif

    if ( TIME_RES_OCEAN_RESTART == TIME_DSTEP_OCEAN_RESTART ) then
       TIME_DOOCEAN_restart   = .true.
       TIME_RES_OCEAN_RESTART = 0
    elseif( TIME_DOend .and. TIME_END_RESTART_OUT ) then
       TIME_DOOCEAN_restart   = .true.
    endif

    if ( TIME_RES_LAND_RESTART  == TIME_DSTEP_LAND_RESTART  ) then
       TIME_DOLAND_restart    = .true.
       TIME_RES_LAND_RESTART  = 0
    elseif( TIME_DOend .and. TIME_END_RESTART_OUT ) then
       TIME_DOLAND_restart    = .true.
    endif

    if ( TIME_RES_URBAN_RESTART == TIME_DSTEP_URBAN_RESTART ) then
       TIME_DOURBAN_restart   = .true.
       TIME_RES_URBAN_RESTART = 0
    elseif( TIME_DOend .and. TIME_END_RESTART_OUT ) then
       TIME_DOURBAN_restart   = .true.
    endif

    if ( TIME_RES_ATMOS_DA == TIME_DSTEP_ATMOS_DA ) then
       TIME_DOATMOS_DA   = .true.
       TIME_RES_ATMOS_DA = 0
    endif

    return
  end subroutine ADMIN_TIME_advance

end module mod_admin_time
