package edu.mit.csail.sdg.alloy4;

import kodkod.ast.Formula;
import kodkod.engine.hol.HOLTranslation;
import kodkod.instance.Bounds;
import kodkod.instance.Instance;
import edu.mit.csail.sdg.alloy4compiler.translator.A4Solution;

public interface IA4Reporter {

    /** This method is called at various points to report the current progress;
     * it is intended as a debugging aid for the developers; the messages are generally not useful for end users.
     */
    public void debug(String msg);

    /** This method is called by the parser to report parser events. */
    public void parse(String msg);

    /** This method is called by the typechecker to report the type for each field/function/predicate/assertion, etc. */
    public void typecheck(String msg);

    /** This method is called by the typechecker to report a nonfatal type error. */
    public void warning(ErrorWarning msg);

    /** This method is called by the ScopeComputer to report the scope chosen for each sig. */
    public void scope(String msg);

    /** This method is called by the BoundsComputer to report the bounds chosen for each sig and each field. */
    public void bound(String msg);

    public void generatingSolution(Formula fgoal, Bounds bounds);

    /** This method is called by the translator just before it begins generating CNF.
     *
     * @param solver - the solver chosen by the user (eg. SAT4J, MiniSat...)
     * @param bitwidth - the integer bitwidth chosen by the user
     * @param maxseq - the scope on seq/Int chosen by the user
     * @param skolemDepth - the skolem function depth chosen by the user (0, 1, 2...)
     * @param symmetry - the amount of symmetry breaking chosen by the user (0...)
     */
    public void translate(String solver, int bitwidth, int maxseq, int skolemDepth, int symmetry);

    /** This method is called by the translator just after it generated the CNF.
     *
     * @param primaryVars - the total number of primary variables
     * @param totalVars - the total number of variables including the number of primary variables
     * @param clauses - the total number of clauses
     */
    public void solve(int primaryVars, int totalVars, int clauses);

    /** If solver==KK or solver==CNF, this method is called by the translator after it constructed the Kodkod or CNF file.
     *
     * @param filename - the Kodkod or CNF file generated by the translator
     */
    public void resultCNF(String filename);

    /** If solver!=KK and solver!=CNF, this method is called by the translator if the formula is satisfiable.
     *
     * @param command - this is the original Command used to generate this solution
     * @param solvingTime - this is the number of milliseconds the solver took to obtain this result
     * @param solution - the satisfying A4Solution object
     */
    public void resultSAT(Object command, long solvingTime, Object solution);

    /** If solver!=KK and solver!=CNF, this method is called by the translator before starting the unsat core minimization.
     *
     * @param command - this is the original Command used to generate this solution
     * @param before - the size of the unsat core before calling minimization
     */
    public void minimizing(Object command, int before);

    /** If solver!=KK and solver!=CNF, this method is called by the translator after performing the unsat core minimization.
     *
     * @param command - this is the original Command used to generate this solution
     * @param before - the size of the unsat core before calling minimization
     * @param after - the size of the unsat core after calling minimization
     */
    public void minimized(Object command, int before, int after);

    /** If solver!=KK and solver!=CNF, this method is called by the translator if the formula is unsatisfiable.
     *
     * @param command - this is the original Command used to generate this solution
     * @param solvingTime - this is the number of milliseconds the solver took to obtain this result
     * @param solution - the unsatisfying A4Solution object
     */
    public void resultUNSAT(Object command, long solvingTime, Object solution);

    /** This method is called by the A4SolutionWriter when it is writing a particular sig, field, or skolem. */
    public void write(Object expr);

    public void holLoopStart(HOLTranslation tr, Formula formula, Bounds bounds);
    public void holCandidateFound(HOLTranslation tr, Instance candidate);
    public void holVerifyingCandidate(HOLTranslation tr, Instance candidate, Formula checkFormula, Bounds bounds);
    public void holCandidateVerified(HOLTranslation tr, Instance candidate);
    public void holCandidateNotVerified(HOLTranslation tr, Instance candidate, Instance cex);
    public void holFindingNextCandidate(HOLTranslation tr, Formula inc);

    public void holSplitStart(HOLTranslation tr, Formula formula);
    public void holSplitChoice(HOLTranslation tr, Formula formula, Bounds bounds);
    public void holSplitChoiceSAT(HOLTranslation tr, Instance inst);
    public void holSplitChoiceUNSAT(HOLTranslation tr);

    public void holFixpointStart(HOLTranslation tr, Formula formula, Bounds bounds);
    public void holFixpointNoSolution(HOLTranslation tr);
    public void holFixpointFirstSolution(HOLTranslation tr, Instance candidate);
    public void holFixpointIncrementing(HOLTranslation tr, Formula inc);
    public void holFixpointIncrementingOutcome(HOLTranslation tr, Instance next);

    // same but with A4Solution instead of Instance
    public void holCandidateFound(HOLTranslation tr, A4Solution candidate);
    public void holVerifyingCandidate(HOLTranslation tr, A4Solution candidate, Formula checkFormula, Bounds bounds);
    public void holCandidateVerified(HOLTranslation tr, A4Solution candidate);
    public void holCandidateNotVerified(HOLTranslation tr, A4Solution candidate, A4Solution cex);
    public void holSplitChoiceSAT(HOLTranslation tr, A4Solution inst);
    public void holFixpointFirstSolution(HOLTranslation tr, A4Solution candidate);
    public void holFixpointIncrementingOutcome(HOLTranslation tr, A4Solution next);

}