import logging
import os
from typing import List, Optional

from fast_downward.driver.main import main as fd_main
from strips_hgn.config import MAX_FD_SEARCH_TIME
from strips_hgn.planning import STRIPSProblem

_log = logging.getLogger(__name__)


def get_optimal_actions_using_fd_with_timeout(
    domain: str, problem: str, timeout: int, sas_name="output.sas", plan_name="sas_plan"
) -> Optional[List[str]]:
    """
    Use Fast-Downward to get the optimal actions for the planning problem

    Parameters
    ----------
    domain: strips domain file
    problem: strips problem file
    timeout: timeout for FD
    sas_name: name of the sas file Fast Downward generates
    plan_name: name of the plan file Fast Downward generates

    Returns
    -------
    Optional[List[str]], sequential list of actions (i.e. the plan),
    or None if we could not find it
    """
    driver_options = ["--sas-file", f"{sas_name}", "--plan-file", f"{plan_name}"]
    search_options = [
        "--search",
        f"astar(lmcut(), max_time={timeout})",
    ]
    exit_code = fd_main(argv=driver_options + [domain, problem] + search_options)

    if exit_code == 12:
        print(f"Search incomplete for {domain}, {problem} within {timeout}s")
        return None
    elif exit_code != 0:
        raise RuntimeError(
            f"Something went wrong, exit code {exit_code} from Fast Downward "
            "(http://www.fast-downward.org/ExitCodes)"
        )

    # Read the plan to get actions, remove \n and ignore final cost line
    plan = open(plan_name, "r").readlines()
    plan = [line[:-1] for line in plan if line.startswith("(") and line.endswith(")\n")]

    # Remove the plan and sas_file from disk
    os.remove(plan_name)
    os.remove(sas_name)
    return plan


def get_optimal_actions_using_fd(
    problem: STRIPSProblem, sas_name: str, plan_name: str
) -> Optional[List[str]]:

    return get_optimal_actions_using_fd_with_timeout(
        problem.domain_pddl,
        problem.problem_pddl,
        MAX_FD_SEARCH_TIME,
        sas_name,
        plan_name,
    )
