#include "overlap_search.h"

#include "../task_proxy.h"

#include "../algorithms/ordered_set.h"
#include "../task_utils/successor_generator.h"
#include "../task_utils/predecessor_generator.h"

#include <algorithm>
#include <fstream>
#include <iostream>
#include <memory>
#include <numeric>
#include <set>
#include <string>

using namespace std;

namespace overlap_search {

template<typename T, typename V>
bool update_current_technique(T &iter, V end) {
    bool increment = false;
    while (iter != end && (*iter)->empty()) {
        iter++;
        increment = true;
    }
    return increment;
}

OverlapSearch::OverlapSearch(const Options &opts)
    : SearchEngine(opts),
      base_techniques(opts.get_list<shared_ptr<sampling_technique::TechniqueGBackwardNone>>("bases")),
      sampling_techniques(opts.get_list<shared_ptr<sampling_technique::SamplingTechnique>>("techniques")),
      max_base_generation_time(opts.get<double>("max_base_generation_time")),
      max_technique_generation_time(opts.get<double>("max_technique_generation_time")),
      timer(max_base_generation_time),
      base_technique(base_techniques.begin()),
      sampling_technique(sampling_techniques.begin()) {

    update_current_technique(base_technique, base_techniques.end());
    if (base_technique == base_techniques.end()){
        cerr << "At least one task has to be samples for the base" << endl;
        utils::exit_with(utils::ExitCode::SEARCH_INPUT_ERROR);
    }

    update_current_technique(sampling_technique, sampling_techniques.end());
    if (sampling_technique == sampling_techniques.end()) {
        cerr << "At least one task has to be samples for the comparison" << endl;
        utils::exit_with(utils::ExitCode::SEARCH_INPUT_ERROR);
    }
    if (any_of(sampling_techniques.begin(), sampling_techniques.end(),
            [](const std::shared_ptr<sampling_technique::SamplingTechnique> &st) {return st->empty();})){
        cerr << "Do not add sampling techniques without tasks to generate" << endl;
        utils::exit_with(utils::ExitCode::SEARCH_INPUT_ERROR);
    }
}


void OverlapSearch::initialize() {
    cout << "Initializing Overlap Manager...";
    cout << "done." << endl;
}

SearchStatus OverlapSearch::step() {
    if (base_technique != base_techniques.end()) {
        bases.push_back((*base_technique)->create_next_initial(task, task_proxy));
        update_current_technique(base_technique, base_techniques.end());
        if (timer.is_expired()) {
            base_technique = base_techniques.end();
        }
        if (base_technique == base_techniques.end()) {
            cout << "Sampled basis. Construct fast lookup" << endl;
            base_generator = make_shared<assignment_cost_generator::AssignmentCostGenerator>(
                    task_proxy, bases, 1);
            cout << "Finished create the basis" << endl;
            timer = utils::CountdownTimer(max_technique_generation_time);
        }
    } else if (sampling_technique != sampling_techniques.end()) {
        bool is_expired = timer.is_expired();
        if (!is_expired) {
            shared_ptr<AbstractTask> new_task = (*sampling_technique)->next(task, task_proxy);
            State s = State(*task, new_task->get_initial_state_values());

            nb_generated++;
            if (base_generator->lookup_cost(s) != -1) {
                nb_overlap++;
            }
        } else {
            sampling_technique = sampling_techniques.end();
        }
        if (is_expired || update_current_technique(sampling_technique, sampling_techniques.end())) {
            output << "Overlap: " << nb_overlap << "/" << nb_generated << "("
                   << nb_overlap/(double)nb_generated << ")" << endl;
            nb_generated = 0;
            nb_overlap = 0;
        }
    } else {
        cout << output.str() << flush;
        return SOLVED;
    }
    return IN_PROGRESS;
}

void OverlapSearch::add_overlap_options(OptionParser &parser) {
    parser.add_list_option<shared_ptr < sampling_technique::TechniqueGBackwardNone >> (
            "bases",
            "base sampling techniques to use");
    parser.add_list_option<shared_ptr < sampling_technique::SamplingTechnique >> (
        "techniques",
        "List of sampling technique definitions to use",
        "[]");
    parser.add_option<double>("max_base_generation_time",
            "Maximum time to use to generate partial assignments"
            "for the base. Use infinity for no limit",
            "infinity");
    parser.add_option<double>("max_technique_generation_time",
            "maximum time to use for generating possible overlapping tasks."
            "Use infinity for no limit",
            "infinity");
}
}
