#ifndef NEURAL_NETWORKS_ABSTRACT_NETWORK_H
#define NEURAL_NETWORKS_ABSTRACT_NETWORK_H

#include "../task_proxy.h"

#include "../algorithms/ordered_set.h"
#include "../utils/system.h"


namespace neural_networks {
enum OutputType {
    /*Different types of basic output a network can produce
     (not how we later interpret the output*/
    Classification, Regression
};

/*Get for a given string the output type it is associated with*/
extern OutputType get_output_type(const std::string &type);

/**
 * Base class for all networks.
 * Add new is_PROPERTY, verify_PROPERTY, get_PROPERTY if adding new properties
 * a network can produce as output (those are interpreted outputs types like
 * heuristic values or preferred operators)
 */
class AbstractNetwork {
protected:
    int verbosity = 0;
public:
    AbstractNetwork() = default;
    AbstractNetwork(const AbstractNetwork &orig) = delete;
    virtual ~AbstractNetwork() = default;

    /**
     * Initialize the network. Call this function after the constructor and
     * prior to the first usage.
     */
    virtual void initialize() {}
    /**
     * Evaluate the given state. To obtain the evaluation results use the
     * get* methods.
     * @param state state object to evaluate in the network
     */
    virtual void evaluate(const State &state) = 0;
    /**
     * Evaluate the given vector of states.  To obtain the evaluation results
     * use the get* methods.
     * @param states
     */
    virtual void evaluate(const std::vector<State> &states) = 0;

    /**
     * Tell if the concrete network class evaluates heuristic values.
     * @return
     */
    virtual bool is_heuristic();
    /**
     * Tell if the concrete network class has confidence values for its
     * heuristic values.
     * @return
     */
    virtual bool is_heuristic_confidence();
    /**
     * Tell if the concrete network class evaluates preferred operators.
     * @return
     */
    virtual bool is_preferred();

    /**
     * Verify that the concrete network class evaluates heuristic values and
     * if not stops execution.
     */
    void verify_heuristic();
    /**
     * Verify that the concrete network class evaluates preferred operators and
     * if not stops execution.
     */
    void verify_preferred();

    /**
     * Return the predicted heuristic value of the last evaluated state.
     * @return heuristic value
     */
    virtual int get_heuristic();


    /**
     * Return the confidence in the predicted heuristic value of the last
     * evaluated state.
     * @return confidence in heuristic value
     */
    virtual double get_heuristic_confidence();

    /**
     * Return the predicted heuristic value of the last batch of evaluated
     * states. Evaluating a single state alone is a batch of size 1.
     * @return predicted heuristic values in the order the states were given
     * for evaluation. If you want a persistent copy of the vector, you have
     * to do this yourself.
     */
    virtual const std::vector<int> &get_heuristics();
    /**
 * Return the confidences of predicted heuristic value of the last batch of evaluated
 * states. Evaluating a single state alone is a batch of size 1.
 * @return predicted heuristic values in the order the states were given
 * for evaluation. If you want a persistent copy of the vector, you have
 * to do this yourself.
 */
    virtual const std::vector<double> &get_heuristic_confidences();
    /**
     * Return the predicted preferred operators of the last evaluated state.
     * @return set of predicted preferred operators.
     */
    virtual ordered_set::OrderedSet<OperatorID> &get_preferred();
    /**
     * Return the predicted preferred operators of the last batch of evaluated
     * states. Evaluating a single state alone is treated as a batch of size 1.
     * @return predicted preferred operators in the order the states were given
     * for evaluation. If you want a persistent copy of the vector, you have
     * to do this yourself.
     */
    virtual std::vector<ordered_set::OrderedSet<OperatorID>> &get_preferreds();

    void set_verbosity(int new_verbosity);
    int get_verbosity();
};
}

#endif /* NEURAL_NETWORKS_NEURAL_NETWORK_H */
