import logging
import os
import sys
import subprocess
from typing import List, Optional

sys.path.append("..")

from driver.main import main as fd_main

_log = logging.getLogger(__name__)


def get_optimal_actions_using_fd_with_timeout(
    domain: str, problem: str, timeout: int, sas_name="output.sas", plan_name="sas_plan"
):
    """
    Use Fast-Downward to get the optimal actions for the planning problem

    Parameters
    ----------
    domain: strips domain file
    problem: strips problem file
    timeout: timeout for FD
    sas_name: name of the sas file Fast Downward generates
    plan_name: name of the plan file Fast Downward generates

    Returns
    -------
    Optional[List[str]], sequential list of actions (i.e. the plan),
    or None if we could not find it
    """
    driver_options = ["--sas-file", f"{sas_name}", "--plan-file", f"{plan_name}"]
    search_options = [
        "--search",
        f"'astar(lmcut(), max_time={timeout})'",
    ]
    options = " ".join(
        ["../fast-downward.py"] + driver_options + [domain, problem] + search_options
    )
    exit_code = 0
    try:
        subprocess.run(args=options, shell=True, check=True)
    except subprocess.CalledProcessError as err:
        exit_code = err.returncode
    # exit_code = fd_main(driver_options + [domain, problem] + search_options)

    if exit_code == 12:
        print(f"Search incomplete for {domain}, {problem} within {timeout}s")
        return None, None
    elif exit_code != 0:
        raise RuntimeError(
            f"Something went wrong, exit code {exit_code} from Fast Downward "
            "(http://www.fast-downward.org/ExitCodes)"
        )

    # Read the plan to get actions and cost, remove \n and ignore final cost line
    lines = open(plan_name, "r").readlines()
    plan = list()
    for line in lines:
        operator = ""
        cost = 0
        if line.startswith("(") and line.endswith(")\n"):
            operator = line[:-1]
        if line.startswith(";") and not line.endswith(")\n"):
            assert line[1:-1].isdigit(), line[1:]
            cost = int(line[1:-1])
            plan.append(tuple((operator, cost)))

    # Read states from the state file and create a trajectory
    state_filename = plan_name + "_states"
    lines = open(state_filename, "r").readlines()
    trajectory = list()

    state = set()
    for line in lines:
        if line.startswith("\n"):
            trajectory.append(frozenset(state))
            state.clear()
        elif line.startswith("Atom"):
            # Remove 'Atom ' and \n
            state.add(line[5:-1])
        elif line.startswith("NegatedAtom"):
            # Remove 'Atom ' and \n
            state.add(line[12:-1])
        else:
            raise RuntimeError(
                f"Proposition {line[:-1]} did not contain 'Atom'. Aborting."
            )

    # Remove the plan and sas_file from disk
    print("FILENAME:", state_filename)
    os.remove(plan_name)
    os.remove(sas_name)
    os.remove(state_filename)
    return plan, trajectory
