#!/usr/bin/env python

import os
import sys

if sys.version_info < (3,):
    class FileExistsError(Exception):
        pass

    class FileNotFoundError(Exception):
        pass

    import subprocess32 as subprocess
else:
    import subprocess
subprocess_decoder = ((lambda x: x) if sys.version_info < (3,)
                      else (lambda x: x.decode()))


class RunningFlag:
    flags = []

    def __init__(self, indicator, id_generator, id_checker):
        self.indicator = indicator.strip()
        self._id_generator = id_generator
        self._id_checker = id_checker

        RunningFlag.flags.append(self)

    def set(self, path, exists_ok=False):
        if not exists_ok and os.path.exists(path):
            raise FileExistsError("Cannot overwrite existing RunningFlag:",
                                  path)

        with open(path, "w") as f:
            f.write(self.indicator)
            f.write("\n")
            f.write(self._id_generator())

    def remove(self, path, remove_other=False):
        if os.path.exists(path):
            if remove_other:
                os.remove(path)
                return

            with open(path, "r") as f:
                content = f.read().split("\n", 1)
                if len(content) <= 1:
                    raise ValueError("Invalid content in RunningFlag:", content)
                if (content[0].strip() != self.indicator or
                        content[1].strip() != self._id_generator().strip()):
                    raise ValueError("Cannot remove RunningFlag created by "
                                     "other process.")
                else:
                    os.remove(path)

    def check_id(self, lines):
        return self._id_checker(lines)

    @staticmethod
    def _get_obj_from_indicator(indicator):
        for flag in RunningFlag.flags:
            if flag.indicator == indicator:
                return flag
        raise ValueError("Found not flag kind associated to the"
                         "kind described in the flag:", indicator)

    @staticmethod
    def _get_objs_from_flag(path):
        with open(path, "r") as f:
            content = f.read().split("\n")
            flag = RunningFlag._get_obj_from_indicator(content[0])
            return flag, content[1:]

    @staticmethod
    def _determine_current_flag():
        if "SLURM_JOB_ID" in os.environ:
            return RunningFlag.SLURM
        else:
            return RunningFlag.PS

    @staticmethod
    def set_flag(path, exists_ok=False):
        RunningFlag._determine_current_flag().set(path, exists_ok=exists_ok)

    @staticmethod
    def remove_flag(path, missing_ok=False, remove_other=False):
        if not os.path.exists(path):
            if missing_ok:
                return
            else:
                raise FileNotFoundError("Cannot remove not existing "
                                        "RunningFlag.")
        else:
            flag, _ = RunningFlag._get_objs_from_flag(path)
            flag.remove(path, remove_other=remove_other)

    @staticmethod
    def check_flag(path, shall_set=False, ok_missing=False):
        """
        Tests if at the given path a valid flag is set. Invalid flags are
        deleted
        :param path: path to the flag
        :param shall_set: if true and no valid flag found, automatically sets
            a flag.
        :param ok_missing: Return false (nothing running) if no file exists at
                           'path', otherwise raises an exception.
        :return: True => present and valid, False => nothing running (absent or
            invalid)
        """
        if not os.path.exists(path):
            if ok_missing:
                return False
            else:
                raise FileNotFoundError("The requested RunningFlag file does "
                                        "not exist.")

        flag, flag_id = RunningFlag._get_objs_from_flag(path)
        ret = flag.check_id(flag_id)

        # No process running => Remove invalid flag
        if not ret:
            if shall_set:
                RunningFlag.set_flag(path)
            else:
                os.remove(path)

        return ret


def _check_line_field(lines, name, command, idx_field):
    if len(lines) != 1:
        raise ValueError("Invalid data for %s RunningFlag:" % name, lines)
    test_id = lines[0].strip()
    out = subprocess_decoder(subprocess.check_output(command))

    for out_line in out.split("\n"):
        line_split = out_line.split()
        if len(line_split) <= idx_field:
            continue
        line_id = line_split[idx_field]
        if line_id == test_id:
            return True
    return False


RunningFlag.SLURM = RunningFlag(
    "SLURM",
    lambda: str(os.environ["SLURM_JOB_ID"]),
    lambda lines: _check_line_field(lines, "slurm", ["squeue"], 0))

RunningFlag.PS = RunningFlag(
    "PS",
    lambda: str(os.getpid()),
    lambda lines: _check_line_field(lines, "ps", ["ps", "-aux"], 1))
