from .base_sampler import saregister, Sampler

from .. import parser
from .. import parser_tools as parset
from .. import SampleBatchData

from ..parser_tools import main_register, ArgumentException
from ..variable import Variable
from ..problem_sorter import ProblemSorter, LexicographicIterableSorter

import os
import re
import time

class IterableFileSampler(Sampler):
    arguments = parset.ClassArguments(
        'IterableFileSampler', Sampler.arguments,
        ('iterable', False, [], str),
        ('batch', True, None, int),
        ('problem_sorter', True, None,
         main_register.get_register(ProblemSorter)),
        ('merge', True, True, parser.convert_bool,
         "If true merges the data of all sampling runs in a single container "
         "otherwise, each run has its own container."),
        order=["sampler_bridge", "iterable", "batch", "problem_sorter", "merge",
               "timeout", "variables", "id"])

    iterable = property(lambda self: tuple(self._iterable))

    def __init__(self, sampler_bridge, iterable=[],
                 batch=None, problem_sorter=None, merge=True,
                 timeout=None, variables={}, id=None):
        Sampler.__init__(self, sampler_bridge, timeout, variables, id)

        self._iterable = [x for x in iterable]  # if it would be a set or s.th
        self._batch = batch
        self._problem_sorter = problem_sorter
        self._merge = merge


    def _initialize(self):
        if self._problem_sorter is not None:
            self._iterable = self._problem_sorter.sort(self._iterable, linearize=True)
        self._next_problem = 0

    def _next_problem_generator(self):
        batch_size = (len(self._iterable) - self._next_problem
                      if self._batch is None
                      else min(len(self._iterable) - self._next_problem, self._batch))
        for i in range(batch_size):
            self._next_problem += 1
            yield self._iterable[self._next_problem - 1]


    def _sample(self, **kwargs):
        problem_generator = self._next_problem_generator()
        bridge_kwargs = {"do_merge": self._merge,
                         "merge_container": kwargs.get("data_container")}
        datas = []
        def sampling_callback(sampling_output):
            new_datas, new_merge_container = sampling_output
            datas.extend(new_datas)
            bridge_kwargs["merge_container"] = new_merge_container

        self._sample_with_timeout_skeleton(problem_generator,
                                           self._problem_path_printer,
                                           bridge_kwargs,
                                           sampling_callback,
                                           verbose=1)

        return [bridge_kwargs["merge_container"]] if self._merge else datas

    def _finalize(self):
        pass

    @staticmethod
    def parse(tree, item_cache):
        return parser.try_whole_obj_parse_process(tree, item_cache,
                                                  IterableFileSampler)


main_register.append_register(IterableFileSampler, "ifsampler")


class DirectorySampler(IterableFileSampler):
    arguments = parset.ClassArguments(
        'DirectorySampler', IterableFileSampler.arguments,
        ('root', False, None, str, "List of directories within which problem files are searched"),
        ("filter_dir", True, None, str),
        ("filter_file", True, None, str),
        ("filter_file_exec", True, None, str,
         "Callable (cannot be given by cmd line currently) which filters additionally the problems"),
        ("max_depth", True, None, int),
        ("selection_depth", True, None, int),
        ("iterable", True, None, str,
        "List of problem files independently of the directory traversing which is sampled"),
        ("ignore", True, None, str,
         "Iterable of file or directory paths to ignore if found"),
        order=["sampler_bridge", "root",
             "filter_dir", "filter_file", "filter_file_exec","ignore",
             "max_depth", "selection_depth",
             "batch", "problem_sorter", "merge",
             "timeout", "variables", "id", "iterable"]
        )


    def __init__(self, sampler_bridge, root, filter_dir=None, filter_file=None,
                 filter_file_exec=None, ignore=None,
                 max_depth=None, selection_depth=None,
                 batch=None, problem_sorter=None, merge=True, timeout=None,
                 variables={}, id=None, iterable=None):
        iterable = [] if iterable is None else iterable
        IterableFileSampler.__init__(
            self, sampler_bridge, iterable, batch, problem_sorter, merge,
            timeout, variables, id)

        if not isinstance(root, list):
            root = [root]
        self._root = root
        self._filter_dir = DirectorySampler.compile_regexes(filter_dir)
        self._filter_file = DirectorySampler.compile_regexes(filter_file)
        self._filter_file_exec = filter_file_exec
        if isinstance(self._filter_file_exec, str):
            raise ValueError("Filter File exec has to be callable and cannot be"
                             "provided via command line currently.")
        self._ignore = set() if ignore is None else set(ignore)
        self._max_depth = max_depth
        self._selection_depth = selection_depth

        for root in self._root:
            todo = [(root, 0)]
            while len(todo) > 0:
                (dir, depth) = todo[0]
                del todo[0]

                for item in os.listdir(dir):
                    path_item = os.path.join(dir, item)
                    if path_item in self._ignore:
                        continue

                    if os.path.isdir(path_item):
                        if ((
                                self._max_depth is None or depth < self._max_depth)
                                and DirectorySampler.check_regexes(
                                    path_item, self._filter_dir)):
                            todo.append((path_item, depth + 1))
                    else:
                        if ((self._selection_depth is None or depth >= self._selection_depth) and
                            DirectorySampler.check_regexes(path_item, self._filter_file) and
                            parser.is_problem_file(path_item) and
                            (self._filter_file_exec is None or self._filter_file_exec(path_item))):
                            self._iterable.append(path_item)


    @staticmethod
    def compile_regexes(regexes):
        if regexes is None:
            return []

        for idx, regex in enumerate(regexes):
            regexes[idx] = (re.compile(regex) if isinstance(regex, str)
                            else regex)

        return regexes

    @staticmethod
    def check_regexes(path, regexes):
        for regex in regexes:
            if not regex.match(path):
                return False
        return True

main_register.append_register(DirectorySampler, "directorysampler")
