from .base_sampler import saregister, Sampler

from .. import parser
from .. import parser_tools as parset
from .. import SampleBatchData

from ..parser_tools import main_register, ArgumentException
from ..variable import Variable
from ..problem_sorter import ProblemSorter, LexicographicIterableSorter

import os
import re
import sys
import threading
import time

if sys.version_info < (3,):
    import subprocess32 as subprocess
    conv_subprocess_string = lambda x: x
else:
    import subprocess
    conv_subprocess_string = lambda x: x.decode()



class GeneratorSampler(Sampler):
    arguments = parset.ClassArguments(
        'GeneratorSampler', Sampler.arguments,
        ('generator', False, None, str),
        ('path', False, None, str,
         "path where to store temporarly the generated problem"),
        ('iterations', False, None, int),
        ('generator_timeout', True, None, int),
        ('batch', True, None, int),
        ('merge', True, True, parser.convert_bool,
         "If true merges the data of all sampling runs in a single container "
         "otherwise, each run has its own container."),
        ('ignore_errors', True, False, parser.convert_bool,
         'If True, then output generated on the stderr of the generator will'
         'not cause the GeneratorSampler to abort.'),
        ('path_domain', True, None, str, 'Path to the domain file'),
        order=["sampler_bridge", "generator", "path", "iterations",
               "generator_timeout", "batch",
               "merge", "ignore_errors", "path_domain",
                "timeout","variables", "id"]
        )

    def __init__(self, sampler_bridge, generator, path, iterations,
                 generator_timeout=None, batch=None,
                 merge=True, ignore_errors=False, path_domain=None,
                 timeout=None, variables={}, id=None):
        Sampler.__init__(self, sampler_bridge, timeout, variables, id)
        self._generator = generator if isinstance(generator, list) else [generator]

        self._path = path
        self._iterations = iterations
        self._iteration = 0
        self._timeout = generator_timeout
        self._batch = batch
        self._merge = merge
        self._ignore_errors = ignore_errors
        self._path_domain = path_domain

        if self._path_domain is not None and not os.path.exists(self._path_domain):
            raise ValueError("The given domain file does not exist: %s" %
                             self._path_domain)


    def _initialize(self):
        self._iteration = 0

    def _run_generator(self):
        def kill_generator(generator_process):
            generator_process.kill()
            kill_generator.killed = True
        kill_generator.killed = False

        p = subprocess.Popen(self._generator,
                             stdout=subprocess.PIPE,
                             stderr=subprocess.PIPE,
                             env=os.environ)

        timeout_timer = (None if self._timeout is None else
                         threading.Timer(self._timeout, kill_generator, [p]))
        try:
            if timeout_timer is not None:
                timeout_timer.start()
            stdout, stderr = p.communicate()
        finally:
            if timeout_timer is not None:
                timeout_timer.cancel()

        if kill_generator.killed:
            raise TimeoutError("Generator run has timed out (%is)." %
                               self._timeout)

        return conv_subprocess_string(stdout), conv_subprocess_string(stderr)

    def _next_problem_generator(self):
        batch_size = ((self._iterations - self._iteration)
                      if self._batch is None else
                      min((self._iterations - self._iteration), self._batch))

        for i in range(batch_size):
            stdout, stderr = self._run_generator()

            if not self._ignore_errors and len(stderr) > 0:
                raise ValueError("Generator has produced an error: %s" % stderr)

            with open(self._path, "w") as f:
                f.write(stdout)

            self._iteration += 1
            yield self._path

    def _sample(self, **kwargs):
        problem_generator = self._next_problem_generator()
        bridge_kwargs = {"path_domain": self._path_domain,
                         "do_merge": self._merge,
                         "merge_container": None}
        datas = []
        def sampling_callback(sampling_output):
            new_datas, new_merge_container = sampling_output
            datas.extend(new_datas)
            bridge_kwargs["merge_container"] = new_merge_container

        self._sample_with_timeout_skeleton(problem_generator,
                                           self._problem_path_printer,
                                           bridge_kwargs,
                                           sampling_callback,
                                           verbose=1)

        return [bridge_kwargs["merge_container"]] if self._merge else datas

    def _finalize(self):
        pass

    @staticmethod
    def parse(tree, item_cache):
        return parser.try_whole_obj_parse_process(tree, item_cache,
                                                  GeneratorSampler)


main_register.append_register(GeneratorSampler, "generator_sampler")
