class BSumTNode(object):
    def __init__(self, left=None, right=None, parent=None,
                 value=None, data=None, is_root=False):
        self.left = left
        self.right = right
        self.parent = parent
        self.is_root = is_root

        self.value = (sum(x for x in (left.value, right.value) if x is not None)
                      if value is None else value)
        self.data = data

        self.check_validity()

    def check_validity(self):
        assert self.is_root or self.parent is not None
        assert ((self.left is not None or self.right is not None) ^
                (self.value is not None))
        assert self.data is None or self.value is not None

    def change_value(self, value):
        assert self.left is None and self.right is None, \
            "Can only set value for leaves"
        self.value = value
        if self.parent is not None:
            self.parent.update_sum()

    def update_sum(self):
        assert self.left is not None or self.right is not None, \
            "Only internal nodes can update their sum"
        self.value = sum(x for x in (self.left.value, self.right.value)
                         if x is not None)
        if self.parent is not None:
            self.parent.update_sum()

    def dump(self, increment="----", depth=0):
        print("%s%s:%s" % (increment * depth, self.value, str(self.data)))
        for child in (self.left, self.right):
            if child is not None:
                child.dump(increment=increment, depth=depth + 1)


class BinarySumTree(object):
    total_priority = property(lambda self: (0 if self.root is None else
                                            self.root.value))

    def __init__(self, size_limit=None):
        self.root = None
        self.leaves = []
        self.depth = 0
        self.nb_leaves_at_final_depth = 0

        self._idx_oldest_leave = 0
        self.size_limit = size_limit

    def _add_node(self, value, data):
        if self.root is None:
            self.root = BSumTNode(value=value, data=data, is_root=True)
            self.leaves.append(self.root)
            self.depth = 1
            self.nb_leaves_at_final_depth = 0
        else:
            max_nb_leaves = 2 ** self.depth
            half = max_nb_leaves / 2
            parent_index = (
                    int(self.nb_leaves_at_final_depth / 2) +
                    (0 if self.nb_leaves_at_final_depth % 2 == 0 else half))

            parent = self.leaves[parent_index]
            left = BSumTNode(parent=parent, value=parent.value, data=parent.data)
            right = BSumTNode(parent=parent, value=value, data=data)
            parent.left, parent.right = left, right
            parent.data, parent.value = None, None
            parent.update_sum()
            self.leaves[parent_index] = left
            self.leaves.append(right)

            self.nb_leaves_at_final_depth += 2
            if self.nb_leaves_at_final_depth >= max_nb_leaves:
                self.depth += 1
                self.nb_leaves_at_final_depth = 0

    def _replace_oldest_node(self, value, data):
        oldest_leave = self.leaves[self._idx_oldest_leave]
        oldest_leave.value = value
        oldest_leave.data = data
        if oldest_leave.parent is not None:
            oldest_leave.parent.update_sum()
        self._idx_oldest_leave += 1
        if self._idx_oldest_leave >= len(self.leaves):
            self._idx_oldest_leave = 0

    def add_node(self, value, data):
        if self.size_limit is None or len(self) < self.size_limit:
            self._add_node(value=value, data=data)
        else:
            self._replace_oldest_node(value=value, data=data)

    def lookup_value(self, value):
        assert self.root is not None
        node = self.root
        while True:
            assert value < node.value
            if node.left is not None and node.right is not None:
                if value < node.left.value:
                    node = node.left
                else:
                    value -= node.left.value
                    node = node.right
            elif node.left is None and node.right is None:
                return node
            elif node.left is None:
                node = node.right
            else:  # aka node.right is None
                node = node.left

    def __len__(self):
        return len(self.leaves)

    def dump(self, increment="----", depth=0):
        if self.root is None:
            print("Empty")
        else:
            self.root.dump(increment=increment, depth=depth)
