from . import SamplerBridge, StateFormat, field_parser
from .common import (generator_context_load_and_convert,
                     generator_load_and_convert_data)

from ... import main_register
from ... import parser
from ... import parser_tools as parset

from ...environments import Environment, SubprocessTask
from ...misc import StreamContext

import hashlib
import os
import random
import shlex


class FastDownwardSamplerBridge(SamplerBridge):
    arguments = parset.ClassArguments(
        'FastDownwardSamplerBridge',
        SamplerBridge.arguments,
        ("search", False, None, str, "Search argument for Fast-Downward"),
        ('streams', True, None, main_register.get_register(StreamContext),
         "StreamContext (usually with StreamDefinitions) for storing the output"),
        ("predefinitions", True, "", str, "Predefinitions for fast downward as string"),
        ("format", True, StateFormat.FD, StateFormat.get,
         "Format to represent the sampled state"),
        ("build", True, "debug64dynamic", str, "Build of Fast-Downward to use"),
        ("memory_limit", True, None, str, "Overall memory limit for Fast Downward"),
        ("fd_path", True, "None", str, "Path to the fast-downward.py script"),
        ("prune", True, True, parser.convert_bool, "Prune duplicate samples"),
        ("provide", True, False, "Does not return the generated samples in a data container (this disables also prune)"),
        ("keep_comments", True, True, "When converting the temporary samples to the output data does not skip the comment lines"),

        order=["search", "predefinitions", "streams", "fields", "parse_kwargs", "unparse_kwargs", "format", "build", "memory_limit",
               "tmp_dir", "provide", "forget", "domain",
               "domain_properties", "domain_properties_loader", "keep_comments",
               "makedir", "fd_path", "prune",
               "environment", "id"]
)

    def __init__(self, search, predefinitions="", streams=None, fields=None,
                 parse_kwargs=None, unparse_kwargs=None, format=StateFormat.FD,
                 build="debug64dynamic", memory_limit=None,
                 tmp_dir=None, provide=True, forget=0.0,
                 domain=None,
                 domain_properties=None, domain_properties_loader=None,
                 keep_comments=True,
                 makedir=False, fd_path=None, prune=True,
                 environment=None, id=None):
        SamplerBridge.__init__(self, tmp_dir, fields, parse_kwargs,
                               unparse_kwargs, provide, forget,
                               domain, domain_properties,
                               domain_properties_loader, makedir, environment, id)

        self._search = search
        self._streams = StreamContext() if streams is None else streams
        self._predefinitions = shlex.split(predefinitions)
        self._format = format
        self._build = build
        self._memory_limit = memory_limit
        self._prune = prune
        self._keep_comments = keep_comments

        if fd_path is None or fd_path == "None":
            fd_path = "."
        if not os.path.isfile(fd_path):
            possible_path = os.path.join(fd_path, "fast-downward.py")
            if not os.path.isfile(possible_path):
                raise parset.ArgumentException("Unable to find fast downward"
                                               "script in/as " + str(fd_path)
                                               + ".")
            else:
                fd_path = possible_path
        self._fd_path = fd_path

    def _initialize(self):
        pass


    def _generator_sample(self, path_problem, path_dir_tmp, path_domain,
                          data_container, skip_lines, count_lines, batch_size,
                          **kwargs):
        assert len(kwargs) == 0, ("Unknown parameters for %s: %s" %
                                  (self.__class__.__name__,
                                   ", ".join(kwargs.keys())))

        path_tmp_samples = os.path.join(path_dir_tmp,
                                        os.path.basename(path_problem)
                                        + "." + str(random.randint(0, 9999))
                                        + ".tmp")

        data_container = (None if not self._provide else
                          self.get_default_container(
                              data_container, path_problem, prune=self._prune))

        if not self._streams.may_reuse(path_problem):
            dynamic_parameters = {}
            with open(path_problem, "rb") as f:
                dynamic_parameters["hash"] = hashlib.md5(f.read()).hexdigest()

            path_common_mutexes = os.path.join(os.path.dirname(path_problem),
                                               "mutexes.sas")
            if os.path.isfile(path_common_mutexes):
                dynamic_parameters["mutexes"] = path_common_mutexes
            else:
                dynamic_parameters["mutexes"] = "none"
            dynamic_parameters["rnd_integer1"] = random.randint(0, 2147483647)

            cmd = ([self._fd_path,
                   "--plan-file", path_tmp_samples,
                   "--build", self._build] +
                   ([] if self._memory_limit is None else ["--overall-memory-limit", str(self._memory_limit)]) +
                   ([] if path_domain is None else [path_domain]) +
                   [path_problem] +
                   self._predefinitions +
                   ["--search", self._search.format(**dynamic_parameters)])

            spt = SubprocessTask("Sampling of " + path_problem, cmd)

            # TODO Add environment again
            #self._environment.queue_push(spt)
            #event.wait()
            spt.run()

            if not os.path.exists(path_tmp_samples):
                return

            for c_samples, c_problems, container in generator_load_and_convert_data(
                path_read=path_tmp_samples,
                format=self._format, field_filter=self._fields,
                prune=self._prune,
                parse_kwargs=self._parse_kwargs,
                unparse_kwargs=self._unparse_kwargs,
                path_problem=path_problem, path_domain=path_domain,
                data_container=data_container,
                delete=True, forget=self._forget, write_context=self._streams,
                domain_properties=self._domain_properties,
                keep_comments=self._keep_comments, batch_size=batch_size,
                skip_lines=skip_lines,
                count_lines=count_lines
            ):
                yield c_samples, c_problems, data_container

        else:
            if self._provide:
                assert False, "TODO: test if generator works"
                for c_samples, c_problems, data_container in generator_context_load_and_convert(
                        self._streams, data_container,
                        format=self._format, field_filter=self._fields,
                        prune=self._prune,
                        parse_kwargs=self._parse_kwargs,
                        unparse_kwargs=self._unparse_kwargs,
                        path_problem=path_problem, path_domain=path_domain,
                        skip=self._skip, skip_magic=self._skip_magic,
                        forget=self._forget,
                        domain_properties=self._domain_properties,
                        batch_size=batch_size,
                        skip_lines=skip_lines,
                        count_lines=count_lines
                ):
                    yield c_samples, c_problems, data_container

    def _finalize(self):
        self._streams.finalize()

    @staticmethod
    def parse(tree, item_cache):
        return parser.try_whole_obj_parse_process(tree, item_cache,
                                                  FastDownwardSamplerBridge)


main_register.append_register(FastDownwardSamplerBridge, "fdbridge")
