from __future__ import print_function

from .field_parser import FIELD_PARSERS
from ... import parser

from ...misc import StreamContext

from .... import translate

import gzip
import json
import os
import psutil
import random
import sys

# Python 2/3 compatibility
gzip_write_converter = lambda x: x.encode()
gzip_read_converter = lambda x: x.decode()
if sys.version_info[0] == 2:
    gzip_write_converter = lambda x: x
    gzip_read_converter = lambda x: x

try:
    FileNotFoundError
except NameError:
    FileNotFoundError = IOError

# Globals
MAGIC_WORD = "# MAGIC FIRST LINE\n"


class MetaFields(object):
    SAMPLE_TYPE = "sample_type"
    PROBLEM_HASH = "problem_hash"
    MODIFICATION_HASH = "modification_hash"
    DELIMITER = "delimiter"
    FIELDS = "fields"
    FIELD_NAME = "name"
    FIELD_TYPE = "type"


class InvalidSampleEntryError(Exception):
    pass


class DataCorruptedError(Exception):
    pass


class ReservoirSampling(object):
    def __init__(self, size):
        self.size = size
        self.reservoir = []
        self.counter = 0

    def _inc_counter(self):
        self.counter += 1

    def _check(self):
        if len(self.reservoir) < self.size:
            return True
        return random.random() < (self.size / float(self.counter))

    def _add(self, *args, **kwargs):
        if len(self.reservoir) == self.size:
            del self.reservoir[int(random.random() * self.size)]
        self.reservoir.append((args, kwargs))
        assert len(self.reservoir) <= self.size, "Exceeded reservoir size"

    def add(self, *args, **kwargs):
        self._inc_counter()
        if self._check():
            self._add(*args, **kwargs)

    def reset(self):
        self.reservoir = []
        self.counter = 0


class SampleReservoirSampling(ReservoirSampling):
    def __init__(self, size, data_container):
        super(SampleReservoirSampling, self).__init__(size)
        self._data_container = data_container
        self._last_key = None

    def __len__(self):
        return len(self._data_container) + len(self.reservoir)

    def reset(self):
        ReservoirSampling.reset(self)
        self._last_key = None

    def submit_reservoir(self, limit=None):
        added = []
        for no, (args, kwargs) in enumerate(self.reservoir):
            if limit is not None and no >= limit:
                break
            assert len(args) == 2 and len(kwargs) == 0 and len(args[1]) == 2
            if self._data_container.add(*args[1][0], **args[1][1]):
                added.append(args[0])
        self.reset()
        return added

    def inc_counter(self, key):
        added = None
        if key != self._last_key:
            if self._last_key is not None:
                added = self.submit_reservoir()
            self._last_key = key
        self._inc_counter()
        return added

    def check(self):
        return self._check()

    def add(self, *args, **kwargs):
        self._add(*args, **kwargs)


def split_meta(line):
    line = line.strip()
    if not line.startswith("{"):
        return None, line

    level = 1
    for i in range(1, len(line)):
        if line[i] == "{":
            level += 1
        elif line[i] == "}":
            level -= 1
            if level == 0:
                return line[: i + 1], line[i + 1:]

    raise DataCorruptedError("Entry has an opening, but not closing tag: "
                             + line)



def extract_from_meta(meta, key, default=None, converter=None):
    if meta is None:
        return default

    if key not in meta:
        return default

    else:
        value = meta[key]
        return value if converter is None else converter(value)


def extract_and_replace_from_meta(meta, key, default=None, new_value=None,
                                  str2val=None):
    if meta is None:
        meta = {}
    value = extract_from_meta(meta, key, default, converter=str2val)

    if value is None:
        if default is None:
            raise ValueError("Unable to determine value for key " + str(key))
        else:
            value = default

    if new_value is not None:
        meta[key] = str(new_value)
    return value, meta

LINE_TYPE_IGNORE = 0
LINE_TYPE_COMMENT = 1
LINE_TYPE_SAMPLE = 2
def check_sample_line_type(line):
    """
    Gets a line from a data file and checks which kind of information the
    line contains.

    :param line: line to process
    :return:
        LINE_TYPE_IGNORE: lines to ignore (e.g. empty lines)
        LINE_TYPE_COMMENT: lines containing comments
        LINE_TYPE_SAMPLE: lines containing data samples
    """
    if line == "":
        return LINE_TYPE_IGNORE
    elif line.startswith("#"):
        return LINE_TYPE_COMMENT
    else:
        return LINE_TYPE_SAMPLE


def parse_reparse_sample(
        meta, smeta, sample_data,
        pddl_task, sas_task,
        state_format, default_state_format,
        field_filter=None,
        domain_properties=None,
        shall_reparse=False,
        parse_kwargs=None,
        reparse_kwargs=None):
    """
    Takes the data of an sample and 1. parses it into the desired format
    and if requested reparses it into a string

    :return: (field_names (names of the fields parsed/reparsed),
              parsed_data,
              reparsed_data)
    """
    data_fields = []
    data_parsed = []
    data_reparsed = "" if shall_reparse else None

    fields = extract_from_meta(meta, MetaFields.FIELDS, None)
    if fields is None:
        raise DataCorruptedError("No field information in the entry to load: "
                                 "%s" % smeta)

    delimiter = extract_from_meta(meta, MetaFields.DELIMITER, ";")
    split_data = [x.strip() for x in sample_data.split(delimiter)]
    if len(split_data) != len(fields):
        raise DataCorruptedError("Entry has more or less fields as described in"
                                 "its meta: %i/%i (present/expected)" %
                                 (len(split_data), len(fields)))

    field_meta_to_remove = []
    for idx_field, (field_meta, field_data) in enumerate(zip(fields, split_data)):
        field_name = extract_from_meta(field_meta, MetaFields.FIELD_NAME, None)
        field_type = extract_from_meta(field_meta, MetaFields.FIELD_TYPE, None)

        if field_filter is not None and field_name not in field_filter:
            field_meta_to_remove.append(idx_field)
            continue

        if field_type not in FIELD_PARSERS:
            raise DataCorruptedError("The type of a field is not known to the "
                                     "framework and cannot be parsed: %s" %
                                     str(field_type))

        data_fields.append(field_name)

        # HACK! format and new_format should only be passed to StateFieldParsers
        new_field_meta, new_parsed_data = FIELD_PARSERS[field_type].parse(
            field_data, field_meta,
            None if parse_kwargs is None else parse_kwargs.get(field_name),
            pddl_task=pddl_task, sas_task=sas_task,
            format=default_state_format, new_format=state_format,
            domain_properties=domain_properties)
        data_parsed.append(new_parsed_data)

        if shall_reparse:
            new_field_meta, new_field_data = FIELD_PARSERS[field_type].unparse(
                data_parsed[-1], new_field_meta,
                None if reparse_kwargs is None else reparse_kwargs.get(field_name),
                pddl_task=pddl_task, sas_task=sas_task,
                format=default_state_format, new_format=state_format)
            meta[MetaFields.FIELDS][idx_field] = new_field_meta
            data_reparsed += new_field_data + delimiter

    if shall_reparse:
        for idx_field in reversed(field_meta_to_remove):
            del meta[MetaFields.FIELDS][idx_field]
        data_reparsed = json.dumps(meta) + data_reparsed[:-len(delimiter)] + "\n"

    return data_fields, tuple(data_parsed), data_reparsed


def load_sample_line(line, data_container, format, pddl_task, sas_task,
                     field_filter=None,
                     pruning_set=None, default_format=None, shall_reparse=False,
                     parse_kwargs=None, unparse_kwargs=None,
                     domain_properties=None,
                     sample_types=None, is_container_sampler=None,
                     reference_states=None):
    """

    :param line: line to load
    :param data_container: data_container to which to add the parsed data or None
    :param format:
    :param pddl_task:
    :param sas_task:
    :param pruning_set:
    :param default_format:
    :param shall_reparse: convert the parsed (and modified) sample back to text
    :param parse_kwargs:
    :param unparse_kwargs:
    :param domain_properties:
    :param field_filter: contains all field names which shall be loaded or None
                         to load all fields (object requires an 'in' operator)

    :return: [(reparsed_data, parsed_data, task_identifier), ...]
             None if using reservoir sampling and samples are not submitted or
                     sample was not added for another reason
             len([..]) == 1 if not using reservoir sampling or reservoir of
                               length 1 was submitted
             len([..]) > 1  if using reservoir sampling and reservoir of
                               length > 2 was submitted.
    """
    smeta, data = split_meta(line)
    # Check meta and data segment are valid
    if smeta is None:
        raise InvalidSampleEntryError("Missing meta tag. Cannot infer entry "
                                      "fields.")
    if data is None or data == "":
        raise DataCorruptedError("Invalid data set entry: " + str(line))

    meta = json.loads(smeta)

    # Check if sample is of required type:
    entry_type = extract_from_meta(meta, MetaFields.SAMPLE_TYPE)
    if sample_types is not None:
        if entry_type is None:
            raise InvalidSampleEntryError("Missing %s field in sample" %
                                          MetaFields.SAMPLE_TYPE)
        elif entry_type not in sample_types:
            return None

    # Pruning of duplicate entries
    # TODO improve pruning, different irrelevant meta prevent pruning
    if pruning_set is not None:
        h_data = (smeta, data)
        if h_data in pruning_set:
            return None
        else:
            pruning_set.add(h_data)

    task_identifier = (
        extract_from_meta(meta, MetaFields.PROBLEM_HASH),
        extract_from_meta(meta, MetaFields.MODIFICATION_HASH))

    # If reservoir sampling: Check if entry is selected (otherwise return)
    return_value = None
    if is_container_sampler:
        return_value = data_container.inc_counter(task_identifier)
        if not data_container.check():
            return return_value

    data_fields, data_parsed, data_reparsed = parse_reparse_sample(
        meta, smeta, data,
        pddl_task=pddl_task,
        sas_task=sas_task,
        state_format=format,
        default_state_format=default_format,
        field_filter=field_filter,
        domain_properties=domain_properties,
        shall_reparse=shall_reparse,
        parse_kwargs=parse_kwargs,
        reparse_kwargs=unparse_kwargs
    )

    if data_container is None:
        assert return_value is None
        return_value = [(data_reparsed, data_parsed, task_identifier)]
    elif is_container_sampler:
        data_container.add(
            (data_reparsed, data_parsed, task_identifier),
            ((data_parsed,), {"type": entry_type, "fields": data_fields}))
    else:  # container adds directly
        assert return_value is None
        if data_container.add(data_parsed, type=entry_type, fields=data_fields):
            return_value = [(data_reparsed, data_parsed, task_identifier)]

    return return_value


def generator_read_data_lines(
        file_data, file_problem,
        write_context,
        delete=False,
        skip_magic_word_check=False):
    write_context = StreamContext() if write_context is None else write_context

    # Reading techniques and settings
    read_format_found = False
    read_techniques = [(open, lambda x: x), (gzip.open, gzip_read_converter)]

    for (read_open, read_conv) in read_techniques:
        try:
            with read_open(file_data, "r") as src:
                with write_context.next(file_problem) as trg:

                    first = False if skip_magic_word_check else True
                    for line in src:
                        line = read_conv(line)

                        if first:
                            first = False
                            read_format_found = line == MAGIC_WORD
                            if read_format_found:
                                trg.write(MAGIC_WORD)
                            else:
                                break
                        yield line, trg

                    if skip_magic_word_check:
                        read_format_found = True
                    if read_format_found:
                        if delete:
                            os.remove(file_data)
                        break

        except UnicodeDecodeError:
            pass
        except InvalidSampleEntryError as e:
            if sys.version_info < (3,):
                pass
            else:
                raise e
        except EOFError:
            print("EOFError in %s" % file_problem, file=sys.stderr)
            return

    if not read_format_found:
        raise ValueError("the given file could not be correctly opened with one "
                         "of the known techniques.")


def generator_load_and_convert_data(
        path_read, format, default_format=None,
        field_filter=None, prune=True,
        parse_kwargs=None, unparse_kwargs=None,
        path_problem=None, path_domain=None,
        pddl_task=None, sas_task=None,
        data_container=None,
        delete=False,
        skip_magic_word_check=False, forget=0.0,
        write_context=None,
        domain_properties=None,
        keep_comments=True,
        max_memory=None,
        sample_types=None,
        samples_per_problem=None,
        max_samples=None,
        batch_size=None,
        skip_lines=None,
        count_lines=False,
        reference_states=None):
    """

    :param path_read: Path to the file containing the samples to load
    :param format: StateFormat into which to convert the samples
    :param default_format: Input format of the loading samples
                           (if not specified within sample entry)
    :param field_filter: contains all field names which shall be loaded or None
                         to load all fields (object requires an 'in' operator)
    :param prune: If true prunes duplicate entries (the meta information is
                  except for the type attribute ignored)
    :param parse_kwargs: {field: [{key:value},{}...], other_field: ...}
                         Map providing for the given fields additional parameters
                         for parsing the data of the named fields.
    :param unparse_kwargs: like parse_kwargs, but for unparsing a field
    :param path_problem: Path to the problem description to which the samples
                         belong. For converting the samples, the problem has to
                         be known. Provide at least one of the following
                         parameter settings:
                         - pddl_task and sas_task
                         - path_problem and path_domain
                         - path_problem and path_domain can be automatically
                           detected
                        If multiple settings are given, the first available one
                        of this list is used.
    :param path_domain: Path to the domain description of the problem
                        (for interactions of this parameter see path_problem)
    :param pddl_task: PDDL Task object of the problem
                     (for interactions of this parameter see path_problem)
    :param sas_task: SAS Task object of the problem
                     (for interactions of this parameter see path_problem)
    :param data_container: Data gathering object for the loaded entries. Object
                           requires an add(entry, type) method
                           (e.g. SizeBatchData). If None is given, the adding
                           is skipped.
    :param delete: Deletes path_input at the end of this method
    :param skip_magic_word_check: Deprecated. Use this to read old sample files.
                                  The outcome when reading a file with a wrong
                                  reading technique (e.g. compressed file via
                                  open) is undetermined.
    :param write_context: If given, this context manager will be used to write
                          the output into. It is required to have the following
                          methods: open(file, mode) which informs about the file
                          we would like to write and returns the context manager,
                          write(*args, **kwargs), and close()
    :param batch_size: In each iteration, batch_size many samples are added to
                       data_container (in the last iteration less might be added.
    :param skip_lines: None -> skip no lines, k -> skip first k sample lines (k>=0)
    :yield: data_container with the samples
    :return:
    """
    assert skip_lines is None or skip_lines >= 0
    assert write_context is None or samples_per_problem is None, \
        "Reservoir sampling and writing the sampled samples is not implemented."

    if max_samples is not None and max_samples <= 0:
        return
    if max_memory is not None:
        process = psutil.Process(os.getpid())
        base_memory = process.memory_info().rss
        iter_memory = 0  # Do not check memory every time, just every X times

    if pddl_task is None or sas_task is None:
        if path_problem is None:
            raise ValueError("Missing needed arguments. Either provide pddl"
                             " and sas task objects or a path to the problem"
                             "file (and if not trivially findable also a path"
                             "to the domain file).")
        if path_domain is None:
            path_domain = parser.find_domain(path_problem)

        (pddl_task, sas_task) = translate.translator.main(
            [path_domain, path_problem,
             "--no-sas-file", "--log-verbosity", "ERROR"])

    shall_reparse = write_context != None
    old_hashs = set() if prune else None
    data_target = (data_container if samples_per_problem is None else
                      SampleReservoirSampling(samples_per_problem,
                                              data_container))

    batch_samples, total_samples = 0, 0
    batch_problems, total_problems = 0, 0
    previous_task_identifier = None

    def process_load_samples(list_samples, owriter, pls_previous_task_identifier):
        if reference_states is not None:
            nb_references_seen = 0
            for _, pls_parsed_data, _ in list_samples:
                assert len(pls_parsed_data) == 1
                if pls_parsed_data[0] in reference_states:
                    nb_references_seen += 1
            return (pls_previous_task_identifier,
                    len(list_samples), nb_references_seen)

        pls_nb_new_samples, pls_nb_new_tasks = 0, 0
        for reparsed_sample, _, task_identifier in list_samples:
            if (shall_reparse and reparsed_sample is not None and
                    task_identifier is not None):
                owriter.write(reparsed_sample)
            pls_nb_new_samples += 1
            if task_identifier != pls_previous_task_identifier:
                pls_nb_new_tasks += 1
                pls_previous_task_identifier = task_identifier
        return (pls_previous_task_identifier, pls_nb_new_samples,
                pls_nb_new_tasks)

    for line, writer in generator_read_data_lines(
        file_data=path_read,
        file_problem=path_problem,
        write_context=write_context,
        delete=delete,
        skip_magic_word_check=skip_magic_word_check
    ):
        if forget != 0.0 and random.random() < forget:
            continue

        line = line.strip()
        line_type = check_sample_line_type(line)

        if line_type == LINE_TYPE_IGNORE:
            pass

        elif line_type == LINE_TYPE_COMMENT:
            if (shall_reparse and keep_comments and
                    not count_lines and
                    (skip_lines is None or skip_lines <= 0)):
                writer.write(line + "\n")

        elif line_type == LINE_TYPE_SAMPLE:
            if skip_lines is not None and skip_lines > 0:
                skip_lines -= 1
                continue

            nb_new_samples = 0
            nb_new_tasks = 0
            if count_lines:
                nb_new_samples = 1
            else:
                load_samples = load_sample_line(
                    line,
                    data_target if reference_states is None else None, format,
                    pddl_task, sas_task, field_filter,
                    old_hashs,
                    default_format,
                    shall_reparse,
                    parse_kwargs,
                    unparse_kwargs,
                    domain_properties=domain_properties,
                    sample_types=sample_types,
                    is_container_sampler=(samples_per_problem is not None),
                    reference_states=reference_states)

                if load_samples is not None:
                    (previous_task_identifier, nb_new_samples,
                     nb_new_tasks) = process_load_samples(
                        load_samples, writer,previous_task_identifier)

            total_samples += nb_new_samples
            batch_samples += nb_new_samples
            total_problems += nb_new_tasks
            batch_problems += nb_new_tasks

            if max_samples is not None and total_samples >= max_samples:
                break
            if batch_size is not None and batch_samples >= batch_size:
                yield batch_samples, batch_problems, data_container
                batch_samples = 0
                batch_problems = 0

        else:
            assert False

        # Check memory limit at end of loop
        if max_memory is not None:
            iter_memory += 1
            if iter_memory >= 100:
                iter_memory = 0
                curr_memory = process.memory_info().rss
                if curr_memory - base_memory > max_memory:
                    print("load_and_convert_data hit memory "
                          "limit: %.2f/%.2f" %
                          (curr_memory - base_memory, max_memory))
                    break

    if samples_per_problem is not None:
        limit = (None if max_samples is None else (max_samples - total_samples))
        (previous_task_identifier, nb_new_samples,nb_new_tasks
         ) = process_load_samples(
            data_target.submit_reservoir(limit), writer,
            previous_task_identifier)
        total_samples += nb_new_samples
        batch_samples += nb_new_samples
        total_problems += nb_new_tasks
        batch_problems += nb_new_tasks

    if batch_samples > 0:
        yield batch_samples, batch_problems, data_container


def load_and_convert_data(path_read, format, default_format=None,
                          field_filter=None, prune=True,
                          parse_kwargs=None, unparse_kwargs=None,
                          path_problem=None, path_domain=None,
                          pddl_task=None, sas_task=None,
                          data_container=None,
                          delete=False,
                          skip_magic_word_check=False, forget=0.0,
                          write_context=None,
                          domain_properties=None,
                          keep_comments=True,
                          max_memory=None,
                          sample_types=None,
                          samples_per_problem=None,
                          max_samples=None,
                          skip_lines=None,
                          count_lines=False):
    """

    :param path_read: Path to the file containing the samples to load
    :param format: StateFormat into which to convert the samples
    :param default_format: Input format of the loading samples
                           (if not specified within sample entry)
    :param field_filter: contains all field names which shall be loaded or None
                         to load all fields (object requires an 'in' operator)
    :param prune: If true prunes duplicate entries (the meta information is
                  except for the type attribute ignored)
    :param parse_kwargs: {field: [{key:value},{}...], other_field: ...}
                         Map providing for the given fields additional parameters
                         for parsing the data of the named fields.
    :param unparse_kwargs: like parse_kwargs, but for unparsing a field
    :param path_problem: Path to the problem description to which the samples
                         belong. For converting the samples, the problem has to
                         be known. Provide at least one of the following
                         parameter settings:
                         - pddl_task and sas_task
                         - path_problem and path_domain
                         - path_problem and path_domain can be automatically
                           detected
                        If multiple settings are given, the first available one
                        of this list is used.
    :param path_domain: Path to the domain description of the problem
                        (for interactions of this parameter see path_problem)
    :param pddl_task: PDDL Task object of the problem
                     (for interactions of this parameter see path_problem)
    :param sas_task: SAS Task object of the problem
                     (for interactions of this parameter see path_problem)
    :param data_container: Data gathering object for the loaded entries. Object
                           requires an add(entry, type) method
                           (e.g. SizeBatchData). If None is given, the adding
                           is skipped.
    :param delete: Deletes path_input at the end of this method
    :param skip_magic_word_check: Deprecated. Use this to read old sample files.
                                  The outcome when reading a file with a wrong
                                  reading technique (e.g. compressed file via
                                  open) is undetermined.
    :param write_context: If given, this context manager will be used to write
                          the output into. It is required to have the following
                          methods: open(file, mode) which informs about the file
                          we would like to write and returns the context manager,
                          write(*args, **kwargs), and close()
    :return:
    """
    first_round = True
    for c_samples, c_problems, _ in generator_load_and_convert_data(
        path_read=path_read, format=format, default_format=default_format,
        field_filter=field_filter, prune=prune, parse_kwargs=parse_kwargs,
        unparse_kwargs=unparse_kwargs,
        path_problem=path_problem, path_domain=path_domain,
        pddl_task=pddl_task, sas_task=sas_task, data_container=data_container,
        delete=delete, skip_magic_word_check=skip_magic_word_check,
        forget=forget, write_context=write_context,
        domain_properties=domain_properties, keep_comments=keep_comments,
        max_memory=max_memory, sample_types=sample_types,
        samples_per_problem=samples_per_problem, max_samples=max_samples,
        batch_size=None, skip_lines=skip_lines, count_lines=count_lines):
        assert first_round
        first_round = False
    assert not first_round
    return c_samples, c_problems


def generator_context_load_and_convert(
        stream_context, data_container, format, field_filter=None,
        prune=True,
        parse_kwargs=None, unparse_kwargs=None,
        path_problem=None, path_domain=None,
        pddl_task=None, sas_task=None,
        default_format=None, skip=True, skip_magic=False, forget=0.0,
        write_context=None,
        domain_properties=None,
        keep_comments=True,
        max_memory=None,
        sample_types=None,
        samples_per_problem=None,
        max_samples=None,
        batch_size=None,
        skip_lines=None,
        count_lines=False,
        reference_states=None):
    if max_memory is not None:
        process = psutil.Process(os.getpid())
        base_memory = process.memory_info().rss
        curr_memory = base_memory

    nb_samples = 0
    nb_problems = 0
    paths_samples = set()
    for stream in stream_context._streams:
        paths_samples.add(stream.get_next_path(path_problem))

    for path_samples in paths_samples:
        if not os.path.exists(path_samples):
            if skip:
                continue
            else:
                raise FileNotFoundError("A sample file to load does not exist:"
                                        + str(path_samples))
        samples_left = None if max_samples is None else (max_samples - nb_samples)
        for c_samples, c_problems, data_container in generator_load_and_convert_data(
                path_read=path_samples,
                format=format, default_format=default_format,
                field_filter=field_filter, prune=prune,
                parse_kwargs=parse_kwargs,
                unparse_kwargs=unparse_kwargs,
                path_problem=path_problem, path_domain=path_domain,
                pddl_task=pddl_task, sas_task=sas_task,
                data_container=data_container,
                skip_magic_word_check=skip_magic, forget=forget,
                write_context=write_context,
                domain_properties=domain_properties,
                keep_comments=keep_comments,
                max_memory=None if max_memory is None else
                (max_memory - min(0, (curr_memory - base_memory))),
                sample_types=sample_types,
                samples_per_problem=samples_per_problem,
                max_samples=samples_left,
                batch_size=batch_size,
                skip_lines=skip_lines,
                count_lines=count_lines,
                reference_states=reference_states,
        ):
            yield c_samples, c_problems, data_container

            nb_samples += c_samples
            nb_problems += c_problems
            #print("TODO: fill batch if one file ends and the next one starts")
            if max_samples is not None and nb_samples >= max_samples:
                break

            if max_memory is not None:
                curr_memory = process.memory_info().rss
                if curr_memory - base_memory > max_memory:
                    print("context_load hit memory limit: %.2f/%.2f" % (
                    curr_memory - base_memory, max_memory))
                    break



def context_load(
        stream_context, data_container, format, field_filter=None,
        prune=True,
        parse_kwargs=None, unparse_kwargs=None,
        path_problem=None, path_domain=None,
        pddl_task=None, sas_task=None,
        default_format=None, skip=True, skip_magic=False, forget=0.0,
        domain_properties=None,
        max_memory=None,
        sample_types=None,
        samples_per_problem=None,
        max_samples=None,
        skip_lines=None,
        count_lines=False
):
    assert False, "seems to be not used anymore"
    first_round=True
    c_samples, c_problems = 0, 0
    for c_samples, c_problems, data_container in generator_context_load_and_convert(
            stream_context=stream_context,
            data_container=data_container,
            format=format,
            field_filter=field_filter,
            prune=prune,
            parse_kwargs=parse_kwargs,
            unparse_kwargs=unparse_kwargs,
            path_problem=path_problem,
            path_domain=path_domain,
            pddl_task=pddl_task,
            sas_task=sas_task,
            default_format=default_format,
            skip=skip,
            skip_magic=skip_magic,
            forget=forget,
            domain_properties=domain_properties,
            max_memory=max_memory,
            sample_types=sample_types,
            samples_per_problem=samples_per_problem,
            max_samples=max_samples,
            batch_size=None,
            skip_lines=skip_lines,
            count_lines=count_lines
    ):
        assert first_round
        first_round = False

    return c_samples, c_problems, data_container




